/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Reader IC command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHHALHW_SAM_CMD_RC_H
#define PHHALHW_SAM_CMD_RC_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_RIC 14_ReaderIC
 * \brief Implementation of SAM Reader IC commands.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_RIC_Defines Defines
 * \brief Definitions for Reader IC commands
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_RIC_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_RC_READ_REGISTER                                   0xEEU   /**< SAM Insturction code for RC_ReadRegister command. */
#define PHHAL_HW_SAM_CMD_INS_RC_WRITE_REGISTER                                  0x1EU   /**< SAM Insturction code for RC_WriteRegister command. */
#define PHHAL_HW_SAM_CMD_INS_RC_RF_CONTROL                                      0xCFU   /**< SAM Insturction code for RC_RFControl command. */
#define PHHAL_HW_SAM_CMD_INS_RC_INIT                                            0xE5U   /**< SAM Insturction code for RC_Int command. */
#define PHHAL_HW_SAM_CMD_INS_RC_LOAD_REGISTER_VALUE_SET                         0x2EU   /**< SAM Insturction code for RC_LoadRegisterValueSet command. */
/**
 * end of group phhalHw_Sam_Cmd_RIC_Defines_CommandCodes
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_RIC_Defines
 * @}
 */

/**
 * \brief Read the content of one or more register(s) of the connected reader chip.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pRegAddr and \b pValue are NULL.
 * \retval #PH_ERR_LENGTH_ERROR         If length received from SAM is not equal
 *                                      to \b bRegAddrLen
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_RC_ReadRegister(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pRegAddr,                                                             /**< [In] Address for the registers to be read. */
        uint8_t bRegAddrLen,                                                            /**< [In] Length of bytes available in \b pRegAddr buffer. */
        uint8_t * pValue                                                                /**< [Out] Register(s) content in the same order as the
                                                                                         *         command data field address(es).
                                                                                         */
    );

/**
 * \brief Write the content of one or more register(s) of the connected reader chip.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData is NULL.
 * \retval #PH_ERR_LENGTH_ERROR         If response is received from SAM
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_RC_WriteRegister(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pData,                                                                /**< [In] The registers to tbe updated. This buffer should contain
                                                                                         *        register address followed by value. \n
                                                                                         *        The format should be as mentioned below. \n
                                                                                         *        RegAdd || Value || RegAdd || Value || ...
                                                                                         *        || RegAdd || Value
                                                                                         */
        uint8_t bDataLen                                                                /**< [In] Length of bytes available in \b pData buffer. */
    );

/**
 * \brief Turn the radio frequency field On or Off
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_RC_RFControl(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wTime                                                                  /**< [In] Shut-down time for the RF field, zero turns the field
                                                                                         *        off.
                                                                                         */
    );

/**
 * \brief Istablishes the serial connection between MIFARE SAM and Reader IC
 * and initializes the reader chip with the register values stored in the
 * selected register value set.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_LENGTH_ERROR         If response is received from SAM
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_RC_Init(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bLoadReg                                                                /**< [In] The Load register information. */
    );

/**
 * \brief Stores a customer defined register value set for the PN512/RC52x/RC663 in the
 * non-volatile memory of the MIFARE SAM
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData is NULL.
 * \retval #PH_ERR_LENGTH_ERROR         If response is received from SAM
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_RC_LoadRegisterValueSet(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bStoreReg,                                                              /**< [In] Number of register value set to be used for storing the values. */
        uint8_t * pData,                                                                /**< [In] List of Register descriptions. This buffer should be as
                                                                                         *        meniotned below.
                                                                                         *          - If RC663: SpeedID || ItemCount || RegAddr || RegValue ...
                                                                                         *            [|| RegAddr || RegValue]
                                                                                         *          - If RC5XX: RegAddr || RegValue ... [|| RegAddr || RegValue]
                                                                                         */
        uint8_t bDataLen                                                                /**< [In] The length of bytes available in \b pData buffer. */
    );

/**
 * end of group phhalHw_Sam_Cmd_RIC
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_RC_H */
