/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Programmable Logic command implementation of Reader
 * Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHHALHW_SAM_CMD_PL_H
#define PHHALHW_SAM_CMD_PL_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

#define PHHAL_HW_SAM_PL_CHUNK_SIZE                                              128U    /* Maximum Chunk size for exchanging informatin to SAM */

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_PL 13_ProgrammableLogic
 * \brief Implementation of SAM Programmable Logic commands.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_PL_Defines Defines
 * \brief Definitions for Programmable Logic commands
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_PL_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_CMD_INS_SAM_PL_EXEC                                            0xBEU   /**< SAM Insturction code for SAM_PLExec command. */
#define PHHAL_HW_CMD_INS_SAM_PL_UPLOAD                                          0xBFU   /**< SAM Insturction code for SAM_PLUpload command. */
/**
 * end of group phhalHw_Sam_Cmd_PL_Defines_CommandCodes
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_PL_Defines
 * @}
 */

/**
 * \brief The command SAM_PLExec is used to trigger the execution of the programmable logic.
 * The command payload PLData is passed to the PL native code for processing. The SAM shall
 * not interpret the command payload.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful with chaining status.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pPLData and \b pPLRespLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_PLExec(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bLFI,                                                                   /**< [In] Option for updating the P1 information of Sam frame.
                                                                                         *          - \ref PHHAL_HW_SAM_ISO7816_LAST_FRAME
                                                                                         *                 "Default or Last Frame"
                                                                                         *          - \ref PHHAL_HW_SAM_ISO7816_CHAINED_FRAME
                                                                                         *                 "Chainned Frame"
                                                                                         */
        uint8_t * pPLData,                                                              /**< [In] Programmable Logic command data. */
        uint8_t bPLDataLen,                                                             /**< [In] Length of bytes available in \b pPLData buffer. */
        uint8_t ** ppPLResp,                                                            /**< [Out] Buffer holding the Programmable Logic response data.
                                                                                         *          - Actual data received from SAM
                                                                                         *          - Actual error data received from SAM. The response
                                                                                         *            will be of 2 bytes.
                                                                                         */
        uint16_t * pPLRespLen                                                           /**< [Out] Length of bytes available in \b ppPLResp buffer. */
    );

/**
 * \brief The command SAM_PLUpload is used to update the code of the programmable logic.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful with chaining status.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pPLCode and \b pPLReKey are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_PLUpload(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bIsFirstFrame,                                                          /**< [In] Option to represent the first frame where the UploadCtr
                                                                                         *        will be exchanged.
                                                                                         *          - \ref PH_OFF "Not First Frame"
                                                                                         *          - \ref PH_ON "First Frame"
                                                                                         */
        uint8_t bIsFinalFrame,                                                          /**< [In] Option to represent the last frame. If set, the LE byte
                                                                                         *        will be exchanged and PLUploadACK will be received from
                                                                                         *        SAM and will be validated internally.
                                                                                         *          - \ref PH_OFF "Not Final Frame"
                                                                                         *          - \ref PH_ON "Final Frame"
                                                                                         */
        uint16_t wUploadCtr,                                                            /**< [In] The upload counter value. This should be greater than
                                                                                         *        the one availabe in SAM.
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Key number of Upload key (Ku) in Software KeyStore. */
        uint8_t bKeyVer,                                                                /**< [In] Key version of Upload key (Ku) in software KeyStore. */
        uint8_t * pPLCode,                                                              /**< [In] Plain Programmable Logic code. */
        uint16_t wPLCodeLen,                                                            /**< [In] Length of bytes available in \b pPLCode buffer. */
        uint8_t * pPLReKey,                                                             /**< [In] The Rekey to be used for next crypto segment. This should
                                                                                         *        have the next SessionENC key followed by the SessionMAC key.
                                                                                         */
        uint8_t bPLReKeyLen                                                             /**< [In] Length of bytes available in \b pPLReKey buffer. The length
                                                                                         *        should be equal to double AES key size.
                                                                                         */
    );

/**
 * end of group phhalHw_Sam_Cmd_PL
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_PL_H */
