/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) MIFARE DESFire command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */


#ifndef PHHALHW_SAM_CMD_DESFIRE_H
#define PHHALHW_SAM_CMD_DESFIRE_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

#define PHHAL_HW_SAM_DESFIRE_SAM_RESPONSE_LEN                                   0x0CU   /* SAM Response length with complete Capabilities */
#define PHHAL_HW_SAM_DESFIRE_PICC_RESPONSE_LEN                                  0x20U   /* MIFARE DESFire PICC response Length */
#define PHHAL_HW_SAM_DESFIRE_PICC_SUCCESS_STATUS_CODE                           0x00U   /* MIFARE DESFire PICC Success status */
#define PHHAL_HW_SAM_DESFIRE_PICC_CAPABALITIES_LEN                              0x06U   /* MIFARE DESFire PICC Capabilities length */
#define PHHAL_HW_SAM_DESFIRE_PICC_CAPABALITIES_USAGE                            0xFFU   /* Default PCD Capabilities usage */
#define PHHAL_HW_SAM_DESFIRE_PICC_NATIVE_STATUS_LEN                             0x01U   /* MIFARE DESFire PICC native status Length */
#define PHHAL_HW_SAM_DESFIRE_PICC_WRAPPED_STATUS_LEN                            0x02U   /* MIFARE DESFire PICC wrapped status Length */

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire 07_MIFARE_DESFire
 * \brief Implementation of SAM MIFARE DESFire commands.
 * Has implementation for both X and S mode communication.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines Defines
 * \brief Definitions common for MIFARE DESFire S and X mode communication
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_Authentication Authenticate
 * \brief Definitions common for MIFARE DESFire S and X mode communication for
 * Authentication interfaces.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part1 "SAM_AuthenticatePICC Part1"
 *  - \ref phhalHw_Sam_Cmd_SAM_IsoAuthenticatePICC_Part1 "SAM_IsoAuthenticatePICC Part1"
 *  - \ref phhalHw_Sam_Cmd_DESFire_AuthenticatePICC "DESFire_AuthenticatePICC"
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_Authentication_AuthType AuthType
 * \brief Definitions for Authentication mode selection.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part1 "SAM_AuthenticatePICC Part1"
 *  - \ref phhalHw_Sam_Cmd_DESFire_AuthenticatePICC "DESFire_AuthenticatePICC"
 * @{
 */
#define PHHAL_HW_CMD_SAM_AUTH_TYPE_D40_EV1                                      0x00U   /**< SAM DESFire Authentication mode as D40 and EV1. */
#define PHHAL_HW_CMD_SAM_AUTH_TYPE_EV2_FIRST_AUTH                               0x80U   /**< SAM DESFire Authentication mode as EV2 First. */
#define PHHAL_HW_CMD_SAM_AUTH_TYPE_EV2_NON_FIRST_AUTH                           0xC0U   /**< SAM DESFire Authentication mode as EV2 Non First. */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_Authentication_AuthType
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_Authentication_SM SecureMessaging
 * \brief Definitions for Secure Messaging.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part1 "SAM_AuthenticatePICC Part1"
 *  - \ref phhalHw_Sam_Cmd_DESFire_AuthenticatePICC "DESFire_AuthenticatePICC"
 * @{
 */
#define PHHAL_HW_CMD_SAM_ALLOW_SECURE_MESSAGING                                 0x00U   /**< SAM DESFire Allow Secure Messaging. */
#define PHHAL_HW_CMD_SAM_SUPPRESS_SECURE_MESSAGING                              0x20U   /**< SAM DESFire Suppress Secure Messaging. To be used if
                                                                                         *       originality keys are used.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_Authentication_SM
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_Authentication_KeyDiv KeyDiversification
 * \brief Definitions for Key diversification mode selection.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part1 "SAM_AuthenticatePICC Part1"
 *  - \ref phhalHw_Sam_Cmd_SAM_IsoAuthenticatePICC_Part1 "SAM_IsoAuthenticatePICC Part1"
 *  - \ref phhalHw_Sam_Cmd_DESFire_AuthenticatePICC "DESFire_AuthenticatePICC"
 * @{
 */
#define PHHAL_HW_CMD_SAM_KDF_AV1                                                0x00U   /**< SAM DESFire key derivation type as AV1. */
#define PHHAL_HW_CMD_SAM_KDF_AV1_DOUBLE_ENCRYPTION                              0x00U   /**< SAM DESFire key derivation type as AV1 double
                                                                                         *   encryption round.
                                                                                         */
#define PHHAL_HW_CMD_SAM_KDF_AV1_SINGLE_ENCRYPTION                              0x08U   /**< SAM DESFire key derivation type as AV1 single
                                                                                         *   encryption round.
                                                                                         */
#define PHHAL_HW_CMD_SAM_KDF_AV2                                                0x10U   /**< SAM DESFire key derivation type as AV2. */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_Authentication_KeyDiv
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_Authentication_KeySel KeySelection
 * \brief Definitions for Key selection.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part1 "SAM_AuthenticatePICC Part1"
 *  - \ref phhalHw_Sam_Cmd_SAM_IsoAuthenticatePICC_Part1 "SAM_IsoAuthenticatePICC Part1"
 *  - \ref phhalHw_Sam_Cmd_DESFire_AuthenticatePICC "DESFire_AuthenticatePICC"
 * @{
 */
#define PHHAL_HW_CMD_SAM_KEY_SELECTION_KEY_ENTRY_NUMBER                         0x00U   /**< SAM DESFire key selection by key entry number. */
#define PHHAL_HW_CMD_SAM_KEY_SELECTION_DESFIRE_KEY_NUMBER                       0x02U   /**< SAM DESFire key selection by DESFIRE key number. */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_Authentication_KeySel
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_Authentication_Div Diversification
 * \brief Definitions for diversification usage.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part1 "SAM_AuthenticatePICC Part1"
 *  - \ref phhalHw_Sam_Cmd_SAM_IsoAuthenticatePICC_Part1 "SAM_IsoAuthenticatePICC Part1"
 *  - \ref phhalHw_Sam_Cmd_DESFire_AuthenticatePICC "DESFire_AuthenticatePICC"
 * @{
 */
#define PHHAL_HW_CMD_SAM_KEY_DIVERSIFICATION_OFF                                0x00U   /**< SAM DESFire key diversification disabled. */
#define PHHAL_HW_CMD_SAM_KEY_DIVERSIFICATION_ON                                 0x01U   /**< SAM DESFire key diversification enabled. */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_Authentication_Div
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_Authentication
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey ChangeKey
 * \brief Definitions common for MIFARE DESFire S and X mode communication for
 * ChangeKey interface.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_ChangeKeyPICC "SAM_ChangeKeyPICC"
 *  - \ref phhalHw_Sam_Cmd_DESFire_ChangeKeyPICC "DESFire_ChangeKeyPICC"
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey_KeyCompilation KeyCompilation
 * \brief Definitions common for MIFARE DESFire S and X mode communication for
 * ChangeKey interface. To be used for updating P1 information byte.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_ChangeKeyPICC "SAM_ChangeKeyPICC"
 *  - \ref phhalHw_Sam_Cmd_DESFire_ChangeKeyPICC "DESFire_ChangeKeyPICC"
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey_KeyCompilation_DivMethod DiversificationMethod
 * \brief Definitions for Key diversification method.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_ChangeKeyPICC "SAM_ChangeKeyPICC"
 *  - \ref phhalHw_Sam_Cmd_DESFire_ChangeKeyPICC "DESFire_ChangeKeyPICC"
 * @{
 */
#define PHHAL_HW_CMD_SAM_DIV_METHOD_AV1                                         0x00U   /**< SAM DESFire diversification method as AV1. */
#define PHHAL_HW_CMD_SAM_DIV_METHOD_AV2                                         0x20U   /**< SAM DESFire diversification method as AV2. */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey_KeyCompilation_DivMethod
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey_KeyCompilation_CurrKey CurrentKey
 * \brief Definitions for Key diversification method for Current Key.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_ChangeKeyPICC "SAM_ChangeKeyPICC"
 *  - \ref phhalHw_Sam_Cmd_DESFire_ChangeKeyPICC "DESFire_ChangeKeyPICC"
 * @{
 */
#define PHHAL_HW_CMD_SAM_KDF_CURRENT_KEY_DOUBLE_ENCRYPTION                      0x00U   /**< SAM DESFire diversification method of current
                                                                                         *   key for AV1 as double encryption.
                                                                                         */
#define PHHAL_HW_CMD_SAM_KDF_CURRENT_KEY_SINGLE_ENCRYPTION                      0x10U   /**< SAM DESFire diversification method of current
                                                                                         *   key for AV1 as single encryption.
                                                                                         */
#define PHHAL_HW_CMD_SAM_KDF_CURRENT_KEY_OFF                                    0x00U   /**< SAM DESFire diversification usage for current
                                                                                         *   key is disabled.
                                                                                         */
#define PHHAL_HW_CMD_SAM_KDF_CURRENT_KEY_ON                                     0x04U   /**< SAM DESFire diversification usage for current
                                                                                         *   key is enabled.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey_KeyCompilation_CurrKey
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey_KeyCompilation_NewKey NewKey
 * \brief Definitions for Key diversification method for New Key.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_ChangeKeyPICC "SAM_ChangeKeyPICC"
 *  - \ref phhalHw_Sam_Cmd_DESFire_ChangeKeyPICC "DESFire_ChangeKeyPICC"
 * @{
 */
#define PHHAL_HW_CMD_SAM_KDF_NEW_KEY_DOUBLE_ENCRYPTION                          0x00U   /**< SAM DESFire diversification method of new key
                                                                                         *   for AV1 as double encryption.
                                                                                         */
#define PHHAL_HW_CMD_SAM_KDF_NEW_KEY_SINGLE_ENCRYPTION                          0x08U   /**< SAM DESFire diversification method of new key
                                                                                         *   for AV1 as single encryption.
                                                                                         */
#define PHHAL_HW_CMD_SAM_KDF_NEW_KEY_OFF                                        0x00U   /**< SAM DESFire diversification usage for new key
                                                                                         *   is disabled.
                                                                                         */
#define PHHAL_HW_CMD_SAM_KDF_NEW_KEY_ON                                         0x02U   /**< SAM DESFire diversification usage for new key
                                                                                         *   is enabled.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey_KeyCompilation_NewKey
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey_KeyCompilation_Crypto CryptoMode
 * \brief Definitions for Cryptogram computation mode.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_ChangeKeyPICC "SAM_ChangeKeyPICC"
 *  - \ref phhalHw_Sam_Cmd_DESFire_ChangeKeyPICC "DESFire_ChangeKeyPICC"
 * @{
 */
#define PHHAL_HW_CMD_SAM_CRYPTO_MODE_DIFFERENT_KEY                              0x00U   /**< SAM DESFire crypto computation mode are different
                                                                                         *   for target and auth keys.
                                                                                         */
#define PHHAL_HW_CMD_SAM_CRYPTO_MODE_SAME_KEY                                   0x01U   /**< SAM DESFire crypto computation mode are same for
                                                                                         *   target and auth keys. The parameters CurrKeyNo and
                                                                                         *   CurrKeyV are ignored
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey_KeyCompilation_Crypto
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey_KeyCompilation
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey_Config Configuration
 * \brief Definitions common for MIFARE DESFire S and X mode communication for
 * ChangeKey interface. To be used for updating P2 information byte.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_ChangeKeyPICC "SAM_ChangeKeyPICC"
 *  - \ref phhalHw_Sam_Cmd_DESFire_ChangeKeyPICC "DESFire_ChangeKeyPICC"
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey_Config_ChangeKey ChangeKey
 * \brief Definitions for PICC Change Key command.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_ChangeKeyPICC "SAM_ChangeKeyPICC"
 *  - \ref phhalHw_Sam_Cmd_DESFire_ChangeKeyPICC "DESFire_ChangeKeyPICC"
 * @{
 */
#define PHHAL_HW_CMD_SAM_CMD_TYPE_CHANGE_KEY                                    0x00U   /**< SAM DESFire Change Key command type as ChangeKey. */
#define PHHAL_HW_CMD_SAM_CMD_TYPE_CHANGE_KEY_EV2                                0x20U   /**< SAM DESFire Change Key command type as ChangeKeyEV2. */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey_Config_ChangeKey
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey_Config_MasterKey MasterKey
 * \brief Definitions for PICC master key update
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_ChangeKeyPICC "SAM_ChangeKeyPICC"
 *  - \ref phhalHw_Sam_Cmd_DESFire_ChangeKeyPICC "DESFire_ChangeKeyPICC"
 * @{
 */
#define PHHAL_HW_CMD_SAM_MASTER_KEY_UPDATE_EXCLUDE_KEYTYPE                      0x00U   /**< SAM DESFire PICC Master key update to exclude key
                                                                                         *   type in cryptogram.
                                                                                         */
#define PHHAL_HW_CMD_SAM_MASTER_KEY_UPDATE_INCLUDE_KEYTYPE                      0x10U   /**< SAM DESFire PICC Master key update to include key
                                                                                         *   type in cryptogram.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey_Config_MasterKey
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey_Config
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_ChangeKey
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_TMFile TransactionMACFile
 * \brief Definitions for File Options and TMKEy options.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_CreateTMFilePICC "SAM_CreateTMFilePICC"
 *  - \ref phhalHw_Sam_Cmd_DESFire_CreateTMFilePICC "DESFire_CreateTMFilePICC"
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_TMFile_Div Diversification
 * \brief Definitions for diversification usage.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_CreateTMFilePICC "SAM_CreateTMFilePICC"
 *  - \ref phhalHw_Sam_Cmd_DESFire_CreateTMFilePICC "DESFire_CreateTMFilePICC"
 * @{
 */
#define PHHAL_HW_CMD_SAM_CREATE_TM_FILE_DIVERSIFICATION_OFF                     0x00U   /**< SAM DESFire key diversification disabled. */
#define PHHAL_HW_CMD_SAM_CREATE_TM_FILE_DIVERSIFICATION_ON                      0x01U   /**< SAM DESFire key diversification enabled. */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_TMFile_Div
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_TMFile_FileOption FileOption
 * \brief Definitions for File Options.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_CreateTMFilePICC "SAM_CreateTMFilePICC"
 *  - \ref phhalHw_Sam_Cmd_DESFire_CreateTMFilePICC "DESFire_CreateTMFilePICC"
 * @{
 */
#define PHHAL_HW_CMD_SAM_CREATE_TM_FILE_COMM_MODE_PLAIN                         0x00U   /**< SAM communication mode as Plain. */
#define PHHAL_HW_CMD_SAM_CREATE_TM_FILE_COMM_MODE_MAC                           0x01U   /**< SAM communication mode as MAC. */
#define PHHAL_HW_CMD_SAM_CREATE_TM_FILE_COMM_MODE_FULL                          0x03U   /**< SAM communication mode as Full. */
#define PHHAL_HW_CMD_SAM_CREATE_TM_FILE_TMI_EXCLUSION_FILEMAP                   0x40U   /**< SAM File Option as TMI Exclusion File Map. */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_TMFile_FileOption
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_Defines_TMFile_TMKey TMKeyOption
 * \brief Definitions for TMKey options.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_CreateTMFilePICC "SAM_CreateTMFilePICC"
 *  - \ref phhalHw_Sam_Cmd_DESFire_CreateTMFilePICC "DESFire_CreateTMFilePICC"
 * @{
 */
#define PHHAL_HW_CMD_SAM_CREATE_TM_FILE_KEYTYPE_AES128                          0x02U   /**< SAM TMKeyOption KeyType as AES128. */
#define PHHAL_HW_CMD_SAM_CREATE_TM_FILE_KEYTYPE_AES256                          0x03U   /**< SAM TMKeyOption KeyType as AES256. */

#define PHHAL_HW_CMD_SAM_CREATE_TM_FILE_MODE_TMAC                               0x00U   /**< SAM TMKeyOption mode as Transaction MAC. */
#define PHHAL_HW_CMD_SAM_CREATE_TM_FILE_MODE_TSIG                               0x80U   /**< SAM TMKeyOption mode as Transaction Signature. */

/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_TMFile_TMKey
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines_TMFile
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_DESFire_Defines
 * @}
 */






/** \defgroup phhalHw_Sam_Cmd_DESFire_SMode S_Mode
 * \brief Implementation of SAM commands used for MIFARE DESFire communication in S-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_SMode_Defines Defines
 * \brief Definitions for SAM commands used for MIFARE DESFire communication in S-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_SMode_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_PICC                              0x0AU   /**< SAM Instruction code for SAM_AuthenticatePICC command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_ISO_AUTHENTICATE_PICC                          0x8EU   /**< SAM Instruction code for SAM_IsoAuthenticatePICC command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_CHANGE_KEY_PICC                                0xC4U   /**< SAM Instruction code for SAM_ChangeKeyPICC command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_CREATE_TM_FILE_PICC                            0xC6U   /**< SAM Instruction code for SAM_CreateTMFilePICC command. */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_SMode_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_SMode_Defines_AuthMode AuthMode
 * \brief Definitions for PICC authentication mode.
 * To be used with \ref phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part1 "SAM_AuthenticatePICC Part1"
 * @{
 */
#define PHHAL_HW_CMD_SAM_AUTH_MODE_EV2                                          0x00U   /**< SAM DESFire Auth mode as EV2. */
#define PHHAL_HW_CMD_SAM_AUTH_MODE_LRP                                          0x01U   /**< SAM DESFire Auth mode as LRP. */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_SMode_Defines_AuthMode
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_DESFire_SMode_Defines
 * @}
 */

/**
 * \brief Performs first part of encryption and decryption of data received from PICC and to be sent to PICC.
 * Here the Encrypted RndB data will be sent to SAM. SAM will Decrypt the data and Encrypt
 * RndA with RndB'. This encrypted RndA and RndB' will be returned to the caller for further transmission to
 * the PICC.
 *
 * \note
 *  - Post calling this interface it's must to call \ref phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part2
 *    "AuthenticatePICC Part2" to place SAM into proper completion state if \ref PH_ERR_SUCCESS_CHAINING
 *    "Chaining Status" is returned from Library. Otherwise error will be returned by SAM for any
 *    subsequent commands.
 *  - If \ref PH_ERR_SUCCESS_CHAINING "Chaining Status" is not returned from Library, no need to call
 *    \ref phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part2 "AuthenticatePICC Part2" interface.
 *
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pDivInput, \b pCardResponse and \b pSamRespLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part1(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Option to update the P1 information. The options can
                                                                                         *        be combined by bitwise OR.
                                                                                         *          - Option for Authentication mode and Authentication type
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_AUTH_TYPE_D40_EV1
                                                                                         *                     "D40 & EV1 authentication type"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_AUTH_TYPE_EV2_FIRST_AUTH
                                                                                         *                     "EVx Authenticate first"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_AUTH_TYPE_EV2_NON_FIRST_AUTH
                                                                                         *                     "EVx Authenticate non first"
                                                                                         *
                                                                                         *          - Option for Suppressing secure messaging
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_ALLOW_SECURE_MESSAGING
                                                                                         *                     "Allow secure messaging"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_SUPPRESS_SECURE_MESSAGING
                                                                                         *                     "Suppress secure messaging" (for example for
                                                                                         *                     originality keys)
                                                                                         *
                                                                                         *          - Option for Key Derivation Information and Key Diversification
                                                                                         *            Types.
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_KDF_AV1
                                                                                         *                     "AV1 compatibility mode" key diversification methods,
                                                                                         *                     3TDEA, AES key
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_KDF_AV1_DOUBLE_ENCRYPTION
                                                                                         *                     "AV1 compatibility mode" key diversification methods,
                                                                                         *                     TDEA Key, diversified using two encryption rounds
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_KDF_AV1_SINGLE_ENCRYPTION
                                                                                         *                     "AV1 compatibility mode" key diversification methods,
                                                                                         *                     TDEA Key, diversified using one encryption round
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_KDF_AV2
                                                                                         *                     "AV2 mode key diversification methods"
                                                                                         *
                                                                                         *          - Option for Key Selection
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_KEY_SELECTION_KEY_ENTRY_NUMBER
                                                                                         *                     "Key selection by key entry number"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_KEY_SELECTION_DESFIRE_KEY_NUMBER
                                                                                         *                     "Key selection by DESFire key number"
                                                                                         *
                                                                                         *          - Option for Key Diversification
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_KEY_DIVERSIFICATION_OFF
                                                                                         *                     "No diversification"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_KEY_DIVERSIFICATION_ON
                                                                                         *                     "Diversify the used key with the given DivInput"
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used in SAM or DESFire Key number.
                                                                                         *          - If Key selection by \ref PHHAL_HW_CMD_SAM_KEY_SELECTION_DESFIRE_KEY_NUMBER
                                                                                         *            "DESFire key number", 0x00 - 0x0D should be used.
                                                                                         *          - If Key selection by \ref PHHAL_HW_CMD_SAM_KEY_SELECTION_KEY_ENTRY_NUMBER
                                                                                         *            "SAM Key Number", then below one should be used
                                                                                         *              - NVM Key: 0x00 - 0x7F
                                                                                         *              - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used in SAM. */
        uint8_t bAuthMode,                                                              /**< [In] The type of key to be used for EV2 authentication.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_AUTH_MODE_EV2
                                                                                         *                 "EV2 authentication"
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_AUTH_MODE_LRP
                                                                                         *                 "LRP authentication"
                                                                                         */
        uint8_t * pDivInput,                                                            /**< [In] Diversification Input used to diversify the key.
                                                                                         *          - If \ref PHHAL_HW_CMD_SAM_KEY_DIVERSIFICATION_ON
                                                                                         *             "Diversification is Enabled", then
                                                                                         *              - 8 bytes (if AV1 key diversification with DES)
                                                                                         *              - 16 bytes (if AV1 key diversification with AES)
                                                                                         *              - 1 to 31 bytes (if AV2 key diversification)
                                                                                         *                diversification input
                                                                                         *          - NULL otherwise
                                                                                         */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t * pCardResponse,                                                        /**< [In] Encrypted Challenge (RndB) data returned by the PICC.
                                                                                         *          - If targeting D40 & EV1 authentication with 2TDEA
                                                                                         *            Key: 8 bytes as E(Kx,RndB)
                                                                                         *          - If targeting D40 & EV1 authentication with 3TDEA, AES
                                                                                         *            Key or if targeting EV2 authentication: 16 bytes as
                                                                                         *            E(Kx,RndB)
                                                                                         *          - If targeting LRP authentication: 16 bytes as RndB
                                                                                         */
        uint8_t bCardRespLen,                                                           /**< [In] Length of bytes available in \b pCardResponse buffer. */
        uint8_t ** ppSamResponse,                                                       /**< [Out] Encrypted PCD Challenge (RndA and RndB') returned by
                                                                                         *         the SAM. One of the following will be available,
                                                                                         *          - If targeting D40 & EV1 authentication with 2TDEA Key:
                                                                                         *            16 bytes as E(Kx,RndA || RndB')
                                                                                         *          - If targeting D40 & EV1 authentication with 3TDEA, AES
                                                                                         *            Key or if targeting EV2 authentication:
                                                                                         *            32 bytes as E(Kx,RndA || RndB')
                                                                                         *          - If targeting LRP authentication: 32 bytes as RndA ||
                                                                                         *            MAC(Km, RndA||RndB)
                                                                                         */
        uint16_t * pSamRespLen                                                          /**< [Out] Length of bytes available in \b ppSamResponse buffer. */
    );

/**
 * \brief Performs second part of decryption of data received from PICC. Here the Encrypted RndA'
 * data will be sent to SAM. SAM will Decrypt the data and extract PCD and PD Capabilities for EV2
 * First Auth and null in case of rest of Authentication modes. This PCD and PD information will be
 * returned to the caller. Also the status code of PICC will be returned to the caller in case of
 * error.
 *
 * \note This interface should be called only if \ref phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part1
 * "AuthenticatePICC Part1" returns \ref PH_ERR_SUCCESS_CHAINING "Chaining Status".
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS               Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pCardResponse, \b pPDcap2 \b pPCDcap2 and
 *                                      \b pPiccRetCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticatePICC_Part2(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bPiccErrorCode,                                                         /**< [In] Status code returned by the PICC in case of failure else
                                                                                         *        00h for success.
                                                                                         */
        uint8_t * pCardResponse,                                                        /**< [In] Encrypted RndA' returned by the SAM. One of the following,
                                                                                         *          - If targeting EV2 first authentication: 32 bytes as
                                                                                         *            E(Kx,TI||RndA'||PDcap2||PCDcap2)
                                                                                         *          - If targeting LRP first authentication: 32 bytes as
                                                                                         *            E(Ke, TI||PDCap2||PCDCap2) || MAC(Km, RndB || RndA
                                                                                         *            E(Ke, TI||PDCap2||PCDCap2))
                                                                                         *          - If targeting LRP non-first authentication: 16 bytes
                                                                                         *            as MAC(Km, RndB || RndA)
                                                                                         *          - If targeting D40 & EV1 authentication with 2TDEA
                                                                                         *            Key: 8 or 16 bytes as E(Kx,RndA')
                                                                                         */
        uint8_t bCardRespLen,                                                           /**< [In] Length of bytes available in \b pCardResponse buffer. */
        uint8_t * pPDcap2,                                                              /**< [Out] Buffer containing the output PD capabilities. This will
                                                                                         *         contain 6 bytes of PD information, if targeting EVx
                                                                                         *         authentication type and first authentication.
                                                                                         */
        uint8_t * pPCDcap2,                                                             /**< [Out] Buffer containing the output PCD capabilities. This will
                                                                                         *         contain 6 bytes of PCD information, if targeting EVx
                                                                                         *         authentication type and first authentication.
                                                                                         */
        uint8_t * pStatusCode                                                           /**< [Out] Status code from MIFARE DESFire PICC if available else
                                                                                         *         zero.
                                                                                         */
    );

/**
 * \brief Performs first part of encryption and decryption of data received from PICC and to be sent to PICC.
 * Here the Encrypted RndB data will be sent to SAM. SAM will Decrypt the data and Encrypt RndA with RndB'.
 * This encrypted RndA and RndB' will be returned to the caller for further transmission to the PICC.
 *
 * \note
 *  - Post calling this interface it's must to call \ref phhalHw_Sam_Cmd_SAM_IsoAuthenticatePICC_Part2
 *    "IsoAuthenticatePICC Part2" to place SAM into proper completion state if \ref PH_ERR_SUCCESS_CHAINING
 *    "Chaining Status" is returned from Library. Otherwise error will be returned by SAM for any
 *    subsequent commands.
 *  - If \ref PH_ERR_SUCCESS_CHAINING "Chaining Status" is not returned from Library, no need to call
 *    \ref phhalHw_Sam_Cmd_SAM_IsoAuthenticatePICC_Part2 "IsoAuthenticatePICC Part2" interface.
 *
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pDivInput, \b pCardResponse and \b pSamRespLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_IsoAuthenticatePICC_Part1(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Option to update the P1 information. The options can
                                                                                         *        be combined by bitwise OR.
                                                                                         *          - Option for Key Derivation Information and Key Diversification
                                                                                         *            Types.
                                                                                         *              - \ref  PHHAL_HW_CMD_SAM_KDF_AV1
                                                                                         *                     "AV1 compatibility mode" key diversification methods,
                                                                                         *                     3TDEA, AES key
                                                                                         *              - \ref  PHHAL_HW_CMD_SAM_KDF_AV1_DOUBLE_ENCRYPTION
                                                                                         *                     "AV1 compatibility mode" key diversification methods,
                                                                                         *                     TDEA Key, diversified using two encryption rounds
                                                                                         *              - \ref  PHHAL_HW_CMD_SAM_KDF_AV1_SINGLE_ENCRYPTION
                                                                                         *                     "AV1 compatibility mode" key diversification methods,
                                                                                         *                     TDEA Key, diversified using one encryption round
                                                                                         *              - \ref  PHHAL_HW_CMD_SAM_KDF_AV2
                                                                                         *                     "AV2 mode key diversification methods"
                                                                                         *
                                                                                         *          - Option for Key Selection
                                                                                         *              - \ref  PHHAL_HW_CMD_SAM_KEY_SELECTION_KEY_ENTRY_NUMBER
                                                                                         *                     "Key selection by key entry number"
                                                                                         *              - \ref  PHHAL_HW_CMD_SAM_KEY_SELECTION_DESFIRE_KEY_NUMBER
                                                                                         *                     "Key selection by DESFire key number"
                                                                                         *
                                                                                         *          - Option for Key Diversification
                                                                                         *              - \ref  PHHAL_HW_CMD_SAM_KEY_DIVERSIFICATION_OFF
                                                                                         *                     "No diversification"
                                                                                         *              - \ref  PHHAL_HW_CMD_SAM_KEY_DIVERSIFICATION_ON
                                                                                         *                     "Diversify the used key with the given DivInput"
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used in SAM or DESFire Key number.
                                                                                         *          - If Key selection by \ref PHHAL_HW_CMD_SAM_KEY_SELECTION_DESFIRE_KEY_NUMBER
                                                                                         *            "DESFire key number", 0x00 - 0x0D should be used.
                                                                                         *          - If Key selection by \ref PHHAL_HW_CMD_SAM_KEY_SELECTION_KEY_ENTRY_NUMBER
                                                                                         *            "SAM Key Number", then below one should be used
                                                                                         *              - NVM Key: 0x00 - 0x7F
                                                                                         *              - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used in SAM. */
        uint8_t * pDivInput,                                                            /**< [In] Diversification Input used to diversify the key.
                                                                                         *          - If \ref PHHAL_HW_CMD_SAM_KEY_DIVERSIFICATION_ON
                                                                                         *             "Diversification is Enabled", then
                                                                                         *              - 8 bytes (if AV1 key diversification with DES)
                                                                                         *              - 16 bytes (if AV1 key diversification with AES)
                                                                                         *              - 1 to 31 bytes (if AV2 key diversification)
                                                                                         *                diversification input
                                                                                         *          - NULL otherwise
                                                                                         */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t * pCardResponse,                                                        /**< [In] PICC challenge in clear as received from
                                                                                         *        PICC's Cmd.ISOGetChallenge.
                                                                                         *          - If targeting TDEA Key: 8 bytes encrypted Challenge
                                                                                         *          - If targeting 3TDEA or AES Key: 16 bytes encrypted
                                                                                         *            Challenge
                                                                                         */
        uint8_t bCardRespLen,                                                           /**< [In] Length of bytes available in \b pCardResponse buffer. */
        uint8_t ** ppSamResponse,                                                       /**< [Out] Encrypted PCD Challenge (RndA and RndB') and response
                                                                                         *         returned by the SAM. One of the following will be
                                                                                         *         available.
                                                                                         *          - If targeting TDEA Key: 24 bytes as E(Kx, RPCD1 ||
                                                                                         *            RPICC1 ) || RPCD2
                                                                                         *          - If targeting 3TDEA or AES Key: 48 bytes as E(Kx,
                                                                                         *            RPCD1 || RPICC1 ) || RPCD2
                                                                                         */
        uint16_t * pSamRespLen                                                          /**< [Out] Length of bytes available in \b ppSamResponse buffer. */
    );

/**
 * Performs second part of decryption of data received from PICC. Here the Encrypted RndA' data
 * will be sent to SAM. SAM will Decrypt the data and extract PCD and PD Capabilities for EV2 First
 * Auth and null in case of rest of Authentication modes. This PCD and PD information will be returned
 * to the caller. Also the status code of PICC will be returned to the caller in case of error.
 *
 * \note This interface should be called only if \ref phhalHw_Sam_Cmd_SAM_IsoAuthenticatePICC_Part1
 * "IsoAuthenticatePICC Part1" returns \ref PH_ERR_SUCCESS_CHAINING "Chaining Status".
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS               Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pCardResponse is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_IsoAuthenticatePICC_Part2(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pCardResponse,                                                        /**< [In] PICC encrypted response as received from
                                                                                         *        Cmd.ISOInternalAuthenticate. One of the following,
                                                                                         *          - If targeting TDEA Key: 16 bytes as E(Kx, RPICC2
                                                                                         *            || RRPCD2 )
                                                                                         *          - If targeting 3TDEA or AES Key: 32 bytes as
                                                                                         *            E(Kx, RPICC2 || RRPCD2 )
                                                                                         */
        uint8_t bCardRespLen                                                            /**< [In] Length of bytes available in \b pCardResponse buffer. */
    );

/**
 * \brief Performs key change for the specified current key to a new key. The crypto operation
 * of the key to be changed will be calculated by SAM hardware. This crypto data will then sent
 * to card to perform Change Key operations.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS               Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pDivInput and \b pSamRespLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_ChangeKeyPICC(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bCryptoMethod,                                                          /**< [In] Key compilation method. Options for P1 information byte.
                                                                                         *        The options can be combined by bitwise OR.
                                                                                         *          - Key diversification method
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_DIV_METHOD_AV1
                                                                                         *                "SAM AV1 compatibility diversification method"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_DIV_METHOD_AV2
                                                                                         *                "SAM AV2 diversification method"
                                                                                         *
                                                                                         *          - SAM AV1 and SAM AV2 Key diversification method
                                                                                         *              - Current Key
                                                                                         *                  - \ref PHHAL_HW_CMD_SAM_KDF_CURRENT_KEY_DOUBLE_ENCRYPTION
                                                                                         *                    "Diversify using two encryption rounds"
                                                                                         *                  - \ref PHHAL_HW_CMD_SAM_KDF_CURRENT_KEY_SINGLE_ENCRYPTION
                                                                                         *                    "Diversify using one encryption round"
                                                                                         *                  - \ref PHHAL_HW_CMD_SAM_KDF_CURRENT_KEY_OFF
                                                                                         *                    "Do not diversified the current key"
                                                                                         *                  - \ref PHHAL_HW_CMD_SAM_KDF_CURRENT_KEY_ON
                                                                                         *                    "Diversified the current key"
                                                                                         *              - New Key
                                                                                         *                  - \ref PHHAL_HW_CMD_SAM_KDF_NEW_KEY_DOUBLE_ENCRYPTION
                                                                                         *                    "Diversify using two encryption rounds"
                                                                                         *                  - \ref PHHAL_HW_CMD_SAM_KDF_NEW_KEY_SINGLE_ENCRYPTION
                                                                                         *                    "Diversify using one encryption round"
                                                                                         *                  - \ref PHHAL_HW_CMD_SAM_KDF_NEW_KEY_OFF
                                                                                         *                    "Do not diversified the new key"
                                                                                         *                  - \ref PHHAL_HW_CMD_SAM_KDF_NEW_KEY_ON
                                                                                         *                    "Diversified the new key"
                                                                                         *
                                                                                         *          - Cryptogram computation mode
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CRYPTO_MODE_DIFFERENT_KEY
                                                                                         *                "PICC targeted key different" from the PICC authenticated
                                                                                         *                 key
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CRYPTO_MODE_SAME_KEY
                                                                                         *                "PICC targeted key equal" to PICC authenticated key.
                                                                                         *                The parameters CurrKeyNo and CurrKeyV are ignored"
                                                                                         */
        uint8_t bConfig,                                                                /**< [In] Options for P2 information byte. The options can be
                                                                                         *        combined by bitwise OR.
                                                                                         *          - PICC Change Key command
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CMD_TYPE_CHANGE_KEY
                                                                                         *                "Cmd.ChangeKey"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CMD_TYPE_CHANGE_KEY_EV2
                                                                                         *                "Cmd.ChangeKeyEV2"
                                                                                         *
                                                                                         *          - PICC master key update
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_MASTER_KEY_UPDATE_EXCLUDE_KEYTYPE
                                                                                         *                "Do not include the key type in the cryptogram"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_MASTER_KEY_UPDATE_INCLUDE_KEYTYPE
                                                                                         *                "Include the key type in the cryptogram"
                                                                                         *
                                                                                         *          - Number of PICC key to be changed. Will be taken from \b bDFKeyNo
                                                                                         *            parameter if \b bConfig has \ref PHHAL_HW_CMD_SAM_CMD_TYPE_CHANGE_KEY
                                                                                         *            "Cmd.ChangeKey"
                                                                                         */
        uint8_t bKeySetNo,                                                              /**< [In] If \b bConfig = \ref PHHAL_HW_CMD_SAM_CMD_TYPE_CHANGE_KEY_EV2
                                                                                         *        "Cmd.ChangeKeyEV2", Key set number to which the key to be
                                                                                         *        changed belongs to.
                                                                                         */
        uint8_t bDFKeyNo,                                                               /**< [In] Block number of the key available in the PICC. This will be used while
                                                                                         *        exchanging the command with PICC.
                                                                                         *          - The lower nibble will be used for P2 information byte if \b bConfig
                                                                                         *            has \ref PHHAL_HW_CMD_SAM_CMD_TYPE_CHANGE_KEY "Cmd.ChangeKey".
                                                                                         *          - Complete byte will be used if \b bConfig has
                                                                                         *            \ref PHHAL_HW_CMD_SAM_CMD_TYPE_CHANGE_KEY_EV2 "Cmd.ChangeKeyEV2".
                                                                                         */
        uint8_t bCurrKeyNo,                                                             /**< [In] Number of key entry holding the current key in SAM.
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bCurrKeyVer,                                                            /**< [In] Key version of the current key in SAM. */
        uint8_t bNewKeyNo,                                                              /**< [In] Number of key entry holding the new key in SAM.
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bNewKeyVer,                                                             /**< [In] Key version of new key in SAM. */
        uint8_t * pDivInput,                                                            /**< [In] Diversification Input used to diversify the key.
                                                                                         *          - If any of diversification option is set in
                                                                                         *            \b bCryptoMethod parameter, then
                                                                                         *              - 8 bytes (if AV1 key diversification with DES)
                                                                                         *              - 16 bytes (if AV1 key diversification with AES)
                                                                                         *              - 1 to 31 bytes (if AV2 key diversification)
                                                                                         *                diversification input
                                                                                         *          - NULL otherwise
                                                                                         */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t ** ppSamResponse,                                                       /**< [Out] Cryptogram holding key data. */
        uint16_t * pSamRespLen                                                          /**< [Out] Length of bytes available in \b ppSamResponse buffer. */
    );

/**
 * \brief The SAM_CreateTMFilePICC command supports the procedure to create a Transaction MAC
 * File in the PICC in S-mode from a key stored in the SAM.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS               Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pAccessRights, \b pDivInput and \b pSamRespLen
 *                                      are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_CreateTMFilePICC(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Key diversification selection. Options for P1
                                                                                         *        information byte.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_DIVERSIFICATION_OFF
                                                                                         *            "No diversification"
                                                                                         *          - \ref #PHHAL_HW_CMD_SAM_CREATE_TM_FILE_DIVERSIFICATION_ON
                                                                                         *            "Diversify the used key with the given DivInput"
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used in SAM. One of the following
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used in SAM. */
        uint8_t bFileNo,                                                                /**< [In] File number of the file to be created. */
        uint8_t bFileOption,                                                            /**< [In] Options for the targeted file.
                                                                                         *          - Communication Mode
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_COMM_MODE_PLAIN
                                                                                         *                "FileOptions as PLAIN"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_COMM_MODE_MAC
                                                                                         *                "FileOptions as MAC"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_COMM_MODE_FULL
                                                                                         *                "FileOptions as FULL"
                                                                                         *
                                                                                         *        - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_TMI_EXCLUSION_FILEMAP
                                                                                         *          "TMI Exclusion file Map". Can bit bitwise OR with above option
                                                                                         */
        uint8_t * pAccessRights,                                                        /**< [In] Access conditions to be applied for the file. Refer
                                                                                         *        respective product DataSheet for access rights
                                                                                         *        information. This should be two bytes long.
                                                                                         */
        uint8_t * pTMIExclFileMap,                                                      /**< [In] TMI exclusion FileMap. Should be 4 byte. */
        uint8_t bTMKeyOptions,                                                          /**< [In] Option for the TransactionMAC Key.
                                                                                         *          - KeyType of the KeyID.AppTransactionMACKey
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_KEYTYPE_AES128
                                                                                         *                     "AES128 Key"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_KEYTYPE_AES256
                                                                                         *                     "AES256 Key"
                                                                                         *
                                                                                         *        - Mode. Can bit bitwise OR with above options
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_MODE_TMAC
                                                                                         *                 "Transaction MAC"
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_MODE_TSIG
                                                                                         *                 "Transaction Signature"
                                                                                         */
        uint8_t bTSIGKeyNo,                                                             /**< [In] Key Number pointing to AppTransactionSIGKey. */
        uint8_t * pDivInput,                                                            /**< [In] Diversification Input used to diversify the key. */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t ** ppSamResponse,                                                       /**< [Out] Cryptogram holding Transaction MAC Key data. */
        uint16_t * pSamRespLen                                                          /**< [Out] Length of bytes available in \b ppSamResponse buffer. */
    );

/**
 * end of group phhalHw_Sam_Cmd_DESFire_SMode
 * @}
 */






/** \defgroup phhalHw_Sam_Cmd_DESFire_XMode X_Mode
 * \brief Implementation of SAM commands used for MIFARE DESFire communication in X-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_XMode_Defines Defines
 * \brief Definitions for SAM commands used for MIFARE DESFire communication in X-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_XMode_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_DESFIRE_AUTHENTICATE                               0xDAU   /**< SAM Instruction code for DESFIRE_Authenticate command. */
#define PHHAL_HW_SAM_CMD_INS_DESFIRE_CHANGE_KEY                                 0xDEU   /**< SAM Instruction code for DESFIRE_ChangeKey command. */
#define PHHAL_HW_SAM_CMD_INS_DESFIRE_WRITE_X                                    0xD3U   /**< SAM Instruction code for DESFIRE_WriteX command. */
#define PHHAL_HW_SAM_CMD_INS_DESFIRE_READ_X                                     0xD2U   /**< SAM Instruction code for DESFIRE_ReadX command. */
#define PHHAL_HW_SAM_CMD_INS_DESFIRE_CREATE_TM_FILE                             0xD1U   /**< SAM Instruction code for DESFIRE_CreateTMFilePICC command. */

/**
 * end of group phhalHw_Sam_Cmd_DESFire_XMode_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_XMode_Defines_ExtendedOffset ExtendedOffset
 * \brief Definitions for Extended Offset.
 * To be used with \ref phhalHw_Sam_Cmd_DESFire_WriteX "DESFire_WriteX" interface
 * @{
 */
#define PHHAL_HW_CMD_SAM_CRYPTO_CONFIG_LEGACY_MODE                              0x00U   /**< SAM DESFire Extended offset mode as legacy mode
                                                                                         *   (MIFARE SAM AV2).
                                                                                         */
#define PHHAL_HW_CMD_SAM_CRYPTO_CONFIG_EXTENDED_OFFSET                          0x80U   /**< SAM DESFire Extended offset mode as EV2 mode. */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_XMode_Defines_ExtendedOffset
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_XMode_Defines_ChainingConfig ChainingConfiguration
 * \brief Definitions for chaining configuration.
 * To be used with \ref phhalHw_Sam_Cmd_DESFire_WriteX "DESFire_WriteX" interface
 * @{
 */
#define PHHAL_HW_CMD_SAM_CRYPTO_CONFIG_DESFIRE_CHAINING                         0x00U   /**< SAM DESFire chaining mode as DESFire application chaining. */
#define PHHAL_HW_CMD_SAM_CRYPTO_CONFIG_ISO_CHAINING                             0x40U   /**< SAM DESFire chaining mode as ISO/IEC 14443-4 chaining. */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_XMode_Defines_ChainingConfig
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DESFire_XMode_Defines_Chaining Chaining
 * \brief Definitions for chaining configuration.
 * To be used with \ref phhalHw_Sam_Cmd_DESFire_ReadX "DESFire_ReadX" interface
 * @{
 */
#define PHHAL_HW_CMD_SAM_NATIVE_CHAINING                                        0x00U   /**< SAM DESFire chaining as Native. */
#define PHHAL_HW_CMD_SAM_ISO_CHAINING                                           0x08U   /**< SAM DESFire chaining as ISO. */
/**
 * end of group phhalHw_Sam_Cmd_DESFire_XMode_Defines_Chaining
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_DESFire_XMode_Defines
 * @}
 */

/**
 * \brief Perform an authentication procedure between SAM AV3 and MIFARE DESFire.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS               Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pPCDcap2In, \b pDivInput, \b pPDcap2
 *                                      \b pPCDcap2 and \b pPiccReturnCode
 *                                      are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_DESFire_AuthenticatePICC(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Option to update the P1 information. The options can be
                                                                                         *        combined by bitwise OR.
                                                                                         *          - Option for Authentication mode and Authentication type
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_AUTH_TYPE_D40_EV1
                                                                                         *                     "D40 & EV1 authentication type"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_AUTH_TYPE_EV2_FIRST_AUTH
                                                                                         *                     "EVx Authenticate first"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_AUTH_TYPE_EV2_NON_FIRST_AUTH
                                                                                         *                     "EVx Authenticate non first"
                                                                                         *
                                                                                         *          - Option for Suppressing secure messaging
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_ALLOW_SECURE_MESSAGING
                                                                                         *                     "Allow secure messaging"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_SUPPRESS_SECURE_MESSAGING
                                                                                         *                     "Suppress secure messaging (for example for
                                                                                         *                     originality keys)"
                                                                                         *
                                                                                         *          - Option for Key Derivation Information and Key Diversification
                                                                                         *            Types.
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_KDF_AV1
                                                                                         *                     "AV1 compatibility mode" key diversification methods,
                                                                                         *                     3TDEA, AES key
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_KDF_AV1_DOUBLE_ENCRYPTION
                                                                                         *                     "AV1 compatibility mode" key diversification methods,
                                                                                         *                     TDEA Key, diversified using two encryption rounds
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_KDF_AV1_SINGLE_ENCRYPTION
                                                                                         *                     "AV1 compatibility mode" key diversification methods,
                                                                                         *                     TDEA Key, diversified using one encryption round
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_KDF_AV2
                                                                                         *                     "AV2 mode key diversification methods"
                                                                                         *
                                                                                         *          - Option for Key Selection
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_KEY_SELECTION_KEY_ENTRY_NUMBER
                                                                                         *                     "Key selection by key entry number"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_KEY_SELECTION_DESFIRE_KEY_NUMBER
                                                                                         *                     "Key selection by DESFire key number"
                                                                                         *
                                                                                         *          - Option for Key Diversification
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_KEY_DIVERSIFICATION_OFF
                                                                                         *                     "No diversification"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_KEY_DIVERSIFICATION_ON
                                                                                         *                     "Diversify the used key with the given DivInput"
                                                                                         */
        uint8_t bISOMode,                                                               /**< [In] ISO mode to be used.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_ISO_MODE_NATIVE
                                                                                         *            "Command will be sent to PICC using native command set"
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_ISO_MODE_ISO7816
                                                                                         *            "Command will be sent to PICC using ISO/IEC 7816-4APDU"
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_ISO_MODE_ISO_AUTHENTICATION
                                                                                         *            "ISO compliant authentication" (Cmd.ISOGetChallenge,
                                                                                                      Cmd.ISOExternalAuthenticate and Cmd.ISOInternalAuthenticate)
                                                                                                      will be performed
                                                                                         */
        uint8_t bDFKeyNo,                                                               /**< [In] DESFire Key Number to be used for authentication. */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used in SAM or DESFire Key number.
                                                                                         *          - If Key selection by \ref PHHAL_HW_CMD_SAM_KEY_SELECTION_DESFIRE_KEY_NUMBER
                                                                                         *            "DESFire key number", 0x00 - 0x0D should be used.
                                                                                         *          - If Key selection by \ref PHHAL_HW_CMD_SAM_KEY_SELECTION_KEY_ENTRY_NUMBER
                                                                                         *            "SAM Key Number", then below one should be used
                                                                                         *              - NVM Key: 0x00 - 0x7F
                                                                                         *              - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used in SAM. */
        uint8_t bPCDcap2InLen,                                                          /**< [In] Length of PCDcap2 field in bytes.
                                                                                         *          - 0x00       : For NonFirst Authentication.
                                                                                         *          - 0x00       : For First Authentication with no input PCDCaps to use.
                                                                                         *          - 0xFF       : For First Authentication with default input PCDCaps to use.
                                                                                         *          - 0x00 - 0x06: For First Authentication with user defined PCDCaps to use.
                                                                                         */
        uint8_t * pPCDcap2In,                                                           /**< [In] Input PCD capabilities to be exchanged.
                                                                                         *          - NonFirstAuth                             : Should be null.
                                                                                         *          - FirstAuth with no Input PCD capabilities : Should be null.
                                                                                         *          - FirstAuth with PCDcap2InLen = 0xFF       : Should be null.
                                                                                         *          - FirstAuth with PCDcap2InLen = 0x00 - 0x06: Should not be null.
                                                                                         *            The PCD input capabilities should be passed.
                                                                                         */
        uint8_t * pDivInput,                                                            /**< [In] Diversification Input used to diversify the key.
                                                                                         *          - If \ref PHHAL_HW_CMD_SAM_KEY_DIVERSIFICATION_ON
                                                                                         *             "Diversification is Enabled", then
                                                                                         *              - 8 bytes (if AV1 key diversification with DES)
                                                                                         *              - 16 bytes (if AV1 key diversification with AES)
                                                                                         *              - 1 to 31 bytes (if AV2 key diversification)
                                                                                         *                diversification input
                                                                                         *          - NULL otherwise
                                                                                         */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t * pPDcap2,                                                              /**< [Out] Buffer containing the output PD capabilities. This will
                                                                                         *         contain 6 bytes of PD information, if targeting EVx
                                                                                         *         authentication type and first authentication.
                                                                                         */
        uint8_t * pPCDcap2,                                                             /**< [Out] Buffer containing the output PCD capabilities. This will
                                                                                         *         contain 6 bytes of PCD information, if targeting EVx
                                                                                         *         authentication type and first authentication.
                                                                                         */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Error code returned by PICC. Will be
                                                                                         *          - 1 byte if \b bISOMode = \ref PHHAL_HW_CMD_SAM_ISO_MODE_NATIVE
                                                                                         *            "Native Format"
                                                                                         *          - 2 bytes if \b bISOMode = \ref PHHAL_HW_CMD_SAM_ISO_MODE_ISO7816
                                                                                         *            "ISO/IEC 7816-4APDU" or
                                                                                         *            \ref PHHAL_HW_CMD_SAM_ISO_MODE_ISO_AUTHENTICATION "ISO compliant"
                                                                                         *            authentication format
                                                                                         */
    );

/**
 * \brief The DESFire_ChangeKeyPICC command supports the procedure to change a key
 * stored in the PICC in X-Mode
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS               Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pDivInput and \b pPiccReturnCode
 *                                      are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_DESFire_ChangeKeyPICC(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bCryptoMethod,                                                          /**< [In] Key compilation method. Options for P1 information byte.
                                                                                         *        The options can be combined by bitwise OR.
                                                                                         *          - Key diversification method
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_DIV_METHOD_AV1
                                                                                         *                "SAM AV1 compatibility diversification method"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_DIV_METHOD_AV2
                                                                                         *                "SAM AV2 diversification method"
                                                                                         *
                                                                                         *          - SAM AV1 and SAM AV2 Key diversification method
                                                                                         *              - Current Key
                                                                                         *                  - \ref PHHAL_HW_CMD_SAM_KDF_CURRENT_KEY_DOUBLE_ENCRYPTION
                                                                                         *                    "Diversify using two encryption rounds"
                                                                                         *                  - \ref PHHAL_HW_CMD_SAM_KDF_CURRENT_KEY_SINGLE_ENCRYPTION
                                                                                         *                    "Diversify using one encryption round"
                                                                                         *                  - \ref PHHAL_HW_CMD_SAM_KDF_CURRENT_KEY_OFF
                                                                                         *                    "Do not diversified the current key"
                                                                                         *                  - \ref PHHAL_HW_CMD_SAM_KDF_CURRENT_KEY_ON
                                                                                         *                    "Diversified the current key"
                                                                                         *              - New Key
                                                                                         *                  - \ref PHHAL_HW_CMD_SAM_KDF_NEW_KEY_DOUBLE_ENCRYPTION
                                                                                         *                    "Diversify using two encryption rounds"
                                                                                         *                  - \ref PHHAL_HW_CMD_SAM_KDF_NEW_KEY_SINGLE_ENCRYPTION
                                                                                         *                    "Diversify using one encryption round"
                                                                                         *                  - \ref PHHAL_HW_CMD_SAM_KDF_NEW_KEY_OFF
                                                                                         *                    "Do not diversified the new key"
                                                                                         *                  - \ref PHHAL_HW_CMD_SAM_KDF_NEW_KEY_ON
                                                                                         *                    "Diversified the new key"
                                                                                         *
                                                                                         *          - Cryptogram computation mode
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CRYPTO_MODE_DIFFERENT_KEY
                                                                                         *                "PICC targeted key different" from the PICC authenticated
                                                                                         *                 key
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CRYPTO_MODE_SAME_KEY
                                                                                         *                "PICC targeted key equal" to PICC authenticated key.
                                                                                         *                The parameters CurrKeyNo and CurrKeyV are ignored"
                                                                                         */
        uint8_t bConfig,                                                                /**< [In] Options for P2 information byte. The options can be
                                                                                         *        combined by bitwise OR.
                                                                                         *          - ISO mode to be used.
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_ISO_MODE_NATIVE
                                                                                         *                "Command will be sent to PICC using native command set"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_ISO_MODE_ISO7816
                                                                                         *                "Command will be sent to PICC using ISO/IEC 7816-4APDU"
                                                                                         *
                                                                                         *          - PICC Change Key command
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CMD_TYPE_CHANGE_KEY
                                                                                         *                "Cmd.ChangeKey"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CMD_TYPE_CHANGE_KEY_EV2
                                                                                         *                "Cmd.ChangeKeyEV2"
                                                                                         *
                                                                                         *          - PICC master key update
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_MASTER_KEY_UPDATE_EXCLUDE_KEYTYPE
                                                                                         *                "Do not include the key type in the cryptogram"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_MASTER_KEY_UPDATE_INCLUDE_KEYTYPE
                                                                                         *                "Include the key type in the cryptogram"
                                                                                         *
                                                                                         *          - Number of PICC key to be changed. Will be taken from \b bDFKeyNo
                                                                                         *            parameter if \b bConfig has \ref  PHHAL_HW_CMD_SAM_CMD_TYPE_CHANGE_KEY
                                                                                         *            "Cmd.ChangeKey"
                                                                                         */
        uint8_t bKeySetNo,                                                              /**< [In] If \b bConfig = \ref  PHHAL_HW_CMD_SAM_CMD_TYPE_CHANGE_KEY_EV2
                                                                                         *        "Cmd.ChangeKeyEV2", Key set number to which the key to be
                                                                                         *        changed belongs to.
                                                                                         */
        uint8_t bDFKeyNo,                                                               /**< [In] Number of DESFire PICC key to be changed. This should be present only if
                                                                                         *        \b bConfig = \ref  PHHAL_HW_CMD_SAM_CMD_TYPE_CHANGE_KEY_EV2
                                                                                         *        "Cmd.ChangeKeyEV2".
                                                                                         */
        uint8_t bCurrKeyNo,                                                             /**< [In] Number of key entry holding the current key in SAM.
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bCurrKeyVer,                                                            /**< [In] Key version of the current key in SAM. */
        uint8_t bNewKeyNo,                                                              /**< [In] Number of key entry holding the new key in SAM.
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bNewKeyVer,                                                             /**< [In] Key version of new key in SAM. */
        uint8_t * pDivInput,                                                            /**< [In] Diversification Input used to diversify the key.
                                                                                         *          - If any of diversification option is set in
                                                                                         *            \b bCryptoMethod parameter, then
                                                                                         *              - 8 bytes (if AV1 key diversification with DES)
                                                                                         *              - 16 bytes (if AV1 key diversification with AES)
                                                                                         *              - 1 to 31 bytes (if AV2 key diversification)
                                                                                         *                diversification input
                                                                                         *          - NULL otherwise
                                                                                         */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Error code returned by PICC. Will be
                                                                                         *          - 1 byte if \b bISOMode = \ref  PHHAL_HW_CMD_SAM_ISO_MODE_NATIVE
                                                                                         *            "Native Format"
                                                                                         *          - 2 bytes if \b bISOMode = \ref PHHAL_HW_CMD_SAM_ISO_MODE_ISO7816
                                                                                         *            "ISO/IEC 7816-4APDU".
                                                                                         */
    );

/**
 * \brief The DESFire_WriteX command is used to apply the DESFire Secure Messaging in X-mode
 * on PICC commands writing data to the DESFire PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS               Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData, \b pPiccReturnCode and \b pErrLen
 *                                      are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_DESFire_WriteX(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Buffering options for exchanging information to SAM.
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT
                                                                                         *                 "Exchange information to SAM and Receive the response"
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_FIRST
                                                                                         *                 "Buffer first" set of information. No exchange is performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_CONT
                                                                                         *                 "Buffer intermediate" set of information. No exchange is performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_LAST
                                                                                         *                 "Buffer last" set of information. Exchange information to SAM and
                                                                                         *                  Receive the response
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_TXCHAINING "Should be used to exchange" chunks of data
                                                                                         *            and receive response. To be ORed with above option and to set LFI flag
                                                                                         *            of P1 byte.
                                                                                         */
        uint8_t bCrypto,                                                                /**< [In] Option to set the P2 information byte. CAn be combined by
                                                                                         *        using bitwise OR operator.
                                                                                         *          - Extended offset
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CRYPTO_CONFIG_LEGACY_MODE
                                                                                         *                "Legacy mode (MIFARE SAM AV2)"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CRYPTO_CONFIG_EXTENDED_OFFSET
                                                                                         *                "Extended offset"
                                                                                         *
                                                                                         *          - Chaining configuration
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CRYPTO_CONFIG_DESFIRE_CHAINING
                                                                                         *                "DESFire application chaining (0xAF)"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CRYPTO_CONFIG_ISO_CHAINING
                                                                                         *                "ISO/IEC 14443-4 chaining"
                                                                                         *
                                                                                         *          - Communication Mode
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_COMM_MODE_PLAIN "CommMode.Plain"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_COMM_MODE_MAC "CommMode.MAC"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_COMM_MODE_FULL "CommMode.Full"
                                                                                         *
                                                                                         *          - Index of byte in data field where crypto operation shall
                                                                                         *            start
                                                                                         *              - If Legacy mode: Full Range (0x00 - 0x0F)
                                                                                         *              - If Extended offset: RFU (0x00)
                                                                                         */
        uint8_t * pData,                                                                /**< [In] The data to be written to the DESFire PICC.
                                                                                         *          - If \ref PHHAL_HW_CMD_SAM_CRYPTO_CONFIG_EXTENDED_OFFSET
                                                                                         *            "Extended offset mode", Communication mode is
                                                                                         *            \ref PHHAL_HW_CMD_SAM_COMM_MODE_FULL "Full" and first frame,
                                                                                         *            the buffer should contain offset followed by the PICC related
                                                                                         *            data.
                                                                                         *          - For any other options, the buffer should contain only the PICC related
                                                                                         *            data.
                                                                                         */
        uint8_t bDataLen,                                                               /**< [In] Length of bytes available in \b pData buffer. */
        uint8_t * pPiccReturnCode,                                                      /**< [Out] Error code returned by PICC. Will be
                                                                                         *          - 1 byte if \b bISOMode = \ref  PHHAL_HW_CMD_SAM_ISO_MODE_NATIVE
                                                                                         *            "Native Format"
                                                                                         *          - 2 bytes if \b bISOMode = \ref PHHAL_HW_CMD_SAM_ISO_MODE_ISO7816
                                                                                         *            "ISO/IEC 7816-4APDU".
                                                                                         */
        uint8_t * pErrLen                                                               /**< [Out] Length of bytes available in \b pPiccReturnCode buffer. */
    );

/**
 * \brief The DESFire_ReadX command is used to remove the DESFire Secure Messaging in X-mode
 * on PICC commands reading data from the DESFire PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS               Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pAppData, \b pRespLen, \b pPiccReturnCode
 *                                      and \b pErrLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_DESFire_ReadX(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Buffering options.
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT "Default": To buffer
                                                                                         *            the initial command information. Use this flag
                                                                                         *            to buffer the Length information also in case
                                                                                         *            of FULL communication mode and more data is
                                                                                         *            expected.
                                                                                         *
                                                                                         *          - To receive data in \ref PH_EXCHANGE_RXCHAINING
                                                                                         *            "Native Chaining" mode
                                                                                         *
                                                                                         *          - To receive data in ISO chaining mode, use
                                                                                         *            \ref PH_EXCHANGE_RXCHAINING "Response Chaining" |
                                                                                         *            \ref PHHAL_HW_CMD_SAM_ISO_CHAINING "ISO Chaining"
                                                                                         *
                                                                                         */
        uint8_t bCrypto,                                                                /**< [In] Crypto configuration. Option to set the P2 information
                                                                                         *        byte.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_COMM_MODE_PLAIN "CommMode.Plain"
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_COMM_MODE_MAC "CommMode.MAC"
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_COMM_MODE_FULL "CommMode.Full"
                                                                                         */
        uint8_t * pAppData,                                                             /**< [In] The following information should be passed.
                                                                                         *          - 3 bytes length information in case if Communication
                                                                                         *            mode is \ref PHHAL_HW_CMD_SAM_COMM_MODE_FULL "Full"
                                                                                         *            and more data is expected.
                                                                                         *          - Complete PICC command header and data to be sent to PICC
                                                                                         *            for initial exchange.
                                                                                         *          - DESFire Chaining command code (0xAF) in case more data is
                                                                                         *            expected.
                                                                                         */
        uint8_t bAppDataLen,                                                            /**< [In] Length of bytes available in \b pAppData buffer. */
        uint8_t ** ppResponse,                                                          /**< [Out] The data received from SAM. */
        uint16_t * pRespLen,                                                            /**< [Out] Length of bytes available in \b ppResponse buffer. */
        uint8_t * pPiccReturnCode,                                                      /**< [Out] Error code returned by PICC. Will be
                                                                                         *          - 1 byte if \b pData information exchanged to SAM is
                                                                                         *            in Native format.
                                                                                         *          - 2 bytes if \b pData information exchanged to SAM is
                                                                                         *            in ISO/IEC 7816-4APDU.
                                                                                         */
        uint8_t * pErrLen                                                               /**< [Out] Length of bytes available in \b pPiccReturnCode buffer. */
    );

/**
 * \brief The DESFire_CreateTMFilePICC command supports the procedure to create a Transaction
 * MAC File in the PICC in X-mode from a key stored in the SAM.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS               Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pAccessRights, \b pDivInput and
 *                                      \b pPiccReturnCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_DESFire_CreateTMFilePICC(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Key diversification selection. Options for P1
                                                                                         *        information byte.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_DIVERSIFICATION_OFF
                                                                                         *            "No diversification"
                                                                                         *          - \ref  #PHHAL_HW_CMD_SAM_CREATE_TM_FILE_DIVERSIFICATION_OFF
                                                                                         *            "Diversify the used key with the given DivInput"
                                                                                         */
        uint8_t bISOMode,                                                               /**< [In] Options for P2 information byte.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_ISO_MODE_NATIVE
                                                                                         *            "Command will be sent to PICC using native command set"
                                                                                         *          - \ref  PHHAL_HW_CMD_SAM_ISO_MODE_ISO7816
                                                                                         *            "Command will be sent to PICC using ISO/IEC 7816-4APDU"
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used in SAM. One of the following
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used in SAM. */
        uint8_t bFileNo,                                                                /**< [In] File number of the file to be created. */
        uint8_t bFileOption,                                                            /**< [In] Options for the targeted file.
                                                                                         *          - Communication Mode
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_COMM_MODE_PLAIN
                                                                                         *                "FileOptions as PLAIN"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_COMM_MODE_MAC
                                                                                         *                "FileOptions as MAC"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_COMM_MODE_FULL
                                                                                         *                "FileOptions as FULL"
                                                                                         *
                                                                                         *        - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_TMI_EXCLUSION_FILEMAP
                                                                                         *          "TMI Exclusion file Map". Can bit bitwise OR with above option
                                                                                         */
        uint8_t * pAccessRights,                                                        /**< [In] Access conditions to be applied for the file. Refer
                                                                                         *        respective product DataSheet for access rights
                                                                                         *        information. This should be two bytes long.
                                                                                         */
        uint8_t * pTMIExclFileMap,                                                      /**< [In] TMI exclusion FileMap. Should be 4 byte. */
        uint8_t bTMKeyOptions,                                                          /**< [In] Option for the TransactionMAC Key. One of the following,
                                                                                         *          - KeyType of the KeyID.AppTransactionMACKey
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_KEYTYPE_AES128
                                                                                         *                     "AES128 Key"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_KEYTYPE_AES256
                                                                                         *                     "AES256 Key"
                                                                                         *
                                                                                         *        - Mode. Can bit bitwise OR with above options
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_MODE_TMAC
                                                                                         *                 "Transaction MAC"
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_CREATE_TM_FILE_MODE_TSIG
                                                                                         *                 "Transaction Signature"
                                                                                         */
        uint8_t bTSIGKeyNo,                                                             /**< [In] Key Number pointing to AppTransactionSIGKey. */
        uint8_t * pDivInput,                                                            /**< [In] Diversification Input used to diversify the key. */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Error code returned by PICC. Will be
                                                                                         *          - 1 byte if \b bISOMode = \ref PHHAL_HW_CMD_SAM_ISO_MODE_NATIVE
                                                                                         *            "Native Format"
                                                                                         *          - 2 bytes if \b bISOMode = \ref PHHAL_HW_CMD_SAM_ISO_MODE_ISO7816
                                                                                         *            "ISO/IEC 7816-4APDU" or
                                                                                         *            \ref PHHAL_HW_CMD_SAM_ISO_MODE_ISO_AUTHENTICATION "ISO compliant"
                                                                                         *            authentication formats
                                                                                         */
    );

/**
 * end of group phhalHw_Sam_Cmd_DESFire_XMode
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_DESFire
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_DESFIRE_H */
