/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Virtual Card command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include "phhalHw_Sam_Cmd_VC.h"

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "../phhalHw_Sam_Cmd.h"

/* S - Mode Commands --------------------------------------------------------------------------------------------------------- */
phStatus_t phhalHw_Sam_Cmd_SAM_SelectVC(phhalHw_Sam_DataParams_t *pDataParams, uint8_t bOption, uint8_t bEncKeyNo,
    uint8_t bEncKeyVer, uint8_t bMacKeyNo, uint8_t bMacKeyVer, uint8_t * pData, uint8_t bDataLen,
    uint8_t * pDivInput, uint8_t  bDivInputLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pData, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRespLen, PH_COMP_HAL);
    if((bDivInputLen != 0) && (bOption != PHHAL_HW_SAM_CMD_SELECT_DIV_DEFAULT))
        PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_SelectVC command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_SELECT_VC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Append the Key number and Key version. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bEncKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bEncKeyVer;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bMacKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bMacKeyVer;

    /* Buffer command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer data information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pData,
        bDataLen,
        NULL,
        NULL));

    /* Buffer the diversification input information to command buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pDivInput,
        bDivInputLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the bufferred information to SAM hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppResponse,
        pRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_ProximityCheck_Part1(phhalHw_Sam_DataParams_t * pDataParams, uint8_t  bOption,
    uint8_t bKeyNo, uint8_t  bKeyVer, uint8_t * pPPCData, uint8_t  bPPCDataLen, uint8_t * pPCData,
    uint8_t bPCDataLen, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t ** ppMac,
    uint16_t * pMacLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPPCData, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPCData, PH_COMP_HAL);
    if(bOption == PHHAL_HW_SAM_CMD_PROXIMITY_CHECK_DIV_ON)
        PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pMacLen, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_ProximityCheck part 1 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_PROXIMITY_CHECK;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add the payload information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVer;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPPCDataLen;

    /* Buffer the command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer PPCData information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pPPCData,
        bPPCDataLen,
        NULL,
        NULL));

    /* Buffer PCData information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pPCData,
        bPCDataLen,
        NULL,
        NULL));

    /* Buffer diversification input information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pDivInput,
        bDivInputLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the bufferred information to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppMac,
        pMacLen);

    /* Check for the Chaining active */
    if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        return wStatus;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_ProximityCheck_Part2(phhalHw_Sam_DataParams_t * pDataParams, uint8_t * pData,
    uint8_t bDataLen, uint8_t * pPiccRetCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pData, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPiccRetCode, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_ProximityCheck part 2 command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_PROXIMITY_CHECK;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add Data inforamtion to command buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pData, bDataLen);
    bCmdLen += bDataLen;

    /* Add LE to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC. */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange SAM_ProximityCheck part 2 command information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Extract the PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_MIFARE_GEN)
    {
        /* Assign the PICC response code the parameter. */
        *pPiccRetCode = pResponse[0];
    }

    return wStatus;
}





/* X - Mode Commands --------------------------------------------------------------------------------------------------------- */
phStatus_t phhalHw_Sam_Cmd_VCA_Select(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bEncKeyNo,
    uint8_t bEncKeyVer, uint8_t bMacKeyNo, uint8_t bMacKeyVer, uint8_t * pIID, uint8_t bIIDLen, uint8_t * pDivInput,
    uint8_t bDivInputLen, uint8_t ** ppResponse, uint16_t * pRespLen, uint16_t * pPiccRetCode)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pIID, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRespLen, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPiccRetCode, PH_COMP_HAL);
    if((bOption  == PHHAL_HW_SAM_CMD_SELECT_ENC_KEY_DIV_INPUT) ||
        (bOption == PHHAL_HW_SAM_CMD_SELECT_MAC_KEY_DIV_INPUT) ||
        (bOption == PHHAL_HW_SAM_CMD_SELECT_MAC_KEY_DIV_INPUT_VCUID))
        PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame VCA_Select part1 variant command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_VCA_SELECT;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Frame the payload information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bEncKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bEncKeyVer;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bMacKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bMacKeyVer;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bIIDLen;

    /* Buffer the command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer the IID information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pIID,
        bIIDLen,
        NULL,
        NULL));

    /* Buffer the diversification input information to exchange buffer. */
    if(bOption & (PHHAL_HW_SAM_CMD_SELECT_ENC_KEY_DIV_INPUT | PHHAL_HW_SAM_CMD_SELECT_MAC_KEY_DIV_INPUT |
        PHHAL_HW_SAM_CMD_SELECT_MAC_KEY_DIV_VCUID | PHHAL_HW_SAM_CMD_SELECT_MAC_KEY_DIV_INPUT_VCUID))
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pDivInput,
            bDivInputLen,
            NULL,
            NULL));
    }

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the bufferred information to SAM hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppResponse,
        pRespLen);

    /* Update the PICC status code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_MIFARE_GEN)
    {
        *pPiccRetCode = PHHAL_HW_SAM_GET_PICC_ERROR(ppResponse[0], *pRespLen);
    }
    else
    {
        /* Nothing to do here
         * The output of Cmd.VCA_Select is already available in response buffers.
         */
    }

    /* Verify the status if 2Part variant and exchange the second command frame. */
    if(bOption & PHHAL_HW_SAM_CMD_VCA_SELECT_VARIANT_PART2)
    {
        /* Check for OK_CHAINING_ACTIVE_EXT */
        if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE_EXT)
        {
            return wStatus;
        }

        /* Reset the command buffer. */
        bCmdLen = 0;
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame VCA_Select part1 variant command information. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_VCA_SELECT;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Buffer the command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            NULL,
            NULL));

        /* Buffer the diversification input information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pDivInput,
            bDivInputLen,
            NULL,
            NULL));

        /* Update LC. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

        /* Buffer LE and exchange the bufferred information to SAM hardware. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            phhalHw_Sam_Cmd_DefaultLe,
            PHHAL_HW_CMD_SAM_LE_LENGTH,
            &pResponse,
            &wRespLen));

        /* Update the PICC status code. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_MIFARE_GEN)
        {
            *pPiccRetCode = PHHAL_HW_SAM_GET_PICC_ERROR(pResponse, wRespLen);
            *pRespLen = wRespLen;
        }
    }
    else
    {
        PH_CHECK_SUCCESS(wStatus);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_VCA_ProximityCheck(phhalHw_Sam_DataParams_t *pDataParams, uint8_t bOption, uint8_t bKeyNo,
    uint8_t bKeyVer, uint8_t bNumOfRand, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bOption & PHHAL_HW_SAM_CMD_PROXIMITY_CHECK_DIV_ON) PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRespLen, PH_COMP_HAL);

    /* Reset the command buffer and its length variable. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame VCA_ProximityCheck command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_VCA_PROXIMITY_CHECK;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add the payload information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVer;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bNumOfRand;

    /* Buffer the command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer Data information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pDivInput,
        bDivInputLen,
        NULL,
        NULL));

    /* Update LC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Buffer LE and exchange the bufferred information to SAM hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppResponse,
        pRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
