/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Public Key Infrastructure command implementation of Reader
 * Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include "phhalHw_Sam_Cmd_PKI.h"

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "../phhalHw_Sam_Cmd.h"
#include "../01_HostCommunication/phhalHw_Sam_Cmd_HC.h"
#include "../02_SecurityConfiguration/phhalHw_Sam_Cmd_SC.h"

/* RSA Commands ---------------------------------------------------------------------------------------------------------- */
phStatus_t phhalHw_Sam_Cmd_PKI_GenerateKeyPair(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bPKI_KeyNo,
    uint16_t wPKI_Set, uint8_t bPKI_KeyNoCEK, uint8_t bPKI_KeyVCEK, uint8_t bPKI_RefNoKUC, uint8_t bPKI_KeyNoAEK,
    uint8_t bPKI_KeyVAEK, uint16_t wPKI_NLen, uint16_t wPKI_eLen, uint8_t * pPKI_e)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bOption & PHHAL_HW_SAM_CMD_PKI_GENERATE_KEY_PAIR_HOST_E)
        PH_ASSERT_NULL_PARAM(pPKI_e, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame the PKI_GenerateKeyPair command header. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_GENERATE_KEY_PAIR;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_PKI_GENERATE_KEY_PAIR_LC_MIN;

    /* Update LC byte if Access Entry Key information is available. */
    if(bOption & PHHAL_HW_SAM_CMD_PKI_INCLUDE_AEK)
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_LC_POS] += 2U /* PKI_KeyNoAEK and PKI_KeyVAEK. */;

    /* Update P2 information and LC byte. */
    if((wPKI_eLen >= PHHAL_HW_SAM_CMD_PKI_E_LEN) && (bOption & PHHAL_HW_SAM_CMD_PKI_GENERATE_KEY_PAIR_HOST_E))
    {
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_LC_POS] += PHHAL_HW_SAM_CMD_PKI_E_LEN /* First part of PKI_e information. */;
    }
    else
    {
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
        if(bOption & PHHAL_HW_SAM_CMD_PKI_GENERATE_KEY_PAIR_HOST_E)
        {
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_LC_POS] += (uint8_t) wPKI_eLen;
        }
    }

    /* Append the payload information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wPKI_Set);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wPKI_Set);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNoCEK;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyVCEK;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_RefNoKUC;

    /* Add AEK key number and version if set in P1.1 */
    if(bOption & PHHAL_HW_SAM_CMD_PKI_INCLUDE_AEK)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNoAEK;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyVAEK;
    }

    /* Append the rest of the Payload information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wPKI_NLen);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wPKI_NLen);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wPKI_eLen);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wPKI_eLen);

    /* Buffer the command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Exchange the final information to SAM hardware in case of last frame. */
    if(PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] == PHHAL_HW_SAM_ISO7816_LAST_FRAME)
    {
        /* Check if User / Host has provided the exponent information. */
        if(bOption & PHHAL_HW_SAM_CMD_PKI_GENERATE_KEY_PAIR_HOST_E)
        {
            /* PKI_e has to be send */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_LAST,
                pPKI_e,
                wPKI_eLen,
                &pResponse,
                &wRespLen));
        }
        else
        {
            /* No PKI_e has to be send. It will be randomly generated by SAM hardware. */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_LAST,
                NULL,
                0,
                &pResponse,
                &wRespLen));
        }
    }

    /* Exchange the final frame to SAM hardware and perform a second part of command / information exchange to SAM hardware. */
    else
    {
        /* Append exponent payload and perform final exchange with SAM hardware. */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            pPKI_e,
            PHHAL_HW_SAM_CMD_PKI_E_LEN, /* LE is not present in that command */
            &pResponse,
            &wRespLen);

        /* Return the status code if its not chaining. */
        if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
            return wStatus;

        /* Reset the command buffer and its length. */
        bCmdLen = 0;
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

        /* Frame the PKI_GenerateKeyPair command header. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_GENERATE_KEY_PAIR;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (wPKI_eLen - PHHAL_HW_SAM_CMD_PKI_E_LEN);

        /* Buffer the header information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            NULL,
            NULL));

        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            pPKI_e + PHHAL_HW_SAM_CMD_PKI_E_LEN,
            (uint16_t) (wPKI_eLen - PHHAL_HW_SAM_CMD_PKI_E_LEN),
            &pResponse,
            &wRespLen));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_PKI_ImportKey(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bPKI_KeyNo,
    uint16_t wPKI_Set, uint8_t bPKI_KeyNoCEK, uint8_t bPKI_KeyVCEK, uint8_t bPKI_RefNoKUC, uint8_t bPKI_KeyNoAEK,
    uint8_t bPKI_KeyVAEK, uint16_t wPKI_NLen, uint16_t wPKI_eLen, uint16_t wPKI_PLen, uint16_t wPKI_QLen,
    uint8_t * pPKI_N, uint8_t * pPKI_e, uint8_t * pPKI_p, uint8_t * pPKI_q, uint8_t * pPKI_dP,
    uint8_t * pPKI_dQ, uint8_t * pPKI_ipq)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;
    uint16_t   PH_MEMLOC_REM wRemainingMsgLen = 0;
    uint8_t    PH_MEMLOC_REM bIsPrivateKeyIncluded = 0;
    uint8_t    PH_MEMLOC_REM bState = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    uint8_t *  PH_MEMLOC_REM pPayload = NULL;
    uint16_t   PH_MEMLOC_REM wPayLoadLen = 0;
    uint16_t   PH_MEMLOC_REM wPayLoadSize = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    /* Check if private key is included. */
    bIsPrivateKeyIncluded = (uint8_t) (wPKI_Set & PH_ON);

    /* Parameter validation. */
    if((bOption & PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_ONLY) != PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_ONLY)
    {
        PH_ASSERT_NULL_PARAM(pPKI_N, PH_COMP_HAL);
        PH_ASSERT_NULL_PARAM(pPKI_e, PH_COMP_HAL);

        if(bIsPrivateKeyIncluded == (uint8_t) PH_ON)
        {
            PH_ASSERT_NULL_PARAM(pPKI_p, PH_COMP_HAL);
            PH_ASSERT_NULL_PARAM(pPKI_q, PH_COMP_HAL);
            PH_ASSERT_NULL_PARAM(pPKI_dP, PH_COMP_HAL);
            PH_ASSERT_NULL_PARAM(pPKI_dQ, PH_COMP_HAL);
            PH_ASSERT_NULL_PARAM(pPKI_ipq, PH_COMP_HAL);
        }
    }

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame the PKI_ImportKey command header. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_IMPORT_KEY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_PKI_IMPORT_KEY_LC_MIN;

    /* Update Remaining message length variable. */
    wRemainingMsgLen = PHHAL_HW_SAM_ISO7816_HEADER_LENGTH;

    /* Update LC byte if Access Entry Key information is available. */
    if((bOption & PHHAL_HW_SAM_CMD_PKI_INCLUDE_AEK) == PHHAL_HW_SAM_CMD_PKI_INCLUDE_AEK)
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_LC_POS] += 2U /* PKI_KeyNoAEK and PKI_KeyVAEK. */;

    /* Append the payload information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wPKI_Set);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wPKI_Set);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNoCEK;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyVCEK;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_RefNoKUC;

    /* Add AEK key number and version if set in P1.1 */
    if((bOption & PHHAL_HW_SAM_CMD_PKI_INCLUDE_AEK) == PHHAL_HW_SAM_CMD_PKI_INCLUDE_AEK)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNoAEK;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyVAEK;
    }

    if((bOption & PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_ONLY) == PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_ONLY)
    {
        /* Update the P2 information byte with Last frame as indicator. */
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;

        /* Buffer the command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            &pResponse,
            &wRespLen));

        /* Only settings have to be sent, so return success status. */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    /* Add modulus and exponent lengths to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wPKI_NLen);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wPKI_NLen);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wPKI_eLen);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wPKI_eLen);

    /* Update remaining message length variable with modulus and exponent length. */
    wRemainingMsgLen += (uint16_t) (wPKI_NLen + wPKI_eLen);

    /* Include Prime P and Q lengths to command buffer if private key is included. */
    if(bIsPrivateKeyIncluded == (uint8_t) PH_ON)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wPKI_PLen);
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wPKI_PLen);
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wPKI_QLen);
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wPKI_QLen);

        /* Update remaining message length variable with Prime P and Q length. */
        wRemainingMsgLen += (uint16_t) ((wPKI_PLen * 2U) + (wPKI_QLen * 3U));
    }

    /* Update Remaining message length variable with the key entry length components (bCmdLen - I7816 Header). */
    wRemainingMsgLen += (uint16_t) (bCmdLen - PHHAL_HW_SAM_ISO7816_HEADER_LENGTH);

    /* Check whether we have to indicate chaining or not */
    if(wRemainingMsgLen > (PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK - 16U))
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;

    /* Buffer the command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Exchange PKI_N, PKI_e, PK_p, PKI_q, PKI_dP, PKI_dQ, PKI_ipq. */
    bState = 0;
    wPayLoadLen = 0;
    wPayLoadSize = 0;
    do
    {
        /* Updated the states to exchange different payloads. */
        switch(bState)
        {
            case 0: /* State to exchange PKI_N information. */
                pPayload = &pPKI_N[wPayLoadLen];
                wPayLoadSize = wPKI_NLen - wPayLoadLen;
                break;

            case 1U: /* State to exchange PKI_e information. */
                pPayload = &pPKI_e[wPayLoadLen];
                wPayLoadSize = wPKI_eLen - wPayLoadLen;
                break;

            case 2U: /* State to exchange PKI_p information. */
                pPayload = &pPKI_p[wPayLoadLen];
                wPayLoadSize = wPKI_PLen - wPayLoadLen;
                break;

            case 3U: /* State to exchange PKI_q information. */
                pPayload = &pPKI_q[wPayLoadLen];
                wPayLoadSize = wPKI_QLen - wPayLoadLen;
                break;

            case 4U: /* State to exchange PKI_dP information. */
                pPayload = &pPKI_dP[wPayLoadLen];
                wPayLoadSize = wPKI_PLen - wPayLoadLen;
                break;

            case 5U: /* State to exchange PKI_dQ information. */
                pPayload = &pPKI_dQ[wPayLoadLen];
                wPayLoadSize = wPKI_QLen - wPayLoadLen;
                break;

            case 6U: /* State to exchange PKI_ipq information. */
                pPayload = &pPKI_ipq[wPayLoadLen];
                wPayLoadSize = wPKI_QLen - wPayLoadLen;
                break;

            default: /* State to reset variables. */
                pPayload = NULL;
                wPayLoadSize = 0;
                break;
        }

        /* Append payload to the buffer */
        if(wPayLoadSize > 0)
        {
            /* Payload fits into current frame without truncation */
            if((bCmdLen + wPayLoadSize) < (PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK - 16U))
            {
                /* Check if its the last frame and perform final exchange. */
                if((wRemainingMsgLen - (bCmdLen + wPayLoadSize)) == 0)
                {
                    /* Append rest of data. */
                    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                        pDataParams,
                        PH_EXCHANGE_BUFFER_CONT,
                        pPayload,
                        wPayLoadSize,
                        NULL,
                        NULL));

                    /* Update LC */
                    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

                    /* perform exchange */
                    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                        pDataParams,
                        PH_EXCHANGE_BUFFER_LAST,
                        NULL,
                        0,
                        &pResponse,
                        &wRespLen));
                }
                /* Just buffer the frame. */
                else
                {
                    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                        pDataParams,
                        PH_EXCHANGE_BUFFER_CONT,
                        pPayload,
                        wPayLoadSize,
                        NULL,
                        NULL));

                    /* Update current frame length */
                    bCmdLen = (uint8_t) (bCmdLen + wPayLoadSize);
                }

                /* Reset payload length */
                wPayLoadLen = 0;

                /* Advance to next payload */
                ++bState;

                /* Set bState to zero if private key is not included. */
                if(!bIsPrivateKeyIncluded && (bState == 2U))
                    bState = 0xFFU;
            }
            /* Else send maximum amount of possible data to SAM. */
            else
            {
                wPayLoadSize = (uint16_t) ((PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK - 16U) - bCmdLen);
                wPayLoadLen = (uint16_t) (wPayLoadLen + wPayLoadSize);

                /* Append rest of data */
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                    pDataParams,
                    PH_EXCHANGE_BUFFER_CONT,
                    pPayload,
                    wPayLoadSize,
                    NULL,
                    NULL));

                /* Update LC */
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

                /* perform exchange */
                wStatus = phhalHw_Sam_Cmd_7816Exchange(
                    pDataParams,
                    PH_EXCHANGE_BUFFER_LAST,
                    NULL,
                    0,
                    &pResponse,
                    &wRespLen);

                /* status check */
                if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
                    return wStatus;

                /* Update current frame length */
                bCmdLen = (uint8_t) (bCmdLen + wPayLoadSize);

                /* Check for internal error */
                if(wRemainingMsgLen < bCmdLen)
                    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);

                /* Update remaining message length */
                wRemainingMsgLen = (uint16_t) (wRemainingMsgLen - bCmdLen);

                /* Check whether we have to indicate chaining or not */
                if(wRemainingMsgLen > (PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK - 16U))
                    PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
                else
                    PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;

                /* Buffer the command header */
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                    pDataParams,
                    PH_EXCHANGE_BUFFER_FIRST,
                    PHHAL_HW_SAM_CMD_BUFFER,
                    PHHAL_HW_SAM_ISO7816_HEADER_LENGTH,
                    NULL,
                    NULL));

                /* Reset current frame length */
                bCmdLen = 0;
            }
        }
    } while(wPayLoadSize > 0);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_PKI_ExportPrivateKey(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t bPKI_KeyNo,
    uint8_t ** ppKeyData, uint16_t *pKeyDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pKeyDataLen, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame the command for PKI_ExportPrivateKey command. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_EXPORT_PRIVATE_KEY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (wOption & PHHAL_HW_SAM_CMD_PKI_RECEIVE_AEK);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update the P1 information with default value. */
    if(wOption & PH_EXCHANGE_RXCHAINING)
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;

    /* Exchange PKI_ExportPrivateKey information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppKeyData,
        pKeyDataLen);

    /* Validate status */
    if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        PH_CHECK_SUCCESS(wStatus);
    }
    else
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_PKI_ExportPublicKey(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t bPKI_KeyNo,
    uint8_t ** ppKeyData, uint16_t *pKeyDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pKeyDataLen, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame the command for PKI_ExportPublicKey command. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_EXPORT_PUBLIC_KEY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (wOption & PHHAL_HW_SAM_CMD_PKI_RECEIVE_AEK);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update the P1 information with default value. */
    if(wOption & PH_EXCHANGE_RXCHAINING)
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;

    /* Exchange PKI_ExportPublicKey information to SAM. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppKeyData,
        pKeyDataLen);

     /* Validate status */
    if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        PH_CHECK_SUCCESS(wStatus);
    }
    else
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_PKI_UpdateKeyEntries(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bNoOfKeyEntries,
    uint8_t bHashingAlg, uint8_t bPKI_KeyNo_Enc, uint8_t bPKI_KeyNo_Sign, uint8_t bPKI_KeyNo_Ack, uint8_t * pKeyFrame,
    uint16_t wKeyFrameLen, uint8_t ** ppUpdateACK, uint16_t * pUpdateACKLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;
    uint16_t   PH_MEMLOC_REM wBufPos = 0;
    uint8_t    PH_MEMLOC_REM bSendHeader = 0;
    uint8_t    PH_MEMLOC_REM bPayloadLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pKeyFrame, PH_COMP_HAL);
    if(bOption & PHHAL_HW_SAM_CMD_PKI_RECEIVE_ACK)
        PH_ASSERT_NULL_PARAM(pUpdateACKLen, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame the PKI_UpdateKeyEntries command. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_UPDATE_KEY_ENTRIES;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bHashingAlg | (uint8_t) (bNoOfKeyEntries << 2U);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;

    /* Add Enc and Sign key numbers to command buffer. */
    if(bOption & PH_EXCHANGE_RXCHAINING)
    {
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    }
    else
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNo_Enc;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNo_Sign;

        /* Add ACK key numbers to command buffer if required. */
        if(bOption & PHHAL_HW_SAM_CMD_PKI_RECEIVE_ACK)
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNo_Ack;
    }

    /* Buffer the command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    if(bOption & PH_EXCHANGE_RXCHAINING)
    {
        /* Include LE if required and exchange the buffered information. */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            phhalHw_Sam_Cmd_DefaultLe,
            (uint8_t) ((bOption & PHHAL_HW_SAM_CMD_PKI_RECEIVE_ACK) ? PH_ON : PH_OFF),
            ppUpdateACK,
            pUpdateACKLen);

        /* Return the chaining code. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
        }
    }
    else
    {
        bPayloadLen = (uint8_t) (bCmdLen - PHHAL_HW_SAM_ISO7816_HEADER_LENGTH);
        wBufPos = 0;
        bSendHeader = (uint8_t) PH_OFF;

        do
        {
            if(bSendHeader == (uint8_t) PH_ON)
            {
                /* Update the header information. */
                PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
                PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
                PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_LC_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

                /* Buffer the command information to exchange buffer. */
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                    pDataParams,
                    PH_EXCHANGE_BUFFER_FIRST,
                    PHHAL_HW_SAM_CMD_BUFFER,
                    PHHAL_HW_SAM_ISO7816_HEADER_LENGTH,
                    NULL,
                    NULL));

                bSendHeader = (uint8_t) PH_OFF;
            }

            if((uint16_t) (wKeyFrameLen - wBufPos) > (uint16_t) (PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK - bPayloadLen))
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateP2(pDataParams, PHHAL_HW_SAM_ISO7816_CHAINED_FRAME));

                /* Buffer the next part of payload to exchange buffer. */
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                    pDataParams,
                    PH_EXCHANGE_BUFFER_CONT,
                    pKeyFrame + wBufPos,
                    (uint16_t) (PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK - bPayloadLen),
                    NULL,
                    NULL));

                /* Update LC */
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

                /* Exchange last frame to Sam hardware. */
                wStatus = phhalHw_Sam_Cmd_7816Exchange(
                    pDataParams,
                    PH_EXCHANGE_BUFFER_LAST,
                    NULL,
                    0,
                    ppUpdateACK,
                    pUpdateACKLen);

                /* Check for Success chaining response. */
                if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
                    return wStatus;

                bSendHeader = PH_ON;
                wBufPos = (uint16_t) (wBufPos + (PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK - bPayloadLen));
                bPayloadLen = 0;
            }
            else
            {
                /* Buffer the next part of payload to exchange buffer. */
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                    pDataParams,
                    PH_EXCHANGE_BUFFER_CONT,
                    pKeyFrame + wBufPos,
                    (uint16_t) (wKeyFrameLen - wBufPos),
                    NULL,
                    NULL));

                /* Update LC */
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

                /* Include LE if required and exchange the buffered information. */
                wStatus = phhalHw_Sam_Cmd_7816Exchange(
                    pDataParams,
                    PH_EXCHANGE_BUFFER_LAST,
                    phhalHw_Sam_Cmd_DefaultLe,
                    (uint8_t) ((bOption & PHHAL_HW_SAM_CMD_PKI_RECEIVE_ACK) ? PH_ON : PH_OFF),
                    ppUpdateACK,
                    pUpdateACKLen);

                wBufPos = wKeyFrameLen;

                /* Check for the Chaining active */
                if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
                {
                    return wStatus;
                }
                else
                {
                    return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
                }
            }

        } while(wBufPos < wKeyFrameLen);

        /* Reset Authentication states. */
        if(pDataParams->bKeyNo == bPKI_KeyNo_Enc)
        {
            /* Perform Kill Authentication. */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_SAM_KillAuthentication(pDataParams, PHHAL_HW_SAM_CMD_SAM_KILL_AUTHENTICATION_FULL));

            /* Invalidate the session keys and set the SM to PLAIN. */
            pDataParams->bAuthType = PHHAL_HW_SAM_CMD_SAM_AUTHENTICATE_HOST_MODE_PLAIN;
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_InvalidateKey(pDataParams->pENCCryptoDataParams));
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_InvalidateKey(pDataParams->pMACCryptoDataParams));
        }
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_PKI_EncipherKeyEntries(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t bNoOfKeyEntries,
    uint8_t bHashingAlg, uint8_t bPKI_KeyNo_Enc, uint8_t bPKI_KeyNo_Sign, uint8_t bPKI_KeyNo_Dec, uint8_t bPKI_KeyNo_Verif,
    uint16_t wPerso_Ctr, uint8_t * pKeyEntries, uint8_t bKeyEntriesLen, uint8_t * pDivInput, uint8_t bDivInputLen,
    uint8_t ** ppEncKeyFrame_Sign, uint16_t * pEncKeyFrame_Sign_Len)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pKeyEntries, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pEncKeyFrame_Sign_Len, PH_COMP_HAL);
    if((wOption & PHHAL_HW_SAM_CMD_PKI_DIVERSIFICATION_ON))
        PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame the command for PKI_EncipherKeyEntries command. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_ENCIPHER_KEY_ENTRIES;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;

    /* Add the key information and perso counter if its first frame. */
    if((wOption & PH_EXCHANGE_DEFAULT) == PH_EXCHANGE_DEFAULT)
    {
        /* Update P1 information byte. */
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = (uint8_t) (wOption & PHHAL_HW_SAM_CMD_PKI_DIVERSIFICATION_ON);
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] |= (uint8_t) ((bNoOfKeyEntries << 2U) | bHashingAlg);

        /* Add Default LC to command buffer. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Add key information to command buffer. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNo_Enc;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNo_Sign;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNo_Dec;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNo_Verif;

        /* Add perso counter to command buffer. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wPerso_Ctr);
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wPerso_Ctr);
    }

    /* Buffer the command to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Add key entries and diversification input to exchange buffer if its first frame. */
    if((wOption & PH_EXCHANGE_DEFAULT) == PH_EXCHANGE_DEFAULT)
    {
        /* Add key entries to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pKeyEntries,
            bKeyEntriesLen,
            NULL,
            NULL));

        /* Add diversification input to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pDivInput,
            bDivInputLen,
            NULL,
            NULL));

        /* Update LC */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));
    }

    /* Add Le byte to exchange buffer and perform the final exchange. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppEncKeyFrame_Sign,
        pEncKeyFrame_Sign_Len);

    /* Update the proper status to be returned in case of chaining. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_PKI_GenerateHash(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t bHashingAlg,
    uint32_t dwMLen, uint8_t * pMessage, uint16_t wMsgLen, uint8_t ** ppHash, uint16_t * pHashLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pMessage, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pHashLen, PH_COMP_HAL);

    /* Reset the command buffer and its length variable. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    if(!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        /* Frame the PKI_GenerateHash command. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_GENERATE_HASH;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Update P1 information byte with hashing algorithm and Message Length to
         * command buffer if its first frame.
         */
        if(wOption & PHHAL_HW_SAM_CMD_PKI_FIRST_FRAME)
        {
            /* Update P1 information byte with hashing algorithm. */
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = bHashingAlg;

            /* Add message length to command buffer. */
            PHHAL_HW_SAM_UINT_BYTE(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, dwMLen);
        }

        /* Update P2 information byte with chaining frame. */
        if(wOption & PH_EXCHANGE_TXCHAINING)
            PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;

        /* Buffer the command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            NULL,
            NULL));
    }

    /* Buffer the message information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pMessage,
        wMsgLen,
        NULL,
        NULL));

    /* Update LC information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Check if the buffered information in exchange buffer should be exchanged with Sam hardware. */
    if(!(wOption & PH_EXCHANGE_BUFFERED_BIT))
    {
        /* Add LE byte to exchange buffer if its the final frame. */
        if(!(wOption & PH_EXCHANGE_TXCHAINING))
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_CONT,
                phhalHw_Sam_Cmd_DefaultLe,
                PHHAL_HW_CMD_SAM_LE_LENGTH,
                NULL,
                NULL));
        }

        /* Exchange the buffered information to Sam hardware. */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            NULL,
            0,
            ppHash,
            pHashLen);

         /* Return Chaining Status. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
        }
    }
    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_PKI_GenerateSignature(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bHashingAlg,
    uint8_t bPKI_KeyNo_Sign, uint8_t * pHash, uint8_t bHashLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pHash, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame the PKI_GenerateSignature command. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_GENERATE_SIGNATURE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bHashingAlg;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add Key number to command frame */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNo_Sign;

    /* Add Has information to command frame */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pHash, bHashLen);
    bCmdLen += bHashLen;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_OFF);

    /* Exchange PKI_GenerateSignature information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_PKI_SendSignature(phhalHw_Sam_DataParams_t * pDataParams, uint8_t ** ppSignature, uint16_t * pSignatureLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSignatureLen, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame then PKI_SendSignature command. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_SEND_SIGNATURE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Exchange PKI_SendSignature information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppSignature,
        pSignatureLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_PKI_VerifySignature(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bPKI_KeyNo_Verif, uint8_t bHashingAlg,
    uint8_t * pHash, uint8_t bHashLen, uint8_t * pSignature, uint16_t wSignatureLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;
    uint8_t    PH_MEMLOC_REM bBufPos = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pHash, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSignature, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame the PKI_VerifySignature command. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_VERIFY_SIGNATURE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bHashingAlg;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (1U /* bPKI_KeyNo_Verif */ + bHashLen + wSignatureLen);

    /* Add Key number to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNo_Verif;

    /* Update P2 information and LC byte with chaining status and length. */
    if((1U /* bPKI_KeyNo_Verif */ + bHashLen + wSignatureLen) > PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK)
    {
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_LC_POS] = PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK;
    }

    /* Buffer the command information to command buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer the Hash information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pHash,
        bHashLen,
        NULL,
        NULL));

    /* Buffer Signature information to exchange buffer and exchange the buffered information to Sam hardware. */
    if(PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] == PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE)
    {
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            pSignature,
            wSignatureLen,
            &pResponse,
            &wRespLen);
    }
    else
    {
        /* Update the position for exchanging next chunk of signature information. */
        bBufPos = (uint8_t) (PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK - 1U - bHashLen);

        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            pSignature,
            bBufPos,
            &pResponse,
            &wRespLen);

        /* Return the status if chaining response is not received from Sam hardware. */
        if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
            return wStatus;

        /* Update the P1 information byte to default value and P2 information byte Last frame.  */
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P1_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_LC_POS] = (uint8_t) (wSignatureLen - bBufPos);

        /* Exchange second part with lower layer */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            PHHAL_HW_SAM_ISO7816_HEADER_LENGTH,
            NULL,
            NULL));

        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            pSignature + bBufPos,
            (uint8_t) (wSignatureLen - bBufPos),
            &pResponse,
            &wRespLen);
    }

    return wStatus;
}

phStatus_t phhalHw_Sam_Cmd_PKI_EncipherData(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bHashingAlg, uint8_t bPKI_KeyNo_Enc,
    uint8_t * pPlainData, uint8_t bPlainDataLen, uint8_t ** ppEncData, uint16_t * pEncDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPlainData, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pEncDataLen, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame PKI_EncipherData command. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_ENCIPHER_DATA;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bHashingAlg;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (1U /* bPKI_KeyNo_Enc */ + bPlainDataLen);

    /* Add Key number to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNo_Enc;

    /* Buffer the command information to command buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer the Plain Data to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pPlainData,
        bPlainDataLen,
        NULL,
        NULL));

    /* Add LE byte to exchange buffer and exchange the information with Sam hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppEncData,
        pEncDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_PKI_DecipherData(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t bHashingAlg,
    uint8_t bPKI_KeyNo_Dec, uint8_t * pEncData, uint16_t wEncDataLen, uint8_t ** ppPlainData, uint16_t * pPlainDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pEncData, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pPlainDataLen, PH_COMP_HAL);

    /* Frame PKI_DecipherData command. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_DECIPHER_DATA;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bHashingAlg;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add key number for the first frame only. */
    if((wOption & PH_EXCHANGE_CUSTOM_BITS_MASK) == PHHAL_HW_SAM_CMD_PKI_FIRST_FRAME)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPKI_KeyNo_Dec;
    }

    /* Set P2 information with chaining flag. */
    if((wOption & PH_EXCHANGE_MODE_MASK) == PH_EXCHANGE_TXCHAINING)
    {
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_P2_POS] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
    }

    /* Buffer the command information to command buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer the Encrypted Data to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pEncData,
        wEncDataLen,
        NULL,
        NULL));

    /* Update LC information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Add LE byte to exchange buffer and exchange the information with Sam hardware. */
    wStatus = phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        PHHAL_HW_CMD_SAM_LE_LENGTH,
        ppPlainData,
        pPlainDataLen);

    /* Update the proper status to be returned in case of chaining. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_HAL);
    }

    return wStatus;
}





/* ECC Commands ---------------------------------------------------------------------------------------------------------- */
phStatus_t phhalHw_Sam_Cmd_PKI_GenerateEccKey(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bECC_KeyNo, uint16_t wECC_Set,
    uint8_t bECC_KeyNoCEK, uint8_t bECC_KeyVCEK, uint8_t bECC_RefNoKUC, uint8_t bECC_KeyNoAEK, uint8_t bECC_KeyVAEK,
    uint8_t bECC_RefNoCurve, uint8_t ** ppECC_xy, uint16_t * pECC_xyLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_PARAM(pECC_xyLen, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame PKI_GenerateEccKey command. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_GENERATE_ECC_KEY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Add key information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wECC_Set);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wECC_Set);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyNoCEK;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyVCEK;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_RefNoKUC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyNoAEK;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyVAEK;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_RefNoCurve;

    /* Exchange the command information to Sam hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppECC_xy,
        pECC_xyLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_PKI_ImportEccKey(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bECC_KeyNo, uint16_t wECC_Set,
    uint8_t bECC_KeyNoCEK, uint8_t bECC_KeyVCEK, uint8_t bECC_RefNoKUC, uint8_t bECC_KeyNoAEK, uint8_t bECC_KeyVAEK, uint16_t wECC_Len,
    uint8_t * pECC_KeyValue, uint8_t bECC_KeyValueLen, uint8_t bECC_RefNoCurve, uint8_t bCRLFile, uint8_t * pIssuer, uint8_t bIssuerLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;
    uint8_t    PH_MEMLOC_REM bLC = 0;
    uint8_t    PH_MEMLOC_REM bRem = 0;
    uint8_t    PH_MEMLOC_REM bOffset  = 0;
    uint8_t    PH_MEMLOC_REM bHasMoreBytes = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    if(bOption == PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_VALUE)
    {
        PH_ASSERT_NULL_PARAM(pECC_KeyValue, PH_COMP_HAL);
    }

    /* Reset the command buffer and its length variable. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame PKI_ImportEccKey command. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_IMPORT_ECC_KEY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_PKI_IMPORT_ECC_KEY_LC_MIN;

    /* Add key information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_MSB(wECC_Set);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_USHORT_MASK_LSB(wECC_Set);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyNoCEK;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyVCEK;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_RefNoKUC;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyNoAEK;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyVAEK;

    /* Check if only key settings should be exchanged to Sam hardware . */
    if(bOption & PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_ONLY)
    {
        /* Exchange the command information to Sam hardware. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            &pResponse,
            &wRespLen));
    }

    /* Add ECC_Len and ECC_xy or ECC_priv to exchange buffer if required. */
    else
    {
        /* Add ECC_Len to command buffer. */
        PHHAL_HW_SAM_USHORT_BYTE(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, wECC_Len);

        /* Add the command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            NULL,
            NULL));

        /* Add ECC_xy or ECC_priv information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pECC_KeyValue,
            bECC_KeyValueLen,
            NULL,
            NULL));

        /* Add ECC_RefNoCurve to command buffer */
        if((wECC_Set & PHHAL_HW_SAM_CMD_PKI_REFNO_CURVE_PRESENT) != 0)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_CONT,
                &bECC_RefNoCurve,
                1U,
                NULL,
                NULL));
        }

        /* Add CRLFile to command buffer */
        if((wECC_Set & PHHAL_HW_SAM_CMD_PKI_CRL_FILE_ID_PRESENT) != 0)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_CONT,
                &bCRLFile,
                1U,
                NULL,
                NULL));
        }

        /* Add Issuer */
        if((wECC_Set & PHHAL_HW_SAM_CMD_PKI_ISSUER_PRESENT) != 0)
        {
            /* Add Issuer Length */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_CONT,
                &bIssuerLen,
                1U,
                NULL,
                NULL));

            /* Update LC */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

            /* Get the length current payload added to SAM frame */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_GetLc(pDataParams, &bLC));

            /* Set Issuer information as remaining one to be exchanged. */
            bRem = bIssuerLen;

            /* Check if More bytes can be added in SAM exchange frame. */
            bHasMoreBytes = (uint8_t) ((bLC + bIssuerLen) > PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK);

            /* Calculate the Remaining bytes to be added in SAM command frame */
            if(bHasMoreBytes != 0)
            {
                bRem = (uint8_t) (PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK - bLC);

                /* Update P2 to chaining flag */
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateP2(pDataParams, PHHAL_HW_SAM_ISO7816_CHAINED_FRAME));
            }

            /* Add Initial Issuer information to command buffer */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_CONT,
                pIssuer,
                bRem,
                NULL,
                NULL));

            /* Calculate Next Offset and length to be exchanged */
            bOffset = bRem;
            bRem = (uint8_t) (bIssuerLen - bRem);
        }

        /* Update LC */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

        /* Exchange the buffered information to Sam hardware. */
        wStatus = phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            NULL,
            0,
            &pResponse,
            &wRespLen);

        /* Validate Status */
        if(bHasMoreBytes != 0)
        {
            if((wStatus & PH_ERR_MASK) != PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE)
            {
                return wStatus;
            }
        }
        else
        {
            PH_CHECK_SUCCESS(wStatus);
        }

        /* Exchange the remaining Issuer information */
        if(((wECC_Set & PHHAL_HW_SAM_CMD_PKI_ISSUER_PRESENT) != 0) &&
            (bHasMoreBytes != 0))
        {
            /* Reset the command buffer and its length variable. */
            PHHAL_HW_SAM_CLEAR_CMD_BUFFER();
            bCmdLen = 0;

            /* Frame PKI_ImportEccKey command. */
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_IMPORT_ECC_KEY;
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_LAST_FRAME;
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bRem;

            /* Add the command information to exchange buffer. */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_FIRST,
                PHHAL_HW_SAM_CMD_BUFFER,
                bCmdLen,
                NULL,
                NULL));

            /* Add Final Issuer to command buffer */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
                pDataParams,
                PH_EXCHANGE_BUFFER_LAST,
                &pIssuer[bOffset],
                bRem,
                &pResponse,
                &wRespLen));
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_PKI_ImportEccCurve(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption, uint8_t bECC_CurveNo,
    uint8_t bECC_KeyNoCCK, uint8_t bECC_KeyVCCK, uint8_t bECC_N, uint8_t bECC_M, uint8_t * pECC_Prime, uint8_t * pECC_ParamA,
    uint8_t * pECC_ParamB, uint8_t * pECC_Px, uint8_t * pECC_Py, uint8_t * pECC_Order)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    if(bOption == PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_VALUE)
    {
        PH_ASSERT_NULL_PARAM(pECC_Prime, PH_COMP_HAL);
        PH_ASSERT_NULL_PARAM(pECC_ParamA, PH_COMP_HAL);
        PH_ASSERT_NULL_PARAM(pECC_ParamB, PH_COMP_HAL);
        PH_ASSERT_NULL_PARAM(pECC_Px, PH_COMP_HAL);
        PH_ASSERT_NULL_PARAM(pECC_Py, PH_COMP_HAL);
        PH_ASSERT_NULL_PARAM(pECC_Order, PH_COMP_HAL);
    }

    /* Reset the command buffer and its length variable. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame PKI_ImportEccCurve command. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_IMPORT_ECC_CURVE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_PKI_IMPORT_ECC_CURVE_LC_MIN;

    /* Add key information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_CurveNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyNoCCK;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyVCCK;

    /* Check if only key settings should be exchanged to Sam hardware . */
    if(bOption & PHHAL_HW_SAM_CMD_PKI_UPDATE_SETTINGS_ONLY)
    {
        /* Exchange the command information to Sam hardware. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            &pResponse,
            &wRespLen));
    }
    /* Add Curve values to exchange buffer if required. */
    else
    {
        /* Update LC */
        PHHAL_HW_SAM_CMD_BUFFER[PHHAL_HW_SAM_ISO7816_LC_POS] += (uint8_t) (2U /* ECC_N, ECC_M */ +
            (5U /* ECC_Prime, ECC_ParamA, ECC_ParamB, ECC_Px, ECC_Py, ECC_Order */ * bECC_N) + bECC_M);

        /* Add ECC_Len to command buffer. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_N;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_M;

        /* Add the command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            NULL,
            NULL));

        /* Add the ECC_Prime information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pECC_Prime,
            bECC_N,
            NULL,
            NULL));

        /* Add the ECC_ParamA information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pECC_ParamA,
            bECC_N,
            NULL,
            NULL));

        /* Add the ECC_ParamB information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pECC_ParamB,
            bECC_N,
            NULL,
            NULL));

        /* Add the ECC_Px information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pECC_Px,
            bECC_N,
            NULL,
            NULL));

        /* Add the ECC_Py information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pECC_Py,
            bECC_N,
            NULL,
            NULL));

        /* Add ECC_Order to exchange buffer and exchange the buffered information to Sam hardware. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            pECC_Order,
            bECC_M,
            &pResponse,
            &wRespLen));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_PKI_ExportEccPrivateKey(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bECC_KeyNo, uint16_t * pECC_Set,
    uint8_t * pECC_KeyNoCEK, uint8_t * pECC_KeyVCEK, uint8_t * pECC_RefNoKUC, uint8_t * pECC_KeyNoAEK, uint8_t * pECC_KeyVAEK,
    uint16_t * pECC_Len, uint8_t * pECC_RefNoCurve, uint8_t ** ppECC_Priv, uint8_t * pECC_PrivLen )
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;
    uint8_t    PH_MEMLOC_REM bOffset = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_PARAM(pECC_Set, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pECC_KeyNoCEK, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pECC_KeyVCEK, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pECC_RefNoKUC, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pECC_KeyNoAEK, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pECC_KeyVAEK, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pECC_Len, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pECC_RefNoCurve, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pECC_PrivLen, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame PKI_ExportEccPrivateKey command. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_EXPORT_ECC_PRIVATE_KEY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Exchange the command information to Sam hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    /* Extract the information from response. */
    *pECC_Set = (uint16_t) (pResponse[bOffset++] << 8U);
    *pECC_Set |= (uint16_t) pResponse[bOffset++];

    *pECC_KeyNoCEK = pResponse[bOffset++];
    *pECC_KeyVCEK = pResponse[bOffset++];

    *pECC_RefNoKUC = pResponse[bOffset++];

    *pECC_KeyNoAEK = pResponse[bOffset++];
    *pECC_KeyVAEK = pResponse[bOffset++];

    *pECC_Len = (uint16_t) (pResponse[bOffset++] << 8U);
    *pECC_Len |= (uint16_t) pResponse[bOffset++];

    *ppECC_Priv = &pResponse[bOffset];
    *pECC_PrivLen = (uint8_t) ((wRespLen - 1U) - bOffset);

    *pECC_RefNoCurve = pResponse[(wRespLen - 1U)];

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_PKI_ExportEccPublicKey(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bECC_KeyNo, uint16_t * pECC_Set,
    uint8_t * pECC_KeyNoCEK, uint8_t * pECC_KeyVCEK, uint8_t * pECC_RefNoKUC, uint8_t * pECC_KeyNoAEK, uint8_t * pECC_KeyVAEK,
    uint16_t * pECC_Len, uint8_t ** ppECC_xy, uint8_t * pECC_xyLen, uint8_t * pCRLFile)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;
    uint8_t    PH_MEMLOC_REM bOffset = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_PARAM(pECC_Set, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pECC_KeyNoCEK, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pECC_KeyVCEK, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pECC_RefNoKUC, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pECC_KeyNoAEK, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pECC_KeyVAEK, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pECC_Len, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(ppECC_xy, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pECC_xyLen, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame PKI_ExportEccPublicKey command. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_EXPORT_ECC_PUBLIC_KEY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Exchange the command information to Sam hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    /* Extract the information from response. */
    *pECC_Set = (uint16_t) (pResponse[bOffset++] << 8U);
    *pECC_Set |= (uint16_t) pResponse[bOffset++];

    *pECC_KeyNoCEK = pResponse[bOffset++];
    *pECC_KeyVCEK = pResponse[bOffset++];

    *pECC_RefNoKUC = pResponse[bOffset++];

    *pECC_KeyNoAEK = pResponse[bOffset++];
    *pECC_KeyVAEK = pResponse[bOffset++];

    *pECC_Len = (uint16_t) (pResponse[bOffset++] << 8U);
    *pECC_Len |= (uint16_t) pResponse[bOffset++];

    *ppECC_xy = &pResponse[bOffset];
    *pECC_xyLen = (uint8_t) (wRespLen - bOffset);

    /* Check if CRLFile is available and extract the information */
    *pCRLFile = 0xFFU;
    if((*pECC_Set & PHHAL_HW_SAM_CMD_PKI_CRL_FILE_ID_PRESENT) != 0)
    {
        *pCRLFile = pResponse[(wRespLen - 1U)];
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_PKI_GenerateEccSignature(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bHashingAlg, uint8_t bECC_KeyNo_Sign,
    uint8_t * pHash, uint8_t bHashLen, uint8_t ** ppSignature, uint16_t * pSigLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_PARAM(pHash, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSigLen, PH_COMP_HAL);

    /* Reset the command buffer and its length variable. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame PKI_GenerateEccSignature command. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_GENERATE_ECC_SIGNATURE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bHashingAlg;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (1U /* ECC_KeyNo_Sign */ + bHashLen);

    /* Add ECC_Key to command frame */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyNo_Sign;

    /* Add command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Add Hash to exchange buffer and exchange the buffered information to Sam hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pHash,
        bHashLen,
        ppSignature,
        pSigLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_PKI_VerifyEccSignature(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bECC_KeyNo, uint8_t bECC_CurveNo,
    uint8_t bLen, uint8_t * pMessage, uint8_t * pSignature, uint16_t wSignatureLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    if(bLen != 0)
        PH_ASSERT_NULL_PARAM(pMessage, PH_COMP_HAL);

    if(wSignatureLen != 0)
        PH_ASSERT_NULL_PARAM(pSignature, PH_COMP_HAL);

    /* Reset the command buffer and its length variable. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame PKI_VerifyEccSignature command. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_VERIFY_ECC_SIGNATURE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (PHHAL_HW_SAM_CMD_PKI_VERIFY_ECC_SIGNATURE_LC_MIN +
        bLen + wSignatureLen);

    /* Add the key and length information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_CurveNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bLen;

    /* Add command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Add message to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pMessage,
        bLen,
        NULL,
        NULL));

    /* Add signature to exchange buffer and exchange the buffered information to Sam hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pSignature,
        wSignatureLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_PKI_ValidateEccCert(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bCertFormat, uint8_t bECC_KeyNo,
    uint8_t bECC_CurveNo, uint8_t bECC_Target, uint8_t * pCertificate, uint16_t wCertLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint16_t   PH_MEMLOC_REM wRem = 0;
    uint16_t   PH_MEMLOC_REM wOffset = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;
    uint8_t    PH_MEMLOC_REM bIsFirstFrame = 1U;
    uint8_t    PH_MEMLOC_REM bFinished = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    PH_ASSERT_NULL_PARAM(pCertificate, PH_COMP_HAL);

    do
    {
        /* Reset the command buffer and its length variable. */
        PHHAL_HW_SAM_CLEAR_CMD_BUFFER();
        bCmdLen = 0;

        /* Frame PKI_ValidateEccCert command. */
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_PKI_VALIDATE_ECC_CERTIFICATE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CHAINED_FRAME;
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

        /* Add information to command buffer. */
        if(bIsFirstFrame == 1U)
        {
            /* Update P1 byte to certificate format */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateP1(pDataParams, bCertFormat));

            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_KeyNo;
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_CurveNo;
            PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bECC_Target;

            /* Clear the flag */
            bIsFirstFrame = 0;

            /* Calculate the Remaining bytes to be added in SAM command frame */
            wRem = (uint16_t) (wCertLen - 3U);
        }

        /* Update P2 information byte */
        if(wRem <= PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateP2(pDataParams,
                PHHAL_HW_SAM_ISO7816_LAST_FRAME));

            bFinished = 1U;
        }
        else
        {
            wRem = (uint16_t) (PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK -
                ((bIsFirstFrame == 1U) ? 3U : 0));
        }

        /* Add command information to exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHHAL_HW_SAM_CMD_BUFFER,
            bCmdLen,
            NULL,
            NULL));

        /* Update LC */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

        /* Add Certificate to command buffer */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            &pCertificate[wOffset],
            wRem,
            NULL,
            NULL));

        /* Add Certificate to command buffer */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            NULL,
            0,
            &pResponse,
            &wRespLen));

        /* Calculate Next Offset and length to be exchanged */
        wOffset = wRem;
        wRem = (uint16_t) (wCertLen - wRem);

    } while(bFinished != 0);

    return wStatus;
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
