/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Key Management command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */


#ifndef PHHALHW_SAM_CMD_KM_H
#define PHHALHW_SAM_CMD_KM_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_KM 03_KeyManagement
 * \brief Implementation of SAM Key Management commands
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_KM_Defines Defines
 * \brief Definitions for SAM Key Management commands.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_KM_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_SAM_CHANGE_KEY_ENTRY                               0xC1U   /**< Sam Instruction code for SAM_ChangeKeyEntry command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_CHANGE_KUC_ENTRY                               0xCCU   /**< Sam Instruction code for SAM_ChangeKUCEntry command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_DISABLE_KEY_ENTRY                              0xD8U   /**< Sam Instruction code for SAM_DisableKeyEntry command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_ENCHIPHER_KEY_ENTRY                            0xE1U   /**< Sam Instruction code for SAM_EnchipherKeyEntry command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_GET_KEY_ENTRY                                  0x64U   /**< Sam Instruction code for SAM_GetKeyEntry command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_GET_KUC_ENTRY                                  0x6CU   /**< Sam Instruction code for SAM_GetKUCEntry command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_DUMP_SESSION_KEY                               0xD5U   /**< Sam Instruction code for SAM_DumpSessionKey command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_DUMP_SECRET_KEY                                0xD6U   /**< Sam Instruction code for SAM_DumpSecretKey command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_DERIVE_KEY                                     0xD7U   /**< Sam Instruction code for SAM_DeriveKey command. */
/**
 * end of group phhalHw_Sam_Cmd_KM_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_HC_Defines_NVM ProMas
 * \brief Definitions for Non-volatile programming mask. To be used with \ref phhalHw_Sam_Cmd_SAM_ChangeKeyEntry
 * "SAM_ChangeKeyEntry" and \ref phhalHw_Sam_Cmd_SAM_ChangeKeyEntryOffline "SAM_ChangeKeyEntryOffline" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_KEY_VA                     0x80U   /**< Option mask for updating key number and version A. */
#define PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_KEY_VB                     0x40U   /**< Option mask for updating key number and version B. */
#define PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_KEY_VC                     0x20U   /**< Option mask for updating key number and version C. */
#define PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_DF_AID                     0x10U   /**< Option mask for updating DESFire application identifier. */
#define PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_KEY_CEK                    0x08U   /**< Option mask for updating key number and version of change entry key (CEK). */
#define PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_REF_NO_KUC                 0x04U   /**< Option mask for updating reference key usage counter. */
#define PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_SET_EXTSET                 0x02U   /**< Option mask for updating SET and Extended SET. */
#define PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_INCLUDE_VERSION                   0x01U   /**< Option mask for specifying the versions in the data field after SET and ExtSET. */
/**
 * end of group phhalHw_Sam_Cmd_HC_Defines_NVM
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_HC_Defines_KUC KUC_ProMas
 * \brief Definitions for Non-volatile programming mask. To be used with \ref phhalHw_Sam_Cmd_SAM_ChangeKUCEntry
 * "SAM_ChangeKUCEntry" and \ref phhalHw_Sam_Cmd_SAM_ChangeKUCEntryOffline "SAM_ChangeKUCEntryOffline" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_SAM_CHANGE_KUC_ENTRY_UPDATE_LIMIT                      0x80U   /**< Option mask for updating the limit. */
#define PHHAL_HW_SAM_CMD_SAM_CHANGE_KUC_ENTRY_UPDATE_KEY_NO_CKUC                0x40U   /**< Option mask for updating CKUC key number. */
#define PHHAL_HW_SAM_CMD_SAM_CHANGE_KUC_ENTRY_UPDATE_VCKUC                      0x20U   /**< Option mask for updating the VCKUC. */
/**
 * end of group phhalHw_Sam_Cmd_HC_Defines_KUC
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_HC_Defines_EncipherKeyEntry EncipherKeyEntry
 * \brief Definitions for Key diversification programming mask. To be used with \ref phhalHw_Sam_Cmd_SAM_EncipherKeyEntry
 * "SAM_EncipherKeyEntry" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_SAM_ENCIPHER_KEY_ENTRY_USE_DIVERSIFICATION             0x01U   /**< Option mask to include the diversification input in command frame. */
#define PHHAL_HW_SAM_CMD_SAM_ENCIPHER_KEY_ENTRY_INCLUDE_SAMUID                  0x02U   /**< Option mask to include the Sam UID in command frame. */
/**
 * end of group phhalHw_Sam_Cmd_HC_Defines_EncipherKeyEntry
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_HC_Defines_DumpMode Dump Mode
 * \brief Definitions for Key diversification programming mask. To be used with \ref phhalHw_Sam_Cmd_SAM_DumpSessionKey
 * "SAM_DumpSessionKey" and \ref phhalHw_Sam_Cmd_SAM_DumpSecretKey "SAM_DumpSecretKey" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_SAM_DUMP_MODE_PLAIN                                    0x00    /**< Option mask for dumping session or secret keys in plain. */
#define PHHAL_HW_SAM_CMD_SAM_DUMP_MODE_ENCIPHERED                               0x01U   /**< Option mask for dumping session or secret keys in enciphered. */
#define PHHAL_HW_SAM_CMD_SAM_DUMP_MODE_DIVERSIFICATION                          0x02U   /**< Option mask for enabling the diversification of the provided key. */
/**
 * end of group phhalHw_Sam_Cmd_HC_Defines_DumpMode
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_KM_Defines
 * @}
 */

/**
 * \brief Change a symmetric key entry in the key table of the SAM.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pKeyData is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_ChangeKeyEntry(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bKeyNo,                                                                 /**< [In] Reference number of the key entry to be changed.
                                                                                         *        One of the following,
                                                                                         *          - NVM Keys: 0x00 - 0x7F
                                                                                         *          - RAM Keys: 0xE0 - 0xE3
                                                                                         */
        uint8_t bProMas,                                                                /**< [In] Program mask indicating the fields that should be changed. All the below
                                                                                         *        option can be combined by using bitwise OR operator.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_KEY_VA
                                                                                         *                 "Update KeyVa"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_KEY_VB
                                                                                         *                 "Update KeyVb" (shall be '0' if RAMKey, P1 >= 0xE0)
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_KEY_VC
                                                                                         *                 "Update KeyVc" (shall be '0' if RAMKey, P1 >= 0xE0)
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_DF_AID
                                                                                         *                 "Update DF_AID", DF_KeyNo (shall be '0' if RAMKey,
                                                                                         *                  P1 >= 0xE0)
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_KEY_CEK
                                                                                         *                 "Update KeyNoCEK, KeyVCEK"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_REF_NO_KUC
                                                                                         *                 "Update RefNoKUC"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_SET_EXTSET
                                                                                         *                 "Update SET, ExtSET and, if present, KeyNoAEK, KeyVAEK"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_INCLUDE_VERSION
                                                                                         *                 "Key versions specified separately"
                                                                                         */
        uint8_t * pKeyData,                                                             /**< [In] Buffer containing the key data information to be updated. */
        uint8_t bKeyDataLen                                                             /**< [In] Length of bytes available in \b pKeyData buffer. */
    );

/**
 * \brief Change a Offline symmetric key entry in the key entry table of the SAM.
 *  - Selected by a change key (KeyNoCEK) of OfflineChange Key Class is restricting the key update to an active
 *    Offline Key Activation.
 *  - The command and its payload are protected by the Offline change protection.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pOfflineCrypto and \b pOfflineAck is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_ChangeKeyEntryOffline(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bKeyNo,                                                                 /**< [In] Reference number of the key entry to be changed.
                                                                                         *        One of the following,
                                                                                         *          - NVM Keys: 0x00 - 0x7F
                                                                                         *          - RAM Keys: 0xE0 - 0xE3
                                                                                         */
        uint8_t bProMas,                                                                /**< [In] Program mask indicating the fields that should be changed. All the below
                                                                                         *        option can be combined by using bitwise OR operator.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_KEY_VA
                                                                                         *                 "Update KeyVa"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_KEY_VB
                                                                                         *                 "Update KeyVb" (shall be '0' if RAMKey, P1 >= 0xE0)
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_KEY_VC
                                                                                         *                 "Update KeyVc" (shall be '0' if RAMKey, P1 >= 0xE0)
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_DF_AID
                                                                                         *                 "Update DF_AID"; DF_KeyNo (shall be '0' if RAMKey,
                                                                                         *                  P1 >= 0xE0)
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_KEY_CEK
                                                                                         *                 "Update KeyNoCEK; KeyVCEK"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_REF_NO_KUC
                                                                                         *                 "Update RefNoKUC"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_UPDATE_SET_EXTSET
                                                                                         *                 "Update SET, ExtSET and, if present, KeyNoAEK; KeyVAEK"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KEY_ENTRY_INCLUDE_VERSION
                                                                                         *                 "Key versions specified separately"
                                                                                         */
        uint16_t wChangeCtr,                                                            /**< [In] Change Counter to avoid replay attacks. */
        uint8_t * pOfflineCrypto,                                                       /**< [In] Offline Cryptogram exchanged to SAM (EncKeyEntry + OfflineMAC)
                                                                                         *          - Encrypted key entry as E(Kce, KeyEntry [|| SAMUID])
                                                                                         *          - Integrity MAC protection as MACt (Kcm, CLA || INS || P1 ||
                                                                                         *            P2 || LC || Change_Ctr || E(Kce, KeyEntry [|| SAMUID]))
                                                                                         */
        uint8_t bOfflineCryptoLen,                                                      /**< [In] Length of bytes available in \b pOfflineCrypto buffer */
        uint8_t bEnableOfflineAck,                                                      /**< [In] To Enable reception of Offline Acknowledge
                                                                                         *          - \ref PH_OFF "Disable reception of Offline ACK"
                                                                                         *            (LE will not be exchanged)
                                                                                         *          - \ref PH_ON "Enable reception of Offline ACK"
                                                                                         *            (LE will be exchanged)
                                                                                         */
        uint8_t ** ppOfflineAck,                                                        /**< [Out] Offline Acknowledge information received from SAM. Will be based on
                                                                                         *           - If \b bEnableOfflineAck = \ref PH_OFF "Disable reception of Offline ACK",
                                                                                         *             buffer will contain NULL.
                                                                                         *           - If \b bEnableOfflineAck = \ref PH_ON "Enable reception of Offline ACK",
                                                                                         *             buffer will contain Offline change acknowledge as MACt (Kcm, 0x90 || 0x00
                                                                                         *             || INS || Change_Ctr || KeyNo || SAMUID).
                                                                                         *
                                                                                         */
        uint16_t * pOfflineAckLen                                                       /**< [Out] Length of bytes available in \b ppOfflineAck buffer */
    );

/**
 * \brief Change the key usage counter (KUC). Selection is done by its reference number.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pKucData is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_ChangeKUCEntry(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bKucNo,                                                                 /**< [In] Reference number of the key usage counter to be updated (00h to 0Fh). */
        uint8_t bProMas,                                                                /**< [In] Program mask indicating the fields that should be changed. All the below
                                                                                         *        option can be combined by using bitwise OR operator.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KUC_ENTRY_UPDATE_LIMIT
                                                                                         *                 "Update limit"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KUC_ENTRY_UPDATE_KEY_NO_CKUC
                                                                                         *                 "Update KeyNoCKUC"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KUC_ENTRY_UPDATE_VCKUC
                                                                                         *                 "Update KeyVCKUC"
                                                                                         */
        uint8_t * pKucData,                                                             /**< [In] Buffer containing the Key Usage Counter data. */
        uint8_t bKucDataLen                                                             /**< [In] Length of bytes available in \b pKucData buffer. */
    );

/**
 * \brief Change a Offline KUC entry in the key entry table of the SAM.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pOfflineCrypto and \b pOfflineAck is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_ChangeKUCEntryOffline(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bKucNo,                                                                 /**< [In] Reference number of the key usage counter to be updated (00h to 0Fh). */
        uint8_t bProMas,                                                                /**< [In] Program mask indicating the fields that should be changed. All the below
                                                                                         *        option can be combined by using bitwise OR operator.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_CHANGE_KUC_ENTRY_UPDATE_LIMIT
                                                                                         *                 "Update limit"
                                                                                         *          - \ref #PHHAL_HW_SAM_CMD_SAM_CHANGE_KUC_ENTRY_UPDATE_KEY_NO_CKUC
                                                                                         *                 "Update KeyNoCKUC"
                                                                                         *          - \ref #PHHAL_HW_SAM_CMD_SAM_CHANGE_KUC_ENTRY_UPDATE_VCKUC
                                                                                         *                 "Update KeyVCKUC"
                                                                                         */
        uint16_t wChangeCtr,                                                            /**< [In] Change Counter to avoid replay attacks. */
        uint8_t * pOfflineCrypto,                                                       /**< [In] Offline Cryptogram exchanged to SAM (EncKUCEntry + OfflineMAC)
                                                                                         *          - Encrypted key entry as E(Kce, KUCEntry [|| SAMUID])
                                                                                         *          - Integrity MAC protection as MACt (Kcm, CLA || INS || P1 ||
                                                                                         *            P2 || LC || Change_Ctr || E(Kce, KUCEntry [|| SAMUID]))
                                                                                         */
        uint8_t bOfflineCryptoLen,                                                      /**< [In] Length of bytes available in \b pOfflineCrypto buffer */
        uint8_t bEnableOfflineAck,                                                      /**< [In] To Enable reception of Offline Acknowledge
                                                                                         *          - \ref PH_OFF "Disable reception of Offline ACK"
                                                                                         *            (LE will not be exchanged)
                                                                                         *          - \ref PH_ON "Enable reception of Offline ACK"
                                                                                         *            (LE will be exchanged)
                                                                                         */
        uint8_t ** ppOfflineAck,                                                        /**< [Out] Offline Acknowledge information received from SAM. Will be based on
                                                                                         *           - If \b bEnableOfflineAck = \ref PH_OFF "Disable reception of Offline ACK",
                                                                                         *             buffer will contain NULL.
                                                                                         *           - If \b bEnableOfflineAck = \ref PH_ON "Enable reception of Offline ACK",
                                                                                         *             buffer will contain Offline change acknowledge as MACt (Kcm, 0x90 || 0x00
                                                                                         *             || INS || Change_Ctr || RefNo KUC || SAMUID).
                                                                                         *
                                                                                         */
        uint16_t * pOfflineAckLen                                                       /**< [Out] Length of bytes available in \b ppOfflineAck buffer */
    );

/**
 * \brief Disable a key entry.
 *  - The SAM_DisableKeyEntry is used to disable any key entry of the KST, hence enabling SETBit9. The plain data
 *    field is not present.
 *  - If there is an active authentication with a key of the targeted key entry, be it a Host Authentication (Logical
 *    Channel Global State), or any PICC Authentication or key activation (XMode State), this will be reset on any LC.
 *  - Note that even if the Host Authentication on the current LC is reset, the current command response will still be
 *    processed with secure messaging.
 *  - After executing this command, the corresponding disable flag in the key entry is set and the key entry cannot be
 *    used anymore for authentication and key change procedures.
 *  - The key entry can still be read by a SAM_GetKeyEntry command.
 *  - To reactivate the entry, a SAM_ChangeKeyEntry command has to be issued.
 *  - All fields in the key entry can still be changed by this command even if the entry has been disabled.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_DisableKeyEntry(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bKeyNo                                                                  /**< [In] Key reference number of the key entry to disable.
                                                                                         *        One of the following,
                                                                                         *          - NVM Keys: 0x00 - 0x7F
                                                                                         *          - RAM Keys: 0xE0 - 0xE3
                                                                                         */
    );

/**
* \brief Disable a key entry using offline cryptogram.
*
* \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pOfflineCrypto and \b pOfflineAck is NULL.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phhalHw_Sam_Cmd_SAM_DisableKeyEntryOffline(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bKeyNo,                                                                 /**< [In] Key reference number of the key entry to disable.
                                                                                         *        One of the following,
                                                                                         *          - NVM Keys: 0x00 - 0x7F
                                                                                         *          - RAM Keys: 0xE0 - 0xE3
                                                                                         */
        uint16_t wChangeCtr,                                                            /**< [In] Change Counter to avoid replay attacks. */
        uint8_t * pOfflineCrypto,                                                       /**< [In] Offline Cryptogram exchanged to SAM (EncKUCEntry + OfflineMAC)
                                                                                         *          - Encrypted GoldField as E(Kce, SAMUID)
                                                                                         *          - Integrity MAC protection as MACt (Kcm, CLA || INS || P1 ||
                                                                                         *            P2 || LC || Change_Ctr [|| E(Kce, SAMUID)])
                                                                                         */
        uint8_t bOfflineCryptoLen,                                                      /**< [In] Length of bytes available in \b pOfflineCrypto buffer */
        uint8_t bEnableOfflineAck,                                                      /**< [In] To Enable reception of Offline Acknowledge
                                                                                         *          - \ref PH_OFF "Disable reception of Offline ACK"
                                                                                         *            (LE will not be exchanged)
                                                                                         *          - \ref PH_ON "Enable reception of Offline ACK"
                                                                                         *            (LE will be exchanged)
                                                                                         */
        uint8_t ** ppOfflineAck,                                                        /**< [Out] Offline Acknowledge information received from SAM. Will be based on
                                                                                         *           - If \b bEnableOfflineAck = \ref PH_OFF "Disable reception of Offline ACK",
                                                                                         *             buffer will contain NULL.
                                                                                         *           - If \b bEnableOfflineAck = \ref PH_ON "Enable reception of Offline ACK",
                                                                                         *             buffer will contain Offline disable acknowledge as MACt (Kcm, 0x90 || 0x00
                                                                                         *             || INS || Change_Ctr || KeyNo || SAMUID).
                                                                                         *
                                                                                         */
        uint16_t * pOfflineAckLen                                                       /**< [Out] Length of bytes available in \b ppOfflineAck buffer */
    );

/**
 * \brief The SAM_EncipherKeyEntry is a Personalization SAM command used to prepare a cryptogram (according
 * to Offline change protection) for the OfflineChange key on a target SAM. The target Offline change counter
 * (Perso_Ctr) are included in the generated cryptogram. It is only relevant for the target SAM and it shall
 * not be compared to the (local) Change_Ctr.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pDivInput and \b pOfflineCryptoLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_EncipherKeyEntry(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bPersoKeyNo,                                                            /**< [In] Key reference number of the KST Key Entry to include
                                                                                         *        in the cryptogram. One of the following,
                                                                                         *          - NVM Keys: 0x00 - 0x7F
                                                                                         *          - RAM Keys: 0xE0 - 0xE3
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Key reference number of the KSTKeyEntry in the target
                                                                                         *        SAM (00h to 7Fh).
                                                                                         */
        uint8_t bOption,                                                                /**< [In] Option to include Key diversification and SAM UID in command.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_ENCIPHER_KEY_ENTRY_USE_DIVERSIFICATION
                                                                                         *                 "Diversify the injected key with the given DivInput"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_ENCIPHER_KEY_ENTRY_INCLUDE_SAMUID
                                                                                         *                 "Include target GoldField (SAM UID)"
                                                                                         */
        uint16_t wPersoCtr,                                                             /**< [In] Change Counter to avoid replay attacks */
        uint8_t * pDivInput,                                                            /**< [In] Diversification Input used to diversify the key. */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t ** ppOfflineCrypto,                                                     /**< [Out] Offline crypto information received from SAM.
                                                                                         *           - EncKeyEntry: Encrypted key entry as E( K_{pe} , NewEntry [|| SAMUID])
                                                                                         *           - OfflineMAC: Integrity MAC protection as
                                                                                         *             MACt ( Kpm , Channel || 0xC1 || KeyNo || ProMas || 2 + 80 + 8 || Perso_Ctr
                                                                                         *             || EncKeyEntry)
                                                                                         *
                                                                                         */
        uint16_t * pOfflineCryptoLen                                                    /**< [Out] Length of bytes available in \b ppOfflineCrypto buffer */
    );

/**
 * \brief Get information about a key entry.
 *  - SAM_GetKeyEntry allows reading the contents of the KST Key Entry specified in \b bKeyNo parameter.
 *  - Instead of the full keys on positions a, b and c, only their key version will be returned,
 *    each packed in one byte.
 *  - This command can be issued without valid (host) authentication.
 *  - Supports retrieval of NVM and RAM Keys.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pKeyEntryLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_GetKeyEntry(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bKeyNo,                                                                 /**< [In] Key reference number of the key entry to get.
                                                                                         *        One of the following,
                                                                                         *          - NVM Keys    : 0x00 - 0x7F
                                                                                         *          - PLUnlock Key: 0xF0
                                                                                         *          - RAM Key     : 0xE0 - 0xE3
                                                                                         */
        uint8_t bMode,                                                                  /**< [In] Key entry format to be used. Not applicable for RAM Keys
                                                                                         *          - \ref PH_OFF "Old Key Entry Format (up to 1 - byte ExtSet)"
                                                                                         *          - \ref PH_ON "New Key Entry Format"
                                                                                         */
        uint8_t ** ppKeyEntry,                                                          /**< [Out] Buffer containing the information about the key entry. */
        uint16_t * pKeyEntryLen                                                         /**< [Out] Length of bytes available in \b ppKeyEntry buffer. */
    );

/**
 * \brief Get information about a key usage counter (KUC).
 *  - SAM_GetKUCEntry allows reading the contents of the Key usage counter specified in \b bKucNo parameter.
 *  - This command can be issued without valid (host) authentication.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pKucEntryLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_GetKUCEntry(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bKucNo,                                                                 /**< [In] Reference number of the key usage counter to be
                                                                                         *        returned (00h to 0Fh).
                                                                                         */
        uint8_t ** ppKucEntry,                                                          /**< [Out] Buffer containing the KUC entry. */
        uint16_t * pKucEntryLen                                                         /**< [Out] Length of bytes available in \b ppKucEntry buffer. */
    );

/**
 * \brief Dump the current session key.
 *  - Retrieve the session key of an established authentication with a DESFire PICC or a MIFARE Plus PICC. In this
 *    case, an active PICC authentication (for these card types) is required.
 *  - Retrieve the derived LRP key material. In this case, an KeyClass.OfflineCrypto activation of KeyType.AES_128LRP
 *    is required, that is the SAM must be in PICCState.OfflineCrypto.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                      Operation successful.
 * \retval #PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE Correct execution - more data available in response.
 * \retval #PH_ERR_INVALID_DATA_PARAMS          \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER            \b pSessionKeyLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_DumpSessionKey(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bDumpMode,                                                              /**< [In] Dump Mode.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_DUMP_MODE_PLAIN "Plain Dump"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_DUMP_MODE_ENCIPHERED
                                                                                         *                 "Enciphered dump"
                                                                                         */
        uint8_t ** ppSessionKey,                                                        /**< [Out] Buffer containing Current session key information. */
        uint16_t * pSessionKeyLen                                                       /**< [Out] Length of bytes available in \b ppSessionKey buffer. */
    );

/**
 * \brief Retrive a PICC or OfflineCrypto key stored in the key table.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pDivInput and \b pSecretKeyLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_DumpSecretKey (
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bDumpMode,                                                              /**< [In] Dump Mode.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_DUMP_MODE_PLAIN "Plain Dump"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_DUMP_MODE_ENCIPHERED
                                                                                         *                 "Enciphered dump"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_DUMP_MODE_DIVERSIFICATION
                                                                                         *                 "Diversify the used key with the given DivInput"
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Reference number of the key entry to be dumped. One of
                                                                                         *        the following
                                                                                         *          - NVM Keys: 0x00 - 0x7F
                                                                                         *          - RAM Keys: 0xE0 - 0xE3
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Reference version of the key entry to be dumped. */
        uint8_t * pDivInput,                                                            /**< [In] Diversification Input used to diversify the key. */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t ** ppSecretKey,                                                         /**< [Out] Buffer containing the plain secret key. */
        uint16_t * pSecretKeyLen                                                        /**< [Out] Length of bytes available in \b ppSecretKey buffer. */
    );

/**
 * \brief Is used to derive a key from a source key (in other contexts often called master key) based
 * on a CMAC operation. In a MIFARE context, this command can be used to support session key generations
 * for the Transaction MAC and Secure Dynamic Messaging features, for back-end (and / or reader)
 * interpretation and validation of the cryptograms created by the PICC.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pDeriveIn is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_DeriveKey(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bSrcKeyNo,                                                              /**< [In] Key number of the source key.
                                                                                         *        One of the following,
                                                                                         *          - NVM Keys: 0x00 - 0x7F
                                                                                         *          - RAM Keys: 0xE0 - 0xE3
                                                                                         */
        uint8_t bSrcKeyVer,                                                             /**< [In] Key version of the source key (00h to FFh). */
        uint8_t bDstKeyNo,                                                              /**< [In] Key number of the destination key entry,
                                                                                         *        only RAM Keys: 0xE0 - 0xE3
                                                                                         */
        uint8_t * pDeriveIn,                                                            /**< [In] The derivation input for deriving the key. */
        uint8_t bDeriveInLen                                                            /**< [In] Length of derivation input used to derive the key. */
    );

/**
 * end of group phhalHw_Sam_Cmd_KM
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_KM_H */
