/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Key Management command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include "phhalHw_Sam_Cmd_KM.h"

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "../phhalHw_Sam_Cmd.h"
#include "../01_HostCommunication/phhalHw_Sam_Cmd_HC.h"
#include "../02_SecurityConfiguration/phhalHw_Sam_Cmd_SC.h"

phStatus_t phhalHw_Sam_Cmd_SAM_ChangeKeyEntry(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bKeyNo, uint8_t bProMas,
    uint8_t * pKeyData, uint8_t bKeyDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pKeyData, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_ChangeKeyEntry command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_CHANGE_KEY_ENTRY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bProMas;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Copy KeyData information to command buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pKeyData, bKeyDataLen);
    bCmdLen += bKeyDataLen;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_OFF);

    /* Exchange SAM_ChangeKeyEntry information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    /* Reset Authentication states if ChangeKey = AuthKey. */
    if(pDataParams->bKeyNo == bKeyNo)
    {
        /* Invalidate the session keys and set the SM to PLAIN. */
        pDataParams->bAuthType = PHHAL_HW_SAM_CMD_SAM_AUTHENTICATE_HOST_MODE_PLAIN;
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_InvalidateKey(pDataParams->pENCCryptoDataParams));
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_InvalidateKey(pDataParams->pMACCryptoDataParams));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_ChangeKeyEntryOffline(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bKeyNo, uint8_t bProMas,
    uint16_t wChangeCtr, uint8_t * pOfflineCrypto, uint8_t bOfflineCryptoLen, uint8_t bEnableOfflineAck, uint8_t ** ppOfflineAck,
    uint16_t * pOfflineAckLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pOfflineCrypto, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pOfflineAckLen, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_ChangeKeyEntryOffline command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_CHANGE_KEY_ENTRY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bProMas;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add Change counter. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) ((wChangeCtr & 0xFF00U) >> 8U);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (wChangeCtr & 0x00FFU);

    /* Copy KeyData information to command buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pOfflineCrypto, bOfflineCryptoLen);
    bCmdLen += bOfflineCryptoLen;

    /* Add LE if Offline Acknowledgment is requested */
    if(bEnableOfflineAck == (uint8_t) PH_ON)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;
    }

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, bEnableOfflineAck);

    /* Exchange SAM_ChangeKeyEntryOffline information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppOfflineAck,
        pOfflineAckLen));

    /* Reset Authentication states if ChangeKey = AuthKey. */
    if(pDataParams->bKeyNo == bKeyNo)
    {
        /* Perform Kill Authentication. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_SAM_KillAuthentication(pDataParams,
            PHHAL_HW_SAM_CMD_SAM_KILL_AUTHENTICATION_FULL));

        /* Invalidate the session keys and set the SM to PLAIN. */
        pDataParams->bAuthType = PHHAL_HW_SAM_CMD_SAM_AUTHENTICATE_HOST_MODE_PLAIN;
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_InvalidateKey(pDataParams->pENCCryptoDataParams));
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_InvalidateKey(pDataParams->pMACCryptoDataParams));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_ChangeKUCEntry(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bKucNo, uint8_t bProMas,
    uint8_t * pKucData, uint8_t bKucDataLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pKucData, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_ChangeKUCEntry command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_CHANGE_KUC_ENTRY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKucNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bProMas;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKucDataLen;

    /* Buffer the command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Buffer KucData to exchange buffer and exchange the buffered information to Sam hardware. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pKucData,
        bKucDataLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_ChangeKUCEntryOffline(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bKucNo, uint8_t bProMas,
    uint16_t wChangeCtr, uint8_t * pOfflineCrypto, uint8_t bOfflineCryptoLen, uint8_t bEnableOfflineAck, uint8_t ** ppOfflineAck,
    uint16_t * pOfflineAckLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pOfflineCrypto, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pOfflineAckLen, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_ChangeKUCEntryOffline command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_CHANGE_KUC_ENTRY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKucNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bProMas;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add Change counter. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) ((wChangeCtr & 0xFF00U) >> 8U);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (wChangeCtr & 0x00FFU);

    /* Copy KeyData information to command buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pOfflineCrypto, bOfflineCryptoLen);
    bCmdLen += bOfflineCryptoLen;

    /* Add LE if Offline Acknowledgment is requested */
    if(bEnableOfflineAck == (uint8_t) PH_ON)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;
    }

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, bEnableOfflineAck);

    /* Exchange SAM_ChangeKUCEntryOffline information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppOfflineAck,
        pOfflineAckLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_DisableKeyEntry(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bKeyNo)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_DisableKeyEntry command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_DISABLE_KEY_ENTRY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;

    /* Exchange SAM_DisableKeyEntry information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    /* Reset Authentication states if ChangeKey = AuthKey. */
    if(pDataParams->bKeyNo == bKeyNo)
    {
        /* Perform Kill Authentication. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_SAM_KillAuthentication(pDataParams,
            PHHAL_HW_SAM_CMD_SAM_KILL_AUTHENTICATION_FULL));

        /* Invalidate the session keys and set the SM to PLAIN. */
        pDataParams->bAuthType = PHHAL_HW_SAM_CMD_SAM_AUTHENTICATE_HOST_MODE_PLAIN;
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_InvalidateKey(pDataParams->pENCCryptoDataParams));
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_InvalidateKey(pDataParams->pMACCryptoDataParams));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_DisableKeyEntryOffline(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bKeyNo, uint16_t wChangeCtr,
    uint8_t * pOfflineCrypto, uint8_t bOfflineCryptoLen, uint8_t bEnableOfflineAck, uint8_t ** ppOfflineAck,
    uint16_t * pOfflineAckLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pOfflineCrypto, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pOfflineAckLen, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_DisableKeyEntryOffline command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_DISABLE_KEY_ENTRY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add Change counter. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) ((wChangeCtr & 0xFF00U) >> 8U);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (wChangeCtr & 0x00FFU);

    /* Copy KeyData information to command buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pOfflineCrypto, bOfflineCryptoLen);
    bCmdLen += bOfflineCryptoLen;

    /* Add LE if Offline Acknowledgment is requested */
    if(bEnableOfflineAck == (uint8_t) PH_ON)
    {
        PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;
    }

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, bEnableOfflineAck);

    /* Exchange SAM_DisableKeyEntryOffline information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppOfflineAck,
        pOfflineAckLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_EncipherKeyEntry(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bPersoKeyNo, uint8_t bKeyNo,
    uint8_t bOption, uint16_t wPersoCtr, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t ** ppOfflineCrypto,
    uint16_t * pOfflineCryptoLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pOfflineCryptoLen, PH_COMP_HAL);
    if(bOption & PHHAL_HW_SAM_CMD_SAM_ENCIPHER_KEY_ENTRY_USE_DIVERSIFICATION)
        PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_EncipherKeyEntry command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_ENCHIPHER_KEY_ENTRY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bPersoKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bOption;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add payload information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (0x80U | pDataParams->bLogicalChannel);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) ((wPersoCtr & 0xFF00U) >> 8U);
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = (uint8_t) (wPersoCtr & 0x00FFU);

    /* Add SAM UID to command buffer. */
    if(bOption & PHHAL_HW_SAM_CMD_SAM_ENCIPHER_KEY_ENTRY_INCLUDE_SAMUID)
    {
        memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pDataParams->bUid, PHHAL_HW_SAM_HC_SAM_UID_SIZE);
        bCmdLen += PHHAL_HW_SAM_HC_SAM_UID_SIZE;
    }

    /* Add Diversification Input to command buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pDivInput, bDivInputLen);
    bCmdLen += bDivInputLen;

    /* Add LE to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange SAM_EncipherKeyEntry information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppOfflineCrypto,
        pOfflineCryptoLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_GetKeyEntry(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bKeyNo, uint8_t bMode,
    uint8_t ** ppKeyEntry, uint16_t * pKeyEntryLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pKeyEntryLen, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_GetKeyEntry command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_GET_KEY_ENTRY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bMode;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Exchange SAM_GetKeyEntry information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppKeyEntry,
        pKeyEntryLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_GetKUCEntry(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bKucNo, uint8_t ** ppKucEntry,
    uint16_t * pKucEntryLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pKucEntryLen, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_GetKUCEntry command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_GET_KUC_ENTRY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKucNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Exchange SAM_GetKUCEntry information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppKucEntry,
        pKucEntryLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_DumpSessionKey(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bDumpMode, uint8_t ** ppSessionKey,
    uint16_t * pSessionKeyLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSessionKeyLen, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_DumpSessionKey command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_DUMP_SESSION_KEY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bDumpMode;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Exchange SAM_DumpSessionKey information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppSessionKey,
        pSessionKeyLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_DumpSecretKey(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bDumpMode, uint8_t bKeyNo,
    uint8_t bKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t ** ppSecretKey, uint16_t * pSecretKeyLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pSecretKeyLen, PH_COMP_HAL);
    if(bDivInputLen != 0)
    {
        PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_HAL);
    }

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_DumpSecretKey command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_DUMP_SECRET_KEY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bDumpMode;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add key number and version to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bKeyVer;

    /* Add Diversification input to Command buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pDivInput, bDivInputLen);
    bCmdLen += bDivInputLen;

    /* Add LE to command buffer */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_ON);

    /* Exchange SAM_DumpSecretKey information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        ppSecretKey,
        pSecretKeyLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Cmd_SAM_DeriveKey(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bSrcKeyNo, uint8_t bSrcKeyVer,
    uint8_t bDstKeyNo, uint8_t * pDeriveIn, uint8_t bDeriveInLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pDeriveIn, PH_COMP_HAL);

    /* Reset the command buffer. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_DeriveKey command information. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_SAM_DERIVE_KEY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add payload information to command buffer. */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bSrcKeyNo;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bSrcKeyVer;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = bDstKeyNo;

    /* Add Derivation input to Command buffer. */
    memcpy(&PHHAL_HW_SAM_CMD_BUFFER[bCmdLen], pDeriveIn, bDeriveInLen);
    bCmdLen += bDeriveInLen;

    /* Update LC */
    phhalHw_Sam_Cmd_UpdateLC(PHHAL_HW_SAM_CMD_BUFFER, bCmdLen, PH_OFF);

    /* Exchange SAM_DeriveKey information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
