/*
 * Copyright 2019 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal definitions for ProxiLAB Reader specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <phTools.h>
#include <string.h>

#import "ex\ProxiLAB.exe" no_namespace

#ifdef NXPBUILD__PHHAL_HW_PROXILAB

#include "phhalHw_ProxiLAB.h"
#include "phhalHw_ProxiLAB_Int.h"
#include "phhalHw_ProxiLAB_Config.h"

phStatus_t phhalHw_ProxiLAB_Int_FieldOn(
                                    phhalHw_ProxiLAB_DataParams_t * pDataParams
                                    )
{
    IProxiLABPtr ProxiLAB = (IProxiLAB*)pDataParams->pBalDataParams->pProxiLAB;

    return phhalHw_ProxiLAB_Int_ConvertError(pDataParams, ProxiLAB->Reader->Power_1024(pDataParams->dwPowerLevel));
}

phStatus_t phhalHw_ProxiLAB_Int_FieldOff(
                                     phhalHw_ProxiLAB_DataParams_t * pDataParams
                                     )
{
    IProxiLABPtr ProxiLAB = (IProxiLAB*)pDataParams->pBalDataParams->pProxiLAB;

    return phhalHw_ProxiLAB_Int_ConvertError(pDataParams, ProxiLAB->Reader->Power_1024(0));
}

phStatus_t phhalHw_ProxiLAB_Int_FieldReset(
                                       phhalHw_ProxiLAB_DataParams_t * pDataParams
                                       )
{
    IProxiLABPtr ProxiLAB = (IProxiLAB*)pDataParams->pBalDataParams->pProxiLAB;
    phStatus_t status;

    //FIXXME Communication timeout???
    ProxiLAB->Reader->ResetDuration = pDataParams->wFieldOffTime * 1000;
    PH_CHECK_SUCCESS_FCT(status, phhalHw_ProxiLAB_Int_ConvertError(pDataParams, ProxiLAB->Reader->RfReset()));
    PH_CHECK_SUCCESS_FCT(status, phhalHw_ProxiLAB_Int_ConvertError(pDataParams, ProxiLAB->Delay(pDataParams->wFieldRecoveryTime)));
    return status;
}

phStatus_t phhalHw_ProxiLAB_Int_Wait(
                                 phhalHw_ProxiLAB_DataParams_t * pDataParams,
                                 uint8_t bUnit,
                                 uint16_t wTimeout
                                 )
{
    IProxiLABPtr ProxiLAB = (IProxiLAB*)pDataParams->pBalDataParams->pProxiLAB;

    if (bUnit == PHHAL_HW_TIME_MICROSECONDS)
    {
        return phhalHw_ProxiLAB_Int_ConvertError(pDataParams, ProxiLAB->Delay(((uint32_t)wTimeout + 999)/1000));
    }

    if (bUnit == PHHAL_HW_TIME_MILLISECONDS)
    {
        return phhalHw_ProxiLAB_Int_ConvertError(pDataParams, ProxiLAB->Delay(wTimeout));
    }

    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
}

phStatus_t phhalHw_ProxiLAB_Int_SetConfig(
                                      phhalHw_ProxiLAB_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint16_t wValue
                                      )
{
    phStatus_t status;
    IProxiLABPtr ProxiLAB = (IProxiLAB *)pDataParams->pBalDataParams->pProxiLAB;

    switch (wConfig)
    {
    case PHHAL_HW_CONFIG_PARITY:
        /* check parameter */
        if (wValue != PH_OFF && wValue != PH_ON)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* FIXXME */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
        /* Write config data into shadow */
        /* pDataParams->wCfgShadow[wConfig] = wValue;
        break; */

    case PHHAL_HW_CONFIG_TXCRC:
        /* check parameter */
        if (wValue != PH_OFF && wValue != PH_ON)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_RXCRC:
        /* check parameter */
        if (wValue != PH_OFF && wValue != PH_ON)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_TXLASTBITS:
        /* check parameter */
        if (wValue > 7)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_RXALIGN:
        /* check parameter */
        if (wValue > 7)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_RXDEAFBITS:
        /* check parameter, maximum 10 seconds */
        //if ((((float)wValue*128.0f)/13.56f) > 10000000.0f)
        //{
        //    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        //}
        ///* check parameter, minimum 30 microseconds or zero*/
        //if ((((float)wValue*128.0f)/13.56f) < 30.0f && wValue != 0)
        //{
        //    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        //}
        //FIXXME
//        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_TXWAIT_US:
        /* FIXXME */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
        /* Write config data into shadow */
        /* pDataParams->wCfgShadow[wConfig] = wValue;
        break; */

    case PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL:
        /* check parameter */
        if (wValue != PH_OFF && wValue != PH_ON)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_TXDATARATE:
        if ((pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A) ||
            (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443B))
        {
            switch (wValue)
            {
            case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_256:
            case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4:
            case PHHAL_HW_RF_RX_DATARATE_LOW:
            case PHHAL_HW_RF_RX_DATARATE_HIGH:
            case PHHAL_HW_RF_RX_DATARATE_FAST_LOW:
            case PHHAL_HW_RF_RX_DATARATE_FAST_HIGH:
                return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);

            case PHHAL_HW_RF_DATARATE_1695:
            case PHHAL_HW_RF_DATARATE_3390:
            case PHHAL_HW_RF_DATARATE_6780:
                if ((ProxiLAB->Settings->License & LICENSE_PCD_VHBR_ASK) != LICENSE_PCD_VHBR_ASK ||
                    (pDataParams->bCardType != PHHAL_HW_CARDTYPE_ISO14443B))
                {
                    return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
                }
            case PHHAL_HW_RF_DATARATE_106:
            case PHHAL_HW_RF_DATARATE_212:
            case PHHAL_HW_RF_DATARATE_424:
            case PHHAL_HW_RF_DATARATE_848:
                break;

            default:
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_RXDATARATE:
        if ((pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A) ||
            (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443B))
        {
            switch (wValue)
            {
            case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_256:
            case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4:
            case PHHAL_HW_RF_RX_DATARATE_LOW:
            case PHHAL_HW_RF_RX_DATARATE_HIGH:
            case PHHAL_HW_RF_RX_DATARATE_FAST_LOW:
            case PHHAL_HW_RF_RX_DATARATE_FAST_HIGH:
                return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);

            case PHHAL_HW_RF_DATARATE_1695:
            case PHHAL_HW_RF_DATARATE_3390:
            case PHHAL_HW_RF_DATARATE_6780:
                if ((ProxiLAB->Settings->License & LICENSE_PCD_VHBR_ASK) != LICENSE_PCD_VHBR_ASK)
                {
                    return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
                }
            case PHHAL_HW_RF_DATARATE_106:
            case PHHAL_HW_RF_DATARATE_212:
            case PHHAL_HW_RF_DATARATE_424:
            case PHHAL_HW_RF_DATARATE_848:
                break;

            default:
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_MODINDEX:
        /* check parameter, maximum 100 percent */
        if (wValue > 100)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* FIXXME */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
        //PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetModulationASK(pDataParams, (uint8_t)wValue));
        /* Write config data into shadow */
        /* pDataParams->wCfgShadow[wConfig] = wValue;
        break; */

    case PHHAL_HW_CONFIG_ASK100:
        /* check parameter */
        if (wValue != PH_OFF && wValue != PH_ON)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* FIXXME */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
        /* Write config data into shadow */
        /* pDataParams->wCfgShadow[wConfig] = wValue;
        break; */

    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_US:
    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS:
        /* Calculate values for Microsecond values */
        if (wConfig == PHHAL_HW_CONFIG_TIMEOUT_VALUE_US)
        {
            pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MICROSECONDS;
        }
        /* Calculate values for Millisecond values */
        else
        {
            pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MILLISECONDS;
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;


    //    case PHHAL_HW_CONFIG_PARITY:
    //    case PHHAL_HW_CONFIG_TXCRC:
    //    case PHHAL_HW_CONFIG_RXCRC:
    //    case PHHAL_HW_CONFIG_TXLASTBITS:
    //    case PHHAL_HW_CONFIG_RXLASTBITS:
    //case PHHAL_HW_CONFIG_RXALIGN:
    //case PHHAL_HW_CONFIG_RXDEAFBITS:
    //case PHHAL_HW_CONFIG_TXWAIT_US:
    //case PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL:
    //case PHHAL_HW_CONFIG_TXDATARATE:
    //case PHHAL_HW_CONFIG_RXDATARATE:
    //case PHHAL_HW_CONFIG_MODINDEX:
    //case PHHAL_HW_CONFIG_ASK100:
    //case PHHAL_HW_CONFIG_TIMEOUT_VALUE_US:
    //case PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS:

        case PHHAL_HW_CONFIG_RXBUFFER_STARTPOS:
            /* Boundary check */
            if (wValue >= pDataParams->wRxBufSize)
            {
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
            }

            /* Set start position */
            pDataParams->wRxBufStartPos = wValue;
            pDataParams->wRxBufLen = wValue;
            /* Preserve RxBuffer contents if needed */
            if (pDataParams->pTxBuffer == pDataParams->pRxBuffer)
            {
                pDataParams->wTxBufStartPos = pDataParams->wRxBufStartPos;
            }
            else
            {
                pDataParams->wTxBufStartPos = 0;
            }
            break;

        case PHHAL_HW_CONFIG_TXBUFFER_LENGTH:
            /* Check parameter */
            if ((pDataParams->wTxBufStartPos + wValue) > pDataParams->wTxBufSize)
            {
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
            }
            /* set buffer length */
            pDataParams->wTxBufLen = wValue;
            break;

        case PHHAL_HW_CONFIG_TXBUFFER:
            /* Check additional info parameter */
            if ((pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo) >= pDataParams->wTxBufSize)
            {
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
            }
            /* Modify TxBuffer byte */
            pDataParams->pTxBuffer[pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo] = (uint8_t)wValue;
            break;

        case PHHAL_HW_CONFIG_FIELD_OFF_TIME:
            /* Parameter Check */
            if (wValue == 0 || wValue > 1000)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            /* Store config data */
            pDataParams->wFieldOffTime = wValue;
            break;

        case PHHAL_HW_CONFIG_FIELD_RECOVERY_TIME:
            /* Store config data */
            pDataParams->wFieldRecoveryTime = wValue;
            break;

        case PHHAL_HW_CONFIG_ADDITIONAL_INFO:
            /* Modify additional info parameter */
            pDataParams->wAdditionalInfo = wValue;
            break;





        case PHHAL_HW_PROXILAB_CONFIG_POWERLEVEL:
            /* check parameter */
            if (wValue > 1023)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            pDataParams->dwPowerLevel = wValue;
            PH_CHECK_SUCCESS_FCT(status, phhalHw_ProxiLAB_Int_ConvertError(pDataParams, ProxiLAB->Reader->Power_1024(pDataParams->dwPowerLevel)));
            break;

        case PHHAL_HW_PROXILAB_CONFIG_MODULATOR_OUTPUT:
            switch (wValue)
            {
            case PHHAL_HW_PROXILAB_MODULATOR_RF_OUT:
                ProxiLAB->Settings->ModulatorOutput = MODULATOR_RF_OUT;
                break;
            case PHHAL_HW_PROXILAB_MODULATOR_AWG_OUT:
                ProxiLAB->Settings->ModulatorOutput = MODULATOR_AWG_OUT;
                break;
            case PHHAL_HW_PROXILAB_MODULATOR_RF_OUT_DUAL:
                ProxiLAB->Settings->ModulatorOutput = MODULATOR_RF_OUT_DUAL;
                break;
            case PHHAL_HW_PROXILAB_MODULATOR_AWG_CARRIER_RF_OUT:
                ProxiLAB->Settings->ModulatorOutput = MODULATOR_AWG_CARRIER_RF_OUT;
                break;
            default:
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_PROXILAB_CONFIG_DEMODULATOR_AUTO_THRESHOLD:
            /* check parameter */
            if (wValue != PH_OFF && wValue != PH_ON)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            ProxiLAB->Settings->DemodulatorAutoThreshold = (wValue == PH_ON) ? 1 : 0;
            break;

        case PHHAL_HW_PROXILAB_CONFIG_DEMODULATOR_THRESHOLD:
            /* check parameter */
            if (wValue > 255)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            ProxiLAB->Settings->DemodulatorThreshold = wValue;
            break;

        case PHHAL_HW_PROXILAB_CONFIG_DEMODULATORINPUT:
            switch (wValue)
            {
            case PHHAL_HW_PROXILAB_DEMODULATOR_RF_OUT:
                if (ProxiLAB->Settings->ActivateSSBFilters == PH_ON)
                {
                    ProxiLAB->Settings->ActivateSSBFilters = PH_OFF;
                }
                ProxiLAB->Settings->DemodulatorInput = DEMODULATOR_RF_OUT;
                break;
            case PHHAL_HW_PROXILAB_DEMODULATOR_RF_IN:
                if (ProxiLAB->Settings->ActivateSSBFilters == PH_ON)
                {
                    ProxiLAB->Settings->ActivateSSBFilters = PH_OFF;
                }
                ProxiLAB->Settings->DemodulatorInput = DEMODULATOR_RF_IN;
                break;
            case PHHAL_HW_PROXILAB_DEMODULATOR_BRIDGE:
                if (ProxiLAB->Settings->ActivateSSBFilters == PH_ON)
                {
                    ProxiLAB->Settings->ActivateSSBFilters = PH_OFF;
                }
                ProxiLAB->Settings->DemodulatorInput = DEMODULATOR_BRIDGE;
                break;
            case PHHAL_HW_PROXILAB_DEMODULATOR_BRIDGE_SSB:
                ProxiLAB->Settings->DemodulatorInput = DEMODULATOR_BRIDGE;
                ProxiLAB->Settings->ActivateSSBFilters = PH_ON;
                break;
            case PHHAL_HW_PROXILAB_DEMODULATOR_SMA2:
                if (ProxiLAB->Settings->ActivateSSBFilters == PH_ON)
                {
                    ProxiLAB->Settings->ActivateSSBFilters = PH_OFF;
                }
                ProxiLAB->Settings->DemodulatorInput = DEMODULATOR_SMA2;
                break;
            default:
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            break;

        case PHHAL_HW_PROXILAB_CONFIG_DISPLAY_VERBOSE:
             /* check parameter */
            if (wValue != PH_OFF && wValue != PH_ON)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            ProxiLAB->Display->Verbose = (wValue == PH_ON) ? 1 : 0;
            break;

        case PHHAL_HW_PROXILAB_CONFIG_CLOCK_SOURCE:
            switch (wValue)
            {
            case PHHAL_HW_PROXILAB_CLOCK_SOURCE_INT:
                ProxiLAB->Settings->ClockSource = CLOCK_SOURCE_INT;
                break;
            case PHHAL_HW_PROXILAB_CLOCK_SOURCE_EXT_DIG:
                ProxiLAB->Settings->ClockSource = CLOCK_SOURCE_EXT_DIG;
                break;
            case PHHAL_HW_PROXILAB_CLOCK_SOURCE_EXT_ANA:
                ProxiLAB->Settings->ClockSource = CLOCK_SOURCE_EXT_ANA;
                break;
            default:
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
            break;

        default:
            /* return success to avoid errors */
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ProxiLAB_Int_GetConfig(
                                      phhalHw_ProxiLAB_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint16_t * pValue
                                      )
{
    IProxiLABPtr ProxiLAB = (IProxiLAB *)pDataParams->pBalDataParams->pProxiLAB;

    switch (wConfig)
    {
    case PHHAL_HW_CONFIG_FIELD_OFF_TIME:
        /* get param from reader (in case changed from another application */
        pDataParams->wFieldOffTime = (uint16_t)((ProxiLAB->Reader->ResetDuration + 500)/1000);
        /* Return parameter */
        *pValue = pDataParams->wFieldOffTime;
        break;

    case PHHAL_HW_CONFIG_FIELD_RECOVERY_TIME:
        /* Return parameter */
        *pValue = pDataParams->wFieldRecoveryTime;
        break;

    case PHHAL_HW_PROXILAB_CONFIG_POWERLEVEL:
        /* get param from reader (in case changed from another application */
        pDataParams->dwPowerLevel = ProxiLAB->Reader->PowerLevel_1024;
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->dwPowerLevel;
        break;

    case PHHAL_HW_PROXILAB_CONFIG_DEMODULATOR_AUTO_THRESHOLD:
        *pValue = (ProxiLAB->Settings->DemodulatorAutoThreshold == 0) ? PH_OFF : PH_ON;
        break;

    case PHHAL_HW_PROXILAB_CONFIG_DEMODULATOR_THRESHOLD:
        *pValue = (uint16_t)ProxiLAB->Settings->DemodulatorThreshold;
        break;

    case PHHAL_HW_CONFIG_RXBUFFER_STARTPOS:
        /* Return parameter */
        *pValue = pDataParams->wRxBufStartPos;
        break;

    case PHHAL_HW_CONFIG_RXBUFFER_BUFSIZE:
        /* Return parameter */
        *pValue = pDataParams->wRxBufSize;
        break;

    case PHHAL_HW_CONFIG_TXBUFFER_BUFSIZE:
        /* Return parameter */
        *pValue = pDataParams->wTxBufSize - pDataParams->wTxBufStartPos;
        break;

    case PHHAL_HW_CONFIG_TXBUFFER_LENGTH:
        /* Return parameter */
        *pValue = pDataParams->wTxBufLen;
        break;

    case PHHAL_HW_CONFIG_TXBUFFER:
        /* Check additional info parameter */
        if ((pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo) >= pDataParams->wTxBufSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* Retrieve TxBuffer byte */
        *pValue = (uint16_t)pDataParams->pTxBuffer[pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo];
        break;

    case PHHAL_HW_CONFIG_ADDITIONAL_INFO:
    case PHHAL_HW_CONFIG_RXLASTBITS:
        *pValue = pDataParams->wAdditionalInfo;
        break;

    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_US:
        if (pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MICROSECONDS)
        {
            *pValue = pDataParams->wCfgShadow[wConfig];
        }
        else
        {
            if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS] > (0xFFFF / 1000))
            {
                return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
            }
            *pValue = pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS] * 1000;
        }
        break;

    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS:
        if (pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MILLISECONDS)
        {
            *pValue = pDataParams->wCfgShadow[wConfig];
        }
        else
        {
            *pValue = pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_US] / 1000;
        }
        break;

    case PHHAL_HW_CONFIG_CARD_TYPE:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->bCardType;
        break;

    case PHHAL_HW_CONFIG_TXDATARATE:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_RXDATARATE:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    default:
        /* return success to avoid errors */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ProxiLAB_Int_Exchange(
                                      phhalHw_ProxiLAB_DataParams_t * pDataParams,
                                      uint8_t * pTxBuffer,
                                      uint16_t wTxLength,
                                      uint16_t wRxBufSize,
                                      uint8_t * pRxBuffer,
                                      uint16_t * pRxLength
                                      )
{
    IProxiLABPtr ProxiLAB = (IProxiLAB *)pDataParams->pBalDataParams->pProxiLAB;
    phStatus_t status;
    uint32_t dwTxDatarate;
    uint32_t dwRxDatarate;
    uint8_t bAppendCRC;
    uint32_t dwTimeout;
    unsigned long dwTxLength = wTxLength;
    unsigned long dwRxLength;
    uint8_t bLastBit;
    uint8_t bRxAlign;
    uint8_t bCollisionFlag;
    uint8_t bCollisionPosition;
    uint16_t wLastUsedByteIndex;

    /* preapare parameter */
    switch(pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXDATARATE])
    {
        case PHHAL_HW_RF_DATARATE_106:
            {
                dwTxDatarate = 106;
                break;
            }
        case PHHAL_HW_RF_DATARATE_212:
            {
                dwTxDatarate = 212;
                break;
            }
        case PHHAL_HW_RF_DATARATE_424:
            {
                dwTxDatarate = 424;
                break;
            }
        case PHHAL_HW_RF_DATARATE_848:
            {
                dwTxDatarate = 847;
                break;
            }
        case PHHAL_HW_RF_DATARATE_1695:
            {
                dwTxDatarate = 1695;
                break;
            }
        case PHHAL_HW_RF_DATARATE_3390:
            {
                dwTxDatarate = 3390;
                break;
            }
        case PHHAL_HW_RF_DATARATE_6780:
            {
                dwTxDatarate = 6780;
                break;
            }
        default:
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    switch(pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE])
    {
        case PHHAL_HW_RF_DATARATE_106:
            {
                dwRxDatarate = 106;
                break;
            }
        case PHHAL_HW_RF_DATARATE_212:
            {
                dwRxDatarate = 212;
                break;
            }
        case PHHAL_HW_RF_DATARATE_424:
            {
                dwRxDatarate = 424;
                break;
            }
        case PHHAL_HW_RF_DATARATE_848:
            {
                dwRxDatarate = 847;
                break;
            }
        case PHHAL_HW_RF_DATARATE_1695:
            {
                dwRxDatarate = 1695;
                break;
            }
        case PHHAL_HW_RF_DATARATE_3390:
            {
                dwRxDatarate = 3390;
                break;
            }
        case PHHAL_HW_RF_DATARATE_6780:
            {
                dwRxDatarate = 6780;
                break;
            }
        default:
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    bAppendCRC = pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXCRC] == PH_ON ? 1 : 0;
    dwTimeout = (pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MILLISECONDS) ?
        pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS] * 1000 :
        pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_US];
    bLastBit = (uint8_t)pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS];
    bRxAlign = (uint8_t)pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXALIGN];

    *pRxLength = 0;

    /* Perform short frame */
    if (dwTxLength == 1 && bLastBit == 7 && bAppendCRC == 0 && bRxAlign == 0)
    {
        PH_CHECK_SUCCESS_FCT(status, phhalHw_ProxiLAB_Int_ConvertError(pDataParams,
            ProxiLAB->Reader->ISO14443->TypeA->SendTransparentCommandShort(dwTxDatarate, dwRxDatarate, dwTimeout, pTxBuffer, &dwTxLength, pRxBuffer, wRxBufSize, &dwRxLength, 0x01)));
        *pRxLength = (uint16_t)dwRxLength;

        /* A collision can also occure at short frame */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_ProxiLAB_Int_ConvertError(pDataParams,
            ProxiLAB->Reader->ISO14443->VerifAnticol(&bCollisionFlag, &bCollisionPosition)));
    }
    else if (bLastBit == 0 && bRxAlign == 0) /* Standard frame */
    {
        unsigned char BufParity = 0;
        unsigned long BufParitySize = 0;
        PH_CHECK_SUCCESS_FCT(status, phhalHw_ProxiLAB_Int_ConvertError(pDataParams,
            ProxiLAB->Reader->ISO14443->TypeA->SendTransparentCommand(dwTxDatarate, dwRxDatarate, bAppendCRC, dwTimeout, pTxBuffer,  &dwTxLength, &BufParity, &BufParitySize, pRxBuffer, wRxBufSize, &dwRxLength, 0x01)));
        *pRxLength = (uint16_t)dwRxLength;

        /* A collision can also occure at standard frame */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_ProxiLAB_Int_ConvertError(pDataParams,
            ProxiLAB->Reader->ISO14443->VerifAnticol(&bCollisionFlag, &bCollisionPosition)));
    }
    else if(bAppendCRC == 0 && bLastBit == bRxAlign) /* Must be a anticollision frame */
    {
        unsigned char BufParity = 0;
        unsigned long BufParitySize = 0;
        if (bLastBit == 0)
        {
            bLastBit = 8;
        }
        PH_CHECK_SUCCESS_FCT(status, phhalHw_ProxiLAB_Int_ConvertError(pDataParams,
            ProxiLAB->Reader->ISO14443->TypeA->SendTransparentCommandAnticollision(dwTxDatarate, dwRxDatarate, bLastBit, dwTimeout, pTxBuffer,  &dwTxLength, &BufParity, &BufParitySize, pRxBuffer, wRxBufSize, &dwRxLength, &bCollisionFlag, &bCollisionPosition, 0x01)));
        *pRxLength = (uint16_t)dwRxLength;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    if(*pRxLength == 0) /* No answer */
    {
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_HAL);
    }

    /* Collision handling */
    if (bCollisionFlag == 1)
    {
        status = PH_ADD_COMPCODE(PH_ERR_COLLISION_ERROR, PH_COMP_HAL);
    }
    if ( (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A) &&
              (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE] == PHHAL_HW_RF_DATARATE_106) &&
              ((status & PH_ERR_MASK) == PH_ERR_COLLISION_ERROR)
        )
    {
        /* Adjust collision position with offset */
        if ( pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXALIGN] != 0 )
        {
            if (bCollisionPosition > 0xFF - pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXALIGN])
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
            }
            bCollisionPosition = bCollisionPosition + (uint8_t)pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXALIGN];
        }
        /* Remove bits after collision */
        if ( pDataParams->wCfgShadow[PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL] == PH_ON )
        {
            wLastUsedByteIndex = (bCollisionPosition>>3);
            pDataParams->wAdditionalInfo = (bCollisionPosition&0x07);
            if (pDataParams->wAdditionalInfo == 0)
            {
                if (wLastUsedByteIndex == 0)
                {
                    *pRxLength = 0;
                }
                else
                {
                    *pRxLength = wLastUsedByteIndex;
                }
            }
            else
            {
                *pRxLength = wLastUsedByteIndex + 1;
            }
        }
    }

    /* If CRC was enabled resize RxBuffer and RxBufferLength */
    if(pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXCRC] == PH_ON)
    {
        return phhalHw_ProxiLAB_Int_CheckCrc(pDataParams, pRxBuffer, pRxLength);
    }
    return status;
}

phStatus_t phhalHw_ProxiLAB_Int_ConvertError(phhalHw_ProxiLAB_DataParams_t * pDataParams,
                                              uint32_t status)
{
    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_ProxiLAB_Int_ConvertError");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bLastErrorMessage);

    phStatus_t convertedStatus;
    IProxiLABPtr ProxiLAB = (IProxiLAB*)pDataParams->pBalDataParams->pProxiLAB;
    if (status == ERR_SUCCESSFUL)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    switch(status)
    {
    case XSMRDR_ERR_CRC:
    case XSMRDR_ERR_PARITY:
        convertedStatus = PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
        break;
    case XSMRDR_ERR_TIMEOUT:
        convertedStatus = PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_HAL);
        break;
    case XSMRDR_ERR_SOFT_TIMEOUT:
    case XSMRDR_ERR_TIMEOUT_PICC_SOF_DETECTED:
    case XSMRDR_ERR_BREAK:
        convertedStatus = PH_ADD_COMPCODE(PH_ERR_FRAMING_ERROR, PH_COMP_HAL);
        break;
    default:
        convertedStatus = PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_HAL);
        break;
    }

    /* Store the error message */
    sprintf_s((char *)pDataParams->pBalDataParams->bLastErrorMessage, PHBAL_REG_PROXILAB_ERROR_MESSAGE_SIZE, (char *)ProxiLAB->GetErrorInfo(status));

    if ((convertedStatus & PH_ERR_MASK) != PH_ERR_IO_TIMEOUT)
    {
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_INFO, bLastErrorMessage_log, (char *)pDataParams->pBalDataParams->bLastErrorMessage, (uint16_t)strlen((char *)pDataParams->pBalDataParams->bLastErrorMessage));
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
    }
    return convertedStatus;
}

phStatus_t phhalHw_ProxiLAB_Int_GetTxBuffer(
                                     phhalHw_ProxiLAB_DataParams_t * pDataParams,
                                     uint8_t ** pTxBuffer,
                                     uint16_t * pTxBufferLen,
                                     uint16_t * pTxBufferSize
                                     )
{
    /* We need to watch that we do not overwrite content below the RxStartPos though */
    if (pDataParams->pTxBuffer == pDataParams->pRxBuffer)
    {
        *pTxBuffer = &pDataParams->pTxBuffer[pDataParams->wRxBufStartPos];
        *pTxBufferSize = pDataParams->wTxBufSize - pDataParams->wRxBufStartPos;
    }
    /* Else just return the actual Buffer */
    else
    {
        *pTxBuffer = pDataParams->pTxBuffer;
        *pTxBufferSize = pDataParams->wTxBufSize;
    }

    /* Return stored length */
    *pTxBufferLen = pDataParams->wTxBufLen;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ProxiLAB_Int_GetRxBuffer(
                                     phhalHw_ProxiLAB_DataParams_t * pDataParams,
                                     uint8_t ** pRxBuffer,
                                     uint16_t * pRxBufferLen,
                                     uint16_t * pRxBufferSize
                                     )
{
    *pRxBuffer = &pDataParams->pRxBuffer[pDataParams->wRxBufStartPos];
    *pRxBufferSize = pDataParams->wRxBufSize - pDataParams->wRxBufStartPos;
    *pRxBufferLen = pDataParams->wRxBufLen;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ProxiLAB_Int_CheckCrc(
                                  phhalHw_ProxiLAB_DataParams_t * pDataParams,
                                  uint8_t * pRxBuffer, uint16_t *pRxLength
                                  )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint16_t   PH_MEMLOC_REM wTmpLength = *pRxLength;
    uint16_t   PH_MEMLOC_REM wCrc;

    if (wTmpLength < 2)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
    }

    if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
            PH_TOOLS_CRC_OPTION_DEFAULT,
            PH_TOOLS_CRC16_PRESET_ISO14443A,
            PH_TOOLS_CRC16_POLY_ISO14443,
            pRxBuffer,
            wTmpLength,
            &wCrc));

        /* CRC over all data must be zero */
        if (wCrc != 0)
        {
            statusTmp = PH_ERR_INTEGRITY_ERROR;
        }

        /* Remove CRC */
        wTmpLength -= 2;
    }
    else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443B)
    {

        wTmpLength -= 2;

        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
            PH_TOOLS_CRC_OPTION_OUPUT_INVERTED,
            PH_TOOLS_CRC16_PRESET_ISO14443B,
            PH_TOOLS_CRC16_POLY_ISO14443,
            pRxBuffer,
            wTmpLength,
            &wCrc));

        if (((uint8_t)wCrc != pRxBuffer[wTmpLength]) || ((uint8_t)(wCrc>>8) != pRxBuffer[wTmpLength+1]))
        {
            statusTmp = PH_ERR_INTEGRITY_ERROR;
        }
    }
    else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_FELICA)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
            PH_TOOLS_CRC_OPTION_MSB_FIRST,
            PH_TOOLS_CRC16_PRESET_FELICA,
            PH_TOOLS_CRC16_POLY_FELICA,
            pRxBuffer,
            wTmpLength,
            &wCrc));

        /* CRC over all data must be zero */
        if (wCrc != 0)
        {
            statusTmp = PH_ERR_INTEGRITY_ERROR;
        }

        /* Remove CRC */
        wTmpLength -= 2;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Correct received data length in dataparams */
    *pRxLength -= 2;

    return PH_ADD_COMPCODE(statusTmp, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_PROXILAB */
