/*
 * Copyright 2019, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * ProxiLAB Reader specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <phhalHw.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHHAL_HW_PROXILAB

#include "phhalHw_ProxiLAB.h"
#include "phhalHw_ProxiLAB_Int.h"

/* Default shadow for ISO14443-3A Mode */
static const uint16_t PH_MEMLOC_CONST_ROM wProxiLAB_DefaultShadow_14443a[][2] =
{
    /* {PHHAL_HW_CONFIG_PARITY,                PH_ON}, FIXXME not supported */
    {PHHAL_HW_CONFIG_TXCRC,                 PH_OFF},
    {PHHAL_HW_CONFIG_RXCRC,                 PH_OFF},
    /* {PHHAL_HW_CONFIG_RXDEAFBITS,            PH_OFF}, FIXXME not supported */
    {PHHAL_HW_CONFIG_TXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_RXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,      PHHAL_HW_PROXILAB_DEFAULT_TIMEOUT_US}
};

/* Default shadow for ISO14443-3B Mode */
static const uint16_t PH_MEMLOC_CONST_ROM wProxiLAB_DefaultShadow_14443b[][2] =
{
    /* {PHHAL_HW_CONFIG_PARITY,                PH_OFF}, FIXXME not supported */
    {PHHAL_HW_CONFIG_TXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_RXCRC,                 PH_ON},
    /* {PHHAL_HW_CONFIG_RXDEAFBITS,            PH_OFF}, FIXXME not supported */
    {PHHAL_HW_CONFIG_TXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_RXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,      PHHAL_HW_PROXILAB_DEFAULT_TIMEOUT_US}
};

phStatus_t phhalHw_ProxiLAB_Init(
                                 phhalHw_ProxiLAB_DataParams_t * pDataParams,
                                 uint16_t wSizeOfDataParams,
                                 phbalReg_ProxiLAB_DataParams_t * pBalDataParams,
                                 uint8_t * pTxBuffer,
                                 uint16_t wTxBufSize,
                                 uint8_t * pRxBuffer,
                                 uint16_t wRxBufSize
                                 )
{
    if (sizeof(phhalHw_ProxiLAB_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_HAL);
    }

    /* wTxBufSize == 0 is allowed */
    if (wRxBufSize == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalDataParams);
    PH_ASSERT_NULL (pTxBuffer);
    PH_ASSERT_NULL (pRxBuffer);

    /* Init. private data */
    memset(pDataParams, 0x00, sizeof(phhalHw_ProxiLAB_DataParams_t));  /* PRQA S 3200 */
    pDataParams->wId                    = PH_COMP_HAL | PHHAL_HW_PROXILAB_ID;
    pDataParams->pBalDataParams         = pBalDataParams;
    pDataParams->pTxBuffer              = pTxBuffer;
    pDataParams->wTxBufSize             = wTxBufSize;
    pDataParams->wTxBufLen              = 0;
    pDataParams->pRxBuffer              = pRxBuffer;
    pDataParams->wRxBufSize             = wRxBufSize;
    pDataParams->wRxBufLen              = 0;
    pDataParams->wRxBufStartPos         = 0;
    pDataParams->wTxBufStartPos         = 0;
    pDataParams->bCardType              = PHHAL_HW_CARDTYPE_ISO14443A;
    pDataParams->bTimeoutUnit           = PHHAL_HW_TIME_MICROSECONDS;
    /* pDataParams->wTimingMode            = PHHAL_HW_TIMING_MODE_OFF; FIXXME not supported */
    pDataParams->wAdditionalInfo        = 0;
    pDataParams->wFieldOffTime          = PHHAL_HW_FIELD_OFF_DEFAULT;
    pDataParams->wFieldRecoveryTime     = PHHAL_HW_FIELD_RECOVERY_DEFAULT;
    pDataParams->dwPowerLevel           = 1023;
    /* Initialize config shadow */
    memset(pDataParams->wCfgShadow, 0x00, sizeof(pDataParams->wCfgShadow));  /* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ProxiLAB_Exchange(
                                     phhalHw_ProxiLAB_DataParams_t * pDataParams,
                                     uint16_t wOption,
                                     uint8_t * pTxBuffer,
                                     uint16_t wTxLength,
                                     uint8_t ** ppRxBuffer,
                                     uint16_t * pRxLength
                                     )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    phStatus_t  PH_MEMLOC_REM statusExchange;

    uint8_t *   PH_MEMLOC_REM pActTxBuffer;
    uint16_t    PH_MEMLOC_REM wActTxBufferLen;
    uint16_t    PH_MEMLOC_REM wActTxBufferSize;
    uint8_t *   PH_MEMLOC_REM pActRxBuffer;
    uint16_t    PH_MEMLOC_REM wActRxBufferLen;
    uint16_t    PH_MEMLOC_REM wActRxBufferSize;

    /* Check options */
    if (wOption & (uint16_t)~(uint16_t)(PH_EXCHANGE_BUFFERED_BIT | PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* clear internal buffer if requested */
    if (!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        pDataParams->wTxBufLen = 0;
    }

    /* set the receive length */
    if (pRxLength != NULL)
    {
        *pRxLength = 0;
    }

    /* Fill the global TxBuffer */
    if (wOption & PH_EXCHANGE_BUFFERED_BIT || wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT)
    {
        /* retrieve transmit buffer */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ProxiLAB_Int_GetTxBuffer(pDataParams, &pActTxBuffer, &wActTxBufferLen, &wActTxBufferSize));

        if (wTxLength != 0)
        {
            /* TxBuffer overflow check */
            if (wTxLength > (wActTxBufferSize - wActTxBufferLen))
            {
                pDataParams->wTxBufLen = 0;
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
            }

            /* copy data */
            memcpy(&pActTxBuffer[wActTxBufferLen], pTxBuffer, wTxLength);  /* PRQA S 3200 */
            pDataParams->wTxBufLen = pDataParams->wTxBufLen + wTxLength;
         }
    }

    /* Shall we already perform the Exchange? */
    if (wOption & PH_EXCHANGE_BUFFERED_BIT)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    /* retrieve receive buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ProxiLAB_Int_GetRxBuffer(pDataParams, &pActRxBuffer, &wActRxBufferLen, &wActRxBufferSize));
    if (wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT)
    {
        /* retrieve transmit buffer */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ProxiLAB_Int_GetTxBuffer(pDataParams, &pActTxBuffer, &wActTxBufferLen, &wActTxBufferSize));

        /* Perform buffered actual exchange */
        statusExchange = phhalHw_ProxiLAB_Int_Exchange(pDataParams,
            pActTxBuffer,
            wActTxBufferLen,
            wActRxBufferSize,
            pActRxBuffer,
            &wActRxBufferLen);
    }
    else
    {
        /* Perform unbuffered actual exchange */
        statusExchange = phhalHw_ProxiLAB_Int_Exchange(pDataParams,
            pTxBuffer,
            wTxLength,
            wActRxBufferSize,
            pActRxBuffer,
            &wActRxBufferLen);
    }
    if ((statusExchange & PH_ERR_MASK) != PH_ERR_IO_TIMEOUT)
    {
        pDataParams->wRxBufLen = pDataParams->wRxBufStartPos + wActRxBufferLen;
    }

    /* Reset buffered bytes */
    pDataParams->wTxBufLen = 0;

    /* Return RxBuffer pointer */
    if (ppRxBuffer != NULL)
    {
        *ppRxBuffer = pDataParams->pRxBuffer;
    }

    /* Return RxBuffer length */
    if (pRxLength != NULL)
    {
        *pRxLength = pDataParams->wRxBufLen;
    }

    return statusExchange;
}

phStatus_t phhalHw_ProxiLAB_WriteRegister(
    phhalHw_ProxiLAB_DataParams_t * pDataParams,
    uint8_t bAddress,
    uint8_t bValue
    )
{
    /* satisfy compiler */
    if (bValue || bAddress || pDataParams);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
}

phStatus_t phhalHw_ProxiLAB_ReadRegister(
    phhalHw_ProxiLAB_DataParams_t * pDataParams,
    uint8_t bAddress,
    uint8_t * pValue
    )
{
   /* satisfy compiler */
   if (pValue || bAddress || pDataParams);
   return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
}

phStatus_t phhalHw_ProxiLAB_ApplyProtocolSettings(
    phhalHw_ProxiLAB_DataParams_t * pDataParams,
    uint8_t bCardType
    )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint16_t    PH_MEMLOC_COUNT wIndex;
    uint16_t *  PH_MEMLOC_REM pShadowDefault;
    uint16_t    PH_MEMLOC_REM wShadowCount;
    uint8_t     PH_MEMLOC_REM bUseDefaultShadow;
    uint16_t    PH_MEMLOC_REM wConfig;

    /* Store new card type */
    if (bCardType != PHHAL_HW_CARDTYPE_CURRENT)
    {
        pDataParams->bCardType = bCardType;
        bUseDefaultShadow = 1;
        pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MICROSECONDS;
    }
    else
    {
        bUseDefaultShadow = 0;
    }

    /* configure reader IC for current card */
    switch (pDataParams->bCardType)
    {
        /* configure hardware for ISO 14443 */
    case PHHAL_HW_CARDTYPE_ISO14443A:
        /* Use 14443a default shadow */
        pShadowDefault = (uint16_t*)wProxiLAB_DefaultShadow_14443a;
        wShadowCount = sizeof(wProxiLAB_DefaultShadow_14443a) / (sizeof(uint16_t) * 2);
        break;

    case PHHAL_HW_CARDTYPE_ISO14443B:
        /* Use 14443b default shadow */
        pShadowDefault = (uint16_t*)wProxiLAB_DefaultShadow_14443b;
        wShadowCount = sizeof(wProxiLAB_DefaultShadow_14443b) / (sizeof(uint16_t) * 2);
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    /* Copy over default shadow contents into current shadow. */
    if (bUseDefaultShadow)
    {
        /* Initialize config shadow */
        memset(pDataParams->wCfgShadow, 0x00, sizeof(pDataParams->wCfgShadow));  /* PRQA S 3200 */

        for (wIndex = 0; wIndex < wShadowCount; ++wIndex)
        {
            wConfig = pShadowDefault[wIndex << 1];
            pDataParams->wCfgShadow[wConfig] = pShadowDefault[(wIndex << 1) + 1];
        }
    }

    /* Apply shadowed registers */
    for (wIndex = 0; wIndex < wShadowCount; ++wIndex)
    {
        /* Get wConfig */
        wConfig = pShadowDefault[wIndex << 1];

        if (wConfig == PHHAL_HW_CONFIG_TIMEOUT_VALUE_US)
        {
            if (pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MICROSECONDS)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ProxiLAB_SetConfig(pDataParams, wConfig, pDataParams->wCfgShadow[wConfig]));
            }
        }
        else if (wConfig == PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS)
        {
            if (pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MILLISECONDS)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ProxiLAB_SetConfig(pDataParams, wConfig, pDataParams->wCfgShadow[wConfig]));
            }
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ProxiLAB_SetConfig(pDataParams, wConfig, pDataParams->wCfgShadow[wConfig]));
        }
    }

    /* MIFARE Crypto1 state is disabled by default */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams, PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1, PH_ON));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ProxiLAB_SetConfig(
                                      phhalHw_ProxiLAB_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint16_t wValue
                                      )
{
    /* Map to C++ code */
    return phhalHw_ProxiLAB_Int_SetConfig(pDataParams, wConfig, wValue);
}

phStatus_t phhalHw_ProxiLAB_GetConfig(
                                      phhalHw_ProxiLAB_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint16_t * pValue
                                      )
{
    /* Map to C++ code */
    return phhalHw_ProxiLAB_Int_GetConfig(pDataParams, wConfig, pValue);
}

phStatus_t phhalHw_ProxiLAB_FieldOn(
                                    phhalHw_ProxiLAB_DataParams_t * pDataParams
                                    )
{
    /* Map to C++ code */
    return phhalHw_ProxiLAB_Int_FieldOn(pDataParams);
}

phStatus_t phhalHw_ProxiLAB_FieldOff(
                                     phhalHw_ProxiLAB_DataParams_t * pDataParams
                                     )
{
    /* Map to C++ code */
    return phhalHw_ProxiLAB_Int_FieldOff(pDataParams);
}

phStatus_t phhalHw_ProxiLAB_FieldReset(
                                       phhalHw_ProxiLAB_DataParams_t * pDataParams
                                       )
{
    /* Map to C++ code */
    phStatus_t  PH_MEMLOC_REM statusTmp;

    switch (pDataParams->bCardType)
    {
    case PHHAL_HW_CARDTYPE_ISO14443A:
    case PHHAL_HW_CARDTYPE_ISO14443B:
    case PHHAL_HW_CARDTYPE_FELICA:
        /* perfrom RF-Reset */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ProxiLAB_Int_FieldReset(pDataParams));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ProxiLAB_Wait(
                                 phhalHw_ProxiLAB_DataParams_t * pDataParams,
                                 uint8_t bUnit,
                                 uint16_t wTimeout
                                 )
{
    /* Map to C++ code */
    return phhalHw_ProxiLAB_Int_Wait(pDataParams, bUnit, wTimeout);
}

phStatus_t phhalHw_ProxiLAB_MfcAuthenticateKeyNo(
    phhalHw_ProxiLAB_DataParams_t * pDataParams,
    uint8_t bBlockNo,
    uint8_t bKeyType,
    uint16_t wKeyNo,
    uint16_t wKeyVersion,
    uint8_t * pUid
    )
{
    /* satisfy compiler */
    if (bBlockNo|| bKeyType || wKeyNo || wKeyVersion || pUid || pDataParams);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
}

phStatus_t phhalHw_ProxiLAB_MfcAuthenticate(
    phhalHw_ProxiLAB_DataParams_t * pDataParams,
    uint8_t bBlockNo,
    uint8_t bKeyType,
    uint8_t * pKey,
    uint8_t * pUid
    )
{
    /* satisfy compiler */
    if (bBlockNo || bKeyType || pKey || pUid || pDataParams);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_PROXILAB */
