/*
 * Copyright 2017, 2020 , 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * PCSC HAL command layer.
 *
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_RefDefs.h>
#include <ph_Status.h>
#include <phhalHw.h>

#ifdef NXPBUILD__PHHAL_HW_PCSC
#include "phhalHw_Pcsc.h"
#include <phhalHw_Pcsc_Cmd.h>
#include "phhalHw_Pcsc_Int.h"

phStatus_t phhalHw_Pcsc_Cmd_GetData(
                                    phhalHw_Pcsc_DataParams_t * pDataParams,
                                    uint8_t bOption,
                                    uint8_t * pData,
                                    uint16_t * pDataLength
                                    )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wRxBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wRxBufferLen = 0U;
    uint8_t    PH_MEMLOC_REM aCmd[5] = {0U};

    /* Check option */
    if ((bOption != PHHAL_HW_PCSC_UID) && (bOption != PHHAL_HW_PCSC_HIST))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Create Command */
    aCmd[0] = 0xFFU;
    aCmd[1] = 0xCAU;
    aCmd[2] = bOption;
    aCmd[3] = 0x00U;
    aCmd[4] = 0x00U;

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wRxBufferLen,
        &wRxBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        5U,
        wRxBufferSize,
        pRxBuffer,
        &wRxBufferLen));

    /* Validate received data */
    if (wRxBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wRxBufferLen - 2]));

    /* Copy UID / Historical bytes */
    (void)memcpy(pData, pRxBuffer, (wRxBufferLen - 2));

    /* Update length */
    *pDataLength = wRxBufferLen - 2;

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Cmd_LoadKeys(
                                     phhalHw_Pcsc_DataParams_t * pDataParams,
                                     uint8_t bKeyType,
                                     uint8_t bTxType,
                                     uint8_t bReaderEncryptionKeyNo,
                                     uint8_t bKeyLocation,
                                     uint8_t bKeyNumber,
                                     uint8_t * pKey,
                                     uint8_t bKeyLength
                                     )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pTxBuffer = NULL;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wBufferLen = 0U;

    /* Check key type */
    if ((bKeyType != PHHAL_HW_PCSC_KEY_CARD) && (bKeyType != PHHAL_HW_PCSC_KEY_READER))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Check transmission type */
    if ((bTxType != PHHAL_HW_PCSC_TX_PLAIN) && (bTxType != PHHAL_HW_PCSC_TX_SECURED))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Check reader encryption key number */
    if ((bReaderEncryptionKeyNo > 0x0FU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Check key location in reader */
    if ((bKeyLocation != PHHAL_HW_PCSC_KEY_LOC_VOLATILE) && (bKeyLocation != PHHAL_HW_PCSC_KEY_LOC_NONVOLATILE))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Get TX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetTxBuffer(
        pDataParams,
        &pTxBuffer,
        &wBufferLen,
        &wBufferSize));

    /* Check TX buffer size */
    if (wBufferSize < (bKeyLength + 5U))
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    /* Create Command */
    pTxBuffer[0] = 0xFFU;
    pTxBuffer[1] = 0x82U;
    pTxBuffer[2] = (uint8_t)(((uint16_t)bKeyType << 7U) | ((uint16_t)bTxType << 6U) | ((uint16_t)bKeyLocation << 5U) | (bReaderEncryptionKeyNo));
    pTxBuffer[3] = bKeyNumber;
    pTxBuffer[4] = bKeyLength;

    /* Copy Key to TX buffer */
    (void)memcpy(&pTxBuffer[5], pKey, bKeyLength);

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wBufferLen,
        &wBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        pTxBuffer,
        (bKeyLength + 5U),
        wBufferSize,
        pRxBuffer,
        &wBufferLen));

    /* Validate received data */
    if (wBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wBufferLen - 2]));

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Cmd_GeneralAuthenticate(
                                                phhalHw_Pcsc_DataParams_t * pDataParams,
                                                uint8_t bKeyType,
                                                uint8_t bKeyNumber,
                                                uint16_t wAddress
                                                )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wRxBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wRxBufferLen = 0U;
    uint8_t    PH_MEMLOC_REM aCmd[10] = {0U};

    /* CHECK: Check bKeyType. Is this check needed? More key types possible? */
    if ((bKeyType != PHHAL_HW_PCSC_KEY_A) && (bKeyType != PHHAL_HW_PCSC_KEY_B))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Create Command */
    aCmd[0] = 0xFFU;
    aCmd[1] = 0x86U;
    aCmd[2] = 0x00U;
    aCmd[3] = 0x00U;
    aCmd[4] = 0x05U;
    aCmd[5] = 0x01U;
    aCmd[6] = (uint8_t)((wAddress & 0xFF00U) >> 8U);
    aCmd[7] = (uint8_t)(wAddress & 0xFFU);
    aCmd[8] = bKeyType;
    aCmd[9] = bKeyNumber;

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wRxBufferLen,
        &wRxBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        10U,
        wRxBufferSize,
        pRxBuffer,
        &wRxBufferLen));

    /* Validate received data */
    if (wRxBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wRxBufferLen - 2]));

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Cmd_Verify(
                                   phhalHw_Pcsc_DataParams_t * pDataParams,
                                   uint8_t bTxType,
                                   uint8_t bReaderEncryptionKeyNo,
                                   uint8_t bRefData,
                                   uint8_t * pPin,
                                   uint8_t bPinLength
                                   )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pTxBuffer = NULL;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wBufferLen = 0U;

    /* Check transmission type */
    if ((bTxType != PHHAL_HW_PCSC_TX_PLAIN) && (bTxType != PHHAL_HW_PCSC_TX_SECURED))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Check reader encryption key number */
    if ((bReaderEncryptionKeyNo > 0x0FU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Get TX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetTxBuffer(
        pDataParams,
        &pTxBuffer,
        &wBufferLen,
        &wBufferSize));

    /* Check TX buffer size */
    if (wBufferSize < (bPinLength + 5U))
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    /* Create Command */
    pTxBuffer[0] = 0xFFU;
    pTxBuffer[1] = 0x20U;
    pTxBuffer[2] = (uint8_t)(((uint16_t)bTxType << 6U) | (bReaderEncryptionKeyNo));
    pTxBuffer[3] = bRefData;
    pTxBuffer[4] = bPinLength;

    /* Copy Key to TX buffer */
    (void)memcpy(&pTxBuffer[5], pPin, bPinLength);

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wBufferLen,
        &wBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        pTxBuffer,
        (bPinLength + 5U),
        wBufferSize,
        pRxBuffer,
        &wBufferLen));

    /* Validate received data */
    if (wBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wBufferLen - 2]));

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Cmd_ReadBinary(
                                       phhalHw_Pcsc_DataParams_t * pDataParams,
                                       uint16_t wAddress,
                                       uint8_t bBytesToRead,
                                       uint8_t ** ppData,
                                       uint16_t * pDataLength
                                       )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wRxBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wRxBufferLen = 0U;
    uint8_t    PH_MEMLOC_REM aCmd[5] = {0U};

    /* Parameter check */
    if ((ppData == NULL) || (pDataLength == NULL))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Create Command */
    aCmd[0] = 0xFFU;
    aCmd[1] = 0xB0U;
    aCmd[2] = (uint8_t)((wAddress & 0xFF00U) >> 8U);
    aCmd[3] = (uint8_t)(wAddress & 0xFFU);

    /* Note: Extended Le field not implemented */
    aCmd[4] = bBytesToRead;

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wRxBufferLen,
        &wRxBufferSize));

    /* Check RX buffer size */
    if (wRxBufferSize < (bBytesToRead + 2U))
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        5U,
        wRxBufferSize,
        pRxBuffer,
        &wRxBufferLen));

    /* Validate received data */
    if (wRxBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wRxBufferLen - 2]));

    /* Received bytes */
    *ppData = pRxBuffer;

    /* Update received length */
    *pDataLength = wRxBufferLen - 2;

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Cmd_UpdateBinary(
                                         phhalHw_Pcsc_DataParams_t * pDataParams,
                                         uint16_t wAddress,
                                         uint8_t * pData,
                                         uint8_t bDataLength
                                         )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pTxBuffer = NULL;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wBufferLen = 0U;

    /* Parameter check */
    if (pData == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Get TX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetTxBuffer(
        pDataParams,
        &pTxBuffer,
        &wBufferLen,
        &wBufferSize));

    /* Check TX buffer size */
    if (wBufferSize < (bDataLength + 5U))
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    /* Create Command */
    pTxBuffer[0] = 0xFFU;
    pTxBuffer[1] = 0xD6U;
    pTxBuffer[2] = (uint8_t)((wAddress & 0xFF00U) >> 8U);
    pTxBuffer[3] = (uint8_t)(wAddress & 0xFFU);
    pTxBuffer[4] = bDataLength;

    /* Copy data to TX buffer */
    (void)memcpy(&pTxBuffer[5], pData, bDataLength);

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wBufferLen,
        &wBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        pTxBuffer,
        (bDataLength + 5U),
        wBufferSize,
        pRxBuffer,
        &wBufferLen));

    /* Validate received data */
    if (wBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wBufferLen - 2]));

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Cmd_Increment(
                                      phhalHw_Pcsc_DataParams_t * pDataParams,
                                      uint8_t bBlockNo,
                                      uint8_t * pValue
                                      )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wRxBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wRxBufferLen = 0U;
    uint8_t    PH_MEMLOC_REM aCmd[17] = {0U};

    /* Parameter check */
    if ((pValue == NULL))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Create Command */
    aCmd[0] = 0xFFU;
    aCmd[1] = 0xC2U;
    aCmd[2] = 0x00U;
    aCmd[3] = 0x03U;

    /* Lc */
    aCmd[4] = 0x0BU;

    /* Increment Tag */
    aCmd[5] = 0xA0U;
    aCmd[6] = 0x09U;

    /* Block Tag */
    aCmd[7] = 0x80U;
    aCmd[8] = 0x01U;
    aCmd[9] = bBlockNo;

    /* Value Tag */
    aCmd[10] = 0x81U;
    aCmd[11] = 0x04U;
    aCmd[12] = pValue[0];
    aCmd[13] = pValue[1];
    aCmd[14] = pValue[2];
    aCmd[15] = pValue[3];

    /* Le */
    aCmd[16] = 0x00U;

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wRxBufferLen,
        &wRxBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        17U,
        wRxBufferSize,
        pRxBuffer,
        &wRxBufferLen));

    /* Validate received data */
    if (wRxBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wRxBufferLen - 2]));

    /* Check response data objects */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckResponse(
        pDataParams,
        pRxBuffer,
        (wRxBufferLen - 2),
        NULL,
        NULL));

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Cmd_Decrement(
                                      phhalHw_Pcsc_DataParams_t * pDataParams,
                                      uint8_t bBlockNo,
                                      uint8_t * pValue
                                      )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wRxBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wRxBufferLen = 0U;
    uint8_t    PH_MEMLOC_REM aCmd[17] = {0U};

    /* Parameter check */
    if ((pValue == NULL))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Create Command */
    aCmd[0] = 0xFFU;
    aCmd[1] = 0xC2U;
    aCmd[2] = 0x00U;
    aCmd[3] = 0x03U;

    /* Lc */
    aCmd[4] = 0x0BU;

    /* Increment Tag */
    aCmd[5] = 0xA1U;
    aCmd[6] = 0x09U;

    /* Block Tag */
    aCmd[7] = 0x80U;
    aCmd[8] = 0x01U;
    aCmd[9] = bBlockNo;

    /* Value Tag */
    aCmd[10] = 0x81U;
    aCmd[11] = 0x04U;
    aCmd[12] = pValue[0];
    aCmd[13] = pValue[1];
    aCmd[14] = pValue[2];
    aCmd[15] = pValue[3];

    /* Le */
    aCmd[16] = 0x00U;

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wRxBufferLen,
        &wRxBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        17U,
        wRxBufferSize,
        pRxBuffer,
        &wRxBufferLen));

    /* Validate received data */
    if (wRxBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wRxBufferLen - 2]));

    /* Check response data objects */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckResponse(
        pDataParams,
        pRxBuffer,
        (wRxBufferLen - 2),
        NULL,
        NULL));

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Cmd_StartTransparentSession(
                                                    phhalHw_Pcsc_DataParams_t * pDataParams
                                                    )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wRxBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wRxBufferLen = 0U;
    uint8_t    PH_MEMLOC_REM bIndex = 0U;
    uint8_t    PH_MEMLOC_REM aCmd[20] = {0U};

    /* Create Command */
    aCmd[bIndex++] = 0xFFU;
    aCmd[bIndex++] = 0xC2U;
    aCmd[bIndex++] = 0x00U;
    aCmd[bIndex++] = 0x00U;

    if(pDataParams->dwTimeoutUs)
    {
        /* Lc */
        aCmd[bIndex++] = 0x09U;
    }
    else
    {
        /* Lc */
        aCmd[bIndex++] = 0x02U;
    }

    /* Tag */
    aCmd[bIndex++] = 0x81U;

    /* Length */
    aCmd[bIndex++] = 0x00U;

    if(pDataParams->dwTimeoutUs)
    {
        /* Timer TLV Tag */
        aCmd[bIndex++] = 0x5FU;
        aCmd[bIndex++] = 0x46U;

        /* Timer TLV Length */
        aCmd[bIndex++] = 0x04U;

        /* Timer TLV Value */
        aCmd[bIndex++] = (uint8_t)((pDataParams->dwTimeoutUs & 0xFF000000U) >> 24U);
        aCmd[bIndex++] = (uint8_t)((pDataParams->dwTimeoutUs & 0xFF0000U) >> 16U);
        aCmd[bIndex++] = (uint8_t)((pDataParams->dwTimeoutUs & 0xFF00U) >> 8U);
        aCmd[bIndex++] = (uint8_t)(pDataParams->dwTimeoutUs & 0xFFU);
    }

    /* Le */
    aCmd[bIndex++] = 0x00U;

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wRxBufferLen,
        &wRxBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        bIndex,
        wRxBufferSize,
        pRxBuffer,
        &wRxBufferLen));

    /* Validate received data */
    if (wRxBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wRxBufferLen - 2]));

    /* Check response data objects */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckResponse(
        pDataParams,
        pRxBuffer,
        (wRxBufferLen - 2),
        NULL,
        NULL));

    /* Update transparent session flag */
    pDataParams->bTransparentSession = PH_ON;

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Cmd_EndTransparentSession(
                                                  phhalHw_Pcsc_DataParams_t * pDataParams
                                                  )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wRxBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wRxBufferLen = 0U;
    uint8_t    PH_MEMLOC_REM aCmd[8] = {0U};

    /* Create Command */
    aCmd[0] = 0xFFU;
    aCmd[1] = 0xC2U;
    aCmd[2] = 0x00U;
    aCmd[3] = 0x00U;

    /* Lc */
    aCmd[4] = 0x02U;

    /* Tag */
    aCmd[5] = 0x82U;

    /* Length */
    aCmd[6] = 0x00U;

    /* Le */
    aCmd[7] = 0x00U;

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wRxBufferLen,
        &wRxBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        8U,
        wRxBufferSize,
        pRxBuffer,
        &wRxBufferLen));

    /* Validate received data */
    if (wRxBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wRxBufferLen - 2]));

    /* Check response data objects */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckResponse(
        pDataParams,
        pRxBuffer,
        (wRxBufferLen - 2),
        NULL,
        NULL));

    /* Update transparent session flag */
    pDataParams->bTransparentSession = PH_OFF;

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Cmd_Version(
                                    phhalHw_Pcsc_DataParams_t * pDataParams,
                                    uint8_t * pVersion
                                    )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wRxBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wRxBufferLen = 0U;
    uint8_t    PH_MEMLOC_REM aCmd[11] = {0U};
    uint8_t *  PH_MEMLOC_REM pOutData = NULL;

    /* Parameter check */
    if (pVersion == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Create Command */
    aCmd[0] = 0xFFU;
    aCmd[1] = 0xC2U;
    aCmd[2] = 0x00U;
    aCmd[3] = 0x00U;

    /* Lc */
    aCmd[4] = 0x05U;

    /* Tag */
    aCmd[5] = 0x80U;

    /* Length */
    aCmd[6] = 0x03U;

    /* Value */
    aCmd[7] = 0x00U;
    aCmd[8] = 0x00U;
    aCmd[9] = 0x00U;

    /* Le */
    aCmd[10] = 0x00U;

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wRxBufferLen,
        &wRxBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        11U,
        wRxBufferSize,
        pRxBuffer,
        &wRxBufferLen));

    /* Validate received data */
    if (wRxBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wRxBufferLen - 2]));

    /* Check response data objects */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckResponse(
        pDataParams,
        pRxBuffer,
        (wRxBufferLen - 2),
        &pOutData,
        NULL));

    /* Copy version */
    (void)memcpy(pVersion, pOutData, 3U);

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Cmd_RfOff(
                                  phhalHw_Pcsc_DataParams_t * pDataParams
                                  )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wRxBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wRxBufferLen = 0U;
    uint8_t    PH_MEMLOC_REM aCmd[8] = {0U};

    /* Is Transparent session on */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_IsTransparentSession(
        pDataParams));

    /* Create Command */
    aCmd[0] = 0xFFU;
    aCmd[1] = 0xC2U;
    aCmd[2] = 0x00U;
    aCmd[3] = 0x00U;

    /* Lc */
    aCmd[4] = 0x02U;

    /* Tag */
    aCmd[5] = 0x83U;

    /* Length */
    aCmd[6] = 0x00U;

    /* Le */
    aCmd[7] = 0x00U;

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wRxBufferLen,
        &wRxBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        8U,
        wRxBufferSize,
        pRxBuffer,
        &wRxBufferLen));

    /* Validate received data */
    if (wRxBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wRxBufferLen - 2]));

    /* Check response data objects */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckResponse(
        pDataParams,
        pRxBuffer,
        (wRxBufferLen - 2),
        NULL,
        NULL));

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Cmd_RfOn(
                                 phhalHw_Pcsc_DataParams_t * pDataParams
                                 )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wRxBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wRxBufferLen = 0U;
    uint8_t    PH_MEMLOC_REM aCmd[8] = {0U};

    /* Is Transparent session on */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_IsTransparentSession(
        pDataParams));

    /* Create Command */
    aCmd[0] = 0xFFU;
    aCmd[1] = 0xC2U;
    aCmd[2] = 0x00U;
    aCmd[3] = 0x00U;

    /* Lc */
    aCmd[4] = 0x02U;

    /* Tag */
    aCmd[5] = 0x84U;

    /* Length */
    aCmd[6] = 0x00U;

    /* Le */
    aCmd[7] = 0x00U;

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wRxBufferLen,
        &wRxBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        8U,
        wRxBufferSize,
        pRxBuffer,
        &wRxBufferLen));

    /* Validate received data */
    if (wRxBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wRxBufferLen - 2]));

    /* Check response data objects */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckResponse(
        pDataParams,
        pRxBuffer,
        (wRxBufferLen - 2),
        NULL,
        NULL));

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Cmd_RfReset(
                                    phhalHw_Pcsc_DataParams_t * pDataParams
                                    )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wRxBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wRxBufferLen = 0U;
    uint8_t    PH_MEMLOC_REM aCmd[17] = {0U};
    uint32_t   PH_MEMLOC_REM dwFieldOffTime = 0U;

    /* Is Transparent session on */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_IsTransparentSession(
        pDataParams));

    /* Field off time in microseconds */
    dwFieldOffTime = pDataParams->wFieldOffTime * 1000U;

    /* Create Command */
    aCmd[0] = 0xFFU;
    aCmd[1] = 0xC2U;
    aCmd[2] = 0x00U;
    aCmd[3] = 0x00U;

    /* Lc */
    aCmd[4] = 0x0BU;

    /* RF OFF TLV */
    aCmd[5] = 0x83U;
    aCmd[6] = 0x00U;

    /* Timer TLV */
    aCmd[7]  = 0x5FU;
    aCmd[8]  = 0x46U;
    aCmd[9]  = 0x04U;
    aCmd[10] = (uint8_t)((dwFieldOffTime & 0xFF000000U) >> 24U);
    aCmd[11] = (uint8_t)((dwFieldOffTime & 0xFF0000U) >> 16U);
    aCmd[12] = (uint8_t)((dwFieldOffTime & 0xFF00U) >> 8U);
    aCmd[13] = (uint8_t)(dwFieldOffTime & 0xFFU);

    /* RF ON TLV */
    aCmd[14] = 0x84U;
    aCmd[15] = 0x00U;

    /* Le */
    aCmd[16] = 0x00U;

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wRxBufferLen,
        &wRxBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        17U,
        wRxBufferSize,
        pRxBuffer,
        &wRxBufferLen));

    /* Validate received data */
    if (wRxBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wRxBufferLen - 2]));

    /* Check response data objects */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckResponse(
        pDataParams,
        pRxBuffer,
        (wRxBufferLen - 2),
        NULL,
        NULL));

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Cmd_GetParam(
                                     phhalHw_Pcsc_DataParams_t * pDataParams,
                                     uint8_t bParamType,
                                     uint8_t * pParamValue,
                                     uint8_t * pParamLength
                                     )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wRxBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wRxBufferLen = 0U;
    uint8_t    PH_MEMLOC_REM aCmd[11] = {0U};
    uint8_t *  PH_MEMLOC_REM pOutData = NULL;
    uint16_t   PH_MEMLOC_REM wOutDataLength = 0U;

    /* Parameter check */
    if ((pParamValue == NULL) || (pParamLength == NULL) || (bParamType > 0x0B))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Create Command */
    aCmd[0] = 0xFFU;
    aCmd[1] = 0xC2U;
    aCmd[2] = 0x00U;
    aCmd[3] = 0x00U;

    /* Lc */
    aCmd[4] = 0x05U;

    /* Tag */
    aCmd[5] = 0xFFU;
    aCmd[6] = 0x6DU;

    /* Length */
    aCmd[7] = 0x02U;

    /* Value */
    aCmd[8] = bParamType;   /* Tag */
    aCmd[9] = 0x00U;        /* Length */

    /* Le */
    aCmd[10] = 0x00U;

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wRxBufferLen,
        &wRxBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        11U,
        wRxBufferSize,
        pRxBuffer,
        &wRxBufferLen));

    /* Validate received data */
    if (wRxBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wRxBufferLen - 2]));

    /* Check response data objects */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckResponse(
        pDataParams,
        pRxBuffer,
        (wRxBufferLen - 2),
        &pOutData,
        &wOutDataLength));

    /* Copy parameter value */
    (void)memcpy(pParamValue, pOutData, wOutDataLength);

    /* Update paremeter length */
    *pParamLength = (uint8_t)wOutDataLength;

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Cmd_SetParam(
                                     phhalHw_Pcsc_DataParams_t * pDataParams,
                                     uint8_t bParamType,
                                     uint8_t * pParamValue,
                                     uint8_t bParamLength
                                     )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wRxBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wRxBufferLen = 0U;
    uint8_t    PH_MEMLOC_REM aCmd[15] = {0U};

    /* Parameter check */
    if ((pParamValue == NULL) || (bParamType > 0x0B))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Check length */
    if ((bParamLength != 4U) && (bParamLength != 1U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Create Command */
    aCmd[0] = 0xFFU;
    aCmd[1] = 0xC2U;
    aCmd[2] = 0x00U;
    aCmd[3] = 0x00U;

    /* Lc */
    aCmd[4] = 0x05U + bParamLength;

    /* Tag */
    aCmd[5] = 0xFFU;
    aCmd[6] = 0x6EU;

    /* Length */
    aCmd[7] = 0x02U + bParamLength;

    /* Value */
    aCmd[8] = bParamType;                               /* Tag */
    aCmd[9] = bParamLength;                             /* Length */
    (void)memcpy(&aCmd[10], pParamValue, bParamLength);  /* Value */

    /* Le */
    aCmd[10U + bParamLength] = 0x00U;

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wRxBufferLen,
        &wRxBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        (11U + bParamLength),
        wRxBufferSize,
        pRxBuffer,
        &wRxBufferLen));

    /* Validate received data */
    if (wRxBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wRxBufferLen - 2]));

    /* Check response data objects */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckResponse(
        pDataParams,
        pRxBuffer,
        (wRxBufferLen - 2),
        NULL,
        NULL));

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Cmd_TxRxFlag(
                                     phhalHw_Pcsc_DataParams_t * pDataParams,
                                     uint16_t wValue
                                     )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wRxBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wRxBufferLen = 0U;
    uint8_t    PH_MEMLOC_REM aCmd[10] = {0U};

    /* Bit 5 - RFU */
    if (wValue & 0x20U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Create Command */
    aCmd[0] = 0xFFU;
    aCmd[1] = 0xC2U;
    aCmd[2] = 0x00U;
    aCmd[3] = 0x01U;

    /* Lc */
    aCmd[4] = 0x04U;

    /* Tag */
    aCmd[5] = 0x90U;

    /* Length */
    aCmd[6] = 0x02U;

    /* Value */
    aCmd[7] = (uint8_t)((wValue & 0xFF00U) >> 8U);
    aCmd[8] = (uint8_t)(wValue & 0xFFU);

    /* Le */
    aCmd[9] = 0x00U;

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wRxBufferLen,
        &wRxBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        10U,
        wRxBufferSize,
        pRxBuffer,
        &wRxBufferLen));

    /* Validate received data */
    if (wRxBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wRxBufferLen - 2]));

    /* Check response data objects */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckResponse(
        pDataParams,
        pRxBuffer,
        (wRxBufferLen - 2),
        NULL,
        NULL));

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Cmd_Transceive(
                                       phhalHw_Pcsc_DataParams_t * pDataParams,
                                       uint16_t wOption,
                                       uint8_t * pTxData,
                                       uint16_t wTxLength,
                                       uint8_t ** ppRxData,
                                       uint16_t * pRxLength
                                       )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pTxBuffer = NULL;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wBufferLen = 0U;
    uint8_t *  PH_MEMLOC_REM pOutData = NULL;
    uint16_t   PH_MEMLOC_REM wOutDataLength = 0U;
    uint8_t    PH_MEMLOC_REM bTxBitsLen = 0U;
    uint16_t   PH_MEMLOC_REM wIndex = 0U;
    uint16_t   PH_MEMLOC_REM wBufferStart = 0U;
    uint16_t   PH_MEMLOC_REM wLc = 0U;
    uint8_t    PH_MEMLOC_REM bExtendedLc = 0U;
    uint16_t   PH_MEMLOC_REM wCmdLength = 0U;
    uint16_t   PH_MEMLOC_REM wHeaderLength = 0U;

    /* Reset RX last bits */
     pDataParams->bRxLastBits = 0U;

    /* Check exchange option */
    if (wOption & (uint16_t)~(uint16_t)(PH_EXCHANGE_BUFFERED_BIT | PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    if ((wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT) == 0x00U)
    {
        /* Reset length field. */
        pDataParams->wTxBufLen = 0U;
    }

    /* Reset the receive length */
    if (pRxLength != NULL)
    {
        *pRxLength = 0;
    }

   /* Retrieve transmit buffer parameters */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetTxBuffer(
        pDataParams,
        &pTxBuffer,
        &wBufferLen,
        &wBufferSize));

    /* Check if there is some data to transmit */
    if (wTxLength != 0U)
    {
        /* Check buffer size (data starts from byte "30") */
        if (wBufferSize < (wTxLength + wBufferLen + 30U))
        {
            /* Can't fit in the HAL buffer */
            pDataParams->wTxBufLen = 0U;

            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* Copy data to TX buffer (data starts from byte "30") */
        (void)memcpy(&pTxBuffer[30 + wBufferLen], pTxData, wTxLength);

        /* Increment TX buffer length */
        pDataParams->wTxBufLen += wTxLength;
    }

    /* Return with success, if only buffering is requested. */
    if (wOption & PH_EXCHANGE_BUFFERED_BIT)
    {
        return PH_ERR_SUCCESS;
    }

    /* Set timeout of FWI = 8U. */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_SetConfig(
        pDataParams,
        PHHAL_HW_PCSC_CONFIG_FWTI,
        0x08U));

    /* Check if Transmission Bit Framing Data Object is needed */
    /*if(pDataParams->bTxLastBits)*/  /* Figured out that uTrust 3700F will keep the last value that was used and did not use all bits valid if absent, so send it always */
    {
        bTxBitsLen = 3U;
    }

    if(pDataParams->dwTimeoutUs)
    {
        bTxBitsLen += 7U;
    }

    /* Get Lc value */
    (void)phhalHw_Pcsc_Int_GetLcValue(
        &wLc,
        (wTxLength + wBufferLen),
        bTxBitsLen);

    /* Calculate command length */
    wCmdLength = (wLc > 0x00FFU) ? (wLc + 9U) : (wLc + 6U);

    /* Bytes count before TX data start position */
    wHeaderLength = (wLc > 0x00FFU) ? (wLc - (wTxLength + wBufferLen) + 7U) : (wLc - (wTxLength + wBufferLen) + 5U);

    /* Buffer start position */
    wBufferStart = 30U - wHeaderLength;

    /* Check TX buffer size */
    if (wBufferSize < (wBufferStart + wCmdLength))
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    /* Initialize TX buffer index */
    wIndex = wBufferStart;

    /* Create Command */
    pTxBuffer[wIndex++] = 0xFFU;
    pTxBuffer[wIndex++] = 0xC2U;
    pTxBuffer[wIndex++] = 0x00U;
    pTxBuffer[wIndex++] = 0x01U;

    /* Lc */
    if (wLc > 0xFFU)
    {
        /* Extended Lc format */
        pTxBuffer[wIndex++] = 0x00U;
        pTxBuffer[wIndex++] = (uint8_t)((wLc & 0xFF00U) >> 8U);
        pTxBuffer[wIndex++] = (uint8_t)(wLc & 0xFFU);

        /* Set Extended Lc flag */
        bExtendedLc = 1;
    }
    else
    {
        pTxBuffer[wIndex++] = (uint8_t)wLc;

        /* Reset Extended Lc flag */
        bExtendedLc = 0;
    }

    if(pDataParams->dwTimeoutUs)
    {
        /* Timer TLV Tag */
        pTxBuffer[wIndex++] = 0x5FU;
        pTxBuffer[wIndex++] = 0x46U;

        /* Timer TLV Length */
        pTxBuffer[wIndex++] = 0x04U;

        /* Timer TLV Value */
        pTxBuffer[wIndex++] = (uint8_t)((pDataParams->dwTimeoutUs & 0xFF000000U) >> 24U);
        pTxBuffer[wIndex++] = (uint8_t)((pDataParams->dwTimeoutUs & 0xFF0000U) >> 16U);
        pTxBuffer[wIndex++] = (uint8_t)((pDataParams->dwTimeoutUs & 0xFF00U) >> 8U);
        pTxBuffer[wIndex++] = (uint8_t)(pDataParams->dwTimeoutUs & 0xFFU);
    }

    /* Add Transmission Bit framing TLV if set. */
    /*if(pDataParams->bTxLastBits)*/ /* Figured out that uTrust 3700F will keep the last value that was used and did not use all bits valid if absent, so send it always */
    {
        /* Transmission Bit Framing TLV Tag */
        pTxBuffer[wIndex++] = 0x91U;

        /* Transmission Bit Framing TLV Length */
        pTxBuffer[wIndex++] = 0x01U;

        /* Transmission Bit Framing TLV Value */
        pTxBuffer[wIndex++] = pDataParams->bTxLastBits;
    }

    /* Transceive TLV Tag */
    pTxBuffer[wIndex++] = 0x95U;

    /* Transceive TLV Length */
    if ((wTxLength + wBufferLen) <= 0x7FU)
    {
        /* BER - TLV one byte length format */
        pTxBuffer[wIndex++] = (uint8_t)(wTxLength + wBufferLen);
    }
    else if (((wTxLength + wBufferLen) > 0x7FU) && ((wTxLength + wBufferLen) <= 0xFFU))
    {
        /* BER - TLV two byte length format */
        pTxBuffer[wIndex++] = 0x81U;
        pTxBuffer[wIndex++] = (uint8_t)(wTxLength + wBufferLen);
    }
    else
    {
        /* BER - TLV three byte length format */
        pTxBuffer[wIndex++] = 0x82U;
        pTxBuffer[wIndex++] = (uint8_t)(((wTxLength + wBufferLen) & 0xFF00U) >> 8U);
        pTxBuffer[wIndex++] = (uint8_t)((wTxLength + wBufferLen) & 0xFFU);
    }

    /* Adjust index (including data count) */
    wIndex = 30U + wBufferLen + wTxLength;

    /* Le */
    if (bExtendedLc)
    {
        /* Extended Le */
        pTxBuffer[wIndex++] = 0x00U;
        pTxBuffer[wIndex++] = 0x00U;
    }
    else
    {
        pTxBuffer[wIndex++] = 0x00U;
    }

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wBufferLen,
        &wBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        &pTxBuffer[wBufferStart],
        wCmdLength,
        wBufferSize,
        pRxBuffer,
        &wBufferLen));

    /* Validate received data */
    if (wBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wBufferLen - 2]));

    /* Check response data objects */
    statusTmp = phhalHw_Pcsc_Int_CheckResponse(
        pDataParams,
        pRxBuffer,
        (wBufferLen - 2),
        &pOutData,
        &wOutDataLength);

    if (pDataParams->bRxLastBits != 0x00U)
    {
        statusTmp = (PH_ERR_SUCCESS_INCOMPLETE_BYTE | PH_COMP_HAL);
    }

    if (((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS_INCOMPLETE_BYTE) ||
        ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS))
    {
        /* Copy the received data from PICC by overwriting the PCSC response received from
           PCSC reader. For both PAL and AL Rx Chaining to work correctly. */
        memcpy(pRxBuffer, pOutData, wOutDataLength);

        /* Return HAL Rx Buffer pointer. */
        *ppRxData = pDataParams->pRxBuffer;

        /* Received data length plus previous Rx Buff start position. */
        *pRxLength = (uint8_t)wOutDataLength + pDataParams->wRxBufStartPos;
    }

    return statusTmp;
}

phStatus_t phhalHw_Pcsc_Cmd_SwitchProtocol(
                                           phhalHw_Pcsc_DataParams_t * pDataParams,
                                           uint8_t bProtocol,
                                           uint8_t bLayer
                                           )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wRxBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wRxBufferLen = 0U;
    uint8_t    PH_MEMLOC_REM aCmd[10] = {0U};

    /* Is Transparent session on */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_IsTransparentSession(
        pDataParams));

    /* Check Parameters - Only 14443 / 15693 supported */
    if ((bProtocol > 0x02U) || (bLayer > 0x04U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Create Command */
    aCmd[0] = 0xFFU;
    aCmd[1] = 0xC2U;
    aCmd[2] = 0x00U;
    aCmd[3] = 0x02U;

    /* Lc */
    aCmd[4] = 0x04U;

    /* Tag */
    aCmd[5] = 0x8FU;

    /* Length */
    aCmd[6] = 0x02U;

    /* Value */
    aCmd[7] = bProtocol;
    aCmd[8] = bLayer;

    /* Le */
    aCmd[9] = 0x00U;

    /* Get RX buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
        pDataParams,
        &pRxBuffer,
        &wRxBufferLen,
        &wRxBufferSize));

    /* Exchange the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        10U,
        wRxBufferSize,
        pRxBuffer,
        &wRxBufferLen));

    /* Validate received data */
    if (wRxBufferLen < 2U)
    {
        /* Invalid response length */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* Check for error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
        pDataParams,
        &pRxBuffer[wRxBufferLen - 2]));

    /* Check response data objects */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckResponse(
        pDataParams,
        pRxBuffer,
        (wRxBufferLen - 2),
        NULL,
        NULL));

    return PH_ERR_SUCCESS;
}

#endif /* NXPBUILD__PHHAL_HW_PCSC */
