/*
 * Copyright 2013, 2015, 2017, 2019 - 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Mp300 specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <phhalHw.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHHAL_HW_MP300

#include "phhalHw_Mp300.h"
#include "phhalHw_Mp300_Int.h"
#include <stdio.h>              /* PRQA S 5124 */

/* Default shadow for ISO14443-3A Mode */
static const uint16_t PH_MEMLOC_CONST_ROM wMp300_DefaultShadow_14443a[][2] =
{
    {PHHAL_HW_CONFIG_PARITY,                PH_ON},
    {PHHAL_HW_CONFIG_TXCRC,                 PH_OFF},
    {PHHAL_HW_CONFIG_RXCRC,                 PH_OFF},
    {PHHAL_HW_CONFIG_RXDEAFBITS,            0x0008},
    {PHHAL_HW_CONFIG_TXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_RXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_ASK100,                PH_ON},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,      0},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,      PHHAL_HW_MP300_DEFAULT_TIMEOUT_US}
};

/* Default shadow for ISO14443-3B Mode */
static const uint16_t PH_MEMLOC_CONST_ROM wMp300_DefaultShadow_14443b[][2] =
{
    {PHHAL_HW_CONFIG_PARITY,                PH_OFF},
    {PHHAL_HW_CONFIG_TXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_RXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_RXDEAFBITS,            0x0008},
    {PHHAL_HW_CONFIG_TXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_RXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_MODINDEX,              PHHAL_HW_MP300_MODINDEX_I14443B},
    {PHHAL_HW_CONFIG_ASK100,                PH_OFF},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,      0},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,      PHHAL_HW_MP300_DEFAULT_TIMEOUT_US}
};

/* Default shadow for Felica Mode */
static const uint16_t PH_MEMLOC_CONST_ROM wMp300_DefaultShadow_Felica[][2] =
{
    {PHHAL_HW_CONFIG_PARITY,                PH_OFF},
    {PHHAL_HW_CONFIG_TXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_RXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_RXDEAFBITS,            0x0000},
    {PHHAL_HW_CONFIG_TXDATARATE,            PHHAL_HW_RF_DATARATE_212},
    {PHHAL_HW_CONFIG_RXDATARATE,            PHHAL_HW_RF_DATARATE_212},
    {PHHAL_HW_CONFIG_MODINDEX,              PHHAL_HW_MP300_MODINDEX_FELICA},
    {PHHAL_HW_CONFIG_ASK100,                PH_OFF},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,      0},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,      PHHAL_HW_MP300_DEFAULT_TIMEOUT_US}
};

/* Default shadow for ISO15693 Mode */
static const uint16_t wMp300_DefaultShadow_15693[][2] =
{
    {PHHAL_HW_CONFIG_PARITY,                PH_OFF},
    {PHHAL_HW_CONFIG_TXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_RXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_RXDEAFBITS,            0x0008},
    {PHHAL_HW_CONFIG_TXDATARATE,            PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4},
    {PHHAL_HW_CONFIG_RXDATARATE,            PHHAL_HW_RF_RX_DATARATE_LOW},
    {PHHAL_HW_CONFIG_SUBCARRIER,            PHHAL_HW_SUBCARRIER_SINGLE},
    {PHHAL_HW_CONFIG_ASK100,                PH_ON},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,      0},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,      PHHAL_HW_MP300_DEFAULT_TIMEOUT_US}
};

phStatus_t phhalHw_Mp300_Init(
                              phhalHw_Mp300_DataParams_t * pDataParams,
                              uint16_t wSizeOfDataParams,
                              void * pBalDataParams,
                              uint8_t * pTxBuffer,
                              uint16_t wTxBufSize,
                              uint8_t * pRxBuffer,
                              uint16_t wRxBufSize,
                              uint8_t * pIntBuffer,
                              uint16_t wIntBufferSize
                              )
{
    if (sizeof(phhalHw_Mp300_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_HAL);
    }
    /* wTxBufSize == 0 is allowed */
    if (wRxBufSize == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalDataParams);
    PH_ASSERT_NULL (pTxBuffer);
    PH_ASSERT_NULL (pRxBuffer);
    PH_ASSERT_NULL (pIntBuffer);

    /* Init. private data */
    pDataParams->wId                    = PH_COMP_HAL | PHHAL_HW_MP300_ID;
    pDataParams->pBalDataParams         = pBalDataParams;
    pDataParams->pTxBuffer              = pTxBuffer;
    pDataParams->wTxBufSize             = wTxBufSize;
    pDataParams->wTxBufLen              = 0;
    pDataParams->pRxBuffer              = pRxBuffer;
    pDataParams->wRxBufSize             = wRxBufSize;
    pDataParams->wRxBufLen              = 0;
    pDataParams->wRxBufStartPos         = 0;
    pDataParams->wTxBufStartPos         = 0;
    pDataParams->bCardType              = PHHAL_HW_CARDTYPE_ISO14443A;
    pDataParams->wTimingMode            = PHHAL_HW_TIMING_MODE_OFF;
    pDataParams->bTimeoutUnit           = PHHAL_HW_TIME_MICROSECONDS;
    pDataParams->qwTiming_ns            = 0;
    pDataParams->qwCommunicationTimePCDtoPICCCyclesAll = 0;
    pDataParams->qwCommunicationTimePICCtoPCDCyclesAll = 0;
    pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent = 0;
    pDataParams->qwCommunicationTimePICCtoPCDCyclesCurrent = 0;
    pDataParams->bLastBit               = 0;
    pDataParams->wAdditionalInfo        = 0;
    pDataParams->wFieldOffTime          = PHHAL_HW_FIELD_OFF_DEFAULT;
    pDataParams->wFieldRecoveryTime     = PHHAL_HW_FIELD_RECOVERY_DEFAULT;
    pDataParams->bSymbolStart           = PH_OFF;
    pDataParams->bSymbolEnd             = PH_OFF;
    pDataParams->wCurrentFieldStrength  = 1000;
    pDataParams->bRfResetAfterTo        = PH_OFF;
    pDataParams->bRfAfterTearing        = PHHAL_HW_MP300_TEARING_FIELD_ON;
    pDataParams->wTearingExchangeCounter = 0;
    pDataParams->pIntBuffer             = pIntBuffer;
    pDataParams->wIntBufferSize         = wIntBufferSize;
    pDataParams->wIntBufferLen          = 0x00;
    pDataParams->wIntBufferReadPos      = 0x00;
    pDataParams->dwCarrierFrequency     = 13560000;
    pDataParams->bRxGain                = 100;
    pDataParams->bRXChannel             = MP300_RXCHANNEL_NORMAL;
    pDataParams->bCouplerNumber         = MP_COUPLER_1;
    pDataParams->bRelay                 = PH_OFF;
    pDataParams->bModulationMode        = (PH_OFF << 4) | PHHAL_HW_MP300_LINEAR_MODULATION;
    pDataParams->wFieldRiseTime         = 0;
    pDataParams->wFallTime              = 0;
    pDataParams->wRiseTime              = 0;
    pDataParams->wPauseWidth            = 2100;
    pDataParams->bSourceOfCarrier       = PHHAL_HW_MP300_CARRIER_INTERNAL;
    pDataParams->bSubCarrierMode        = PHHAL_HW_SUBCARRIER_SINGLE;
    pDataParams->SpyThreadHandle        = 0;
    pDataParams->GEvtReady              = 0;
    pDataParams->GEvtFinished           = 0;

    memset(pDataParams->bSpyFilename, 0x00, sizeof(pDataParams->bSpyFilename)); /* PRQA S 3200 */
    memset(pDataParams->pIntBuffer, 0x00, wIntBufferSize); /* PRQA S 3200 */

    /* Data Rate = 106 */
    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_SOF_LOW_CYCLES] = (uint16_t)(10*128);
    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_SOF_HIGH_CYCLES] = (uint16_t)(2*128);
    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_EGT_CYCLES] = (uint16_t)(0*128);
    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_EOF_CYCLES] = (uint16_t)(10*128);
    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_START_BIT_CYCLES] = (uint16_t)(1*128);
    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_0_CYCLES] = (uint16_t)(1*128);
    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_1_CYCLES] = (uint16_t)(1*128);
    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_2_CYCLES] = (uint16_t)(1*128);
    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_3_CYCLES] = (uint16_t)(1*128);
    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_4_CYCLES] = (uint16_t)(1*128);
    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_5_CYCLES] = (uint16_t)(1*128);
    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_6_CYCLES] = (uint16_t)(1*128);
    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_7_CYCLES] = (uint16_t)(1*128);
    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_STOP_BIT_CYCLES] = (uint16_t)(1*128);

    pDataParams->wCouplerResourceId = NULL;
    pDataParams->bAntiEMD = PH_OFF;
    pDataParams->bForceFdtMethod1 = PH_OFF;
    /* Set to invalid value */
    pDataParams->wTargetRxGainLevel = 0xFFFF;

    pDataParams->bSerialNumber[0] = 0;
    pDataParams->bDeviceFamily[0] = 0;
    pDataParams->bDeviceReferenceWithinFamily[0] = 0;
    pDataParams->bMotherboardSystemVersion[0] = 0;
    pDataParams->bMotherboardBootVersion[0] = 0;
    pDataParams->bCouplerName[0] = 0;
    pDataParams->bCouplerSerialNumber[0] = 0;
    pDataParams->bCouplerDriverVersion[0] = 0;
    pDataParams->wCouplerFPGAVersion = 0;

    pDataParams->dwScenarioID = 0;
    pDataParams->dwScenarioPcdId = 0;

    pDataParams->pBalTimeoutHandling = NULL;
    pDataParams->pBalTimeoutHandlingContext = NULL;

    pDataParams->wTriggerExchangeCounters[MP300_TRIGGER_PORT_1 - 1] = 0;
    pDataParams->wTriggerExchangeCounters[MP300_TRIGGER_PORT_2 - 1] = 0;
    pDataParams->wTriggerExchangeCounters[MP300_TRIGGER_PORT_3 - 1] = 0;
    pDataParams->wTriggerExchangeCounters[MP300_TRIGGER_PORT_4 - 1] = 0;

    memset(pDataParams->dwTriggerConfigs, 0x00, sizeof(pDataParams->dwTriggerConfigs)); /* PRQA S 3200 */
    memset(pDataParams->dwTriggerValues, 0x00, sizeof(pDataParams->dwTriggerValues)); /* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_Exchange(
                                  phhalHw_Mp300_DataParams_t * pDataParams,
                                  uint16_t wOption,
                                  uint8_t * pTxBuffer,
                                  uint16_t wTxLength,
                                  uint8_t ** ppRxBuffer,
                                  uint16_t * pRxLength
                                  )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    phStatus_t  PH_MEMLOC_REM statusDisableTrigger;
    phStatus_t  PH_MEMLOC_REM statusTime;
    phStatus_t  PH_MEMLOC_REM statusExchange;

    uint8_t *   PH_MEMLOC_REM pTmpBuffer;
    uint16_t    PH_MEMLOC_REM wTmpBufferLen;
    uint16_t    PH_MEMLOC_REM wTmpBufferSize;
    uint16_t    PH_MEMLOC_REM wTransmitLen;

    uint8_t     PH_MEMLOC_REM pCrc[2];

    uint16_t    PH_MEMLOC_REM wNumTransmitBits;
    uint16_t    PH_MEMLOC_REM wNumReceivedBits = 0;

    uint32_t    PH_MEMLOC_REM dwPosColl;

    uint32_t    PH_MEMLOC_REM dwTmp;
    uint8_t     PH_MEMLOC_COUNT bLoop;

    /* Check options */
    if (wOption & (uint16_t)~(uint16_t)(PH_EXCHANGE_BUFFERED_BIT | PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* clear internal buffer if requested */
    if (!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        pDataParams->wTxBufLen = 0;
    }

    /* set the receive length */
    if (pRxLength != NULL)
    {
        *pRxLength = 0;
    }

    /* Fill the global TxBuffer */
    if (wOption & PH_EXCHANGE_BUFFERED_BIT)
    {
        /* retrieve transmit buffer */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_GetTxBuffer(pDataParams, &pTmpBuffer, &wTmpBufferLen, &wTmpBufferSize));

        if (wTxLength != 0)
        {
            /* TxBuffer overflow check */
            if (wTxLength > (wTmpBufferSize - wTmpBufferLen))
            {
                pDataParams->wTxBufLen = 0;
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
            }

            /* copy data */
            memcpy(&pTmpBuffer[wTmpBufferLen], pTxBuffer, wTxLength);  /* PRQA S 3200 */
            pDataParams->wTxBufLen = pDataParams->wTxBufLen + wTxLength;
        }

        /* Buffer operation -> finished */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    /* retrieve transmit buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_GetTxBuffer(pDataParams, &pTmpBuffer, &wTmpBufferLen, &wTmpBufferSize));

    /* Calc size of data to transmit */
    if (wTxLength + wTmpBufferLen > 0xFFFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }
    wTransmitLen = wTxLength + wTmpBufferLen;

    /* Suppress SOF (send isolated EOF) is only supported when no data is send */
    if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO15693 && pDataParams->bSymbolStart == PH_OFF && wTransmitLen != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Add CRC */
    if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXCRC] == PH_ON)
    {
        if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
        }
        /* check overflow */
        if (wTransmitLen + 2 > 0xFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
        wTransmitLen += 2;

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_CalcCrc(pDataParams, pTxBuffer, wTxLength, pCrc));
    }

    /* Special handling if an empty frame is send with suppressed SOF a isolated EOF should be send, this requires bytes to send = 0
     * See MPC_ExchangeCmdVicinity doku of Micropross:
     * If NbByteTx is set to 0, an isolated EOF is generated. */
    if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO15693 && pDataParams->bSymbolStart == PH_OFF)
    {
        wTransmitLen = 0;
    }

    /* Timing */
    if (wTransmitLen > 0 && pDataParams->wTimingMode != PHHAL_HW_TIMING_MODE_OFF)
    {
        PH_CHECK_SUCCESS_FCT(statusTime, phhalHw_Mp300_SetTimingParameters(pDataParams, MP300_TIMING_CLEAR));
    }

    /* check max transmit len */
    if (wTransmitLen > 4096)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    /* Calculate number of bits transmitted */
    wNumTransmitBits = wTransmitLen*8;
    if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] != 0)
    {
        if (wNumTransmitBits < 8)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
        }
        wNumTransmitBits = wNumTransmitBits + pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] - 8;
        if ( wTransmitLen != 0 ) /* Should everytime be true */
        {
            wTransmitLen = wTransmitLen - 1;
        }
    }

    /* Calculate time for transmitting the command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_CalcCommTimePCDToPICC(pDataParams, wNumTransmitBits, wTransmitLen));

    /* Check if tearing should be enabled for current exchange */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_EnableTearing(pDataParams));

    /* Check if any trigger port should be enabled for this exchange, this must be done at this point because the IntBuffer is used to build the exchange command */
    PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_Int_EnableTrigger(pDataParams), phhalHw_Mp300_Int_CleanupTearingAndTrigger(pDataParams));

    /* In case any of the next commands would return the trigger or tearing needs to be disabled again so use cleanup function
       This needs to be performed until the exchange response is extracted. */

    /* Build Exchange command */
    if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO15693)
    {
        PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_EXCHANGE_ISO15693), phhalHw_Mp300_Int_CleanupTearingAndTrigger(pDataParams));
    }
    else
    {
        PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_SetCommand(pDataParams, MP300_CMD_EXCHANGE), phhalHw_Mp300_Int_CleanupTearingAndTrigger(pDataParams));
    }

    PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_AddParam_Uint8(pDataParams, pDataParams->bCouplerNumber), phhalHw_Mp300_Int_CleanupTearingAndTrigger(pDataParams));

    if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO15693)
    {
        PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_AddParam_Uint16(pDataParams, wTransmitLen), phhalHw_Mp300_Int_CleanupTearingAndTrigger(pDataParams));
    }
    else
    {
        PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_AddParam_Uint16(pDataParams, wNumTransmitBits), phhalHw_Mp300_Int_CleanupTearingAndTrigger(pDataParams));
    }

    PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_AddParam_HexString(pDataParams, pTmpBuffer, wTmpBufferLen), phhalHw_Mp300_Int_CleanupTearingAndTrigger(pDataParams));
    PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_CatParam_HexString(pDataParams, pTxBuffer, wTxLength), phhalHw_Mp300_Int_CleanupTearingAndTrigger(pDataParams));
    if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXCRC] == PH_ON)
    {
        PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_CatParam_HexString(pDataParams, pCrc, 2), phhalHw_Mp300_Int_CleanupTearingAndTrigger(pDataParams));
    }

    /* Calculate last bit */
    /* Get last byte */
    pTmpBuffer = &pDataParams->pIntBuffer[pDataParams->wIntBufferLen-2];
    sscanf((char*)pTmpBuffer,"%02x",&dwTmp);

    if (wNumTransmitBits % 8 != 0 )
    {
        /* Last byte not full so no parity is added */
        pDataParams->bLastBit = (dwTmp>>((wNumTransmitBits % 8)-1))&0x01;
    }
    else if ( pDataParams->wCfgShadow[PHHAL_HW_CONFIG_PARITY] == PH_OFF)
    {
        /* No Parity is used but last byte uses 8 bit */
        pDataParams->bLastBit = (dwTmp>>7)&0x01;
    }
    else /* Last bit is full and also add parity */
    {
        pDataParams->bLastBit = 0x01;

        for (bLoop = 0; bLoop < 8; ++bLoop)
        {
            if (dwTmp & (uint8_t)(1 << bLoop))
            {
                ++pDataParams->bLastBit;
            }
        }
        pDataParams->bLastBit = pDataParams->bLastBit & 0x01;
    }

    statusExchange = phhalHw_Mp300_SendCommand(pDataParams);

    if (statusExchange != PH_ERR_SUCCESS)
    {
        phhalHw_Mp300_Int_CleanupTearingAndTrigger(pDataParams);
        return statusExchange;
    }

    /* Reset TxLastBits */
    if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] != 0)
    {
        /* This can be done because there is no exchange with the reader so the IntBuffer is not overwritten */
        PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_SetConfig(pDataParams, PHHAL_HW_CONFIG_TXLASTBITS, 0x00), phhalHw_Mp300_Int_CleanupTearingAndTrigger(pDataParams));
    }

    /* Reset buffered bytes */
    pDataParams->wTxBufLen = 0;

    /* get the return status */
    if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO15693)
    {
        statusExchange = phhalHw_Mp300_GetStatus(pDataParams,MP300_CMD_EXCHANGE_ISO15693);
    }
    else
    {
        statusExchange = phhalHw_Mp300_GetStatus(pDataParams,MP300_CMD_EXCHANGE);
    }

    /* Extract Data before any other action is done because buffer is reused (only everything is ok or only a collision occurs)*/
    if ( ((statusExchange & PH_ERR_MASK) == PH_ERR_SUCCESS) ||
            ( (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A) &&
              (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE] == PHHAL_HW_RF_DATARATE_106) &&
              ((statusExchange & PH_ERR_MASK) == PH_ERR_COLLISION_ERROR)
            )
        )
    {
        /* get number of received bits */
        PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_GetParam_Uint16(pDataParams, &wNumReceivedBits), phhalHw_Mp300_Int_CleanupTearingAndTrigger(pDataParams));

        if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO15693)
        {
            /* for ISO15693 micropross exchange command parameter is sending and receiven bytes instead of bits */
            wNumReceivedBits *= 8;
        }

        /* Reset receive buffer length */
        pDataParams->wRxBufLen = 0;

        /* Retrieve receive buffer properties */
        PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_GetRxBuffer(pDataParams, &pTmpBuffer, &wTmpBufferLen, &wTmpBufferSize), phhalHw_Mp300_Int_CleanupTearingAndTrigger(pDataParams));

        /* Add the 2 reserved bytes for the CRC */
        wTmpBufferSize += 2;

        /* Extract the received data */
        PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phhalHw_Mp300_GetParam_HexString(pDataParams, pTmpBuffer, wTmpBufferSize, &wTmpBufferLen), phhalHw_Mp300_Int_CleanupTearingAndTrigger(pDataParams));
    }

    /* Now all data has be extracted, trigger and tearing can be disabled */
    statusDisableTrigger = phhalHw_Mp300_Int_DisableTrigger(pDataParams);
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_DisableTearing(pDataParams, &statusExchange));
    PH_CHECK_SUCCESS(statusDisableTrigger);

    /* Extract Data only everything is ok or only a collision occurs*/
    if ( ((statusExchange & PH_ERR_MASK) == PH_ERR_SUCCESS) ||
            ( (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A) &&
              (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE] == PHHAL_HW_RF_DATARATE_106) &&
              ((statusExchange & PH_ERR_MASK) == PH_ERR_COLLISION_ERROR)
            )
        )
    {
        /* Calulate time for transmitting the response */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_CalcCommTimePICCToPCD(pDataParams, wNumReceivedBits));

        if ( pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXALIGN] != 0 )
        {
            wNumReceivedBits = wNumReceivedBits + pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXALIGN];
        }

        /* Check received bit count match with extraced data */
        if ((wNumReceivedBits+7)>>3 != wTmpBufferLen)
            return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);

        /* If a collision was detected get the collision position */
        if ( (statusExchange & PH_ERR_MASK) == PH_ERR_COLLISION_ERROR )
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_GetPosColl(pDataParams, &dwPosColl));

            if ( pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXALIGN] != 0 )
            {
                dwPosColl = dwPosColl + pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXALIGN];
            }

            if ( pDataParams->wCfgShadow[PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL] == PH_ON )
            {
                /* Resize the number of bytes received to the collision position */
                if ( dwPosColl <= wNumReceivedBits && dwPosColl != 0 )
                {
                    wNumReceivedBits = (uint16_t)(dwPosColl-1);
                    wTmpBufferLen = (wNumReceivedBits+7)>>3;

                    /* Clear bits after collision */
                    pTmpBuffer[wTmpBufferLen-1] = pTmpBuffer[wTmpBufferLen-1]&(uint8_t)((2<<((wNumReceivedBits-1)&7))-1);
                }
            }
        }
        else
        {
            dwPosColl = 0;
        }

        /* Store received data length in dataparams */
        pDataParams->wRxBufLen = pDataParams->wRxBufStartPos + wTmpBufferLen;

        /* Check and remove CRC in software if required */
        if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXCRC] == PH_ON)
        {
            if (wNumReceivedBits&0x07)
            {
                return PH_ADD_COMPCODE(PH_ERR_FRAMING_ERROR, PH_COMP_HAL);
            }
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_CheckCrc(pDataParams));
        }

        /* Set RxLastBits */
        pDataParams->wAdditionalInfo = wNumReceivedBits&0x07;

        /* If collision */
        if ( (statusExchange & PH_ERR_MASK) == PH_ERR_COLLISION_ERROR && dwPosColl != 0 )
        {
            statusTmp = PH_ERR_COLLISION_ERROR;
        }
        /* Set incomplete byte status if applicable */
        else if (pDataParams->wAdditionalInfo != 0x00)
        {
            statusTmp = PH_ERR_SUCCESS_INCOMPLETE_BYTE;
        }
        else
        {
            statusTmp = PH_ERR_SUCCESS;
        }
    }
    else if ((statusExchange & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        /* Store received data length in dataparams */
        pDataParams->wRxBufLen = pDataParams->wRxBufStartPos + 0;

        /* If timeout then the response time is Zero */
        pDataParams->qwCommunicationTimePICCtoPCDCyclesCurrent = 0;
        statusTmp = PH_ERR_IO_TIMEOUT;
    }
    else
    {
        /* Just delegate error */
        statusTmp = statusExchange;
    }

    /* Return RxBuffer pointer */
    if (ppRxBuffer != NULL)
    {
        *ppRxBuffer = pDataParams->pRxBuffer;
    }

    /* Return RxBuffer length */
    if (pRxLength != NULL)
    {
        *pRxLength = pDataParams->wRxBufLen;
    }

    /* Timing */
    if (pDataParams->wTimingMode != PHHAL_HW_TIMING_MODE_OFF)
    {
        if (pDataParams->bForceFdtMethod1 == PH_OFF  && pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A && wNumReceivedBits < 8)
        {
            if ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS || (statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS_INCOMPLETE_BYTE)
            {
                return PH_ADD_COMPCODE(PHHAL_HW_MP300_ERR_FDT_METHODE2_NOT_SUPPORTED, PH_COMP_HAL);
            }
            else
            {
                return PH_ADD_COMPCODE(statusTmp, PH_COMP_HAL);
            }
        }
        PH_CHECK_SUCCESS_FCT(statusTime, phhalHw_Mp300_SetTimingParameters(pDataParams, MP300_TIMING_GET));
    }

    return PH_ADD_COMPCODE(statusTmp, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_WriteRegister(
                                       phhalHw_Mp300_DataParams_t * pDataParams,
                                       uint8_t bAddress,
                                       uint8_t bValue
                                       )
{
    if (pDataParams || bAddress || bValue);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_ReadRegister(
                                      phhalHw_Mp300_DataParams_t * pDataParams,
                                      uint8_t bAddress,
                                      uint8_t * pValue
                                      )
{
    if (pDataParams || bAddress || pValue);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_ApplyProtocolSettings(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint8_t bCardType
    )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint16_t    PH_MEMLOC_COUNT wIndex;
    uint16_t *  PH_MEMLOC_REM pShadowDefault;
    uint16_t    PH_MEMLOC_REM wShadowCount;
    uint8_t     PH_MEMLOC_REM bUseDefaultShadow;
    uint16_t    PH_MEMLOC_REM wConfig;

    /* If resource not is open open it */
    if( pDataParams->wCouplerResourceId == NULL )
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_OpenResource(pDataParams));
        /* Set default Rx Gain of not TCL3 */
        if (pDataParams->wCouplerResourceId != PHHAL_HW_MP300_RESSOURCE_TCL3)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams, PHHAL_HW_MP300_CONFIG_RX_GAIN_TARGET, 750));
        }
    }

    /* Store new card type */
    if (bCardType != PHHAL_HW_CARDTYPE_CURRENT)
    {
        pDataParams->bCardType = bCardType;
        bUseDefaultShadow = 1;
        pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MICROSECONDS;
    }
    else
    {
        bUseDefaultShadow = 0;
    }

    /* configure reader IC for current card */
    switch (pDataParams->bCardType)
    {
        /* configure hardware for ISO 14443 */
    case PHHAL_HW_CARDTYPE_ISO14443A:
        /* Use 14443a default shadow */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetCardType(pDataParams, MP300_TYPE_A));
        /* Pause Width is set to 2100 if Select Card Type A */
        pDataParams->wPauseWidth = 2100;
        /* Fall and Rise Time are set to zero if a Select Card is done */
        pDataParams->wRiseTime = 0;
        pDataParams->wRiseTime = 0;
        pShadowDefault = (uint16_t*)wMp300_DefaultShadow_14443a;
        wShadowCount = sizeof(wMp300_DefaultShadow_14443a) / (sizeof(uint16_t) * 2);
        break;

    case PHHAL_HW_CARDTYPE_ISO14443B:
        /* Use 14443b default shadow */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetCardType(pDataParams, MP300_TYPE_B));
        /* Fall and Rise Time are set to zero if a Select Card is done */
        pDataParams->wRiseTime = 0;
        pDataParams->wRiseTime = 0;

        /* Data Rate = 106 */
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_SOF_LOW_CYCLES] = (uint16_t)(10*128);
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_SOF_HIGH_CYCLES] = (uint16_t)(2*128);
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_EGT_CYCLES] = (uint16_t)(0*128);
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_EOF_CYCLES] = (uint16_t)(10*128);
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_START_BIT_CYCLES] = (uint16_t)(1*128);
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_0_CYCLES] = (uint16_t)(1*128);
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_1_CYCLES] = (uint16_t)(1*128);
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_2_CYCLES] = (uint16_t)(1*128);
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_3_CYCLES] = (uint16_t)(1*128);
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_4_CYCLES] = (uint16_t)(1*128);
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_5_CYCLES] = (uint16_t)(1*128);
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_6_CYCLES] = (uint16_t)(1*128);
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_7_CYCLES] = (uint16_t)(1*128);
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_STOP_BIT_CYCLES] = (uint16_t)(1*128);

        pShadowDefault = (uint16_t*)wMp300_DefaultShadow_14443b;
        wShadowCount = sizeof(wMp300_DefaultShadow_14443b) / (sizeof(uint16_t) * 2);
        break;

    case PHHAL_HW_CARDTYPE_FELICA:
        /* Use Felica default shadow */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetCardType(pDataParams, MP300_TYPE_FELICA));
        /* Fall and Rise Time are set to zero if a Select Card is done */
        pDataParams->wRiseTime = 0;
        pDataParams->wRiseTime = 0;
        pShadowDefault = (uint16_t*)wMp300_DefaultShadow_Felica;
        wShadowCount = sizeof(wMp300_DefaultShadow_Felica) / (sizeof(uint16_t) * 2);
        break;

    case PHHAL_HW_CARDTYPE_ISO15693:
        /* Use ISO15693 default shadow */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetCardType(pDataParams, MP300_TYPE_ISO15693));
        /* Fall and Rise Time are set to zero if a Select Card is done */
        pDataParams->wRiseTime = 0;
        pDataParams->wRiseTime = 0;
        pShadowDefault = (uint16_t*)wMp300_DefaultShadow_15693;
        wShadowCount = sizeof(wMp300_DefaultShadow_15693) / (sizeof(uint16_t) * 2);

        /* Set default SOF */
        pDataParams->bSymbolStart = PHHAL_HW_SYMBOL_I15693_SOF;

        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    /* Copy over default shadow contents into current shadow. */
    if (bUseDefaultShadow)
    {
        /* Initialize config shadow */
        memset(pDataParams->wCfgShadow, 0x00, sizeof(pDataParams->wCfgShadow));  /* PRQA S 3200 */

        for (wIndex = 0; wIndex < wShadowCount; ++wIndex)
        {
            wConfig = pShadowDefault[wIndex << 1];
            pDataParams->wCfgShadow[wConfig] = pShadowDefault[(wIndex << 1) + 1];
        }
    }

    /* Apply shadowed registers */
    for (wIndex = 0; wIndex < wShadowCount; ++wIndex)
    {
        /* Get wConfig */
        wConfig = pShadowDefault[wIndex << 1];

        if (wConfig == PHHAL_HW_CONFIG_TIMEOUT_VALUE_US)
        {
            if (pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MICROSECONDS)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetConfig(pDataParams, wConfig, pDataParams->wCfgShadow[wConfig]));
            }
        }
        else if (wConfig == PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS)
        {
            if (pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MILLISECONDS)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetConfig(pDataParams, wConfig, pDataParams->wCfgShadow[wConfig]));
            }
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetConfig(pDataParams, wConfig, pDataParams->wCfgShadow[wConfig]));
        }
    }

    /* MIFARE Crypto1 state is disabled by default */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams, PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1, PH_ON));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_SetConfig(
                                   phhalHw_Mp300_DataParams_t * pDataParams,
                                   uint16_t wConfig,
                                   uint16_t wValue
                                   )
{

    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t PH_MEMLOC_REM bTmp;
    uint16_t PH_MEMLOC_REM wTmp;
    uint32_t PH_MEMLOC_REM dwTmp;
    float32_t PH_MEMLOC_REM fDriverVersion;

    switch (wConfig)
    {
    case PHHAL_HW_CONFIG_PARITY:
        /* Update the flags (enabled is disabled in this case) */
        if (wValue == PH_OFF)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_ActivateParity(pDataParams, PH_OFF));
        }
        else if (wValue == PH_ON)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_ActivateParity(pDataParams, PH_ON));
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_TXCRC:
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_RXCRC:
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_TXLASTBITS:
        /* check parameter */
        if (wValue > 7)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_RXALIGN:
        /* check parameter */
        if (wValue > 7)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* The Rx Align is done automatic with the last bits */
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_RXDEAFBITS:
        /* check parameter, maximum 10 seconds */
        if ((((float)wValue*128.0f)/13.56f) > 10000000.0f)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* check parameter, minimum 30 microseconds or zero*/
        if ((((float)wValue*128.0f)/13.56f) < 30.0f && wValue != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_ConfigRxDeafbits(pDataParams, wValue));

        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL:
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_TXDATARATE:
        if ((pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A) ||
            (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443B))
        {
            switch (wValue)
            {
            case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_256:
            case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4:
            case PHHAL_HW_RF_RX_DATARATE_LOW:
            case PHHAL_HW_RF_RX_DATARATE_HIGH:
            case PHHAL_HW_RF_RX_DATARATE_FAST_LOW:
            case PHHAL_HW_RF_RX_DATARATE_FAST_HIGH:
                return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);

            case PHHAL_HW_RF_DATARATE_1695:
            case PHHAL_HW_RF_DATARATE_3390:
            case PHHAL_HW_RF_DATARATE_6780:
                if (pDataParams->bCardType != PHHAL_HW_CARDTYPE_ISO14443B)
                {
                    return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
                }
            case PHHAL_HW_RF_DATARATE_106:
            case PHHAL_HW_RF_DATARATE_212:
            case PHHAL_HW_RF_DATARATE_424:
            case PHHAL_HW_RF_DATARATE_848:
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetDatarate(pDataParams, wValue, pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE]));

                if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443B)
                {
                    switch (wValue)
                    {
                    case PHHAL_HW_RF_DATARATE_106:
                        bTmp = 128;
                        break;
                    case PHHAL_HW_RF_DATARATE_212:
                        bTmp = 64;
                        break;
                    case PHHAL_HW_RF_DATARATE_424:
                        bTmp = 32;
                        break;
                    case PHHAL_HW_RF_DATARATE_848:
                        bTmp = 16;
                        break;
                    case PHHAL_HW_RF_DATARATE_1695:
                        bTmp = 8;
                        break;
                    case PHHAL_HW_RF_DATARATE_3390:
                        bTmp = 4;
                        break;
                    case PHHAL_HW_RF_DATARATE_6780:
                        bTmp = 2;
                        break;
                    default:
                        bTmp = 128;
                    }
                    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_SOF_LOW_CYCLES] = (uint16_t)(10*bTmp);
                    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_SOF_HIGH_CYCLES] = (uint16_t)(2*bTmp);
                    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_EGT_CYCLES] = (uint16_t)(0*bTmp);
                    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_EOF_CYCLES] = (uint16_t)(10*bTmp);
                    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_START_BIT_CYCLES] = (uint16_t)(1*bTmp);
                    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_0_CYCLES] = (uint16_t)(1*bTmp);
                    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_1_CYCLES] = (uint16_t)(1*bTmp);
                    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_2_CYCLES] = (uint16_t)(1*bTmp);
                    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_3_CYCLES] = (uint16_t)(1*bTmp);
                    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_4_CYCLES] = (uint16_t)(1*bTmp);
                    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_5_CYCLES] = (uint16_t)(1*bTmp);
                    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_6_CYCLES] = (uint16_t)(1*bTmp);
                    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_7_CYCLES] = (uint16_t)(1*bTmp);
                    pDataParams->wTypeBTimingParameter[MP300_TYPE_B_STOP_BIT_CYCLES] = (uint16_t)(1*bTmp);
                }
                break;

            default:
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_FELICA)
        {
            switch (wValue)
            {
            case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_256:
            case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4:
            case PHHAL_HW_RF_RX_DATARATE_LOW:
            case PHHAL_HW_RF_RX_DATARATE_HIGH:
            case PHHAL_HW_RF_RX_DATARATE_FAST_LOW:
            case PHHAL_HW_RF_RX_DATARATE_FAST_HIGH:
            case PHHAL_HW_RF_DATARATE_106:
            case PHHAL_HW_RF_DATARATE_848:
                return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);

            case PHHAL_HW_RF_DATARATE_212:
            case PHHAL_HW_RF_DATARATE_424:
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetDatarate(pDataParams, wValue, wValue));
                pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE] = wValue;
                break;

            default:
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO15693)
        {
            switch (wValue)
            {
            case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_256:
            case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4:
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetDatarate_ISO15693(pDataParams, wValue, pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE]));
                break;

            default:
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_RXDATARATE:
        if ((pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A) ||
            (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443B))
        {
            switch (wValue)
            {
            case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_256:
            case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4:
            case PHHAL_HW_RF_RX_DATARATE_LOW:
            case PHHAL_HW_RF_RX_DATARATE_HIGH:
            case PHHAL_HW_RF_RX_DATARATE_FAST_LOW:
            case PHHAL_HW_RF_RX_DATARATE_FAST_HIGH:
                return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);

            case PHHAL_HW_RF_DATARATE_1695:
            case PHHAL_HW_RF_DATARATE_3390:
            case PHHAL_HW_RF_DATARATE_6780:
                if( pDataParams->wCouplerResourceId != PHHAL_HW_MP300_RESSOURCE_TCL3 )
                {
                    return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
                }
            case PHHAL_HW_RF_DATARATE_106:
            case PHHAL_HW_RF_DATARATE_212:
            case PHHAL_HW_RF_DATARATE_424:
            case PHHAL_HW_RF_DATARATE_848:
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetDatarate(pDataParams, pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXDATARATE], wValue));
                break;

            default:
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_FELICA)
        {
            switch (wValue)
            {
            case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_256:
            case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4:
            case PHHAL_HW_RF_RX_DATARATE_LOW:
            case PHHAL_HW_RF_RX_DATARATE_HIGH:
            case PHHAL_HW_RF_RX_DATARATE_FAST_LOW:
            case PHHAL_HW_RF_RX_DATARATE_FAST_HIGH:
            case PHHAL_HW_RF_DATARATE_106:
            case PHHAL_HW_RF_DATARATE_848:
                return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);

            case PHHAL_HW_RF_DATARATE_212:
            case PHHAL_HW_RF_DATARATE_424:
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetDatarate(pDataParams, pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXDATARATE], wValue));
                pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXDATARATE] = wValue;
                break;

            default:
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO15693)
        {
            switch (wValue)
            {
            case PHHAL_HW_RF_DATARATE_106: /* X4 */
            case PHHAL_HW_RF_DATARATE_212: /* X8 */
            case PHHAL_HW_RF_RX_DATARATE_FAST_HIGH: /* X2 */
                if (pDataParams->wCouplerResourceId != PHHAL_HW_MP300_RESSOURCE_TCL3)
                {
                    return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
                }
                if (sscanf((const char *)pDataParams->bCouplerDriverVersion, "%f", &fDriverVersion) != 1)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
                }
                /* New in version 1.11: High Data Rate X2 and X4 */
                /* New in version 1.17: High Data Rate X8 */
                if ((fDriverVersion < 1.11) || (wValue == PHHAL_HW_RF_DATARATE_212 && fDriverVersion < 1.17))
                {
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
                }
            case PHHAL_HW_RF_RX_DATARATE_LOW:
            case PHHAL_HW_RF_RX_DATARATE_HIGH:
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetDatarate_ISO15693(pDataParams, pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXDATARATE], wValue));
                break;

            default:
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_MODINDEX:
        /* check parameter, maximum 100 percent */
        if (wValue > 100)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetModulationASK(pDataParams, (uint8_t)wValue));
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_ASK100:
        /* switch on 100% ASK */
        if (wValue != PH_OFF)
        {
            /* Setting MP300 to 100% */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetModulationASK(pDataParams, 100));
        }
        /* switch off 100% ASK */
        else
        {
            /* Setting MP300 to value set with PHHAL_HW_CONFIG_MODINDEX */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetModulationASK(pDataParams, (uint8_t)pDataParams->wCfgShadow[PHHAL_HW_CONFIG_MODINDEX]));
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_US:
    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS:
        /* Calculate values for Microsecond values */
        if (wConfig == PHHAL_HW_CONFIG_TIMEOUT_VALUE_US)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTimeout(pDataParams, (uint32_t)wValue));
            pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MICROSECONDS;
        }
        /* Calculate values for Millisecond values */
        else
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTimeout(pDataParams, (uint32_t)(wValue * 1000)));
            pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MILLISECONDS;
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_TIMING_MODE:
        /* Check supported option bits */
        switch (wValue & PHHAL_HW_TIMING_MODE_OPTION_MASK)
        {
        case PHHAL_HW_TIMING_MODE_OPTION_DEFAULT:
        case PHHAL_HW_TIMING_MODE_OPTION_AUTOCLEAR:
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Check supported timing modes */
        switch (wValue & (uint16_t)~(uint16_t)PHHAL_HW_TIMING_MODE_OPTION_MASK)
        {
        case PHHAL_HW_TIMING_MODE_OFF:
            pDataParams->qwTiming_ns = 0;
            pDataParams->qwCommunicationTimePCDtoPICCCyclesAll = 0;
            pDataParams->qwCommunicationTimePICCtoPCDCyclesAll = 0;
            pDataParams->qwCommunicationTimePCDtoPICCCyclesCurrent = 0;
            pDataParams->qwCommunicationTimePICCtoPCDCyclesCurrent = 0;

            if ((pDataParams->wTimingMode & (uint16_t)~(uint16_t)PHHAL_HW_TIMING_MODE_OPTION_MASK) !=
                                  (wValue & (uint16_t)~(uint16_t)PHHAL_HW_TIMING_MODE_OPTION_MASK))
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTimingParameters(pDataParams, MP300_TIMING_DISABLE));
            }
            pDataParams->wTimingMode = wValue;
            break;

        case PHHAL_HW_TIMING_MODE_FDT:
        case PHHAL_HW_TIMING_MODE_COMM:
            pDataParams->qwTiming_ns = 0;
            if ((pDataParams->wTimingMode & (uint16_t)~(uint16_t)PHHAL_HW_TIMING_MODE_OPTION_MASK) ==
                                  PHHAL_HW_TIMING_MODE_OFF)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTimingParameters(pDataParams, MP300_TIMING_ENABLE));
            }
            pDataParams->wTimingMode = wValue;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        break;

    case PHHAL_HW_CONFIG_FIELD_OFF_TIME:
        /* Parameter Check, MP support only up to 21 seconds for SCN_DO_RF_RESET */
        if (wValue == 0 || wValue > 21000)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Store config data */
        pDataParams->wFieldOffTime = wValue;
        break;

    case PHHAL_HW_CONFIG_FIELD_RECOVERY_TIME:
        /* Store config data */
        pDataParams->wFieldRecoveryTime = wValue;
        break;

    case PHHAL_HW_CONFIG_SYMBOL_START:
        if (pDataParams->bCardType != PHHAL_HW_CARDTYPE_ISO15693)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
        }
        if (wValue != PHHAL_HW_SYMBOL_I15693_SOF && wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        pDataParams->bSymbolStart = (uint8_t)wValue;
        break;

    case PHHAL_HW_CONFIG_SYMBOL_END:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);

    case PHHAL_HW_CONFIG_SUBCARRIER:
        if (wValue > PHHAL_HW_SUBCARRIER_QUAD)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        if (wValue > PHHAL_HW_SUBCARRIER_DUAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
        }
        pDataParams->bSubCarrierMode = (uint8_t)wValue;
        /* Apply value will be done togeher with datarate */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetDatarate_ISO15693(pDataParams, pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXDATARATE], pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE]));
        break;

    case PHHAL_HW_CONFIG_RXBUFFER_STARTPOS:
        /* Boundary check */
        if (wValue >= (pDataParams->wRxBufSize - PHHAL_HW_MP300_CRC_BUFFER_LEN))
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
        /* Set start position */
        pDataParams->wRxBufStartPos = wValue;
        pDataParams->wRxBufLen = wValue;
        /* Preserve RxBuffer contents if needed */
        if (pDataParams->pTxBuffer == pDataParams->pRxBuffer)
        {
            pDataParams->wTxBufStartPos = pDataParams->wRxBufStartPos;
        }
        else
        {
            pDataParams->wTxBufStartPos = 0;
        }
        break;

    case PHHAL_HW_CONFIG_TXBUFFER_LENGTH:
        /* Check parameter */
        if ((pDataParams->wTxBufStartPos + wValue) > pDataParams->wTxBufSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
        /* set buffer length */
        pDataParams->wTxBufLen = wValue;
        break;

    case PHHAL_HW_CONFIG_TXBUFFER:
        /* Check additional info parameter */
        if ((pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo) >= pDataParams->wTxBufSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
        /* Modify TxBuffer byte */
        pDataParams->pTxBuffer[pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo] = (uint8_t)wValue;
        break;

    case PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1:
        /* do nothing implement for compatibility reasons */
        break;

    case PHHAL_HW_CONFIG_ADDITIONAL_INFO:
        /* Modify additional info parameter */
        pDataParams->wAdditionalInfo = wValue;
        break;

    case PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT:
        if (wValue == PH_OFF)
        {
            pDataParams->bRfResetAfterTo = PH_OFF;
        }
        else
        {
            pDataParams->bRfResetAfterTo = PH_ON;
        }
        break;

    case PHHAL_HW_MP300_CONFIG_RFAFTERTEARING:
        if ((wValue != PHHAL_HW_MP300_TEARING_FIELD_OFF) &&
            (wValue != PHHAL_HW_MP300_TEARING_FIELD_ON) &&
            (wValue != PHHAL_HW_MP300_TEARING_FIELD_OFF_AFTER_TEARING) &&
            (wValue != PHHAL_HW_MP300_TEARING_FIELD_ON_AFTER_TEARING))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        pDataParams->bRfAfterTearing = (uint8_t)wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_TEARING_EXCHANGE_COUNTER:
        pDataParams->wTearingExchangeCounter = wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_FIELDSTRENGTH:
        PH_CHECK_SUCCESS_FCT(statusTmp,
            phhalHw_GetConfig(pDataParams, PHHAL_HW_MP300_CONFIG_MAX_FIELDSTRENGTH_THOUSAND, &wTmp));

        /* Fieldstrength has to be in the range of 0 to 150/200 percent */
        if (wValue > wTmp / 10)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        wValue = wValue * 10;
    case PHHAL_HW_MP300_CONFIG_FIELDSTRENGTH_THOUSAND:
        PH_CHECK_SUCCESS_FCT(statusTmp,
            phhalHw_GetConfig(pDataParams, PHHAL_HW_MP300_CONFIG_MAX_FIELDSTRENGTH_THOUSAND, &wTmp));

        /* Fieldstrength has to be in the range of 0 to 1500/2000 Permil */
        if (wValue > wTmp)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetFieldStrength(pDataParams, wValue));
        pDataParams->wCurrentFieldStrength = wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_WAVEFORM_GENERATION_MODE:
        /* Parameter Check */
        if ((wValue != PHHAL_HW_MP300_LINEAR_MODULATION) &&
            (wValue != PHHAL_HW_MP300_CUSTOMIZED_MODULATION) &&
            (wValue != PHHAL_HW_MP300_CUSTOMIZED_MODULATION_CLK) &&
            (wValue != PHHAL_HW_MP300_CUSTOMIZED_MODULATION_10CLK))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        if(pDataParams->wCouplerResourceId != PHHAL_HW_MP300_RESSOURCE_TCL3 && wValue == PHHAL_HW_MP300_CUSTOMIZED_MODULATION_10CLK)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        if ( (((pDataParams->bModulationMode&0xF0) >> 4) == PH_OFF) && (wValue != PHHAL_HW_MP300_LINEAR_MODULATION) )
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetWaveformGenerationMode(pDataParams, (uint8_t)wValue));
        pDataParams->bModulationMode = (pDataParams->bModulationMode & 0xF0) | (uint8_t)wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_RX_GAIN_TARGET:
        /* Parameter Check */
        if (pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL3)
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
        }
        if (wValue > 0xFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Set the value only if it has changed */
        if (pDataParams->wTargetRxGainLevel != wValue)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetRxGainTarget(pDataParams, wValue));
        }
        pDataParams->wTargetRxGainLevel = wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_CARRIER_FREQUENCY_SHIFT:
        if ((int16_t)wValue < -10000 || (int16_t)wValue > 10000)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetCarrierFrequency(pDataParams, (uint32_t)(13560000 + ((int16_t)wValue * 100))));
        pDataParams->dwCarrierFrequency = (uint32_t)(13560000 + ((int16_t)wValue * 100));
        break;

    case PHHAL_HW_MP300_CONFIG_RX_GAIN:
        /* Gain has to be in the range of 0 to 100 percent
         * or for TCL3 with Driver >= 1.11 0 to 200 */
        if (sscanf((const char *)pDataParams->bCouplerDriverVersion, "%f", &fDriverVersion) != 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
        }

        if (wValue > 200 || (fDriverVersion < 1.11 && wValue > 100) )
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetRxGain(pDataParams, wValue));
        pDataParams->bRxGain = (uint8_t)wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_RX_CHANNEL:
        if (((uint8_t)wValue != MP300_RXCHANNEL_NORMAL) && ((uint8_t)wValue != MP300_RXCHANNEL_EXTERNAL))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SelectRXChannel(pDataParams, wValue));
        pDataParams->bRXChannel = (uint8_t)wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_COUPLER_NUMBER:
        if ( wValue > MP_COUPLER_3 )
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Check if this coupler exists */
        statusTmp = phhalHw_Mp300_GetResourceID(pDataParams, (uint8_t)wValue, &dwTmp);
        if ((statusTmp) != PH_ERR_SUCCESS)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        pDataParams->bCouplerNumber = (uint8_t)wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_RELAY:
        if ((wValue != PH_ON) && (wValue != PH_OFF))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_ConfigureRelayPort(pDataParams, (uint8_t)wValue));
        pDataParams->bRelay = (uint8_t)wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_FIELD_RISE_TIME:
        if ((pDataParams->wCouplerResourceId != PHHAL_HW_MP300_RESSOURCE_TCL3 && wValue > 5000) ||
            (pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL3 && wValue > 25000))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SelectFieldRiseTime(pDataParams, (uint32_t)wValue));
        pDataParams->wFieldRiseTime = wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_PAUSE_WIDTH:
        if ((wValue < 100) || (wValue > 4500))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SelectPauseWidth(pDataParams, (uint16_t)wValue));
        pDataParams->wPauseWidth = (uint16_t)wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_RISE_TIME:
        if (wValue > 10000)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SelectFallAndRiseTime(pDataParams, pDataParams->wFallTime, (uint16_t)wValue));
        pDataParams->wRiseTime = (uint16_t)wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_FALL_TIME:
        if (wValue > 10000)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SelectFallAndRiseTime(pDataParams, (uint16_t)wValue, pDataParams->wRiseTime));
        pDataParams->wFallTime = (uint16_t)wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_SOURCE_OF_CARRIER:
        if(pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL3)
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
        }
        if ((wValue != PHHAL_HW_MP300_CARRIER_INTERNAL) &&
            (wValue != PHHAL_HW_MP300_CARRIER_EXTERNAL) &&
            (wValue != PHHAL_HW_MP300_CARRIER_INTERNAL_FIXED))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SelectSourceOfCarrier(pDataParams, (uint8_t)wValue));
        pDataParams->bSourceOfCarrier = (uint8_t)wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_SOF_LOW_CYCLES:
        if (wValue == 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTypeBModulationTimingParameter(pDataParams, wValue, MP300_TYPE_B_SOF_LOW_CYCLES));
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_SOF_LOW_CYCLES] = wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_SOF_HIGH_CYCLES:
        if (wValue == 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTypeBModulationTimingParameter(pDataParams, wValue, MP300_TYPE_B_SOF_HIGH_CYCLES));
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_SOF_HIGH_CYCLES] = wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_EGT_CYCLES:
        if (wValue == 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTypeBModulationTimingParameter(pDataParams, wValue, MP300_TYPE_B_EGT_CYCLES));
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_EGT_CYCLES] = wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_EOF_CYCLES:
        if (wValue == 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTypeBModulationTimingParameter(pDataParams, wValue, MP300_TYPE_B_EOF_CYCLES));
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_EOF_CYCLES] = wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_START_BIT_CYCLES:
        if (wValue == 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTypeBModulationTimingParameter(pDataParams, wValue, MP300_TYPE_B_START_BIT_CYCLES));
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_START_BIT_CYCLES] = wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_0_CYCLES:
        if (wValue <= 2 || wValue >= 256)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTypeBModulationTimingParameter(pDataParams, wValue, MP300_TYPE_B_BIT_0_CYCLES));
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_0_CYCLES] = wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_1_CYCLES:
        if (wValue <= 2 || wValue >= 256)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTypeBModulationTimingParameter(pDataParams, wValue, MP300_TYPE_B_BIT_1_CYCLES));
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_1_CYCLES] = wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_2_CYCLES:
        if (wValue <= 2 || wValue >= 256)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTypeBModulationTimingParameter(pDataParams, wValue, MP300_TYPE_B_BIT_2_CYCLES));
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_2_CYCLES] = wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_3_CYCLES:
        if (wValue <= 2 || wValue >= 256)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTypeBModulationTimingParameter(pDataParams, wValue, MP300_TYPE_B_BIT_3_CYCLES));
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_3_CYCLES] = wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_4_CYCLES:
        if (wValue <= 2 || wValue >= 256)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTypeBModulationTimingParameter(pDataParams, wValue, MP300_TYPE_B_BIT_4_CYCLES));
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_4_CYCLES] = wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_5_CYCLES:
        if (wValue <= 2 || wValue >= 256)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTypeBModulationTimingParameter(pDataParams, wValue, MP300_TYPE_B_BIT_5_CYCLES));
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_5_CYCLES] = wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_6_CYCLES:
        if (wValue <= 2 || wValue >= 256)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTypeBModulationTimingParameter(pDataParams, wValue, MP300_TYPE_B_BIT_6_CYCLES));
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_6_CYCLES] = wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_7_CYCLES:
        if (wValue <= 2 || wValue >= 256)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTypeBModulationTimingParameter(pDataParams, wValue, MP300_TYPE_B_BIT_7_CYCLES));
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_7_CYCLES] = wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_STOP_BIT_CYCLES:
        if (wValue == 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTypeBModulationTimingParameter(pDataParams, wValue, MP300_TYPE_B_STOP_BIT_CYCLES));
        pDataParams->wTypeBTimingParameter[MP300_TYPE_B_STOP_BIT_CYCLES] = wValue;
        break;

    /* TCL 2 specific configs */
    case PHHAL_HW_MP300_CONFIG_ANTI_EMD:
        if(pDataParams->wCouplerResourceId != PHHAL_HW_MP300_RESSOURCE_TCL2 && pDataParams->wCouplerResourceId != PHHAL_HW_MP300_RESSOURCE_TCL3)
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
        }

        if ((wValue != PH_ON) && (wValue != PH_OFF))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_ActivateAntiEMD(pDataParams, (uint8_t)wValue));
        pDataParams->bAntiEMD = (uint8_t)wValue;
        break;

    case PHHAL_HW_MP300_CONFIG_FORCE_FDT_METHOD_1:
        if ((wValue != PH_ON) && (wValue != PH_OFF))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        pDataParams->bForceFdtMethod1 = (uint8_t)wValue;
        /* If FDT enabled update the mode */
        if ((pDataParams->wTimingMode & (uint16_t)~(uint16_t)PHHAL_HW_TIMING_MODE_OPTION_MASK) !=
                                  PHHAL_HW_TIMING_MODE_OFF)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_SetTimingParameters(pDataParams, MP300_TIMING_ENABLE));
        }
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_GetConfig(
                                   phhalHw_Mp300_DataParams_t * pDataParams,
                                   uint16_t wConfig,
                                   uint16_t * pValue
                                   )
{
    switch (wConfig)
    {
    case PHHAL_HW_CONFIG_PARITY:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_TXCRC:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_RXCRC:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_TXLASTBITS:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_ADDITIONAL_INFO:
    case PHHAL_HW_CONFIG_RXLASTBITS:
        *pValue = pDataParams->wAdditionalInfo;
        break;

    case PHHAL_HW_CONFIG_RXDEAFBITS:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_TXDATARATE:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_RXDATARATE:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_MODINDEX:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_ASK100:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_US:
        if (pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MICROSECONDS)
        {
            *pValue = pDataParams->wCfgShadow[wConfig];
        }
        else
        {
            if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS] > (0xFFFF / 1000))
            {
                return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
            }
            *pValue = pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS] * 1000;
        }
        break;

    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS:
        if (pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MILLISECONDS)
        {
            *pValue = pDataParams->wCfgShadow[wConfig];
        }
        else
        {
            *pValue = pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_US] / 1000;
        }
        break;

    case PHHAL_HW_CONFIG_TIMING_MODE:
        /* Return parameter */
        *pValue = pDataParams->wTimingMode;
        break;

    case PHHAL_HW_CONFIG_TIMING_US:
        if ((pDataParams->qwTiming_ns / 1000) > 0xFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }

        *pValue = (uint16_t)(pDataParams->qwTiming_ns / 1000);
        pDataParams->qwTiming_ns = 0;
        break;

    case PHHAL_HW_CONFIG_TIMING_MS:
        if ((pDataParams->qwTiming_ns / 1000000) > 0xFFFF)
        {
            pDataParams->qwTiming_ns = 0;
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }

        *pValue = (uint16_t)(pDataParams->qwTiming_ns / 1000000);
        pDataParams->qwTiming_ns = 0;
        break;

    case PHHAL_HW_CONFIG_FIELD_OFF_TIME:
        /* Return parameter */
        *pValue = pDataParams->wFieldOffTime;
        break;

    case PHHAL_HW_CONFIG_FIELD_RECOVERY_TIME:
        /* Return parameter */
        *pValue = pDataParams->wFieldRecoveryTime;
        break;

    case PHHAL_HW_CONFIG_SYMBOL_START:
        /* Return parameter */
        *pValue = pDataParams->bSymbolStart;
        break;

    case PHHAL_HW_CONFIG_SYMBOL_END:
        /* Return parameter */
        *pValue = pDataParams->bSymbolEnd;
        break;

    case PHHAL_HW_CONFIG_SUBCARRIER:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->bSubCarrierMode;
        break;

    case PHHAL_HW_CONFIG_RXBUFFER_STARTPOS:
        /* Return parameter */
        *pValue = pDataParams->wRxBufStartPos;
        break;

    case PHHAL_HW_CONFIG_RXBUFFER_BUFSIZE:
        /* Return parameter */
        *pValue = pDataParams->wRxBufSize - PHHAL_HW_MP300_CRC_BUFFER_LEN;
        break;

    case PHHAL_HW_CONFIG_TXBUFFER_BUFSIZE:
        /* Return parameter */
        *pValue = pDataParams->wTxBufSize - pDataParams->wTxBufStartPos;
        break;

    case PHHAL_HW_CONFIG_TXBUFFER_LENGTH:
        /* Return parameter */
        *pValue = pDataParams->wTxBufLen;
        break;

    case PHHAL_HW_CONFIG_TXBUFFER:
        /* Check additional info parameter */
        if ((pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo) >= pDataParams->wTxBufSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* Retrieve TxBuffer byte */
        *pValue = (uint16_t)pDataParams->pTxBuffer[pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo];
        break;

    case PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1:
        /* return 1 as crypto is not supported */
        *pValue = PH_ON;
        break;

    case PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->bRfResetAfterTo;
        break;

    case PHHAL_HW_MP300_CONFIG_RFAFTERTEARING:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->bRfAfterTearing;
        break;

    case PHHAL_HW_MP300_CONFIG_TEARING_EXCHANGE_COUNTER:
        /* Return parameter */
        *pValue = pDataParams->wTearingExchangeCounter;
        break;

    case PHHAL_HW_CONFIG_CARD_TYPE:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->bCardType;
        break;

    case PHHAL_HW_MP300_CONFIG_FIELDSTRENGTH:
        /* Return parameter */
        *pValue = (uint16_t)((pDataParams->wCurrentFieldStrength+5)/10);
        break;

    case PHHAL_HW_MP300_CONFIG_FIELDSTRENGTH_THOUSAND:
        /* Return parameter */
        *pValue = pDataParams->wCurrentFieldStrength;
        break;

    case PHHAL_HW_MP300_CONFIG_WAVEFORM_GENERATION_MODE:
        /* Return parameter */
        *pValue = pDataParams->bModulationMode&0x0F;
        break;

    case PHHAL_HW_MP300_CONFIG_RX_GAIN_TARGET:
        if (pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL3)
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
        }
        /* Return parameter */
        *pValue = pDataParams->wTargetRxGainLevel;
        break;

    case PHHAL_HW_MP300_CONFIG_CARRIER_FREQUENCY_SHIFT:
        /* Return parameter */
        *pValue = (uint16_t)((pDataParams->dwCarrierFrequency - 13560000) / 100);
        break;

    case PHHAL_HW_MP300_CONFIG_RX_GAIN:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->bRxGain;
        break;

    case PHHAL_HW_MP300_CONFIG_RX_CHANNEL:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->bRXChannel;
        break;

    case PHHAL_HW_MP300_CONFIG_COUPLER_NUMBER:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->bCouplerNumber;
        break;

    case PHHAL_HW_MP300_CONFIG_RELAY:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->bRelay;
        break;

    case PHHAL_HW_MP300_CONFIG_FIELD_RISE_TIME:
        /* Return parameter */
        *pValue = pDataParams->wFieldRiseTime;
        break;

    case PHHAL_HW_MP300_CONFIG_PAUSE_WIDTH:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wPauseWidth;
        break;

    case PHHAL_HW_MP300_CONFIG_FALL_TIME:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wFallTime;
        break;

    case PHHAL_HW_MP300_CONFIG_RISE_TIME:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wRiseTime;
        break;

    case PHHAL_HW_MP300_CONFIG_SOURCE_OF_CARRIER:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->bSourceOfCarrier;
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_SOF_LOW_CYCLES:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wTypeBTimingParameter[MP300_TYPE_B_SOF_LOW_CYCLES];
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_SOF_HIGH_CYCLES:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wTypeBTimingParameter[MP300_TYPE_B_SOF_HIGH_CYCLES];
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_EGT_CYCLES:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wTypeBTimingParameter[MP300_TYPE_B_EGT_CYCLES];
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_EOF_CYCLES:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wTypeBTimingParameter[MP300_TYPE_B_EOF_CYCLES];
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_START_BIT_CYCLES:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wTypeBTimingParameter[MP300_TYPE_B_START_BIT_CYCLES];
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_0_CYCLES:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_0_CYCLES];
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_1_CYCLES:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_1_CYCLES];
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_2_CYCLES:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_2_CYCLES];
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_3_CYCLES:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_3_CYCLES];
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_4_CYCLES:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_4_CYCLES];
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_5_CYCLES:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_5_CYCLES];
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_6_CYCLES:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_6_CYCLES];
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_BIT_7_CYCLES:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wTypeBTimingParameter[MP300_TYPE_B_BIT_7_CYCLES];
        break;

    case PHHAL_HW_MP300_CONFIG_TYPE_B_STOP_BIT_CYCLES:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wTypeBTimingParameter[MP300_TYPE_B_STOP_BIT_CYCLES];
        break;

    /* TCL 2 specific configs */
    case PHHAL_HW_MP300_CONFIG_ANTI_EMD:
        if(pDataParams->wCouplerResourceId != PHHAL_HW_MP300_RESSOURCE_TCL2 && pDataParams->wCouplerResourceId != PHHAL_HW_MP300_RESSOURCE_TCL3)
        {
            *pValue = NULL;
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
        }
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->bAntiEMD;
        break;

    case PHHAL_HW_MP300_CONFIG_FORCE_FDT_METHOD_1:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->bForceFdtMethod1;
        break;

    case PHHAL_HW_MP300_CONFIG_MAX_FIELDSTRENGTH_THOUSAND:
        if(pDataParams->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL3)
        {
            *pValue = MP500_MAX_FIELDSTRENGTH_THOUSAND;
        }
        else
        {
            *pValue = MP300_MAX_FIELDSTRENGTH_THOUSAND;
        }
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_FieldOn(
                                 phhalHw_Mp300_DataParams_t * pDataParams
                                 )
{
    return phhalHw_Mp300_SetFieldStrength(pDataParams, pDataParams->wCurrentFieldStrength);
}

phStatus_t phhalHw_Mp300_FieldOff(
                                  phhalHw_Mp300_DataParams_t * pDataParams
                                  )
{
    return phhalHw_Mp300_SetFieldStrength(pDataParams, 0);
}

phStatus_t phhalHw_Mp300_FieldReset(
                                    phhalHw_Mp300_DataParams_t * pDataParams
                                    )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;

    switch (pDataParams->bCardType)
    {
    case PHHAL_HW_CARDTYPE_ISO14443A:
    case PHHAL_HW_CARDTYPE_ISO14443B:
    case PHHAL_HW_CARDTYPE_FELICA:
    case PHHAL_HW_CARDTYPE_ISO15693:
        /* perfrom RF-Reset */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Mp300_Int_FieldReset(pDataParams));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_Wait(
                              phhalHw_Mp300_DataParams_t * pDataParams,
                              uint8_t bUnit,
                              uint16_t wTimeout
                              )
{
    if (bUnit == PHHAL_HW_TIME_MICROSECONDS)
    {
        return phhalHw_Mp300_Sleep(pDataParams,wTimeout);
    }

    if (bUnit == PHHAL_HW_TIME_MILLISECONDS)
    {
        return phhalHw_Mp300_Sleep(pDataParams,wTimeout*1000);
    }

    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_MfcAuthenticateKeyNo(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint8_t bBlockNo,
    uint8_t bKeyType,
    uint16_t wKeyNo,
    uint16_t wKeyVersion,
    uint8_t * pUid
    )
{
    if (pUid || bBlockNo || pDataParams || bKeyType || wKeyNo || wKeyVersion);

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
}

phStatus_t phhalHw_Mp300_MfcAuthenticate(
    phhalHw_Mp300_DataParams_t * pDataParams,
    uint8_t bBlockNo,
    uint8_t bKeyType,
    uint8_t * pKey,
    uint8_t * pUid
    )
{
    if (bBlockNo || bKeyType || pKey || pUid || pDataParams);

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_MP300 */
