/*
 * Copyright 2019 - 2020, 2022, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * ISO3 specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phhalHw.h>

#ifndef PHHALHW_ISO3_INT_H
#define PHHALHW_ISO3_INT_H

#define PHHAL_HW_ISO3_CARD_TYPE_A                    0           /**< Type A card */
#define PHHAL_HW_ISO3_CARD_TYPE_B                    1           /**< Type B card */
#define PHHAL_HW_ISO3_CARD_TYPE_F                    2           /**< Type Felica card */
#define PHHAL_HW_ISO3_CARD_TYPE_ISO_15693            3           /**< ISO15693 card */
#define PHHAL_HW_ISO3_DATARATE_106K                  0           /**< Data rate 106k */
#define PHHAL_HW_ISO3_DATARATE_212K                  1           /**< Data rate 212k */
#define PHHAL_HW_ISO3_DATARATE_424K                  2           /**< Data rate 424k */
#define PHHAL_HW_ISO3_DATARATE_848K                  3           /**< Data rate 848k */
#define PHHAL_HW_ISO3_DATARATE_15693_256_1K          0           /**< Data rate 15693 1/256 1.6k */
#define PHHAL_HW_ISO3_DATARATE_15693_4_26K           0           /**< Data rate 15693 1/4 26.48k */
#define PHHAL_HW_ISO3_DATARATE_15693_256_13K         1           /**< Data rate 15693 1/256 13.24k */
#define PHHAL_HW_ISO3_DATARATE_15693_4_53K           1           /**< Data rate 15693 1/4 53k */
#define PHHAL_HW_ISO3_DATARATE_15693_256_26K         2           /**< Data rate 15693 1/256 26.48k */
#define PHHAL_HW_ISO3_DATARATE_15693_4_106K          2           /**< Data rate 15693 1/4 106k */
#define PHHAL_HW_ISO3_DATARATE_15693_256_53K         3           /**< Data rate 15693 1/256 53k */
#define PHHAL_HW_ISO3_DATARATE_15693_4_212K          3           /**< Data rate 15693 1/4 212k */
#define PHHAL_HW_ISO3_ISO_15693_MODE_256             0           /**< 1 out of 256 mode for ISO 15693 */
#define PHHAL_HW_ISO3_ISO_15693_MODE_4               1           /**< 1 out of 4 mode for ISO 15693 */

#define PHHAL_HW_ISO3_DAC_MIN_GAIN                   0x00050     /**< Measured value */
#define PHHAL_HW_ISO3_STEERING_MAX_VAL               0x1ffff     /**< Value for oversteering */
#define PHHAL_HW_ISO3_DAC_MAX_GAIN                   0x20000     /**<  */

/**
* \brief ISO3 default register values
*
*/
/*@{*/
#define PHHAL_HW_ISO3_FPGA_PCD2PICC_TIMEOUT_DEFAULT 0x0000846c  /**< 250us default timeout */
#define PHHAL_HW_ISO3_FPGA_PICC2PCD_DELAY_DEFAULT   0x0000      /**< 0 ETUs default delay */
#define PHHAL_HW_ISO3_DGRM_SIG_DET_TH_DEFAULT       1000        /**< default dgrm signal detection threshold */
/*@}*/

/**
* \name Step Attenuator Default Bit Settings for Dampings
*/
/*@{*/
#define PHHAL_HW_ISO3_STEP_ATT_DEFAULT_0DB           0x8040      /**< Default attenuator value for 0db */
#define PHHAL_HW_ISO3_STEP_ATT_DEFAULT_6DB           0x9048      /**< Default attenuator value for 6db */
#define PHHAL_HW_ISO3_STEP_ATT_DEFAULT_12DB          0x9060      /**< Default attenuator value for 12db */
#define PHHAL_HW_ISO3_STEP_ATT_DEFAULT_18DB          0xb058      /**< Default attenuator value for 18db */
#define PHHAL_HW_ISO3_STEP_ATT_DEFAULT_24DB          0xb070      /**< Default attenuator value for 24db */
/*@}*/

/**
* \brief ISO3 TX Commands
*/
/*@{*/
#define PHHAL_HW_ISO3_TX_STATE_POWER_DOWN            0x0000      /**< State value within tx control register for power down */
#define PHHAL_HW_ISO3_TX_STATE_POWER_UP              0x0001      /**< State value within tx control register for power up. This plays the waveshape loaded in the waveshape buffer */
#define PHHAL_HW_ISO3_TX_STATE_TRANSMIT              0x0003      /**< State value within tx control register for transmit */
/*@}*/

/**
* \brief ISO3 MISC Command
*/
#define PHHAL_HW_ISO3_MISC_CMD_START_DAC             0x0051      /**< Start DAC */
#define PHHAL_HW_ISO3_MISC_CMD_STOP_DAC              0x0011      /**< Stop DAC */

/**
* \brief ISO3 FPGA Commands
*/
/*@{*/
#define PHHAL_HW_ISO3_FPGA_OPERAND_MASK             0x0FFFFFFF
#define PHHAL_HW_ISO3_FPGA_CMD_LOOP                 0x2       /**< Marks the beginning of a loop block and sets loop counter to a value of 1 */
#define PHHAL_HW_ISO3_FPGA_CMD_WAIT_FOR_CODE        0x4       /**< Suspend program execution until at least n items of program code are already commited to the code buffer */
#define PHHAL_HW_ISO3_FPGA_CMD_WAIT_FOR_OUTBUF      0x6       /**< copies n data items from following address to the output buffer */
#define PHHAL_HW_ISO3_FPGA_CMD_OUT_IMM              0x8       /**< copies n following words to the output buffer */
#define PHHAL_HW_ISO3_FPGA_CMD_MOV_DST_IMM          0xA       /**< Copies n words to the memory map starting at following address */
/*@}*/

/**
* \brief ISO3 Makros to build Fpga Commands
*/
#define PHHAL_HW_ISO3_BUILD_CMD(cmd, operand)       ((cmd << 28) | (PHHAL_HW_ISO3_FPGA_OPERAND_MASK & operand))
#define PHHAL_HW_ISO3_WRITE_REGISTER                (PHHAL_HW_ISO3_BUILD_CMD(PHHAL_HW_ISO3_FPGA_CMD_MOV_DST_IMM, 1))
#define PHHAL_HW_ISO3_READ_REGISTER                 (PHHAL_HW_ISO3_BUILD_CMD(PHHAL_HW_ISO3_FPGA_CMD_WAIT_FOR_OUTBUF, 1))
#define PHHAL_HW_ISO3_WRITE_WORDS(count)            (PHHAL_HW_ISO3_BUILD_CMD(PHHAL_HW_ISO3_FPGA_CMD_MOV_DST_IMM, count))
#define PHHAL_HW_ISO3_READ_WORDS(count)             (PHHAL_HW_ISO3_BUILD_CMD(PHHAL_HW_ISO3_FPGA_CMD_WAIT_FOR_OUTBUF, count))
#define PHHAL_HW_ISO3_WRITE_OUTBUF(count)           (PHHAL_HW_ISO3_BUILD_CMD(PHHAL_HW_ISO3_FPGA_CMD_OUT_IMM, count))
#define PHHAL_HW_ISO3_OUTBUF_WRITE_WORDS(count)     (PHHAL_HW_ISO3_BUILD_CMD(PHHAL_HW_ISO3_FPGA_CMD_OUT_IMM, count))

#define PHHAL_HW_ISO3_REGISTER_WRITE_BYTES(count)   (PHHAL_HW_ISO3_BUILD_CMD(PHHAL_HW_ISO3_FPGA_CMD_MOV_DST_IMM, (count > 0) ? ((count + 3) / 4) : 0))
#define PHHAL_HW_ISO3_REGISTER_READ_BYTES(count)    (PHHAL_HW_ISO3_BUILD_CMD(PHHAL_HW_ISO3_FPGA_CMD_WAIT_FOR_OUTBUF, (count > 0) ? ((count + 3) / 4) : 0))

#define PHHAL_HW_ISO3_TX_CTRL_STATE(txCtrl, state)                 ((state & 0xC0000000) | txCtrl)
#define PHHAL_HW_ISO3_TX_CTRL_CARD_TYPE(txCtrl, cardType)          ((cardType & 0x20000000) | txCtrl)
#define PHHAL_HW_ISO3_TX_CTRL_DATA_RATE(txCtrl, dataRate)          ((dataRate & 0x18000000) | txCtrl)
#define PHHAL_HW_ISO3_TX_CTRL_REPEAT_WAVESHAPE(txCtrl, repeat)     ((repeat & 0x04000000) | txCtrl)
#define PHHAL_HW_ISO3_TX_CTRL_OPERAND(txCtrl, operand)             ((operand & 0x00003FFF) | txCtrl)

/**
* \brief PLS Header Fields
*/
#define ISO3_PLS_HOST_SENDER_ID                     0x01      /**< Pls header value for sender ID */
#define ISO3_PLS_HEADER_SIGNATURE                   0xaa      /**< Pls header signature */
#define ISO3_PLS_EXCHANGE_BIT_TRUE                  0x01      /**< Pls header flag indicates picc exchange within this tx package */
#define ISO3_PLS_EXCHANGE_BIT_FALSE                 0x00      /**< Pls header flag for data only for fpga */

/**
* \brief PLS header
*/
typedef union
{
    struct
    {
        unsigned int bSignature              :  8;      /**< header signature to identify the valid pls header */
        unsigned int bSenderId               :  8;      /**< senderId      */
        unsigned int wExpectedSize           : 16;      /**< Size of expected response package in bytes */
        unsigned int dwPackageSize           : 32;      /**< Size of package in bytes (excluding header size) */
    } headerFields;
    uint64_t header;
} phhalHw_ISO3_PlsHeaderPackage_t;

/**
* \brief ISO3 waveshape defines
*/
#define ISO3_WAVESHAPE_SLOT_GET_IN_USE(sID)           ((pDataParams->bWaveShapeBufferSlotsBitmap[sID/8] >> (sID%8)) & 1)  /** Makro for wave shape slot */
#define ISO3_WAVESHAPE_SLOT_SET_IN_USE(sID)           ( pDataParams->bWaveShapeBufferSlotsBitmap[sID/8] |=  (1 << (sID%8)))  /** Makro for wave shape slot */
#define ISO3_WAVESHAPE_SLOT_INVALID                   0xffU   /**< Invalid Waveshape Slot ID */

#define ISO3_OVS_FILTER_SLOT_GET_IN_USE(sID)          ((pDataParams->pbOvsFilterSlotsBitmap[sID/8] >> (sID%8)) & 1)  /** Makro to get overshoot slot usage */
#define ISO3_OVS_FILTER_SLOT_SET_IN_USE(sID)          ( pDataParams->pbOvsFilterSlotsBitmap[sID/8] |=  (1 << (sID%8)))
#define ISO3_OVS_FILTER_SLOT_SET_CLEAR(sID)           ( pDataParams->pbOvsFilterSlotsBitmap[sID/8] &=  ~(1 << (sID%8)))

#define ISO3_WAVESHAPE_SAMPLE_SIZE_BYTES              2       /**< Size of single waveshape sample in bytes */

#define ISO3_WAVESHAPE_TYPE_INVALID                   0x00U   /**< Invalid Waveshape ID */
#define ISO3_WAVESHAPE_TYPE_FIELD_ON                  0x01U   /**< Field On Waveshape ID*/
#define ISO3_WAVESHAPE_TYPE_EXCHANGE                  0x02U   /**< Exchange Waveshape ID */
#define ISO3_WAVESHAPE_TYPE_FIELD_OFF                 0x03U   /**< Field Off Waveshape ID */

#define ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB   20     /**< Number of samples for the default field on waveshape */
#define ISO3_WAVESHAPE_DEFAULT_FIELD_ON_OFFSET        0x0000  /**< Sample offset used for the standard field on wave shape */
#define ISO3_WAVESHAPE_DEFAULT_FIELD_OFF_SAMPLE_NUMB  20     /**< Number of samples for the default field off waveshape */
#define ISO3_WAVESHAPE_DEFAULT_FIELD_OFF_OFFSET       (ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB / 2) /*< Word offset used for the standard field off wave shape (sample number * sample size(2Bytes) / adress size(4Bytes)) */

#define ISO3_WAVESHAPE_BUFFER_BASE_ADDRESS            0x0800   /*< Start address of wave shape buffer withing the fpga */
#define ISO3_WAVESHAPE_RESERVED_BUFFER_OFFSET         ((ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB + ISO3_WAVESHAPE_DEFAULT_FIELD_OFF_SAMPLE_NUMB) / 2)     /*< Word offset for reserved area at the begin of fpga wave shape buffer */
#define ISO3_WAVESHAPE_BUFFER_SIZE_BYTES              0xffff   /*< Size of wave shape buffer in bytes */
#define ISO3_WAVESHAPE_BUFFER_AVAIL_BYTES             (ISO3_WAVESHAPE_BUFFER_SIZE_BYTES - ISO3_WAVESHAPE_RESERVED_BUFFER_OFFSET * 4)   /*< Size of available buffer in byte */

/**
* \brief ISO3 HAL Internal Functions
*/

phStatus_t phhalHw_ISO3_Int_GetTxBuffer(
                                     phhalHw_ISO3_DataParams_t * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
                                     uint8_t ** pTxBuffer,                      /**< [Out] Pointer to the TxBuffer. */
                                     uint16_t * pTxBufferLen,                   /**< [Out] Number of bytes already in the TxBuffer. */
                                     uint16_t * pTxBufferSize                   /**< [Out] Size of the TxBuffer. */
                                     );

phStatus_t phhalHw_ISO3_Int_GetRxBuffer(
                                     phhalHw_ISO3_DataParams_t * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
                                     uint8_t ** pRxBuffer,                      /**< [Out] Pointer to the RxBuffer. */
                                     uint16_t * pRxBufferLen,                   /**< [Out] Number of bytes already in the RxBuffer. */
                                     uint16_t * pRxBufferSize                   /**< [Out] Size of the RxBuffer. */
                                     );

phStatus_t phhalHw_ISO3_Int_CalcCrc(
                                 phhalHw_ISO3_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                                 uint8_t * pTxBuffer,                           /**< [In] Pointer to the transmit buffer. */
                                 uint16_t wTxLength,                            /**< [In] Len of the transmit buffer. */
                                 uint8_t * pCrc                                 /**< [Out] Calculated Crc. */
                                 );

phStatus_t phhalHw_ISO3_Int_CheckCrc(
                                 phhalHw_ISO3_DataParams_t * pDataParams       /**< [In] Pointer to this layer's parameter structure. */
                                 );

phStatus_t phhalHw_ISO3_Int_DirectExchange(
                            phhalHw_ISO3_DataParams_t * pDataParams        /**< [In] Pointer to this layer's parameter structure. */
                            );

phStatus_t phhalHw_ISO3_Int_SetCommand(
                            phhalHw_ISO3_DataParams_t * pDataParams,        /**< [In] Pointer to this layer's parameter structure. */
                            uint32_t dwCommand                              /**< [In] Fpga Command, see above */
                            );

phStatus_t phhalHw_ISO3_Int_AddParam(
                            phhalHw_ISO3_DataParams_t * pDataParams,        /**< [In] Pointer to this layer's parameter structure. */
                            uint8_t * pbParam,                              /**< [In] Parameter belonging to Command */
                            uint16_t wSizeBytes                             /**< [In] Size in bytes of parameter */
                            );

phStatus_t phhalHw_ISO3_Int_BuildRegisterWrite(
                            phhalHw_ISO3_DataParams_t * pDataParams,        /**< [In] Pointer to this layer's parameter structure. */
                            uint32_t dwAddr,                                /**< [In] Fpga Register Address, see phhalHw_ISO3_regs.h */
                            uint32_t dwValue                                /**< [In] 4byte data written to fpga address */
                            );

phStatus_t phhalHw_ISO3_Int_BuildRegisterWritePtr(
                            phhalHw_ISO3_DataParams_t * pDataParams,        /**< [In] Pointer to this layer's parameter structure. */
                            uint32_t dwAddr,                                /**< [In] Fpga Register Address, see phhalHw_ISO3_regs.h */
                            uint8_t * pbValue,                              /**< [In] Pointer to transmition data */
                            uint16_t wSizeBytes                             /**< [In] Size in bytes of transmitting data */
                            );

phStatus_t phhalHw_ISO3_Int_BuildRegisterRead(
                            phhalHw_ISO3_DataParams_t * pDataParams,        /**< [In] Pointer to this layer's parameter structure. */
                            uint32_t dwAddr,                                /**< [In] Fpga Register Address, see phhalHw_ISO3_regs.h */
                            uint16_t wSizeBytes                             /**< [In] Size in bytes of data at address */
                            );

phStatus_t phhalHw_ISO3_Int_BuildOutBufferWrite(
                            phhalHw_ISO3_DataParams_t * pDataParams,        /**< [In] Pointer to this layer's parameter structure. */
                            uint8_t * pbValue,                              /**< [In] Data written to fpga output buffer */
                            uint16_t wSizeBytes                             /**< [In] Size in bytes of written to fpga output buffer */
                            );
phStatus_t phhalHw_ISO3_Int_UpdateStepAttenuators(
                            phhalHw_ISO3_DataParams_t * pDataParams,        /**< [In] Pointer to this layer's parameter structure. */
                            uint16_t wValue                                 /**< [In] Damping (0db, 6db, 12db, 24db) */
                            );

phStatus_t phhalHw_ISO3_Int_ConfigMillerTest(
                            phhalHw_ISO3_DataParams_t * pDataParams,        /**< [In] Pointer to this layer's parameter structure. */
                            int8_t bAdjustment                              /**< [In] bAdjustment value of carrier cycles adjusted for miller test */
                            );

phStatus_t phhalHw_ISO3_Int_BuildSetWaveShapePointer(
                            phhalHw_ISO3_DataParams_t * pDataParams,        /**< [In] Pointer to this layer's parameter structure. */
                            uint16_t wWaveShapeType                         /**< [In] Wave shape type (field on, field off, exchange, invalid) */
                            );

phStatus_t phhalHw_ISO3_Int_VenusConfig(
                            phhalHw_ISO3_DataParams_t * pDataParams,        /**< [In] Pointer to this layer's parameter structure. */
                            uint8_t cardType,                               /**< [In] card type identifier. One out of PHHAL_HW_CARDTYPE_ISO14443A, PHHAL_HW_CARDTYPE_ISO14443B, PHHAL_HW_CARDTYPE_FELICA */
                            uint8_t rxDataRate);                            /**< [In] data rate identifier. One out of PHHAL_HW_RF_DATARATE_106 to PHHAL_HW_RF_DATARATE_848 */

/**
* \brief Configuration for loading scaled waveshapes for IIRFilter functionality
* \return Status code
*/
phStatus_t phhalHw_ISO3_Int_LoadPrescaledWaveshapes(
                            phhalHw_ISO3_DataParams_t * pDataParams         /**< [In] Pointer to this layer's parameter structure */
                            );

#endif /* PHHALHW_ISO3_INT_H */
