/*
 * Copyright 2019 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * ISO3 specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <phhalHw.h>

#ifdef NXPBUILD__PHHAL_HW_ISO3

#include "phhalHw_ISO3_int.h"
#include "phhalHw_ISO3.h"
#include "phhalHw_ISO3_Cmd.h"
#include "phhalHw_ISO3_reg.h"
#include <phTools.h>

phStatus_t phhalHw_ISO3_Int_GetTxBuffer(
                                     phhalHw_ISO3_DataParams_t * pDataParams,
                                     uint8_t ** pTxBuffer,
                                     uint16_t * pTxBufferLen,
                                     uint16_t * pTxBufferSize
                                     )
{
    /* We need to watch that we do not overwrite content below the RxStartPos though */
    if (pDataParams->pTxBuffer == pDataParams->pRxBuffer)
    {
        *pTxBuffer = &pDataParams->pTxBuffer[pDataParams->wRxBufStartPos];
        *pTxBufferSize = pDataParams->wTxBufSize - pDataParams->wRxBufStartPos;
    }
    /* Else just return the actual Buffer */
    else
    {
        *pTxBuffer = pDataParams->pTxBuffer;
        *pTxBufferSize = pDataParams->wTxBufSize;
    }

    /* Return stored length */
    *pTxBufferLen = pDataParams->wTxBufLen;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Int_GetRxBuffer(
                                     phhalHw_ISO3_DataParams_t * pDataParams,
                                     uint8_t ** pRxBuffer,
                                     uint16_t * pRxBufferLen,
                                     uint16_t * pRxBufferSize
                                     )
{
    *pRxBuffer = &pDataParams->pRxBuffer[pDataParams->wRxBufStartPos];
    *pRxBufferSize = pDataParams->wRxBufSize - pDataParams->wRxBufStartPos - 2;
    *pRxBufferLen = pDataParams->wRxBufLen;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Int_CalcCrc(
                                 phhalHw_ISO3_DataParams_t * pDataParams,
                                 uint8_t * pTxBuffer,
                                 uint16_t wTxLength,
                                 uint8_t * pCrc
                                 )
{
    phStatus_t statusTmp;
    uint8_t *  pTmpBuffer;
    uint16_t   wTmpBufferLen;
    uint16_t   wTmpBufferSize;
    uint16_t   wCrc;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_GetTxBuffer(pDataParams, &pTmpBuffer, &wTmpBufferLen, &wTmpBufferSize));

    if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A)
    {
        /* Preset CRC */
        wCrc = PH_TOOLS_CRC16_PRESET_ISO14443A;

        /* If internal buffer is used */
        if (wTmpBufferLen != 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_DEFAULT,
                wCrc,
                PH_TOOLS_CRC16_POLY_ISO14443,
                pTmpBuffer,
                wTmpBufferLen,
                &wCrc));
        }
        /* if unbuffed transmit is used */
        if (wTxLength != 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_DEFAULT,
                wCrc,
                PH_TOOLS_CRC16_POLY_ISO14443,
                pTxBuffer,
                wTxLength,
                &wCrc));
        }
    }
    else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443B)
    {
        /* Preset CRC */
        wCrc = PH_TOOLS_CRC16_PRESET_ISO14443B;

        /* If internal buffer is used */
        if (wTmpBufferLen != 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_DEFAULT,
                wCrc,
                PH_TOOLS_CRC16_POLY_ISO14443,
                pTmpBuffer,
                wTmpBufferLen,
                &wCrc));
        }
        /* if unbuffed transmit is used */
        if (wTxLength != 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_DEFAULT,
                wCrc,
                PH_TOOLS_CRC16_POLY_ISO14443,
                pTxBuffer,
                wTxLength,
                &wCrc));
        }

        /* invert output */
        wCrc ^= 0xFFFFU;
    }
    else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_FELICA)
    {
        /* Preset CRC */
        wCrc = PH_TOOLS_CRC16_PRESET_FELICA;

        /* If internal buffer is used */
        if (wTmpBufferLen != 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_MSB_FIRST,
                wCrc,
                PH_TOOLS_CRC16_POLY_FELICA,
                pTmpBuffer,
                wTmpBufferLen,
                &wCrc));
        }
        /* if unbuffed transmit is used */
        if (wTxLength != 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_MSB_FIRST,
                wCrc,
                PH_TOOLS_CRC16_POLY_FELICA,
                pTxBuffer,
                wTxLength,
                &wCrc));
        }
        /* swap bits */
        wCrc = (uint16_t)(wCrc >> 8)|((wCrc&0xFF) << 8);
    }
    else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO15693)
    {
        /* Preset CRC */
        wCrc = PH_TOOLS_CRC16_PRESET_ISO15693;

        /* If internal buffer is used */
        if (wTmpBufferLen != 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_DEFAULT,
                wCrc,
                PH_TOOLS_CRC16_POLY_ISO15693,
                pTmpBuffer,
                wTmpBufferLen,
                &wCrc));
        }
        /* if unbuffed transmit is used */
        if (wTxLength != 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_DEFAULT,
                wCrc,
                PH_TOOLS_CRC16_POLY_ISO15693,
                pTxBuffer,
                wTxLength,
                &wCrc));
        }

        /* invert output */
        wCrc ^= 0xFFFFU;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    pCrc[0] = (uint8_t)(wCrc);
    pCrc[1] = (uint8_t)(wCrc >> 8);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Int_CheckCrc(
                                  phhalHw_ISO3_DataParams_t * pDataParams
                                  )
{
    phStatus_t statusTmp;
    uint8_t *  pTmpBuffer;
    uint16_t   wTmpBufferLen;
    uint16_t   wTmpBufferSize;
    uint16_t   wCrc;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_GetRxBuffer(pDataParams, &pTmpBuffer, &wTmpBufferLen, &wTmpBufferSize));

    /* Calculate actuel received rx buffer len */
    wTmpBufferLen = wTmpBufferLen - pDataParams->wRxBufStartPos;

    if (wTmpBufferLen < 2)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
    }

    if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
            PH_TOOLS_CRC_OPTION_DEFAULT,
            PH_TOOLS_CRC16_PRESET_ISO14443A,
            PH_TOOLS_CRC16_POLY_ISO14443,
            pTmpBuffer,
            wTmpBufferLen,
            &wCrc));

        /* CRC over all data must be zero */
        if (wCrc != 0)
        {
            statusTmp = PH_ERR_INTEGRITY_ERROR;
        }

        /* Remove CRC */
        wTmpBufferLen -= 2;
    }
    else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443B)
    {

        wTmpBufferLen -= 2;

        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
            PH_TOOLS_CRC_OPTION_OUPUT_INVERTED,
            PH_TOOLS_CRC16_PRESET_ISO14443B,
            PH_TOOLS_CRC16_POLY_ISO14443,
            pTmpBuffer,
            wTmpBufferLen,
            &wCrc));

        if (((uint8_t)wCrc != pTmpBuffer[wTmpBufferLen]) || ((uint8_t)(wCrc>>8) != pTmpBuffer[wTmpBufferLen+1]))
        {
            statusTmp = PH_ERR_INTEGRITY_ERROR;
        }
    }
    else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_FELICA)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
            PH_TOOLS_CRC_OPTION_MSB_FIRST,
            PH_TOOLS_CRC16_PRESET_FELICA,
            PH_TOOLS_CRC16_POLY_FELICA,
            pTmpBuffer,
            wTmpBufferLen,
            &wCrc));

        /* CRC over all data must be zero */
        if (wCrc != 0)
        {
            statusTmp = PH_ERR_INTEGRITY_ERROR;
        }

        /* Remove CRC */
        wTmpBufferLen -= 2;
    }
    else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO15693)
    {

        wTmpBufferLen -= 2;

        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
            PH_TOOLS_CRC_OPTION_OUPUT_INVERTED,
            PH_TOOLS_CRC16_PRESET_ISO15693,
            PH_TOOLS_CRC16_POLY_ISO15693,
            pTmpBuffer,
            wTmpBufferLen,
            &wCrc));

        if (((uint8_t)wCrc != pTmpBuffer[wTmpBufferLen]) || ((uint8_t)(wCrc>>8) != pTmpBuffer[wTmpBufferLen+1]))
        {
            statusTmp = PH_ERR_INTEGRITY_ERROR;
        }
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Correct received data length in dataparams */
    pDataParams->wRxBufLen = pDataParams->wRxBufStartPos + wTmpBufferLen;

    return PH_ADD_COMPCODE(statusTmp, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Int_DirectExchange(
                            phhalHw_ISO3_DataParams_t * pDataParams
                            )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_PlsHeaderPackage_t plsSendHeader;

    PH_ASSERT_NULL(pDataParams);

    /* check tx length */
    if (pDataParams->wIntNumValidBytes == 0 && pDataParams->wIntNumValidBytes % sizeof(uint32_t) == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* create pls send header */
    plsSendHeader.headerFields.bSenderId         = (uint8_t)ISO3_PLS_HOST_SENDER_ID;
    plsSendHeader.headerFields.bSignature        = (uint8_t)ISO3_PLS_HEADER_SIGNATURE;
    plsSendHeader.headerFields.wExpectedSize     = (uint16_t)pDataParams->wRxExpectedInputBytes;
    plsSendHeader.headerFields.dwPackageSize     = (uint32_t)pDataParams->wIntNumValidBytes;

    /* Send pls header first */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDirectDataParams,                      /* [IN] BAL params */
        PH_EXCHANGE_DEFAULT,                                    /* [IN] option */
        (uint8_t*) &plsSendHeader,                              /* [IN] Tx Buffer */
        sizeof(plsSendHeader),                                  /* [IN] Tx Length */
        0,                                                      /* [IN] Rx Length / Expected bytes */
        NULL,                                                   /* [IN] Rx Buffer */
        NULL));                                                 /* [OUT] Recv length */

    /* Reseting pDataParams->wIntNumValidRxBytes before next communication */
    pDataParams->wIntNumValidRxBytes = 0;

    /* Transmit and receive data over BAL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDirectDataParams,                  /* [IN] BAL params */
        PH_EXCHANGE_DEFAULT,                                /* [IN] option */
        pDataParams->pIntBuffer,                            /* [IN] Tx Buffer */
        pDataParams->wIntNumValidBytes,                     /* [IN] Tx Length */
        pDataParams->wRxExpectedInputBytes,                 /* [IN] Rx Length / Expected bytes */
        pDataParams->pIntBuffer,                            /* [IN] Rx Buffer is also the int buffer */
        &(pDataParams->wIntNumValidRxBytes)));              /* [OUT] Recv length is stored as int buffer rx valid */

    /* reset buffers */
    pDataParams->wIntNumValidBytes     = 0;
    pDataParams->wRxExpectedInputBytes = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Int_SetCommand(
                                phhalHw_ISO3_DataParams_t * pDataParams,
                                uint32_t dwCommand
                                )
{
    PH_ASSERT_NULL (pDataParams);

    if(pDataParams->wIntNumValidBytes + sizeof(uint32_t) > pDataParams->wMaxIntBufferSizeBytes)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    *(uint32_t*)(pDataParams->pIntBuffer + pDataParams->wIntNumValidBytes) = dwCommand;
    pDataParams->wIntNumValidBytes += sizeof(uint32_t);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Int_AddParam(
                                phhalHw_ISO3_DataParams_t * pDataParams,
                                uint8_t * pParam,
                                uint16_t wSizeBytes
                                )
{
    uint16_t wStep;
    uint16_t wWords;
    uint32_t* pStartAddress;

    PH_ASSERT_NULL (pDataParams);

    if (wSizeBytes <= 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);;
    }

    if(pDataParams->wIntNumValidBytes + wSizeBytes + sizeof(uint32_t) > pDataParams->wMaxIntBufferSizeBytes)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    wWords = wSizeBytes / sizeof(uint32_t);
    pStartAddress = (uint32_t*)(pDataParams->pIntBuffer + pDataParams->wIntNumValidBytes);

    for(wStep = 0; wStep < wWords; wStep++)
    {
        *(pStartAddress + wStep) = *((uint32_t*)pParam + wStep);
    }

    pDataParams->wIntNumValidBytes += wWords * sizeof(uint32_t);

    if (wSizeBytes % sizeof(uint32_t))
    {
        *(pStartAddress + wWords) = 0x0;

        for(wStep = 0; wStep < wSizeBytes % sizeof(uint32_t); wStep++)
        {
            *(pDataParams->pIntBuffer + (pDataParams->wIntNumValidBytes + wStep)) =
                *(pParam + (wWords * 4 + wStep));
        }
        pDataParams->wIntNumValidBytes += sizeof(uint32_t);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Int_BuildRegisterWrite(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint32_t dwAddr,
                            uint32_t dwValue
                            )
{
    phStatus_t statusTmp;
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_SetCommand(pDataParams, (uint32_t)PHHAL_HW_ISO3_WRITE_REGISTER));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_AddParam(pDataParams, (uint8_t*) &dwAddr, sizeof(uint32_t)));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_AddParam(pDataParams, (uint8_t*) &dwValue, sizeof(uint32_t)));
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Int_BuildRegisterWritePtr(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint32_t dwAddr,
                            uint8_t * pbValue,
                            uint16_t wSizeBytes
                            )
{
    phStatus_t statusTmp;
    uint32_t dwWords;
    PH_ASSERT_NULL (pDataParams);

    if (wSizeBytes == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    if (wSizeBytes % sizeof(uint32_t) == 0)
    {
        dwWords = wSizeBytes / sizeof(uint32_t);
    }
    else
    {
        dwWords = (wSizeBytes / sizeof(uint32_t)) + 1;
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_SetCommand(pDataParams, PHHAL_HW_ISO3_WRITE_WORDS(dwWords)));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_AddParam(pDataParams, (uint8_t*) &dwAddr, sizeof(uint32_t)));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_AddParam(pDataParams, pbValue, wSizeBytes));
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Int_BuildRegisterRead(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint32_t dwAddr,
                            uint16_t wSizeBytes
                            )
{
    phStatus_t statusTmp;
    uint16_t wWords;
    PH_ASSERT_NULL (pDataParams);

    if (wSizeBytes == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    if (wSizeBytes % sizeof(uint32_t) == 0)
    {
        wWords = wSizeBytes / sizeof(uint32_t);
    }
    else
    {
        wWords = (wSizeBytes / sizeof(uint32_t)) + 1;
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_SetCommand(pDataParams, PHHAL_HW_ISO3_READ_WORDS(wWords)));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_AddParam(pDataParams, (uint8_t*) &dwAddr, sizeof(uint32_t)));

    if(pDataParams->wRxExpectedInputBytes + (wWords * sizeof(uint32_t)) > 0xffff)
    {
        PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }
    pDataParams->wRxExpectedInputBytes += wWords * sizeof(uint32_t);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}


phStatus_t phhalHw_ISO3_Int_BuildOutBufferWrite(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint8_t * pbValue,
                            uint16_t wSizeBytes)
{
    uint32_t dwWords;
    PH_ASSERT_NULL (pDataParams);

    if (wSizeBytes == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    if (wSizeBytes % sizeof(uint32_t) == 0)
    {
        dwWords = wSizeBytes / sizeof(uint32_t);
    }
    else
    {
        dwWords = (wSizeBytes / sizeof(uint32_t)) + 1;
    }

    phhalHw_ISO3_Int_SetCommand(pDataParams, PHHAL_HW_ISO3_OUTBUF_WRITE_WORDS(dwWords));
    phhalHw_ISO3_Int_AddParam(pDataParams, pbValue, wSizeBytes);

    pDataParams->wRxExpectedInputBytes += (uint16_t)(dwWords * sizeof(uint32_t));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Int_UpdateStepAttenuators(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint16_t wValue
                            )
{
    phStatus_t statusTmp;

    PH_ASSERT_NULL (pDataParams);

    /* freeze DAC Gain control before changing DSA */
    phhalHw_ISO3_Int_BuildRegisterWrite(
        pDataParams,
        PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL,
        PH_ON);
    phhalHw_ISO3_Int_BuildRegisterWrite(
        pDataParams,
        PHHAL_HW_ISO3_REG_STEP_ATTEN_CTRL,
        (uint32_t)(wValue >> 8));
    phhalHw_ISO3_Int_BuildRegisterWrite(
        pDataParams,
        PHHAL_HW_ISO3_REG_STEP_ATTEN_CTRL,
        (uint32_t)(wValue & 0x00FF));
    phhalHw_ISO3_Int_BuildRegisterWrite(
        pDataParams,
        PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL,
        PH_OFF);
    statusTmp = phhalHw_ISO3_Int_DirectExchange(pDataParams);
    return PH_ADD_COMPCODE(statusTmp, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Int_ConfigMillerTest(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            int8_t bAdjustment
                            )
{
    phStatus_t statusTmp;
    /* Calculate the new values for the bitgrid prescaler.
    Only items from 1..8 are used to adjust the second half of the bit */
    int8_t adjustOffset = bAdjustment >= 0 ? bAdjustment : -bAdjustment;
    int8_t adjustAll = adjustOffset/8;
    int8_t adjustRest = adjustOffset%8;
    int8_t i;
    int8_t offset;
    uint8_t newTarget;

    phhalHw_ISO3_PrescalerCtrlReg_t ctrl;
    phhalHw_ISO3_PrescalerTiming0to7Reg_t timing0to7;
    phhalHw_ISO3_PrescalerTiming8to15Reg_t timing8to15;

    /* Default values */
    ctrl.Struct.bModeBitgrid = PH_OFF;
    ctrl.Struct.dwRfu = 0;
    timing0to7.Value = 0x77777777;
    timing8to15.Value = 0x77777777;

    if (bAdjustment == 0)
    {
        ctrl.Struct.bEnabled = PH_OFF;
    }
    else
    {
        ctrl.Struct.bEnabled = PH_ON;
        for (i = 0; i < 8; i++)
        {
            offset = adjustAll;
            if (i < adjustRest)
            {
                offset++;
            }
            newTarget = (bAdjustment < 0) ? 7 - offset : 7 + offset;
            switch(i+1)
            {
                case 1: timing0to7.Struct.bPrescaler1 = newTarget; break;
                case 2: timing0to7.Struct.bPrescaler2 = newTarget; break;
                case 3: timing0to7.Struct.bPrescaler3 = newTarget; break;
                case 4: timing0to7.Struct.bPrescaler4 = newTarget; break;
                case 5: timing0to7.Struct.bPrescaler5 = newTarget; break;
                case 6: timing0to7.Struct.bPrescaler6 = newTarget; break;
                case 7: timing0to7.Struct.bPrescaler7 = newTarget; break;
                case 8: timing8to15.Struct.bPrescaler8 = newTarget; break;
            }

        }
    }

    /* reset buffers */
    pDataParams->wIntNumValidBytes = 0;
    pDataParams->wIntNumValidRxBytes = 0;
    pDataParams->wRxExpectedInputBytes = 0;

    /* Set prescaler config */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_PRESCALER_TIMING_0TO7_REG, timing0to7.Value));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams,  PHHAL_HW_ISO3_REG_PRESCALER_TIMING_8TO15_REG, timing8to15.Value));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_PRESCALER_CTRL_REG, ctrl.Value));
    statusTmp = phhalHw_ISO3_Int_DirectExchange(pDataParams);
    return PH_ADD_COMPCODE(statusTmp, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Int_BuildSetWaveShapePointer(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint16_t wWaveShapeType
                            )
{
    phStatus_t statusTmp;
    uint16_t wSlotID;
    uint16_t wStartAddress = 0;
    uint16_t wSlotSizeBytes;
    uint32_t waveshapeRegister;

    PH_ASSERT_NULL (pDataParams);

    switch (wWaveShapeType)
    {
    case ISO3_WAVESHAPE_TYPE_FIELD_ON:
        phhalHw_ISO3_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOT_FIELD_ON, &wSlotID);
        if (wSlotID == PHHAL_HW_ISO3_WAVESHAPE_SLOT_DEFAULT_ID)
        {
            wStartAddress = ISO3_WAVESHAPE_DEFAULT_FIELD_ON_OFFSET;
        }
        waveshapeRegister = PHHAL_HW_ISO3_REG_RF_ON_WAVESHAPE_POINTER;
        break;
    case ISO3_WAVESHAPE_TYPE_FIELD_OFF:
        phhalHw_ISO3_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOT_FIELD_OFF, &wSlotID);
        if (wSlotID == PHHAL_HW_ISO3_WAVESHAPE_SLOT_DEFAULT_ID)
        {
            wStartAddress = ISO3_WAVESHAPE_DEFAULT_FIELD_OFF_OFFSET;
        }
        waveshapeRegister = PHHAL_HW_ISO3_REG_RF_OFF_WAVESHAPE_POINTER;
        break;
    case ISO3_WAVESHAPE_TYPE_EXCHANGE:
        phhalHw_ISO3_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOT_EXCHANGE, &wSlotID);
        waveshapeRegister = PHHAL_HW_ISO3_REG_WAVESHAPE_POINTER;
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    if(wWaveShapeType == ISO3_WAVESHAPE_TYPE_EXCHANGE ||
       wSlotID != PHHAL_HW_ISO3_WAVESHAPE_SLOT_DEFAULT_ID)
    {
        /* WAVESHAPE SLOT */
        phhalHw_ISO3_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_SIZE, &wSlotSizeBytes);

        /* [ RESERVED | SLOT1 | SLOT2 | ... | SLOTn  ] */
        /* calculate relative start offset */

        /* If sequence8 is enabled, set the waveshape pointer at the base address of the waveshape buffer (Offset = 0) */
        if ((ISO3_REG_TX_CTRL_0.cardType == PHHAL_HW_ISO3_CARD_TYPE_A) && (ISO3_REG_TX_CTRL_0.iso15693mode == PH_ON))
        {
            wStartAddress = 0;
        }
        else
        {
            wStartAddress = (uint16_t)(ISO3_WAVESHAPE_RESERVED_BUFFER_OFFSET + wSlotID * (wSlotSizeBytes / sizeof(uint32_t)));
        }
    }

    /* Set wave shape start address */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(
        pDataParams,
        waveshapeRegister,
        wStartAddress));

    pDataParams->dwCurrentWaveShapeType = wWaveShapeType;
    if (wWaveShapeType == ISO3_WAVESHAPE_TYPE_EXCHANGE)
    {
        pDataParams->bCurrentWaveshapeSlot = (uint8_t)wSlotID;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Int_VenusConfig(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint8_t cardType,
                            uint8_t rxDataRate) {
    phStatus_t statusTmp;
    uint32_t venusConfigValues[6];
    uint32_t venusConfigAddresses[6];
    uint8_t i;
    uint16_t status;

    PH_ASSERT_NULL (pDataParams);

    status = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);

    switch(cardType) {
    case PHHAL_HW_CARDTYPE_ISO14443A:
        {
            switch(rxDataRate) {
            case PHHAL_HW_RF_DATARATE_106:
                venusConfigValues[0] = PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_A106;
                venusConfigValues[1] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_A106;
                venusConfigValues[2] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_A106;
                venusConfigValues[3] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_A106;
                venusConfigValues[4] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_A106;
                venusConfigValues[5] = PHHAL_HW_ISO3_REG_VENUS_RM_REG_A106;
                break;
            case PHHAL_HW_RF_DATARATE_212:
                venusConfigValues[0] = PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_A212;
                venusConfigValues[1] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_A212;
                venusConfigValues[2] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_A212;
                venusConfigValues[3] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_A212;
                venusConfigValues[4] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_A212;
                venusConfigValues[5] = PHHAL_HW_ISO3_REG_VENUS_RM_REG_A212;
                break;
            case PHHAL_HW_RF_DATARATE_424:
                venusConfigValues[0] = PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_A424;
                venusConfigValues[1] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_A424;
                venusConfigValues[2] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_A424;
                venusConfigValues[3] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_A424;
                venusConfigValues[4] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_A424;
                venusConfigValues[5] = PHHAL_HW_ISO3_REG_VENUS_RM_REG_A424;
                break;
            case PHHAL_HW_RF_DATARATE_848:
                venusConfigValues[0] = PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_A848;
                venusConfigValues[1] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_A848;
                venusConfigValues[2] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_A848;
                venusConfigValues[3] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_A848;
                venusConfigValues[4] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_A848;
                venusConfigValues[5] = PHHAL_HW_ISO3_REG_VENUS_RM_REG_A848;
                break;
            default:
                /* unsuppported data rate */
                status = PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        break;
    case PHHAL_HW_CARDTYPE_ISO14443B:
        {
            switch(rxDataRate) {
            case PHHAL_HW_RF_DATARATE_106:
                venusConfigValues[0] = PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_B106;
                venusConfigValues[1] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_B106;
                venusConfigValues[2] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_B106;
                venusConfigValues[3] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_B106;
                venusConfigValues[4] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_B106;
                venusConfigValues[5] = PHHAL_HW_ISO3_REG_VENUS_RM_REG_B106;
                break;
            case PHHAL_HW_RF_DATARATE_212:
                venusConfigValues[0] = PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_B212;
                venusConfigValues[1] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_B212;
                venusConfigValues[2] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_B212;
                venusConfigValues[3] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_B212;
                venusConfigValues[4] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_B212;
                venusConfigValues[5] = PHHAL_HW_ISO3_REG_VENUS_RM_REG_B212;
                break;
            case PHHAL_HW_RF_DATARATE_424:
                venusConfigValues[0] = PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_B424;
                venusConfigValues[1] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_B424;
                venusConfigValues[2] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_B424;
                venusConfigValues[3] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_B424;
                venusConfigValues[4] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_B424;
                venusConfigValues[5] = PHHAL_HW_ISO3_REG_VENUS_RM_REG_B424;
                break;
            case PHHAL_HW_RF_DATARATE_848:
                venusConfigValues[0] = PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_B848;
                venusConfigValues[1] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_B848;
                venusConfigValues[2] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_B848;
                venusConfigValues[3] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_B848;
                venusConfigValues[4] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_B848;
                venusConfigValues[5] = PHHAL_HW_ISO3_REG_VENUS_RM_REG_B848;
                break;
            default:
                /* unsuppported data rate */
                status = PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        break;
    case PHHAL_HW_CARDTYPE_FELICA:
        {
            switch(rxDataRate) {
            case PHHAL_HW_RF_DATARATE_212:
                venusConfigValues[0] = PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_F212;
                venusConfigValues[1] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_F212;
                venusConfigValues[2] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_F212;
                venusConfigValues[3] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_F212;
                venusConfigValues[4] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_F212;
                venusConfigValues[5] = PHHAL_HW_ISO3_REG_VENUS_RM_REG_F212;
                break;
            case PHHAL_HW_RF_DATARATE_424:
                venusConfigValues[0] = PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_F424;
                venusConfigValues[1] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_F424;
                venusConfigValues[2] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_F424;
                venusConfigValues[3] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_F424;
                venusConfigValues[4] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_F424;
                venusConfigValues[5] = PHHAL_HW_ISO3_REG_VENUS_RM_REG_F424;
                break;
            default:
                /* unsuppported data rate */
                status = PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        break;
    case PHHAL_HW_CARDTYPE_ISO15693:
        {
            switch(rxDataRate) {
            case PHHAL_HW_RF_RX_DATARATE_LOW:
                venusConfigValues[0] = PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_I006;
                venusConfigValues[1] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_I006;
                venusConfigValues[2] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_I006;
                venusConfigValues[3] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_I006;
                venusConfigValues[4] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_I006;
                venusConfigValues[5] = PHHAL_HW_ISO3_REG_VENUS_RM_REG_I006;
                break;
            case PHHAL_HW_RF_RX_DATARATE_HIGH:
                venusConfigValues[0] = PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_I026;
                venusConfigValues[1] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_I026;
                venusConfigValues[2] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_I026;
                venusConfigValues[3] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_I026;
                venusConfigValues[4] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_I026;
                venusConfigValues[5] = PHHAL_HW_ISO3_REG_VENUS_RM_REG_I026;
                break;
            case PHHAL_HW_RF_RX_DATARATE_FAST_HIGH:
                venusConfigValues[0] = PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_I053;
                venusConfigValues[1] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG_I053;
                venusConfigValues[2] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG_I053;
                venusConfigValues[3] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG_I053;
                venusConfigValues[4] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG_I053;
                venusConfigValues[5] = PHHAL_HW_ISO3_REG_VENUS_RM_REG_I053;
                break;
            default:
                /* unsuppported data rate */
                status = PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        break;

    default:
        /* unsupported card type */
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS(status);

    venusConfigAddresses[0] = PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG;
    venusConfigAddresses[1] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_CONFIG_REG;
    venusConfigAddresses[2] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_PATTERN_REG;
    venusConfigAddresses[3] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_ENABLES_REG;
    venusConfigAddresses[4] = PHHAL_HW_ISO3_REG_VENUS_CLIF_SIGPRO_RM_TECH_REG;
    venusConfigAddresses[5] = PHHAL_HW_ISO3_REG_VENUS_RM_REG;

    pDataParams->dwVenusClifRxConfigReg = venusConfigValues[0];

    for (i = 0; i < 6; i++) {
        phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, venusConfigAddresses[i], venusConfigValues[i]);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(pDataParams));

    return status;
}

phStatus_t phhalHw_ISO3_Int_LoadPrescaledWaveshapes(phhalHw_ISO3_DataParams_t * pDataParams)
{
    phStatus_t statusTmp;
    uint16_t wCounter;
    uint16_t pwFieldOn[ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB];
    uint16_t pwFieldOff[ISO3_WAVESHAPE_DEFAULT_FIELD_OFF_SAMPLE_NUMB];
    uint16_t wWaveShapePrescalePercent;
    phhalHw_ISO3_TxCtrlRegister_t tmp;
    uint8_t  bCounter;

    /* Check param */
    PH_ASSERT_NULL(pDataParams);

    /* save previous state to temporary variable */
    memcpy(&tmp, &pDataParams->dwTxCtrl, sizeof(phhalHw_ISO3_TxCtrlRegister_t));

    /* Invalidate all waveshape slots */
    for(bCounter = 0; bCounter < ((PHHAL_HW_ISO3_MAX_NUMB_WS_SLOTS + 7) / 8); bCounter++)
    {
        pDataParams->bWaveShapeBufferSlotsBitmap[bCounter] = (uint8_t)0x0;
    }

    /* Turn the field OFF */
    if (ISO3_REG_TX_CTRL_0.state != PHHAL_HW_ISO3_TX_STATE_POWER_DOWN)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_FieldOff(pDataParams));
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_PRESCALE_PERCENT, &wWaveShapePrescalePercent));

    /* Create and load the custom field on and field of wave shape */
    for(wCounter = 0; wCounter < ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB; wCounter++)
    {
        pwFieldOn[wCounter] = (uint16_t)((double)0xfffe * ((double)wWaveShapePrescalePercent/100) * (double)wCounter / (double)(ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB - 1));
    }

    pwFieldOn[ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB - 3] = pwFieldOn[ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB - 1];
    pwFieldOn[ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB - 4] = pwFieldOn[ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB - 1];

    for(wCounter = 0; wCounter < ISO3_WAVESHAPE_DEFAULT_FIELD_OFF_SAMPLE_NUMB; wCounter++)
    {
        pwFieldOff[wCounter] = pwFieldOn[ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB - 1 - wCounter];
    }

    pwFieldOff[ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB - 4] = pwFieldOff[ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB - 1];
    pwFieldOff[ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB - 3] = pwFieldOff[ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB - 1];

    pwFieldOn[ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB - 2] = 0xffff;
    pwFieldOff[ISO3_WAVESHAPE_DEFAULT_FIELD_OFF_SAMPLE_NUMB - 2] = 0xffff;

    phhalHw_ISO3_SetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_NUMB,  PHHAL_HW_ISO3_WAVESHAPE_SLOT_DEFAULT_NUMBER);

    /* Upload custom field on/off waveshapes into fpga buffer */
    phhalHw_ISO3_Int_BuildRegisterWritePtr(pDataParams, ISO3_WAVESHAPE_BUFFER_BASE_ADDRESS, (uint8_t*)&pwFieldOn, ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB * sizeof(uint16_t));
    phhalHw_ISO3_Int_BuildRegisterWritePtr(pDataParams, ISO3_WAVESHAPE_BUFFER_BASE_ADDRESS + ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB / sizeof(uint16_t), (uint8_t*)&pwFieldOff, ISO3_WAVESHAPE_DEFAULT_FIELD_OFF_SAMPLE_NUMB * sizeof(uint16_t));

    for ( wCounter = 0; wCounter < ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB * sizeof(uint16_t); wCounter++ )
    {
        pDataParams->pWaveShapeShadowBuffer[wCounter] = *(((uint8_t*)pwFieldOn) + wCounter);
    }
    for ( wCounter = 0; wCounter < ISO3_WAVESHAPE_DEFAULT_FIELD_OFF_SAMPLE_NUMB * sizeof(uint16_t); wCounter++ )
    {
        pDataParams->pWaveShapeShadowBuffer[wCounter + ISO3_WAVESHAPE_DEFAULT_FIELD_ON_SAMPLE_NUMB * sizeof(uint16_t)] = *(((uint8_t*)pwFieldOff) + wCounter);
    }

    phhalHw_ISO3_SetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOT_FIELD_ON,  PHHAL_HW_ISO3_WAVESHAPE_SLOT_DEFAULT_ID);
    phhalHw_ISO3_SetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOT_FIELD_OFF, PHHAL_HW_ISO3_WAVESHAPE_SLOT_DEFAULT_ID);

    /* Send and reset buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(pDataParams));

    /* check for field off */
    if (tmp.state != PHHAL_HW_ISO3_TX_STATE_POWER_DOWN )
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_FieldOn(pDataParams));
    }

    /* restore the previous state */
    ISO3_REG_TX_CTRL_0.cardType = tmp.cardType;
    ISO3_REG_TX_CTRL_0.iso15693mode = tmp.iso15693mode;
    ISO3_REG_TX_CTRL_0.datarate = tmp.datarate;
    ISO3_REG_TX_CTRL_0.operand = tmp.operand;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(
        pDataParams,
        PHHAL_HW_ISO3_REG_TX_CTRL_0,
        pDataParams->dwTxCtrl));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(
        pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}
#endif /* NXPBUILD__PHHAL_HW_ISO3 */
