/*
 * Copyright 2019 - 2020, 2022 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * ISO3 specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <phhalHw.h>
#include <phTools.h>

#ifdef NXPBUILD__PHHAL_HW_ISO3

#include "phhalHw_ISO3.h"
#include "phhalHw_ISO3_int.h"
#include "phhalHw_ISO3_cmd.h"
#include "phhalHw_ISO3_reg.h"


/* Default shadow for ISO14443-3A Mode */
static const uint16_t wISO3_DefaultShadow_14443a[][2] =
{
    {PHHAL_HW_CONFIG_PARITY,                PH_ON},
    {PHHAL_HW_CONFIG_TXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_RXCRC,                 PH_OFF},
    {PHHAL_HW_CONFIG_RXDEAFBITS,            0x0008},
    {PHHAL_HW_CONFIG_TXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_RXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_ASK100,                PH_ON},
    {PHHAL_HW_CONFIG_TXWAIT_US,             0},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,      0},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,      PHHAL_HW_ISO3_DEFAULT_TIMEOUT_US}
};

/* Default shadow for ISO14443-3B Mode */
static const uint16_t wISO3_DefaultShadow_14443b[][2] =
{
    {PHHAL_HW_CONFIG_PARITY,                PH_OFF},
    {PHHAL_HW_CONFIG_TXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_RXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_RXDEAFBITS,            0x0008},
    {PHHAL_HW_CONFIG_TXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_RXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_MODINDEX,              /* TODO */ 0},
    {PHHAL_HW_CONFIG_ASK100,                PH_OFF},
    {PHHAL_HW_CONFIG_TXWAIT_US,             0},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,      0},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,      PHHAL_HW_ISO3_DEFAULT_TIMEOUT_US}
};

/* Default shadow for ISO15693 Mode */
static const uint16_t wISO3_DefaultShadow_15693[][2] =
{
    {PHHAL_HW_CONFIG_PARITY,                PH_OFF},
    {PHHAL_HW_CONFIG_TXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_RXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_TXDATARATE,            PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4},
    {PHHAL_HW_CONFIG_RXDATARATE,            PHHAL_HW_RF_RX_DATARATE_LOW},
    {PHHAL_HW_CONFIG_ASK100,                PH_ON},
    {PHHAL_HW_CONFIG_TXWAIT_US,             0},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,      0},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,      PHHAL_HW_ISO3_DEFAULT_TIMEOUT_US}
};

/* Default shadow for Felica Mode */
static const uint16_t wISO3_DefaultShadow_Felica[][2] =
{
    {PHHAL_HW_CONFIG_PARITY,                PH_OFF},
    {PHHAL_HW_CONFIG_TXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_RXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_RXDEAFBITS,            0x0000},
    {PHHAL_HW_CONFIG_TXDATARATE,            PHHAL_HW_RF_DATARATE_212},
    {PHHAL_HW_CONFIG_RXDATARATE,            PHHAL_HW_RF_DATARATE_212},
    {PHHAL_HW_CONFIG_ASK100,                PH_OFF},
    {PHHAL_HW_CONFIG_TXWAIT_US,             0},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,      0},
    {PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,      PHHAL_HW_ISO3_DEFAULT_TIMEOUT_US}
};

/* ------------------- */
/* Interface functions */
/* ------------------- */

phStatus_t phhalHw_ISO3_Exchange(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint16_t wOption,
                            uint8_t * pTxBuffer,
                            uint16_t wTxLength,
                            uint8_t ** ppRxBuffer,
                            uint16_t * pRxLength
                            )
{
    phStatus_t  statusTmp;
    phStatus_t  statusExchange;

    uint8_t *   pTmpBuffer;
    uint16_t    wTmpBufferLen;
    uint16_t    wTmpBufferSize;

    uint8_t *pRxBufferInt;
    uint16_t wRxLengthInt;

    uint8_t     pCrc[2];
    uint8_t     bLoop;

    uint32_t    dwPosColl;
    uint32_t    dwNumReceivedBits = 0;
    uint16_t    wShiftIndex = 0;
    uint8_t     bShiftOver = 0;
    uint8_t     bTmpShiftOver = 0;
    uint16_t    wTmpReceivedByte;
    uint8_t     bTmpLastBit;
    uint8_t     bOneCnt;


    /* Check options */
    if (wOption & (uint16_t)~(uint16_t)(PH_EXCHANGE_BUFFERED_BIT | PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* clear internal buffer if requested */
    if (!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        pDataParams->wTxBufLen = 0;
    }

    /* set the receive length */
    if (pRxLength != NULL)
    {
        *pRxLength = 0;
    }

    /* Fill the global TxBuffer always because separated buffers are not handeled*/
    /* retrieve transmit buffer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_GetTxBuffer(pDataParams, &pTmpBuffer, &wTmpBufferLen, &wTmpBufferSize));

    if (wTxLength != 0)
    {
        /* TxBuffer overflow check */
        if (wTxLength > (wTmpBufferSize - wTmpBufferLen))
        {
            pDataParams->wTxBufLen = 0;
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* copy data */
        memcpy(&pTmpBuffer[wTmpBufferLen], pTxBuffer, wTxLength);  /* PRQA S 3200 */
        pDataParams->wTxBufLen = pDataParams->wTxBufLen + wTxLength;
        /* Update also wTmpBufferLen */
        wTmpBufferLen = wTmpBufferLen + wTxLength;
    }

    if (wOption & PH_EXCHANGE_BUFFERED_BIT)
    {
        /* Buffer operation -> finished */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    /* Add CRC */
    if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXCRC] == PH_ON)
    {
        if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
        }
        /* TxBuffer overflow check */
        if (sizeof(uint16_t) > (wTmpBufferSize - wTmpBufferLen))
        {
            pDataParams->wTxBufLen = 0;
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_CalcCrc(pDataParams, NULL, 0, pCrc)); /* separated Tx Buffers are not supported */

        memcpy(&pTmpBuffer[wTmpBufferLen], pCrc, sizeof(uint16_t));  /* PRQA S 3200 */

        /* Update length of data */
        pDataParams->wTxBufLen = pDataParams->wTxBufLen + sizeof(uint16_t);
        wTmpBufferLen += sizeof(uint16_t);
    }

    /* Saves last bit for FDT measurement Type A */
    if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A)
    {
        if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] != 0) /* Short frame - no parity */
        {
            pDataParams->bLastBit = (pTmpBuffer[wTmpBufferLen - 1]>>(pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] - 1)) & 0x01;
        }
        else if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_PARITY] == PH_OFF)
        {
            pDataParams->bLastBit = (pTmpBuffer[wTmpBufferLen - 1]>>7) & 0x01;
        }
        else
        {
            pDataParams->bLastBit = 0x01;

            for (bLoop = 0; bLoop < 8; ++bLoop)
            {
                if (pTmpBuffer[wTmpBufferLen - 1] & (uint8_t)(1 << bLoop))
                {
                    ++pDataParams->bLastBit;
                }
            }
            pDataParams->bLastBit = pDataParams->bLastBit & 0x01;
        }
    }
    else if(pDataParams->bCardType == PHHAL_HW_CARDTYPE_FELICA) /* For Felica parity is not used */
    {
        /* Felica transmit MSB first so last bit it LSB */
        pDataParams->bLastBit = pTmpBuffer[wTmpBufferLen - 1] & 0x01;
    }
    else /* For other types last bit is not needed */
    {
        pDataParams->bLastBit = 0x00;
    }

    statusExchange = phhalHw_ISO3_Cmd_PiccExchange(
        pDataParams,
        pTmpBuffer,
        wTmpBufferLen,
        &pRxBufferInt,
        &wRxLengthInt);

    /* Reset buffered bytes */
    pDataParams->wTxBufLen = 0;

    /* Extract Data only everything is ok */
    if ((statusExchange & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* Reset receive buffer length */
        pDataParams->wRxBufLen = 0;

        /* Retrieve receive buffer properties */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_GetRxBuffer(pDataParams, &pTmpBuffer, &wTmpBufferLen, &wTmpBufferSize));

        /* Add the 2 reserved bytes for the CRC */
        wTmpBufferSize += 2;

        /* Check size */
        if (wRxLengthInt > wTmpBufferSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
        /* Extract the received data */
        memcpy(pTmpBuffer, pRxBufferInt, wRxLengthInt);  /* PRQA S 3200 */

         /* FIXXME: Workaround for Venus receiver issue that rx_align did not work correct.
               Parity check is working on the correct bits but data is received without shift an wrong bits are removed instead if parity.
               Workaround: Disable parity and get plain values, perform shift and parity calculation in SW */
        if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXALIGN] != 0)
        {
            /* Adjust size */
            dwNumReceivedBits = ISO3_REG_RX_STAT.wNumBytesReceived * 8;
            if (ISO3_REG_RX_STAT.bRxNumLastBits != 0)
            {
                dwNumReceivedBits = dwNumReceivedBits - 8 + ISO3_REG_RX_STAT.bRxNumLastBits;
            }
            dwNumReceivedBits = dwNumReceivedBits + pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXALIGN]; /*Calc new number of bits after shift */
            /* Check size */
            if (dwNumReceivedBits > (uint32_t)(wTmpBufferSize*8))
            {
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
            }

            /* Shift data */
            bShiftOver = 0;
            bTmpShiftOver = 0;
            for (wShiftIndex = 0; wShiftIndex < (uint16_t)(dwNumReceivedBits + 7)/8; wShiftIndex++) /* Count up to new length */
            {
                bShiftOver = pTmpBuffer[wShiftIndex] & ((1<<pDataParams->wCfgShadow[0x0005U])-1)<<(8-pDataParams->wCfgShadow[0x0005U]);
                pTmpBuffer[wShiftIndex] = (pTmpBuffer[wShiftIndex] << pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXALIGN]) | bTmpShiftOver;
                bTmpShiftOver = bShiftOver >> (8-pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXALIGN]);
            }

            /* Update data size and collision position with the shift */
            wTmpReceivedByte = (uint16_t)(dwNumReceivedBits + 7)/8;
            bTmpLastBit = dwNumReceivedBits & 0x07;
            wRxLengthInt = wTmpReceivedByte;
            ISO3_REG_RX_STAT.bRxNumLastBits = bTmpLastBit;
            ISO3_REG_RX_STAT.wNumBytesReceived = wTmpReceivedByte;
            if (ISO3_REG_RX_STAT.bRxCollDet == PH_ON)
            {
                ISO3_REG_RX_STAT.bRxCollPos = ISO3_REG_RX_STAT.bRxCollPos + pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXALIGN];
            }

            /* Check/remove parity if enabled */
            if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_PARITY] == PH_ON)
            {
                /* Adjust parity for first byte because it dont care for if Rx Align is set */
                if (wTmpReceivedByte > 1)
                {
                    bOneCnt = 0;
                    for (bLoop = 0; bLoop < 8; bLoop++)
                    {
                        if (((pTmpBuffer[0] >> bLoop) & 0x01) == 0x01)
                            bOneCnt++;
                    }
                    if ((bOneCnt & 0x01) == 0x01)
                    {
                        pTmpBuffer[1] = (pTmpBuffer[1]&0xFE) | 0x00;
                    }
                    else
                    {
                        pTmpBuffer[1] = (pTmpBuffer[1]&0xFE) | 0x01;
                    }
                }

                /* Verify parity (ignore parities if collision was detected) */
                PH_CHECK_SUCCESS_FCT(statusTmp, phTools_DecodeParity(
                    (ISO3_REG_RX_STAT.bRxCollDet == PH_OFF) ? PH_TOOLS_PARITY_OPTION_ODD : PH_TOOLS_PARITY_OPTION_DONT_CARE,
                    pTmpBuffer,
                    wTmpReceivedByte,
                    bTmpLastBit,
                    wTmpBufferSize,
                    pTmpBuffer,
                    &wTmpReceivedByte,
                    &bTmpLastBit));

                /* Update data size */
                wRxLengthInt = wTmpReceivedByte;
                ISO3_REG_RX_STAT.bRxNumLastBits = bTmpLastBit;
                ISO3_REG_RX_STAT.wNumBytesReceived = wTmpReceivedByte;
                if (ISO3_REG_RX_STAT.bRxCollDet == PH_ON)
                {
                    /* Adjust for removed paritys for collision position */
                    ISO3_REG_RX_STAT.bRxCollPos = ISO3_REG_RX_STAT.bRxCollPos - ISO3_REG_RX_STAT.bRxCollPos/9;
                }
            }
        }

        /* Store received data length in dataparams */
        pDataParams->wRxBufLen = pDataParams->wRxBufStartPos + wRxLengthInt;

        /* Check and remove CRC in software if required */
        if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXCRC] == PH_ON)
        {
            if (ISO3_REG_RX_STAT.bRxNumLastBits&0x07)
            {
                return PH_ADD_COMPCODE(PH_ERR_FRAMING_ERROR, PH_COMP_HAL);
            }
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_CheckCrc(pDataParams));
        }

        /* Set RxLastBits */
        pDataParams->wAdditionalInfo = (uint16_t)ISO3_REG_RX_STAT.bRxNumLastBits;

        /* Check collision */
        if (ISO3_REG_RX_STAT.bRxCollDet == PH_ON)
        {
            dwPosColl = ISO3_REG_RX_STAT.bRxCollPos + 1;
            if ( pDataParams->wCfgShadow[PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL] == PH_ON )
            {
                /* Resize the number of bytes received to the collision position */
                dwNumReceivedBits = (uint16_t)(dwPosColl-1);
                pDataParams->wRxBufLen = (uint16_t)(dwNumReceivedBits+7)>>3;
                pDataParams->wAdditionalInfo = dwNumReceivedBits & 0x7;
            }
        }
        else
        {
            dwPosColl = 0;
        }

        /* If collision */
        if (ISO3_REG_RX_STAT.bRxCollDet == PH_ON && dwPosColl != 0)
        {
            statusTmp = PH_ERR_COLLISION_ERROR;
        }
        /* Set incomplete byte status if applicable */
        else if (pDataParams->wAdditionalInfo != 0x00)
        {
            statusTmp = PH_ERR_SUCCESS_INCOMPLETE_BYTE;
        }
        else
        {
            statusTmp = PH_ERR_SUCCESS;
        }
    }
    else if ((statusExchange & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        /* Store received data length in dataparams */
        pDataParams->wRxBufLen = pDataParams->wRxBufStartPos + 0;
        statusTmp = PH_ERR_IO_TIMEOUT;
    }
    else
    {
        /* Just delegate error */
        statusTmp = statusExchange;
    }

    /* Return RxBuffer pointer */
    if (ppRxBuffer != NULL)
    {
        *ppRxBuffer = pDataParams->pRxBuffer;
    }

    /* Return RxBuffer length */
    if (pRxLength != NULL)
    {
        *pRxLength = pDataParams->wRxBufLen;
    }

    return PH_ADD_COMPCODE(statusTmp, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Init(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint16_t wSizeOfDataParams,
                            void * pBalDataParams,
                            uint8_t * pTxBuffer,
                            uint16_t wTxBufSize,
                            uint8_t * pRxBuffer,
                            uint16_t wRxBufSize,
                            uint32_t * pIntBuffer,
                            uint32_t wIntBufferSize,
                            uint8_t * pWaveShapeBuffer,
                            uint32_t dwWaveShapeBufferSize
                            )
{
    /* Check params */
    if (sizeof(phhalHw_ISO3_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_HAL);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalDataParams);
    PH_ASSERT_NULL (pTxBuffer);
    PH_ASSERT_NULL (pRxBuffer);
    PH_ASSERT_NULL (pIntBuffer);

    /* wTxBufSize == 0 is allowed */
    if (wRxBufSize == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Init data params with default values */
    pDataParams->wId                            = PH_COMP_HAL | PHHAL_HW_ISO3_ID;
    pDataParams->pBalDirectDataParams           = (void*)     pBalDataParams;
    pDataParams->pTxBuffer                      = pTxBuffer;
    pDataParams->wTxBufSize                     = wTxBufSize;
    pDataParams->wTxBufLen                      = (uint16_t)  0;
    pDataParams->pRxBuffer                      = pRxBuffer;
    pDataParams->wRxBufSize                     = wRxBufSize;
    pDataParams->wRxBufLen                      = (uint16_t)  0;
    pDataParams->wRxBufStartPos                 = (uint16_t)  0;
    pDataParams->wTxBufStartPos                 = (uint16_t)  0;
    pDataParams->wRxExpectedInputBytes          = (uint16_t)  0;
    pDataParams->bCardType                      = (uint8_t)   0;
    pDataParams->wTimingMode                    = (uint16_t)  0;
    pDataParams->bLastBit                       = (uint8_t)   0;
    pDataParams->wCurrentFieldStrength          = (uint16_t)  0;
    pDataParams->pIntBuffer                     = (uint8_t*)  pIntBuffer;
    pDataParams->wMaxIntBufferSizeBytes         = (uint16_t)  wIntBufferSize;
    pDataParams->wIntNumValidBytes              = (uint16_t)  0;
    pDataParams->wIntNumValidRxBytes            = (uint16_t)  0;
    pDataParams->dwTxCtrl                       = (uint32_t)  0;
    pDataParams->dwTxCtrl1                      = (uint32_t)  0;
    pDataParams->dwRxCtrl2                      = (uint32_t)  0;
    pDataParams->dwVenusClifRxConfigReg         = (uint32_t)  PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG_A106;
    pDataParams->dwExchangeCounter              = (uint32_t)  PHHAL_HW_ISO3_EXCHANGE_COUNT_START;
    pDataParams->dwNextTxBufferAddress          = (uint32_t)  PHHAL_HW_ISO3_REG_TX_DATA;
    pDataParams->dwCurrentWaveShapeType         = (uint32_t)  ISO3_WAVESHAPE_TYPE_INVALID;
    pDataParams->wLastFdtValueT1                = (uint32_t)  0;
    pDataParams->wLastFdtValueT2                = (uint32_t)  0;
    pDataParams->dwRxStatReg                    = (uint32_t)  0;
    pDataParams->dwDacGain                      = (uint32_t)  PHHAL_HW_ISO3_DAC_MIN_GAIN;
    pDataParams->bCurrentWaveshapeSlot          = (uint8_t)   ISO3_WAVESHAPE_SLOT_INVALID;
    pDataParams->dwIIRFilterCtrl                = (uint32_t)  PH_OFF;
    pDataParams->dwDgrmSigDetTh                 = (uint32_t)  0;
    pDataParams->dwFieldOffTimeCycles           = (uint32_t)((float32_t) PHHAL_HW_FIELD_OFF_DEFAULT * 13560.0f);
    pDataParams->bDisableAGC                    = (uint8_t)   0;
    pDataParams->dwFieldRecoveryTimeCycles      = (uint32_t)((float32_t) PHHAL_HW_FIELD_RECOVERY_DEFAULT * 13560.0f);
    pDataParams->pWaveShapeShadowBuffer         = (uint8_t*) pWaveShapeBuffer;
    pDataParams->dwWaveShapeShadowBufferSizeByte = (uint32_t) dwWaveShapeBufferSize;
    pDataParams->dwPcd2PiccTimeout              = (uint32_t) PHHAL_HW_ISO3_FPGA_PCD2PICC_TIMEOUT_DEFAULT;
    pDataParams->wDebugConfig0                  = 0x0017; /* Register 0x17 is always 0. This is to avoid inteference and reduce noise. */
    pDataParams->wDebugConfig1                  = 0x0017; /* Register 0x17 is always 0. This is to avoid inteference and reduce noise. */
    pDataParams->wDebugConfig2                  = 0x0017; /* Register 0x17 is always 0. This is to avoid inteference and reduce noise. */
    pDataParams->wDebugConfig3                  = 0x0017; /* Register 0x17 is always 0. This is to avoid inteference and reduce noise. */
    pDataParams->bTimeoutUnit                   = PHHAL_HW_TIME_MICROSECONDS;
    pDataParams->WaveShapePrescalePercent       = (uint16_t)80;

    memset(&(pDataParams->wCfgShadow), 0x00, sizeof(pDataParams->wCfgShadow));
    memset(&(pDataParams->wCfgShadowCustom), 0x00, sizeof(pDataParams->wCfgShadowCustom));
    memset(&(pDataParams->pbOvsFilterSlotsBitmap), 0x00, sizeof(pDataParams->pbOvsFilterSlotsBitmap));

    /* Init Transmitter Settings */
    ISO3_REG_TX_CTRL_0.state = PHHAL_HW_ISO3_TX_STATE_POWER_DOWN;
    ISO3_REG_TX_CTRL_0.cardType = PHHAL_HW_ISO3_CARD_TYPE_A;
    ISO3_REG_TX_CTRL_0.datarate = PHHAL_HW_ISO3_DATARATE_106K;
    ISO3_REG_TX_CTRL_0.operand = (uint32_t)((5000 * 13.56 + 0.5)); /* 5 ms */

    ISO3_REG_TX_CTRL_1.bitBoundary = 0;
    ISO3_REG_TX_CTRL_1.egt = 0;
    ISO3_REG_TX_CTRL_1.dutyCycle = PHHAL_HW_ISO3_TX_DUTY_CYCLE_50;
    ISO3_REG_TX_CTRL_1.double_data_rate = PH_OFF;
    ISO3_REG_TX_CTRL_1.disableParity = 0;
    ISO3_REG_TX_CTRL_1.rfu = 0;

    /* Init Receiver Settings */
    ISO3_REG_RX_CTRL_2.rxActivityTracker = 0;
    ISO3_REG_RX_CTRL_2.symbolTracking = 0;
    ISO3_REG_RX_CTRL_2.noisePowerThreshold = 0x8101;
    ISO3_REG_RX_CTRL_2.noisePowerDetector = PH_ON;
    ISO3_REG_RX_CTRL_2.parityDecode = PH_OFF;
    ISO3_REG_RX_CTRL_2.rfu = 0;

    /* Init Filter settings */
    ISO3_REG_IIRFILTER_CTRL.enable = PH_OFF;
    ISO3_REG_IIRFILTER_CTRL.rfu = 0;

    ISO3_REG_MLS_WHITE_NOISE_CTRL.mlsSeed = 0x1111;
    ISO3_REG_MLS_WHITE_NOISE_CTRL.rfu = 0;
    ISO3_REG_MLS_WHITE_NOISE_CTRL.enableMls = PH_OFF;
    ISO3_REG_MLS_WHITE_NOISE_CTRL.switchTx = PH_OFF;

    ISO3_REG_VENUS_DGRM_SIG_DET_TH.dgrm_signal_detection_th = PHHAL_HW_ISO3_DGRM_SIG_DET_TH_DEFAULT;

    ISO3_REG_VENUS_RX_TIMEOUT.rxTimeoutMax = 0; /* deprecated */
    ISO3_REG_VENUS_RX_TIMEOUT.rxEof2txSoftime = PHHAL_HW_ISO3_FPGA_PICC2PCD_DELAY_DEFAULT;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}


phStatus_t phhalHw_ISO3_InitAll(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint16_t wSizeOfDataParams,
                            void * pBalDirectDataParams,
                            void * pBalDispatchDataParams,
                            uint8_t * pTxBuffer,
                            uint16_t wTxBufSize,
                            uint8_t * pRxBuffer,
                            uint16_t wRxBufSize,
                            uint32_t * pIntBuffer,
                            uint32_t wIntBufferSize,
                            uint8_t * pWaveShapeShadowBuffer,
                            uint32_t dwWaveShapeBufferSizeByte
                            )
{
    phStatus_t statusTmp;
    /* Check params */
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalDispatchDataParams);

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Init(pDataParams, wSizeOfDataParams, pBalDirectDataParams, pTxBuffer,
        wTxBufSize, pRxBuffer, wRxBufSize, pIntBuffer, wIntBufferSize, pWaveShapeShadowBuffer, dwWaveShapeBufferSizeByte));

    pDataParams->pBalDispatchDataParams         = (void*) pBalDispatchDataParams;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}


phStatus_t phhalHw_ISO3_WriteRegister(
                            phhalHw_ISO3_DataParams_t *  pDataParams,
                            uint8_t bAddress,
                            uint8_t bValue
                            )
{
    /* Unsupported command due size of bAddress and bValue (too small) */
    if(pDataParams || bAddress || bValue);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_ReadRegister(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint8_t bAddress,
                            uint8_t * pValue
                            )
{
    /* Unsupported command due size of bAddress and pValue (too small) */
    if(pDataParams || bAddress || pValue);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_ApplyProtocolSettings(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint8_t bCardType
                            )
{
    /* Declare variable */
    phStatus_t statusTmp;
    uint32_t dwTxCtrlTmp = 0;
    uint16_t * pDefaultShadow;
    uint16_t wConfig;
    uint16_t  wIndex;
    uint16_t numSlots = 0;
    uint16_t wShadowCount;
    uint8_t  bUseDefaultShadow;

    /* Check params */
    PH_ASSERT_NULL(pDataParams);

    /* Store new card type */
    if (bCardType != PHHAL_HW_CARDTYPE_CURRENT)
    {
        pDataParams->bCardType = bCardType;
        bUseDefaultShadow = 1;
        pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MICROSECONDS;
    }
    else
    {
        bUseDefaultShadow = 0;
    }

    switch (bCardType)
    {
    case PHHAL_HW_CARDTYPE_ISO14443A:
        /* Set default shadow */
        pDefaultShadow = (uint16_t *) wISO3_DefaultShadow_14443a;
        wShadowCount = sizeof(wISO3_DefaultShadow_14443a) / (sizeof(uint16_t) * 2);

        dwTxCtrlTmp = pDataParams->dwTxCtrl;
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_NUMB, &numSlots));
        if (numSlots != 7)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_SetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_NUMB, 7));
        }

        /* Store config in data params */
        ISO3_REG_TX_CTRL_0.cardType = PHHAL_HW_ISO3_CARD_TYPE_A;
        ISO3_REG_TX_CTRL_0.datarate = PHHAL_HW_ISO3_DATARATE_106K;
        ISO3_REG_TX_CTRL_0.iso15693mode = PH_OFF;

        /* TODO: Parity, TXCRC, RXCRC, DeafBits */

        /* Send to fpga if something has changed */
        if (dwTxCtrlTmp != pDataParams->dwTxCtrl)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_0, pDataParams->dwTxCtrl));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(pDataParams));
        }
        /* configure receiver too */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_VenusConfig(pDataParams, PHHAL_HW_CARDTYPE_ISO14443A, PHHAL_HW_ISO3_DATARATE_106K));

        break;
    case PHHAL_HW_CARDTYPE_ISO14443B:
        /* Set default shadow */
        pDefaultShadow = (uint16_t *) wISO3_DefaultShadow_14443b;
        wShadowCount = sizeof(wISO3_DefaultShadow_14443b) / (sizeof(uint16_t) * 2);

        dwTxCtrlTmp = pDataParams->dwTxCtrl;

        /* Store config in data params */
        ISO3_REG_TX_CTRL_0.cardType = PHHAL_HW_ISO3_CARD_TYPE_B;
        ISO3_REG_TX_CTRL_0.datarate = PHHAL_HW_ISO3_DATARATE_106K;
        ISO3_REG_TX_CTRL_0.iso15693mode = PH_OFF;

        /* TODO: Parity, TXCRC, RXCRC, DeafBits */

        /* Send to fpga if something has changed */
        if (dwTxCtrlTmp != pDataParams->dwTxCtrl)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_0, pDataParams->dwTxCtrl));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(pDataParams));
        }
        /* configure receiver too */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_VenusConfig(pDataParams, PHHAL_HW_CARDTYPE_ISO14443B, PHHAL_HW_ISO3_DATARATE_106K));

        phhalHw_ISO3_SetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_NUMB, 7);

        break;
    case PHHAL_HW_CARDTYPE_FELICA:
        pDefaultShadow = (uint16_t*) wISO3_DefaultShadow_Felica;
        wShadowCount = sizeof(wISO3_DefaultShadow_Felica) / (sizeof(uint16_t) * 2);

        dwTxCtrlTmp = pDataParams->dwTxCtrl;
        ISO3_REG_TX_CTRL_0.cardType = PHHAL_HW_ISO3_CARD_TYPE_F;
        ISO3_REG_TX_CTRL_0.datarate = PHHAL_HW_ISO3_DATARATE_212K;
        ISO3_REG_TX_CTRL_0.iso15693mode = PH_OFF;

        /* Send to fpga if something has changed */
        if (dwTxCtrlTmp != pDataParams->dwTxCtrl)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_0, pDataParams->dwTxCtrl));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(pDataParams));
        }
        /* configure receiver too */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_VenusConfig(pDataParams, PHHAL_HW_CARDTYPE_FELICA, PHHAL_HW_ISO3_DATARATE_212K));

        phhalHw_ISO3_SetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_NUMB, 3);

        break;
    case PHHAL_HW_CARDTYPE_ISO15693:
        /* Set default shadow */
        pDefaultShadow = (uint16_t *) wISO3_DefaultShadow_15693;
        wShadowCount = sizeof(wISO3_DefaultShadow_15693) / (sizeof(uint16_t) * 2);

        dwTxCtrlTmp = pDataParams->dwTxCtrl;

        /* Store config in data params */
        ISO3_REG_TX_CTRL_0.cardType = PHHAL_HW_ISO3_CARD_TYPE_ISO_15693;
        ISO3_REG_TX_CTRL_0.iso15693mode = PHHAL_HW_ISO3_ISO_15693_MODE_4;
        ISO3_REG_TX_CTRL_0.datarate = PHHAL_HW_ISO3_DATARATE_15693_4_26K;

        /* TODO: Parity, TXCRC, RXCRC, DeafBits */

        /* Send to fpga if something has changed */
        if (dwTxCtrlTmp != pDataParams->dwTxCtrl)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_0, pDataParams->dwTxCtrl));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(pDataParams));
        }
        /* configure receiver too */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_VenusConfig(pDataParams, PHHAL_HW_CARDTYPE_ISO15693, PHHAL_HW_RF_RX_DATARATE_HIGH));

        /* TODO: ISO15693 wave shapes tend to be longer. Change number of slots accordingly */
        phhalHw_ISO3_SetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_NUMB, 1);

        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    /* Copy over default shadow contents into current shadow. */
    if (bUseDefaultShadow)
    {
        /* Initialize config shadow */
        memset(pDataParams->wCfgShadow, 0x00, sizeof(pDataParams->wCfgShadow));  /* PRQA S 3200 */
        memset(pDataParams->wCfgShadowCustom, 0x00, sizeof(pDataParams->wCfgShadowCustom));  /* PRQA S 3200 */

        for (wIndex = 0; wIndex < wShadowCount; ++wIndex)
        {
            wConfig = pDefaultShadow[wIndex << 1];
            if (wConfig < PH_CONFIG_CUSTOM_BEGIN)
            {
                pDataParams->wCfgShadow[wConfig] = pDefaultShadow[(wIndex << 1) + 1];
            }
            else
            {
                pDataParams->wCfgShadowCustom[wConfig - PH_CONFIG_CUSTOM_BEGIN] = pDefaultShadow[(wIndex << 1) + 1];
            }
        }
    }

    /* Apply shadowed registers */
    for (wIndex = 0; wIndex < wShadowCount; ++wIndex)
    {
        /* Get wConfig */
        wConfig = pDefaultShadow[wIndex << 1];

        if (wConfig == PHHAL_HW_CONFIG_TIMEOUT_VALUE_US)
        {
            if (pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MICROSECONDS)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_SetConfig(pDataParams, wConfig, pDataParams->wCfgShadow[wConfig]));
            }
        }
        else if (wConfig == PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS)
        {
            if (pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MILLISECONDS)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_SetConfig(pDataParams, wConfig, pDataParams->wCfgShadow[wConfig]));
            }
        }
        else if (wConfig < PH_CONFIG_CUSTOM_BEGIN)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_SetConfig(pDataParams, wConfig, pDataParams->wCfgShadow[wConfig]));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_SetConfig(pDataParams, wConfig, pDataParams->wCfgShadowCustom[wConfig - PH_CONFIG_CUSTOM_BEGIN]));
        }
    }

    /* MIFARE Crypto1 state is disabled by default */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams, PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1, PH_ON));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_SetConfig(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint16_t wConfig,
                            uint16_t wValue
                            )
{
    phStatus_t statusTmp;
	uint16_t wWaveShapeSlotNumb;
    uint8_t  bCounter;
    uint32_t dwTmpValue = 0;
    uint16_t wFilterOverflowFlag;
    phhalHw_ISO3_TxCtrlRegister_t tmp;

    /* Check params */
    PH_ASSERT_NULL(pDataParams);

    switch (wConfig)
    {
    case PHHAL_HW_CONFIG_PARITY:
        if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443B)
        {
            if (wValue == PH_OFF)
            {
                /* For Type B parity is used as StopBit so the setting always needs to be enabled */
                if (ISO3_REG_TX_CTRL_1.disableParity != PH_OFF)
                {
                    dwTmpValue = pDataParams->dwTxCtrl1;
                    (*((phhalHw_ISO3_TxCtrlRegister_1_t*)&(dwTmpValue))).disableParity = PH_OFF;
                    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                        pDataParams,
                        PHHAL_HW_ISO3_REG_TX_CTRL_1,
                        (uint8_t*) &dwTmpValue,
                        sizeof(dwTmpValue)));
                    pDataParams->dwTxCtrl1 = dwTmpValue;
                }
                if (ISO3_REG_VENUS_CLIF_RX_CONFIG.RX_PARITY_ENABLE != PH_OFF)
                {
                    dwTmpValue = pDataParams->dwVenusClifRxConfigReg;
                    (*((phhalHw_ISO3_VenusClifRxConfigRegStruct_t*)&(dwTmpValue))).RX_PARITY_ENABLE = PH_OFF;
                    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                        pDataParams,
                        PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG,
                        (uint8_t*) &dwTmpValue,
                        sizeof(dwTmpValue)));
                    pDataParams->dwVenusClifRxConfigReg = dwTmpValue;
                }
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
            }
        }
        else
        {
            if (wValue == PH_OFF)
            {
                /* if parity changed, update transmitter control 1 register */
                if (ISO3_REG_TX_CTRL_1.disableParity != PH_ON)
                {
                    dwTmpValue = pDataParams->dwTxCtrl1;
                    (*((phhalHw_ISO3_TxCtrlRegister_1_t*)&(dwTmpValue))).disableParity = PH_ON;
                    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                        pDataParams,
                        PHHAL_HW_ISO3_REG_TX_CTRL_1,
                        (uint8_t*) &dwTmpValue,
                        sizeof(dwTmpValue)));
                    pDataParams->dwTxCtrl1 = dwTmpValue;
                }

                if (ISO3_REG_VENUS_CLIF_RX_CONFIG.RX_PARITY_ENABLE != PH_OFF)
                {
                    dwTmpValue = pDataParams->dwVenusClifRxConfigReg;
                    (*((phhalHw_ISO3_VenusClifRxConfigRegStruct_t*)&(dwTmpValue))).RX_PARITY_ENABLE = PH_OFF;
                    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                        pDataParams,
                        PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG,
                        (uint8_t*) &dwTmpValue,
                        sizeof(dwTmpValue)));
                    pDataParams->dwVenusClifRxConfigReg = dwTmpValue;
                }
            }
            else if (wValue == PH_ON)
            {
                /* if parity changed, update transmitter control 1 register */
                if (ISO3_REG_TX_CTRL_1.disableParity != PH_OFF)
                {
                    dwTmpValue = pDataParams->dwTxCtrl1;
                    (*((phhalHw_ISO3_TxCtrlRegister_1_t*)&(dwTmpValue))).disableParity = PH_OFF;
                    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                        pDataParams,
                        PHHAL_HW_ISO3_REG_TX_CTRL_1,
                        (uint8_t*) &dwTmpValue,
                        sizeof(dwTmpValue)));
                    pDataParams->dwTxCtrl1 = dwTmpValue;
                }

                /* FIXXME: Workaround for Venus receiver issue that rx_align did not work correct.
                   Parity check is working on the correct bits but data is received without shift an wrong bits are removed instead if parity.
                   Workaround: Disable parity and get plain values, perform shift and parity calculation in SW */
                /* Only apply if rx align is not set */
                if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXALIGN] == 0);
                {
                    if (ISO3_REG_VENUS_CLIF_RX_CONFIG.RX_PARITY_ENABLE != PH_ON)
                    {
                        dwTmpValue = pDataParams->dwVenusClifRxConfigReg;
                        (*((phhalHw_ISO3_VenusClifRxConfigRegStruct_t*)&(dwTmpValue))).RX_PARITY_ENABLE = PH_ON;
                        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                            pDataParams,
                            PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG,
                            (uint8_t*) &dwTmpValue,
                            sizeof(dwTmpValue)));
                        pDataParams->dwVenusClifRxConfigReg = dwTmpValue;
                    }
                }
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_TXCRC:
        /* Store config in shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        /* Just indicate that everything is allright */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_RXCRC:
        /* Store config in shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        /* Just indicate that everything is allright */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_TXLASTBITS:
        /* Store config in shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        /* Just indicate that everything is allright */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_RXALIGN:
        /* check parameter */
        if (wValue > 7)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        dwTmpValue = pDataParams->dwVenusClifRxConfigReg;
        (*((phhalHw_ISO3_VenusClifRxConfigRegStruct_t*)&(dwTmpValue))).RX_BIT_ALIGN = 0; /* Set to zero in any case as shift anyway did not work */
        /* FIXXME: Workaround for Venus receiver issue that rx_align did not work correct.
            Parity check is working on the correct bits but data is received without shift an wrong bits are removed instead if parity.
            Workaround: Disable parity and get plain values, perform shift and parity calculation in SW */
        /* Disable Parity to perform check in SW */
        if (wValue != 0)
        {
            /* Disable parity */
            (*((phhalHw_ISO3_VenusClifRxConfigRegStruct_t*)&(dwTmpValue))).RX_PARITY_ENABLE = PH_OFF;
        }
        else
        {
            /* Restore Parity */
            (*((phhalHw_ISO3_VenusClifRxConfigRegStruct_t*)&(dwTmpValue))).RX_PARITY_ENABLE = pDataParams->wCfgShadow[PHHAL_HW_CONFIG_PARITY];
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
            pDataParams,
            PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG,
            (uint8_t*) &dwTmpValue,
            sizeof(dwTmpValue)));
        pDataParams->dwVenusClifRxConfigReg = dwTmpValue;
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_RXDEAFBITS:
        if (pDataParams->bCardType != PHHAL_HW_CARDTYPE_ISO14443A &&
            pDataParams->bCardType != PHHAL_HW_CARDTYPE_ISO14443B &&
            wValue != 0) /* Other card types support only 0 */
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
        }

        dwTmpValue = 0; /* Init with zero for not supported datarates */

        switch (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXDATARATE])
        {
        case PHHAL_HW_RF_DATARATE_106:
            dwTmpValue = 1280*wValue;
            break;
        case PHHAL_HW_RF_DATARATE_212:
            dwTmpValue = 640*wValue;
            break;
        case PHHAL_HW_RF_DATARATE_424:
            dwTmpValue = 320*wValue;
            break;
        case PHHAL_HW_RF_DATARATE_848:
            dwTmpValue = 160*wValue;
            break;
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
            pDataParams,
            PHHAL_HW_ISO3_REG_RX_DELAY,
            (uint8_t*) &dwTmpValue,
            sizeof(dwTmpValue)));

        /* Store config in shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_TXWAIT_US:
        /* Store config in shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        /* Just indicate that everything is allright */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL:
        if (wValue == PH_OFF)
        {
            if (ISO3_REG_VENUS_CLIF_RX_CONFIG.VALUES_AFTER_COLLISION != PH_ON)
            {
                dwTmpValue = pDataParams->dwVenusClifRxConfigReg;
                (*((phhalHw_ISO3_VenusClifRxConfigRegStruct_t*)&(dwTmpValue))).VALUES_AFTER_COLLISION = PH_ON;
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                    pDataParams,
                    PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG,
                    (uint8_t*) &dwTmpValue,
                    sizeof(dwTmpValue)));
                pDataParams->dwVenusClifRxConfigReg = dwTmpValue;
            }
        }
        else if (wValue == PH_ON)
        {
            if (ISO3_REG_VENUS_CLIF_RX_CONFIG.VALUES_AFTER_COLLISION != PH_OFF)
            {
                dwTmpValue = pDataParams->dwVenusClifRxConfigReg;
                (*((phhalHw_ISO3_VenusClifRxConfigRegStruct_t*)&(dwTmpValue))).VALUES_AFTER_COLLISION = PH_OFF;
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                    pDataParams,
                    PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG,
                    (uint8_t*) &dwTmpValue,
                    sizeof(dwTmpValue)));
                pDataParams->dwVenusClifRxConfigReg = dwTmpValue;
            }
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Store config in shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        /* Just indicate that everything is allright */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);

    case PHHAL_HW_CONFIG_TXDATARATE:
        /* check card type first. different card types support different data rates */
        if ((pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A ||
             pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443B))
        {
            /* valid settings for card types 14443A and 14443B */
            if ((wValue == PHHAL_HW_RF_DATARATE_106) ||
                (wValue == PHHAL_HW_RF_DATARATE_212) ||
                (wValue == PHHAL_HW_RF_DATARATE_424) ||
                (wValue == PHHAL_HW_RF_DATARATE_848))
            {
                /* if datarate changed, update transmitter control register and wave shape */
                if (ISO3_REG_TX_CTRL_0.datarate != (wValue & 0x3u))
                {
                    ISO3_REG_TX_CTRL_0.datarate = wValue & 0x3u;
                    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                        pDataParams,
                        PHHAL_HW_ISO3_REG_TX_CTRL_0,
                        (uint8_t*) &pDataParams->dwTxCtrl,
                        sizeof(pDataParams->dwTxCtrl)));

                    /* un-set the double-data-rate flag if it is set */
                    if (ISO3_REG_TX_CTRL_1.double_data_rate == PH_ON)
                    {
                        ISO3_REG_TX_CTRL_1.double_data_rate = PH_OFF;
                        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                            pDataParams,
                            PHHAL_HW_ISO3_REG_TX_CTRL_1,
                            (uint8_t*) &pDataParams->dwTxCtrl1,
                            sizeof(pDataParams->dwTxCtrl1)));
                    }
                }
                /* Store config in shadow */
                pDataParams->wCfgShadow[wConfig] = wValue;
                return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
            }
            else
            {
                /* invalid data rate for the current card type */
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_FELICA)
        {
            /* valid settings for card type Felica */
            if ((wValue == PHHAL_HW_RF_DATARATE_212) ||
                (wValue == PHHAL_HW_RF_DATARATE_424))
            {
                /* if datarate changed, update transmitter control register and wave shape */
                if (ISO3_REG_TX_CTRL_0.datarate != (wValue & 0x3u))
                {
                    ISO3_REG_TX_CTRL_0.datarate = wValue & 0x3u;
                    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                        pDataParams,
                        PHHAL_HW_ISO3_REG_TX_CTRL_0,
                        (uint8_t*) &pDataParams->dwTxCtrl,
                        sizeof(pDataParams->dwTxCtrl)));

                    /* un-set the double-data-rate flag if it is set */
                    if (ISO3_REG_TX_CTRL_1.double_data_rate == PH_ON)
                    {
                        ISO3_REG_TX_CTRL_1.double_data_rate = PH_OFF;
                        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                            pDataParams,
                            PHHAL_HW_ISO3_REG_TX_CTRL_1,
                            (uint8_t*) &pDataParams->dwTxCtrl1,
                            sizeof(pDataParams->dwTxCtrl1)));
                    }
                }
                /* Store config in shadow */
                pDataParams->wCfgShadow[wConfig] = wValue;
                return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
            }
            else
            {
                /* invalid data rate for the current card type */
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO15693)
        {
            /* valid settings for card type ISO15693 */
            /* for ICODE 1 out of 256 coding only the basic data rate (1.6kb/s) is used in operation.
               The data rates 13.24 kb/s, 26.48 kb/s and 53kb/s can be configured explicitly for 1 out of 256 coding if necessary. */
            if (wValue == PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_256)
            {
                /* only this data rate uses 1 out of 256 coding */
                ISO3_REG_TX_CTRL_0.datarate = 0;
                ISO3_REG_TX_CTRL_0.iso15693mode = PHHAL_HW_ISO3_ISO_15693_MODE_256;
                ISO3_REG_TX_CTRL_1.double_data_rate = PH_OFF;
            }
            else if ((wValue == PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4) ||
                     (wValue == PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4_53) ||
                     (wValue == PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4_106) ||
                     (wValue == PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4_212))
            {
                /* lowest data rate uses 1 out of 256 coding. all others use 1 out of 4 coding */
                ISO3_REG_TX_CTRL_0.iso15693mode = PHHAL_HW_ISO3_ISO_15693_MODE_4;
                switch (wValue)
                {
                case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4: ISO3_REG_TX_CTRL_0.datarate = 0;
                    break;
                case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4_53: ISO3_REG_TX_CTRL_0.datarate = 1;
                    break;
                case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4_106: ISO3_REG_TX_CTRL_0.datarate = 2;
                    break;
                case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4_212: ISO3_REG_TX_CTRL_0.datarate = 3;
                }

                /* un-set the double-data-rate flag if it is set */
                if (ISO3_REG_TX_CTRL_1.double_data_rate == PH_ON)
                {
                    ISO3_REG_TX_CTRL_1.double_data_rate = PH_OFF;
                }

                /* Store config in shadow */
                pDataParams->wCfgShadow[wConfig] = wValue;
            }
            else
            {
                /* invalid data rate for the current card type */
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        else
        {
            /* current card type is not supported */
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
        }
        /* Store config in shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_ISO3_CONFIG_BIT_BOUNDARY:
        if (ISO3_REG_TX_CTRL_1.bitBoundary != (wValue & 0xFFu))
        {
            ISO3_REG_TX_CTRL_1.bitBoundary = wValue & 0xFFu;
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_1, (uint8_t*) &pDataParams->dwTxCtrl1,
                sizeof(pDataParams->dwTxCtrl1)));
        }
        pDataParams->wCfgShadowCustom[wConfig - PH_CONFIG_CUSTOM_BEGIN] = wValue;  /* Use custom config shadow */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_ISO3_CONFIG_EGT:
        if (ISO3_REG_TX_CTRL_1.egt != (wValue & 0x1FFFu))
        {
            ISO3_REG_TX_CTRL_1.egt = wValue & 0x1FFFu;
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_1, (uint8_t*) &pDataParams->dwTxCtrl1,
                sizeof(pDataParams->dwTxCtrl1)));
        }
        pDataParams->wCfgShadowCustom[wConfig - PH_CONFIG_CUSTOM_BEGIN] = wValue;  /* Use custom config shadow */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_ISO3_CONFIG_DUTY_CYCLE:
        if (ISO3_REG_TX_CTRL_1.dutyCycle != (wValue & 0x07u))
        {
            ISO3_REG_TX_CTRL_1.dutyCycle = wValue & 0x07u;
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_1, (uint8_t*) &pDataParams->dwTxCtrl1,
                sizeof(pDataParams->dwTxCtrl1)));
        }
        pDataParams->wCfgShadowCustom[wConfig - PH_CONFIG_CUSTOM_BEGIN] = wValue;  /* Use custom config shadow */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_RXDATARATE:
        /* configure venus receiver */
        if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443A ||
            pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO14443B)
        {
            if (wValue == PHHAL_HW_RF_DATARATE_106 || wValue == PHHAL_HW_RF_DATARATE_212 ||
                wValue == PHHAL_HW_RF_DATARATE_424 || wValue == PHHAL_HW_RF_DATARATE_848)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_VenusConfig(pDataParams, pDataParams->bCardType, (uint8_t)wValue));
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_FELICA)
        {
            if (wValue == PHHAL_HW_RF_DATARATE_212 || wValue == PHHAL_HW_RF_DATARATE_424)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_VenusConfig(pDataParams, pDataParams->bCardType, (uint8_t)wValue));
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        else if (pDataParams->bCardType == PHHAL_HW_CARDTYPE_ISO15693)
        {
            if (wValue == PHHAL_HW_RF_RX_DATARATE_HIGH)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_VenusConfig(pDataParams, pDataParams->bCardType,
                    (uint8_t)PHHAL_HW_RF_RX_DATARATE_HIGH));
            }
            else if (wValue == PHHAL_HW_RF_RX_DATARATE_FAST_HIGH)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_VenusConfig(pDataParams, pDataParams->bCardType,
                    (uint8_t)PHHAL_HW_RF_RX_DATARATE_FAST_HIGH));
            }
            else if (wValue == PHHAL_HW_RF_RX_DATARATE_LOW)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_VenusConfig(pDataParams, pDataParams->bCardType,
                    (uint8_t)PHHAL_HW_RF_RX_DATARATE_LOW));
            }
            else if (wValue == PHHAL_HW_RF_RX_DATARATE_FAST_LOW)
            {
                /* data rate not supported */
                return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* Store config in shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_MODINDEX:
        /* Store config in shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        /* Just indicate that everything is allright */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_ASK100:
        /* Store config in shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        /* Just indicate that everything is allright */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_US:
        /* Store config in shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS] = 0; /* Time unit is mutual exclusive. microsec or milisec not both */
        pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MICROSECONDS;
        /* Just indicate that everything is allright */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS:
        /* Store config in shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_US] = 0; /* Time unit is mutual exclusive. microsec or milisec not both */
        pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MILLISECONDS;
        /* Just indicate that everything is allright */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_TIMING_MODE:
        pDataParams->wTimingMode = wValue;
        /* Just indicate that everything is allright */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_TIMING_US:
        /* Just indicate that the parameter is not supported */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_TIMING_MS:
        /* Just indicate that the parameter is not supported */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_FIELD_OFF_TIME:
        /* Check if the given Field Off duration in clk cycles can fit in transmitter TX_CTRL0.operand 25bit register */
        if ((double)((double)wValue * 13560.0) > 0x1FFFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        if (wValue < 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        pDataParams->dwFieldOffTimeCycles = (uint32_t) ((float32_t)wValue * 13560.0);
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_ISO3_CONFIG_FIELD_OFF_TIME_US:
        /* Check if the given Field Off duration in clk cycles can fit in transmitter TX_CTRL0.operand 25bit register */
        if ((double)((double)wValue * 13.56) > 0x1FFFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        if (wValue < 3) /*Limit is 35 clock cycles */
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        pDataParams->dwFieldOffTimeCycles = (uint32_t) (((float32_t)wValue * 13.56) + 0.5);
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_ISO3_CONFIG_FIELD_OFF_TIME_CYCLES:
        /* Check if the given Field Off duration in clk cycles can fit in transmitter TX_CTRL0.operand 25bit register */
        if ((double)wValue > 0x1FFFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        if (wValue < 35) /*Limit is 35 clock cycles */
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        pDataParams->dwFieldOffTimeCycles = (uint32_t)wValue;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_FIELD_RECOVERY_TIME:
        /* Check if the given recovery time duration in ETU can fit in transmitter TX_CTRL0.operand 25bit register */
        if ((double)((double)wValue * 13560.0) > 0x1FFFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        /* sets the field recovery time (or startup time, if a request is transmitted after fieldOn) */
        pDataParams->dwFieldRecoveryTimeCycles = (uint32_t) ((float32_t)wValue * 13560.0);
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_ISO3_CONFIG_FIELD_RECOVERY_TIME_US:
        /* Check if the given recovery time duration in ETU can fit in transmitter TX_CTRL0.operand 25bit register */
        if ((double)((double)wValue * 13.56) > 0x1FFFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        pDataParams->dwFieldRecoveryTimeCycles = (uint32_t) (((float32_t)wValue * 13.56) + 0.5);
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_ISO3_CONFIG_DEBUG_OUTPUT0:
        if (wValue > 31)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        pDataParams->wDebugConfig0 = (uint16_t)wValue;
        dwTmpValue = ((0xFF & pDataParams->wDebugConfig3) << 24) |
                     ((0xFF & pDataParams->wDebugConfig2) << 16) |
                     ((0xFF & pDataParams->wDebugConfig1) << 8) |
                     ((0xFF & pDataParams->wDebugConfig0) << 0);
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
            pDataParams,
            PHHAL_HW_ISO3_REG_DEBUG_CONFIG,
            (uint8_t*) &dwTmpValue,
            sizeof(dwTmpValue)));
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_ISO3_CONFIG_DEBUG_OUTPUT1:
        if (wValue > 31)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        pDataParams->wDebugConfig1 = (uint16_t)wValue;
        dwTmpValue = ((0xFF & pDataParams->wDebugConfig3) << 24) |
                     ((0xFF & pDataParams->wDebugConfig2) << 16) |
                     ((0xFF & pDataParams->wDebugConfig1) << 8) |
                     ((0xFF & pDataParams->wDebugConfig0) << 0);
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
            pDataParams,
            PHHAL_HW_ISO3_REG_DEBUG_CONFIG,
            (uint8_t*) &dwTmpValue,
            sizeof(dwTmpValue)));
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_ISO3_CONFIG_DEBUG_OUTPUT2:
        if (wValue > 31)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        pDataParams->wDebugConfig2 = (uint16_t)wValue;
        dwTmpValue = ((0xFF & pDataParams->wDebugConfig3) << 24) |
                     ((0xFF & pDataParams->wDebugConfig2) << 16) |
                     ((0xFF & pDataParams->wDebugConfig1) << 8) |
                     ((0xFF & pDataParams->wDebugConfig0) << 0);
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
            pDataParams,
            PHHAL_HW_ISO3_REG_DEBUG_CONFIG,
            (uint8_t*) &dwTmpValue,
            sizeof(dwTmpValue)));
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_ISO3_CONFIG_DEBUG_OUTPUT3:
        if (wValue > 31)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        pDataParams->wDebugConfig3 = (uint16_t)wValue;
        dwTmpValue = ((0xFF & pDataParams->wDebugConfig3) << 24) |
                     ((0xFF & pDataParams->wDebugConfig2) << 16) |
                     ((0xFF & pDataParams->wDebugConfig1) << 8) |
                     ((0xFF & pDataParams->wDebugConfig0) << 0);
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
            pDataParams,
            PHHAL_HW_ISO3_REG_DEBUG_CONFIG,
            (uint8_t*) &dwTmpValue,
            sizeof(dwTmpValue)));
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);

    case PHHAL_HW_CONFIG_SYMBOL_START:
        /* Just indicate that the parameter is not supported */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_SYMBOL_END:
        /* Just indicate that the parameter is not supported */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_CARD_TYPE:
        /* The card type is changed in ApplyProtocolSettings. Just indicate that everything is allright */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_SUBCARRIER:
        /* Just indicate that the parameter is not supported */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_RXBUFFER_STARTPOS:
        /* Boundary check */
        if (wValue >= (pDataParams->wRxBufSize - PHHAL_HW_ISO3_CRC_BUFFER_LEN))
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
        /* Set start position */
        pDataParams->wRxBufStartPos = wValue;
        pDataParams->wRxBufLen = wValue;
        /* Preserve RxBuffer contents if needed */
        if (pDataParams->pTxBuffer == pDataParams->pRxBuffer)
        {
            pDataParams->wTxBufStartPos = pDataParams->wRxBufStartPos;
        }
        else
        {
            pDataParams->wTxBufStartPos = 0;
        }
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_TXBUFFER_LENGTH:
        /* Check parameter */
        if ((pDataParams->wTxBufStartPos + wValue) > pDataParams->wTxBufSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
        /* set buffer length */
        pDataParams->wTxBufLen = wValue;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_TXBUFFER:
        /* Check additional info parameter */
        if ((pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo) >= pDataParams->wTxBufSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
        /* Modify TxBuffer byte */
        pDataParams->pTxBuffer[pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo] = (uint8_t)wValue;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_MAX_PRECACHED_BYTES:
        /* Just indicate that the parameter is not supported */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_BAL_CONNECTION:
        /* Just indicate that the parameter is not supported */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_SERIAL_BITRATE:
        /* Just indicate that the parameter is not supported */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1:
        /* Just indicate that everything is allright */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_ADDITIONAL_INFO:
        /* Modify additional info parameter */
        pDataParams->wAdditionalInfo = wValue;
        /* Just indicate that everything is allright */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT:
        if (wValue == PH_OFF)
        {
            pDataParams->bRfResetAfterTo = PH_OFF;
        }
        else
        {
            pDataParams->bRfResetAfterTo = PH_ON;
        }
        /* Just indicate that everything is allright */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_SETMINFDT:
        /* Just indicate that the parameter is not supported */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_FRAME_SYNC:
        /* Just indicate that the parameter is not supported */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_TXDATANUM:
        /* Just indicate that the parameter is not supported */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_SET_READER_IC:
        /* Just indicate that the parameter is not supported */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    case PHHAL_HW_CONFIG_COMMUNICATION_CONFIG:
        /* Just indicate that the parameter is not supported */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    case PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_NUMB:
        if(wValue > PHHAL_HW_ISO3_MAX_NUMB_WS_SLOTS || wValue == 0x0U )
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* Store config from shadow */
        pDataParams->wWaveShapeBufferConfigs[PHHAL_HW_ISO3_WAVESHAPE_SLOTS_NUMB] = wValue;

        pDataParams->bCurrentWaveshapeSlot = ISO3_WAVESHAPE_SLOT_INVALID;

        /* Calculate size of single partition (make size fit an even number of samples) */
        pDataParams->wWaveShapeBufferConfigs[PHHAL_HW_ISO3_WAVESHAPE_SLOT_SIZE] =
            (ISO3_WAVESHAPE_BUFFER_SIZE_BYTES - ISO3_WAVESHAPE_RESERVED_BUFFER_OFFSET * 4) / wValue;

        pDataParams->wWaveShapeBufferConfigs[PHHAL_HW_ISO3_WAVESHAPE_SLOT_SIZE] -=
            pDataParams->wWaveShapeBufferConfigs[PHHAL_HW_ISO3_WAVESHAPE_SLOT_SIZE] %
            sizeof(uint32_t);

        /* Reset all waveshape slots */
        for(bCounter = 0; bCounter < ((PHHAL_HW_ISO3_MAX_NUMB_WS_SLOTS + 7) / 8); bCounter++)
        {
            pDataParams->bWaveShapeBufferSlotsBitmap[bCounter] = (uint8_t)0x0;
        }

        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOT_FIELD_ON:
    case PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOT_FIELD_OFF:
        phhalHw_ISO3_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_NUMB, &wWaveShapeSlotNumb);
        /* check slots */
        if(wWaveShapeSlotNumb == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
        }
        if(wValue >= wWaveShapeSlotNumb && wValue != PHHAL_HW_ISO3_WAVESHAPE_SLOT_DEFAULT_ID)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Check if slot is in use (except DEFAULT slot) */
        if(wValue != PHHAL_HW_ISO3_WAVESHAPE_SLOT_DEFAULT_ID && ISO3_WAVESHAPE_SLOT_GET_IN_USE(wValue) == false)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        if(wConfig == PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOT_FIELD_ON)
        {
            pDataParams->wWaveShapeBufferConfigs[PHHAL_HW_ISO3_WAVESHAPE_SLOT_FIELD_ON] = wValue;
        }
        else
        {
            pDataParams->wWaveShapeBufferConfigs[PHHAL_HW_ISO3_WAVESHAPE_SLOT_FIELD_OFF] = wValue;
        }
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOT_EXCHANGE:
        phhalHw_ISO3_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_NUMB, &wWaveShapeSlotNumb);
        /* check slots */
        if(wWaveShapeSlotNumb == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
        }
        if(wValue >= wWaveShapeSlotNumb)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* Check if slot is in use */
        if(ISO3_WAVESHAPE_SLOT_GET_IN_USE(wValue) == false)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
        }

        /* Store config in shadow */
        pDataParams->wWaveShapeBufferConfigs[PHHAL_HW_ISO3_WAVESHAPE_SLOT_EXCHANGE] = wValue;
        /* Just indicate that everything is allright */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);

    case PHHAL_HW_ISO3_CONFIG_RX_NOISE_POWER_TH:
        /* the threshold is a 10bit value */
        if (wValue > 0xffff)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* return success, if the register value did not change */
        if (wValue == ISO3_REG_VENUS_DGRM_SIG_DET_TH.dgrm_signal_detection_th)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
        }

        /* set updated threshold in the FPGA */
        ISO3_REG_VENUS_DGRM_SIG_DET_TH.dgrm_signal_detection_th = (uint32_t)(wValue & 0xffff);
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
            pDataParams,
            PHHAL_HW_ISO3_REG_DGRM_SIG_DET,
            (uint8_t*)&pDataParams->dwDgrmSigDetTh,
            sizeof(pDataParams->dwDgrmSigDetTh)));

        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);

    case PHHAL_HW_ISO3_CONFIG_IIR_FILTER_CONTROL:
        if ((wValue != PH_ON) && (wValue != PH_OFF))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* save previous state to temporary variable */
        memcpy(&tmp, &pDataParams->dwTxCtrl, sizeof(phhalHw_ISO3_TxCtrlRegister_t));

        /* Turn the field OFF */
        if (ISO3_REG_TX_CTRL_0.state != PHHAL_HW_ISO3_TX_STATE_POWER_DOWN)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_FieldOff(pDataParams));
        }

        /* Toggle IIR */
        ISO3_REG_IIRFILTER_CTRL.enable = (0x01u & 0);
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(pDataParams, PHHAL_HW_ISO3_REG_IIRFILTER_CTRL,
            (uint8_t*) &pDataParams->dwIIRFilterCtrl, sizeof(pDataParams->dwIIRFilterCtrl)));


        ISO3_REG_IIRFILTER_CTRL.enable = (0x01u & wValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(pDataParams, PHHAL_HW_ISO3_REG_IIRFILTER_CTRL,
            (uint8_t*) &pDataParams->dwIIRFilterCtrl, sizeof(pDataParams->dwIIRFilterCtrl)));

        /* Switch ON the field, check filter overflow and then turn OFF the field */
        if (wValue == PH_ON)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_FieldOn(pDataParams));

            /* toggle the overflow reset switch */
            dwTmpValue = 1;
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(pDataParams, PHHAL_HW_ISO3_REG_IIRFILTER_OVERFLOW_RESET, (uint8_t *)&dwTmpValue, sizeof(dwTmpValue)));
            dwTmpValue = 0;
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(pDataParams, PHHAL_HW_ISO3_REG_IIRFILTER_OVERFLOW_RESET, (uint8_t *)&dwTmpValue, sizeof(dwTmpValue)));

            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_IIR_FILTER_OVERFLOW, &wFilterOverflowFlag));
            if(wFilterOverflowFlag == PHHAL_HW_ISO3_IIR_FILTER_OVERFLOW)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_FieldOff(pDataParams));
                return PH_ADD_COMPCODE(PHHAL_HW_ISO3_ERR_IIR_FILTER_OVERFLOW, PH_COMP_HAL);
            }

            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_FieldOff(pDataParams));
        }

        /* pcheck for field off */
        if (tmp.state != PHHAL_HW_ISO3_TX_STATE_POWER_DOWN )
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_FieldOn(pDataParams));
        }

        /* restore the previous state */
        ISO3_REG_TX_CTRL_0.cardType = tmp.cardType;
        ISO3_REG_TX_CTRL_0.iso15693mode = tmp.iso15693mode;
        ISO3_REG_TX_CTRL_0.datarate = tmp.datarate;
        ISO3_REG_TX_CTRL_0.operand = tmp.operand;

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(
            pDataParams,
            PHHAL_HW_ISO3_REG_TX_CTRL_0,
            pDataParams->dwTxCtrl));

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(
            pDataParams));

        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);

    case PHHAL_HW_ISO3_CONFIG_DOUBLE_DATA_RATE:
        if ((wValue != PH_ON) && (wValue != PH_OFF))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        if (ISO3_REG_TX_CTRL_1.double_data_rate != (uint32_t)(wValue & 0x0001))
        {
            ISO3_REG_TX_CTRL_1.double_data_rate = wValue;
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_1, (uint8_t*)&pDataParams->dwTxCtrl1, sizeof(pDataParams->dwTxCtrl1)));
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
        }
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_ISO3_CONFIG_SEQ8_MODE:
        if (wValue == PH_ON)
        {
            /* Configure transmitter control register for triggering sequence 8 mode */
            ISO3_REG_TX_CTRL_0.state = PHHAL_HW_ISO3_TX_STATE_POWER_DOWN;
            ISO3_REG_TX_CTRL_0.cardType = PHHAL_HW_ISO3_CARD_TYPE_A;
            ISO3_REG_TX_CTRL_0.datarate = PHHAL_HW_RF_DATARATE_106;
            ISO3_REG_TX_CTRL_0.iso15693mode = PH_ON;
            ISO3_REG_TX_CTRL_0.operand = 0;

            /* Configure the wave shape buffer so that the whole memory can be used for the wave shape */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_SetConfig(
                pDataParams,
                PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_NUMB,
                0x01));

            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
        }
        else
        {
            pDataParams->dwTxCtrl = 0;

            /* reset transmitter and wave shape buffer to initial values */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_ApplyProtocolSettings(pDataParams, PHHAL_HW_CARDTYPE_ISO14443A));

            /* to reset registers and load Field-On and Field-Off wave shapes */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_InitFpga(pDataParams));

            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
        }

    case PHHAL_HW_ISO3_CONFIG_DISABLE_AGC:
        /* This is used to disable AGC control in several other APIs */
        if ((wValue != PH_ON) && (wValue != PH_OFF)) {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        if (wValue == PH_ON) {
            uint8_t agc_ctrl[] = {0x01, 0x00, 0x00, 0x00};

            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                pDataParams,
                PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL,
                agc_ctrl,
                sizeof(agc_ctrl)));

        } else {
            uint8_t agc_ctrl[] = {0x00, 0x00, 0x00, 0x00};

            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                pDataParams,
                PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL,
                agc_ctrl,
                sizeof(agc_ctrl)));

        }
        pDataParams->bDisableAGC = (uint8_t)wValue;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);

    case PHHAL_HW_ISO3_CONFIG_IIR_FILTER_OVERFLOW:
        /* Just indicate that the parameter is not supported */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);

    case PHHAL_HW_ISO3_CONFIG_WAVESHAPE_PRESCALE_PERCENT:
        if (wValue > 100)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        pDataParams->WaveShapePrescalePercent = wValue;
        /* This will load the field on/off waveshape according to the set waveshape percentage */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_LoadPrescaledWaveshapes(pDataParams));
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);

    case PHHAL_HW_ISO3_CONFIG_OVS_FILTER_SLOT_ENABLE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_GetConfig(
            pDataParams,
            PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_NUMB,
            &wWaveShapeSlotNumb));

        if(wValue > PHHAL_HW_ISO3_OVS_FILTER_MAX_SLOT_NUM || wValue > wWaveShapeSlotNumb)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        ISO3_OVS_FILTER_SLOT_SET_IN_USE(wValue);
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_ISO3_CONFIG_OVS_FILTER_SLOT_DISABLE:
        if(wValue > PHHAL_HW_ISO3_OVS_FILTER_MAX_SLOT_NUM)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        ISO3_OVS_FILTER_SLOT_SET_CLEAR(wValue);
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case PHHAL_HW_ISO3_CONFIG_MILLER_OFFSET:
        if ((int16_t)wValue > 64 || (int16_t)wValue < -56)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        return phhalHw_ISO3_Int_ConfigMillerTest(pDataParams, (int8_t)wValue);
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }
}

phStatus_t phhalHw_ISO3_GetConfig(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint16_t wConfig,
                            uint16_t * pValue
                            )
{
    phStatus_t statusTmp;
    uint32_t dwTmp;

    /* Check params */
    PH_ASSERT_NULL(pDataParams);

    switch (wConfig)
    {
    case PHHAL_HW_CONFIG_PARITY:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;
    case PHHAL_HW_CONFIG_TXCRC:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;
    case PHHAL_HW_CONFIG_RXCRC:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;
    case PHHAL_HW_CONFIG_TXLASTBITS:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;
    case PHHAL_HW_CONFIG_RXLASTBITS:
        *pValue = pDataParams->wAdditionalInfo;
        break;
    case PHHAL_HW_CONFIG_RXALIGN:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;
    case PHHAL_HW_CONFIG_RXDEAFBITS:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;
    case PHHAL_HW_CONFIG_TXWAIT_US:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;
    case PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;
    case PHHAL_HW_CONFIG_TXDATARATE:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;
    case PHHAL_HW_CONFIG_RXDATARATE:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;
    case PHHAL_HW_CONFIG_MODINDEX:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;
    case PHHAL_HW_CONFIG_ASK100:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_US:
        if (pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MICROSECONDS)
        {
            *pValue = pDataParams->wCfgShadow[wConfig];
        }
        else
        {
            if (pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS] > (0xFFFF / 1000))
            {
                return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
            }
            *pValue = pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS] * 1000;
        }
        break;

    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS:
        if (pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MILLISECONDS)
        {
            *pValue = pDataParams->wCfgShadow[wConfig];
        }
        else
        {
            *pValue = pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_US] / 1000;
        }
        break;

    case PHHAL_HW_CONFIG_TIMING_MODE:
        /* Return parameter */
        *pValue = pDataParams->wTimingMode;
        break;
    case PHHAL_HW_CONFIG_TIMING_US:
        /* Return 1 as this is not supported */
        *pValue = PH_ON;
        break;
    case PHHAL_HW_CONFIG_TIMING_MS:
        /* Return 1 as this is not supported */
        *pValue = PH_ON;
        break;
    case PHHAL_HW_CONFIG_FIELD_OFF_TIME:
        if (pDataParams->dwFieldOffTimeCycles > 0xFFFF * 13560.0)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        /* Return parameter */
        *pValue = (uint16_t)((pDataParams->dwFieldOffTimeCycles / 13560.0) + 0.5);
        break;
    case PHHAL_HW_ISO3_CONFIG_FIELD_OFF_TIME_US:
        if (pDataParams->dwFieldOffTimeCycles > 0xFFFF * 13.56)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        /* Return parameter */
        *pValue = (uint16_t)((pDataParams->dwFieldOffTimeCycles / 13.56) + 0.5);
        break;
    case PHHAL_HW_ISO3_CONFIG_FIELD_OFF_TIME_CYCLES:
        if (pDataParams->dwFieldOffTimeCycles > 0xFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->dwFieldOffTimeCycles;
        break;
    case PHHAL_HW_CONFIG_FIELD_RECOVERY_TIME:
        if (pDataParams->dwFieldRecoveryTimeCycles > 0xFFFF * 13560.0)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        /* Return parameter */
        *pValue = (uint16_t)((pDataParams->dwFieldRecoveryTimeCycles / 13560.0) + 0.5);
        break;
    case PHHAL_HW_ISO3_CONFIG_FIELD_RECOVERY_TIME_US:
        if (pDataParams->dwFieldRecoveryTimeCycles > 0xFFFF * 13.56)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        /* Return parameter */
        *pValue = (uint16_t)((pDataParams->dwFieldRecoveryTimeCycles / 13.56) + 0.5);
        break;
    case PHHAL_HW_ISO3_CONFIG_DEBUG_OUTPUT0:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wDebugConfig0;
        break;
    case PHHAL_HW_ISO3_CONFIG_DEBUG_OUTPUT1:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wDebugConfig1;
        break;
    case PHHAL_HW_ISO3_CONFIG_DEBUG_OUTPUT2:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wDebugConfig2;
        break;
    case PHHAL_HW_ISO3_CONFIG_DEBUG_OUTPUT3:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->wDebugConfig3;
        break;
    case PHHAL_HW_CONFIG_SYMBOL_START:
        /* Return 1 as this is not supported */
        *pValue = PH_ON;
        break;
    case PHHAL_HW_CONFIG_SYMBOL_END:
        /* Return 1 as this is not supported */
        *pValue = PH_ON;
        break;
    case PHHAL_HW_CONFIG_CARD_TYPE:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->bCardType;
        break;
    case PHHAL_HW_CONFIG_SUBCARRIER:
        /* Return parameter */
        *pValue = PHHAL_HW_SUBCARRIER_SINGLE;
        break;
    case PHHAL_HW_CONFIG_RXBUFFER_STARTPOS:
        /* Return parameter */
        *pValue = pDataParams->wRxBufStartPos;
        break;
    case PHHAL_HW_CONFIG_RXBUFFER_BUFSIZE:
        /* Return parameter */
        *pValue = pDataParams->wRxBufSize - PHHAL_HW_ISO3_CRC_BUFFER_LEN;
        break;
    case PHHAL_HW_CONFIG_TXBUFFER_BUFSIZE:
        /* Return parameter */
        *pValue = pDataParams->wTxBufSize - pDataParams->wTxBufStartPos;
        break;
    case PHHAL_HW_CONFIG_TXBUFFER_LENGTH:
        /* Return parameter */
        *pValue = pDataParams->wTxBufLen;
        break;
    case PHHAL_HW_CONFIG_TXBUFFER:
        /* Check additional info parameter */
        if ((pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo) >= pDataParams->wTxBufSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
        /* Retrieve TxBuffer byte */
        *pValue = (uint16_t)pDataParams->pTxBuffer[pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo];
        break;
    case PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1:
        /* Return 1 as crypto is not supported */
        *pValue = PH_ON;
        break;
    case PHHAL_HW_CONFIG_MAX_PRECACHED_BYTES:
        /* Return 1 as this is not supported */
        *pValue = PH_ON;
        break;
    case PHHAL_HW_CONFIG_BAL_CONNECTION:
        /* Return 1 as this is not supported */
        *pValue = PH_ON;
        break;
    case PHHAL_HW_CONFIG_SERIAL_BITRATE:
        /* Return 1 as this is not supported */
        *pValue = PH_ON;
        break;
    case PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->bRfResetAfterTo;
        break;
    case PHHAL_HW_CONFIG_SETMINFDT:
        /* Return 1 as this is not supported */
        *pValue = PH_ON;
        break;
    case PHHAL_HW_CONFIG_FRAME_SYNC:
        /* Return 1 as this is not supported */
        *pValue = PH_ON;
        break;
    case PHHAL_HW_CONFIG_TXDATANUM:
        /* Return 1 as this is not supported */
        *pValue = PH_ON;
        break;
    case PHHAL_HW_CONFIG_ADDITIONAL_INFO:
        *pValue = pDataParams->wAdditionalInfo;
        break;
    case PHHAL_HW_CONFIG_SET_READER_IC:
        /* Return 1 as this is not supported */
        *pValue = PH_ON;
        break;
    case PHHAL_HW_CONFIG_COMMUNICATION_CONFIG:
        /* Return 1 as this is not supported */
        *pValue = PH_ON;
        break;
    case PHHAL_HW_ISO3_CONFIG_FIELD_STATE:
        if (ISO3_REG_TX_CTRL_0.state == PHHAL_HW_ISO3_TX_STATE_POWER_DOWN)
        {
            *pValue = PH_OFF;
        }
        else
        {
            *pValue = PH_ON;
        }
        break;
    case PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_NUMB:
        /* Read config from shadow */
        *pValue = pDataParams->wWaveShapeBufferConfigs[PHHAL_HW_ISO3_WAVESHAPE_SLOTS_NUMB];
        break;
    case PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOTS_SIZE:
        /* Read config from shadow */
        *pValue = pDataParams->wWaveShapeBufferConfigs[PHHAL_HW_ISO3_WAVESHAPE_SLOT_SIZE];
        break;
    case PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOT_FIELD_ON:
        /* Read config from shadow */
        *pValue = pDataParams->wWaveShapeBufferConfigs[PHHAL_HW_ISO3_WAVESHAPE_SLOT_FIELD_ON];
        break;
    case PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOT_FIELD_OFF:
        /* Read config from shadow */
        *pValue = pDataParams->wWaveShapeBufferConfigs[PHHAL_HW_ISO3_WAVESHAPE_SLOT_FIELD_OFF];
        break;
    case PHHAL_HW_ISO3_CONFIG_WAVESHAPE_SLOT_EXCHANGE:
        /* Read config from shadow */
        *pValue = pDataParams->wWaveShapeBufferConfigs[PHHAL_HW_ISO3_WAVESHAPE_SLOT_EXCHANGE];
        break;
    case PHHAL_HW_ISO3_CONFIG_FPGA_VERSION:
        /* Read config from fpga register */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(
            pDataParams,
            PHHAL_HW_ISO3_REG_FPGA_VERSION,
            (uint8_t*) &dwTmp,
            sizeof(uint32_t)));
        *pValue = (uint16_t)dwTmp;

        if(dwTmp > (uint16_t) -1)
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);

        break;
    case PHHAL_HW_ISO3_CONFIG_BIT_BOUNDARY:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadowCustom[wConfig - PH_CONFIG_CUSTOM_BEGIN]; /* Use custom config shadow */
        break;
    case PHHAL_HW_ISO3_CONFIG_EGT:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadowCustom[wConfig - PH_CONFIG_CUSTOM_BEGIN]; /* Use custom config shadow */
        break;
    case PHHAL_HW_ISO3_CONFIG_DUTY_CYCLE:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadowCustom[wConfig - PH_CONFIG_CUSTOM_BEGIN]; /* Use custom config shadow */
        break;
    case PHHAL_HW_ISO3_CONFIG_RX_NOISE_POWER_TH:
        *pValue = (uint16_t)(ISO3_REG_VENUS_DGRM_SIG_DET_TH.dgrm_signal_detection_th);
        break;
    case PHHAL_HW_ISO3_CONFIG_IIR_FILTER_CONTROL:
        *pValue = (uint16_t)(ISO3_REG_IIRFILTER_CTRL.enable);
        break;
    case PHHAL_HW_ISO3_CONFIG_DOUBLE_DATA_RATE:
        *pValue = (uint16_t)(ISO3_REG_TX_CTRL_1.double_data_rate);
        break;
    case PHHAL_HW_ISO3_CONFIG_SEQ8_MODE:
        if ((ISO3_REG_TX_CTRL_0.cardType == PHHAL_HW_ISO3_CARD_TYPE_A) && (ISO3_REG_TX_CTRL_0.iso15693mode == PH_ON))
        {
            *pValue = PH_ON;
        }
        else
        {
            *pValue = PH_OFF;
        }
        break;
    case PHHAL_HW_ISO3_CONFIG_IIR_FILTER_OVERFLOW:
        /* Check here if IIR overflow bit was set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(
            pDataParams,
            PHHAL_HW_ISO3_REG_IIRFILTER_OVERFLOW,
            (uint8_t*) &dwTmp,
            sizeof(uint32_t)));
        *pValue = (uint16_t)dwTmp;
        break;
    case PHHAL_HW_ISO3_CONFIG_WAVESHAPE_PRESCALE_PERCENT:
        *pValue = pDataParams->WaveShapePrescalePercent;
        break;
    case PHHAL_HW_ISO3_CONFIG_TX_LAST_BIT:
        *pValue = pDataParams->bLastBit;
        break;
    case PHHAL_HW_ISO3_CONFIG_PEAK_SENS_SIGNAL:
        *pValue = (uint16_t)pDataParams->dwPeakSensSignal;
        break;
    case PHHAL_HW_ISO3_CONFIG_PEAK_CAL_SIGNAL:
        *pValue = (uint16_t)pDataParams->dwPeakCalSignal;
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_FieldOn(
                            phhalHw_ISO3_DataParams_t * pDataParams
                            )
{
    phStatus_t statusTmp;
    /* convert milliseconds to ETUs */
    uint16_t wValue;
    uint16_t wFilterOverflowFlag;
    uint32_t dwTmpValue = 0;

    /* Check params */
    PH_ASSERT_NULL(pDataParams);

    /* if field is already on, return success */
    if (ISO3_REG_TX_CTRL_0.state != PHHAL_HW_ISO3_TX_STATE_POWER_DOWN)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    /* set pointer to field on waveshape */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildSetWaveShapePointer(
        pDataParams,
        ISO3_WAVESHAPE_TYPE_FIELD_ON));

    if (pDataParams->bDisableAGC == PH_OFF) {
        /* deactivate gain control */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(
            pDataParams,
            PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL,
            PH_ON));
    }

    ISO3_REG_TX_CTRL_0.state = PHHAL_HW_ISO3_TX_STATE_POWER_DOWN;
    ISO3_REG_TX_CTRL_0.operand = 0x0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(
        pDataParams,
        PHHAL_HW_ISO3_REG_TX_CTRL_0,
        pDataParams->dwTxCtrl));

    /* build field on command */
    ISO3_REG_TX_CTRL_0.state = PHHAL_HW_ISO3_TX_STATE_POWER_UP;
    ISO3_REG_TX_CTRL_0.operand = pDataParams->dwFieldRecoveryTimeCycles;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(
        pDataParams,
        PHHAL_HW_ISO3_REG_TX_CTRL_0,
        pDataParams->dwTxCtrl));

    if (pDataParams->bDisableAGC == PH_OFF) {
        /* reactivate gain control */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(
            pDataParams,
            PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL,
            PH_OFF));
    }

    /* perform field on */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(
        pDataParams));

    /* Check IIR filter overflow if the filter is enabled */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_IIR_FILTER_CONTROL, &wValue));
    if (wValue == PH_ON)
    {
        /* toggle the overflow reset switch */
        dwTmpValue = 1;
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(pDataParams, PHHAL_HW_ISO3_REG_IIRFILTER_OVERFLOW_RESET, (uint8_t *)&dwTmpValue, sizeof(dwTmpValue)));
        dwTmpValue = 0;
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(pDataParams, PHHAL_HW_ISO3_REG_IIRFILTER_OVERFLOW_RESET, (uint8_t *)&dwTmpValue, sizeof(dwTmpValue)));

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_IIR_FILTER_OVERFLOW, &wFilterOverflowFlag));
        if(wFilterOverflowFlag == PHHAL_HW_ISO3_IIR_FILTER_OVERFLOW)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_FieldOff(pDataParams));
            return PH_ADD_COMPCODE(PHHAL_HW_ISO3_ERR_IIR_FILTER_OVERFLOW, PH_COMP_HAL);
        }
    }

    if (pDataParams->bDisableAGC == PH_OFF)
    {
        phTools_Sleep((uint32_t)((pDataParams->dwFieldRecoveryTimeCycles/13560) + 0.5));
        phhalHw_ISO3_Cmd_GetAgcStatistic(pDataParams, PH_ON, &dwTmpValue, &dwTmpValue);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_FieldOff(
                            phhalHw_ISO3_DataParams_t * pDataParams
                            )
{
    phStatus_t statusTmp;
    /* convert milliseconds to ETUs */
    double f = 13560.0;
    uint16_t wResetTime;
    uint16_t wNumbETUs = 0x1f;
    phhalHw_ISO3_TxCtrlRegister_t tmp;

    /* Check params */
    PH_ASSERT_NULL(pDataParams);

    /* save previous state to temporary variable */
    memcpy(&tmp, &pDataParams->dwTxCtrl, sizeof(phhalHw_ISO3_TxCtrlRegister_t));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(pDataParams, PHHAL_HW_CONFIG_FIELD_OFF_TIME, &wResetTime));
    wNumbETUs = (uint16_t)( (double)wResetTime * f / (double)(128 >> 3) ); /* 3 is ISO3_REG_TX_CTRL_0.datarate */

    /* in sequence8 mode the state field in TX_CTRL is 0, but the field is on. */
    /*phhalHw_ISO3_GetConfig(pDataParams, PHHAL_HW_ISO3_CONFIG_SEQ8_MODE, &wSeq8Mode);*/

    /* if field is already off, return success */
    if (ISO3_REG_TX_CTRL_0.state == PHHAL_HW_ISO3_TX_STATE_POWER_DOWN) /*  && wSeq8Mode == PH_OFF -> Seq8 not used */
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    if (pDataParams->bDisableAGC == PH_OFF) {
    /* deactivate gain control */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(
        pDataParams,
        PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL,
        PH_ON));
    }

    /* set pointer to field off waveshape */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildSetWaveShapePointer(
        pDataParams,
        ISO3_WAVESHAPE_TYPE_FIELD_OFF));

    /* Rf Off (The operand value does not matter) */
    ISO3_REG_TX_CTRL_0.state = PHHAL_HW_ISO3_TX_STATE_POWER_DOWN;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(
        pDataParams,
        PHHAL_HW_ISO3_REG_TX_CTRL_0,
        pDataParams->dwTxCtrl));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(
        pDataParams));

    /* need to reset tx ping-pong buffer in field-off state */
    pDataParams->dwNextTxBufferAddress = (uint32_t) PHHAL_HW_ISO3_REG_TX_DATA;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_FieldReset(
                            phhalHw_ISO3_DataParams_t * pDataParams
                            )
{
    phStatus_t statusTmp;
    uint32_t dwTmpValue;

    phhalHw_ISO3_TxCtrlRegister_t tmp;

    /* Check params */
    PH_ASSERT_NULL(pDataParams);

    /* save previous state to temporary variable */
    memcpy(&tmp, &pDataParams->dwTxCtrl, sizeof(phhalHw_ISO3_TxCtrlRegister_t));

    /* Check params */
    PH_ASSERT_NULL(pDataParams);

    if (ISO3_REG_TX_CTRL_0.state == PHHAL_HW_ISO3_TX_STATE_POWER_DOWN)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    if (pDataParams->bDisableAGC == PH_OFF) {
    /* deactivate gain control */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL, PH_ON));
    }

    /* set pointer to field Off waveshape */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildSetWaveShapePointer(
        pDataParams,
        ISO3_WAVESHAPE_TYPE_FIELD_OFF));

    /* Configure to Type A card */
    ISO3_REG_TX_CTRL_0.cardType = PHHAL_HW_ISO3_CARD_TYPE_A;
    ISO3_REG_TX_CTRL_0.iso15693mode = 0;
    ISO3_REG_TX_CTRL_0.datarate = 3;

    /* Rf Off */
    ISO3_REG_TX_CTRL_0.state = PHHAL_HW_ISO3_TX_STATE_POWER_DOWN;
    ISO3_REG_TX_CTRL_0.operand = pDataParams->dwFieldOffTimeCycles;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(
        pDataParams,
        PHHAL_HW_ISO3_REG_TX_CTRL_0,
        pDataParams->dwTxCtrl));

    /* set pointer to field On waveshape */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildSetWaveShapePointer(
        pDataParams, ISO3_WAVESHAPE_TYPE_FIELD_ON));

    /* turn field on again */
    ISO3_REG_TX_CTRL_0.state = PHHAL_HW_ISO3_TX_STATE_POWER_UP;
    ISO3_REG_TX_CTRL_0.operand = pDataParams->dwFieldRecoveryTimeCycles;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(
        pDataParams,PHHAL_HW_ISO3_REG_TX_CTRL_0,
        pDataParams->dwTxCtrl));

    /* restore the previous state */
    ISO3_REG_TX_CTRL_0.cardType = tmp.cardType;
    ISO3_REG_TX_CTRL_0.iso15693mode = tmp.iso15693mode;
    ISO3_REG_TX_CTRL_0.datarate = tmp.datarate;
    ISO3_REG_TX_CTRL_0.operand = tmp.operand;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(
        pDataParams,
        PHHAL_HW_ISO3_REG_TX_CTRL_0,
        pDataParams->dwTxCtrl));

    /* send data to fpga */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(pDataParams));

    if (pDataParams->bDisableAGC == PH_OFF) {
        /* re-activate gain control */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterWrite(
            pDataParams,
            PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL,
            PH_OFF));
        /* send data to fpga */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(pDataParams));
    }

    /* need to be reset because of FIELD_OFF */
    pDataParams->dwNextTxBufferAddress = PHHAL_HW_ISO3_REG_TX_DATA;

    if (pDataParams->bDisableAGC == PH_OFF)
    {
        phTools_Sleep((uint32_t)((pDataParams->dwFieldRecoveryTimeCycles/13560) + 0.5));
        phhalHw_ISO3_Cmd_GetAgcStatistic(pDataParams, PH_ON, &dwTmpValue, &dwTmpValue);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_Wait(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint8_t bUnit,
                            uint16_t wTimeout
                            )
{
    PH_ASSERT_NULL(pDataParams);

    /* Suppress unused warning for release build */
    PH_UNUSED_VARIABLE(pDataParams);

    /* Code below ask FPGA to wait for given time, so that next read/write command to FPGA will be executed after given time. However it doesn't work!*/
    /*
    uint16_t numEtus;
    double f = 0.0;
    if (bUnit == PHHAL_HW_TIME_MILLISECONDS)
    {
        f = 13560.0;
    }
    if (bUnit == PHHAL_HW_TIME_MICROSECONDS)
    {
        f = 13.56;
    }

    numEtus = (uint16_t)((double)wTimeout * f / (double)(128 >> ISO3_REG_TX_CTRL_0.datarate));
    ISO3_REG_TX_CTRL_0.operand = numEtus;

    PH_ASSERT_NULL(pDataParams);

    phhalHw_ISO3_Int_BuildRegisterWrite(pDataParams, PHHAL_HW_ISO3_REG_TX_CTRL_0, pDataParams->dwTxCtrl);

    phhalHw_ISO3_Int_DirectExchange(pDataParams, false);
    */

    if (bUnit == PHHAL_HW_TIME_MILLISECONDS)
    {
        phTools_Sleep(wTimeout);
    }
   else if (bUnit == PHHAL_HW_TIME_MICROSECONDS)
    {
        if (wTimeout <= 1000)
        {
            /* Can't wait shorter then 1 ms */
            phTools_Sleep(1);
        }
        else
        {
            phTools_Sleep(wTimeout/1000);
        }
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_MfcAuthenticateKeyNo(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint8_t bBlockNo,
                            uint8_t bKeyType,
                            uint16_t wKeyNo,
                            uint16_t wKeyVersion,
                            uint8_t * pUid
                            )
{
    /* TODO Implement */
    if(pDataParams || bBlockNo || bKeyType || wKeyNo || wKeyVersion || pUid);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
}

phStatus_t phhalHw_ISO3_MfcAuthenticate(
                            phhalHw_ISO3_DataParams_t * pDataParams,
                            uint8_t bBlockNo,
                            uint8_t bKeyType,
                            uint8_t * pKey,
                            uint8_t * pUid
                            )
{
    /* TODO Implement */
    if(pDataParams || bBlockNo || bKeyType || pKey || pUid);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_ISO3 */
