/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * DUT Hal specific definition for KeyStore commands.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7463 $
 * $Date: 2025-08-29 15:56:52 +0530 (Fri, 29 Aug 2025) $
 *
 * History:
 *  Generated 12 October, 2022
 *
 */


#include <ph_Status.h>

#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_KEYSTORE
#include <ph_RefDefs.h>
#include <phhalHw_DUT_Cmd.h>

#include "../phhalHw_DUT.h"

phStatus_t phhalHw_DUT_Cmd_KeyStore_Init(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wNoOfKeyEntries)
{
    uint8_t PH_MEMLOC_REM aCmd[4];
    uint8_t PH_MEMLOC_REM bCmdLen = 0;

    /* Validate Parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wNoOfKeyEntries, 2);
    bCmdLen = 2;

    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_INIT,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        bCmdLen,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_KeyStore_DeInit(phhalHw_DUT_DataParams_t * pDataParams)
{
    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_DEINIT,
        PH_EXCHANGE_DEFAULT,
        NULL,
        0,
        NULL,
        NULL);
}





/* Common Interfaces ------------------------------------------------------------------------------------------------------------------- */
phStatus_t phhalHw_DUT_Cmd_KeyStore_FormatKeyEntry(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t wNewKeyType)
{
    uint8_t PH_MEMLOC_REM aCmd[4];
    uint8_t PH_MEMLOC_REM bCmdLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyNo, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wNewKeyType, 2);
    bCmdLen += 2;

    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_FORMAT_KEY_ENTRY,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        bCmdLen,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_KeyStore_SetKUC(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t wRefNoKUC)
{
    uint8_t PH_MEMLOC_REM aCmd[4];
    uint8_t PH_MEMLOC_REM bCmdLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyNo, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wRefNoKUC, 2);
    bCmdLen += 2;

    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_SET_KUC,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        bCmdLen,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_KeyStore_GetKUC(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wRefNoKUC, uint32_t * pLimit,
    uint32_t * pCurVal)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM aCmd[2];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    uint8_t     PH_MEMLOC_REM * pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wRefNoKUC, 2);
    bCmdLen = 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_GET_KUC,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        bCmdLen,
        &pResponse,
        &wRspLen));

    /* Validate Status. */
    if(wRspLen < 8U)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);

    /* Copy the status to response buffer. */
    (void) memcpy((uint8_t *) pLimit, &pResponse[0], 4);
    (void) memcpy((uint8_t *) pCurVal, &pResponse[4], 4);

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_KeyStore_ChangeKUC(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wRefNoKUC, uint32_t dwLimit)
{
    uint8_t PH_MEMLOC_REM aCmd[6];
    uint8_t PH_MEMLOC_REM bCmdLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wRefNoKUC, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &dwLimit, 4);
    bCmdLen += 4;

    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_CHANGE_KUC,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        bCmdLen,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_KeyStore_SetConfig(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    uint8_t PH_MEMLOC_REM aCmd[4];
    uint8_t PH_MEMLOC_REM bCmdLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wConfig, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wValue, 2);
    bCmdLen += 2;

    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_SET_CONFIG,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        bCmdLen,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_KeyStore_SetConfigStr(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wConfig, uint8_t *pBuffer,
    uint16_t wBuffLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM aCmd[2];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wConfig, 2);
    bCmdLen = 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_SET_CONFIG_BUFFER,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmd,
        bCmdLen,
        NULL,
        NULL));

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_SET_CONFIG_BUFFER,
        PH_EXCHANGE_BUFFER_LAST,
        pBuffer,
        wBuffLen,
        NULL,
        NULL));

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_KeyStore_GetConfig(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM aCmd[2];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    uint8_t     PH_MEMLOC_REM * pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wConfig, 2);
    bCmdLen = 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_GET_CONFIG,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        bCmdLen,
        &pResponse,
        &wRspLen));

    /* Validate Status. */
    if(wRspLen < 2U)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);

    /* Copy the status to response buffer. */
    (void) memcpy(pValue, &pResponse[0], 2);

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_KeyStore_GetConfigStr(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wConfig, uint8_t ** ppBuffer,
    uint16_t * pBuffLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM aCmd[2];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    uint8_t     PH_MEMLOC_REM * pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wConfig, 2);
    bCmdLen = 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_GET_CONFIG_BUFFER,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        bCmdLen,
        &pResponse,
        &wRspLen));

    /* Copy the status to response buffer. */
    ppBuffer = &pResponse;
    *pBuffLen = wRspLen;

    return wStatus;
}





/* Interfaces for Symmetric Keys ------------------------------------------------------------------------------------------------------- */
phStatus_t phhalHw_DUT_Cmd_KeyStore_SetKey(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t wKeyVer, uint16_t wKeyType,
    uint8_t * pNewKey, uint16_t wNewKeyVer)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmd[8];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint8_t     PH_MEMLOC_REM bKeyLen = 0;

    /* Compute Key Size */
    switch(wKeyType)
    {
        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_DES:
            bKeyLen = 8;
            break;

        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_2K3DES:
        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_AES128:
            bKeyLen = 16;
            break;

        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_3K3DES:
        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_AES192:
            bKeyLen = 24;
            break;

        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_AES256:
            bKeyLen = 32;
            break;

        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_MIFARE:
            bKeyLen = 12;
            break;

        default:
            bKeyLen = 0;
            break;
    }

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyNo, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyVer, 2);
    bCmdLen += 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyType, 2);
    bCmdLen += 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wNewKeyVer, 2);
    bCmdLen += 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_SET_KEY,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmd,
        bCmdLen,
        NULL,
        NULL));

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_SET_KEY,
        PH_EXCHANGE_BUFFER_LAST,
        pNewKey,
        bKeyLen,
        NULL,
        NULL));

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_KeyStore_SetKeyAtPos(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t wPos, uint16_t wKeyType,
    uint8_t * pNewKey, uint16_t wNewKeyVer)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmd[8];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint8_t     PH_MEMLOC_REM bKeyLen = 0;

    /* Compute Key Size */
    switch(wKeyType)
    {
        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_DES:
            bKeyLen = 8;
            break;

        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_2K3DES:
        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_AES128:
            bKeyLen = 16;
            break;

        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_3K3DES:
        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_AES192:
            bKeyLen = 24;
            break;

        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_AES256:
            bKeyLen = 32;
            break;

        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_MIFARE:
            bKeyLen = 12;
            break;

        default:
            bKeyLen = 0;
            break;
    }

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyNo, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wPos, 2);
    bCmdLen += 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyType, 2);
    bCmdLen += 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wNewKeyVer, 2);
    bCmdLen += 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_SET_KEY_AT_POS,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmd,
        bCmdLen,
        NULL,
        NULL));

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_SET_KEY_AT_POS,
        PH_EXCHANGE_BUFFER_LAST,
        pNewKey,
        bKeyLen,
        NULL,
        NULL));

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_KeyStore_SetFullKeyEntry(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wNoOfKeys, uint16_t wKeyNo,
    uint16_t wNewRefNoKUC, uint16_t wNewKeyType, uint8_t * pNewKeys, uint16_t * pNewKeyVerList)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmd[8];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint8_t     PH_MEMLOC_REM bKeyLen = 0;

    /* Compute Key Size */
    switch(wNewKeyType)
    {
        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_DES:
            bKeyLen = 8;
            break;

        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_2K3DES:
        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_AES128:
            bKeyLen = 16;
            break;

        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_3K3DES:
        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_AES192:
            bKeyLen = 24;
            break;

        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_AES256:
            bKeyLen = 32;
            break;

        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_MIFARE:
            bKeyLen = 12;
            break;

        default:
            bKeyLen = 0;
            break;
    }

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wNoOfKeys, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyNo, 2);
    bCmdLen += 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wNewRefNoKUC, 2);
    bCmdLen += 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wNewKeyType, 2);
    bCmdLen += 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_SET_FULL_KEY_ENTRY,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmd,
        bCmdLen,
        NULL,
        NULL));

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_SET_FULL_KEY_ENTRY,
        PH_EXCHANGE_BUFFER_CONT,
        pNewKeys,
        (wNoOfKeys * bKeyLen),
        NULL,
        NULL));

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_SET_FULL_KEY_ENTRY,
        PH_EXCHANGE_BUFFER_LAST,
        (uint8_t *) pNewKeyVerList,
        (wNoOfKeys * 2),
        NULL,
        NULL));

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_KeyStore_GetKeyEntry(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t wKeyVerBufSize,
    uint16_t * pKeyVer, uint16_t * pKeyVerLen, uint16_t * pKeyType)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM aCmd[4];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    uint8_t     PH_MEMLOC_REM * pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyNo, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyVerBufSize, 2);
    bCmdLen += 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_GET_KEY_ENTRY,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        bCmdLen,
        &pResponse,
        &wRspLen));

    /* Validate Status. */
    if(wRspLen < 6U)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);

    /* Copy the status to response buffer. */
    (void) memcpy((uint8_t *)pKeyVer, &pResponse[0], 2);
    (void) memcpy((uint8_t *) pKeyVerLen, &pResponse[2], 2);
    (void) memcpy((uint8_t *) pKeyType, &pResponse[4], 2);

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_KeyStore_GetKey(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bKeyBufSize,
    uint8_t * pKey, uint16_t * pKeyType)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bKeyLen = 0;

    uint8_t     PH_MEMLOC_REM aCmd[5];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    uint8_t     PH_MEMLOC_REM * pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyNo, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyVer, 2);
    bCmdLen += 2;

    aCmd[bCmdLen] = bKeyBufSize;
    bCmdLen += 1;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_GET_KEY,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        bCmdLen,
        &pResponse,
        &wRspLen));

    /* Validate Status. */
    if(wRspLen < 6U)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);

    /* Copy the status to response buffer. */
    (void) memcpy((uint8_t *) pKeyType, &pResponse[0], 2);

    /* Compute Key Size */
    switch(*pKeyType)
    {
        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_DES:
            bKeyLen = 8;
            break;

        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_2K3DES:
        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_AES128:
            bKeyLen = 16;
            break;

        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_3K3DES:
        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_AES192:
            bKeyLen = 24;
            break;

        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_AES256:
            bKeyLen = 32;
            break;

        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_MIFARE:
            bKeyLen = 12;
            break;

        default:
            bKeyLen = 0;
            break;
    }
    (void) memcpy(pKey, &pResponse[2], bKeyLen);

    return wStatus;
}





/* Interfaces for ASymmetric Keys ------------------------------------------------------------------------------------------------------ */
phStatus_t phhalHw_DUT_Cmd_KeyStore_SetKeyASym(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t wPos, uint16_t wKeyType,
    uint16_t wKeyInfo, uint8_t * pKey, uint16_t wKeyLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmd[10];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyNo, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wPos, 2);
    bCmdLen += 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyType, 2);
    bCmdLen += 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyInfo, 2);
    bCmdLen += 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyLen, 2);
    bCmdLen += 2;

     PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_SET_KEY_ASYM,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmd,
        bCmdLen,
        NULL,
        NULL));

     PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
         pDataParams,
         PHHAL_HW_DUT_CMD_KEYSTORE_SET_KEY_ASYM,
         PH_EXCHANGE_BUFFER_LAST,
         pKey,
         wKeyLen,
         NULL,
         NULL));

     return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_DUT_Cmd_KeyStore_GetKeyASym(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t wPos, uint16_t wKeyPairType,
    uint16_t * pKeyType, uint8_t * pCurveID, uint8_t * pKey, uint16_t * pKeyLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM aCmd[6];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    uint8_t     PH_MEMLOC_REM * pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyNo, 2);
    bCmdLen = 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wPos, 2);
    bCmdLen += 2;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyPairType, 2);
    bCmdLen += 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_GET_KEY_ASYM,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        bCmdLen,
        &pResponse,
        &wRspLen));

    /* Validate Status. */
    if(wRspLen < 3U)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);

    /* Copy the status to response buffer. */
    (void) memcpy((uint8_t *) pKeyType, &pResponse[0], 2);
    *pCurveID = pResponse[2];

    (void) memcpy(pKey, &pResponse[3], wRspLen - 3);
    *pKeyLen = (uint16_t) (wRspLen - 3);

    return wStatus;
}





#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_KEYSTORE_NDA
phStatus_t phhalHw_DUT_Cmd_KeyStore_Provision_Init(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wOption, uint8_t *pTransportKey_AES128,
    uint8_t *pTransportKey_AES256, uint8_t *pAppRootKey_AES128, uint8_t *pAppRootKey_AES256, uint8_t *pExpDecData, uint8_t * pDervMsg,
    uint8_t * pWrapIV)
{
    uint8_t     PH_MEMLOC_REM aCmd[155];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wOption, 2);
    bCmdLen = 2;

    if(pTransportKey_AES128 != NULL)
    {
        (void) memcpy(&aCmd[bCmdLen], pTransportKey_AES128, PHHAL_HW_DUT_KEYSTORE_AES128_KEY_SIZE);
        bCmdLen += PHHAL_HW_DUT_KEYSTORE_AES128_KEY_SIZE;
    }

    if(pTransportKey_AES256 != NULL)
    {
        (void) memcpy(&aCmd[bCmdLen], pTransportKey_AES256, PHHAL_HW_DUT_KEYSTORE_AES256_KEY_SIZE);
        bCmdLen += PHHAL_HW_DUT_KEYSTORE_AES256_KEY_SIZE;
    }

    if(pAppRootKey_AES128 != NULL)
    {
        (void) memcpy(&aCmd[bCmdLen], pAppRootKey_AES128, PHHAL_HW_DUT_KEYSTORE_AES128_KEY_SIZE);
        bCmdLen += PHHAL_HW_DUT_KEYSTORE_AES128_KEY_SIZE;
    }

    if(pAppRootKey_AES256 != NULL)
    {
        (void) memcpy(&aCmd[bCmdLen], pAppRootKey_AES256, PHHAL_HW_DUT_KEYSTORE_AES256_KEY_SIZE);
        bCmdLen += PHHAL_HW_DUT_KEYSTORE_AES256_KEY_SIZE;
    }

    if(pExpDecData != NULL)
    {
        (void) memcpy(&aCmd[bCmdLen], pExpDecData, PHHAL_HW_DUT_KEYSTORE_AES128_KEY_SIZE);
        bCmdLen += PHHAL_HW_DUT_KEYSTORE_AES128_KEY_SIZE;
    }

    if(pDervMsg != NULL)
    {
        (void) memcpy(&aCmd[bCmdLen], pDervMsg, 24U);
        bCmdLen += 24U;
    }

    if(pWrapIV != NULL)
    {
        (void) memcpy(&aCmd[bCmdLen], pWrapIV, PHHAL_HW_DUT_KEYSTORE_AES128_KEY_SIZE);
        bCmdLen += PHHAL_HW_DUT_KEYSTORE_AES128_KEY_SIZE;
    }

    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_INIT,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        bCmdLen,
        NULL,
        NULL);
}

void phhalHw_DUT_Cmd_KeyStore_Provision_DeInit(phhalHw_DUT_DataParams_t * pDataParams)
{
    PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_DEINIT,
        PH_EXCHANGE_DEFAULT,
        NULL,
        0,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_KeyStore_SetConfig_Ext(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wConfig, uint8_t * pValue, uint16_t wValueLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM aCmd[2];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wConfig, 2);
    bCmdLen = 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_SET_CONFIG,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmd,
        2,
        NULL,
        NULL));

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_SET_CONFIG,
        PH_EXCHANGE_BUFFER_LAST,
        pValue,
        wValueLen,
        NULL,
        NULL));

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_KeyStore_GetConfig_Ext(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wConfig, uint8_t * pValue, uint16_t * pValueLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM aCmd[2];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    uint8_t     PH_MEMLOC_REM * pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Prepare the command. */
    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wConfig, 2);
    bCmdLen = 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_GET_CONFIG,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        bCmdLen,
        &pResponse,
        &wRspLen));

    /* Copy the status to response buffer. */
    (void) memcpy(pValue, pResponse, wRspLen);
    *pValueLen = wRspLen;

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_KeyStore_Provision_AppFixedKeys(phhalHw_DUT_DataParams_t * pDataParams, uint8_t bKeyIndex, uint16_t wKeyType,
    uint8_t * pNewKey)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmd[3];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint8_t     PH_MEMLOC_REM bKeyLen = 0;

    /* Compute Key Size */
    switch(wKeyType)
    {
        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_AES128:
            bKeyLen = 16;
            break;

        case PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_AES256:
            bKeyLen = 32;
            break;

        default:
            bKeyLen = 0;
            break;
    }

    /* Prepare the command. */
    aCmd[bCmdLen++] = bKeyIndex;

    (void) memcpy(&aCmd[bCmdLen], (uint8_t *) &wKeyType, 2);
    bCmdLen += 2;

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_APP_FIXED_KEYS,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmd,
        bCmdLen,
        NULL,
        NULL));

    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_APP_FIXED_KEYS,
        PH_EXCHANGE_BUFFER_LAST,
        pNewKey,
        bKeyLen,
        NULL,
        NULL));

    return wStatus;
}
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_KEYSTORE_NDA */

#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_KEYSTORE */
