/*
 * Copyright 2013, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Generic Thermostream Device Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phdlThstrm.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_THSTRM

#include "phdlThstrm_Int.h"

phStatus_t phdlThstrm_Int_Send(
                               void * pBalRegDataParams,
                               uint8_t * pTxBuffer,
                               uint16_t wTxLength
                               )
{
    phStatus_t statusTmp;
    phStatus_t statusTmp2;
    uint16_t wValue = 0;
    uint8_t bRx;
    uint16_t wRxLen;

    /* read current IO timeout */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wValue));
    /* set IO timeout to 1ms  to speed up timeout error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 500));

    /*send data*/
    statusTmp = phbalReg_Exchange(
        pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        pTxBuffer,
        wTxLength,
        1,
        &bRx,
        &wRxLen);

    /* restore IO timeout use statusTmp2 to not overwrite response from send */
    PH_CHECK_SUCCESS_FCT(statusTmp2, phbalReg_SetConfig(pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wValue));

    /* handle expected IO-timeouterror occured and ignore error as no response is excpected */
    if ( (statusTmp & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        statusTmp = PH_ERR_SUCCESS;
    }

    return PH_ADD_COMPCODE(statusTmp, PH_COMP_DL_AMP);
}

phStatus_t phdlThstrm_Int_SendWait(
                                   void * pBalRegDataParams,
                                   uint8_t * pTxBuffer,
                                   uint16_t wTxLength,
                                   uint16_t wWaittime
                                   )
{
    phStatus_t statusTmp;
    phStatus_t statusTmp2;
    uint16_t wValue=0;
    uint8_t bRx;
    uint16_t wRxLen;

    /* read current IO timeout */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wValue));
    /* set IO timeout to zero to speed up timeout error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wWaittime));

    /* send data*/
    statusTmp = phbalReg_Exchange(
        pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        pTxBuffer,
        wTxLength,
        1,
        &bRx,
        &wRxLen);

    /* restore IO timeout use statusTmp2 to not overwrite response from send */
    PH_CHECK_SUCCESS_FCT(statusTmp2, phbalReg_SetConfig(pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wValue));

    /* handle expected IO-timeouterror occured and ignore error as no response is excpected */
    if ( (statusTmp & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        statusTmp = PH_ERR_SUCCESS;
    }

    return PH_ADD_COMPCODE(statusTmp, PH_COMP_DL_THSTRM);
}

/* helperfunction asci to integer */
phStatus_t phdlThstrm_Int_Atoui16(
                                  int8_t * pAsci,
                                  uint16_t * wURetVal
                                  )
{
    uint16_t wTemp = 0;

    while (*pAsci >= '0' && *pAsci <= '9')
    {
        if(wTemp > 6553)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_THSTRM);
        }
        wTemp *= 10;
        wTemp += (*pAsci-'0');
        pAsci++;
    }

    *wURetVal = wTemp;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_THSTRM);
}

/* helperfunction asci to integer */
phStatus_t phdlThstrm_Int_Atoi32(
                               int8_t * pAsci,
                               int32_t * dwIRetVal
                               )
{
    int64_t qwTemp=0;

    if ((*pAsci >= '0' && *pAsci <= '9') || *pAsci == '-' || *pAsci == '+')
    {
        if (*pAsci == '-')
        {
            /* increse index, so that minus sign is ignored */
            pAsci++;
            do
            {
                if (*pAsci != '.')
                {
                    qwTemp *= 10;
                    qwTemp -= (*pAsci-'0');
                }
                pAsci++;
                if (qwTemp < -2147483648LL)
                {
                    return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_OSCI);
                }
            } while ((*pAsci >= '0' && *pAsci <= '9') || *pAsci == '.');
        }
        else
        {
            /* increse index, so that plus sign is ignored */
            if (*pAsci == '+')
                pAsci++;

            do
            {
                if (*pAsci != '.')
                {
                    qwTemp *= 10;
                    qwTemp += (*pAsci-'0');
                }
                pAsci++;
                if (qwTemp > 2147483647LL)
                {
                    return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_OSCI);
                }
            }while ((*pAsci >= '0' && *pAsci <= '9') || *pAsci == '.');
        }
    }
    *dwIRetVal = (int32_t)qwTemp;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

/* Helperfunction to multiply the value by ten to remove the dot */
phStatus_t phdlThstrm_Int_MultTeni16(
                                 int8_t * pAsci,
                                 int16_t * wIRetVal
                                 )
{
    int16_t wTemp=0;

    if ( (*pAsci >= '0' && *pAsci <= '9') || *pAsci == '-')
    {
        if ( *pAsci == '-')
        {
            /* increse index, so that minus sign is ignored */
            pAsci++;
            do
            {
                if (*pAsci != '.')
                {
                    if(wTemp > 3276 || wTemp < -3276)
                    {
                        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_THSTRM);
                    }
                    wTemp *= 10;
                    wTemp -= (*pAsci-'0');
                }
                else
                {
                    pAsci++;
                    if(*pAsci < '0' || *pAsci > '9')
                    {
                        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_THSTRM);
                    }
                    wTemp *= 10;
                    wTemp -= (*pAsci-'0');
                    pAsci++;
                    if(*pAsci >= '5' && *pAsci <= '9')
                    {
                        wTemp -= 1;
                    }
                    break;
                }
                pAsci++;
            } while( (*pAsci >= '0' && *pAsci <= '9') || *pAsci == '.' );
        }
        else
        {
            do
            {
                if (*pAsci != '.')
                {
                    if(wTemp > 3276 || wTemp < -3276)
                    {
                        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_THSTRM);
                    }
                    wTemp *= 10;
                    wTemp += (*pAsci-'0');
                }
                else
                {
                    pAsci++;
                    if(*pAsci < '0' || *pAsci > '9')
                    {
                        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_THSTRM);
                    }
                    wTemp *= 10;
                    wTemp += (*pAsci-'0');
                    pAsci++;
                    if(*pAsci >= '5' && *pAsci <= '9')
                    {
                        wTemp += 1;
                    }
                    break;
                }
                pAsci++;
            } while( (*pAsci >= '0' && *pAsci <= '9') || *pAsci == '.');
        }
    }
    *wIRetVal = wTemp;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_THSTRM);
}

#endif /* NXPBUILD__PHDL_THSTRM */
