/*
 * Copyright 2019 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Stepper Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <phdlStepper.h>
#include <ph_RefDefs.h>
#include <phTools.h>
#include <ph_Status.h>

#ifdef NXPBUILD__PHDL_STEPPER_WACHLER

#include "phdlStepper_Wachler.h"
#include "phdlStepper_Wachler_Int.h"
#include "../phdlStepper_Int.h"

#ifndef _WIN32
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>              /* PRQA S 5124 */
#else
#pragma warning(push)           /* PRQA S 3116 */
#pragma warning(disable:4001)   /* PRQA S 3116 */
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>              /* PRQA S 5124 */
#pragma warning(pop)            /* PRQA S 3116 */
#endif

phStatus_t phdlStepper_Wachler_Int_Reset(
                                        phdlStepper_Wachler_DataParams_t * pDataParams
                                        )
{
    phStatus_t statusTmp;
    uint8_t bMode = 0;

    /* Wait for 3 sec to get wachler startup after reset */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Int_Wait(3000));

    /* Get current mode */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_GetMode(pDataParams, &bMode));

    /* If not in calibration mode start mode */
    if (bMode != PHDL_STEPPER_WACHLER_MODE_CAL && bMode != PHDL_STEPPER_WACHLER_MODE_PWMCAL)
    {
        /* Send Calibrate command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_SetMode(pDataParams, PHDL_STEPPER_WACHLER_MODE_CAL));
    }
    /* Wait until pwm calibrate is done */
    do
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_GetMode(pDataParams, &bMode));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Int_Wait(10));
    }
    while (bMode == PHDL_STEPPER_WACHLER_MODE_PWMCAL);
    /* Wait until calibrate is done */
    do
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_GetMode(pDataParams, &bMode));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Int_Wait(10));
    }
    while (bMode == PHDL_STEPPER_WACHLER_MODE_CAL);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_SetMode(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        uint8_t bMode
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    uint8_t *pMode;

    switch(bMode)
    {
    case PHDL_STEPPER_WACHLER_MODE_RUN:
        {
            pMode = PHDL_STEPPER_WACHLER_INT_MODE_RUN;
            break;
        }
    case PHDL_STEPPER_WACHLER_MODE_STOP:
        {
            pMode = PHDL_STEPPER_WACHLER_INT_MODE_STOP;
            break;
        }
    case PHDL_STEPPER_WACHLER_MODE_CAL:
        {
            pMode = PHDL_STEPPER_WACHLER_INT_MODE_CAL;
            break;
        }
    case PHDL_STEPPER_WACHLER_MODE_PWMCAL:
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_STEPPER);
        }
    case PHDL_STEPPER_WACHLER_MODE_AUTO:
        {
            pMode = PHDL_STEPPER_WACHLER_INT_MODE_AUTO;
            break;
        }
    case PHDL_STEPPER_WACHLER_MODE_BSTOP:
        {
            pMode = PHDL_STEPPER_WACHLER_INT_MODE_BSTOP;
            break;
        }
    default:
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
    }

    sprintf((char *)buf, (char *)(PHDL_STEPPER_WACHLER_INT_CMD_MODE), pMode);

    /* Send Mode command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        buf,
        (uint16_t)strlen((char *)buf),
        sizeof(buf),
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseStatus(buf));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_GetMode(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        uint8_t * pbMode
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint16_t wRxLen = 0;

    /* Check if already calibrating */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_WACHLER_INT_CMD_GET_MODE,
        (uint16_t)strlen((char *)PHDL_STEPPER_WACHLER_INT_CMD_GET_MODE),
        sizeof(buf),
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseString(buf));

    /* If not in calibration mode start mode */
    if (!_strcmpi((char *)buf, (char *)PHDL_STEPPER_WACHLER_INT_MODE_RUN))
    {
        *pbMode = PHDL_STEPPER_WACHLER_MODE_RUN;
    }
    else if (!_strcmpi((char *)buf, (char *)PHDL_STEPPER_WACHLER_INT_MODE_STOP))
    {
        *pbMode = PHDL_STEPPER_WACHLER_MODE_STOP;
    }
    else if (!_strcmpi((char *)buf, (char *)PHDL_STEPPER_WACHLER_INT_MODE_CAL))
    {
        *pbMode = PHDL_STEPPER_WACHLER_MODE_CAL;
    }
    else if (!_strcmpi((char *)buf, (char *)PHDL_STEPPER_WACHLER_INT_MODE_PWMCAL))
    {
        *pbMode = PHDL_STEPPER_WACHLER_MODE_PWMCAL;
    }
    else if (!_strcmpi((char *)buf, (char *)PHDL_STEPPER_WACHLER_INT_MODE_AUTO))
    {
        *pbMode = PHDL_STEPPER_WACHLER_MODE_AUTO;
    }
    else if (!_strcmpi((char *)buf, (char *)PHDL_STEPPER_WACHLER_INT_MODE_BSTOP))
    {
        *pbMode = PHDL_STEPPER_WACHLER_MODE_BSTOP;
    }
    else if (!_strcmpi((char *)buf, (char *)PHDL_STEPPER_WACHLER_INT_MODE_POS))
    {
        *pbMode = PHDL_STEPPER_WACHLER_MODE_POS;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_MoveUp(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        uint8_t bBlocking
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    int32_t dwIsUp = 0;

    /* Send Calibrate command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_WACHLER_INT_CMD_POS_UP,
        (uint16_t)strlen((char *)PHDL_STEPPER_WACHLER_INT_CMD_POS_UP),
        sizeof(buf),
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseStatus(buf));

    if (bBlocking == PH_ON)
    {
        /* Wait until position is reached */
        do
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
                PH_EXCHANGE_DEFAULT,
                PHDL_STEPPER_WACHLER_INT_CMD_GET_POS_UP,
                (uint16_t)strlen((char *)PHDL_STEPPER_WACHLER_INT_CMD_GET_POS_UP),
                sizeof(buf),
                buf,
                &wRxLen));
            buf[wRxLen] = 0;

            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseInt(buf, &dwIsUp));

            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Int_Wait(10));
        }
        while (dwIsUp != 1);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_MoveDown(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        uint8_t bBlocking
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    int32_t dwIsDown = 0;

    /* Send Calibrate command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_WACHLER_INT_CMD_POS_DOWN,
        (uint16_t)strlen((char *)PHDL_STEPPER_WACHLER_INT_CMD_POS_DOWN),
        sizeof(buf),
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseStatus(buf));

    if (bBlocking == PH_ON)
    {
        /* Wait until position is reached */
        do
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
                PH_EXCHANGE_DEFAULT,
                PHDL_STEPPER_WACHLER_INT_CMD_GET_POS_DOWN,
                (uint16_t)strlen((char *)PHDL_STEPPER_WACHLER_INT_CMD_GET_POS_DOWN),
                sizeof(buf),
                buf,
                &wRxLen));
            buf[wRxLen] = 0;

            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseInt(buf, &dwIsDown));

            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Int_Wait(10));
        }
        while (dwIsDown != 1);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_SetPos(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        float fPosition,
                                        uint8_t bBlocking
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint8_t expected[50];
    uint16_t wRxLen = 0;
    float fPositionSet;

    sprintf((char *)buf, (char *)(PHDL_STEPPER_WACHLER_INT_CMD_POS), fPosition);
    sprintf((char *)expected, "%.2f\r\n", fPosition);

    /* Send Calibrate command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        buf,
        (uint16_t)strlen((char *)buf),
        (uint16_t)strlen((char *)expected), /* Expect length of expected data */
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseFloat(buf, &fPositionSet));

    /* Check if resonse match with set value */
    if (strcmp((char *)expected, (char *)buf))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }

    if (bBlocking == PH_ON)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_WaitMoveFinished(pDataParams));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_GetPos(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        float *pFloatPosition
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint16_t wRxLen = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_WACHLER_INT_CMD_GET_POS,
        (uint16_t)strlen((char *)PHDL_STEPPER_WACHLER_INT_CMD_GET_POS),
        sizeof(buf),
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseFloat(buf, pFloatPosition));
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_GetPosMin(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        float *pFloatPosition
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint16_t wRxLen = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_WACHLER_INT_CMD_GET_POS_MIN,
        (uint16_t)strlen((char *)PHDL_STEPPER_WACHLER_INT_CMD_GET_POS_MIN),
        sizeof(buf),
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseFloat(buf, pFloatPosition));
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_GetPosMax(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        float *pFloatPosition
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint16_t wRxLen = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_WACHLER_INT_CMD_GET_POS_MAX,
        (uint16_t)strlen((char *)PHDL_STEPPER_WACHLER_INT_CMD_GET_POS_MAX),
        sizeof(buf),
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseFloat(buf, pFloatPosition));
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_GetState(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        int32_t *pIntValue
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[4];
    uint16_t wRxLen = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_WACHLER_INT_CMD_GET_STATE,
        (uint16_t)strlen((char *)PHDL_STEPPER_WACHLER_INT_CMD_GET_STATE),
        3, /* Only 3 bytes expected */
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseInt(buf, pIntValue));
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_SetDelay(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        uint16_t wDelay
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    int32_t dwDelaySet = 0;

    sprintf((char *)buf, (char *)(PHDL_STEPPER_WACHLER_INT_CMD_DELAY), wDelay);

    /* Send Calibrate command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        buf,
        (uint16_t)strlen((char *)buf),
        sizeof(buf),
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseInt(buf, &dwDelaySet));
    if (dwDelaySet != wDelay)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_GetDelay(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        uint16_t * pwDelay
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    int32_t dwDelay = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_WACHLER_INT_CMD_GET_DELAY,
        (uint16_t)strlen((char *)PHDL_STEPPER_WACHLER_INT_CMD_GET_DELAY),
        sizeof(buf),
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseInt(buf, &dwDelay));
    if (dwDelay > 0xFFFF || dwDelay < 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_STEPPER);
    }
    *pwDelay = (uint16_t)(dwDelay);
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_SetPwm(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        uint8_t bPwmEnabled
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    int32_t dwPwmEnabledSet = 0;

    sprintf((char *)buf, (char *)(PHDL_STEPPER_WACHLER_INT_CMD_PWM), bPwmEnabled);

    /* Send Calibrate command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        buf,
        (uint16_t)strlen((char *)buf),
        sizeof(buf),
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseInt(buf, &dwPwmEnabledSet));
    if (dwPwmEnabledSet != bPwmEnabled)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_SetPwmSpeed(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        uint8_t bPwmSpeed
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    int32_t dwPwmSpeedSet = 0;

    sprintf((char *)buf, (char *)(PHDL_STEPPER_WACHLER_INT_CMD_PWM_SPEED), bPwmSpeed);

    /* Send Calibrate command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        buf,
        (uint16_t)strlen((char *)buf),
        sizeof(buf),
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseInt(buf, &dwPwmSpeedSet));
    if (dwPwmSpeedSet != bPwmSpeed)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_SetPwmCalibration(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        uint8_t bPwmCalibration
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    int32_t dwPwmCalibrationSet = 0;

    sprintf((char *)buf, (char *)(PHDL_STEPPER_WACHLER_INT_CMD_PWM_CALIBRATION), bPwmCalibration);

    /* Send Calibrate command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        buf,
        (uint16_t)strlen((char *)buf),
        sizeof(buf),
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseInt(buf, &dwPwmCalibrationSet));
    if (dwPwmCalibrationSet != bPwmCalibration)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_SetLevel(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        float fLevel
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint8_t expected[50];
    uint16_t wRxLen = 0;
    float fLevelSet = 0.0;

    sprintf((char *)buf, (char *)(PHDL_STEPPER_WACHLER_INT_CMD_LEVEL), fLevel);
    sprintf((char *)expected, "%.2f\r\n", fLevel);

    /* Send Calibrate command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        buf,
        (uint16_t)strlen((char *)buf),
        (uint16_t)strlen((char *)expected), /* Expect length of expected data */
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseFloat(buf, &fLevelSet));
    /* Check if resonse match with set value */
    if (strcmp((char *)expected, (char *)buf))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_GetLevel(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        float * pfLevel
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint16_t wRxLen = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_WACHLER_INT_CMD_GET_LEVEL,
        (uint16_t)strlen((char *)PHDL_STEPPER_WACHLER_INT_CMD_GET_LEVEL),
        sizeof(buf),
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseFloat(buf, pfLevel));
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_GetCounter(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        uint32_t * pdwCounter
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    int32_t dwCounter = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_WACHLER_INT_CMD_GET_COUNTER,
        (uint16_t)strlen((char *)PHDL_STEPPER_WACHLER_INT_CMD_GET_COUNTER),
        sizeof(buf),
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseInt(buf, &dwCounter));
    if (dwCounter < 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_STEPPER);
    }
    *pdwCounter = (uint16_t)(dwCounter);
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_ResetCounter(
                                        phdlStepper_Wachler_DataParams_t * pDataParams
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint16_t wRxLen = 0;

    /* Send Reset Counter command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_WACHLER_INT_CMD_COUNTER_RESET,
        (uint16_t)strlen((char *)PHDL_STEPPER_WACHLER_INT_CMD_COUNTER_RESET),
        sizeof(buf),
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseStatus(buf));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_SetRemote(
                                        phdlStepper_Wachler_DataParams_t * pDataParams,
                                        uint8_t bRemote
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    int32_t dwRemoteSet = 0;

    sprintf((char *)buf, (char *)(PHDL_STEPPER_WACHLER_INT_CMD_REMOTE), bRemote);

    /* Send Calibrate command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        buf,
        (uint16_t)strlen((char *)buf),
        sizeof(buf),
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseInt(buf, &dwRemoteSet));
    if (dwRemoteSet != bRemote)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_GetIdn(
                                        phdlStepper_Wachler_DataParams_t * pDataParams
                                        )
{
    phStatus_t statusTmp;
    uint8_t buf[128];
    uint16_t wRxLen = 0;

    /* Check if already calibrating */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_WACHLER_INT_CMD_GET_IDN,
        (uint16_t)strlen((char *)PHDL_STEPPER_WACHLER_INT_CMD_GET_IDN),
        sizeof(buf),
        buf,
        &wRxLen));
    buf[wRxLen] = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_CheckResponseString(buf));

    strncpy((char *)pDataParams->bIdn, (char *)buf, PH_COMP_DL_STEPPER_WACHLER_IDN_SIZE);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_WaitMoveFinished(
                                        phdlStepper_Wachler_DataParams_t * pDataParams
                                        )
{
    phStatus_t statusTmp;
    int32_t dwState = 0;

    /* Wait until position is reached (motor stop) */
    do
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_GetState(pDataParams, &dwState));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Int_Wait(10));
    }
    while (dwState == 1);
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_CheckResponseStatus(
                                       uint8_t * pResponse
                                       )
{
    if (!_strcmpi((char *)pResponse, PHDL_STEPPER_WACHLER_INT_RESP_OK))
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
    }
    if (!_strcmpi((char *)pResponse, PHDL_STEPPER_WACHLER_INT_RESP_INV))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }
    if (!_strcmpi((char *)pResponse, PHDL_STEPPER_WACHLER_INT_RESP_ERR))
    {
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_CheckResponseString(
                                       uint8_t * pResponse
                                       )
{
    if (!_strcmpi((char *)pResponse, PHDL_STEPPER_WACHLER_INT_RESP_OK))
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
    }
    if (!_strcmpi((char *)pResponse, PHDL_STEPPER_WACHLER_INT_RESP_INV))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }
    if (!_strcmpi((char *)pResponse, PHDL_STEPPER_WACHLER_INT_RESP_ERR))
    {
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }
    /* Any string is success */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_CheckResponseInt(
                                       uint8_t * pResponse,
                                       int32_t *pInt32Value
                                       )
{
    if (!_strcmpi((char *)pResponse, PHDL_STEPPER_WACHLER_INT_RESP_INV))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }
    if (!_strcmpi((char *)pResponse, PHDL_STEPPER_WACHLER_INT_RESP_ERR))
    {
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }

    if (sscanf((char *)pResponse, "%d", pInt32Value) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_Int_CheckResponseFloat(
                                       uint8_t * pResponse,
                                       float *pfValue
                                       )
{
    if (!_strcmpi((char *)pResponse, PHDL_STEPPER_WACHLER_INT_RESP_INV))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }
    if (!_strcmpi((char *)pResponse, PHDL_STEPPER_WACHLER_INT_RESP_ERR))
    {
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }

    if (sscanf((char *)pResponse, "%f", pfValue) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

#endif /* NXPBUILD__PHDL_STEPPER_WACHLER */
