/*
 * Copyright 2019 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hardware Steppper Wachler Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <math.h>

#ifdef NXPBUILD__PHDL_STEPPER_WACHLER

#include <phdlStepper.h>
#include "phdlStepper_Wachler.h"
#include "phdlStepper_Wachler_Int.h"
#include "../phdlStepper_Int.h"

phStatus_t phdlStepper_Wachler_Init(
                               phdlStepper_Wachler_DataParams_t * pDataParams,
                               uint16_t wSizeOfDataParams,
                               void * pBalRegDataParams
                               )
{
    if (sizeof(phdlStepper_Wachler_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalRegDataParams);

    /* init private data */
    pDataParams->wId                    = PH_COMP_DL_STEPPER | PH_COMP_DL_STEPPER_WACHLER_ID;
    pDataParams->pBalRegDataParams      = pBalRegDataParams;
    pDataParams->bInit                  = PH_OFF;
    pDataParams->fMaxPositionDegree     = 0.0f;
    pDataParams->fMinPositionDegree     = 0.0f;
    pDataParams->actPosition            = 0;
    pDataParams->bNumSteps              = PH_COMP_DL_STEPPER_WACHLER_DEFAULT_NUM_STEPS;
    pDataParams->fModeUpDownLevel       = PH_COMP_DL_STEPPER_WACHLER_DEFAULT_UP_DOWN_LEVEL;
    pDataParams->fModePosLevel          = PH_COMP_DL_STEPPER_WACHLER_DEFAULT_POS_LEVEL;
    pDataParams->bPwmSpeed              = 100;
    memset(pDataParams->bIdn, 0, PH_COMP_DL_STEPPER_WACHLER_IDN_SIZE);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t  phdlStepper_Wachler_Initialize(
                                      phdlStepper_Wachler_DataParams_t * pDataParams
                                      )
{
    phStatus_t PH_MEMLOC_REM status;
    float PH_MEMLOC_REM fTemp;

    PH_ASSERT_NULL (pDataParams);

    /* Reset Wachler and wait until top position is reached */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Wachler_Int_Reset(pDataParams));

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Wachler_Int_SetRemote(pDataParams, PH_ON));

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Wachler_Int_SetMode(pDataParams, PHDL_STEPPER_WACHLER_MODE_STOP));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Wachler_Int_SetLevel(pDataParams, pDataParams->fModeUpDownLevel));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Wachler_Int_MoveDown(pDataParams, PH_ON));
    pDataParams->actPosition = 0;

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Wachler_Int_GetPosMin(pDataParams, &fTemp));
    pDataParams->fMinPositionDegree = fTemp;
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Wachler_Int_GetPosMax(pDataParams, &fTemp));
    pDataParams->fMaxPositionDegree = fTemp;

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Wachler_Int_ResetCounter(pDataParams));

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Wachler_Int_GetIdn(pDataParams));

    pDataParams->bInit = PH_ON;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_MoveDistance(
                                       phdlStepper_Wachler_DataParams_t * pDataParams,
                                       uint16_t wSpeed,
                                       uint8_t bDirection,
                                       uint32_t dwDistance,
                                       uint8_t bBlocking
                                       )
{
    PH_UNUSED_VARIABLE (wSpeed);
    PH_UNUSED_VARIABLE (bDirection);
    PH_UNUSED_VARIABLE (dwDistance);
    PH_UNUSED_VARIABLE (bBlocking);

    PH_ASSERT_NULL (pDataParams);

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_MoveSteps(
                                    phdlStepper_Wachler_DataParams_t * pDataParams,
                                    uint16_t wSpeed,
                                    uint8_t bDirection,
                                    uint32_t dwSteps,
                                    uint8_t bBlocking
                                    )
{
    PH_UNUSED_VARIABLE (wSpeed);
    PH_UNUSED_VARIABLE (bDirection);
    PH_UNUSED_VARIABLE (dwSteps);
    PH_UNUSED_VARIABLE (bBlocking);

    PH_ASSERT_NULL (pDataParams);
    if( bDirection != PHDL_STEPPER_DIR_DOWN && bDirection != PHDL_STEPPER_DIR_UP )
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Wachler_GoToPosition(
                                       phdlStepper_Wachler_DataParams_t * pDataParams,
                                       uint16_t wSpeed,
                                       uint32_t dwPosition,
                                       uint8_t bBlocking
                                       )
{
    phStatus_t  PH_MEMLOC_REM status;
    int32_t     PH_MEMLOC_REM dwSignedPosition;
    float fDegree = 0.0f;
    float fStepDegreeHalf = 0.0f;
    uint32_t oldSpeed = 0;

    PH_ASSERT_NULL (pDataParams);

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    /* Cast the value of the position */
    dwSignedPosition = (int32_t)dwPosition;

    if (dwSignedPosition < 0 || dwSignedPosition > pDataParams->bNumSteps)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }

    if (bBlocking != PH_ON && bBlocking != PH_OFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }

    fStepDegreeHalf = (pDataParams->fMaxPositionDegree - pDataParams->fMinPositionDegree)/(float)(2.0f * pDataParams->bNumSteps);
    if (fStepDegreeHalf < 1.0f)
    {
        fStepDegreeHalf = 1.0f;
    }

    /* Calc Degree */
    fDegree = pDataParams->fMinPositionDegree + ((pDataParams->fMaxPositionDegree - pDataParams->fMinPositionDegree)/(float)(pDataParams->bNumSteps)*(float)dwSignedPosition);

    if (pDataParams->fModePosLevel < fStepDegreeHalf)
    {
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_Wachler_Int_SetLevel(pDataParams, pDataParams->fModePosLevel));
    }
    else
    {
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_Wachler_Int_SetLevel(pDataParams, fStepDegreeHalf));
    }

    /* backup old speed and set speed */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Wachler_GetConfig32(pDataParams, PHDL_STEPPER_WACHLER_CONFIG_PWM_SPEED, &oldSpeed));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Wachler_SetConfig32(pDataParams, PHDL_STEPPER_WACHLER_CONFIG_PWM_SPEED, wSpeed));

    status = phdlStepper_Wachler_Int_SetPos(pDataParams, fDegree, bBlocking);

    /* restore speed */
    phdlStepper_Wachler_SetConfig32(pDataParams, PHDL_STEPPER_WACHLER_CONFIG_PWM_SPEED, oldSpeed);

    return status;
}

phStatus_t phdlStepper_Wachler_SetConfig(
                                    phdlStepper_Wachler_DataParams_t * pDataParams,
                                    uint16_t wConfig,
                                    uint16_t wValue
                                    )
{
    PH_ASSERT_NULL (pDataParams);

    return phdlStepper_Wachler_SetConfig32(pDataParams, wConfig, wValue);
}

phStatus_t phdlStepper_Wachler_SetConfig32(
                                      phdlStepper_Wachler_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint32_t dwValue
                                      )
{
    phStatus_t  PH_MEMLOC_REM status;

    PH_ASSERT_NULL (pDataParams);

    switch (wConfig)
    {
    case PHDL_STEPPER_WACHLER_CONFIG_MODE:
        if (dwValue > PHDL_STEPPER_WACHLER_MODE_BSTOP)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        return phdlStepper_Wachler_Int_SetMode(pDataParams, (uint8_t)dwValue);

    case PHDL_STEPPER_WACHLER_CONFIG_REMOTE:
        if (dwValue != PH_ON && dwValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        return phdlStepper_Wachler_Int_SetRemote(pDataParams, (uint8_t)dwValue);

    case PHDL_STEPPER_WACHLER_CONFIG_NUMBER_STEPS:
        if (dwValue > PH_COMP_DL_STEPPER_WACHLER_MAX_NUM_STEPS)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->bNumSteps = (uint8_t)dwValue;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);

    case PHDL_STEPPER_WACHLER_CONFIG_UP_DOWN_LEVEL:
        if (dwValue < 50 || (pDataParams->bInit == PH_ON && (dwValue > ((pDataParams->fMaxPositionDegree - pDataParams->fMinPositionDegree)/2 -3)*100)))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->fModeUpDownLevel = (float)(dwValue/100.0);
        return phdlStepper_Wachler_Int_SetLevel(pDataParams, pDataParams->fModeUpDownLevel);

    case PHDL_STEPPER_WACHLER_CONFIG_POS_LEVEL:
        if (dwValue < 50 || dwValue > 400)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->fModePosLevel = (float)(dwValue/100.0);
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);

    case PHDL_STEPPER_WACHLER_CONFIG_DELAY:
        if (dwValue > 0xFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        return phdlStepper_Wachler_Int_SetDelay(pDataParams, (uint16_t)dwValue);

    case PHDL_STEPPER_WACHLER_CONFIG_PWM:
        if (dwValue != PH_OFF && dwValue != PH_ON)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        return phdlStepper_Wachler_Int_SetPwm(pDataParams, (uint8_t)dwValue);

    case PHDL_STEPPER_WACHLER_CONFIG_PWM_SPEED:
        if (dwValue > 100)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        status = phdlStepper_Wachler_Int_SetPwmSpeed(pDataParams, (uint8_t)dwValue);
        if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            pDataParams->bPwmSpeed = (uint8_t)dwValue;
        }
        return status;

    case PHDL_STEPPER_WACHLER_CONFIG_PWM_CALIBRATION:
        if (dwValue > 100)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        status = phdlStepper_Wachler_Int_SetPwmCalibration(pDataParams, (uint8_t)dwValue);
        if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            pDataParams->bPwmSpeed = (uint8_t)dwValue;
        }
        return status;

    case PHDL_STEPPER_WACHLER_CONFIG_COUNTER:
        /* Only set zero is possible for counter */
        if (dwValue != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        return phdlStepper_Wachler_Int_ResetCounter(pDataParams);

    case PHDL_STEPPER_WACHLER_CONFIG_POSITION_Z:
        return phdlStepper_Wachler_GoToPosition(pDataParams, 0, dwValue, PH_ON);

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_STEPPER);
    }
}

phStatus_t phdlStepper_Wachler_GetConfig(
                                    phdlStepper_Wachler_DataParams_t * pDataParams,
                                    uint16_t wConfig,
                                    uint16_t * pValue
                                    )
{
    uint32_t dwValue;
    phStatus_t statusTmp;

    PH_ASSERT_NULL (pDataParams);

    switch(wConfig)
    {
    case PHDL_STEPPER_WACHLER_CONFIG_POS_MIN:
        *pValue = (uint16_t)(pDataParams->fMinPositionDegree*100);
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);

    case PHDL_STEPPER_WACHLER_CONFIG_POS_MAX:
        *pValue = (uint16_t)(pDataParams->fMaxPositionDegree*100);
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);

    default:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_GetConfig32(pDataParams, wConfig, &dwValue));
        if (dwValue > 0xFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_STEPPER);
        }
        *pValue = (uint16_t)dwValue;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
    }
}

phStatus_t phdlStepper_Wachler_GetConfig32(
                                      phdlStepper_Wachler_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint32_t * pValue
                                      )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint32_t    PH_MEMLOC_REM dwCounter = 0;
    uint16_t    PH_MEMLOC_REM wDelay = 0;
    uint8_t     PH_MEMLOC_REM bMode = 0;

    PH_ASSERT_NULL (pDataParams);

    switch(wConfig)
    {
    case PHDL_STEPPER_WACHLER_CONFIG_MODE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_GetMode(pDataParams, &bMode));
        *pValue = (uint32_t)bMode;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);

    case PHDL_STEPPER_WACHLER_CONFIG_NUMBER_STEPS:
        *pValue = (uint32_t)pDataParams->bNumSteps;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);

    case PHDL_STEPPER_WACHLER_CONFIG_UP_DOWN_LEVEL:
        *pValue = (uint32_t)(pDataParams->fModeUpDownLevel*100);
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);

    case PHDL_STEPPER_WACHLER_CONFIG_POS_LEVEL:
        *pValue = (uint32_t)(pDataParams->fModePosLevel*100);
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);

    case PHDL_STEPPER_WACHLER_CONFIG_DELAY:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_GetDelay(pDataParams, &wDelay));
        *pValue = (uint32_t)wDelay;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);

    case PHDL_STEPPER_WACHLER_CONFIG_POS_MIN:
        *pValue = (uint32_t)(pDataParams->fMinPositionDegree*100);
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);

    case PHDL_STEPPER_WACHLER_CONFIG_POS_MAX:
        *pValue = (uint32_t)(pDataParams->fMaxPositionDegree*100);
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);

    case PHDL_STEPPER_WACHLER_CONFIG_COUNTER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Wachler_Int_GetCounter(pDataParams, &dwCounter));
        *pValue = (uint32_t)dwCounter;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);

    case PHDL_STEPPER_WACHLER_CONFIG_PWM_SPEED:
        *pValue = (uint16_t)pDataParams->bPwmSpeed;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_STEPPER);
    }
}

#endif /* NXPBUILD__PHDL_STEPPER_WACHLER */
