/*
 * Copyright 2017, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Stepper Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHDLSTEPPER_DENSOVS60_EXTERNAL_H
#define PHDLSTEPPER_DENSOVS60_EXTERNAL_H


#ifdef __cplusplus
#define EXTERNC extern "C"
#else
#define EXTERNC
#endif

#include <ph_Status.h>
#include <phdlStepper.h>
#include <WTypes.h>
#include "external\CAO.h"

/** \cond INCLUDE_INTERNALS \defgroup phdlStepper_V1 Component : V1
* @{
*/

/* CONSTANT DEFINES */
#define X_INDEX                           0x0000U /**< X coordinate index */
#define Y_INDEX                           0x0001U /**< Y coordinate index */
#define Z_INDEX                           0x0002U /**< Z coordinate index */
#define BASE_RADIUS                       0x008CU /**< Radius value: 140mm */
#define J1__ANGLE_LIMIT                   0x00A9U /**< Maximum allowed angle to which the base can rotate: 169 */

/* FUNCTION DECLARATION */

/*Create workspace and start/stop functions */

/**
* \brief Primary function for initialization of the robots engine, workspace and controller
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_CompleteInternalInitialization(
                                                            phdlStepper_DensoVS60_DataParams_t * pDataParams                /**< [In] Pointer to this layer's parameter structure. */
                                                                                );
/**
* \brief Create the robot controller
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_CreateController(
                                                            phdlStepper_DensoVS60_DataParams_t * pDataParams                /**< [In] Pointer to this layer's parameter structure. */
                                                                   );
/**
* \brief Start the robot motor
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_StartMotor(
                                                             phdlStepper_DensoVS60_DataParams_t * pDataParams               /**< [In] Pointer to this layer's parameter structure. */
                                                            );

/**
* \brief Stop the robot motor
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_StopMotor(
                                                             phdlStepper_DensoVS60_DataParams_t * pDataParams               /**< [In] Pointer to this layer's parameter structure. */
                                                           );

/**
* \brief Create the robot arm object
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/

EXTERNC phStatus_t phdlStepper_DensoVS60_External_CreateRobotArm(
                                                              phdlStepper_DensoVS60_DataParams_t * pDataParams              /**< [In] Pointer to this layer's parameter structure. */
                                                                );




/**
* \brief Take control over the arm by the user
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_TakeArmControl(
                                                               phdlStepper_DensoVS60_DataParams_t * pDataParams             /**< [In] Pointer to this layer's parameter structure. */
                                                               );

/**
* \brief Release control over the arm by the user
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_GiveArmContol(
                                                               phdlStepper_DensoVS60_DataParams_t * pDataParams             /**< [In] Pointer to this layer's parameter structure. */
                                                               );

/**
* \brief Attach default I/O components of the robot
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_AttachDefaultIOVariables(
                                                                  phdlStepper_DensoVS60_DataParams_t * pDataParams          /**< [In] Pointer to this layer's parameter structure. */
                                                                  );

/**
* \brief Attach additional I/O components of the robot (laser,pump, headsensor, etc.)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlStepper_DensoVS60_External_AttachOptionalIOVariables(
                                                                   phdlStepper_DensoVS60_DataParams_t * pDataParams         /**< [In] Pointer to this layer's parameter structure. */
                                                                   );



/**
* \brief Turn on/off specific component of the robot
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_SetIODeviceState(
                                                             phdlStepper_DensoVS60_DataParams_t * pDataParams,              /**< [In] Pointer to this layer's parameter structure. */
                                                             int devIndex,                                                  /**< [In] Pointer to the I/O device of the robot */
                                                             int valueOnOff                                                 /**< [In] Turn on - 1 ; Turn-off - 0 */
                                                             );

/**
* \brief Get the current value of the device (if it on or off)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_GetIODeviceState(
                                                              phdlStepper_DensoVS60_DataParams_t * pDataParams,             /**< [In] Pointer to this layer's parameter structure. */
                                                              int devIndex                                                  /**< [In] Pointer to the I/O device of the robot */
                                                          );

/**
* \brief Set the (de)acceleration value of the robot movement speed
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_SetRobotAcceleration  (
                                                               phdlStepper_DensoVS60_DataParams_t * pDataParams,            /**< [In] Pointer to this layer's parameter structure. */
                                                               float wAcceleration                                          /**< [In] Acceleration values in percentages (if negative than deacceleration) */
                                                                        );


/**
* \brief Set the internal speed of the robot
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_SetRobotSpeedInternal(
                                                                    phdlStepper_DensoVS60_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                                                                    uint16_t wRobotSpeed                                       /**< [In] Internal speed value of the robot */
                                                                    );

/**
* \brief Set the external speed of the robot
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_SetRobotSpeedExternal
                                                               (
                                                                     phdlStepper_DensoVS60_DataParams_t * pDataParams,      /**< [In] Pointer to this layer's parameter structure. */
                                                                     float wRobotSpeed                                      /**< [In] Internal speed value of the robot */
                                                               );

/*------------------------------------------------------------------------*/
/*------------------ RELEASE MEMORY FUNCTIONS DECLARATION ----------------*/
/*------------------------------------------------------------------------*/

/**
* \brief  Deallocate all memory occupied by the program is there is a problem
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_EndProc(
                                                            phdlStepper_DensoVS60_DataParams_t * pDataParams,               /**< [In] Pointer to this layer's parameter structure. */
                                                            HRESULT argHr                                                   /**< [In] Status of the executed CAO function */
                                                         );

/**
* \brief Disconnect arm from the controller and release all used memory, stop program
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_DisconnectController(
                                                               phdlStepper_DensoVS60_DataParams_t * pDataParams              /**< [In] Pointer to this layer's parameter structure. */
                                                                      );

/*------------------------------------------------------------------------*/
/*----------------------- MOVE FUNCTIONS DECLARATION ---------------------*/
/*------------------------------------------------------------------------*/

/**
* \brief Updates target coordinates of the robot and checks if the move is possible
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_MoveToTargetCoordinates(
                                                                          phdlStepper_DensoVS60_DataParams_t * pDataParams,                     /**< [In] Pointer to this layer's parameter structure.         */
                                                                          double* arrAbosolutePosition,                                         /**< [In] Distance that should be traveled                     */
                                                                          int wAxisType,                                                        /**< [In] Type of axis for movement 0 - X/Y ; 1 - Joint        */
                                                                          int bBlocking                                                         /**< [In] Flag that sets the type of blocking for move command */
                                                                          );

/**
* \brief  Executes the move command with the provided parameteres
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_ExecuteMove(
                                                            phdlStepper_DensoVS60_DataParams_t * pDataParams,                                   /**< [In] Pointer to this layer's parameter structure.  */
                                                            wchar_t* wStructCommand,                                                            /**< [In] Array with values of the target position cooridantes. */
                                                            int dBlocking                                                                       /**< [In] Blocking flag which shows if the move command can be
                                                                                                                                                          stoped during execution*/
                                                             );

/**
* \brief  Executes the rotation command with the provided parameteres
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_ExecuteRotation(
                                                          phdlStepper_DensoVS60_DataParams_t * pDataParams,                                     /**< [In] Pointer to this layer's parameter structure.  */
                                                          double dwAngle                                                                        /**< [In] Angle of the target position                  */
                                                         );

/**
* \brief  Move down the Z-axis until the obstacle is reached(or the robot limit)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_GoDownUntilContact(
                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams,                                      /**< [In] Pointer to this layer's parameter structure. */
                                                         long dwMinZPoint,                                                                      /**< [In] End point to which the Z axis should go down. */
                                                         uint16_t dwRobotSpeed                                                                      /**< [In] Sets the speed of the robot */
                                                        );

/*------------------------------------------------------------------------*/
/*----------------- SPLINE MOVE COMMAND DECLARATION  ---------------------*/
/*------------------------------------------------------------------------*/

/**
* \brief Execute the spline movement after the addition of the intermediate points
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
* \retval Other Invalid parameters or conditions - the error status is returned by the underlayed functions. If the parameters are correct the function
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_SplineExecuteMove(
                                                    phdlStepper_DensoVS60_DataParams_t * pDataParams,                                           /**< [In] Pointer to this layer's parameter structure. */
                                                    uint8_t bPathIndex,                                                                         /**< [In] Index of the choosen path (range 1 to 20) */
                                                    uint8_t bBlocking                                                                           /**< [In] Flag that sets the movement on blocking(1) or non-blocking(0) */
                                                    );

/**
* \brief Add path point to the internal memory of the choosen path
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
* \retval Other Invalid parameters or conditions - the error status is returned by the underlayed functions. If the parameters are correct the function
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_SplineAddPathPoint(
                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams,                                      /**< [In] Pointer to this layer's parameter structure. */
                                                         uint8_t  bPathIndex,                                                                   /**< [In] Index of the choosen path (range 1 to 20) */
                                                         int32_t dwPositionX,                                                                    /**< [In] Value of X coodinate in micrometers */
                                                         int32_t dwPositionY,                                                                    /**< [In] Value of Y coodinate in micrometers */
                                                         int32_t dwPositionZ                                                                     /**< [In] Value of Z coodinate in micrometers */
                                                        );

/**
* \brief Clear all points from the internal memory of the choosen path
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
* \retval Other Invalid parameters or conditions - the error status is returned by the underlayed functions. If the parameters are correct the function
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_SplineClearPath(
                                                    phdlStepper_DensoVS60_DataParams_t * pDataParams,                                           /**< [In] Pointer to this layer's parameter structure. */
                                                    uint8_t bPathIndex                                                                          /**< [In] Index of the choosen path (range 1 to 20) */
                                                        );

/**
* \brief Load the inverse path of the selected path in the last internal slot(20)
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
* \retval Other Invalid parameters or conditions - the error status is returned by the underlayed functions. If the parameters are correct the function
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_SplineLoadInversePath(
                                                                        phdlStepper_DensoVS60_DataParams_t * pDataParams,                       /**< [In] Pointer to this layer's parameter structure. */
                                                                        uint8_t bPathIndex                                                      /**< [In] Index of the choosen path (range 1 to 20) */
                                                                       );


/**
* \brief Get the coorindates of the choosen point of the path
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
* \retval Other Invalid parameters or conditions - the error status is returned by the underlayed functions. If the parameters are correct the function
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_SplineGetPathPoint(
                                                                     phdlStepper_DensoVS60_DataParams_t * pDataParams,                          /**< [In] Pointer to this layer's parameter structure. */
                                                                     phdlStepper_DensoVS60_DataParams_XY_coord_t * sLocalStructTargetXY,        /**< [In] Pointer to the XY stucture used for local saving */
                                                                     uint8_t bPathIndex,                                                        /**< [In] Index of the selected path (range 1 to 20) */
                                                                     uint16_t wPointIndex                                                       /**< [In] Index of selected point (range 1 to 5000) */
                                                                    );

/**
* \brief Get the number of points from the selected path
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
* \retval Other Invalid parameters or conditions - the error status is returned by the underlayed functions. If the parameters are correct the function
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_SplineGetPathPointCount(
                                                                   phdlStepper_DensoVS60_DataParams_t * pDataParams,                            /**< [In] Pointer to this layer's parameter structure. */
                                                                   uint8_t bPathIndex,                                                          /**< [In] Index of the selected path (range 1 to 20) */
                                                                   uint16_t *pwNumberOfPoints                                                   /**< [In] Reference to the number of points for the selected path */
                                                                  );

/**
* \brief  Load all points prior to the last one
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_SplineLoadOnlyReachablePoints(
                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams,                                      /**< [In] Pointer to this layer's parameter structure. */
                                                         uint8_t bPathIndex                                                                      /**< [In] Index of the selected path (range 1 to 20) */
                                                                 );

/*------------------------------------------------------------------------*/
/*--------------------- GETTER FUNCTIONS DECLARATION ---------------------*/
/*------------------------------------------------------------------------*/

/**
* \brief  Get the cooridnates of the current positions for the Jointaxis
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_GetCurrentJointPositionWithoutOffset(
                                                              phdlStepper_DensoVS60_DataParams_t * pDataParams,                                 /**< [In] Pointer to this layer's parameter structure. */
                                                              phdlStepper_DensoVS60_DataParams_Joint_coord_t* structCurrentPositionJointCoords  /**< [In] Pointer the struct of the Joint current coordinates structure  */
                                                              );

/**
* \brief  Get the cooridnates of the current positions for the X-Y axis
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_GetCurrentXYPositionWithoutOffset(
                                                              phdlStepper_DensoVS60_DataParams_t * pDataParams,                                 /**< [In] Pointer to this layer's parameter structure.   */
                                                              phdlStepper_DensoVS60_DataParams_XY_coord_t* structCurrentPositionXYCoords        /**< [In] Pointer the struct of the Joint current coordinates structure  */
                                                              );

/**
* \brief  Get the current values of the selected cooridnate from both modes
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition(
                                                              phdlStepper_DensoVS60_DataParams_t * pDataParams,                                 /**< [In] Pointer to this layer's parameter structure.   */
                                                              int bAxisType,                                                                    /**< [In] Type of axis for movement 0 - X/Y ; 1 - Joint  */
                                                              int bAxisNumber,                                                                  /**< [In] Number of axis which value is needed  */
                                                              double* dwAbsoluteCurrentCoodinate                                                /**< [In] Pointer to double value in which the value is saved  */
                                                              );

/*------------------------------------------------------------------------*/
/*--------------------- SETTER FUNCTIONS DECLARATION ---------------------*/
/*------------------------------------------------------------------------*/

/**
* \brief  Set current coordinates as offset, when that position is set as zero point
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_SetCurrentCoordsAsOffset(
                                                                            phdlStepper_DensoVS60_DataParams_t * pDataParams,                   /**< [In] Pointer to this layer's parameter structure.   */
                                                                            int bAxisType                                                       /**< [In] Type of axis for movement 0 - X/Y ; 1 - Joint */
                                                                          );
/**
* \brief  Reset offset values to zero
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_ResetOffset(
                                                                            phdlStepper_DensoVS60_DataParams_t * pDataParams,                   /**< [In] Pointer to this layer's parameter structure. */
                                                                            int bAxisType                                                       /**< [In] Type of axis for movement 0 - X/Y ; 1 - Joint */
                                                             );

/**
* \brief  Reset current coordinates to zero if the target point is not reachable
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_ResetCurrentCooridnates(
                                                      phdlStepper_DensoVS60_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
                                                      int bAxisType                                                                             /**< [In] Type of axis for movement 0 - X/Y ; 1 - Joint */
                                                      );

/**
* \brief  Sets the given tool number
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_SetTool(
                                                          phdlStepper_DensoVS60_DataParams_t * pDataParams,                                     /**< [In] Pointer to this layer's parameter structure. */
                                                          uint16_t wtoolNumber                                                                  /**< [In] Tool number which is set. */
                                                          );

EXTERNC phStatus_t phdlStepper_DensoVS60_External_SetToolDef(
                                                             phdlStepper_DensoVS60_DataParams_t * pDataParams,                                  /**< [In] Pointer to this layer's parameter structure. */
                                                             float64_t x,                                                                       /**< [In] Coordinate to set the new ToolDef. */
                                                             float64_t y,                                                                       /**< [In] Coordinate to set the new ToolDef. */
                                                             float64_t z,                                                                       /**< [In] Coordinate to set the new ToolDef. */
                                                             float64_t rx,                                                                      /**< [In] Coordinate to set the new ToolDef. */
                                                             float64_t ry,                                                                      /**< [In] Coordinate to set the new ToolDef. */
                                                             float64_t rz                                                                       /**< [In] Coordinate to set the new ToolDef. */
                                                             );

/**
* \brief  Set the absolute coordinates of the current position to the used XY coords structure
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_SetCurrentXYPositionWithOffset(
                                                              phdlStepper_DensoVS60_DataParams_t * pDataParams,                                 /**< [In] Pointer to this layer's parameter structure. */
                                                              phdlStepper_DensoVS60_DataParams_XY_coord_t* structCurrentPositionXYCoords        /**< [In] Pointer to used Joint paramter structure */
                                                              );

/**
* \brief  Set the absolute coordinates of the current position to the used Joint coords structure
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_SetCurrentJointPositionWithOffset(
                                                              phdlStepper_DensoVS60_DataParams_t * pDataParams,                                 /**< [In] Pointer to this layer's parameter structure. */
                                                              phdlStepper_DensoVS60_DataParams_Joint_coord_t* structCurrentPositionJointCoords  /**< [In] Pointer to used Joint paramter structure */
                                                              );

/*------------------------------------------------------------------------*/
/*-------------------- CHECK FUNCTIONS DECLARATION -----------------------*/
/*------------------------------------------------------------------------*/

/**
* \brief Alows the rotation the robot Rz(J6)axis in the range between -180 to 180 degree
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_CheckIfRotationSafeMove(
                                                               phdlStepper_DensoVS60_DataParams_t * pDataParams,                                /**< [In] Pointer to this layer's parameter structure.          */
                                                               double dwDistance,                                                               /**< [In] Distance in angle between target and current position */
                                                               double* arrTargetPostition,                                                      /**< [In] Array to the target position                          */
                                                               int bAxisType                                                                    /**< [In] Type of axis for movement 0 - X/Y ; 1 - Joint         */
                                                               );

/**
* \brief  Check if the move is allowed before moving( inside of given field)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_CheckBeforeMove(
                                                                  phdlStepper_DensoVS60_DataParams_t * pDataParams                             /**< [In] Pointer to this layer's parameter structure. */
                                                                 );


/**
* \brief  Check if the move between two XY points is possible without rotation ( no robot base in between two points)
* \return Integer status code
* \retval (0) - Is possible , (-1) - is not possible.
*/
EXTERNC int phdlStepper_DensoVS60_External_CheckIfMoveAllowed(
                                                              double* startPoints,                                                              /**< [In] Pointer to the cooridantes of the start point */
                                                              double* endPoint                                                                  /**< [In] Pointer to the coordinates of the end point   */
                                                             );

/**
* \brief  Check if the move is allowed before moving( inside of given field)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_CheckIfHeadAngleInRange(
                                                                phdlStepper_DensoVS60_DataParams_t * pDataParams,                               /**< [In] Pointer to the cooridantes of the start point */
                                                                double* arrAbosolutePosition,                                                   /**< [In] Array of absoulte target positions cooridantes */
                                                                int bAxisType                                                                   /**< [In] Type of axis for movement 0 - X/Y ; 1 - Joint         */
                                                               );

/**
* \brief  Check if motion is completed when move type is non-blocking
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_CheckMotionComplete(
                                                                        phdlStepper_DensoVS60_DataParams_t * pDataParams,                        /**< [In] Pointer to this layer's parameter structure. */
                                                                        uint8_t *pMotionCompleted                                                /**< [Out] Status if the motion is completed is PH_ON if in motion PH_OFF. */
                                                                        );


/**
* \brief  Check if target position lies out of range
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_CheckOutOfRange(
                                                      phdlStepper_DensoVS60_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
                                                      wchar_t* wStructCommand                                                                   /**< [In] Array with values of the target position cooridantes. */
                                                     );


/**
* \brief  Halt the motion when move type is non-blocking
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_ExecuteHaltMove(
                                                            phdlStepper_DensoVS60_DataParams_t * pDataParams                                    /**< [In] Pointer to this layer's parameter structure. */
                                                          );

/*------------------------------------------------------------------------*/
/*-------------------- HELP FUNCTIONS DECLARATION  -----------------------*/
/*------------------------------------------------------------------------*/

/**
* \brief  Parse the error code to check if the motor should still be turned on afterwards
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_parseErrorCode(
                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams,                                      /**< [In] Pointer to this layer's parameter structure. */
                                                         HRESULT hr                                                                             /**< [In] Return value of the executed move command */
                                                        );

/**
* \brief  Clear the error from the log and the remote controller
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_ClearErrorCode(
                                                         phdlStepper_DensoVS60_DataParams_t * pDataParams                                       /**< [In] Pointer to this layer's parameter structure. */
                                                        );

/**
* \brief  Initialize coordinates for both axis
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_InitializeCoordsStructures(
                                                                             phdlStepper_DensoVS60_DataParams_XY_coord_t* structXYCoords,       /**< [In] Pointer to the XY axis coordinates structure.    */
                                                                             phdlStepper_DensoVS60_DataParams_Joint_coord_t* structJointCoords  /**< [In] Pointer to the Joint axis coordiantes structure. */
                                                                            );

/**
* \brief  Normalize the Rz coordinate in order to prevent possible rotation over 360
* (only range of one rotation should be allowed)
* \return Double value
* \retval Converted int value given in micrometers from the provided double value in milimeters
*/
double phdlStepper_DensoVS60_External_NormalizeTargetedValue(
                                                             double dwCurrentValue,                                                             /**< [In] Current value of the rz cooridnate */
                                                             double * pTargetValue                                                              /**< [In] Target value of the rz cooridnate */
                                                            );

/**
* \brief  Find the interim angles between two points in there is more than 180 between them
* \return Array of double values
* \retval Array with double values which represented the interim angles
*/
double*  phdlStepper_DensoVS60_External_ArrayOfInterimAngles(
                                                             double dwDistance                                                                  /**< [In] Angle width between current and target Rz point */
                                                            );

/**
* \brief  Converts the cooridantes value given in doubles to the int32_t in order to be used higher in hierarchy
* \return Integer value
* \retval Converted int value given in micrometers from the provided double value in milimeters
*/
EXTERNC int32_t phdlStepper_DensoVS60_External_RoundDoubleValue(
                                                                double dwValue                                                                  /**< [In] Cooridante value given as double value */
                                                               );

/**
* \brief  Check if the target x-y position is inside the basis radius (robot should not go inside that area)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
EXTERNC phStatus_t phdlStepper_DensoVS60_External_CheckIfPointInsideBase(
                                                                        double dwXcoord,                                                        /**< [In] Target value of the x cooridante */
                                                                        double dwYcoord,                                                        /**< [In] Target value of the y cooridnate */
                                                                        int wInnerLimit                                                         /**< [In] Radius of the basis */
                                                                        );
/* TEST FUNCTIONS */

#ifdef _DEBUG
/**
* \brief  Logger function that is used in the debugging mode to log errors
* \return None
*/
EXTERNC void phdlStepper_DensoVS60_External_LogErrorCodes(
                                                  HRESULT hr                                                                                    /**< [In] Status of the executed CAO function */
                                                 );
#endif

/** @} \endcond */

#undef EXTERNC


#endif /* PHDLSTEPPER_DENSOVS60_EXTERNAL_H */
