/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Generic Spectrum Analyzer R&S FSV Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>

#ifdef NXPBUILD__PHDL_SPA_FSV
#include "FSV/phdlSPA_FSV.h"
#include "FSV\phdlSPA_FSV_Int.h"
#endif /* NXPBUILD__PHDL_SPA_FSV */

#ifdef NXPBUILD__PHDL_SPA

#include "phdlSPA_Int.h"
#pragma warning(push)           /* PRQA S 3116 */
#pragma warning(disable:4001)   /* PRQA S 3116 */
#include <stdio.h>              /* PRQA S 5124 */
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#pragma warning(pop)            /* PRQA S 3116 */

#include "../../phbalReg/src/Visa/external/visa.h"
#include "../../phbalReg/src/Visa/phbalReg_Visa_Cmd.h"


phStatus_t phdlSPA_Int_Send(
    phbalReg_Visa_DataParams_t * pBalRegDataParam,
    uint8_t * pCommand,
    uint8_t bWaitFinished
    )
{
    phStatus_t statusTmp;
    uint32_t dwValue=0;

    /* send data */
    statusTmp = phhalReg_Visa_Cmd_Transmit(
        pBalRegDataParam,
        PH_EXCHANGE_DEFAULT,
        pCommand,
        (uint16_t)strlen((char *)pCommand));

    if(bWaitFinished == PH_ON)
    {
        /* And send the wait finished command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_ExchangeGetUInt(pBalRegDataParam, (uint8_t *)"*OPC?\n", &dwValue, 1));
        /* *OPC? should always return 1 */
        if (dwValue != 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_SPA);
        }
    }
    /* Check if there was an error */
    return phdlSPA_Int_CheckError(pBalRegDataParam);
}

phStatus_t phdlSPA_Int_ExchangeGetUInt(
    phbalReg_Visa_DataParams_t * pBalRegDataParam,
    uint8_t * pCommand,
    uint32_t * dwResponse,
    uint8_t numberDigits
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxLen;

    if (numberDigits > sizeof(bRx) - 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_SPA);
    }

    /* send data */
    statusTmp = phbalReg_Exchange(
        pBalRegDataParam,
        PH_EXCHANGE_DEFAULT,
        pCommand,
        (uint16_t)strlen((char *)pCommand),
        numberDigits+1,
        bRx,
        &wRxLen);

    /* Check if there was only numberDigits received */
    if (bRx[numberDigits] != '\n')
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_SPA);
    }

    PH_CHECK_SUCCESS(statusTmp);

    if (sscanf((const char *)bRx, "%d\n", dwResponse) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_SPA);
    }

    /* Check if there was an error */
    return phdlSPA_Int_CheckError(pBalRegDataParam);
}

phStatus_t phdlSPA_Int_ExchangeGetResp(
    phbalReg_Visa_DataParams_t * pBalRegDataParam,
    uint8_t * pCommand,
    uint8_t * pResponse,
    uint16_t wRespSize
    )
{
    phStatus_t statusTmp;
    uint16_t wRxLen = 0;

    /* send data */
    statusTmp = phbalReg_Exchange(
        pBalRegDataParam,
        PH_EXCHANGE_DEFAULT,
        pCommand,
        (uint16_t)strlen((char *)pCommand),
        wRespSize,
        pResponse,
        &wRxLen);

    if (((statusTmp & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT) && wRxLen > 0)
    {
        statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
    }

    PH_CHECK_SUCCESS(statusTmp);

    /* Check if there was an error */
    return phdlSPA_Int_CheckError(pBalRegDataParam);
}

phStatus_t phdlSPA_Int_CheckError(
    phbalReg_Visa_DataParams_t * pBalRegDataParam
    )
{
    phStatus_t statusTmp;
    uint8_t errorBuffer[100];
    uint32_t errorCode;
    uint16_t wRxLen;
    errorCode = 0;

    /* send data */
    statusTmp = phbalReg_Exchange(
        pBalRegDataParam,
        PH_EXCHANGE_DEFAULT,
        (uint8_t *)("SYSTem:ERRor?\n"),
        (uint16_t)strlen("SYSTem:ERRor?\n"),
        sizeof(errorBuffer),
        errorBuffer,
        &wRxLen);

    /* If there was an timeout or success and data are available */
    if ((((statusTmp & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT) || ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS)) && wRxLen > 0)
    {
        statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
        if (sscanf((const char *)errorBuffer, "%d,\n", &errorCode) != 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_SPA);
        }
    }

    if (errorCode != 0)
    {
        /* -201 Instrument cannot carry out operation because it is in local mode Switch to remote mode */
        if (errorCode == (uint32_t)(-201))
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_SPA);
        }
        else if(errorCode == (uint32_t)(-410)) /*Query Unteterminated ignored */
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
        }
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

phStatus_t phdlSPA_Int_FindInstrumentId(
    phdlSPA_FSV_DataParams_t * pDataParams,
    int8_t * pbIdBuff,
    uint8_t bIdBuffSize,
    int8_t * pbSeperator
    )
{
    uint8_t bRefCount = 0, bJ = 0, sI = 0;
    uint8_t bseperatorId [10];
    uint8_t bmanufacturerId [21];

    if(bIdBuffSize > 0)
    {
        for(bJ = 0; bJ < bIdBuffSize; bJ++)
        {
            if(*(pbIdBuff + bJ) == *pbSeperator)
            {
                bseperatorId [sI] = bJ;
                sI++;
            }
        }

        for (bJ = 0; bJ < bseperatorId[0]; bJ++)
        {
            bmanufacturerId[bRefCount] = pbIdBuff[bJ];
            bRefCount++;
        }

        bmanufacturerId[bRefCount] = '\n';

        if(strncmp((char *)PHDL_SPA_RS_ID_STR, (char *)bmanufacturerId, strlen((char *)PHDL_SPA_RS_ID_STR)) == 0)
        {
            phdlSPA_FSV_Int_Set_Instrument_Parameter(pDataParams, pbIdBuff, bseperatorId);
        }
        else if(strncmp((char *)PHDL_SPA_AGILENT_ID_STR, (char *)bmanufacturerId, strlen((char *)PHDL_SPA_AGILENT_ID_STR)) == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_SPA);
        }
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

phStatus_t phdlSPA_Int_ExchangeGetUIntWithoutRespLeng(
    phbalReg_Visa_DataParams_t * pBalRegDataParam,
    uint8_t * pCommand,
    uint32_t * dwResponse
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxLen;

    /* send data */
    statusTmp = phbalReg_Exchange(
        pBalRegDataParam,
        PH_EXCHANGE_DEFAULT,
        pCommand,
        (uint16_t)strlen((char *)pCommand),
        10,
        bRx,
        &wRxLen);

    PH_CHECK_SUCCESS(statusTmp);

    if (sscanf((const char *)bRx, "%d\n", dwResponse) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_SPA);
    }

    /* Check if there was an error */
    return phdlSPA_Int_CheckError(pBalRegDataParam);
}

phStatus_t phdlSPA_Int_ExchangeGetUInt16WithoutRespLeng(
                                                        phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                                        uint8_t * pCommand,
                                                        uint16_t * pwResponse
                                                       )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxLen;
    uint32_t tempVar = 0, maxUint16Value = 0xFFFF;

    /* send data */
    statusTmp = phbalReg_Exchange(
        pBalRegDataParam,
        PH_EXCHANGE_DEFAULT,
        pCommand,
        (uint16_t)strlen((char *)pCommand),
        10,
        bRx,
        &wRxLen);

    PH_CHECK_SUCCESS(statusTmp);

    if (sscanf((const char *)bRx, "%d\n", &tempVar) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_SPA);
    }

    if(tempVar <= maxUint16Value)
    {
        *pwResponse = (uint16_t)tempVar;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    /* Check if there was an error */
    return phdlSPA_Int_CheckError(pBalRegDataParam);
}

phStatus_t phdlSPA_Int_ExchangeGetUInt64WithoutRespLeng(
    phbalReg_Visa_DataParams_t * pBalRegDataParam,
    uint8_t * pCommand,
    uint64_t * pqResponse
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxLen;

    /* send data */
    statusTmp = phbalReg_Exchange(
        pBalRegDataParam,
        PH_EXCHANGE_DEFAULT,
        pCommand,
        (uint16_t)strlen((char *)pCommand),
        10,
        bRx,
        &wRxLen);

    PH_CHECK_SUCCESS(statusTmp);

    if (sscanf((const char *)bRx, "%llu\n", pqResponse) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_SPA);
    }

    /* Check if there was an error */
    return phdlSPA_Int_CheckError(pBalRegDataParam);
}

phStatus_t phdlSPA_Int_ExchangeGetInt64WithoutRespLeng(
    phbalReg_Visa_DataParams_t * pBalRegDataParam,
    uint8_t * pCommand,
    uint8_t  bIsFloat,
    int64_t * pqResponse
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[15];
    uint16_t wRxLen;

    /* send data */
    statusTmp = phbalReg_Exchange(
        pBalRegDataParam,
        PH_EXCHANGE_DEFAULT,
        pCommand,
        (uint16_t)strlen((char *)pCommand),
        10,
        bRx,
        &wRxLen);

    PH_CHECK_SUCCESS(statusTmp);


    if(bIsFloat == PH_ON)
    {
        convertFloatToInt(bRx, pqResponse);
    }
    else
    {
        if (sscanf((const char *)bRx, "%lli\n", pqResponse) != 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_SPA);
        }
    }
    /* Check if there was an error */
    return phdlSPA_Int_CheckError(pBalRegDataParam);
}

phStatus_t phdlSPA_Int_GetInstrumentCalData(
                                            uint8_t * pbIdBuff,
                                            uint16_t bIdBuffSize,
                                            uint8_t * pbRefIdString,
                                            uint8_t bRefIdSize,
                                            uint8_t * pbSeperator,
                                            uint8_t * pbIdFound
                                           )
{
    uint8_t bI = 0, bRefCount = 0, bJ = 0, sI = 0, dI = 0, bDateCount = 0;
    uint8_t bseperatorId [10];
    struct tm * lTimeDate;
    int day, month, year, calDay;
    char dayString [5];

    /* Get current Date from system */
    time_t currentDate = time(0);
    lTimeDate = localtime(&currentDate);
    day = lTimeDate->tm_mday;
    month = lTimeDate->tm_mon;
    year = lTimeDate->tm_year;

    if(bIdBuffSize > bRefIdSize)
    {
        for(bJ = 0; bJ < bIdBuffSize; bJ++)
        {
            if(*(pbIdBuff + bJ) == *pbSeperator)
            {
                bseperatorId [sI] = bJ;
                sI++;
            }
            else if(*(pbIdBuff + bJ) == 'O')
                break;
        }

        for(bI = 0; bI < bseperatorId [0]; bI++)
        {
            for(bJ = 0; bJ < bRefIdSize; bJ++)
            {
                if(*(pbIdBuff + bI) == *(pbRefIdString + bJ))
                {
                    bRefCount++;
                }
            }
        }

        for(bI = bseperatorId[1] + 21; bI < bseperatorId[2] - 24; bI++)
        {
            dayString[dI] = pbIdBuff[bI];
            dI++;
        }

        calDay = atoi(dayString);

        if(day == calDay)
        {
            bDateCount++;
        }

        if(bRefCount == bRefIdSize && bDateCount > 0)
        {
            *pbIdFound = PH_ON;
        }
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

phStatus_t phdlSPA_Int_BalReg_VisaExchangeLarge(
    phdlSPA_FSV_DataParams_t * pDataParams,
    uint8_t * pbTxBuffer,
    uint16_t wTxLength,
    uint32_t dwRxBufSize,
    uint8_t * pbRxBuffer,
    uint32_t * pdwRxLength
    )
{
#ifdef _WIN32
    ViStatus statusVi;
    uint32_t dwBytesWritten, dwBytesRead;

    /* check if port is open */
    if (pDataParams->pBalRegDataParams->instr == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* check if response buffer is not zero */
    if (dwRxBufSize == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
    }

     /* Send command to instrument*/
    statusVi = (pDataParams->pBalRegDataParams->pviWrite)(pDataParams->pBalRegDataParams->instr, (ViBuf)pbTxBuffer, (ViUInt32)wTxLength, &dwBytesWritten);

    /* error handling */
    if (statusVi < VI_SUCCESS)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }
    if (dwBytesWritten != wTxLength)
    {
        /* write error not all data send */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    /* Read trace data*/
    statusVi = (pDataParams->pBalRegDataParams->pviRead)(pDataParams->pBalRegDataParams->instr, (ViBuf)pbRxBuffer, (ViUInt32)dwRxBufSize, &dwBytesRead);

    *pdwRxLength = dwBytesRead;

    /* Check for timeout error */
    if (dwBytesRead == 0 || statusVi == VI_ERROR_TMO)
    {
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_BAL);
    }
    /* error handling */
    if (statusVi < VI_SUCCESS)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
 #else
/* satisfy compiler */
pDataParams = NULL;
pTxBuffer = NULL;
wTxLength = 0;
wRxBufSize = 0;
pRxBuffer = NULL;
pRxLength = NULL;

return PH_ADD_COMPCODE(PH_ERR_NOT_AVAILABLE, PH_COMP_BAL);
#endif
}

/* Defines the trigger Source */
phStatus_t phdlSPA_Int_SetTriggerSource(
                                        phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                        uint16_t wValue
                                       )
{
    phStatus_t statusTmp;

    switch(wValue)
    {
    case PHDL_SPA_TRIGGER_SOURCE_EXTERNAL:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "TRIGger:SOURce EXTernal", PH_ON));
        break;

    case PHDL_SPA_TRIGGER_SOURCE_IFPOWER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "TRIGger:SOURce IFPower",PH_ON));
        break;

    case PHDL_SPA_TRIGGER_SOURCE_FREERUN:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "TRIGger:SOURce IMMediate",PH_ON));
        break;

    case PHDL_SPA_TRIGGER_SOURCE_MASK:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_SPA);

    case PHDL_SPA_TRIGGER_SOURCE_TIMEDOM:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_SPA);

    case PHDL_SPA_TRIGGER_SOURCE_TIME:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "TRIGger:SOURce TIME",PH_ON));
        break;

    case PHDL_SPA_TRIGGER_SOURCE_VIDEO:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "TRIGger:SOURce VIDeo", PH_ON));
        break;

    case PHDL_SPA_TRIGGER_SOURCE_AFPOWER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "TRIGger:SOURce AF", PH_ON));
        break;

    case PHDL_SPA_TRIGGER_SOURCE_FMPOWER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "TRIGger:SOURce FM", PH_ON));
        break;

    case PHDL_SPA_TRIGGER_SOURCE_AMPOWER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "TRIGger:SOURce AM", PH_ON));
        break;

    case PHDL_SPA_TRIGGER_SOURCE_AMRELATIV:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "TRIGger:SOURce AMRelative", PH_ON));
        break;

    case PHDL_SPA_TRIGGER_SOURCE_PMPOWER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "TRIGger:SOURce PM", PH_ON));
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Deffines the trigger slope */
phStatus_t phdlSPA_Int_SetTriggerSlope(
                                       phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                       uint16_t wValue
                                      )
{
    phStatus_t statusTmp;

    switch(wValue)
    {
    case PHDL_SPA_TRIGGER_SLOPE_POSITIVE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "TRIGger:SLOPe POSitive", PH_ON));
        break;

    case PHDL_SPA_TRIGGER_SLOPE_NEGATIVE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "TRIGger:SLOPe NEGative", PH_ON));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set the level for if power trigger */
phStatus_t phdlSPA_Int_SetTriggerLevelIfP(
                                          phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                          uint16_t wValue
                                         )
{
    phStatus_t statusTmp;
    uint8_t bIfPwCmd [30];

    if(wValue < -50 || wValue >= 20)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        sprintf_s((char *)bIfPwCmd, sizeof(bIfPwCmd), "TRIGger:LEVel:IFPower %dDBM", wValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, bIfPwCmd,PH_ON));
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set the level for the external trigger */
phStatus_t phdlSPA_Int_SetExtTriggerLevelMV(
                                            phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                            uint16_t wValue
                                           )
{
    phStatus_t statusTmp;
    uint8_t bExtLevCmd [30];
    uint16_t fullVolt, partVolt;

    if(wValue <= 500 || wValue >= 3500)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        if(wValue < 1000)
        {
            fullVolt = wValue / 100;
            sprintf_s((char *)bExtLevCmd, sizeof(bExtLevCmd), "TRIGger:LEVel:EXTernal 0.%dV", fullVolt);
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, bExtLevCmd, PH_ON));
        }
        else
        {
            fullVolt = wValue / 1000;
            partVolt = (wValue % 1000) / 100;
            sprintf_s((char *)bExtLevCmd, sizeof(bExtLevCmd), "TRIGger:LEVel:EXTernal %d.%dV", fullVolt, partVolt);
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, bExtLevCmd, PH_ON));
        }
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set the level for Rf power trigger */
phStatus_t phdlSPA_Int_SetTriggerLevelRfPower(
                                              phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                              uint16_t wValue
                                             )
{
    phStatus_t statusTmp;
    uint8_t bFmPowCmd [30];


    if(wValue < 10 || wValue > 50)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        sprintf_s((char *)bFmPowCmd, sizeof(bFmPowCmd), "TRIGger:LEVel:RFPower -%ddBm", wValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, bFmPowCmd, PH_ON));
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set the level for Video trigger */
phStatus_t phdlSPA_Int_SetTriggerLevelVideo(
                                            phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                            uint16_t wValue
                                           )
{
    phStatus_t statusTmp;
    uint8_t bFmPowCmd [30];


    if(wValue < 0 || wValue > 100)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        sprintf_s((char *)bFmPowCmd, sizeof(bFmPowCmd), "TRIGger:LEVel:VIDeo %dPCT", wValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, bFmPowCmd, PH_ON));
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set the mask conditions */
phStatus_t phdlSPA_Int_SetTriggerMaskCondition(
                                               phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                               uint16_t wValue
                                              )
{
    phStatus_t statusTmp;

    switch(wValue)
    {
    case PHDL_SPA_TRIGGER_CONDITION_MASKCOND_ENTER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "TRIGger:MASK:CONDition ENTer", PH_ON));
        break;

    case PHDL_SPA_TRIGGER_CONDITION_MASKCOND_LEAVE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "TRIGger:MASK:CONDition LEAVing", PH_ON));
        break;

    case PHDL_SPA_TRIGGER_CONDITION_MASKCOND_IN:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "TRIGger:MASK:CONDition INSide", PH_ON));
        break;

    case PHDL_SPA_TRIGGER_CONDITION_MASKCOND_OUT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "TRIGger:MASK:CONDition OUTSide", PH_ON));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}
/* Set the operation mode of the Analyzer */
phStatus_t phdlSPA_Int_SetSPAMode(
                                  phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                  uint16_t wValue,
                                  uint8_t * pbDevId
                                 )
{
    phStatus_t statusTmp;

    switch(wValue)
    {
    case PHDL_SPA_MODE_SPECTRUM:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "INSTrument SANalyzer", PH_ON));
        break;

    case PHDL_SPA_MODE_REALTIME:
        if(strncmp((char *)PHDL_SPA_FSV_STR, (char *)pbDevId, strlen((char *)PHDL_SPA_FSV_STR)) == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_SPA);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "INSTrument RTIMe", PH_ON));
        break;

    case PHDL_SPA_MODE_ANALOGDEMOD:
        if(strncmp((char *)PHDL_SPA_FSV_STR, (char *)pbDevId, strlen((char *)PHDL_SPA_FSV_STR)) == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_SPA);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "INSTrument ADEMod", PH_ON));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Defines the output format */
phStatus_t phdlSPA_Int_SetOutputFormat(
                                       phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                       uint16_t wValue,
                                       uint8_t * pbUseHeaderForTrace
                                      )
{
    phStatus_t statusTmp;

    switch(wValue)
    {
    case PHDL_SPA_OUTPUT_FORMAT_ASCII:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "FORMat ASCii", PH_ON));
        *pbUseHeaderForTrace = PH_OFF;
        break;

    case PHDL_SPA_OUTPUT_FORMAT_REAL:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "FORMat REAL", PH_ON));
        *pbUseHeaderForTrace = PH_OFF;
        break;

    case PHDL_SPA_OUTPUT_FORMAT_REAL_HEADER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "FORMat REAL", PH_ON));
        *pbUseHeaderForTrace = PH_ON;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set the center frequency */
phStatus_t phdlSPA_Int_SetCenterFreq_HZ(
                                        phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                        uint64_t qValue,
                                        uint64_t qMaxFreq
                                       )
{
    phStatus_t statusTmp;
    uint8_t bCentFreqCmd [30];

    if(qValue < 0 || qValue > qMaxFreq)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        sprintf_s((char *)bCentFreqCmd, sizeof(bCentFreqCmd), "FREQuency:CENTer %llu", qValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam,bCentFreqCmd, PH_ON));
    }

    /* Default Unit is Hz */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set frequenzy mode to zero span */
phStatus_t phdlSPA_Int_SetFreqMode(
                                   phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                   uint16_t wValue
                                  )
{
     phStatus_t statusTmp;

    switch(wValue)
    {
    case PHDL_SPA_FREQUENCY_MODE_CW:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "FREQuency:MODE CW", PH_ON));
        break;

    case PHDL_SPA_FREQUENCY_MODE_FIXED:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "FREQuency:MODE FIXed", PH_ON));
        break;

    case PHDL_SPA_FREQUENCY_MODE_SWEEP:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "FREQuency:MODE SWEep", PH_ON));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set the frequenzy span */
phStatus_t phdlSPA_Int_SetFreqSpan_HZ(
                                      phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                      uint64_t qValue
                                     )
{
    phStatus_t statusTmp;
    uint8_t bCentFreqCmd [30];

     if(qValue < 100 || qValue > 40e6)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        sprintf_s((char *)bCentFreqCmd, sizeof(bCentFreqCmd), "FREQuency:SPAN %llu", qValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam,bCentFreqCmd, PH_ON));
    }

    /* Default Unit is Hz */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set frequenzy offset */
phStatus_t phdlSPA_Int_SetFreqOffSet_HZ(
                                        phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                        int64_t qValue
                                       )
{
    phStatus_t statusTmp;
    uint8_t bCentFreqCmd [30];

     if(qValue < -100e9 || qValue > 100e9)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        sprintf_s((char *)bCentFreqCmd, sizeof(bCentFreqCmd), "FREQuency:OFFSet %lli", qValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam,bCentFreqCmd, PH_ON));
    }

    /* Default Unit is Hz */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);

}

/* Set Start frequenzy */
phStatus_t phdlSPA_Int_SetFreqStart_HZ(
                                       phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                       uint64_t qValue
                                      )
{
    phStatus_t statusTmp;
    uint8_t bCentFreqCmd [30];

     if(qValue < 100 || qValue > 40e6)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        sprintf_s((char *)bCentFreqCmd, sizeof(bCentFreqCmd), "FREQuency:STARt %llu", qValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam,bCentFreqCmd, PH_ON));
    }

    /* Default Unit is Hz */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set Stop frequenzy */
phStatus_t phdlSPA_Int_SetFreqStop_HZ(
                                      phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                      uint64_t qValue
                                     )
{
    phStatus_t statusTmp;
    uint8_t bCentFreqCmd [30];

     if(qValue < 100 || qValue > 40e6)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        sprintf_s((char *)bCentFreqCmd, sizeof(bCentFreqCmd), "FREQuency:STOP %llu", qValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam,bCentFreqCmd,PH_ON));
    }

    /* Default Unit is Hz */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set Sweep Mode */
phStatus_t phdlSPA_Int_SetSweepMode(
                                    phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                    uint16_t wValue
                                   )
{
    phStatus_t statusTmp;

    switch(wValue)
    {
    case PHDL_SPA_SWEEP_MODE_AUTO:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "SWEep:MODE AUTO", PH_ON));
        break;

    case PHDL_SPA_SWEEP_MODE_SPECTRUM_EMMISOIN_MASK:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "SWEep:MODE ESPectrum", PH_ON));
        break;

    case PHDL_SPA_SWEEP_MODE_SPURIOUS_EMMISIONS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "SWEep:MODE LIST", PH_ON));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set Sweep Type */
phStatus_t phdlSPA_Int_SetSweepType(
                                    phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                    uint16_t wValue
                                   )
{
    phStatus_t statusTmp;

    switch(wValue)
    {
    case PHDL_SPA_SWEEP_TYPE_SWEEPLIST:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "SWEep:TYPE SWE", PH_ON));
        break;

    case PHDL_SPA_SWEEP_TYPE_AUTO:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "SWEep:TYPE AUTO", PH_ON));
        break;

    case PHDL_SPA_SWEEP_TYPE_FFT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "SWEep:TYPE FFT", PH_ON));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Defines single mode sweep count */
phStatus_t phdlSPA_Int_SetSweepCount(
                                     phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                     uint16_t wValue
                                    )
{
    phStatus_t statusTmp;
    uint8_t bIfPwCmd [30];

    if(wValue < 0 || wValue >= 32767)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        sprintf_s((char *)bIfPwCmd, sizeof(bIfPwCmd), "SWEep:COUNt %d", wValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, bIfPwCmd, PH_ON));
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Defines the number of sweep points */
phStatus_t phdlSPA_Int_SetSweepPoints(
                                      phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                      uint16_t wValue
                                     )
{
    phStatus_t statusTmp;
    uint8_t bIfPwCmd [30];

    if(wValue < 101 || wValue > 32001)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        sprintf_s((char *)bIfPwCmd, sizeof(bIfPwCmd), "SWEep:POINts %d", wValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, bIfPwCmd, PH_ON));
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set averageing state to on or off */
phStatus_t phdlSPA_Int_SetAverageState(
                                       phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                       uint16_t wValue
                                      )
{
    phStatus_t statusTmp;

    switch(wValue)
    {
    case PH_ON:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "AVERage ON", PH_ON));
        break;

    case PH_OFF:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "AVERage OFF", PH_ON));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Define the number of points used for averaging */
phStatus_t phdlSPA_Int_SetAverageCount(
                                       phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                       uint16_t wValue
                                      )
{
    phStatus_t statusTmp;
    uint8_t bIfPwCmd [30];

    if(wValue < 0 || wValue >= 32767)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        sprintf_s((char *)bIfPwCmd, sizeof(bIfPwCmd), "AVERage:COUNt %d", wValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, bIfPwCmd, PH_ON));
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Define the averaging algorithm type */
phStatus_t phdlSPA_Int_SetAverageType(
                                      phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                      uint16_t wValue
                                     )
{
    phStatus_t statusTmp;

    switch(wValue)
    {
    case PHDL_SPA_AVERAGE_TYPE_VIDEO:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "AVERage:TYPE VIDeo", PH_ON));
        break;

    case PHDL_SPA_AVERAGE_TYPE_LINEAR:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "AVERage:TYPE LINear", PH_ON));
        break;

    case PHDL_SPA_AVERAGE_TYPE_POWER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "AVERage:TYPE POWer", PH_ON));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Defines the input coupling type */
phStatus_t phdlSPA_Int_SetInputCoupling(
                                        phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                        uint16_t wValue
                                       )
{
    phStatus_t statusTmp;

    switch(wValue)
    {
    case PHDL_SPA_INPUT_COUPLING_AC:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "INPut:COUPling AC", PH_ON));
        break;

    case PHDL_SPA_INPUT_COUPLING_DC:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "INPut:COUPling DC", PH_ON));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Define the input impedanze for the input channel */
phStatus_t phdlSPA_Int_SetInputImpedance(
                                         phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                         uint16_t wValue
                                        )
{
    phStatus_t statusTmp;

    switch(wValue)
    {
    case 50:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "INPut:IMPedance 50", PH_ON));
        break;

    case 75:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "INPut:IMPedance 75", PH_ON));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Define a input attenuation */
phStatus_t phdlSPA_Int_SetInputAttVal(
                                      phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                      uint16_t wValue,
                                      uint16_t wMaxAttValue
                                     )
{
    phStatus_t statusTmp;
    uint8_t bIfPwCmd [30];

    if(wValue < 0 || wValue > wMaxAttValue)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        sprintf_s((char *)bIfPwCmd, sizeof(bIfPwCmd), "INPut:ATTenuation %ddb", wValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, bIfPwCmd, PH_ON));
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set the automatic attenuation calculation to on or off */
phStatus_t phdlSPA_Int_SetInputAttAuto(
                                       phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                       uint16_t wValue
                                      )
{
    phStatus_t statusTmp;

    switch(wValue)
    {
    case PH_ON:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "INPut:ATTenuation:AUTO ON", PH_ON));
        break;

    case PH_OFF:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "INPut:ATTenuation:AUTO OFF", PH_ON));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Defines the resolution bandwidth */
phStatus_t phdlSPA_Int_SetBandwidthRes_HZ(
                                          phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                          uint64_t qValue
                                         )
{
    phStatus_t statusTmp;
    uint8_t bBandWidResCmd [30];

    if(qValue < 1 || qValue > 10e6)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        sprintf_s((char *)bBandWidResCmd, sizeof(bBandWidResCmd), "BANDwidth %llu", qValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam,bBandWidResCmd, PH_ON));
    }

    /* Default Unit is Hz */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Defines the bandwidth filter for fft mode */
phStatus_t phdlSPA_Int_SetBandwidthFilter(
                                          phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                          uint16_t wValue
                                         )
{
    phStatus_t statusTmp;

    switch(wValue)
    {
    case PHDL_SPA_BWIDTH_FILTER_TYPE_NORMAL:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "BANDwidth:TYPE NORMal", PH_ON));
        break;

    case PHDL_SPA_BWIDTH_FILTER_TYPE_CHANNEL:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "BANDwidth:TYPE CFILter", PH_ON));
        break;

    case PHDL_SPA_BWIDTH_FILTER_TYPE_RRC:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "BANDwidth:TYPE RRC", PH_ON));
        break;

    case PHDL_SPA_BWIDTH_FILTER_TYPE_P5:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "BANDwidth:TYPE P5", PH_ON));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

phStatus_t phdlSPA_Int_SetBandwidthResVideo_HZ(
                                               phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                               uint64_t qValue,
                                               uint64_t qMaxVideoBandwidth
                                              )
{
    phStatus_t statusTmp;
    uint8_t bBandResVidCmd [30];

    if(qValue < 100 || qValue > qMaxVideoBandwidth)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        sprintf_s((char *)bBandResVidCmd, sizeof(bBandResVidCmd), "BANDwidth:VIDeo %llu", qValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam,bBandResVidCmd, PH_ON));
    }

    /* Default Unit is Hz */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Defines the detector settings */
phStatus_t phdlSPA_Int_SetTraceDetectorMode(
                                            phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                            /*uint8_t * pTrace, */
                                            uint16_t wValue
                                           )
{
    phStatus_t statusTmp;

    switch(wValue)
    {
    case PHDL_SPA_TRACE_DETECTOR_APEAK:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "DETector APEak", PH_ON));
        break;

    case PHDL_SPA_TRACE_DETECTOR_NEGATIVE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "DETector NEGative", PH_ON));
        break;

    case PHDL_SPA_TRACE_DETECTOR_POSITIVE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "DETector POSitive", PH_ON));
        break;

    case PHDL_SPA_TRACE_DETECTOR_SAMPLE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "DETector SAMPle", PH_ON));
        break;

    case PHDL_SPA_TRACE_DETECTOR_RMS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "DETector RMS", PH_ON));
        break;

    case PHDL_SPA_TRACE_DETECTOR_AVERAGE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "DETector AVERage", PH_ON));
        break;

    case PHDL_SPA_TRACE_DETECTOR_QPEAK:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_SPA);

    case PHDL_SPA_TRACE_DETECTOR_CAVERAGE:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_SPA);
    case PHDL_SPA_TRACE_DETECTOR_CRMS:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_SPA);
    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set trace mode for display */
phStatus_t phdlSPA_Int_SetTraceMode(
                                    phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                    uint16_t wValue
                                   )
{
    phStatus_t statusTmp;

    switch(wValue)
    {
    case PHDL_SPA_TRACE_MODE_WRITE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "DISPlay:TRACe:MODE WRITe", PH_ON));
        break;

    case PHDL_SPA_TRACE_MODE_VIEW:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "DISPlay:TRACe:MODE VIEW", PH_ON));
        break;

    case PHDL_SPA_TRACE_MODE_AVERAGE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "DISPlay:TRACe:MODE AVERage", PH_ON));
        break;

    case PHDL_SPA_TRACE_MODE_MAXHOLD:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "DISPlay:TRACe:MODE MAXHold", PH_ON));
        break;

    case PHDL_SPA_TRACE_MODE_MINHOLD:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "DISPlay:TRACe:MODE MINHold", PH_ON));
        break;

    case PHDL_SPA_TRACE_MODE_BLANK:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "DISPlay:TRACe:MODE BLANk", PH_ON));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Defines the unit used for the power */
phStatus_t phdlSPA_Int_SetPowerUnit(
                                    phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                    uint16_t wValue
                                   )
{
    phStatus_t statusTmp;

    switch(wValue)
    {
    case PHDL_SPA_POWER_UNIT_DBM:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "CALCulate:UNIT:POWer DBM", PH_ON));
        break;

    case PHDL_SPA_POWER_UNIT_DBMV:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "CALCulate:UNIT:POWer DBMV", PH_ON));
        break;

    case PHDL_SPA_POWER_UNIT_DBUV:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "CALCulate:UNIT:POWer DBUV", PH_ON));
        break;

    case PHDL_SPA_POWER_UNIT_DBUA:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "CALCulate:UNIT:POWer DBUA", PH_ON));
        break;

    case PHDL_SPA_POWER_UNIT_DBPW:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "CALCulate:UNIT:POWer DBPW", PH_ON));
        break;

    case PHDL_SPA_POWER_UNIT_VOLT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "CALCulate:UNIT:POWer Volt", PH_ON));
        break;

    case PHDL_SPA_POWER_UNIT_AMPERE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "CALCulate:UNIT:POWer Ampere", PH_ON));
        break;

    case PHDL_SPA_POWER_UNIT_WATT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "CALCulate:UNIT:POWer Watt", PH_ON));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set reference level */
phStatus_t phdlSPA_Int_SetReferenceLevel(
                                         phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                         int64_t wValue,
                                         int64_t maxRefLevelValue
                                        )
{
    phStatus_t statusTmp;
    uint8_t bRefLevCmd [30];

    if(wValue < -124 || wValue > maxRefLevelValue)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        sprintf_s((char *)bRefLevCmd, sizeof(bRefLevCmd), "DISPlay:TRACe:Y:RLEVel %lli", wValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam,bRefLevCmd, PH_ON));
    }

    /* Default Unit is Hz */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set reference level offset*/
phStatus_t phdlSPA_Int_SetReferenceLevelOffset(
                                               phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                               int64_t wValue
                                              )
{
    phStatus_t statusTmp;
    uint8_t bRefLevCmd [35];

     if(wValue < -20 || wValue > 6)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        sprintf_s((char *)bRefLevCmd, sizeof(bRefLevCmd), "DISPlay:TRACe:Y:RLEVel:OFFSet %lli", wValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam,bRefLevCmd, PH_ON));
    }

    /* Default Unit is Hz */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}
/* Defines the run mode */
phStatus_t phdlSPA_Int_SetRunMode(
                                  phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                  uint16_t wValue
                                 )
{
    phStatus_t statusTmp;

    switch(wValue)
    {
    case PHDL_SPA_RUN_CONTINOUS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "INITiate:CONTinuous ON", PH_OFF));
        break;

    case PHDL_SPA_RUN_SINGLE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "INITiate:CONTinuous OFF", PH_OFF));
        break;

    case PHDL_SPA_RUN_ABORT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *)"ABORt\n", PH_ON));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/*Continue the measuremnt*/
phStatus_t phdlSPA_Int_SetRunModeContinue(
                                          phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                          uint16_t wValue
                                         )
{
    phStatus_t statusTmp;
    if(wValue == PH_ON)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *)"INITiate:CONMeas", PH_OFF));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set the Holdoff of the trigger*/
phStatus_t phdlSPA_Int_SetTriggerHoldoff_NS(
                                            phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                            int64_t qValue
                                           )
{
    phStatus_t statusTmp;
    uint8_t bHoldOffCmd [45];
    uint16_t zeroSpanTrue;
    int64_t sweepTime = 0;

    phdlSPA_Int_GetFreqMode(pBalRegDataParam, &zeroSpanTrue);
    phdlSPA_Int_GetSweepTime_NS(pBalRegDataParam, &sweepTime);

    if(zeroSpanTrue != PHDL_SPA_FREQUENCY_MODE_FIXED)
    {
        if(qValue < 0 || qValue > 30e9)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
        }
        else
        {
            sprintf_s((char *)bHoldOffCmd, sizeof(bHoldOffCmd), "TRIGger:HOLDoff %llins", qValue);
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam,bHoldOffCmd, PH_ON));
        }
    }
    else
    {
        if(qValue < -sweepTime || qValue > 30e9)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
        }
        else
        {
            sprintf_s((char *)bHoldOffCmd, sizeof(bHoldOffCmd), "TRIGger:HOLDoff %llins", qValue);
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam,bHoldOffCmd, PH_ON));
        }
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Set the sweep time*/
phStatus_t phdlSPA_Int_SetSweepTime_NS(
                                       phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                       int64_t qValue
                                      )
{
    phStatus_t statusTmp;
    uint8_t bSweepTimeCmd [45];
    uint16_t wVisaTimeout = 0;

    if(qValue < 1000 || qValue > 16e12)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    else
    {
        if (qValue > 32e9) /* If we use more than 65 seconds sweep time -> also adjust the VISA timeout to max */
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pBalRegDataParam, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 65535));
        }
        if (qValue > 1e9) /* If we use more than 1 seconds sweep time -> also adjust the VISA timeout to twice the time */
        {
            wVisaTimeout = 2 * (uint16_t)(qValue / 1000000);
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pBalRegDataParam, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wVisaTimeout));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pBalRegDataParam, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 2000));
        }

        sprintf_s((char *)bSweepTimeCmd, sizeof(bSweepTimeCmd), "SWEep:TIME %llins", qValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam,bSweepTimeCmd, PH_ON));
    }

    /* Default Unit is ns */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

phStatus_t phdlSPA_Int_Preset(
                              phbalReg_Visa_DataParams_t * pBalRegDataParam,
                              uint16_t wValue
                             )
{
    phStatus_t statusTmp;
    if(wValue == PH_ON)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *)"*RST\n", PH_ON));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

phStatus_t phdlSPA_Int_SetMeasuremntType(
                                         phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                         uint16_t wValue
                                        )
{
     phStatus_t statusTmp;

    switch(wValue)
    {
    case PHDL_SPA_MEASURE_TYPE_PEAK:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "CALCulate:MARKer:FUNCtion:SUMMary:PPEak ON", PH_ON));
        break;

    case PHDL_SPA_MEASURE_TYPE_MEAN:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *) "CALCulate:MARKer:FUNCtion:SUMMary:MEAN ON", PH_ON));
        break;

    case PHDL_SPA_MEASURE_TYPE_RMS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *)"CALCulate:MARKer:FUNCtion:SUMMary:RMS ON", PH_ON));
        break;

    case PHDL_SPA_MEASURE_TYPE_SDEV:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pBalRegDataParam, (uint8_t *)"CALCulate:MARKer:FUNCtion:SUMMary:SDEViation ON", PH_ON));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/*
* Start of the gettings helper functions
*/

/* Get actual trigger slope */
phStatus_t phdlSPA_Int_GetTriggerSlope(
                                       phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                       uint16_t * pwValue
                                      )
{
    phStatus_t statusTmp;
    uint8_t bRespBuff[6] = {0};
    uint8_t bclearedString[10] = {0};

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_ExchangeGetResp(pBalRegDataParam, (uint8_t *)"TRIGger:SLOPe?", bRespBuff, sizeof(bRespBuff)));

    if(bRespBuff[0] == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_DL_SPA);
    }

    removeWhiteSpaceConvertToUpper(bRespBuff, bclearedString);

    if (strncmp((char *)PHDL_SPA_INT_TRIGGER_SLOPE_POSITIVE_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_INT_TRIGGER_SLOPE_POSITIVE_STR)) == 0)
    {
        *pwValue = PHDL_SPA_TRIGGER_SLOPE_POSITIVE;
    }
    else if (strncmp((char *)PHDL_SPA_INT_TRIGGER_SLOPE_NEGATIVE_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_INT_TRIGGER_SLOPE_NEGATIVE_STR)) == 0)
    {
        *pwValue = PHDL_SPA_TRIGGER_SLOPE_NEGATIVE;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Get actual operationmode of the SPA */
phStatus_t phdlSPA_Int_GetSPAMode(
                                  phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                  uint16_t * pwValue
                                 )
{
    phStatus_t statusTmp;
    uint8_t bRespBuff[6] = {0};
    uint8_t bclearedString[10] = {0};

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_ExchangeGetResp(pBalRegDataParam, (uint8_t *)"INSTrument?", bRespBuff, sizeof(bRespBuff)));

    if(bRespBuff[0] == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_DL_SPA);
    }

    removeWhiteSpaceConvertToUpper(bRespBuff, bclearedString);

    if (strncmp((char *)PHDL_SPA_MODE_SPECTRUM_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_MODE_SPECTRUM_STR)) == 0)
    {
        *pwValue = PHDL_SPA_MODE_SPECTRUM;
    }
    else if (strncmp((char *)PHDL_SPA_MODE_REALTIME_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_MODE_REALTIME_STR)) == 0)
    {
        *pwValue = PHDL_SPA_MODE_REALTIME;
    }
    else if (strncmp((char *)PHDL_SPA_MODE_ANALOGDEMOD_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_MODE_ANALOGDEMOD_STR)) == 0)
    {
        *pwValue = PHDL_SPA_MODE_ANALOGDEMOD;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Get actual output format */
phStatus_t phdlSPA_Int_GetOutputFormat(
                                       phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                       uint16_t * pwValue,
                                       uint8_t bUseHeaderForTrace
                                      )
{
    phStatus_t statusTmp;
    uint8_t bRespBuff[10] = {0};
    uint8_t bclearedString[10] = {0};

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_ExchangeGetResp(pBalRegDataParam, (uint8_t *)"FORMat?", bRespBuff, sizeof(bRespBuff)));

    if(bRespBuff[0] == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_DL_SPA);
    }

    removeWhiteSpaceConvertToUpper(bRespBuff, bclearedString);

    if (strncmp((char *)PHDL_SPA_OUTPUT_FORMAT_ASCII_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_OUTPUT_FORMAT_ASCII_STR)) == 0)
    {
        *pwValue = PHDL_SPA_OUTPUT_FORMAT_ASCII;
    }
    else if (strncmp((char *)PHDL_SPA_OUTPUT_FORMAT_REAL_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_OUTPUT_FORMAT_REAL_STR)) == 0)
    {
        if (bUseHeaderForTrace == PH_ON)
        {
            *pwValue = PHDL_SPA_OUTPUT_FORMAT_REAL_HEADER;
        }
        else
        {
            *pwValue = PHDL_SPA_OUTPUT_FORMAT_REAL;
        }
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Get actual frequency mode */
phStatus_t phdlSPA_Int_GetFreqMode(
                                   phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                   uint16_t * pwValue
                                  )
{
    phStatus_t statusTmp;
    uint8_t bRespBuff[10] = {0};
    uint8_t bclearedString[10] = {0};

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_ExchangeGetResp(pBalRegDataParam, (uint8_t *)"FREQuency:MODE?", bRespBuff, sizeof(bRespBuff)));

    if(bRespBuff[0] == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_DL_SPA);
    }

    removeWhiteSpaceConvertToUpper(bRespBuff, bclearedString);

    if (strncmp((char *)PHDL_SPA_FREQUENCY_MODE_CW_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_FREQUENCY_MODE_CW_STR)) == 0)
    {
        *pwValue = PHDL_SPA_FREQUENCY_MODE_CW;
    }
    else if (strncmp((char *)PHDL_SPA_FREQUENCY_MODE_FIXED_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_FREQUENCY_MODE_FIXED_STR)) == 0)
    {
        *pwValue = PHDL_SPA_FREQUENCY_MODE_FIXED;
    }
    else if (strncmp((char *)PHDL_SPA_FREQUENCY_MODE_SWEEP_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_FREQUENCY_MODE_SWEEP_STR)) == 0)
    {
        *pwValue = PHDL_SPA_FREQUENCY_MODE_SWEEP;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Get center frequency value */
phStatus_t phdlSPA_Int_GetCenterFreq_HZ(
                                        phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                        int64_t * pqValue
                                       )
{
    return phdlSPA_Int_ExchangeGetInt64WithoutRespLeng(pBalRegDataParam,(uint8_t *)"FREQuency:CENTer?" , PH_OFF, pqValue);
}

/* Get span value */
phStatus_t phdlSPA_Int_GetFreqSpan_HZ(
                                      phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                      int64_t * pqValue
                                     )
{
    return phdlSPA_Int_ExchangeGetInt64WithoutRespLeng(pBalRegDataParam,(uint8_t *)"FREQuency:SPAN?" , PH_OFF, pqValue);
}

/* Get start frequency value */
phStatus_t phdlSPA_Int_GetFreqStart_HZ(
                                       phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                       int64_t * pqValue
                                      )
{
    return phdlSPA_Int_ExchangeGetInt64WithoutRespLeng(pBalRegDataParam,(uint8_t *)"FREQuency:STARt?" , PH_OFF, pqValue);
}

/* Get stop frequency value */
phStatus_t phdlSPA_Int_GetFreqStop_HZ(
                                      phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                      int64_t * pqValue
                                     )
{
    return phdlSPA_Int_ExchangeGetInt64WithoutRespLeng(pBalRegDataParam,(uint8_t *)"FREQuency:STOP?" , PH_OFF, pqValue);
}

/* Get sweep count */
phStatus_t phdlSPA_Int_GetSweepCount(
                                     phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                     uint16_t * pwValue
                                    )
{
    return phdlSPA_Int_ExchangeGetUInt16WithoutRespLeng(pBalRegDataParam,(uint8_t *)"SWEep:COUNt?" , pwValue);
}

/* Get sweep points */
phStatus_t phdlSPA_Int_GetSweepPoints(
                                      phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                      uint16_t * pwValue
                                     )
{
    return phdlSPA_Int_ExchangeGetUInt16WithoutRespLeng(pBalRegDataParam,(uint8_t *)"SWEep:POINts?" , pwValue);
}

/* Get average state */
phStatus_t phdlSPA_Int_GetAverageState(
                                       phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                       uint16_t * pwValue
                                      )
{
    return phdlSPA_Int_ExchangeGetUInt16WithoutRespLeng(pBalRegDataParam, (uint8_t *)"AVERage?", pwValue);
}

/* Get number of averaging points */
phStatus_t phdlSPA_Int_GetAverageCount(
                                       phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                       uint16_t * pwValue
                                      )
{
    return phdlSPA_Int_ExchangeGetUInt16WithoutRespLeng(pBalRegDataParam, (uint8_t *)"AVERage:COUNt?", pwValue);
}

/* Get averaging type */
phStatus_t phdlSPA_Int_GetAverageType(
                                      phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                      uint16_t * pwValue
                                     )
{
    phStatus_t statusTmp;
    uint8_t bRespBuff[10] = {0};
    uint8_t bclearedString[10] = {0};

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_ExchangeGetResp(pBalRegDataParam, (uint8_t *)"AVERage:TYPE?", bRespBuff, sizeof(bRespBuff)));

    if(bRespBuff[0] == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_DL_SPA);
    }

    removeWhiteSpaceConvertToUpper(bRespBuff, bclearedString);

    if (strncmp((char *)PHDL_SPA_AVERAGE_TYPE_VIDEO_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_AVERAGE_TYPE_VIDEO_STR)) == 0)
    {
        *pwValue = PHDL_SPA_AVERAGE_TYPE_VIDEO;
    }
    else if (strncmp((char *)PHDL_SPA_AVERAGE_TYPE_LINEAR_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_AVERAGE_TYPE_LINEAR_STR)) == 0)
    {
        *pwValue = PHDL_SPA_AVERAGE_TYPE_LINEAR;
    }
    else if (strncmp((char *)PHDL_SPA_AVERAGE_TYPE_POWER_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_AVERAGE_TYPE_POWER_STR)) == 0)
    {
        *pwValue = PHDL_SPA_AVERAGE_TYPE_POWER;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Get the actual bandwidth resolution */
phStatus_t phdlSPA_Int_GetBandwidthRes_HZ(
                                          phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                          int64_t * pqValue
                                         )
{
    return phdlSPA_Int_ExchangeGetInt64WithoutRespLeng(pBalRegDataParam,(uint8_t *)"BANDwidth?" , PH_OFF, pqValue);
}

phStatus_t phdlSPA_Int_GetBandwidthResVideo_HZ(
                                               phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                               int64_t * pqValue
                                              )
{
    return phdlSPA_Int_ExchangeGetInt64WithoutRespLeng(pBalRegDataParam,(uint8_t *)"BANDwidth:VIDeo?" , PH_OFF, pqValue);
}

/* Get actual input attenuation value */
phStatus_t phdlSPA_Int_GetInputAttVal(
                                      phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                      uint16_t * pwValue
                                     )
{
    return phdlSPA_Int_ExchangeGetUInt16WithoutRespLeng(pBalRegDataParam, (uint8_t *)"INPut:ATTenuation?", pwValue);
}

/* Get trigger holdoff value */
phStatus_t phdlSPA_Int_GetTriggerHoldoff_NS(
                                            phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                            int64_t * pqValue
                                           )
{
    return phdlSPA_Int_ExchangeGetInt64WithoutRespLeng(pBalRegDataParam,(uint8_t *)"TRIGger:HOLDoff?" , PH_ON, pqValue);
}

/* Get sweep time value */
phStatus_t phdlSPA_Int_GetSweepTime_NS(
                                       phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                       int64_t * pqValue
                                      )
{
    return phdlSPA_Int_ExchangeGetInt64WithoutRespLeng(pBalRegDataParam,(uint8_t *)"SWEep:TIME?" , PH_ON, pqValue);
}

phStatus_t phdlSPA_Int_GetMeasurementData(
                                          phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                          uint8_t * pCommand,
                                          float64_t * pfValue
                                         )
{
    phStatus_t statusTmp;
    uint8_t bResponse[16];
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_ExchangeGetResp(pBalRegDataParam, pCommand, bResponse, sizeof(bResponse)));
    *pfValue = atof((const char *)bResponse);
    return statusTmp;
}

/* Get the Operation completed sate*/
phStatus_t phdlSPA_Int_GetOperationCompleted(
                                             phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                             uint16_t * bCompleted
                                            )
{
    return phdlSPA_Int_ExchangeGetUInt16WithoutRespLeng(pBalRegDataParam, (uint8_t *)"*OPC?\n", bCompleted);
}

/* Get Actual Power Unit*/
phStatus_t phdlSPA_Int_GetPowerUnit(
                                    phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                    uint16_t * pwValue
                                   )
{
    phStatus_t statusTmp;
    uint8_t bRespBuff[10] = {0};
    uint8_t bclearedString[10] = {0};

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_ExchangeGetResp(pBalRegDataParam, (uint8_t *)"CALCulate:UNIT:POWer?", bRespBuff, sizeof(bRespBuff)));

    if(bRespBuff[0] == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_DL_SPA);
    }

    removeWhiteSpaceConvertToUpper(bRespBuff, bclearedString);

    if (strncmp((char *)PHDL_SPA_POWER_UNIT_DBM_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_POWER_UNIT_DBM_STR)) == 0)
    {
        *pwValue = PHDL_SPA_POWER_UNIT_DBM;
    }
    else if (strncmp((char *)PHDL_SPA_POWER_UNIT_DBMUV_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_POWER_UNIT_DBMUV_STR)) == 0)
    {
        *pwValue = PHDL_SPA_POWER_UNIT_DBMV;
    }
    else if (strncmp((char *)PHDL_SPA_POWER_UNIT_DBMUA_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_POWER_UNIT_DBMUA_STR)) == 0)
    {
        *pwValue = PHDL_SPA_POWER_UNIT_DBUV;
    }
    else if (strncmp((char *)PHDL_SPA_POWER_UNIT_DBMUA_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_POWER_UNIT_DBMUA_STR)) == 0)
    {
        *pwValue = PHDL_SPA_POWER_UNIT_DBUA;
    }
    else if (strncmp((char *)PHDL_SPA_POWER_UNIT_DBPW_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_POWER_UNIT_DBPW_STR)) == 0)
    {
        *pwValue = PHDL_SPA_POWER_UNIT_DBPW;
    }
    else if (strncmp((char *)PHDL_SPA_POWER_UNIT_VOLT_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_POWER_UNIT_VOLT_STR)) == 0)
    {
        *pwValue = PHDL_SPA_POWER_UNIT_VOLT;
    }
    else if (strncmp((char *)PHDL_SPA_POWER_UNIT_AMPE_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_POWER_UNIT_AMPE_STR)) == 0)
    {
        *pwValue = PHDL_SPA_POWER_UNIT_AMPERE;
    }
    else if (strncmp((char *)PHDL_SPA_POWER_UNIT_WATT_STR, (char *)bclearedString, strlen((char *)PHDL_SPA_POWER_UNIT_WATT_STR)) == 0)
    {
        *pwValue = PHDL_SPA_POWER_UNIT_WATT;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_SPA);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

/* Get reference level */
phStatus_t phdlSPA_Int_GetReferenceLevel(
                                         phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                         int64_t * pqValue
                                        )
{
    return phdlSPA_Int_ExchangeGetInt64WithoutRespLeng(pBalRegDataParam,(uint8_t *)"DISPlay:TRACe:Y:RLEVel?" , PH_ON, pqValue);
}

/* Get reference level offset */
phStatus_t phdlSPA_Int_GetReferenceLevelOffset(
                                               phbalReg_Visa_DataParams_t * pBalRegDataParam,
                                               int64_t * pqValue
                                              )
{
    return phdlSPA_Int_ExchangeGetInt64WithoutRespLeng(pBalRegDataParam,(uint8_t *)"DISPlay:TRACe:Y:RLEVel:OFFSet?" , PH_ON, pqValue);
}
void removeWhiteSpaceConvertToUpper(
                                    uint8_t * pbConvBuff,
                                    uint8_t * pbReturnBuff
                                   )
{
    int16_t wI = 0;
    uint8_t spaces_removed = PH_OFF;

    for (wI = (int16_t) strlen((char *)pbConvBuff) -1; wI >= 0; wI--)
    {
        if (spaces_removed == PH_OFF) /* Only remove spaces at te end until a non whitespace is found */
        {
            if (isspace(pbConvBuff[wI]) != 0)
            {
                /* We found a whitespace remove it */
                pbReturnBuff[wI] = 0;
            }
            else /* First non whitespace so disable search */
            {
                spaces_removed = PH_ON;
                pbReturnBuff[wI] = (uint8_t)toupper(pbConvBuff[wI]);
            }
        }
        else
        {
            pbReturnBuff[wI] = (uint8_t)toupper(pbConvBuff[wI]);
        }
    }
}

void convertFloatToInt(
                       uint8_t * pbRxBuff,
                       int64_t * pqConvertedNum
                      )
{
    double temp = 0.0;

    temp = atof((const char *)pbRxBuff);
    temp = temp * 1e9; /* Convert to nano seconds */
    *pqConvertedNum = (int64_t)temp;
}
#endif /* NXPBUILD__PHDL_SPA */
