/*
 * Copyright 2019 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * ISO3 FPGA soft scope.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE

#include <phhalHw.h> /* Softscope uses ISO3 HAL as its BAL.*/
#include "phdlOsci_ISO3SoftScope.h"
#include "phdlOsci_ISO3SoftScope_Int.h"
#include "phdlOsci_ISO3SoftScope_Cmd.h"
#include "../../../../intfs/phhalHw_ISO3_cmd.h"             /* Requires Hal ISO3 interface function */
#include "../../../phhalHw/src/ISO3/phhalHw_ISO3_reg.h"     /* Register definitions for ISO3 Reader */

phStatus_t phdlOsci_ISO3SoftScope_Cmd_GetTriggerCounter(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bChannel,
    uint32_t * dwRiseCounter,
    uint32_t * dwFallCounter
    )
{
    return phdlOsci_ISO3SoftScope_Int_GetTriggerCounter(
        pDataParams,
        bChannel,
        dwRiseCounter,
        dwFallCounter);
}

phStatus_t phdlOsci_ISO3SoftScope_Cmd_Reset(
                                           phdlOsci_ISO3SoftScope_DataParams_t * pDataParams
                                           )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);

    pHalISO3DataParams = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;

    /* Init Trigger settings */
    ISO3_REG_TRIGGER_CTRL_0.conditionT1Rise = PHHAL_HW_ISO3_TRIG_MODE_TX_SOC;
    ISO3_REG_TRIGGER_CTRL_0.conditionT1Fall = PHHAL_HW_ISO3_TRIG_MODE_TX_LAST_BIT;
    ISO3_REG_TRIGGER_CTRL_0.conditionT2Rise = PHHAL_HW_ISO3_TRIG_MODE_RX_SOC;
    ISO3_REG_TRIGGER_CTRL_0.conditionT2Fall = PHHAL_HW_ISO3_TRIG_MODE_RX_EOC;
    ISO3_REG_TRIGGER_CTRL_0.externEnableT1 = PH_ON;
    ISO3_REG_TRIGGER_CTRL_0.externEnableT2 = PH_ON;
    ISO3_REG_TRIGGER_CTRL_0.signalIdT1 = PHHAL_HW_ISO3_TRIG1_SIGNAL_ADC_INPUT;
    ISO3_REG_TRIGGER_CTRL_0.signalIdT2 = PHHAL_HW_ISO3_TRIG2_SIGNAL_ADC_INPUT;
    ISO3_REG_TRIGGER_CTRL_0.fdtEnable = PH_OFF;

    ISO3_REG_TRIGGER_CTRL_1.triggerRiseOnBitNumber = 3;
    ISO3_REG_TRIGGER_CTRL_1.triggerFallOnBitNumber = 3;

    /* Init soft scope settings */
    ISO3_REG_LOG_CTRL.trig0     = 0;
    ISO3_REG_LOG_CTRL.trig1     = 0;
    ISO3_REG_LOG_CTRL.trigForce = 0;
    ISO3_REG_LOG_CTRL.trigMls   = 0;
    ISO3_REG_LOG_CTRL.rfu1      = 0;
    ISO3_REG_LOG_CTRL.logRx     = 0;
    ISO3_REG_LOG_CTRL.logCal    = 0;
    ISO3_REG_LOG_CTRL.logCos    = 0;
    ISO3_REG_LOG_CTRL.logSin    = 0;
    ISO3_REG_LOG_CTRL.logAgcO   = 0;
    ISO3_REG_LOG_CTRL.logIqO    = 0;
    ISO3_REG_LOG_CTRL.logDacO   = 0;
    ISO3_REG_LOG_CTRL.rfu2      = 0;

    /* Write config to FPGA register */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_TRIG1_CTRL0,
        (uint8_t*)&pDataParams->dwTriggerCtrl0Reg,
        sizeof(uint32_t)));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_TRIG1_CTRL1,
        (uint8_t*)&pDataParams->dwTriggerCtrl1Reg,
        sizeof(uint32_t)));

    /* set log size to maximum size */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_SetBufferSize(pDataParams, PHDL_OSCI_ISO3SOFTSCOPE_MAX_BUFFER_SIZE/PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_LOG_CTRL,
        (uint8_t*)&pDataParams->dwLogCtrlReg,
        sizeof(uint32_t)));

    /* Resets trigger counters */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_ResetSoftScope(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_ISO3);
}

phStatus_t phdlOsci_ISO3SoftScope_Cmd_StoreBigBufferDataInRemoteBuffer(
                                           phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
                                           uint8_t bRemoteBufferID,
                                           uint32_t dwMaxBufferSizeBytes
                                           )
{
    return phdlOsci_ISO3SoftScope_Int_StoreBigBufferDataInRemoteBuffer(
        pDataParams,
        bRemoteBufferID,
        dwMaxBufferSizeBytes);
}

phStatus_t phdlOsci_ISO3SoftScope_Cmd_ForceTrigger(phdlOsci_ISO3SoftScope_DataParams_t * pDataParams)
{
    return phdlOsci_ISO3SoftScope_Int_ForceTrigger(pDataParams);
}

phStatus_t phdlOsci_ISO3SoftScope_Cmd_IsSignalInRange(phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
                                                      uint32_t dwDataSource,
                                                      uint8_t * overflow)
{
    return phdlOsci_ISO3SoftScope_Int_IsSignalInRange(pDataParams, dwDataSource, overflow);
}

phStatus_t phdlOsci_ISO3SoftScope_Cmd_GetWaveFormBlocking(
                                       phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
                                       uint8_t  bChannel,
                                       uint32_t dwWaveFormBufferSize,
                                       uint32_t * pdwWaveFormLength,
                                       uint8_t * pbWaveFormBuffer,
                                       uint8_t * pbHeaderOffset
                                       )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;

    /* Error checks */
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);
    pHalISO3DataParams = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;

    /* Wait until softscope finishes acquiring trace */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_CompleteAcquisition(pDataParams));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_GetWaveForm(pDataParams, bChannel, dwWaveFormBufferSize,
        pdwWaveFormLength, pbWaveFormBuffer, pbHeaderOffset));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}
#endif /* NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE */
