/*
 * Copyright 2017 - 2018, 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Hardware Oscilloscope Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <math.h>
#include <stdio.h>              /* PRQA S 5124 */

#ifdef NXPBUILD__PHDL_OSCI_DSO7052A

/* for visa functionality displaced function exchange large */
#include "../../../phbalReg/src/Visa/external/visa.h"
#include "../../../phbalReg/src/Visa/phbalReg_Visa_Cmd.h"

#include "phdlOsci_DSO7052A.h"
#include "phdlOsci_DSO7052A_Int.h"
#include "../phdlOsci_Int.h"
#include <phdlOsci_DSO7052A_Cmd.h>

phStatus_t phdlOsci_DSO7052A_Cmd_SaveSetupSlotNumber(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t bSlotNumber
    )
{
    phStatus_t statusTmp;
    int32_t dwPrintLength = 0;
    uint8_t abCmd[20];

    /* check slot range */
    if(bSlotNumber < 0 || bSlotNumber > 9)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    /* build and send command */
    dwPrintLength = sprintf((char *)abCmd, ":SAV:SET:STAR %d\n", bSlotNumber);
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, (uint16_t)dwPrintLength));

    /* Wait for command and check execution status */
    PH_CHECK_SUCCESS_FCT(statusTmp,phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Cmd_RecallSetupSlotNumber(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t bSlotNumber
    )
{
    phStatus_t statusTmp;
    int32_t dwPrintLength = 0;
    uint8_t abCmd[20];

    /* check slot range */
    if(bSlotNumber < 0 || bSlotNumber > 9)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    /* build and send command */
    dwPrintLength = sprintf((char *)abCmd, ":REC:SET:STAR %d\n", bSlotNumber);
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, (uint16_t)dwPrintLength));

    /* Wait for command and check execution status */
    PH_CHECK_SUCCESS_FCT(statusTmp,phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Cmd_SaveSetupFileName(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t *pFileName
    )
{
    phStatus_t statusTmp;
    int32_t dwPrintLength = 0;
    uint8_t abCmd[300];

    if(pFileName == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    /* check file name length and extension */
    dwPrintLength = (int32_t)strlen((char *)pFileName);
    if(dwPrintLength < 0 || dwPrintLength > 256)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    /* build and send command */
    dwPrintLength = sprintf((char *)abCmd, ":SAV:SET:STAR \"/usb0/%s\"\n", pFileName);
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, (uint16_t)dwPrintLength));

    /* Wait for command and check execution status */
    PH_CHECK_SUCCESS_FCT(statusTmp,phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Cmd_RecallSetupFileName(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t *pFileName
    )
{
    phStatus_t statusTmp;
    int32_t dwPrintLength = 0;
    uint8_t abCmd[300];

    if(pFileName == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    /* check file name length and extension */
    dwPrintLength = (int32_t)strlen((char *)pFileName);
    if(dwPrintLength < 0 || dwPrintLength > 256)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    /* build and send command */
    dwPrintLength = sprintf((char *)abCmd, ":REC:SET:STAR \"/usb0/%s\"\n", pFileName);
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, (uint16_t)dwPrintLength));

    /* Wait for command and check execution status */
    PH_CHECK_SUCCESS_FCT(statusTmp,phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Cmd_GetAverage(
                                   phdlOsci_DSO7052A_DataParams_t * pDataParams,
                                   uint8_t  bChannel,
                                   uint16_t * pwAverage
                                   )
{
    phStatus_t statusTmp = PH_ERR_SUCCESS, statusRestore;
    float32_t fTemp = 0.0;
    uint8_t bFSAdjustCnt = 0;
    uint16_t wOrigMinRange, wOrigMaxRange;
    uint16_t wTriggerMode;

    if (bChannel < 1 || bChannel > pDataParams->bNumberOfChannels)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    wOrigMinRange = pDataParams->wRangeMin;
    wOrigMaxRange = pDataParams->wRangeMax;
    wTriggerMode = pDataParams->wCurrentTriggerMode;
    /* check if mode is not set to auto */
    if (wTriggerMode != PHDL_OSCI_TRIGGER_AUTO)
    {
        /* set mode to auto to be able to get in correct range in any case */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerMode(pDataParams, PHDL_OSCI_TRIGGER_AUTO));
    }

    /* We try to adjust the osci range 5 times if we have some measurment problems */
    for (bFSAdjustCnt = 0; bFSAdjustCnt < 5; bFSAdjustCnt++)
    {
        /* If we have at least two retries that fails -> we adjust the scale on the scope */
        if (bFSAdjustCnt > 1)
        {
            pDataParams->wRangeMax = (uint16_t)max(25, (int32_t)wOrigMaxRange - ((int32_t)bFSAdjustCnt-1) * 10);
            pDataParams->wRangeMin = (uint16_t)max(20, (int32_t)wOrigMinRange - ((int32_t)bFSAdjustCnt-1) * 10);
        }


        statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
        if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_RANGE) == PHDL_OSCI_MEASUREMENT_ADJUST_RANGE)
        {
            /* get in correct range */
            statusTmp = phdlOsci_DSO7052A_Int_SetCorrRange(pDataParams, bChannel);
            if ((statusTmp & PH_ERR_MASK) == PH_ERR_PARAMETER_OVERFLOW)
            {
                continue;
            }
        }

        /* Only perform command if previous was success */
        if ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* get the VRMS of channel */
            statusTmp = phdlOsci_DSO7052A_Int_GetAverage(pDataParams , bChannel, &fTemp);
            if ((statusTmp & PH_ERR_MASK) == PH_ERR_PARAMETER_OVERFLOW)
            {
                continue;
            }
        }
        break;
    }

    /* Restore scaling */
    pDataParams->wRangeMin = wOrigMinRange;
    pDataParams->wRangeMax = wOrigMaxRange;
    /* Restore triggermode if it was not set to auto */
    if (wTriggerMode != PHDL_OSCI_TRIGGER_AUTO)
    {
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_DSO7052A_Int_SetTriggerMode(pDataParams, wTriggerMode));
    }

    /* Check the status */
    PH_CHECK_SUCCESS(statusTmp);

    /* assign value */
    *pwAverage = (uint16_t) (fTemp * PHDL_OSCI_VOLTAGE_DIVISOR);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

#endif /* NXPBUILD__PHDL_OSCI_DSO7052A */
