/*
 * Copyright 2017, 2019 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software Master Amplifier Oscilloscope Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <math.h>

#ifdef NXPBUILD__PHDL_MSTAMPOSC_MP300

#include <phdlMstAmpOsc.h>
#include "phdlMstAmpOsc_Mp300.h"
#include "phdlMstAmpOsc_MP300_Int.h"

#include <phdlAmp.h>
#include <phdlOsci.h>
#include <phTools.h>


phStatus_t phdlMstAmpOsc_Mp300_Init(
                                 phdlMstAmpOsc_Mp300_DataParams_t * pDataParams,
                                 uint16_t wSizeOfDataParams,
                                 void * pDlAmpDataParams,
                                 void * pDlOsciDataParams,
                                 void * pDlRdGainDataParams,
                                 uint8_t bChannel,
                                 uint8_t bNumberPoints,
                                 double *pGainValues,
                                 double *pFieldstrengthValues,
                                 double *pSplineParameterStorage,
                                 uint16_t wSizeOfSplineParameterStorage
                                 )
{
    phStatus_t statusTmp;
    uint8_t bIndex = 0;
    uint16_t wMaxGain = 0;
    float32_t fGainStep = 0.0;
    float32_t fActGain = 0.0;
    uint8_t bZeroValueFound = PH_OFF;

    if (sizeof(phdlMstAmpOsc_Mp300_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_MSTAMPOSC);
    }

    if (wSizeOfSplineParameterStorage < 18*sizeof(double)*bNumberPoints)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
    }

    PH_ASSERT_NULL (pDataParams);

    /* at least a amplfier or a mp300 must supplied */
    if (pDlRdGainDataParams == NULL)
    {
        PH_ASSERT_NULL (pDlAmpDataParams);
    }
    if (pDlAmpDataParams == NULL)
    {
        PH_ASSERT_NULL (pDlRdGainDataParams);
    }

    /* Scope must only be supplied if no fieldstrength values are supplied */
    if (pFieldstrengthValues == NULL)
    {
        PH_ASSERT_NULL (pDlOsciDataParams);
    }
    PH_ASSERT_NULL (pSplineParameterStorage);

    /* check channel parameter (only if scope available) */
    if ((pDlOsciDataParams != NULL) && (bChannel < 1 || bChannel > 4))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
    }

    /* check bNumberPoints parameter */
    if (bNumberPoints < 2)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
    }

    /* if FS values are given also gain values must be supplied */
    if (pFieldstrengthValues != NULL)
    {
        PH_ASSERT_NULL (pGainValues);
    }

    /* Gain values must be monotonically increasing */
    if (pGainValues != NULL)
    {
        for( bIndex = 0; bIndex < bNumberPoints-1; bIndex++ )
        {
            if (pGainValues[bIndex] >= pGainValues[bIndex+1])
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
            }
        }
    }
    /* Fieldstrength values must be monotonically increasing */
    if (pFieldstrengthValues != NULL)
    {
        /* at least the second supplied point must be non zero */
        if (pFieldstrengthValues[1] == 0.0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        bZeroValueFound = PH_OFF;
        for( bIndex = 0; bIndex < bNumberPoints-1; bIndex++ )
        {
            /* 0 values are allowed that are not measured values. But all values after this has to be set to a 0 */
            if (pFieldstrengthValues[bIndex+1] == 0.0)
            {
                bZeroValueFound = PH_ON;
                continue;
            }
            if (bZeroValueFound == PH_ON)
            {
                if (pFieldstrengthValues[bIndex+1] != 0.0)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
                }
                continue;
            }
            if (pFieldstrengthValues[bIndex] >= pFieldstrengthValues[bIndex+1])
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
            }
        }
    }

    /* Get Min and Max gain (if amplifier supplied from amplifier else use the mp300 gains) */
    if (pDlAmpDataParams != NULL)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_GetConfig(pDlAmpDataParams, PHDL_AMP_CONFIG_MIN_GAIN, (uint16_t * UNALIGNED)&pDataParams->wMinAmpGain));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_GetConfig(pDlAmpDataParams, PHDL_AMP_CONFIG_MAX_GAIN, (uint16_t * UNALIGNED)&wMaxGain));
    }
    else if(pDlRdGainDataParams != NULL)
    {
        /* Get Values supported by MP300/500 or FPGA */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_GetMaxOutputGain(pDlRdGainDataParams, &wMaxGain));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_GetMinOutputGain(pDlRdGainDataParams, (uint16_t * UNALIGNED)&pDataParams->wMinAmpGain));
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
    }

    /* Check min and max gain */
    if (pGainValues != NULL)
    {
        if ((pGainValues[0] < pDataParams->wMinAmpGain) || (pGainValues[bNumberPoints-1] > wMaxGain))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
    }
    else /* Out from min gain, max gain and bNumberPoints calc the stepsize */
    {
        fGainStep = ((float32_t)(wMaxGain-pDataParams->wMinAmpGain))/((float32_t)(bNumberPoints-1));
        fActGain = (float32_t)pDataParams->wMinAmpGain;
    }
    /* init private data */
    pDataParams->wId                  = PH_COMP_DL_MSTAMPOSC | PHDL_MSTAMPOSC_MP300_ID;
    pDataParams->pDlAmpDataParams     = pDlAmpDataParams;
    pDataParams->pDlOsciDataParams    = pDlOsciDataParams;
    pDataParams->pDlRdGainDataParams  = pDlRdGainDataParams;

    /* Set Correct Pointers */
    pDataParams->pdGainValues = pSplineParameterStorage;
    pDataParams->pdFieldstrength = pDataParams->pdGainValues + 2*bNumberPoints;
    pDataParams->pdbH2Gain = pDataParams->pdFieldstrength + 2*bNumberPoints;
    pDataParams->pdcH2Gain = pDataParams->pdbH2Gain + 2*bNumberPoints;
    pDataParams->pddH2Gain = pDataParams->pdcH2Gain + 2*bNumberPoints;
    pDataParams->pdbGain2H = pDataParams->pddH2Gain + 2*bNumberPoints;
    pDataParams->pdcGain2H = pDataParams->pdbGain2H + 2*bNumberPoints;
    pDataParams->pddGain2H = pDataParams->pdcGain2H + 2*bNumberPoints;
    pDataParams->pdTempBuffer = pDataParams->pddGain2H + 2*bNumberPoints;

    /* reset the spline parameter */
    for( bIndex = 0; bIndex < 2*bNumberPoints; bIndex++ )
    {
        pDataParams->pdGainValues[bIndex] = 0.0;
        pDataParams->pdFieldstrength[bIndex] = 0.0;
        pDataParams->pdbH2Gain[bIndex] = 0.0;
        pDataParams->pdcH2Gain[bIndex] = 0.0;
        pDataParams->pddH2Gain[bIndex] = 0.0;
        pDataParams->pdbGain2H[bIndex] = 0.0;
        pDataParams->pdcGain2H[bIndex] = 0.0;
        pDataParams->pddGain2H[bIndex] = 0.0;
        pDataParams->pdTempBuffer[bIndex] = 0.0;
    }

    /* Copy supplied values or set linear gain points */
    for( bIndex = 0; bIndex < bNumberPoints; bIndex++ )
    {
        if (pGainValues != NULL)
        {
            pDataParams->pdGainValues[bIndex] = pGainValues[bIndex];
        }
        else
        {
            pDataParams->pdGainValues[bIndex] = (uint16_t)(fActGain + 0.5);
            fActGain += fGainStep;
        }
        if (pFieldstrengthValues != NULL)
        {
            /* Check if the supplied fieldstrengt does not exit limits */
            if (pFieldstrengthValues[bIndex] > 65535.0)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
            }
            pDataParams->pdFieldstrength[bIndex] = pFieldstrengthValues[bIndex];
        }
        else
        {
            pDataParams->pdFieldstrength[bIndex] = 0.0;
        }
    }

    pDataParams->wMaxFieldstrengthUsed = 8500;
    pDataParams->bNumberPoints = bNumberPoints;
    pDataParams->bMaxPoints = 2*bNumberPoints;
    pDataParams->bFieldstrengthValuesSupplied = (pFieldstrengthValues == NULL) ? PH_OFF : PH_ON;
    pDataParams->bMaxMeasuredInterpolationPoint = 0;
    pDataParams->bChannel = bChannel;
    pDataParams->bPrecision = 2;
    pDataParams->bCheckPrecision = PH_OFF;
    pDataParams->bFastRecalibrateEnabled = PH_ON;
    pDataParams->bUseNormalCalIfFastFail = PH_ON;
    pDataParams->bFineControl = PH_OFF;
    pDataParams->bCal = PH_OFF;
    pDataParams->wDefaultMp300Gain = 1000;
    pDataParams->wCurrAmpGain = pDataParams->wMinAmpGain;
    pDataParams->bRxGainMode = PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_DISABLED;
    pDataParams->wMinMp300RxGainFS = 750;
    pDataParams->wMaxMp300RxGainFS = 7500;
    pDataParams->wMinMp300RxGain = 750;
    pDataParams->wMaxMp300RxGain = 1500;
    pDataParams->bSenseCoilBalanceChannel = 0;
    pDataParams->bSenseCoilBalanceCheckLimit = 500; /* starting compare from 500mV */
    pDataParams->bCancellationToken = PH_OFF;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_Mp300_InitMstAmpOsc(
    phdlMstAmpOsc_Mp300_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;

    /* Init Osci */
    if (pDataParams->pDlOsciDataParams != NULL)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_InitOsci(pDataParams->pDlOsciDataParams));
    }

    if (pDataParams->pDlAmpDataParams != NULL)
    {
        /* set amp gain to minimum */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wMinAmpGain));
        pDataParams->wCurrAmpGain = pDataParams->wMinAmpGain;

        /* switch on amplifier */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetPower(pDataParams->pDlAmpDataParams, PH_ON));
    }
    else /* Only Mp300 is used so switch gain do Min */
    {
        /* set gain to minimum */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetOutputGain(pDataParams->pDlRdGainDataParams, pDataParams->wMinAmpGain));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_Mp300_SetConfig(
                                      phdlMstAmpOsc_Mp300_DataParams_t * pDataParams,
                                      uint16_t wIdentifier,
                                      uint16_t wValue
                                      )
{
    phStatus_t statusTmp;
    uint16_t wMaxMPGain = 0;
    uint16_t wGainLimit = 0;

    switch (wIdentifier)
    {
    case   PHDL_MSTAMPOSC_CONFIG_PRECISION:
        /* check value*/
        if (wValue > 100)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        /* Fine Control only available if precision <= 10% */
        if (pDataParams->bFineControl == PH_ON && wValue > 10)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_MSTAMPOSC);
        }

        /* store value */
        pDataParams->bPrecision = (uint8_t)wValue;
        break;

    case PHDL_MSTAMPOSC_CONFIG_CHECK_PRECISION:
        /* check value */
        if (wValue != PH_ON && wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        /* Fine Control only available if check precision is enabled */
        if (pDataParams->bFineControl == PH_ON && wValue != PH_ON)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_MSTAMPOSC);
        }

        /* If check precision will be switched on a osci must be present */
        if (wValue == PH_ON && pDataParams->pDlOsciDataParams == NULL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_MSTAMPOSC);
        }
        /* store value */
        pDataParams->bCheckPrecision = (uint8_t)wValue;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_FAST_RECALIBRATE:
        /* check value */
        if (wValue != PH_ON && wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        pDataParams->bFastRecalibrateEnabled = (uint8_t)wValue;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_USE_NORMAL_CAL_IF_FAST_FAIL:
        /* check value */
        if (wValue != PH_ON && wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        pDataParams->bUseNormalCalIfFastFail = (uint8_t)wValue;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_SENSE_COIL_BALANCE_CHANNEL:
        /* check value */
        if (wValue > PHDL_OSCI_CHANNEL_4)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        pDataParams->bSenseCoilBalanceChannel = (uint8_t)wValue;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_CANCELLATION_TOKEN:
        /* check value */
        if (wValue != PH_ON && wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        pDataParams->bCancellationToken = (uint8_t)wValue;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_MAX_FIELDSTRENGTH_USED:
        /* check value not more than 30 A/m allowed*/
        if (wValue > 30000)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        /* check if calibration become invalid */
        if (wValue > pDataParams->wMaxFieldstrengthUsed)
        {
            pDataParams->bCal = PH_OFF;
        }
        else if (pDataParams->bCal == PH_ON && wValue < pDataParams->wMaxFieldstrengthUsed) /* Max Gain is deactivated so reset also the points above wValue */
        {
            while ((pDataParams->bMaxMeasuredInterpolationPoint > 1) && (pDataParams->pdFieldstrength[pDataParams->bMaxMeasuredInterpolationPoint-1] > wValue))
            {
                pDataParams->bMaxMeasuredInterpolationPoint--;
            }
        }
        /* store value */
        pDataParams->wMaxFieldstrengthUsed = wValue;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_USE_FINE_CONTROL:
        /* check value */
        if (wValue != PH_ON && wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        /* If fine control will be switched on a osci and a reader must be present */
        if (wValue == PH_ON && (pDataParams->pDlOsciDataParams == NULL || pDataParams->pDlRdGainDataParams == NULL))
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_MSTAMPOSC);
        }
        /* Fine Control only available if precision <= 10% and CheckPrecision is enabled */
        if (wValue == PH_ON && (pDataParams->bPrecision > 10 || pDataParams->bCheckPrecision == PH_OFF))
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_MSTAMPOSC);
        }

        /* store value */
        pDataParams->bFineControl = (uint8_t)wValue;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_DEFAULT_MP300_GAIN:
        /* check value */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_GetMaxOutputGain(pDataParams->pDlRdGainDataParams, &wMaxMPGain));

        if (wValue > (uint16_t)(0.978*wMaxMPGain)) /* We need margin to the upper limit but want do use 1000 also for ISO setup */
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        pDataParams->wDefaultMp300Gain = wValue;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_RX_GAIN_TARGET_MODE:
        /* check value */
        if (wValue > PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_LINEAR_LIMIT_DOWN)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        pDataParams->bRxGainMode = (uint8_t)wValue;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_RX_GAIN_TARGET_MIN_FS:
        pDataParams->wMinMp300RxGainFS = wValue;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_RX_GAIN_TARGET_MAX_FS:
        pDataParams->wMaxMp300RxGainFS = wValue;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_RX_GAIN_TARGET_MIN:
        /* check value */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_GetMinRxGain(pDataParams->pDlRdGainDataParams, &wGainLimit));
        if (wValue < wGainLimit)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_GetMaxRxGain(pDataParams->pDlRdGainDataParams, &wGainLimit));
        if (wValue > wGainLimit)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        pDataParams->wMinMp300RxGain = wValue;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_RX_GAIN_TARGET_MAX:
        /* check value */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_GetMinRxGain(pDataParams->pDlRdGainDataParams, &wGainLimit));
        if (wValue < wGainLimit)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_GetMaxRxGain(pDataParams->pDlRdGainDataParams, &wGainLimit));
        if (wValue > wGainLimit)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        pDataParams->wMaxMp300RxGain = wValue;
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_Mp300_GetConfig(
                                      phdlMstAmpOsc_Mp300_DataParams_t * pDataParams,
                                      uint16_t wIdentifier,
                                      uint16_t * pwValue
                                      )
{
    switch (wIdentifier)
    {
    case PHDL_MSTAMPOSC_CONFIG_PRECISION:
        /* return precision */
        *pwValue = (uint16_t)pDataParams->bPrecision;
        break;

    case PHDL_MSTAMPOSC_CONFIG_CHECK_PRECISION:
        /* return check precision */
        *pwValue = (uint16_t)pDataParams->bCheckPrecision;
        break;

    case PHDL_MSTAMPOSC_CONFIG_MIN_FIELD:
        /* check if calibrated */
        if (pDataParams->bCal == PH_OFF)
        {
            /*return error code*/
            return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_NOT_CAL, PH_COMP_DL_MSTAMPOSC);
        }

        if (pDataParams->pDlRdGainDataParams == NULL)
        {
            if (pDataParams->pdFieldstrength[0] >= 65535.5)
            {   /* There must be an error in because it is not possible that there are more than 65 A/m */
                return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_MSTAMPOSC);
            }
            *pwValue = (uint16_t)(pDataParams->pdFieldstrength[0] + 0.5);
        }
        else
        {
            *pwValue = 100; /*Min Field at least 100 mA/m to reach also the precission because the noise may be to high if the FS is to low. */
        }
        break;

    case PHDL_MSTAMPOSC_CONFIG_MAX_FIELD:
        /* check if calibrated */
        if (pDataParams->bCal == PH_OFF)
        {
            /*return error code*/
            return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_NOT_CAL, PH_COMP_DL_MSTAMPOSC);
        }

        if (pDataParams->pdFieldstrength[pDataParams->bMaxMeasuredInterpolationPoint] >= 65535.5)
        {   /* There must be an error in because it is not possible that there are more than 65 A/m */
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_MSTAMPOSC);
        }
        *pwValue = (uint16_t)(pDataParams->pdFieldstrength[pDataParams->bMaxMeasuredInterpolationPoint] + 0.5);
        break;

    case PHDL_MSTAMPOSC_CONFIG_AMPLIFIER_GAIN:
        /* return current amplifier gain */
        *pwValue = (uint16_t)pDataParams->wCurrAmpGain;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_MAX_FIELDSTRENGTH_USED:
        /* return use max gain */
        *pwValue = pDataParams->wMaxFieldstrengthUsed;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_USE_FINE_CONTROL:
        /* return use fine control */
        *pwValue = (uint16_t)pDataParams->bFineControl;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_DEFAULT_MP300_GAIN:
        *pwValue = pDataParams->wDefaultMp300Gain;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_RX_GAIN_TARGET_MODE:
        *pwValue = (uint16_t)pDataParams->bRxGainMode;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_RX_GAIN_TARGET_MIN_FS:
        *pwValue = pDataParams->wMinMp300RxGainFS;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_RX_GAIN_TARGET_MAX_FS:
        *pwValue = pDataParams->wMaxMp300RxGainFS;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_RX_GAIN_TARGET_MIN:
        *pwValue = pDataParams->wMinMp300RxGain;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_RX_GAIN_TARGET_MAX:
        *pwValue = pDataParams->wMaxMp300RxGain;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_FAST_RECALIBRATE:
        *pwValue = pDataParams->bFastRecalibrateEnabled;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_USE_NORMAL_CAL_IF_FAST_FAIL:
        *pwValue = pDataParams->bUseNormalCalIfFastFail;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_SENSE_COIL_BALANCE_CHANNEL:
        *pwValue = pDataParams->bSenseCoilBalanceChannel;
        break;

    case PHDL_MSTAMPOSC_MP300_CONFIG_CANCELLATION_TOKEN:
        *pwValue = pDataParams->bCancellationToken;
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        break;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_Mp300_Cal(
                                phdlMstAmpOsc_Mp300_DataParams_t * pDataParams
                                )
{
    phStatus_t statusTmp;
    phStatus_t statusCalResult = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
    uint16_t wFieldStrengthMeas;
    uint8_t bActInterpolationPoint;
    uint8_t bCalibrationRetryCnt = 0;

    /* Set MP300 Fieldstrength to pDataParams->wDefaultMp300Gain (just if also amplifier is used) */
    if (pDataParams->pDlRdGainDataParams != NULL && pDataParams->pDlAmpDataParams != NULL)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetOutputGain(pDataParams->pDlRdGainDataParams, pDataParams->wDefaultMp300Gain));
    }
    bCalibrationRetryCnt = 0;
    for( bActInterpolationPoint = 0; bActInterpolationPoint < pDataParams->bNumberPoints; )
    {
        if (pDataParams->bFieldstrengthValuesSupplied == PH_ON)
        {
            /* Supplied value 0 means that this point is not measured so break the calibration loop at this point */
            if (bActInterpolationPoint > 0 && pDataParams->pdFieldstrength[bActInterpolationPoint] == 0.0)
            {
                break;
            }
            if (pDataParams->pdFieldstrength[bActInterpolationPoint] > 65535.0)
            {
                statusCalResult = PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_MSTAMPOSC);
                break;
            }
            wFieldStrengthMeas = (uint16_t)(pDataParams->pdFieldstrength[bActInterpolationPoint] + 0.5);
        }
        else
        {
            /* Check for cancellation */
            if (pDataParams->bCancellationToken == PH_ON)
            {
                pDataParams->bCancellationToken = PH_OFF;
                statusCalResult = PH_ADD_COMPCODE(PH_ERR_CANCELED, PH_COMP_DL_MSTAMPOSC);
                break;
            }

            statusCalResult = phdlMstAmpOsc_Mp300_Int_MeasureCalibrationPoint(pDataParams, (uint16_t)pDataParams->pdGainValues[bActInterpolationPoint], &wFieldStrengthMeas);
            if ((statusCalResult & PH_ERR_MASK) != PH_ERR_SUCCESS)
            {
                break;
            }
        }

        /* Check if the FS is monoton rising */
        if ((bActInterpolationPoint != 0) && (wFieldStrengthMeas <= pDataParams->pdFieldstrength[bActInterpolationPoint-1]))
        {
            bCalibrationRetryCnt++;
            if (bCalibrationRetryCnt == 3)
            {
                statusCalResult = PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_COULD_NOT_FIND_GAIN_VALS, PH_COMP_DL_MSTAMPOSC);
                break;
            }
            bActInterpolationPoint = 0;
            continue;
        }
        pDataParams->pdFieldstrength[bActInterpolationPoint] = wFieldStrengthMeas;
        pDataParams->bMaxMeasuredInterpolationPoint = bActInterpolationPoint;

        /* damper check */
        if (wFieldStrengthMeas > pDataParams->wMaxFieldstrengthUsed)
        {
            break;
        }
        bActInterpolationPoint++;
    }

    /* Reset Gain to wMinAmpGain to not kill the card in very high field */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetSecureState(pDataParams));

    /* Check for error in calibration */
    PH_CHECK_SUCCESS(statusCalResult);

    /* Calc spline */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_CalcSplineValues(
        pDataParams->bMaxMeasuredInterpolationPoint, pDataParams->pdFieldstrength, pDataParams->pdGainValues,
        pDataParams->pdbH2Gain, pDataParams->pdcH2Gain, pDataParams->pddH2Gain,
        pDataParams->pdbGain2H, pDataParams->pdcGain2H, pDataParams->pddGain2H));

    /* set calibrated flag */
    pDataParams->bCal = PH_ON;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}


phStatus_t phdlMstAmpOsc_Mp300_SetFieldStrength(
    phdlMstAmpOsc_Mp300_DataParams_t * pDataParams,
    uint16_t wFieldStrength
    )
{
    phStatus_t statusTmp;
    uint16_t wGainValue;
    uint16_t wMp300GainValueThousand;
    uint16_t wFieldStrengthMeas = 0;
    uint16_t wPrecisionLimitLower = 0;
    uint16_t wPrecisionLimitUpper = 0;
    uint16_t wRecalibrationLimitLower = 0;
    uint16_t wRecalibrationLimitUpper = 0;
    uint8_t bRecalCounter = 0;
    uint8_t bReaderAdjustCounter = 0;
    uint16_t wMaxMPGain = 0;
    float64_t dfReaderTopRoom = 0.0;
    float64_t dfMaxTopRoom = 0.0;
    float64_t dfPrecisionAbs = 0.0; /* Absolute size of the precision im mA/m */

    /* Calc precision limit */
    dfPrecisionAbs = (float64_t)wFieldStrength * (float64_t)pDataParams->bPrecision / 100.0;
    /* We limit the precision to minimum 10 mA/m */
    if (dfPrecisionAbs < 10.0)
        dfPrecisionAbs = 10.0;
    wPrecisionLimitUpper = wFieldStrength + (uint16_t)(dfPrecisionAbs + 0.5);
    if ((float32_t)wFieldStrength - dfPrecisionAbs < 0.0)
    {
        wPrecisionLimitLower = 0;
    }
    else
    {
        wPrecisionLimitLower = wFieldStrength - (uint16_t)(dfPrecisionAbs + 0.5);
    }

    /* If no reader is used use same limit for Recalibration and Precision */
    if (pDataParams->pDlRdGainDataParams == NULL)
    {
        wRecalibrationLimitLower = wPrecisionLimitLower;
        wRecalibrationLimitUpper = wPrecisionLimitUpper;
    }
    else
    {
        /* Calc recalibration limit */
        /* The upper limit is 10% higher than the precision limit because the reader can scale down easy */
        wRecalibrationLimitUpper = (uint16_t)(((float32_t)wPrecisionLimitUpper * 1.1) + 0.5);
        /* Calc recal limit bottom (take into acount the available values for the reader) */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_GetMaxOutputGain(pDataParams->pDlRdGainDataParams, &wMaxMPGain));

        /* If a fieldstrength below the min FS of the amplifier is used allow more headroom */
        if (wFieldStrength < pDataParams->pdFieldstrength[0])
        {
            /* Calc the possible headroom depending on the reader gain that is used for the target fieldstrength */
            wMp300GainValueThousand = (uint16_t)((float32_t)wFieldStrength / ((float32_t)pDataParams->pdFieldstrength[0] / (float32_t)pDataParams->wDefaultMp300Gain)+0.5);
            dfReaderTopRoom = (((double)wMaxMPGain/(double)wMp300GainValueThousand));

            /* Calc max toproom out of the fs (1.1 to 2.0) */
            dfMaxTopRoom = 2.0 - (0.9 * (float64_t)wFieldStrength / (float64_t)pDataParams->pdFieldstrength[0]);

            /* limit maximum head room */
            if (dfReaderTopRoom > dfMaxTopRoom)
                dfReaderTopRoom = dfMaxTopRoom;
        }
        else
        {
            dfReaderTopRoom = (((double)wMaxMPGain/(double)pDataParams->wDefaultMp300Gain));
            /* maximum is 10% from precision limit */
            if (dfReaderTopRoom > 1.1)
                dfReaderTopRoom = 1.1;
        }

        wRecalibrationLimitLower = (uint16_t)(((float32_t)wPrecisionLimitLower / dfReaderTopRoom) + 0.5);
    }

    /* check if calibrated */
    if (pDataParams->bCal == PH_OFF)
    {
        /*return error code*/
        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_NOT_CAL, PH_COMP_DL_MSTAMPOSC);
    }

    /* check field strength limits */
    if ( (wFieldStrength > pDataParams->pdFieldstrength[pDataParams->bMaxMeasuredInterpolationPoint]) || (wFieldStrength > pDataParams->wMaxFieldstrengthUsed) )
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetSecureState(pDataParams));
        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_TOO_HIGH, PH_COMP_DL_MSTAMPOSC);
    }
    if ( (pDataParams->pDlRdGainDataParams == NULL) && (wFieldStrength < pDataParams->pdFieldstrength[0]) )
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetSecureState(pDataParams));
        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_TOO_LOW, PH_COMP_DL_MSTAMPOSC);
    }

    /* Adjust the reader reception gain */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_AdjustReaderRxGain(pDataParams, wFieldStrength));

    bRecalCounter = 0;
    do
    {
        /* calculate gain value */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_CalcGainValues(pDataParams, wFieldStrength, &wGainValue, &wMp300GainValueThousand));

        /* Apply Value */
        if (pDataParams->pDlAmpDataParams != NULL)
        {
            /* set gain value */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, wGainValue));
            pDataParams->wCurrAmpGain = wGainValue;
            if (pDataParams->pDlRdGainDataParams != NULL)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetOutputGain(pDataParams->pDlRdGainDataParams, wMp300GainValueThousand));
            }
        }
        else /* Only Mp300 */
        {
            wMp300GainValueThousand = wGainValue; /* The gain value is used for the mp300 */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetOutputGain(pDataParams->pDlRdGainDataParams, wMp300GainValueThousand));
        }

        /* if we did not check just return */
        if (pDataParams->bCheckPrecision == PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
        }

        /* Wait so FS can be stabile */
        phTools_Sleep(50);

        /* read field strength */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_GetFieldStrengthRangeChecked(pDataParams, wRecalibrationLimitLower, wRecalibrationLimitUpper, &wFieldStrengthMeas));

        /* If outside recal limits perform a recalibration and perform next loop */
        if (wFieldStrengthMeas < wRecalibrationLimitLower || wFieldStrengthMeas > wRecalibrationLimitUpper)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_ReCalibrate(pDataParams, wFieldStrength, bRecalCounter));
            bRecalCounter++;
            continue;
        }

        /* If we are in precsion and fine contol is not used -> finsihed */
        if (wPrecisionLimitLower <= wFieldStrengthMeas && wFieldStrengthMeas <= wPrecisionLimitUpper && pDataParams->bFineControl == PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
        }

        /* Calc the best reader output gain */
        bReaderAdjustCounter = 0;
        do
        {
            /* Calc new gain Value to adjust */
            wMp300GainValueThousand = (uint16_t)((float32_t)wMp300GainValueThousand*((float32_t)wFieldStrength/(float32_t)wFieldStrengthMeas)+0.5);

            /* get Limit of MP and check if within limit */
            PH_CHECK_SUCCESS_FCT(statusTmp,
                phdlMstAmpOsc_Mp300_Int_GetMaxOutputGain(pDataParams->pDlRdGainDataParams, &wMaxMPGain));
            if (wMp300GainValueThousand > wMaxMPGain)
            {
                /* This can occure if we are complite at the edge if the calibration limits
                   if we use the max gain it should be possible to accive the presission limits */
                wMp300GainValueThousand = wMaxMPGain;
            }

            PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetOutputGain(pDataParams->pDlRdGainDataParams, wMp300GainValueThousand));

            /* Wait so FS can be stabile */
            phTools_Sleep(50);

            /* read new field again strength */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_GetFieldStrengthRangeChecked(pDataParams, wRecalibrationLimitLower, wRecalibrationLimitUpper, &wFieldStrengthMeas));

            /* If we are outside the recalibration limits something realy went wrong so calibrate again */
            if (wFieldStrengthMeas < wRecalibrationLimitLower || wFieldStrengthMeas > wRecalibrationLimitUpper)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_ReCalibrate(pDataParams, wFieldStrength, bRecalCounter));
                bRecalCounter++;
                /* Break reader adjust loop to come again to begin */
                break;
            }

            /* Of now inside precision -> return */
            if (wPrecisionLimitLower <= wFieldStrengthMeas && wFieldStrengthMeas <= wPrecisionLimitUpper)
            {
                return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
            }

            /* Inside recal outside precision -> perform another try */
            bReaderAdjustCounter++;
            /* If limit of retries is reached -> preform calibration again */
            if (bReaderAdjustCounter > PHDL_MSTAMPOSC_NUM_OF_RECAL)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_ReCalibrate(pDataParams, wFieldStrength, bRecalCounter));
                bRecalCounter++;
                /* Break reader adjust loop to come again to begin */
                break;
            }
            /* Continue with next try */
        } /* Should be always true because if we are within the limits we will return befor */
        while( wFieldStrengthMeas < wPrecisionLimitLower || wFieldStrengthMeas > wPrecisionLimitUpper );

        continue;
    } /* Should always be true */
    while( bRecalCounter >= 0 );

    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_MSTAMPOSC);
}

#endif /* NXPBUILD__PHDL_MSTAMPOSC_MP300 */
