/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * ISO Setup Software Master Amplifier Oscilloscope Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_MSTAMPOSC_ISOSETUP
#include <phdlMstAmpOsc.h>

#include "phdlMstAmpOsc_ISOSetup.h"
#include "phdlMstAmpOsc_ISOSetup_Int.h"

#include <phdlAmp.h>
#include <phdlOsci.h>
#include <phdlRdFpga.h>

phStatus_t phdlMstAmpOsc_ISOSetup_Int_CheckFieldRange(
    phdlMstAmpOsc_ISOSetup_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    uint16_t wTemp;

    /* in the highest range it is not allowed to exceed 12 A/m */
    if (pDataParams->bCurrGainId == PHDL_MSTAMPOSC_ISO_MAX_ID)
    {
        /* set gain value to in the middle of Potirange */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, (pDataParams->wMaxPotiVal - pDataParams->wSNRLimitPotiVal) / 2 + pDataParams->wSNRLimitPotiVal ));
    }
    else
    {
        /* set gain value to Max */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMaxPotiVal));
    }

    /* read field strength */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_GetFieldStrength(pDataParams->pDlOsciDataParams, pDataParams->bChannel, &wTemp));

    pDataParams->wMaxFieldStrength[pDataParams->bCurrGainId] = wTemp;

    /* set gain value to Min */
    PH_CHECK_SUCCESS_FCT(statusTmp,phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wSNRLimitPotiVal));

    /* read field strength */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_GetFieldStrength(pDataParams->pDlOsciDataParams, pDataParams->bChannel, &wTemp ));

    pDataParams->wMinFieldStrength[pDataParams->bCurrGainId] = wTemp;

    /* in the highest range we have to interpolate  max. Field */
    if (pDataParams->bCurrGainId == PHDL_MSTAMPOSC_ISO_MAX_ID)
    {
        /* interpolate to highest Setting */
        pDataParams->wMaxFieldStrength[pDataParams->bCurrGainId] = pDataParams->wMaxFieldStrength[pDataParams->bCurrGainId] * 2 - pDataParams->wMinFieldStrength[pDataParams->bCurrGainId];
    }

    /* if in low field region the lowest part has to be checked as well */
    if (pDataParams->bCurrGainId == PHDL_MSTAMPOSC_ISO_MIN_ID)
    {
        /* set gain value to Min */
        PH_CHECK_SUCCESS_FCT(statusTmp,phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal));

        /* read field strength */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_GetFieldStrength(pDataParams->pDlOsciDataParams, pDataParams->bChannel, &wTemp ));

        pDataParams->wMinFieldStrengthLowField = wTemp;
    }

    /* check if the read field strength is plaussible by checking the upper against the lower value */
    if ( (uint16_t) ((float32_t) pDataParams->wMinFieldStrength[pDataParams->bCurrGainId] * 1.5) > pDataParams->wMaxFieldStrength[pDataParams->bCurrGainId])
    {
        statusTmp = phdlAmp_SetPower(pDataParams->pDlAmpDataParams, PH_OFF);
        statusTmp = phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wMinGainVal);
        statusTmp = phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal);

        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_TOO_HIGH, PH_COMP_DL_MSTAMPOSC);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_ISOSetup_Int_CalcPotiPos(
    phdlMstAmpOsc_ISOSetup_DataParams_t * pDataParams,
    uint16_t wFieldStrength,
    uint16_t * pwPotiPos
    )
{
    phStatus_t statusTmp;
    uint16_t wFieldRange;

    /* check field strength limits, if calc is possible */
    if ( (wFieldStrength < pDataParams->wMinFieldStrength[pDataParams->bCurrGainId] && pDataParams->bCurrGainId != PHDL_MSTAMPOSC_ISO_MIN_ID) || \
        (wFieldStrength < pDataParams->wMinFieldStrengthLowField && pDataParams->bCurrGainId == PHDL_MSTAMPOSC_ISO_MIN_ID))
    {
        /* set gain value to Min */
        statusTmp = phdlAmp_SetPower(pDataParams->pDlAmpDataParams, PH_OFF);
        statusTmp = phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wMinGainVal);
        statusTmp = phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal);

        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_TOO_LOW, PH_COMP_DL_MSTAMPOSC);
    }
    if(wFieldStrength > pDataParams->wMaxFieldStrength[pDataParams->bCurrGainId])
    {
        /* set gain value to Min */
        statusTmp = phdlAmp_SetPower(pDataParams->pDlAmpDataParams, PH_OFF);
        statusTmp = phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wMinGainVal);
        statusTmp = phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal);

        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_TOO_HIGH, PH_COMP_DL_MSTAMPOSC);
    }

    /* check if we are in low field region */
    if (wFieldStrength < pDataParams->wMinFieldStrength[pDataParams->bCurrGainId])
    {
        /* get field range */
        wFieldRange = pDataParams->wMinFieldStrength[pDataParams->bCurrGainId] - pDataParams->wMinFieldStrengthLowField;

        /* calculate potiposition (interpolate) */
        *pwPotiPos = pDataParams->wMinPotiVal+ (uint16_t) ((float32_t)(wFieldStrength - pDataParams->wMinFieldStrengthLowField)/ (float32_t) wFieldRange * (float32_t) (pDataParams->wSNRLimitPotiVal - pDataParams->wMinPotiVal) );

    }
    else
    {
        /* get field range */
        wFieldRange = pDataParams->wMaxFieldStrength[pDataParams->bCurrGainId] - pDataParams->wMinFieldStrength[pDataParams->bCurrGainId];

        /* calculate potiposition (interpolate) */
        *pwPotiPos = pDataParams->wSNRLimitPotiVal + (uint16_t) ((float32_t)(wFieldStrength - pDataParams->wMinFieldStrength[pDataParams->bCurrGainId])/ (float32_t) wFieldRange * (float32_t) (pDataParams->wMaxPotiVal - pDataParams->wSNRLimitPotiVal) );
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}



phStatus_t phdlMstAmpOsc_ISOSetup_Int_SetAmpGainRange(
    phdlMstAmpOsc_ISOSetup_DataParams_t * pDataParams,
    uint16_t wFieldStrength
    )
{
    phStatus_t statusTmp;
    uint16_t wMinFieldOld;
    uint16_t wMaxFieldOld;

    if (wFieldStrength < pDataParams->wMinFieldStrength[pDataParams->bCurrGainId]) /* field strength too high */
    {
        do
        {
            /* store old parameters */
            wMinFieldOld = pDataParams->wMinFieldStrength[pDataParams->bCurrGainId];
            wMaxFieldOld = pDataParams->wMaxFieldStrength[pDataParams->bCurrGainId];

            /* decrease Gain ID */
            if (pDataParams->bCurrGainId > PHDL_MSTAMPOSC_ISO_MIN_ID)
            {
                pDataParams->bCurrGainId--;
            }

            /* check for gap in field strength settings */
            if (wFieldStrength < wMinFieldOld && wFieldStrength > pDataParams->wMaxFieldStrength[pDataParams->bCurrGainId])
            {
                /* set amplification to minimum if there is a gap */
                statusTmp = phdlAmp_SetPower(pDataParams->pDlAmpDataParams, PH_OFF);
                statusTmp = phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wMinGainVal);
                statusTmp = phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal);

                return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_GAP, PH_COMP_DL_MSTAMPOSC);
            }

            /* if we are in Low field region we can leave */
            if (pDataParams->bCurrGainId == PHDL_MSTAMPOSC_ISO_MIN_ID)
            {
                break;
            }

        }while(wFieldStrength < pDataParams->wMinFieldStrength[pDataParams->bCurrGainId]);
    }
    else /* field strength too low */
    {
        do
        {
            /* store old parameters */
            wMinFieldOld = pDataParams->wMinFieldStrength[pDataParams->bCurrGainId];
            wMaxFieldOld = pDataParams->wMaxFieldStrength[pDataParams->bCurrGainId];

            /* increase Gain ID */
            if ((pDataParams->bCurrGainId < PHDL_MSTAMPOSC_ISO_MAX_ID && pDataParams->bUseMaxGain == PH_ON) ||
                (pDataParams->bCurrGainId < (PHDL_MSTAMPOSC_ISO_MAX_ID - 1 ) && pDataParams->bUseMaxGain == PH_OFF))
            {
                pDataParams->bCurrGainId++;
            }

            /* check if out of range */
            if (((pDataParams->bCurrGainId == PHDL_MSTAMPOSC_ISO_MAX_ID && pDataParams->bUseMaxGain == PH_ON) ||
                (pDataParams->bCurrGainId == (PHDL_MSTAMPOSC_ISO_MAX_ID - 1 ) && pDataParams->bUseMaxGain == PH_OFF)) &&
                wFieldStrength > pDataParams->wMaxFieldStrength[pDataParams->bCurrGainId])
            {
                /* set amplification to minimum if there is a gap and switch off amplifier */
                statusTmp = phdlAmp_SetPower(pDataParams->pDlAmpDataParams, PH_OFF);
                statusTmp = phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wMinGainVal);
                statusTmp = phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal);

                /* return error code */
                return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_TOO_HIGH, PH_COMP_DL_MSTAMPOSC);
            }

            /* check for gap in field strength settings */
            if (wFieldStrength < pDataParams->wMinFieldStrength[pDataParams->bCurrGainId] && wFieldStrength > wMaxFieldOld)
            {
                /* set amplification to minimum if there is a gap and switch off amplifier */
                statusTmp = phdlAmp_SetPower(pDataParams->pDlAmpDataParams, PH_OFF);
                statusTmp = phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wMinGainVal);
                statusTmp = phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal);

                return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_GAP, PH_COMP_DL_MSTAMPOSC);
            }
        }while (wFieldStrength > pDataParams->wMaxFieldStrength[pDataParams->bCurrGainId]);
    }

    /* set poti to zero position to minimize thermal stress on equipment */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal));

    /* set gain value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wGainValue[pDataParams->bCurrGainId]));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}


phStatus_t phdlMstAmpOsc_ISOSetup_Int_Cal(
    phdlMstAmpOsc_ISOSetup_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    uint8_t bI;

    for (bI = PHDL_MSTAMPOSC_ISO_MIN_ID; bI <= PHDL_MSTAMPOSC_ISO_MAX_ID; bI++)
    {
        /* if maximum field strength is not enabled to not use it */
        if (pDataParams->bUseMaxGain == 0 && bI == PHDL_MSTAMPOSC_ISO_MAX_ID)
        {
            break;
        }

        /* set Gain ID */
        pDataParams->bCurrGainId = bI;

        /* set Poti val to min to minimize thermal stress on components */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal));

        /* set gain value */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wGainValue[pDataParams->bCurrGainId]));

        /* get range */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISOSetup_Int_CheckFieldRange(pDataParams));
    }

    /* reset to lowest gain */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wGainValue[PHDL_MSTAMPOSC_ISO_MIN_ID]));
    pDataParams->bCurrGainId = PHDL_MSTAMPOSC_ISO_MIN_ID;

    /* set calibrated precision flag */
    pDataParams->bCal = PH_ON;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_ISO_Int_Setup_FindGainValues(
    phdlMstAmpOsc_ISOSetup_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    uint8_t bI;
    uint8_t bCounter;
    uint16_t wStepSize;

    /* calculate stepSize */
    wStepSize = (uint16_t) ((float32_t) (pDataParams->wMaxGainVal - pDataParams->wMinGainVal) / 160);

    /* set Minimum Gain Id */
    pDataParams->bCurrGainId = PHDL_MSTAMPOSC_ISO_MIN_ID;

    /* set Poti val to min to minimize thermal stress on components */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal));

    /* set gain value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wGainValue[pDataParams->bCurrGainId]));

    /* get range */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISOSetup_Int_CheckFieldRange(pDataParams));

    /* find gain for ID1 and ID2 */
    for (bI = PHDL_MSTAMPOSC_ISO_MIN_ID + 1; bI <= PHDL_MSTAMPOSC_ISO_MAX_ID - 1 ; bI++)
    {
        /* set current Gain Id */
        pDataParams->bCurrGainId = bI;

        /* decrease current gain value by 150 to ensure to be below gap */
        pDataParams->wGainValue[pDataParams->bCurrGainId] -= wStepSize*6;

        /* reset counter */
        bCounter = 0;
        do
        {
            /* increase counter */
            bCounter++;

            /* check if got stuck */
            if (bCounter >50)
            {
                /* set amplification to minimum and switch off amplifier */
                statusTmp = phdlAmp_SetPower(pDataParams->pDlAmpDataParams, PH_OFF);
                statusTmp = phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wMinGainVal);
                statusTmp = phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal);

                return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_COULD_NOT_FIND_GAIN_VALS, PH_COMP_DL_MSTAMPOSC);
            }

            /* check if maxGainVal is not exceeded */
            if (pDataParams->wGainValue[pDataParams->bCurrGainId] + wStepSize*2 > pDataParams->wMaxGainVal)
            {
                break;
            }
            /* increase currentgain value by 50 */
            pDataParams->wGainValue[pDataParams->bCurrGainId] += wStepSize*2;

            /* set Poti val to min to minimize thermal stress on components */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal));

            /* set gain value */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wGainValue[pDataParams->bCurrGainId]));

            /* get range */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISOSetup_Int_CheckFieldRange(pDataParams));

            /* do as long as min field strenght is below max fieldstrength from lower Gain Id minus gap */
        }while ((pDataParams->wMinFieldStrength[pDataParams->bCurrGainId] + pDataParams->wMinOverlap) < pDataParams->wMaxFieldStrength[pDataParams->bCurrGainId -1] &&
                 pDataParams->wMaxFieldStrength[pDataParams->bCurrGainId] < pDataParams->wMaxFindFieldStrength);

        /* once passed Gap is too small so Gain Value has to be decreased by 50 */
        pDataParams->wGainValue[pDataParams->bCurrGainId] -= wStepSize * 2;

        /* set Poti val to min to minimize thermal stress on components */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal));

        /* set gain value */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wGainValue[pDataParams->bCurrGainId]));

        /* get range */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISOSetup_Int_CheckFieldRange(pDataParams));
    }

    /* find gain for ID4 (Max) maximum Gain has to be below 13 A/m */
    /* if enabled */
    if (pDataParams->bUseMaxGain == 1)
    {
        /* set maximum gain ID */
        pDataParams->bCurrGainId = PHDL_MSTAMPOSC_ISO_MAX_ID;

        /* decrease current gain value by 150 to ensure to be below gap */
        pDataParams->wGainValue[pDataParams->bCurrGainId] -= wStepSize * 4;

        /* reset counter */
        bCounter = 0;

        do
        {
            /* increase counter */
            bCounter++;

            /* check if got stuck */
            if (bCounter >50)
            {
                /* set amplification to minimum and switch off amplifier */
                statusTmp = phdlAmp_SetPower(pDataParams->pDlAmpDataParams, PH_OFF);
                statusTmp = phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wMinGainVal);
                statusTmp = phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal);

                return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_COULD_NOT_FIND_GAIN_VALS, PH_COMP_DL_MSTAMPOSC);
            }

            /* check if maxGainVal is not exceeded */
            if (pDataParams->wGainValue[pDataParams->bCurrGainId] + wStepSize > pDataParams->wMaxGainVal)
            {
                break;
            }
            /* increase currentgain value by 25 */
            pDataParams->wGainValue[pDataParams->bCurrGainId] = pDataParams->wGainValue[pDataParams->bCurrGainId] + wStepSize;

            /* set Poti val to min to minimize thermal stress on components */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal));

            /* set gain value */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wGainValue[pDataParams->bCurrGainId]));

            /* get range */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISOSetup_Int_CheckFieldRange(pDataParams));

        } while (pDataParams->wMaxFieldStrength[pDataParams->bCurrGainId] < pDataParams->wMaxFindFieldStrength);

        /* once field strenght is too high, so decrease by 25 (to ensure that we can reach 12 A/m) */
        pDataParams->wGainValue[pDataParams->bCurrGainId] = (uint16_t) (pDataParams->wGainValue[pDataParams->bCurrGainId] - wStepSize);

        /* set Poti val to min to minimize thermal stress on components */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal));

        /* set gain value */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wGainValue[pDataParams->bCurrGainId]));

        /* get range */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISOSetup_Int_CheckFieldRange(pDataParams));
    }

    /* reset to lowest gain */
    pDataParams->bCurrGainId = PHDL_MSTAMPOSC_ISO_MIN_ID;
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wGainValue[pDataParams->bCurrGainId]));

    /* set calibrated precision flag (calibration implicitely done) */
    pDataParams->bCal = PH_ON;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

#endif /* NXPBUILD__PHDL_MSTAMPOSC_ISOSETUP */
