/*
 * Copyright 2013, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * ISO Setup Master Amplifier Oscilloscope Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_MSTAMPOSC_ISOSETUP

#include <phdlMstAmpOsc.h>
#include "phdlMstAmpOsc_ISOSetup.h"
#include "phdlMstAmpOsc_ISOSetup_Int.h"

#include <phdlAmp.h>
#include <phdlOsci.h>
#include <phdlRdFpga.h>

phStatus_t phdlMstAmpOsc_ISOSetup_Init(
                                        phdlMstAmpOsc_ISOSetup_DataParams_t * pDataParams,
                                        uint16_t wSizeOfDataParams,
                                        void * pDlAmpDataParams,
                                        void * pDlOsciDataParams,
                                        void * pDlRdFpgaDataParams,
                                        uint8_t bChannel,
                                        uint16_t wFieldRxGainLowerThresh,
                                        uint16_t wFieldRxGainUpperThresh,
                                        uint8_t bRxGainStandard
                                        )
{
    if (sizeof(phdlMstAmpOsc_ISOSetup_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_MSTAMPOSC);
    }

    /* check channel parameter */
    if (bChannel < 1 || bChannel > 4)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
    }

    /* check RxGain Parameters */
    if ((wFieldRxGainLowerThresh < (uint16_t) (0.01*PHDL_MSTAMPOSC_FIELD_STRENGTH_DIVISOR) || wFieldRxGainUpperThresh > (uint16_t) (30*PHDL_MSTAMPOSC_FIELD_STRENGTH_DIVISOR)) ||
        (wFieldRxGainLowerThresh > wFieldRxGainUpperThresh) ||
        (bRxGainStandard < 2 || bRxGainStandard > 7))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
    }

    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pDlAmpDataParams);
    PH_ASSERT_NULL (pDlOsciDataParams);
    PH_ASSERT_NULL (pDlRdFpgaDataParams);

    /* init private data */
    pDataParams->wId                    = PH_COMP_DL_MSTAMPOSC | PHDL_MSTAMPOSC_ISOSETUP_ID;
    pDataParams->pDlAmpDataParams   =	pDlAmpDataParams;
    pDataParams->pDlOsciDataParams  =	pDlOsciDataParams;
    pDataParams->pDlRdFpgaDataParams = pDlRdFpgaDataParams;
    pDataParams->wGainValue[PHDL_MSTAMPOSC_ISO_ID0] = 0;
    pDataParams->wGainValue[PHDL_MSTAMPOSC_ISO_ID1] = 2600;
    pDataParams->wGainValue[PHDL_MSTAMPOSC_ISO_ID2] = 3500;
    pDataParams->wGainValue[PHDL_MSTAMPOSC_ISO_ID3] = 3850;
    pDataParams->bCurrGainId = PHDL_MSTAMPOSC_ISO_MIN_ID;
    pDataParams->bUseMaxGain = PH_OFF;
    pDataParams->bChannel = bChannel;
    pDataParams->bPrecision = 2;
    pDataParams->bCheckPrecision = PH_OFF;
    pDataParams->bCal = PH_OFF;
    pDataParams->wFieldRxGainLowerThresh = wFieldRxGainLowerThresh;
    pDataParams->wFieldRxGainUpperThresh = wFieldRxGainUpperThresh;
    pDataParams->bRxGainStandard = bRxGainStandard;
    pDataParams->bFindGainVals = PH_OFF;
    pDataParams->wMinPotiVal = 0;
    pDataParams->wMaxPotiVal = 1023;
    pDataParams->wSNRLimitPotiVal = 300;
    pDataParams->wMinGainVal = 0;
    pDataParams->wMaxGainVal = 4095;
    pDataParams->wMaxFindFieldStrength = 13000;
    pDataParams->wMinOverlap = 100;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_ISOSetup_InitMstAmpOsc(
    phdlMstAmpOsc_ISOSetup_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    uint16_t wMinPotiVal;

    /* Init Osci */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_InitOsci(pDataParams->pDlOsciDataParams));

    /* read the maximum and minimum settings for the RdFpga poti value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_GetConfig(pDataParams->pDlRdFpgaDataParams, PHDL_RDFPGA_CONFIG_MAX_POTI, &pDataParams->wMaxPotiVal));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_GetConfig(pDataParams->pDlRdFpgaDataParams, PHDL_RDFPGA_CONFIG_SNR_LIMIT_POTI, &pDataParams->wSNRLimitPotiVal));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_GetConfig(pDataParams->pDlRdFpgaDataParams, PHDL_RDFPGA_CONFIG_MIN_POTI, &wMinPotiVal));

    /* do not use the very limit as the SNR is very bad otherwise */
    pDataParams->wMinPotiVal = wMinPotiVal + (uint16_t) ((float32_t) (pDataParams->wMaxPotiVal - wMinPotiVal) * 0.05);

    /* read the maximum and minimum amplifier settings */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_GetConfig(pDataParams->pDlAmpDataParams, PHDL_AMP_CONFIG_MIN_GAIN, &pDataParams->wMinGainVal));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_GetConfig(pDataParams->pDlAmpDataParams, PHDL_AMP_CONFIG_MAX_GAIN, &pDataParams->wMaxGainVal));

    /* set amp gain to minimum */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wGainValue[PHDL_MSTAMPOSC_ISO_ID0]));
    pDataParams->bCurrGainId = PHDL_MSTAMPOSC_ISO_ID0;

    /* switch on amplifier */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetPower(pDataParams->pDlAmpDataParams, PH_ON));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_ISOSetup_SetConfig(
    phdlMstAmpOsc_ISOSetup_DataParams_t * pDataParams,
    uint16_t wIdentifier,
    uint16_t wValue
    )
{
    switch (wIdentifier)
    {
    case   PHDL_MSTAMPOSC_CONFIG_PRECISION:
        /* check value */
        if (wValue > 100)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        /* store value */
        pDataParams->bPrecision = (uint8_t)wValue;
        break;

    case PHDL_MSTAMPOSC_CONFIG_CHECK_PRECISION:
        /* check value */
        if (wValue != PH_ON && wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        /* store value */
        pDataParams->bCheckPrecision = (uint8_t)wValue;
        break;

    case PHDL_MSTAMPOSC_ISOSETUP_CONFIG_USE_MAX_GAIN:
        /* check value */
        if (wValue != PH_ON && wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        /* check if calibration become invalid */
        if (wValue == PH_ON)
        {
            pDataParams->bCal = PH_OFF;
        }
        /* store value */
        pDataParams->bUseMaxGain = (uint8_t)wValue;
        break;

    case PHDL_MSTAMPOSC_ISOSETUP_CONFIG_FIND_GAIN_VALS:
        /* check value */
        if (wValue != PH_ON && wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        /* store value */
        pDataParams->bFindGainVals = (uint8_t)wValue;
        break;

    case  PHDL_MSTAMPOSC_ISOSETUP_CONFIG_GAIN_0:
        /* check value */
        if (wValue < pDataParams->wMinGainVal || wValue > pDataParams->wMaxGainVal)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        /* store value */
        pDataParams->wGainValue[PHDL_MSTAMPOSC_ISO_ID0] = wValue;
        break;

    case  PHDL_MSTAMPOSC_ISOSETUP_CONFIG_GAIN_1:
        /* check value */
        if (wValue < pDataParams->wMinGainVal || wValue > pDataParams->wMaxGainVal)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        /* store value */
        pDataParams->wGainValue[PHDL_MSTAMPOSC_ISO_ID1] = wValue;
        break;

    case  PHDL_MSTAMPOSC_ISOSETUP_CONFIG_GAIN_2:
        /* check value */
        if (wValue < pDataParams->wMinGainVal || wValue > pDataParams->wMaxGainVal)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        /* store value */
        pDataParams->wGainValue[PHDL_MSTAMPOSC_ISO_ID2] = wValue;
        break;

    case  PHDL_MSTAMPOSC_ISOSETUP_CONFIG_GAIN_3:
        /* check value */
        if (wValue < pDataParams->wMinGainVal || wValue > pDataParams->wMaxGainVal)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        /* store value */
        pDataParams->wGainValue[PHDL_MSTAMPOSC_ISO_ID3] = wValue;
        break;

    case PHDL_MSTAMPOSC_ISOSETUP_CONFIG_MAX_FIELD_FIND:
        /* store value */
        pDataParams->wMaxFindFieldStrength = wValue;
        break;

    case PHDL_MSTAMPOSC_ISOSETUP_CONFIG_MIN_FIELD_OVERLAP:
        /* check value */
        if (wValue > 2000)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        pDataParams->wMinOverlap = wValue;
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_ISOSetup_GetConfig(
    phdlMstAmpOsc_ISOSetup_DataParams_t * pDataParams,
    uint16_t wIdentifier,
    uint16_t * pwValue
    )
{
    switch (wIdentifier)
    {
    case PHDL_MSTAMPOSC_CONFIG_PRECISION:
        /* return precision */
        *pwValue = (uint16_t)pDataParams->bPrecision;
        break;

    case PHDL_MSTAMPOSC_CONFIG_CHECK_PRECISION:
        /* return check precision */
        *pwValue = (uint16_t)pDataParams->bCheckPrecision;
        break;

    case PHDL_MSTAMPOSC_CONFIG_MIN_FIELD:
        /* return min field */
        *pwValue = pDataParams->wMinFieldStrengthLowField;
        break;

    case PHDL_MSTAMPOSC_CONFIG_MAX_FIELD:
        /* return max field */
        if (pDataParams->bUseMaxGain == PH_ON)
        {
            *pwValue = pDataParams->wMaxFieldStrength[PHDL_MSTAMPOSC_ISO_ID3];
        }
        else
        {
            *pwValue = pDataParams->wMaxFieldStrength[PHDL_MSTAMPOSC_ISO_ID2];
        }
        break;

    case PHDL_MSTAMPOSC_ISOSETUP_CONFIG_USE_MAX_GAIN:
        /* return use max gain */
        *pwValue = (uint16_t)pDataParams->bUseMaxGain;
        break;

    case PHDL_MSTAMPOSC_ISOSETUP_CONFIG_FIND_GAIN_VALS:
        /* return use find gain vals */
        *pwValue = (uint16_t) pDataParams->bFindGainVals;
        break;

    case  PHDL_MSTAMPOSC_ISOSETUP_CONFIG_GAIN_0:
        /* return use gain 0 */
        *pwValue = pDataParams->wGainValue[PHDL_MSTAMPOSC_ISO_ID0];
        break;

    case  PHDL_MSTAMPOSC_ISOSETUP_CONFIG_GAIN_1:
        /* return use gain 1 */
        *pwValue = pDataParams->wGainValue[PHDL_MSTAMPOSC_ISO_ID1];
        break;

    case  PHDL_MSTAMPOSC_ISOSETUP_CONFIG_GAIN_2:
        /* return use gain 2 */
        *pwValue = pDataParams->wGainValue[PHDL_MSTAMPOSC_ISO_ID2];
        break;

    case  PHDL_MSTAMPOSC_ISOSETUP_CONFIG_GAIN_3:
        /* return use gain 3 */
        *pwValue = pDataParams->wGainValue[PHDL_MSTAMPOSC_ISO_ID3];
        break;

    case PHDL_MSTAMPOSC_ISOSETUP_CONFIG_MAX_FIELD_FIND:
        /* return value */
        *pwValue = pDataParams->wMaxFindFieldStrength;
        break;

    case PHDL_MSTAMPOSC_ISOSETUP_CONFIG_MIN_FIELD_OVERLAP:
        /* return value */
        *pwValue = pDataParams->wMinOverlap;
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        break;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_ISOSetup_SetFieldStrength(
    phdlMstAmpOsc_ISOSetup_DataParams_t * pDataParams,
    uint16_t wFieldStrength
    )
{
    phStatus_t statusTmp;
    uint16_t wPotiPos;
    uint16_t wFieldStrengthMeas;
    uint16_t wPrecsionsLimitLower = wFieldStrength - (uint16_t)((float32_t)wFieldStrength * (float32_t)pDataParams->bPrecision / 100 );
    uint16_t wPrecsionsLimitUpper = wFieldStrength + (uint16_t)((float32_t)wFieldStrength * (float32_t)pDataParams->bPrecision / 100 );
    uint8_t bCounter = 0;

    /* check if calibrated */
    if ( pDataParams->bCal == PH_OFF )
    {

        /* set amplification to minimum */
        statusTmp = phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wMinGainVal);
        statusTmp = phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal);

        /* return error code */
        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_NOT_CAL, PH_COMP_DL_MSTAMPOSC);
    }

    do
    {
        /* check for low Field region boundaries */
        if (wFieldStrength < pDataParams->wMinFieldStrengthLowField &&
            pDataParams->bCurrGainId == PHDL_MSTAMPOSC_ISO_MIN_ID)
        {
            /* set amplification to minimum and switch off amplifier */
            statusTmp = phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wMinGainVal);
            statusTmp = phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal);

            /* return error code */
            return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_TOO_LOW, PH_COMP_DL_MSTAMPOSC);
        }

        /* check field strength limits, if amplifier Gain has to be changed */
        if (wFieldStrength < pDataParams->wMinFieldStrength[pDataParams->bCurrGainId] ||
            wFieldStrength > pDataParams->wMaxFieldStrength[pDataParams->bCurrGainId])
        {
            /* find right amplifier gain */
            PH_CHECK_SUCCESS_FCT(statusTmp,	phdlMstAmpOsc_ISOSetup_Int_SetAmpGainRange(pDataParams, wFieldStrength));
        }

        /* calculate poti position */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISOSetup_Int_CalcPotiPos(pDataParams, wFieldStrength, &wPotiPos));

        /* set poti position */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, wPotiPos));

        /* check precision */
        if (pDataParams->bCheckPrecision ==1)
        {
            /* read field strength */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_GetFieldStrength(pDataParams->pDlOsciDataParams, pDataParams->bChannel, &wFieldStrengthMeas));

            /* check if within the precision limits */
            if (wFieldStrengthMeas < wPrecsionsLimitLower || wFieldStrengthMeas > wPrecsionsLimitUpper)
            {
                /* if we tried to recalibrate 3 times we assume it is not possible */
                if (bCounter >= PHDL_MSTAMPOSC_NUM_OF_RECAL)
                {
                    /* set amplification to minimum if there is a gap and switch off amplifier */
                    statusTmp = phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wMinGainVal);
                    statusTmp = phdlRdFpga_SetPotiPos(pDataParams->pDlRdFpgaDataParams, pDataParams->wMinPotiVal);

                    /* allready tried to recalibrate, precision not achieveable */
                    return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_PRECISION_NOT_ACHIEVEABLE, PH_COMP_DL_MSTAMPOSC);
                }
                /* try to recalibrate, assume thermal drift */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISOSetup_Int_CheckFieldRange(pDataParams));
                /* increase counter */
                bCounter++;
            }
            else
            {
                /* everthing is fine, we can leave */
                break;
            }
        }
    }
    while (pDataParams->bCheckPrecision == 1);

    /* set joiner RX gain */
    if (wFieldStrength <= pDataParams->wFieldRxGainLowerThresh)
    {
        if (pDataParams->bRxGainStandard <= 2)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_SetJoinerRxGain(pDataParams->pDlRdFpgaDataParams,(uint8_t) pDataParams->bRxGainStandard));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_SetJoinerRxGain(pDataParams->pDlRdFpgaDataParams, (uint8_t) (pDataParams->bRxGainStandard -1)));
        }
    }
    else if (wFieldStrength <= pDataParams->wFieldRxGainUpperThresh)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_SetJoinerRxGain(pDataParams->pDlRdFpgaDataParams,(uint8_t) pDataParams->bRxGainStandard));
    }
    else
    {
        if (pDataParams->bRxGainStandard >= 7)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_SetJoinerRxGain(pDataParams->pDlRdFpgaDataParams,(uint8_t) pDataParams->bRxGainStandard));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_SetJoinerRxGain(pDataParams->pDlRdFpgaDataParams,(uint8_t) (pDataParams->bRxGainStandard +1)));
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_ISOSetup_Cal(
                                       phdlMstAmpOsc_ISOSetup_DataParams_t * pDataParams
                                       )
{
    if ( pDataParams->bFindGainVals == PH_ON )
    {
        return phdlMstAmpOsc_ISO_Int_Setup_FindGainValues(pDataParams);
    }
    else
    {
        return phdlMstAmpOsc_ISOSetup_Int_Cal(pDataParams);
    }
}

#endif /* NXPBUILD__PHDL_MSTAMPOSC_ISOSETUP */
