/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
* Internal FPGA BOX BAL Component of Reader Library Framework.
* $Author: Rajendran Kumar (nxp99556) $
* $Revision: 7467 $
* $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
*/

#include <ph_Status.h>
#include <phbalReg.h>

#ifdef NXPBUILD__PHBAL_REG_RDSCR01

#include "phbalReg_RdScr01.h"
#include "phbalReg_RdScr01_Int.h"
#include "external\phbalReg_RdScr01_Ex.h"

phStatus_t phbalReg_RdScr01_Init(
                                 phbalReg_RdScr01_DataParams_t * pDataParams,
                                 uint16_t wSizeOfDataParams,
                                 uint32_t dwInitVccmv,
                                 uint32_t dwInitClockFrequencykHz
                                 )
{
    if (sizeof(phbalReg_RdScr01_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_BAL);
    }

    /* initialize the data parameters to default values */
    pDataParams->wId = PH_COMP_BAL | PHBAL_REG_RDSCR01_ID;
    pDataParams->dwBitRate = PHBAL_REG_RDSCR01_BITRATE_DEFAULT;
    pDataParams->dwTimeoutValue = PHBAL_REG_RDSCR01_TIMEOUT_DEFAULT;
    pDataParams->dwInitVccmv = dwInitVccmv;
    pDataParams->dwInitClockFrequencykHz = dwInitClockFrequencykHz;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}


phStatus_t phbalReg_RdScr01_GetPortList(
                                        phbalReg_RdScr01_DataParams_t * pDataParams,
                                        uint16_t wPortBufSize,
                                        uint8_t * pPortNames,
                                        uint16_t * pNumOfPorts
                                        )
{
    uint8_t abPortName[] = "Scr01Dll USB";
    uint16_t wLen;
    /* Backup old values */
    uint32_t dwtmpInitVccmv = pDataParams->dwInitVccmv;
    uint32_t dwtmpInitClockFrequencykHz = pDataParams->dwInitClockFrequencykHz;

    /* Set temp values */
    pDataParams->dwInitVccmv = 0;
    pDataParams->dwInitClockFrequencykHz = 5000;

    *pNumOfPorts = 0;

    /* Try to open port if successful return reader */
    if(phbalReg_OpenPort(pDataParams) == PH_ERR_SUCCESS)
    {
        /* restore values */
        pDataParams->dwInitVccmv = dwtmpInitVccmv;
        pDataParams->dwInitClockFrequencykHz = dwtmpInitClockFrequencykHz;

        phbalReg_ClosePort(pDataParams);
        if (wPortBufSize)
        {
            wLen = (uint16_t)strlen( (const char *) abPortName);
            memcpy(pPortNames, abPortName, wLen); /* PRQA S 3200 */
            *pNumOfPorts = 1;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
        }
    }
    /* restore values */
    pDataParams->dwInitVccmv = dwtmpInitVccmv;
    pDataParams->dwInitClockFrequencykHz = dwtmpInitClockFrequencykHz;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}


phStatus_t phbalReg_RdScr01_SetPort(
                                    phbalReg_RdScr01_DataParams_t * pDataParams,
                                    uint8_t * pPortName
                                    )
{
    if(pDataParams || pPortName);
    /* No port needed for SCR01 therefore just return SUCCESS */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_RdScr01_OpenPort(
                                     phbalReg_RdScr01_DataParams_t * pDataParams
                                     )
{
    phStatus_t statusTmp;
    float64_t vcc = (float64_t)(pDataParams->dwInitVccmv / 1000.0);
    float64_t clk = (float64_t)(pDataParams->dwInitClockFrequencykHz / 1000.0);

    /* load the function pointers */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_RdScr01_Ex_LoadFunctions(pDataParams));
    if(statusTmp != PH_ERR_SUCCESS)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_RdScr01_Ex_UnLoadFunctions(pDataParams));
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_RdScr01_Int_ConvertError(pDataParams->pfInitVF(vcc, clk)));
    if(statusTmp != PH_ERR_SUCCESS)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_RdScr01_Ex_UnLoadFunctions(pDataParams));
    }

#if 1
    /* WORKAROUND_SLOW_FPGA */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_RdScr01_Int_ConvertError(pDataParams->pf7816ModifyProtocol(ISO7816_PROTOCOL_ATR_WAIT_CLOCKS, 100000)));
    if(statusTmp != PH_ERR_SUCCESS)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_RdScr01_Ex_UnLoadFunctions(pDataParams));
    }
#endif

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}


phStatus_t phbalReg_RdScr01_ClosePort(
                                      phbalReg_RdScr01_DataParams_t * pDataParams
                                      )
{
    /* Close and Unload Function Pointers */
    if(pDataParams->pfClose != NULL)
    {
        pDataParams->pfClose();
    }

    phbalReg_RdScr01_Ex_UnLoadFunctions(pDataParams);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_RdScr01_Exchange(
                                     phbalReg_RdScr01_DataParams_t * pDataParams,
                                     uint16_t wOption,
                                     uint8_t * pTxBuffer,
                                     uint16_t wTxLength,
                                     uint16_t wRxBufSize,
                                     uint8_t * pRxBuffer,
                                     uint16_t * pRxLength
                                     )
{
    uint8_t    buffer[1024];
    uint32_t   size;
    phStatus_t statusTmp;

    /* Check options */
    if (wOption != PH_EXCHANGE_DEFAULT)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    if(wTxLength >= 1024)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
    }

    size = wTxLength;
    memcpy(buffer, pTxBuffer, wTxLength);

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_RdScr01_Int_ConvertError(pDataParams->pfRawExchange(buffer, &size, pDataParams->dwTimeoutValue)));

    *pRxLength = size & 0xFFFF;

    if (*pRxLength > 0)
    {
        if(size <= wRxBufSize)
        {
            memcpy(pRxBuffer, buffer, *pRxLength);
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_BAL);
    }
}


phStatus_t phbalReg_RdScr01_SetConfig(
                                      phbalReg_RdScr01_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint16_t wValue
                                      )
{
    phStatus_t statusTmp;

    switch(wConfig)
    {
        case PHBAL_REG_CONFIG_WRITE_TIMEOUT_MS:
        case PHBAL_REG_CONFIG_READ_TIMEOUT_MS:
            pDataParams->dwTimeoutValue = wValue;
            break;

        case PHBAL_REG_RDSCR01_CONFIG_BITRATE:
            pDataParams->dwBitRate = wValue;
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_RdScr01_Int_ConvertError(phbalReg_RdScr01_Int_SetBitRate(pDataParams, pDataParams->dwBitRate)));
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_RdScr01_GetConfig(
    phbalReg_RdScr01_DataParams_t * pDataParams,
    uint16_t wConfig,
    uint16_t * pValue )
{
    switch( wConfig )
    {
        case PHBAL_REG_CONFIG_WRITE_TIMEOUT_MS:
        case PHBAL_REG_CONFIG_READ_TIMEOUT_MS:
            *pValue = (uint16_t)pDataParams->dwTimeoutValue;
            break;

        case PHBAL_REG_RDSCR01_CONFIG_BITRATE:
            *pValue = (uint16_t)pDataParams->dwBitRate;
            break;

        default:
            *pValue = 0;
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

#endif /* NXPBUILD__PHBAL_REG_RDSCR01 */
