/*
 * Copyright 2017, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * MP300 USB BAL Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHBAL_REG_MP300USB

#include "phbalReg_Mp300Usb.h"
#include "phbalReg_Mp300Usb_If.h"
#include "external/lusb0_usb.h"
#include "external/phbalReg_Mp300Usb_Ex.h"

phStatus_t phbalReg_Mp300Usb_Init(phbalReg_Mp300Usb_DataParams_t * pDataParams,
                                    uint16_t wSizeOfDataParams)
{
    if (sizeof(phbalReg_Mp300Usb_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_BAL);
    }
    PH_ASSERT_NULL (pDataParams);

    /* initialize the data parameters to default values */
    pDataParams->wId = PH_COMP_BAL | PHBAL_REG_MP300USB_ID;
    pDataParams->wExchangeDelayCount = 0xFFFF;
    pDataParams->wTxTimeout = DEFAULT_TIMEOUT_IN_MILLISECONDS;
    pDataParams->wRxTimeout = DEFAULT_TIMEOUT_IN_MILLISECONDS;
    pDataParams->pPortName = NULL;
    pDataParams->pUsbDevHandle = INVALID_HANDLE_VALUE;
    pDataParams->bInterfaceNumber = INVALID_USB_HANDLE;
    pDataParams->bUsbWriteEndpointAddress = INVALID_USB_HANDLE;
    pDataParams->bUsbReadEndpointAddress = INVALID_USB_HANDLE;
    pDataParams->wVendorIdFilter = 0;
    pDataParams->wProductIdFilter = 0;
    pDataParams->bExchangeMode = PHBAL_REG_MP300USB_VALUE_EXCHANGEMODE_INTERRUPT;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_Mp300Usb_GetPortList(
    phbalReg_Mp300Usb_DataParams_t * pDataParams,
    uint16_t wPortBufSize,
    uint8_t * pPortNames,
    uint16_t * pNumOfPorts
    )
{
#ifdef WIN64_BUILD
	/* This is a macro Define for X64 build only
	 * Dummy return since usb_XXX_XX interfaces throw Reference not found error in case of WIN64 Build
	 */
	/* satisfy compiler */
    if (pDataParams || wPortBufSize || pPortNames || pNumOfPorts);
		return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#else
    struct usb_bus    PH_MEMLOC_COUNT * pUsbBus = NULL;
    struct usb_device PH_MEMLOC_COUNT * pUsbDev = NULL;
    usb_dev_handle    PH_MEMLOC_REM * pUsbDevHandle = NULL;
    uint16_t          PH_MEMLOC_REM wBufferPos = 0;
    phStatus_t        PH_MEMLOC_REM statusTmp;
    int32_t           PH_MEMLOC_REM dwError = 0;
    uint8_t           PH_MEMLOC_REM bDllLoaded = 0;

    /* Parameter Check */
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pPortNames);
    PH_ASSERT_NULL (pNumOfPorts);

    if (wPortBufSize == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
    }

    /* load the function pointers */
    if(pDataParams->pLibraryHandleLibUsb == NULL)
    {
        PH_CHECK_SUCCESS_FCT_CLEANUP(statusTmp, phbalReg_Mp300Usb_Ex_LoadFunctions(pDataParams), phbalReg_Mp300Usb_Ex_UnLoadFunctions(pDataParams));
        bDllLoaded = 1;
    }

    /* Reset number of found ports */
    *pNumOfPorts = 0;

    /* Init Usb and find devices */
    pDataParams->pfInit();
    dwError = pDataParams->pfFindBusses(); /* find all busses */
    if (dwError < 0)
    {
        if(bDllLoaded)
        {
            phbalReg_Mp300Usb_Ex_UnLoadFunctions(pDataParams);
        }
        return phbalReg_Mp300Usb_If_ConvertError(dwError);
    }
    dwError = pDataParams->pfFindDevices(); /* find all connected devices */
    if (dwError < 0)
    {
        if(bDllLoaded)
        {
            phbalReg_Mp300Usb_Ex_UnLoadFunctions(pDataParams);
        }
        return phbalReg_Mp300Usb_If_ConvertError(dwError);
    }

    /* Go through all devices found */
    for (pUsbBus = (struct usb_bus *)pDataParams->pfGetBusses(); pUsbBus; pUsbBus = pUsbBus->next)
    {
        for (pUsbDev = pUsbBus->devices; pUsbDev; pUsbDev = pUsbDev->next)
        {
            /* If filter is set check it */
            if (pDataParams->wVendorIdFilter != 0 || pDataParams->wProductIdFilter != 0)
            {
                if (pUsbDev->descriptor.idVendor != pDataParams->wVendorIdFilter ||
                    pUsbDev->descriptor.idProduct != pDataParams->wProductIdFilter)
                {
                    continue;
                }
            }

            /* Try to open handle */
            pUsbDevHandle = (usb_dev_handle *)pDataParams->pfOpen(pUsbDev);
            if (pUsbDevHandle > 0)
            {
                /* Get name of sucessfull opened device and add name to available usb device list */
                statusTmp = phbalReg_Mp300Usb_If_GetDeviceName(pDataParams, pUsbDev, pUsbDevHandle, &pPortNames[wBufferPos], wPortBufSize-wBufferPos);

                if(statusTmp != PH_ERR_SUCCESS)
                {
                    if(bDllLoaded)
                    {
                        phbalReg_Mp300Usb_Ex_UnLoadFunctions(pDataParams);
                    }
                    return statusTmp;
                }

                wBufferPos += (uint16_t)strlen((char*)&pPortNames[wBufferPos]) + 1;
                if (*pNumOfPorts < 65535)
                {
                    ++(*pNumOfPorts);
                }
                else
                {
                    if(bDllLoaded)
                    {
                        phbalReg_Mp300Usb_Ex_UnLoadFunctions(pDataParams);
                    }
                    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
                }

                dwError = pDataParams->pfClose(pUsbDevHandle);
                if (dwError != 0)
                {
                    if(bDllLoaded)
                    {
                        phbalReg_Mp300Usb_Ex_UnLoadFunctions(pDataParams);
                    }
                    return phbalReg_Mp300Usb_If_ConvertError(dwError);
                }
            }
        }
    }

    if(bDllLoaded)
    {
        PH_CHECK_SUCCESS(phbalReg_Mp300Usb_Ex_UnLoadFunctions(pDataParams));
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_Mp300Usb_SetPort(
                                      phbalReg_Mp300Usb_DataParams_t * pDataParams,
                                      uint8_t * pPortName
                                      )
{
#ifdef _WIN32
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pPortName);

    pDataParams->pPortName = pPortName;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams || pPortName);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_Mp300Usb_OpenPort(phbalReg_Mp300Usb_DataParams_t * pDataParams)
{
    phStatus_t statusTmp;
    PH_ASSERT_NULL (pDataParams);

    /* load the function pointers */
    if(pDataParams->pLibraryHandleLibUsb == NULL)
    {
        statusTmp = phbalReg_Mp300Usb_Ex_LoadFunctions(pDataParams);
        if(statusTmp != PH_ERR_SUCCESS)
        {
            phbalReg_Mp300Usb_Ex_UnLoadFunctions(pDataParams);
            return PH_ADD_COMPCODE(statusTmp, PH_COMP_HAL);
        }
    }

    return phbalReg_Mp300Usb_If_Open(pDataParams);
}

phStatus_t phbalReg_Mp300Usb_ClosePort(phbalReg_Mp300Usb_DataParams_t * pDataParams)
{
    phStatus_t statusTmp;
    PH_ASSERT_NULL (pDataParams);
    statusTmp = phbalReg_Mp300Usb_If_Close(pDataParams);

    if(statusTmp != PH_ERR_SUCCESS)
    {
        phbalReg_Mp300Usb_Ex_UnLoadFunctions(pDataParams);
        return PH_ADD_COMPCODE(statusTmp, PH_COMP_HAL);
    }
    else
    {
        return phbalReg_Mp300Usb_Ex_UnLoadFunctions(pDataParams);
    }
}

phStatus_t phbalReg_Mp300Usb_Exchange(phbalReg_Mp300Usb_DataParams_t * pDataParams,
                                       uint16_t wOption,
                                       uint8_t * pTxBuffer,
                                       uint16_t wTxLength,
                                       uint16_t wRxBufSize,
                                       uint8_t * pRxBuffer,
                                       uint16_t * pRxLength)
{
    PH_ASSERT_NULL (pDataParams);

    /* Check options */
    if (wOption != PH_EXCHANGE_DEFAULT)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }
    return phbalReg_Mp300Usb_If_Exchange(pDataParams, pTxBuffer, wTxLength, wRxBufSize, pRxBuffer, pRxLength);
}

phStatus_t phbalReg_Mp300Usb_SetConfig(phbalReg_Mp300Usb_DataParams_t * pDataParams,
                                        uint16_t wConfig,
                                        uint16_t wValue)
{
    PH_ASSERT_NULL (pDataParams);

    switch (wConfig)
    {
    case PHBAL_REG_CONFIG_WRITE_TIMEOUT_MS:
        pDataParams->wTxTimeout = wValue;
        break;

    case PHBAL_REG_CONFIG_READ_TIMEOUT_MS:
        pDataParams->wRxTimeout = wValue;
        break;

    case PHBAL_REG_MP300USB_CONFIG_EXCHANGE_DELAY_COUNT:
        pDataParams->wExchangeDelayCount = wValue;
        break;

    case PHBAL_REG_MP300USB_CONFIG_VENDOR_ID_FILTER:
        pDataParams->wVendorIdFilter = wValue;
        break;

    case PHBAL_REG_MP300USB_CONFIG_PRODUCT_ID_FILTER:
        pDataParams->wProductIdFilter = wValue;
        break;

    case PHBAL_REG_MP300USB_CONFIG_EXCHANGE_MODE:
        switch (wValue)
        {
        case PHBAL_REG_MP300USB_VALUE_EXCHANGEMODE_INTERRUPT:
        case PHBAL_REG_MP300USB_VALUE_EXCHANGEMODE_BULK:
            pDataParams->bExchangeMode = (uint8_t)wValue;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
        }
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_Mp300Usb_GetConfig( phbalReg_Mp300Usb_DataParams_t * pDataParams,
                                        uint16_t wConfig,
                                        uint16_t * pValue
                                        )
{
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pValue);

    switch (wConfig)
    {
    case PHBAL_REG_CONFIG_WRITE_TIMEOUT_MS:
        *pValue = pDataParams->wTxTimeout;
        break;

    case PHBAL_REG_CONFIG_READ_TIMEOUT_MS:
        *pValue = pDataParams->wRxTimeout;
        break;

    case PHBAL_REG_MP300USB_CONFIG_EXCHANGE_DELAY_COUNT:
        *pValue = pDataParams->wExchangeDelayCount;
        break;

    case PHBAL_REG_MP300USB_CONFIG_VENDOR_ID_FILTER:
        *pValue = pDataParams->wVendorIdFilter;
        break;

    case PHBAL_REG_MP300USB_CONFIG_PRODUCT_ID_FILTER:
        *pValue = pDataParams->wProductIdFilter;
        break;

    case PHBAL_REG_MP300USB_CONFIG_EXCHANGE_MODE:
        *pValue = pDataParams->bExchangeMode;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}
#endif /* NXPBUILD__PHBAL_REG_MP300USB */
