/*
 * Copyright 2013, 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal definitions for Software ISO7816-4 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHALP40CMDPUB_SW_INT_H
#define PHALP40CMDPUB_SW_INT_H

#include <ph_Status.h>
#include <phalP40CmdPub.h>

/** \addtogroup ph_Private
* @{
*/

/** \name 7816p4 Exchange Options
** \c bit definitions must conform to PH_EXCHANGE_CUSTOM_BITS_MASK field 0x00F0U
Assists in interpretation of the Le field value.
*/
/*@{*/
#define PHALP40CMDPUB_I7816P4_CLASS_CHAINING_BIT  0x10U
/**
Specify use of extended length mode.
Use when Le Value is zero but expected length is 64k
*/
#define PHALP40CMDPUB_I7816P4_EXCHANGE_EXTENDED_LENGTH   0x0010U

/**
Interpret Le = 0 as MAX value dependant on PHALP40CMDPUB_I7816P4_EXCHANGE_EXTENDED_LENGTH flag
Only checked on first call to exchange  i.e PH_EXCHANGE_LEAVE_BUFFER_BIT not set
*/
#define PHALP40CMDPUB_I7816P4_EXCHANGE_LE_MAX				0x0020U
/**
composite flag
Specify 256 bytes expected response length
Only checked on first call to exchange  i.e PH_EXCHANGE_LEAVE_BUFFER_BIT not set
Le value must be set to zero
Aternatively, set Le to 256 directly and do not use this flag
*/
#define PHALP40CMDPUB_I7816P4_EXCHANGE_LE_256    PHALP40CMDPUB_I7816P4_EXCHANGE_LE_MAX

/**
composite flag
Specify 64k bytes expected response length
Only checked on first call to exchange  i.e PH_EXCHANGE_LEAVE_BUFFER_BIT not set
Le value must be set to zero
*/
#define PHALP40CMDPUB_I7816P4_EXCHANGE_LE_64K		(PHALP40CMDPUB_I7816P4_EXCHANGE_LE_MAX |  \
    PHALP40CMDPUB_I7816P4_EXCHANGE_EXTENDED_LENGTH)
/*@}*/

/** \name defines used for phalCmdPub_Sw_CurrentApdu_t.bFlags
*/
/*@{*/
#define PHPALCMDPUB_APDUFLAG_DEFAULT               0x00U  /**< Reset value of the flag byte. */
#define PHPALCMDPUB_APDUFLAG_LC_PRESENT        0x01U  /**< Set if the Lc field has been filled. */
#define PHPALCMDPUB_APDUFLAG_USE_EXTENDED  0x02U  /**< Set if either Lc field or Le field requires it. */
#define PHPALCMDPUB_APDUFLAG_MAX_LE                  0x04U /**< Set if  Le= 0 field specifies max value instead of zero. */
/*@}*/

/**
* \brief clears data in Current command buffer
* \return
*/
void phalP40CmdPub_Sw_Int_Reset_Current_Cmd (
    phalP40CmdPub_Sw_CurrentApdu_t * pCurrentCmd	    /**< [In]  Pointer to Current command saved structure. */
    );

/**
* \brief Formats the Lc bytes of a 7816 command
* \return number of bytes written at Dest
*/
uint8_t phalP40CmdPub_Sw_Int_EncodeLc (
                                       phalP40CmdPub_Sw_CurrentApdu_t * pCurrentCmd,   /**< [In]  Pointer to Current command saved structure. */
                                       uint8_t * pDest  /**< [In]  Location to write the encoded Lc bytes. up to 3 bytes */
                                       );

/**
* \brief Encodes the Le bytes of a 7816 command
* \return number of bytes written at Dest
*/
uint8_t phalP40CmdPub_Sw_Int_EncodeLe (
                                       phalP40CmdPub_Sw_CurrentApdu_t * pCurrentCmd,   /**< [In]  Pointer to Current command saved structure. */
                                       uint8_t*  pDest  /**< [Out] location to write the encoded Le bytes. */
                                       );

/**
* \brief Specifies if either LC or LE need to be in extended mode. Stores Lc and Le in raw form
* \ must be called prior to either EncodeLc or EncodeLe
* \effect sets PHPALCMDPUB_APDUFLAG_USE_EXTENDED bit in phalP40CmdPub_Sw_CurrentApdu_t.bFlags accordingly
*/
phStatus_t phalP40CmdPub_Sw_Int_CheckForExtendedLength (
            phalP40CmdPub_Sw_CurrentApdu_t * pCurrentCmd,  /**< [In]  Pointer to Current command saved structure. */
            uint16_t wOptions,  /**< [In] Options flag including formatting information. */
            uint16_t wLc,    /**< [In] command length . */
            uint16_t wLe    /**< [In] Expected response length. */
            );

/**
* \brief Specifies if either LC or LE need to be in extended mode.
* Stores Lc and Le in raw form.
* Must be called prior to either EncodeLc or EncodeLe.
* Effect sets PHPALCMDPUB_APDUFLAG_USE_EXTENDED bit in phalP40CmdPub_Sw_CurrentApdu_t.bFlags accordingly.
*/
phStatus_t phalP40CmdPub_Sw_Int_Exchange(
    phalP40CmdPub_Sw_DataParams_t * pDataParams,   /**< [In] Pointer to this layers parameter structure. */
    uint16_t wOption,       /**< [In] Exchange options parameter. */
    uint8_t  bUseL3,         /**< [In] non zero if level 3 is used. */
    uint8_t bCla,                /**< [In] Command class byte. */
    uint8_t bIns,	                /**< [In] Command Instruction. */
    uint8_t bP1,                 /**< [In] Command first parameter. */
    uint8_t bP2,                 /**< [In] Command second parameter. */
    uint16_t wLc,               /**< [In] Command length parameter. */
    uint16_t wLe,               /**< [In] Expected response length parameter. */
    uint8_t * pTxBuffer,     /**< [In] Data field to transmit. */
    uint16_t wTxLength,   /**< [In] Number of Data bytes used in pTxBuffer. */
    uint8_t ** ppRxBuffer, /**< [Out] Response data storage. */
    uint16_t * pRxLength /**< [Out] Size of response data field (MAX return bytes expected). */
    );
/**
* \brief encapsulation layer for transport.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other predefined or user defined status code, depending on implementation .
*/
phStatus_t phalP40CmdPub_Sw_Int_ExchangeRaw(
    phalP40CmdPub_Sw_DataParams_t * pDataParams, /**< [In] Pointer to this layers parameter structure. */
    uint16_t wOption,   /**< [In] Exchange options parameter. */
    uint8_t   bUseL3,	    /**< [In] non zero if level 3 is used. */
    uint8_t * pTxBuffer, /**< [In] Data field to send. */
    uint16_t wTxLength,   /**< [In] Number of Data bytes used in pTxBuffer. */
    uint8_t ** ppRxBuffer,   /**< [Out] Response data storage. */
    uint16_t * pRxLength  /**< [Out] Size of response data field (MAX return bytes expected). */
    );

phStatus_t phalP40CmdPub_Sw_Int_CheckReturnResult (
    phalP40CmdPub_Sw_DataParams_t *  pDataParams, /**< [In] Pointer to this layers parameter structure. */
    phStatus_t status     /**< [In] result of last executed exchange. */
    );

uint8_t phalP40CmdPub_Sw_Int_LastSW1Result(
    phalP40CmdPub_Sw_DataParams_t * pDataParams	     /**< [In] Pointer to this layers parameter structure. */
    );

uint8_t phalP40CmdPub_Sw_Int_LastSW2Result(
    phalP40CmdPub_Sw_DataParams_t * pDataParams	     /**< [In] Pointer to this layers parameter structure. */
    );

phStatus_t phalP40CmdPub_Sw_Int_Download (
    phalP40CmdPub_Sw_DataParams_t *  pDataParams,
    uint8_t bCmd,
    uint16_t wStartAddr,
    uint8_t* pData,
    uint16_t wDataLength);

void phalP40CmdPub_Sw_Int_ResetAuthStatus(phalP40CmdPub_Sw_DataParams_t * pDataParams);

/** @}
* end of ph_Private
*/

#endif /* PHALP40CMDPUB_SW_INT_H */
