/*
 * Copyright 2016 - 2017, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal functions of Software implementation of MIFARE PRIME NTAG application layer.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHALMFPRIMENTAG_INT_H
#define PHALMFPRIMENTAG_INT_H

#include <ph_Status.h>
#include <phalMfprimeNtag.h>

/** \addtogroup ph_Private
* @{
*/

/**
* \name MIFARE NTAG 413 Native Commands
*/

/** @{ */

#define PHAL_MFPRIMENTAG_CMD_AUTHENTICATE_EV2_FIRST        0x71    /**< MFPRIMENTAG Authenticate Ev2 First command. */
#define PHAL_MFPRIMENTAG_CMD_AUTHENTICATE_EV2_NON_FIRST    0x77    /**< MFPRIMENTAG Authenticate Ev2 Non First command. */
#define PHAL_MFPRIMENTAG_CMD_CHANGE_KEY                    0xC4    /**< MFPRIMENTAG Change key cmd. */
#define PHAL_MFPRIMENTAG_CMD_GET_KEY_VERSION               0x64    /**< MFPRIMENTAG Get Key Version. */
#define PHAL_MFPRIMENTAG_CMD_GET_VERSION                   0x60    /**< MFPRIMENTAG Get Version cmd. */
#define PHAL_MFPRIMENTAG_CMD_SET_CONFIG                    0x5C    /**< MFPRIMENTAG Set Configuration Cmd. */
#define PHAL_MFPRIMENTAG_CMD_GET_FILE_SETTINGS             0xF5    /**< MFPRIMENTAG Get File settings cmd. */
#define PHAL_MFPRIMENTAG_CMD_GET_SDM_FILE_COUNTER          0xF6    /**< MFPRIMENTAG Get SDM Counter cmd. */
#define PHAL_MFPRIMENTAG_CMD_CHANGE_FILE_SETTINGS          0x5F    /**< MFPRIMENTAG Change file settings cmd. */
#define PHAL_MFPRIMENTAG_CMD_READ_DATA                     0xBD    /**< MFPRIMENTAG Read Data cmd. */
#define PHAL_MFPRIMENTAG_CMD_READ_DATA_ISO                 0xAD    /**< MFPRIMENTAG Read Data cmd using ISO chaining. */
#define PHAL_MFPRIMENTAG_CMD_WRITE_DATA                    0x3D    /**< MFPRIMENTAG Write data cmd. */
#define PHAL_MFPRIMENTAG_CMD_WRITE_DATA_ISO                0x8D    /**< MFPRIMENTAG Write data cmd using ISO chaining. */
#define PHAL_MFPRIMENTAG_CMD_READ_SIG					   0x3C	   /**< MFPRIMENTAG Verify read signature command. */

/** @} */

/**
* \name MIFARE NTAG 413 Response Codes
*/

/** @{ */

#define PHAL_MFPRIMENTAG_RESP_OPERATION_OK                 0x00U    /**< MFPRIMENTAG Response - Successful operation. */
#define PHAL_MFPRIMENTAG_RESP_OK			               0x90U    /**< MFPRIMENTAG Response - Successful operation. */
#define PHAL_MFPRIMENTAG_RESP_NO_CHANGES                   0x0CU    /**< MFPRIMENTAG Response - No changes done to backup files. */
#define PHAL_MFPRIMENTAG_RESP_ERR_NOT_SUP				   0x0DU     /*< MFPRIMENTAG Not Supported Error. */
#define PHAL_MFPRIMENTAG_RESP_ILLEGAL_COMMAND_CODE         0x1CU    /**< MFPRIMENTAG command code not supported. */
#define PHAL_MFPRIMENTAG_RESP_ERR_INTEGRITY_ERROR          0x1EU    /**< MFPRIMENTAG CRC or MAC does not match data padding bytes not valid. */
#define PHAL_MFPRIMENTAG_RESP_NO_SUCH_KEY                  0x40U    /**< MFPRIMENTAG Invalid key number specified. */
#define PHAL_MFPRIMENTAG_RESP_CHAINING                     0x71U    /**< MFPRIMENTAG ISO Chaining Status. */
#define PHAL_MFPRIMENTAG_RESP_ERR_LENGTH_ERROR             0x7EU    /**< MFPRIMENTAG Length of command string invalid. */
#define PHAL_MFPRIMENTAG_RESP_PERMISSION_DENIED            0x9DU    /**< MFPRIMENTAG Current configuration/status does not allow the requested command. */
#define PHAL_MFPRIMENTAG_RESP_ERR_PARAMETER_ERROR          0x9EU    /**< MFPRIMENTAG Value of params invalid. */
#define PHAL_MFPRIMENTAG_RESP_ERR_AUTHENTICATION_ERROR     0xAEU    /**< MFPRIMENTAG Current authentication status does not allow the requested cmd. */
#define PHAL_MFPRIMENTAG_RESP_ADDITIONAL_FRAME             0xAFU    /**< MFPRIMENTAG Additional data frame is expected to be sent. */
#define PHAL_MFPRIMENTAG_RESP_ERR_BOUNDARY_ERROR           0xBEU    /**< MFPRIMENTAG Attempt to read/write data from/to beyond the files/record's limits. */
#define PHAL_MFPRIMENTAG_RESP_ERR_COMMAND_ABORTED          0xCAU    /**< MFPRIMENTAG Previous cmd not fully completed. Not all frames were requested or provided by the PCD. */
#define PHAL_MFPRIMENTAG_RESP_ERR_FILE_NOT_FOUND           0xF0U    /**< MFPRIMENTAG Specified file number does not exist. */

/** @} */

/**
* \name ISO 7816 Instructions
*/

/** @{ */
#define PHAL_MFPRIMENTAG_CMD_ISO7816_SELECT_FILE       0xA4    /**< ISO Select File. */
#define PHAL_MFPRIMENTAG_CMD_ISO7816_READ_BINARY       0xB0    /**< ISO Read Binary. */
#define PHAL_MFPRIMENTAG_CMD_ISO7816_UPDATE_BINARY     0xD6    /**< ISO UPDATE Binary. */

/** @} */

/**
* \name MIFARE NTAG 413 specific defines
*/

/** @{ */
#define PHAL_MFPRIMENTAG_MAXWRAPPEDAPDU_SIZE   0x37u   /**< Maximum size of wrapped APDU 55 bytes. */
#define PHAL_MFPRIMENTAG_MAXDFAPDU_SIZE        0x3Cu   /**< Maximum size of DESFire APDU 60 bytes. */
#define PHAL_MFPRIMENTAG_DFAPPID_SIZE          0x03u   /**< Size of MFPRIMENTAG application Id. */
#define PHAL_MFPRIMENTAG_DATA_BLOCK_SIZE       0x10u   /**< Data block size need for internal purposes. */
#define PHAL_MFPRIMENTAG_MAX_FRAME_SIZE        0x40u   /**< Max size in a ISO 14443-4 frame. */
/** @} */

/**
* \name ISO 7816 NTAG 413 return Codes
*/
/** @{ */
#define PHAL_MFPRIMENTAG_ISO7816_SUCCESS                        0x9000U /**< Correct execution. */
#define PHAL_MFPRIMENTAG_ISO7816_ERR_WRONG_LENGTH               0x6700U /**< Wrong length. */
#define PHAL_MFPRIMENTAG_ISO7816_ERR_INVALID_APPLN              0x6A82U /**< Application / file not found. */
#define PHAL_MFPRIMENTAG_ISO7816_ERR_WRONG_PARAMS               0x6A86U /**< Wrong parameters P1 and/or P2. */
#define PHAL_MFPRIMENTAG_ISO7816_ERR_WRONG_LC			        0x6A87U /**< Lc inconsistent with P1/p2. */
#define PHAL_MFPRIMENTAG_ISO7816_ERR_WRONG_LE                   0x6C00U /**< Wrong Le. */
#define PHAL_MFPRIMENTAG_ISO7816_ERR_NO_PRECISE_DIAGNOSTICS     0x6F00U /**< No precise diagnostics. */
#define PHAL_MFPRIMENTAG_ISO7816_ERR_EOF_REACHED		        0x6282U /**< End of File reached. */
#define PHAL_MFPRIMENTAG_ISO7816_ERR_FILE_ACCESS		        0x6982U	/**< File access not allowed. */
#define PHAL_MFPRIMENTAG_ISO7816_ERR_FILE_EMPTY		            0x6985U	/**< File empty or access conditions not satisfied. */
#define PHAL_MFPRIMENTAG_ISO7816_ERR_FILE_NOT_FOUND		        0x6A82U /**< File not found. */
#define PHAL_MFPRIMENTAG_ISO7816_ERR_MEMORY_FAILURE		        0x6581U /**< Memory failure (unsuccessful update). */
#define PHAL_MFPRIMENTAG_ISO7816_ERR_INCORRECT_PARAMS		    0x6B00U /**< Wrong parameter p1 or p2. READ RECORDS. */
#define PHAL_MFPRIMENTAG_ISO7816_ERR_WRONG_CLA			        0x6E00U /**< Wrong Class byte. */
#define PHAL_MFPRIMENTAG_ISO7816_ERR_UNSUPPORTED_INS		    0x6D00U /**< Instruction not supported. */
/** @} */

#define PHAL_MFPRIMENTAG_WRAP_HDR_LEN      0x05U   /* Wrapped APDU header length */
#define PHAL_MFPRIMENTAG_WRAPPEDAPDU_CLA   0x90U   /* Wrapped APDU default class. */
#define PHAL_MFPRIMENTAG_WRAPPEDAPDU_P1    0x00U   /* Wrapped APDU default P1. */
#define PHAL_MFPRIMENTAG_WRAPPEDAPDU_P2    0x00U   /* Wrapped APDU default P2. */
#define PHAL_MFPRIMENTAG_WRAPPEDAPDU_LE    0x00U   /* Wrapped APDU default LE. */

#define PHAL_MFPRIMENTAG_TRUNCATED_MAC_SIZE		8       /**< Size of the truncated MAC. */
#define PHAL_MFPRIMENTAG_KEYSETVERSIONS			0x40U   /**< 6th bit KeyNo used to retrieve all keyset versions. */
#define PHAL_MFPRIMENTAG_ISO_CHAINING_MODE		0x04    /**< Option to enable ISO chaining. */
#define PHAL_MFPRIMENTAG_DEFAULT_MODE			0x00    /**< Native Chaining. */
#define PHAL_MFPRIMENTAG_SIG_LENGTH				0x38	/**< NXP Originality Signature length */
#define PHAL_MFPRIMENTAG_DEFAULT_UID_LENGTH		0x07	/**< By default NTAG card is configures as 7 byte UID */
#define PHAL_MFPRIMENTAG_10B_UID_LENGTH		    0x09    /**< EV2 can also be configured as 10 byte UID. */
#define PHAL_MFPRIMENTAG_4B_UID_LENGTH		    0x06    /**< EV2 can also be configured as 4 byte UID. */
#define PHAL_MFPRIMENTAG_SDM_FILE_CTR_LENGTH	0x05	/**< SDMReadCounter(3B) + reserved(2B) */
#define PHAL_MFPRIMENTAG_DEF_VERSION_LENGTH		0x1C	/**< Version String is of 28 bytes by default(If its 7 Byte UID) */
#define PHAL_MFPRIMENTAG_10B_VERSION_LENGTH	    0x1E	/**< Version String is of 30 bytes If its 10B Byte UID */
#define PHAL_MFPRIMENTAG_4B_VERSION_LENGTH	    0x1B	/**< Version String is of 27 bytes If its 4B Byte UID */
#define PHAL_MFPRIMENTAG_ISO_CHAINING_MODE_MAPPED	(PHAL_MFPRIMENTAG_ISO_CHAINING_MODE << 2)	/**< Option to enable ISO chaining mapped for internal use. */

#define PHAL_MFPRIMENTAG_ISO_7816_NO_LC_LE		 0x00								  /**< Check if Lc=0 and Le=0 */
#define PHAL_MFPRIMENTAG_ISO_7816_LC_SHORT_APDU	 0x01								  /**< Indicates Lc=1 byte */
#define PHAL_MFPRIMENTAG_ISO_7816_LC_EXT_APDU_3B 0x03								  /**< Indicates Lc=3 bytes */
#define PHAL_MFPRIMENTAG_ISO_7816_LE_SHORT_APDU	 PHAL_MFPRIMENTAG_ISO_7816_LC_SHORT_APDU  /**< Indicates Le=1 byte */
#define PHAL_MFPRIMENTAG_ISO_7816_LE_EXT_APDU_2B 0x02								  /**< Indicates Le=2 bytes */
#define PHAL_MFPRIMENTAG_ISO_7816_LE_EXT_APDU_3B PHAL_MFPRIMENTAG_ISO_7816_LC_EXT_APDU_3B /**< Indicates Le=3 bytes */
/**
* \name Proximity Check return Codes
*/
/** @{ */

#define PHAL_MFPRIMENTAG_RESP_ERR_CMD_INVALID   0x0BU   /*< MFPRIMENTAG Invalid Command Error. */
#define PHAL_MFPRIMENTAG_RESP_ERR_FORMAT        0x0CU   /*< MFPRIMENTAG Format Error. */

/** @} */

/* To avoid unused variable warnings. */
#define PHAL_MFPRIMENTAG_UNUSED_VARIABLE(x)  {for( ( x ) = ( x ) ; ( x ) != ( x ) ; );}
#define PHAL_MFPRIMENTAG_UNUSED_ARRAY(x)  {for( ( (x)[0] ) = ( (x)[0] ) ; ( (x)[0] ) != ( (x)[0] ) ; );}

#define PHAL_MFPRIMENTAG_SDM_FILE_ID			0x02	/*< MFPRIMENTAG File Id where NDEF data is stored. */
#define PHAL_MFPRIMENTAG_STANDARD_FILE_ID		0x01 	/*< MFPRIMENTAG File Id standard data is stored. */
#define PHAL_MFPRIMENTAG_PLAIN_MIRRORING		0x0E 	/*< Plain Mirroring. */
#define PHAL_MFPRIMENTAG_NO_SDM_FOR_READING		0x0F 	/*< No SDM For Reading. */

phStatus_t phalMfprimeNtag_ExchangeCmd(
                                   void * pDataParams,         /* Pointer to params data structure */
                                   void * pPalMifareDataParams,  /*  Pointer to mifare protocol layer */
                                   uint8_t bWrappedMode,
                                   uint8_t * pCmd,
                                   uint16_t wCmdLen,
                                   uint8_t ** ppResponse,
                                   uint16_t * wRxlen
                                   );

phStatus_t phalMfprimeNtag_Int_ComputeErrorResponse(
    void * pDataParams,
    uint16_t wStatus
    );

phStatus_t phalMfprimeNtag_Int_Send7816Apdu(
                                        void * pDataParams,
                                        void * pPalMifareDataParams,
                                        uint8_t bOption,
                                        uint8_t bIns,
                                        uint8_t p1,
                                        uint8_t p2,
                                        uint32_t Lc,
										uint8_t bExtendedLenApdu,
                                        uint8_t * pDataIn,  /*  Data In of Lc bytes. */
                                        uint32_t bLe,
                                        uint8_t ** pDataOut,  /*  Data received from the card. */
                                        uint16_t *pDataLen  /*  Length of data returned. */
                                        );

phStatus_t phalMfprimeNtag_Int_ISOSelectFile(
											void * pDataParams,
											void * pPalMifareDataParams,
											uint8_t bOption,
											uint8_t * bCmdBuff,
											uint16_t wCmdLen,
											uint8_t bLc,
											uint8_t bExtendedLenApdu,
											uint8_t * pDataIn,
											uint8_t bLe,
											uint8_t ** ppDataOut,
											uint16_t *pDataLen
										);

phStatus_t phalMfprimeNtag_Int_ISOUpdateBinary(
											void * pDataParams,
											void * pPalMifareDataParams,
											uint8_t bOption,
											uint8_t * bCmdBuff,
											uint16_t wCmdLen,
											uint32_t dwLc,
											uint8_t bExtendedLenApdu,
											uint8_t * pDataIn,
											uint8_t ** ppDataOut,
											uint16_t *pDataLen
										  );

/** @} */

#endif /* PHALMFPRIMENTAG_INT_H */
