/*
 * Copyright 2016 - 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic MIFARE PRIME NTAG Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phalMfprimeNtag.h>
#include <ph_Status.h>
#include <ph_TypeDefs.h>

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
#include "Sw/phalMfprimeNtag_Sw.h"
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
#include "Sam_NonX\phalMfPrimeNtag_Sam_NonX.h"
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
#include "Sam_X\phalMfPrimeNtag_Sam_X.h"
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG

#ifdef NXPBUILD__PH_CRYPTOSYM
/* MIFARE PRIME NTAG secure messaging related commands. ------------------------------------------------------------------------------ */
phStatus_t phalMfprimeNtag_AuthenticateEv2(void *pDataParams, uint8_t bFirstAuth, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bKeyNoCard,
	uint8_t * pDivInput, uint8_t bDivLen, uint8_t bLenPcdCapsIn, uint8_t *bPcdCapsIn, uint8_t *bPcdCapsOut, uint8_t *bPdCapsOut)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_AuthenticatEv2");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFirstAuth);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyVer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bKeyNoCard);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDivInput);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bPcdCapsIn);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bPcdCapsOut);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bPdCapsOut);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFirstAuth_log, &bFirstAuth);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wOption_log, &wOption);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyNo_log, &wKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyVer_log, &wKeyVer);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bKeyNoCard_log, &bKeyNoCard);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pDivInput_log, pDivInput, bDivLen);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, bPcdCapsIn_log, bPcdCapsIn, bLenPcdCapsIn);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);
	if(bDivLen) PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_AL_MFPRIMENTAG);
	if(bLenPcdCapsIn)  PH_ASSERT_NULL_PARAM(bPcdCapsIn, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_AuthenticateEv2((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, bFirstAuth, wOption, wKeyNo, wKeyVer,
				bKeyNoCard, pDivInput, bDivLen, bLenPcdCapsIn, bPcdCapsIn, bPcdCapsOut, bPdCapsOut);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_AuthenticateEv2(pDataParams, bFirstAuth, wOption, wKeyNo, wKeyVer, bKeyNoCard, pDivInput, bDivLen,
				bPcdCapsIn, bLenPcdCapsIn, bPcdCapsOut, bPdCapsOut);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_AuthenticateEv2(pDataParams, bFirstAuth, wOption, wKeyNo, wKeyVer, bKeyNoCard, pDivInput, bDivLen,
				bPcdCapsIn, bLenPcdCapsIn, bPcdCapsOut, bPdCapsOut);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	if(bFirstAuth)
	{
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, bPcdCapsOut_log, bPcdCapsOut, 6);
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, bPdCapsOut_log, bPdCapsOut, 6);
	}
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */




/* MIFARE PRIME NTAG Memory and Configuration mamangement commands. ------------------------------------------------------------------ */
#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalMfprimeNtag_SetConfiguration(void * pDataParams, uint8_t bOption, uint8_t * pData, uint8_t bDataLen)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_SetConfiguration");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pData);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDataLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOption_log, &bOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDataLen_log, &bDataLen);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pData_log, pData, bDataLen);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(pData, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_SetConfiguration((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, bOption, pData, bDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_SetConfiguration(pDataParams, bOption, pData, bDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_SetConfiguration(pDataParams, bOption, pData, bDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalMfprimeNtag_GetVersion(void * pDataParams, uint8_t * pVerInfo)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_GetVersion");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pVerInfo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(pVerInfo, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_GetVersion((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, pVerInfo);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_GetVersion(pDataParams, pVerInfo);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_GetVersion(pDataParams, pVerInfo);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pVerInfo_log, pVerInfo, 28);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}



/* MIFARE PRIME NTAG Key mamangement commands. -------------------------------------------------------------------------------------- */
#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalMfprimeNtag_ChangeKey(void * pDataParams, uint16_t wOption, uint16_t wOldKeyNo, uint16_t wOldKeyVer, uint16_t wNewKeyNo, uint16_t wNewKeyVer,
	uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivLen)
{
	/**
	* This  function will handle all the three authentication modes: 0x0A, 1A and AA.
	* and all crypto modes i.e., DES, 3DES, 3K3DES, AES
	* The previous authentication status including key number and session key is
	* present in the params  structure.
	* Successful auth. with PICC master key is required if AID = 0x00 else
	* an auth. with the application master key is required.
	*/
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_ChangeKey");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOldKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wNewKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOldKeyVer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wNewKeyVer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bKeyNoCard);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDivInput);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDivLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wOption_log, &wOption);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wOldKeyVer_log, &wOldKeyVer);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wNewKeyVer_log, &wNewKeyVer);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pDivInput_log, pDivInput, bDivLen);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wOldKeyNo_log, &wOldKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wNewKeyNo_log, &wNewKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bKeyNoCard_log, &bKeyNoCard);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDivLen_log, &bDivLen);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);
	if(bDivLen > 0) PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_ChangeKey((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, wOption, wOldKeyNo, wOldKeyVer, wNewKeyNo,
				wNewKeyVer, bKeyNoCard, pDivInput, bDivLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_ChangeKey(pDataParams, wOption, wOldKeyNo, wOldKeyVer, wNewKeyNo, wNewKeyVer, bKeyNoCard, pDivInput, bDivLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_ChangeKey(pDataParams, wOption, wOldKeyNo, wOldKeyVer, wNewKeyNo, wNewKeyVer, bKeyNoCard, pDivInput, bDivLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
	return status;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalMfprimeNtag_GetKeyVersion(void * pDataParams, uint8_t bKeyNo, uint8_t bKeySetNo, uint8_t * pKeyVersion, uint8_t * bRxLen)
{
	/**
	* This command can be issued without valid authentication
	*/
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_GetKeyVersion");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bKeySetNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pKeyVersion);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bRxLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bKeyNo_log, &bKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bKeySetNo_log, &bKeySetNo);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(pKeyVersion, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_GetKeyVersion((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, bKeyNo, bKeySetNo, pKeyVersion, bRxLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_GetKeyVersion(pDataParams, bKeyNo, pKeyVersion, bRxLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_GetKeyVersion(pDataParams, bKeyNo, pKeyVersion, bRxLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pKeyVersion_log, pKeyVersion, *bRxLen);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bRxLen_log, bRxLen);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}




/* MIFARE PRIME NTAG File mamangement commands. -------------------------------------------------------------------------------------- */
phStatus_t phalMfprimeNtag_GetFileSettings(void * pDataParams, uint8_t bFileNo, uint8_t * pFSBuffer, uint8_t * bBufferLen)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_GetFileSettings");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pFSBuffer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bBufferLen);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileNo_log, &bFileNo);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(pFSBuffer, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(bBufferLen, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_GetFileSettings((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, bFileNo, pFSBuffer, bBufferLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_GetFileSettings(pDataParams, bFileNo, pFSBuffer, bBufferLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_GetFileSettings(pDataParams, bFileNo, pFSBuffer, bBufferLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pFSBuffer_log, pFSBuffer, (*bBufferLen));
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, bBufferLen_log, bBufferLen, 1);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalMfprimeNtag_GetFileCountersSDM(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pFileCounters, uint8_t * pRxLen)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_GetFileCountersSDM");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pFileCounters);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRxLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOption_log, &bOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileNo_log, &bFileNo);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_GetFileCountersSDM((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, bOption, bFileNo, pFileCounters, pRxLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_GetFileCountersSDM(pDataParams, bOption, bFileNo, pFileCounters, pRxLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_GetFileCountersSDM(pDataParams, bOption, bFileNo, pFileCounters, pRxLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
	if((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
	{
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pFileCounters_log, pFileCounters, (*pRxLen));
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pRxLen_log, pRxLen, 1);
	}
#endif
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalMfprimeNtag_ChangeFileSettings(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t bCommSett, uint8_t *pAccessRights,
	uint8_t bNumAddARs, uint8_t * pAddARs)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_ChangeFileSettings");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bCommSett);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pAccessRights);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bNumAddARs);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pAddARs);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOption_log, &bOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileNo_log, &bFileNo);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bCommSett_log, &bCommSett);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bNumAddARs_log, &bNumAddARs);
	if(pAccessRights != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pAccessRights_log, pAccessRights, 2);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pAddARs_log, pAddARs, bNumAddARs);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(pAccessRights, PH_COMP_AL_MFPRIMENTAG);
	if(bNumAddARs)
		PH_ASSERT_NULL_PARAM(pAddARs, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_ChangeFileSettings((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, bOption, bFileNo, bCommSett, pAccessRights,
				bNumAddARs, pAddARs);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_ChangeFileSettings(pDataParams, bOption, bFileNo, bCommSett, pAccessRights);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_ChangeFileSettings(pDataParams, bOption, bFileNo, bCommSett, pAccessRights);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfprimeNtag_ChangeFileSettingsSDM(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t bFileOption, uint8_t *pAccessRights,
	uint8_t bSdmOptions, uint8_t *pSdmAccessRights, uint8_t *pVCUIDOffset, uint8_t *pSDMReadCtrOffset, uint8_t *pSDMMACInputOffset, uint8_t *pSDMMACOffset)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_ChangeFileSettingsSDM");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pAccessRights);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bSdmOptions);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pSdmAccessRights);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pVCUIDOffset);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pSDMReadCtrOffset);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pSDMMACInputOffset);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pSDMMACOffset);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOption_log, &bOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileOption_log, &bFileOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileNo_log, &bFileNo);
	PH_ASSERT_NULL_PARAM(pAccessRights, PH_COMP_AL_MFPRIMENTAG);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pAccessRights_log, pAccessRights, 2);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bSdmOptions_log, &bSdmOptions);
	PH_ASSERT_NULL_PARAM(pSdmAccessRights, PH_COMP_AL_MFPRIMENTAG);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pSdmAccessRights_log, pSdmAccessRights, 2);
	if(pVCUIDOffset != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pVCUIDOffset_log, pVCUIDOffset, 3);
	if(pSDMReadCtrOffset != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pSDMReadCtrOffset_log, pSDMReadCtrOffset, 3);
	PH_ASSERT_NULL_PARAM(pSDMMACInputOffset, PH_COMP_AL_MFPRIMENTAG);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pSDMMACInputOffset_log, pSDMMACInputOffset, 3);
	PH_ASSERT_NULL_PARAM(pSDMMACOffset, PH_COMP_AL_MFPRIMENTAG);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pSDMMACOffset_log, pSDMMACOffset, 3);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_ChangeFileSettingsSDM((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, bOption, bFileNo, bFileOption, pAccessRights,
				bSdmOptions, pSdmAccessRights, pVCUIDOffset, pSDMReadCtrOffset, pSDMMACInputOffset, pSDMMACOffset);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_ChangeFileSettingsSDM(pDataParams, bOption, bFileNo, bFileOption, pAccessRights, bSdmOptions, pSdmAccessRights,
				pVCUIDOffset, pSDMReadCtrOffset, pSDMMACInputOffset, pSDMMACOffset);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_ChangeFileSettingsSDM(pDataParams, bOption, bFileNo, bFileOption, pAccessRights, bSdmOptions, pSdmAccessRights,
				pVCUIDOffset, pSDMReadCtrOffset, pSDMMACInputOffset, pSDMMACOffset);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */
		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}




/* MIFARE PRIME NTAG Data mamangement commands. -------------------------------------------------------------------------------------- */
phStatus_t phalMfprimeNtag_ReadData(void * pDataParams, uint8_t bOption, uint8_t bIns, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pLength, uint8_t ** ppRxdata,
	uint16_t * pRxdataLen)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_ReadData");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bIns);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pOffset);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pLength);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(ppRxdata);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRxdataLen);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOption_log, &bOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bIns_log, &bIns);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileNo_log, &bFileNo);
	if(pOffset != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pOffset_log, pOffset, 3);
	if(pLength != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pLength_log, pLength, 3);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(pOffset, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(pLength, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(ppRxdata, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(pRxdataLen, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_ReadData((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, bOption, bIns, bFileNo, pOffset, pLength,
				ppRxdata, pRxdataLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_ReadData(pDataParams, bOption, bFileNo, pOffset, pLength, ppRxdata, pRxdataLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_ReadData(pDataParams, bOption, bFileNo, pOffset, pLength, ppRxdata, pRxdataLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	if(*pRxdataLen != 0)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, ppRxdata_log, *ppRxdata, (*pRxdataLen));
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, pRxdataLen_log, pRxdataLen);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfprimeNtag_WriteData(void * pDataParams, uint8_t bOption, uint8_t bIns, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pTxData, uint8_t * pTxDataLen)
{
	phStatus_t  PH_MEMLOC_REM status;
	uint16_t    PH_MEMLOC_REM wDataLen = 0;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_WriteData");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bIns);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pOffset);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTxData);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTxDataLen);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOption_log, &bOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bIns_log, &bIns);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileNo_log, &bFileNo);
	if(pOffset != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pOffset_log, pOffset, 3);
	if(pTxDataLen != NULL)
	{
		wDataLen = (uint16_t) pTxDataLen[1];
		wDataLen = wDataLen << 8;
		wDataLen |= pTxDataLen[0];
	}
	if(pTxData != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pTxData_log, pTxData, wDataLen);
	if(pTxDataLen != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pTxDataLen_log, pTxDataLen, 3);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(pOffset, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(pTxData, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(pTxDataLen, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_WriteData((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, bOption, bIns, bFileNo, pOffset, pTxData, pTxDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_WriteData(pDataParams, bOption, bFileNo, pOffset, pTxData, pTxDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_WriteData(pDataParams, bOption, bFileNo, pOffset, pTxData, pTxDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}




/* MIFARE PRIME NTAG ISO7816-4 commands. --------------------------------------------------------------------------------------------- */
phStatus_t phalMfprimeNtag_IsoSelectFile(void * pDataParams, uint8_t bOption, uint8_t bSelector, uint8_t * pFid, uint8_t * pDFname, uint8_t bDFnameLen,
	uint8_t bExtendedLenApdu, uint8_t **ppFCI, uint16_t *pwFCILen)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_IsoSelectFile");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bSelector);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pFid);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDFname);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDFnameLen);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOption_log, &bOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bSelector_log, &bSelector);
	if(pFid != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pFid_log, pFid, 2);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDFnameLen_log, &bDFnameLen);
	if(pDFname != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pDFname_log, pDFname, (uint16_t) bDFnameLen);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_IsoSelectFile((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, bOption, bSelector, pFid, pDFname, bDFnameLen,
				bExtendedLenApdu, ppFCI, pwFCILen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_IsoSelectFile(pDataParams, bOption, bSelector, pFid, pDFname, bDFnameLen, bExtendedLenApdu, ppFCI, pwFCILen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_IsoSelectFile(pDataParams, bOption, bSelector, pFid, pDFname, bDFnameLen, bExtendedLenApdu, ppFCI, pwFCILen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfprimeNtag_IsoReadBinary(void * pDataParams, uint16_t wOption, uint8_t bOffset, uint8_t bSfid, uint32_t dwBytesToRead, uint8_t bExtendedLenApdu,
	uint8_t ** ppRxBuffer, uint32_t * pBytesRead)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_IsoReadBinary");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOffset);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bSfid);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(dwBytesToRead);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(ppRxBuffer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pBytesRead);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOffset_log, &bOffset);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bSfid_log, &bSfid);
	PH_LOG_HELPER_ADDPARAM_UINT32(PH_LOG_LOGTYPE_DEBUG, dwBytesToRead_log, &dwBytesToRead);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(ppRxBuffer, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(pBytesRead, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_IsoReadBinary((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, wOption, bOffset, bSfid, dwBytesToRead, bExtendedLenApdu,
				ppRxBuffer, pBytesRead);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_IsoReadBinary(pDataParams, wOption, bOffset, bSfid, dwBytesToRead, bExtendedLenApdu, ppRxBuffer, pBytesRead);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_IsoReadBinary(pDataParams, wOption, bOffset, bSfid, dwBytesToRead, bExtendedLenApdu, ppRxBuffer, pBytesRead);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, ppRxBuffer_log, *ppRxBuffer, (uint16_t) (*pBytesRead));
	PH_LOG_HELPER_ADDPARAM_UINT32(PH_LOG_LOGTYPE_INFO, pBytesRead_log, pBytesRead);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfprimeNtag_IsoUpdateBinary(void * pDataParams, uint8_t bOffset, uint8_t bSfid, uint8_t bExtendedLenApdu, uint8_t * pData, uint32_t dwDataLen)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_IsoUpdateBinary");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOffset);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bSfid);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pData);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(dwDataLen);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);

	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOffset_log, &bOffset);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bSfid_log, &bSfid);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pData_log, pData, (uint16_t) dwDataLen);
	PH_LOG_HELPER_ADDPARAM_UINT32(PH_LOG_LOGTYPE_INFO, dwDataLen_log, &dwDataLen);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(pData, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_IsoUpdateBinary((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, bOffset, bSfid, bExtendedLenApdu, pData, dwDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_IsoUpdateBinary(pDataParams, bOffset, bSfid, bExtendedLenApdu, pData, dwDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_IsoUpdateBinary(pDataParams, bOffset, bSfid, bExtendedLenApdu, pData, dwDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}




/* MIFARE PRIME NTAG Originality Check functions. ------------------------------------------------------------------------------------ */
phStatus_t phalMfprimeNtag_ReadSign(void * pDataParams, uint8_t bAddr, uint8_t ** pSignature)
{
	phStatus_t PH_MEMLOC_REM status = 0;
	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_ReadSign");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bAddr);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pSignature);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bAddr_log, &bAddr);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(pSignature, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return  PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_ReadSign((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, bAddr, pSignature);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_ReadSign(pDataParams, bAddr, pSignature);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_ReadSign(pDataParams, bAddr, pSignature);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	if(*pSignature != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pSignature_log, *pSignature, 56);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}




/* MIFARE PRIME NTAG Miscellaneous functions. ---------------------------------------------------------------------------------------- */
phStatus_t phalMfprimeNtag_GetConfig(void * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_GetConfig");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wConfig);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pValue);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wConfig_log, &wConfig);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(pValue, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_GetConfig((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, wConfig, pValue);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_GetConfig(pDataParams, wConfig, pValue);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_GetConfig(pDataParams, wConfig, pValue);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
	if((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
	{
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, pValue_log, pValue);
	}
#endif
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfprimeNtag_SetConfig(void * pDataParams, uint16_t wConfig, uint16_t wValue)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_GetConfig");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wConfig);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wValue);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wConfig_log, &wConfig);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wValue_log, &wValue);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_SetConfig((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, wConfig, wValue);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_SetConfig(pDataParams, wConfig, wValue);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_SetConfig(pDataParams, wConfig, wValue);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);

	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfprimeNtag_ResetAuthentication(void * pDataParams)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_ResetAuthentication");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_ResetAuthentication((phalMfprimeNtag_Sw_DataParams_t *) pDataParams);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_ResetAuthentication(pDataParams);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_ResetAuthentication(pDataParams);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);

	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalMfprimeNtag_CalculateMACSDM(void * pDataParams, uint16_t wOption, uint8_t  bSdmOption, uint16_t wSDMFileReadKeyNo, uint16_t wSDMFileReadKeyVer,
	uint16_t wRamKeyNo, uint16_t wRamKeyVer, uint8_t  *pDivInput, uint8_t  bDivInputLen, uint8_t  *pUid, uint8_t  bUidLen, uint8_t  *pSDMReadCtr,
	uint8_t  *pIndata, uint16_t wInDataLen, uint8_t  *pRespMac)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfprimeNtag_CalculateMACSDM");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bSdmOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wSDMFileReadKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wSDMFileReadKeyVer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wRamKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wRamKeyVer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDivInput);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDivInputLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bUidLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pSDMReadCtr);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pIndata);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wInDataLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRespMac);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wOption_log, &wOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bSdmOption_log, &bSdmOption);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wSDMFileReadKeyNo_log, &wSDMFileReadKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wSDMFileReadKeyVer_log, &wSDMFileReadKeyVer);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wRamKeyNo_log, &wRamKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wRamKeyVer_log, &wRamKeyVer);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bUidLen_log, &bUidLen);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDivInputLen_log, &bDivInputLen);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pDivInput_log, pDivInput, bDivInputLen);
	if(pSDMReadCtr != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pSDMReadCtr_log, pSDMReadCtr, 3);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wInDataLen_log, &wInDataLen);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pIndata_log, pIndata, wInDataLen);
	if(pUid != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pUid_log, pUid, bUidLen);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);
	if(bDivInputLen) PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_AL_MFPRIMENTAG);
	if(wInDataLen) PH_ASSERT_NULL_PARAM(pIndata, PH_COMP_AL_MFPRIMENTAG);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFPRIMENTAG)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW
		case PHAL_MFPRIMENTAG_SW_ID:
			status = phalMfprimeNtag_Sw_CalculateMACSDM((phalMfprimeNtag_Sw_DataParams_t *) pDataParams, wOption, bSdmOption, wSDMFileReadKeyNo,
				wSDMFileReadKeyVer, pDivInput, bDivInputLen, pUid, bUidLen, pSDMReadCtr, pIndata, wInDataLen, pRespMac);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
		case PHAL_MFPRIMENTAG_SAMAV3_NONX_ID:
			status = phalMfPrimeNtag_Sam_NonX_CalculateMACSDM(pDataParams, bSdmOption, wSDMFileReadKeyNo, wSDMFileReadKeyVer, wRamKeyNo, wRamKeyVer,
				pUid, bUidLen, pSDMReadCtr, pIndata, wInDataLen, pRespMac);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X
		case PHAL_MFPRIMENTAG_SAMAV3_X_ID:
			status = phalMfPrimeNtag_Sam_X_CalculateMACSDM(pDataParams, bSdmOption, wSDMFileReadKeyNo, wSDMFileReadKeyVer, wRamKeyNo, wRamKeyVer,
				pUid, bUidLen, pSDMReadCtr, pIndata, wInDataLen, pRespMac);
			break;
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pRespMac_log, pRespMac, 8);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

#endif /* NXPBUILD__PHAL_MFPRIMENTAG */
