/*
 * Copyright 2009 - 2018, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal functions for MIFARE (R) application layer.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 *
 * History:
 *  CHu: Generated 05. November 2009
 *
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <phalMfc.h>
#include "phalMfc_Int.h"
#include <phpalMifare.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHAL_MFC

/*
 * Perform MIFARE(R) Transfer command with MIFARE Picc. This interface will be called by Software and
 * Sam component.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to PAL Mifare parameter structure.
 *		bBlockNo				: The block number to be used for transferring the value.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Int_Transfer(void * pPalMifareDataParams, uint8_t bBlockNo)
{
	uint8_t     PH_MEMLOC_REM aCommand[2];
	uint8_t *   PH_MEMLOC_REM pRxBuffer;
	uint16_t    PH_MEMLOC_REM wRxLength;

	/* Build command frame */
	aCommand[0] = PHAL_MFC_CMD_TRANSFER;
	aCommand[1] = bBlockNo;

	/* Transmit the command frame */
	return phpalMifare_ExchangeL3(
		pPalMifareDataParams,
		PH_EXCHANGE_DEFAULT,
		aCommand,
		2,
		&pRxBuffer,
		&wRxLength
		);
}

/*
 * Perform a MIFARE(R) Value operation with MIFARE Picc. This is used by INCREMENT, DECREMENT and RESTORE commands.
 * This interface will be called by Software and Sam component.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to PAL Mifare layer's parameter structure.
 *		bBlockNo				: The block number to be used for decrementing the value.
 *		pValue					: The value to be Decremented in a MIFARE PICC.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Int_Value(void * pPalMifareDataParams, uint8_t bCmdCode, uint8_t bBlockNo,
	uint8_t * pValue)
{
	phStatus_t  PH_MEMLOC_REM status;
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM aCommand[2];
	uint8_t *   PH_MEMLOC_REM pRxBuffer;
	uint16_t    PH_MEMLOC_REM wRxLength;

	/* Build command frame */
	aCommand[0] = bCmdCode;
	aCommand[1] = bBlockNo;

	/* Transmit the command frame (first part) */
	PH_CHECK_SUCCESS_FCT(statusTmp, phpalMifare_ExchangeL3(
		pPalMifareDataParams,
		PH_EXCHANGE_DEFAULT,
		aCommand,
		2,
		&pRxBuffer,
		&wRxLength
		));

	/* Transmit the data (second part) */
	status = phpalMifare_ExchangeL3(
		pPalMifareDataParams,
		PH_EXCHANGE_DEFAULT,
		pValue,
		PHAL_MFC_VALUE_BLOCK_LENGTH,
		&pRxBuffer,
		&wRxLength
		);

	/* There should be no response in case of successful operation */
	if ((status & PH_ERR_MASK) != PH_ERR_IO_TIMEOUT)
	{
		/* Resolve NAK Code */
		PH_CHECK_SUCCESS(status);

		/* ACK means protocol error */
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFC);
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}

/*
 * Create a Value block for a given value/addr pair.
 *
 * Input Parameters:
 *		pValue					: The value to be used for valueblock format creation.
 *		bAddrData				: The address to be used for valueblock format creation.
 *		pBlock					: The block to be used for valueblock format creation.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Int_CreateValueBlock(uint8_t * pValue, uint8_t bAddrData, uint8_t * pBlock)
{
	pBlock[0]  = (uint8_t)pValue[0];
	pBlock[1]  = (uint8_t)pValue[1];
	pBlock[2]  = (uint8_t)pValue[2];
	pBlock[3]  = (uint8_t)pValue[3];
	pBlock[4]  = (uint8_t)~(uint8_t)pValue[0];
	pBlock[5]  = (uint8_t)~(uint8_t)pValue[1];
	pBlock[6]  = (uint8_t)~(uint8_t)pValue[2];
	pBlock[7]  = (uint8_t)~(uint8_t)pValue[3];
	pBlock[8]  = (uint8_t)pValue[0];
	pBlock[9]  = (uint8_t)pValue[1];
	pBlock[10] = (uint8_t)pValue[2];
	pBlock[11] = (uint8_t)pValue[3];
	pBlock[12] = (uint8_t)bAddrData;
	pBlock[13] = (uint8_t)~(uint8_t)bAddrData;
	pBlock[14] = (uint8_t)bAddrData;
	pBlock[15] = (uint8_t)~(uint8_t)bAddrData;
	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}

/*
 * Check value block format of a given block.
 *
 * Input Parameters:
 *		pBlock					: The block to be used for valueblock format verification.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Int_CheckValueBlockFormat(uint8_t * pBlock)
{
	/* Check format of value block */
	if ((pBlock[0] != pBlock[8]) ||
		(pBlock[1] != pBlock[9]) ||
		(pBlock[2] != pBlock[10]) ||
		(pBlock[3] != pBlock[11]) ||
		(pBlock[4] != (pBlock[0] ^ 0xFF)) ||
		(pBlock[5] != (pBlock[1] ^ 0xFF)) ||
		(pBlock[6] != (pBlock[2] ^ 0xFF)) ||
		(pBlock[7] != (pBlock[3] ^ 0xFF)) ||
		(pBlock[12] != pBlock[14]) ||
		(pBlock[13] != pBlock[15]) ||
		(pBlock[12] != (pBlock[13]^ 0xFF)))
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFC);
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}

#endif /* NXPBUILD__PHAL_MFC */
