/*
*         Copyright (c), NXP Semiconductors Bangalore / India
*
*                     (C)NXP Semiconductors
*       All rights are reserved. Reproduction in whole or in part is 
*      prohibited without the written consent of the copyright owner.
*  NXP reserves the right to make changes without notice at any time.
* NXP makes no warranty, expressed, implied or statutory, including but
* not limited to any implied warranty of merchantability or fitness for any
* particular purpose, or that the use will not infringe any third party patent,
* copyright or trademark. NXP must not be liable for any loss or damage
*                          arising from its use.
*/

/**
* \file
* DESFire IDENTITY application SamAV3 NonX component of Reader Library framework.
* $Author: Rajendran Kumar (nxp99556) $
* $Revision: 7230 $
* $Date: 2025-03-14 10:27:23 +0530 (Fri, 14 Mar 2025) $
*
* History:
*/

#include <ph_Status.h>
#include <phhalHw.h>
#include <phCryptoSym.h>
#include <phCryptoRng.h>
#include <phKeyStore.h>
#include <ph_RefDefs.h>
#include <string.h>
#include <ph_TypeDefs.h>

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
#include "../phalMfIdentity_Int.h"
#include "phalMfIdentity_Sam_NonX_Int.h"

phStatus_t phalMfIdentity_Sam_NonX_Int_ValidateResponse(void * pDataParams, uint16_t wStatus, uint16_t wPiccRetCode)
{
	uint16_t	PH_MEMLOC_REM wPiccStat = 0;

	/* Evaluate the response. */
	if ((wStatus == PH_ERR_SUCCESS) || 
		((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_GEN) ||
		((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_DESFIRE_GEN))
	{
		/* Evaluate the PICC Status. */
		wPiccStat = (uint16_t) (((wPiccRetCode & 0xFF00) == 0x9100) ? (wPiccRetCode & 0x00FF) : wPiccRetCode);

		/* Validate the PICC Status. */
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Int_ComputeErrorResponse(pDataParams, wPiccStat));
	}
	else
	{
		if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
		{
			wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_AL_MFIDENTITY);
		}

		PH_CHECK_SUCCESS(wStatus);
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_NonX_Int_CardExchange(void * pDataParams, uint16_t bWrapCmd, uint16_t wBufferOption, uint16_t wTotDataLen, uint8_t bExchangeLE, 
	uint8_t * pData, uint16_t wDataLen, uint8_t ** ppResponse, uint16_t * pRespLen, uint8_t * pPiccErrCode)
{
	phStatus_t		PH_MEMLOC_REM wStatus = 0;
	phStatus_t		PH_MEMLOC_REM wPICCStatus = 0;
	uint16_t		PH_MEMLOC_REM wLc = 0;
	uint16_t		PH_MEMLOC_REM wRespLen = 0;
	uint8_t*		PH_MEMLOC_REM pResponse = NULL;
		
	uint8_t			PH_MEMLOC_REM aLc[1] = {0x00};
	uint8_t			PH_MEMLOC_REM aLe[1] = {0x00};
	uint8_t			PH_MEMLOC_REM aISO7816Header[8] = {PHAL_MFIDENTITY_WRAPPEDAPDU_CLA, 0x00, PHAL_MFIDENTITY_WRAPPEDAPDU_P1, PHAL_MFIDENTITY_WRAPPEDAPDU_P2};
	uint8_t			PH_MEMLOC_REM bISO7816HeaderLen = 4;
	
	/* Exchange the command in Iso7816 wrapped formmat. ----------------------------------------------------------------- */
	if((wBufferOption == PH_EXCHANGE_BUFFER_FIRST) || (wBufferOption == PH_EXCHANGE_DEFAULT) || (wBufferOption == PH_EXCHANGE_RXCHAINING))
	{
		if(bWrapCmd)
		{
			/* Set the LC information. */
			wLc = (uint16_t) (wTotDataLen - 1 /* Excluding the command code. */);

			/* Update the command code to Iso7816 header */
			aISO7816Header[1] = pData[0];

			/* Add the ISO 7816 header to layer 4 buffer. */
			PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
				PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
				PH_EXCHANGE_BUFFER_FIRST,
				&aISO7816Header[0],
				bISO7816HeaderLen,
				NULL,
				NULL));

			/* Add Lc if available */
			if(wLc)
			{
				aLc[0] = (uint8_t) (wLc & 0x00FF);

				/* Add the Lc to layer 4 buffer. */
				PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
					PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
					PH_EXCHANGE_BUFFER_CONT,
					aLc,
					1,
					NULL,
					NULL));

				/* Add the data to layer 4 buffer. */
				PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
					PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
					PH_EXCHANGE_BUFFER_CONT,
					&pData[1],	/* Exclude the command code because it is added to INS. */
					(uint16_t) (wDataLen - 1),
					NULL,
					NULL));
			}
		}
	}
	
	if((wBufferOption == PH_EXCHANGE_BUFFER_CONT))
	{
		/* Add the data to layer 4 buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
			PH_EXCHANGE_BUFFER_CONT,
			pData,
			wDataLen,
			NULL,
			NULL));
	}

	if((wBufferOption == PH_EXCHANGE_BUFFER_LAST) || (wBufferOption == PH_EXCHANGE_DEFAULT) || (wBufferOption == PH_EXCHANGE_RXCHAINING))
	{
		if((wBufferOption == PH_EXCHANGE_BUFFER_LAST) || !bWrapCmd)
		{
			/* Add the data to layer 4 buffer. */
			PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
				PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
				(uint16_t) (bWrapCmd ? PH_EXCHANGE_BUFFER_CONT : PH_EXCHANGE_BUFFER_CONT),
				pData,
				wDataLen,
				NULL,
				NULL));
		}

		/* Add Le to L4 buffer and exchange the command. */
		wStatus = phpalMifare_ExchangeL4(
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
			PH_EXCHANGE_BUFFER_LAST,
			aLe,
			(uint8_t) (bExchangeLE ?  1 : 0),
			&pResponse,
			&wRespLen);
		
		/* Create memory for updating the response of ISO 14443 format. */
		*ppResponse = pResponse;

		/* Proceed with validation for SUCCESS case. */
		if(wStatus != PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_PAL_ISO14443P4))
		{
			/* Combine Sw1 and Sw2 status codes. */
			wPICCStatus = (uint16_t) ((pResponse[wRespLen - 2] << 8) | pResponse[wRespLen - 1]);

			/* Evaluate the response. */
			wStatus = phalMfIdentity_Sam_NonX_Int_ValidateResponse(pDataParams, wStatus, wPICCStatus);

			/* Update the response buffer length excluding SW1SW2. */
			*pRespLen = wRespLen - 2;
					
			/* Copy the second byte of response (SW2) to RxBuffer */
			*pPiccErrCode = pResponse[wRespLen - 1];
		}
		else
		{
			*pRespLen = wRespLen;
		}
	}

	return wStatus;
}

phStatus_t phalMfIdentity_Sam_NonX_Int_AuthenticatePICC(void * pDataParams, uint8_t bAuthType, uint8_t bAuthMode, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer, 
	uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pPcdCapsIn, uint8_t bPcdCapsInLen, uint8_t * pPCDCap2, uint8_t * pPDCap2)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	phStatus_t	PH_MEMLOC_REM wStatus1 = 0;
	uint8_t     PH_MEMLOC_REM bP1 = 0;
	uint16_t	PH_MEMLOC_REM wCmdLen = 0;
	uint8_t		PH_MEMLOC_REM bIsPcdCapsInAvailable = PH_OFF;
	uint8_t *	PH_MEMLOC_REM pCardResponse = NULL;
	uint16_t	PH_MEMLOC_REM wCardRespLen = 0;
	uint16_t	PH_MEMLOC_REM wCardRespOffset = 0;
	uint8_t*	PH_MEMLOC_REM pSamResponse = NULL;
	uint16_t	PH_MEMLOC_REM wSamRespLen = 0;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;
	uint8_t		PH_MEMLOC_REM aCmdBuff[34];
	uint8_t		PH_MEMLOC_REM aAppId[3] = {0x00, 0x00, 0x00};
	
#ifdef RDR_LIB_PARAM_CHECK
	if(memcmp(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00)
	{
		/* At PICC level valid key numbes are 1 - 4. */
		if((bKeyNoCard > PHAL_MFIDENTITY_ORIGINALITY_KEY_LAST ) || (bKeyNoCard < PHAL_MFIDENTITY_ORIGINALITY_KEY_FIRST))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}
	}
	else
	{
		/* At App level valid key numbers are 0 - 4. */
		if((bKeyNoCard > PHAL_MFIDENTITY_APP_KEY_LAST))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}
	}

	/* Change for valid diversification options. */
	if ((wOption != PHAL_MFIDENTITY_NO_DIVERSIFICATION) &&      
		(wOption != PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}

	/* Validate diversification input length. */
	if ((wOption != PHAL_MFIDENTITY_NO_DIVERSIFICATION) && (bDivInputLen > 31 ))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}
#endif

    /* Check for valid SAM keystore number and version. */
    if ((wKeyNo > 0x7f) || (wKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }
		
	/* Clear the command buffer and length. */
	wCmdLen = 0;
	memset(aCmdBuff, 0x00, sizeof(aCmdBuff));	/* PRQA S 3200 */

/* Frame the command buffer to be exchanged with PICC---------------------------------------------------------------------------------- */ 

	/* Add the Auth code to Command Buffer . */
	aCmdBuff[wCmdLen++] = bAuthType;
	aCmdBuff[wCmdLen++] = bKeyNoCard;

	/* Append PCD imput capabilitites in case of EV2 First auth. */
	if (bAuthType == PHAL_MFIDENTITY_CMD_AUTHENTICATE_EV2_FIRST)
	{
		/* Append LenCap Byte. */
		aCmdBuff[wCmdLen++] = bPcdCapsInLen;

		/* Set the PCD input availability flag. */
		bIsPcdCapsInAvailable = PH_ON;
		
		/* Reset the Authentication state. */
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_NonX_Int_ResetAuthStatus(pDataParams));
	}

	/* Buffer the command with the card. */
	wStatus = phalMfIdentity_Sam_NonX_Int_CardExchange(
		pDataParams, 
		PH_ON,
		PH_EXCHANGE_BUFFER_FIRST,
		(uint16_t) (wCmdLen + (bIsPcdCapsInAvailable ? bPcdCapsInLen : 0)),
		PH_OFF,
		aCmdBuff,
		wCmdLen, 
		&pCardResponse, 
		&wCardRespLen, 
		&bPiccErrCode);

	/* Buffer PCD Input capabilities and exchange the bufferred information to card. */
	wStatus = phalMfIdentity_Sam_NonX_Int_CardExchange(
		pDataParams, 
		PH_ON,
		PH_EXCHANGE_BUFFER_LAST,
		0,
		PH_ON,
		pPcdCapsIn,
		(uint16_t) (bIsPcdCapsInAvailable ? bPcdCapsInLen : 0), 
		&pCardResponse, 
		&wCardRespLen, 
		&bPiccErrCode);

	/* Validate the response for chaining. */
	if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
	{
		return wStatus;
	}

/* First part of Exchange with Sam hardware. ------------------------------------------------------------------------------------------- */

	/* Set Auth Type. */
	bP1 = (uint8_t) ((bAuthType == PHAL_MFIDENTITY_CMD_AUTHENTICATE_EV2_FIRST) ? PHHAL_HW_CMD_SAMAV3_AUTH_MODE_EV2_FIRST_AUTH :
						   PHHAL_HW_CMD_SAMAV3_AUTH_MODE_EV2_NON_FIRST_AUTH);

	/* Set Auth mode with diversification enabled. */
	bP1 |= (uint8_t) ((wOption != PHAL_MFIDENTITY_NO_DIVERSIFICATION) ? PHHAL_HW_CMD_SAMAV3_KEY_DIVERSIFICATION_ON : 
		PHHAL_HW_CMD_SAMAV3_KEY_DIVERSIFICATION_OFF);

	/* Set Diversification flags. 
	 * AV2 compatibility mode key diversification methods, 3TDEA, AES key
	 */
    if (wOption == PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS)
    {
        bP1 |= (uint8_t) PHHAL_HW_CMD_SAMAV3_KDF_AV2;
	}

	/* Set the secure messaging. */
	if((bKeyNoCard >= PHAL_MFIDENTITY_ORIGINALITY_KEY_FIRST) && (bKeyNoCard <= PHAL_MFIDENTITY_ORIGINALITY_KEY_LAST) &&
		(memcmp(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00))
	{
		bP1 = (uint8_t) (bP1 | PHHAL_HW_CMD_SAMAV3_SUPPRESS_SECURE_MESSAGING);
	}
		
	/* Set the Offset to use. */
	wCardRespOffset = (uint16_t) ((bAuthMode == PHHAL_HW_CMD_SAMAV3_AUTH_MODE_LRP) ? 1 : 0);

	wStatus1 = phhalHw_SamAV3_Cmd_SAM_AuthenticatePICC_Part1(
		PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
		bP1,
		(uint8_t) wKeyNo,
		(uint8_t) wKeyVer,
		bAuthMode,
		pDivInput,
		bDivInputLen,
		&pCardResponse[wCardRespOffset],
		(uint8_t) (wCardRespLen - wCardRespOffset),
		&pSamResponse,
		&wSamRespLen);

	/* Check for the Chaining active */
	if ((wStatus1 & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
	{
		return wStatus1;
	}




/* Second part of Exchange with card. -------------------------------------------------------------------------------------------------- */
	wCmdLen = 0;
	bPiccErrCode = 0;
	memset(aCmdBuff, 0x00, sizeof(aCmdBuff));	/* PRQA S 3200 */

	/* Frame the command for Exchange to card. */
	aCmdBuff[wCmdLen++] = 0xAF;
	
	/* Copy the response received from SAM to Command buffer. */
	memcpy(&aCmdBuff[wCmdLen], pSamResponse, wSamRespLen);	/* PRQA S 3200 */
	wCmdLen += wSamRespLen;

	/* Exchange the command with the card. */
	wStatus = phalMfIdentity_Sam_NonX_Int_CardExchange(
		pDataParams, 
		PH_ON,
		PH_EXCHANGE_DEFAULT,
		wCmdLen,
		PH_ON,
		aCmdBuff,
		wCmdLen, 
		&pCardResponse, 
		&wCardRespLen, 
		&bPiccErrCode);

	/* Second part of Exchange with Sam hardware. ----------------------------------------------------------------- */
	wStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticatePICC_Part2(
		PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
		bPiccErrCode,
		pCardResponse,
		(uint8_t) wCardRespLen,
		pPDCap2,
		pPCDCap2,
		&bPiccErrCode);

	/* Validate the PICC error. */
	wStatus = phalMfIdentity_Sam_NonX_Int_ValidateResponse(pDataParams, wStatus, bPiccErrCode);
	
	if(wStatus == PH_ERR_SUCCESS)
	{
		/* Do not update the auth state if originality keys are used. */
		if((bKeyNoCard >= PHAL_MFIDENTITY_ORIGINALITY_KEY_FIRST) && (bKeyNoCard <= PHAL_MFIDENTITY_ORIGINALITY_KEY_LAST) &&
			(memcmp(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00))
		{
			PH_CHECK_SUCCESS_FCT(wStatus1, phalMfIdentity_Sam_NonX_Int_ResetAuthStatus(pDataParams));
		}
		else
		{
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode = PHAL_MFIDENTITY_AUTHENTICATEEV2;
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo = bKeyNoCard;
		}
	}

	return wStatus;
}

phStatus_t phalMfIdentity_Sam_NonX_Int_ChangeKeyPICC(void * pDataParams, uint16_t wOption, uint8_t bKeyNoCard, uint16_t wCurrKeyNo, uint16_t wCurrKeyVer, uint16_t wNewKeyNo, 
	uint16_t wNewKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	phStatus_t	PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bKeyCompMeth = 0;
    uint8_t     PH_MEMLOC_REM bCfg = 0;
	uint8_t		PH_MEMLOC_REM aCmdBuff[43];
	uint16_t	PH_MEMLOC_REM wCmdLen = 0;
	uint8_t *	PH_MEMLOC_REM pCardResponse = NULL;
	uint16_t	PH_MEMLOC_REM wCardRespLen = 0;
	uint16_t	PH_MEMLOC_REM wSamRespLen = 0;
	uint8_t *	PH_MEMLOC_REM pSamResponse = NULL;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;

#ifdef RDR_LIB_PARAM_CHECK
	uint8_t		PH_MEMLOC_REM aAppId[3] = {0x00, 0x00, 0x00};

	/* Only if seleted Aid is 0x000000. */
	if (memcmp(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00)
    {
		/* At PICC level ChangeKey cannot be performed. */
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }
    else
    {
		/* At App level, 0,1,2,3,4 are valid ones. */
		if(bKeyNoCard > PHAL_MFIDENTITY_APP_KEY_LAST)
		{
			/* Invalid application key specified */
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}
    }

	if (PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_NOT_AUTHENTICATED)
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFIDENTITY);
	}
#endif

    /* Check for valid SAM key number and version. */
    if ((wCurrKeyNo > 0x7f) || (wCurrKeyVer > 0xff) || (wNewKeyNo > 0x7f) || (wNewKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	
/* Command Exchange with SAM. ---------------------------------------------------------------------------------------------------------- */
    /* Set the key compilation method. */
	if(wOption == PHAL_MFIDENTITY_NO_DIVERSIFICATION)
	{
		bKeyCompMeth = 0x00;
	}
	else
	{
#ifdef RDR_LIB_PARAM_CHECK
		/* Change for valid diversification options. */
		if ((wOption & PHHAL_HW_CMD_SAMAV3_DIV_METHOD_AV2) != PHHAL_HW_CMD_SAMAV3_DIV_METHOD_AV2)
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}

		/* Validate diversification input length. */
		if ( bDivInputLen > 31 )
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}
#endif

		/* Assin the option to local variable. */
		bKeyCompMeth = (uint8_t) wOption;
	}

    /* Set if PICC targeted key equal to PICC authenticated key. */
    if ((PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo) == bKeyNoCard)
    {
        bKeyCompMeth = (uint8_t) (bKeyCompMeth | PHHAL_HW_CMD_SAMAV3_CRYPTO_MODE_SAME_KEY);
    }

	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ChangeKeyPICC(
		PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
		bKeyCompMeth,
		bCfg,
		0x00,
		(uint8_t) bKeyNoCard,
		(uint8_t) wCurrKeyNo,
		(uint8_t) wCurrKeyVer,
		(uint8_t) wNewKeyNo,
		(uint8_t) wNewKeyVer,
		pDivInput,
		bDivInputLen,
		&pSamResponse,
		&wSamRespLen));

	/* Command Exchange with Card. -------------------------------------------------------------------------------- */
	wCmdLen = 0;
	memset(aCmdBuff, 0x00, sizeof(aCmdBuff));	/* PRQA S 3200 */

	/* Frame the command information with command type. */
	aCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_CHANGE_KEY;

	/* Add CardKey number to command buffer. */
	aCmdBuff[wCmdLen++] = bKeyNoCard;

	/* Copy the response received from SAM to Command buffer. */
	memcpy(&aCmdBuff[wCmdLen], pSamResponse, wSamRespLen);	/* PRQA S 3200 */
	wCmdLen += wSamRespLen;

	/* Exchange the command with the card. */
	wStatus = phalMfIdentity_Sam_NonX_Int_CardExchange(
		pDataParams, 
		PH_ON,
		PH_EXCHANGE_DEFAULT,
		wCmdLen,
		PH_ON,
		aCmdBuff,
		wCmdLen, 
		&pCardResponse,
		&wCardRespLen, 
		&bPiccErrCode);
		
    /* Evaluate the response. */
	wStatus = phalMfIdentity_Sam_NonX_Int_ValidateResponse(pDataParams, wStatus, bPiccErrCode);

	/* Reset the Auth state. */
	if(wStatus != PH_ERR_SUCCESS)
	{
		/* Kill the PICC Authentication. */
		PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SamAV3_Cmd_SAM_KillAuthentication(
			PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
			PHHAL_HW_SAMAV3_CMD_SAM_KILL_AUTHENTICATION_PARTIAL));

		return wStatus;
	}
	else
	{
		/* Verify the MAC. */
		if(wCardRespLen)
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_NonX_Int_VerifySM(
				pDataParams,
				(uint16_t) (PH_EXCHANGE_DEFAULT | PHALMFIDENTITY_SAM_NONX_EXCHANGE_PICC_STATUS),
				PHAL_MFIDENTITY_COMMUNICATION_MACD,
				NULL,
				0,
				bPiccErrCode,
				pCardResponse,
				wCardRespLen,
				&pSamResponse,
				&wSamRespLen));
		}

		/* Reset authentication status only if the key authenticated with is changed. */
		if ((PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo) == bKeyNoCard)
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_NonX_Int_ResetAuthStatus(pDataParams));
		}
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_NonX_Int_GenerateSM(void * pDataParams, uint16_t wOption, uint8_t bCommMode, uint8_t * pCmdBuff, uint16_t wCmdLen, uint8_t * pData, 
	uint16_t wDataLen, uint8_t ** ppOutBuffer, uint16_t * pOutBufLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bOffset = 0;
	uint16_t	PH_MEMLOC_REM wBuffOption = 0;

	/* Exchange the information to Sam hardware to get the MAC information. 
	 * This computed MAC might not be exchanged. This is computed to initial crypto informationin SAM which will be used for MAC verification.
	 */
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode != PHAL_MFIDENTITY_NOT_AUTHENTICATED)
	{
		if(bCommMode == PHAL_MFIDENTITY_COMMUNICATION_PLAIN)
		{
			/* Buffer command counter information to SAM. */
			if(pCmdBuff[0] != 0xAF)
			{
				PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ApplySM(
					PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
					PH_EXCHANGE_DEFAULT,
					PHHAL_HW_SAMAV3_CMD_APPLY_SM_COMM_MODE_PLAIN,
					0x00,
					0x01,
					NULL,
					0,
					ppOutBuffer,
					pOutBufLen));
			}
		}
		else
		{
			/* Set the bufferring option. */
			wBuffOption = PH_EXCHANGE_BUFFER_FIRST;
			wBuffOption |= (uint16_t) (((wOption & 0xFF0F) == PH_EXCHANGE_DEFAULT) ? wBuffOption : PH_EXCHANGE_TXCHAINING);

			/* Compute the Offset to be exchagned. */
			if((bCommMode == PHAL_MFIDENTITY_COMMUNICATION_ENC) && (pCmdBuff[0] != 0xAF))
			{
				bOffset = (uint8_t) wCmdLen;
				wBuffOption |= PHHAL_HW_SAMAV3_CMD_APPLY_SM_INCLUDE_OFFSET;
			}

			/* Buffer command information to SAM. */
			PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ApplySM(
				PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
				wBuffOption,
				bCommMode,
				bOffset,
				0x00,
				pCmdBuff,
				(uint8_t) ((pCmdBuff[0] != 0xAF) ? wCmdLen : 0),
				ppOutBuffer,
				pOutBufLen));

			/* Buffer data information to SAM and exchange the infomration to received the SM information. */
			PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ApplySM(
				PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
				PH_EXCHANGE_BUFFER_LAST,
				bCommMode,
				0,
				0,
				pData,
				(uint8_t) wDataLen,
				ppOutBuffer,
				pOutBufLen));
		}
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_NonX_Int_VerifySM(void * pDataParams, uint16_t wOption, uint8_t bCommMode, uint8_t * pResponse, uint16_t wRespLen, uint8_t bPiccStat, 
	uint8_t * pRespMac, uint16_t wRespMacLen, uint8_t ** ppOutBuffer, uint16_t * pOutBufLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint16_t	PH_MEMLOC_REM wBuffOption = 0;

	/* Exchange the information to Sam hardware to get the MAC information. */
	if((PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode != PHAL_MFIDENTITY_NOT_AUTHENTICATED) && 
		(bCommMode != PHAL_MFIDENTITY_COMMUNICATION_PLAIN))
	{
		/* Set the buffering flag to Default. */
		wBuffOption = PH_EXCHANGE_BUFFER_FIRST;
		wBuffOption |= (uint16_t) (((wOption & 0xFF0F) == PH_EXCHANGE_DEFAULT) ? wBuffOption : PH_EXCHANGE_TXCHAINING);

		/* Set PICC Status to be exchanged for the first frame. */
		if(((wOption & PH_EXCHANGE_CUSTOM_BITS_MASK) == PHALMFIDENTITY_SAM_NONX_EXCHANGE_PICC_STATUS) && 
			(bPiccStat == PHAL_MFIDENTITY_RESP_ADDITIONAL_FRAME))
		{
			bPiccStat = PHAL_MFIDENTITY_RESP_OPERATION_OK;
		}

		/* Buffer the PICC status information to Sam buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_RemoveSM(
			PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
			wBuffOption,
			bCommMode,
			&bPiccStat,
			(uint8_t) (((wOption & PH_EXCHANGE_CUSTOM_BITS_MASK) == PHALMFIDENTITY_SAM_NONX_EXCHANGE_PICC_STATUS)  ? 1 : 0),
			NULL,
			NULL));

		/* Buffer the Plain response information to Sam buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_RemoveSM(
			PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
			PH_EXCHANGE_BUFFER_CONT,
			bCommMode,
			pResponse,
			(uint8_t) wRespLen,
			NULL,
			NULL));

		/* Buffer Mac and Exchagne the bufferred information to Sam hardware. */
		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_RemoveSM(
			PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
			PH_EXCHANGE_BUFFER_LAST,
			0x00,
			pRespMac,
			(uint8_t) wRespMacLen,
			ppOutBuffer,
			pOutBufLen));
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_NonX_Int_ReadData(void * pDataParams, uint8_t bCmd_ComMode, uint8_t bResp_ComMode, uint8_t * pCmdBuff, uint16_t wCmdLen, uint8_t ** ppResponse, 
	uint16_t * pRespLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	phStatus_t	PH_MEMLOC_REM wStatus1 = 0;
	uint16_t	PH_MEMLOC_REM wBuffOption = 0;
	uint16_t	PH_MEMLOC_REM wBuffOption1 = 0;
	uint8_t		PH_MEMLOC_REM bFirstFrame = 0;
	uint8_t		PH_MEMLOC_REM bLastFrame = 0;
	uint8_t		PH_MEMLOC_REM bLargeData = 0;
	uint8_t		PH_MEMLOC_REM bFinished = 0;
	uint8_t		PH_MEMLOC_REM bFinished1 = 0;
	uint8_t	*	PH_MEMLOC_REM pMac = NULL;
	uint16_t	PH_MEMLOC_REM wMacLen = 0;
	uint16_t	PH_MEMLOC_REM wOffset = 0;
	uint16_t	PH_MEMLOC_REM wTotLen = 0;
	uint16_t	PH_MEMLOC_REM wRemData = 0;
	uint8_t	*	PH_MEMLOC_REM pCardResponse = NULL;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wCardRespLen = 0;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;
	
#ifdef RDR_LIB_PARAM_CHECK
	if ((bCmd_ComMode == PHAL_MFIDENTITY_COMMUNICATION_ENC) && (PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_NOT_AUTHENTICATED))
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFIDENTITY);
	}
#endif 

	/* Secure the information to be exchanged. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_NonX_Int_GenerateSM(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		bCmd_ComMode,
		pCmdBuff,
		wCmdLen,
		NULL,
		0,
		&pMac,
		&wMacLen));

	/* Frame the total length. */
	wTotLen = (uint16_t) (wCmdLen + wMacLen);

	/* Set exchange optionto First. */
	wBuffOption = PH_EXCHANGE_BUFFER_FIRST;

	do
	{
		/* Buffer the command information. */
		wStatus1 = phalMfIdentity_Sam_NonX_Int_CardExchange(
			pDataParams, 
			PH_ON,
			wBuffOption,
			wTotLen,
			PH_ON,
			pCmdBuff,
			wCmdLen, 
			&pCardResponse,
			&wCardRespLen, 
			&bPiccErrCode);
	
		/* Buffer the Mac information and exchange the complete information to PICC. */
		if(wBuffOption != PH_EXCHANGE_DEFAULT)
		{
			wStatus1 = phalMfIdentity_Sam_NonX_Int_CardExchange(
				pDataParams, 
				PH_ON,
				PH_EXCHANGE_BUFFER_LAST,
				0,
				PH_ON,
				pMac,
				wMacLen, 
				&pCardResponse,
				&wCardRespLen, 
				&bPiccErrCode);

			/* Set First Frame. */
			bFirstFrame = PH_ON;

			/* Subtract the total length with MAC. */
			wTotLen -= wMacLen;
		}
		
		/* Evaluate the response. */
		wStatus1 = phalMfIdentity_Sam_NonX_Int_ValidateResponse(pDataParams, wStatus1, bPiccErrCode);

		/* Set the last frame to end the looping. */
		bLastFrame = (uint8_t) ((wStatus1 == PH_ERR_SUCCESS) ? PH_ON : PH_OFF);

		/* Reset the command length. */
		wCmdLen = 0;

		/* Update command information. */
		if(!bLastFrame)
		{
			pCmdBuff[0] = PHAL_MFIDENTITY_RESP_ADDITIONAL_FRAME;
			wCmdLen = 1;

			/* Update the Total length. */
			wTotLen = 1;
		}

		/* Set the exchange option to RxChaining if there is still more information to be exchanged. */
		wBuffOption = PH_EXCHANGE_DEFAULT;

		/* Update the variables and parameters. */
		if(ppResponse != NULL)
		{
			if(ppResponse[0] != NULL)
			{
				memcpy(&ppResponse[0][wOffset], pCardResponse, (bLastFrame ? (wCardRespLen - wMacLen ) : wCardRespLen));	/* PRQA S 3200 */
			}
			else
			{
				ppResponse[0] = pCardResponse;
			}

			*pRespLen += wCardRespLen;
			wOffset += (uint16_t) wCardRespLen;
		}

		/* Set Largedata flag. */
		bLargeData = (uint8_t) ((wCardRespLen > PHALMFIDENTITY_SAM_DATA_FRAME_LENGTH) ? PH_ON : PH_OFF);

		/* Reset the Auth state of PICC only. */
		if((wStatus1 != PH_ERR_SUCCESS) && ((wStatus1 & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_NonX_Int_ResetAuthStatus(pDataParams));
			bFinished = PH_ON;
		}
		else
		{
			/* Exchange the data to SAM in chunks in case of large amount of data. */
			if(bLargeData)
			{
				bFirstFrame = PH_ON;
				bLastFrame = PH_OFF;
				wTotLen = wCardRespLen;
				wRemData = wTotLen;
				wOffset = 0;
				wCardRespLen = PHALMFIDENTITY_SAM_DATA_FRAME_LENGTH;
			}

			do
			{
				/* Set the information for the last frame to be exchanged. */
				if(bLargeData)
				{
					if(wRemData < PHALMFIDENTITY_SAM_DATA_FRAME_LENGTH)
					{
						wCardRespLen = wRemData;
						bLastFrame = PH_ON;
						bFinished1 = PH_ON;
					}
				}
				else
				{
					bFinished1 = PH_ON;
				}

				/* Set the buffering options. */
				wBuffOption1 = (uint16_t) (bLastFrame ? PH_EXCHANGE_DEFAULT : PH_EXCHANGE_RXCHAINING);

				/* Set the PICC status utilization. */
				if(bFirstFrame)
				{
					wBuffOption1 |= (uint16_t) PHALMFIDENTITY_SAM_NONX_EXCHANGE_PICC_STATUS;
				}
					
				/* Set the Mac Length. */
				if(bResp_ComMode != PHAL_MFIDENTITY_COMMUNICATION_PLAIN)
				{
					wMacLen = (uint16_t) (((PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_NOT_AUTHENTICATED) ||
						!bLastFrame) ? 0 : 8);
				}

				/* Verify the security of the received information. */
				wStatus = phalMfIdentity_Sam_NonX_Int_VerifySM(
					pDataParams,
					wBuffOption1,
					bResp_ComMode,
					&pCardResponse[bLargeData ? wOffset : 0],
					(uint16_t) (wCardRespLen - (bLastFrame ? wMacLen : 0)),
					bPiccErrCode,
					&pCardResponse[bLargeData ? (wTotLen - (bLastFrame ? wMacLen : 0)) : (wCardRespLen - (bLastFrame ? wMacLen : 0))],
					(uint16_t) (bLastFrame ? wMacLen : 0),
					&pResponse,
					&wRespLen);

				/* Copy the response to the buffer. */
				if((wStatus == PH_ERR_SUCCESS) || ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING))
				{
					if(bResp_ComMode == PHAL_MFIDENTITY_COMMUNICATION_ENC)
					{
						/* Reset the length buffer. */
						if(bFirstFrame)
						{
							*pRespLen = 0;
						}

						memcpy(&ppResponse[0][*pRespLen], pResponse, wRespLen);
						*pRespLen = (uint16_t) (bLargeData ? (*pRespLen + wRespLen) : wRespLen);
					}
				}

				/* Subtract if Mac is available. */
				if(pRespLen != NULL)
				{
					*pRespLen -= (uint16_t) ((bLastFrame && (bResp_ComMode != PHAL_MFIDENTITY_COMMUNICATION_ENC)) ? wMacLen : 0);
				}

				/* Validate the status. */
				if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
				{
					PH_CHECK_SUCCESS(wStatus);
				}

				/* Update offset for large amount of data only. */
				if(bLargeData)
				{
					/* Update the offsets and length. */
					wOffset += PHALMFIDENTITY_SAM_DATA_FRAME_LENGTH;

					/* Set the remaining data length to be exchanged. */
					wRemData -= PHALMFIDENTITY_SAM_DATA_FRAME_LENGTH;	
				}
					
				/* Clear First Frame. */
				bFirstFrame = PH_OFF;
			}while(!bFinished1);
		}

		/* Set finished flag. */
		if(wStatus1 == PH_ERR_SUCCESS)
		{
			bFinished = PH_ON;
		}
	}while(!bFinished);

	return wStatus1;
}

phStatus_t phalMfIdentity_Sam_NonX_Int_WriteData(void * pDataParams, uint16_t wOption, uint8_t bIsDataCmd, uint8_t bCmd_ComMode, uint8_t bResp_ComMode, uint8_t bResetAuth, 
	uint8_t * pCmdBuff, uint16_t wCmdLen, uint8_t * pData, uint32_t dwDataLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	phStatus_t	PH_MEMLOC_REM wStatus1 = 0;
	uint16_t	PH_MEMLOC_REM wTotLen = 0;
	uint8_t		PH_MEMLOC_REM aCmdBuf_Tmp[50];
	uint16_t	PH_MEMLOC_REM wCmdLen_Tmp= 0;
	uint8_t		PH_MEMLOC_REM bExchangeMac = 0;
	uint8_t	*	PH_MEMLOC_REM pSMData = NULL;
	uint16_t	PH_MEMLOC_REM wSMDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pCardResponse = NULL;
	uint16_t	PH_MEMLOC_REM wCardRespLen = 0;
	uint16_t	PH_MEMLOC_REM wCardRespOffset = 0;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;
	uint16_t	PH_MEMLOC_REM wPICCFrameLen = 0;
	uint8_t		PH_MEMLOC_REM bFirstFrame = PH_ON;
	uint8_t		PH_MEMLOC_REM bLastFrame = PH_OFF;
	uint8_t		PH_MEMLOC_REM bDataLen = 0;
	uint8_t		PH_MEMLOC_REM bIns = 0;
	uint8_t		PH_MEMLOC_REM bIsLargeData = 0;
	uint32_t	PH_MEMLOC_REM dwRemLen = 0;
		
	uint16_t	PH_MEMLOC_REM wBuffOption_PICC = 0;
	uint8_t		PH_MEMLOC_REM bFinished_PICC = PH_OFF;
	uint8_t		PH_MEMLOC_REM bPiccExchangeComplete = PH_OFF;
	uint32_t	PH_MEMLOC_REM dwOffset_PICC = 0;
	
	uint16_t	PH_MEMLOC_REM wBuffOption_SAM = 0;
	uint8_t		PH_MEMLOC_REM bFrameLen_SAM = 0;
	uint32_t	PH_MEMLOC_REM dwOffset_SAM = 0;
	uint32_t	PH_MEMLOC_REM dwRemLen_SAM = 0;
	uint8_t		PH_MEMLOC_REM bSamExchangeComplete = PH_OFF;
	uint8_t		PH_MEMLOC_REM bFinished_SAM = PH_OFF;
	
#ifdef RDR_LIB_PARAM_CHECK
	if ((bCmd_ComMode == PHAL_MFIDENTITY_COMMUNICATION_ENC) && (PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_NOT_AUTHENTICATED))
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFIDENTITY);
	}
#endif 

	/* Save the command information and it length because in course . */
	memcpy(aCmdBuf_Tmp, pCmdBuff, wCmdLen);
	wCmdLen_Tmp = wCmdLen;
		
	/* Set the Initial Frame length. */
	bFrameLen_SAM = (uint8_t) dwDataLen;
	dwRemLen_SAM = dwDataLen;

	/* Get the PICC Frame length. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_NonX_Int_GetFrameLen(
		pDataParams,
		&wPICCFrameLen));

	/* Set INS flag if Write command are ISO chaining based.*/
	bIns = (uint8_t) (((pCmdBuff[0] == PHAL_MFIDENTITY_CMD_WRITE_DATA_ISO) || (pCmdBuff[0] == PHAL_MFIDENTITY_CMD_WRITE_RECORD_ISO)) ? PH_ON : PH_OFF);

	do
	{
		/* Encrypt the information. */
		if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode != PHAL_MFIDENTITY_NOT_AUTHENTICATED)
		{
			if((dwRemLen_SAM > PHALMFIDENTITY_SAM_DATA_FRAME_LENGTH))
			{
				bFrameLen_SAM = PHALMFIDENTITY_SAM_DATA_FRAME_LENGTH;
				wBuffOption_SAM = PH_EXCHANGE_TXCHAINING;
			}
			else
			{
				bFrameLen_SAM = (uint8_t) dwRemLen_SAM;
				wBuffOption_SAM = (uint16_t) ((wOption & PH_EXCHANGE_TXCHAINING) ? PH_EXCHANGE_TXCHAINING : PH_EXCHANGE_DEFAULT);
				bFinished_SAM = PH_ON;
			}

			wStatus = phalMfIdentity_Sam_NonX_Int_GenerateSM(
				pDataParams,
				wBuffOption_SAM,
				bCmd_ComMode,
				aCmdBuf_Tmp,
				wCmdLen_Tmp,
				&pData[dwOffset_SAM],
				bFrameLen_SAM, 
				&pSMData,
				&wSMDataLen);

			/* Validate the status. */
			if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
			{
				return wStatus;
			}
			else
			{
				if(wStatus == PH_ERR_SUCCESS)
				{
					/* Set Sam complete exchagne flag. */
					bSamExchangeComplete = PH_ON;

					if(wSMDataLen < wPICCFrameLen)
					{
						bIsLargeData = PH_OFF;
						dwRemLen = (uint8_t) ((wOption & PHALMFIDENTITY_SAM_NONX_EXCHANGE_DATA_PICC) ? dwDataLen : 0);

						/* Set if Mac on command is required. */
						bExchangeMac = (uint8_t) ((bCmd_ComMode == PHAL_MFIDENTITY_COMMUNICATION_MACD) ? PHALMFIDENTITY_SAM_NONX_MAC_ON_CMD :
							PHALMFIDENTITY_SAM_NONX_NO_MAC_ON_CMD);
					}
					else
					{
						bFirstFrame = (uint8_t) ((bCmd_ComMode == PHAL_MFIDENTITY_COMMUNICATION_ENC) ? PH_ON : PH_OFF);
					}
				}
			}

			/* Update the lengths. */
			dwRemLen_SAM = dwRemLen_SAM - PHALMFIDENTITY_SAM_DATA_FRAME_LENGTH;
			dwOffset_SAM += PHALMFIDENTITY_SAM_DATA_FRAME_LENGTH;
		}
		else
		{
			bFinished_SAM = PH_ON;
			bSamExchangeComplete = PH_ON;
		}
		
		/* Set First Frame. */
		if(bCmd_ComMode == PHAL_MFIDENTITY_COMMUNICATION_ENC)
		{
			bFirstFrame = PH_ON;
		}

		if(!bPiccExchangeComplete)
		{
			do
			{
				if(bIsDataCmd)
				{
					/* Get the frame size that can be transmitted to PICC. */
					if(bFirstFrame)
					{
						/* Set the lengths. */
						dwRemLen = (bCmd_ComMode == PHAL_MFIDENTITY_COMMUNICATION_ENC) ? wSMDataLen : dwDataLen;
					}

					/* Performing chunk exchange if large data flag is set. */
					if(bIsLargeData)
					{
						bDataLen = (uint8_t) (wPICCFrameLen - wCmdLen_Tmp - 6 /* ISO7816 Header */);

						/* Set the completion flag. */
						if(dwRemLen <= bDataLen)
						{
							bDataLen = (uint8_t) dwRemLen;
							bFinished_PICC = PH_ON;
							bLastFrame = PH_ON;
							dwRemLen = 0;
						}
					}
					else
					{
						bFinished_PICC = PH_ON;
						bLastFrame = PH_ON;
						bDataLen = (uint8_t) dwRemLen;
					}

					/* Set PICC Exchange complete for MAC and PLAIN communication. */
					bPiccExchangeComplete = (uint8_t) ((bCmd_ComMode == PHAL_MFIDENTITY_COMMUNICATION_ENC) ? PH_OFF : PH_ON);
				}
				else
				{
					bFinished_PICC = PH_ON;
					bLastFrame = PH_ON;
					bDataLen = (uint8_t) dwRemLen;

					if(!bIsDataCmd && (bCmd_ComMode == PHAL_MFIDENTITY_COMMUNICATION_ENC))
					{
						bDataLen = (uint8_t) wSMDataLen;
					}
				}

				/* Frame the total length. */
				wTotLen = 0;
				wTotLen = (uint16_t) (wCmdLen_Tmp + ((bCmd_ComMode == PHAL_MFIDENTITY_COMMUNICATION_ENC) ? bDataLen : dwDataLen));
				wTotLen = (uint16_t) ((bExchangeMac && bLastFrame) ? ( wTotLen + wSMDataLen) : wTotLen);

				/* Update the total length. This will be used to update the actual LC information of ISO7816 format. */
				if(bIns)
				{
					if(bCmd_ComMode == PHAL_MFIDENTITY_COMMUNICATION_MACD)
					{
						wTotLen = (uint16_t) (wCmdLen_Tmp + dwRemLen);
						wTotLen = (uint16_t) (wTotLen + 8);
					}
				}

				/* Set the Bufferring option. */
				wBuffOption_PICC = PH_EXCHANGE_BUFFER_FIRST;
				wBuffOption_PICC = (uint16_t) ((bIns && !wCmdLen_Tmp) ? PH_EXCHANGE_BUFFER_CONT : wBuffOption_PICC);

				/* Buffer the command information. */
				PH_CHECK_SUCCESS_FCT(wStatus1, phalMfIdentity_Sam_NonX_Int_CardExchange(
					pDataParams, 
					PH_ON,
					wBuffOption_PICC,
					wTotLen,
					PH_OFF,
					aCmdBuf_Tmp,
					wCmdLen_Tmp, 
					NULL,
					NULL, 
					NULL));

				/* Buffer the data information. */
				if(bCmd_ComMode != PHAL_MFIDENTITY_COMMUNICATION_ENC)
				{
					PH_CHECK_SUCCESS_FCT(wStatus1, phalMfIdentity_Sam_NonX_Int_CardExchange(
						pDataParams, 
						PH_ON,
						PH_EXCHANGE_BUFFER_CONT,
						0,
						PH_OFF,
						&pData[dwOffset_PICC],
						(uint16_t) (bIsLargeData ? bDataLen : dwRemLen), 
						NULL,
						NULL, 
						NULL));
				}
	
				/* Set the Bufferring option. */
				wBuffOption_PICC = (uint16_t) ((bIns && bLastFrame && bSamExchangeComplete) ? PH_EXCHANGE_BUFFER_LAST : PH_EXCHANGE_BUFFER_CONT);
				wBuffOption_PICC = (uint16_t) (bIns ? wBuffOption_PICC : PH_EXCHANGE_BUFFER_LAST);

				/* Buffer the Mac information exchange the complete information to PICC. */
				wStatus1 = phalMfIdentity_Sam_NonX_Int_CardExchange(
					pDataParams, 
					PH_ON,
					wBuffOption_PICC,
					0,
					PH_ON,
					&pSMData[(bCmd_ComMode == PHAL_MFIDENTITY_COMMUNICATION_ENC) ? dwOffset_PICC : 0],
					(uint16_t) ((bCmd_ComMode == PHAL_MFIDENTITY_COMMUNICATION_ENC) ? bDataLen : ((bLastFrame && bExchangeMac) ? wSMDataLen : 0)),
					&pCardResponse,
					&wCardRespLen, 
					&bPiccErrCode);

				/* Validate the status. */
				if((wStatus1 != PH_ERR_SUCCESS) && ((wStatus1 & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
				{
					bFinished_PICC = PH_ON;
				}

				/* Reset the command information. */
				wCmdLen_Tmp = 0;
				aCmdBuf_Tmp[0] = PHAL_MFIDENTITY_RESP_ADDITIONAL_FRAME;

				if(bIsDataCmd && bIsLargeData)
				{
					/* Clear the First frame flag. */
					bFirstFrame = PH_OFF;

					/* Update length. */
					dwOffset_PICC += bDataLen;
					dwRemLen = (uint32_t) (dwRemLen - bDataLen);
					
					if(!bIns)
					{
						wCmdLen_Tmp++;
					}
				}
			}while(!bFinished_PICC);

			/* Reset the variables. */
			dwOffset_PICC = 0;
			bDataLen = 0;
			bFinished_PICC = PH_OFF;
		}
		else
		{
			if(wSMDataLen && (bCmd_ComMode != PHAL_MFIDENTITY_COMMUNICATION_PLAIN))
			{
				wCmdLen_Tmp = 0;

				/* Add the additional frame information. */
				wBuffOption_PICC = PH_EXCHANGE_BUFFER_LAST;
				if(!bIns)
				{
					aCmdBuf_Tmp[wCmdLen_Tmp++] = PHAL_MFIDENTITY_RESP_ADDITIONAL_FRAME;
					wBuffOption_PICC = PH_EXCHANGE_DEFAULT;
				}

				/* Copy the MAC information */
				memcpy(&aCmdBuf_Tmp[wCmdLen_Tmp], pSMData, wSMDataLen);	/* PRQA S 3200 */
				wCmdLen_Tmp += wSMDataLen;

				/* Buffer the command information. */
				wStatus1 = phalMfIdentity_Sam_NonX_Int_CardExchange(
					pDataParams, 
					PH_ON,
					wBuffOption_PICC,
					wCmdLen_Tmp,
					PH_ON,
					aCmdBuf_Tmp,
					wCmdLen_Tmp,
					&pCardResponse,
					&wCardRespLen, 
					&bPiccErrCode);
			}
		}
	}while(!bFinished_SAM);

	/* Reset the Authentication. */
	if(bResetAuth && (wStatus1 == PH_ERR_SUCCESS))
	{
		if (PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_NonX_Int_ResetAuthStatus(pDataParams));
		}
	}

	/* Verify the security of the received information. */
	else
	{
		/* Reset the Authentication state if there is an error. */
		if(wStatus1 != PH_ERR_SUCCESS)
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_NonX_Int_ResetAuthStatus(pDataParams));
		}
		else
		{
			if(bResp_ComMode != PHAL_MFIDENTITY_COMMUNICATION_PLAIN)
			{
				/* Update the Additional Information flag. */
				wCardRespOffset = (uint16_t) (((pCmdBuff[0] == PHAL_MFIDENTITY_CMD_COMMIT_TXN) && (wCmdLen == 2) && (pCmdBuff[1] & 0x01)) ? 12 : 0);

				PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_NonX_Int_VerifySM(
					pDataParams,
					(uint16_t) (PH_EXCHANGE_DEFAULT | PHALMFIDENTITY_SAM_NONX_EXCHANGE_PICC_STATUS),
					bResp_ComMode,
					pCardResponse,
					wCardRespOffset,
					bPiccErrCode,
					&pCardResponse[wCardRespOffset],
					(uint16_t) (wCardRespLen - wCardRespOffset),
					ppResponse,
					pRespLen));

				if((pRespLen != NULL) && (bResp_ComMode != PHAL_MFIDENTITY_COMMUNICATION_ENC))
				{
					*ppResponse = pCardResponse;
					*pRespLen = wCardRespLen;
				}
			}
		}
	}
		
	/* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_NonX_Int_ValidateResponse(pDataParams, wStatus1, bPiccErrCode));

	return wStatus1;
}

phStatus_t phalMfIdentity_Sam_NonX_Int_CreateTMFilePICC(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t bFileOption, uint8_t * pAccessRights, uint8_t bTMKeyOption, 
	uint8_t bKeyNo, uint8_t bKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	phStatus_t	PH_MEMLOC_REM wStatus1 = 0;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;
	uint16_t	PH_MEMLOC_REM wCmdLen = 0;
	uint16_t	PH_MEMLOC_REM wCardRespLen = 0;
	uint16_t	PH_MEMLOC_REM wSamRespLen = 0;

	uint8_t*	PH_MEMLOC_REM pCardResponse = NULL;
	uint8_t*	PH_MEMLOC_REM pSamResponse = NULL;
	uint8_t		PH_MEMLOC_REM aCmdBuff[8];
	
#ifdef RDR_LIB_PARAM_CHECK
    /* Check for valid file no. and KeyType == AES128 */
	if (bTMKeyOption != PHAL_MFIDENTITY_KEY_TYPE_AES128)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

    if ((bFileOption != PHAL_MFIDENTITY_COMMUNICATION_PLAIN) &&
		(bFileOption != PHAL_MFIDENTITY_COMMUNICATION_PLAIN_1) &&
        (bFileOption != PHAL_MFIDENTITY_COMMUNICATION_ENC) &&
        (bFileOption != PHAL_MFIDENTITY_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }
#endif

	/* Check for valid SAM keystore number and version. */
    if (bKeyNo > 0x7f)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	/* Command Exchange with Card. -------------------------------------------------------------------------------- */
	wCmdLen = 0;
	memset(aCmdBuff, 0x00, sizeof(aCmdBuff));	/* PRQA S 3200 */

	/* Add the command code. */
	aCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_CREATE_TRANSACTION_MACFILE;
	
	/* Add the file number to command buffer. */
	aCmdBuff[wCmdLen++] = bFileNo;

	/* Add the file options to command buffer. */
	aCmdBuff[wCmdLen++] = (uint8_t) (bFileOption >> 4);

	/* Append access rights. */
	memcpy(&aCmdBuff[wCmdLen], pAccessRights, 2); /* PRQA S 3200 */
	wCmdLen += 2;

	/* Add the TM key options to command buffer. */
	aCmdBuff[wCmdLen++] = bTMKeyOption;

	/* Command Exchange with SAM. ----------------------------------------------------------------------------- */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_CreateTMFilePICC(
		PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
		bOption,
		bKeyNo,
		bKeyVer,
		bFileNo,
		(uint8_t) (bFileOption >> 4),
		pAccessRights,
		bTMKeyOption,
		pDivInput,
		bDivInputLen,
		&pSamResponse,
		&wSamRespLen));

	/* Buffer the data. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_NonX_Int_CardExchange(
		pDataParams, 
		PH_ON,
		PH_EXCHANGE_BUFFER_FIRST,
		(uint16_t) (wCmdLen + wSamRespLen),
		PH_OFF,
		aCmdBuff,
		wCmdLen, 
		&pCardResponse,
		&wCardRespLen, 
		&bPiccErrCode));

	/* Add the response received from SAM and perform final exchange with the card. */
	wStatus = phalMfIdentity_Sam_NonX_Int_CardExchange(
		pDataParams, 
		PH_ON,
		PH_EXCHANGE_BUFFER_LAST,
		0,
		PH_ON,
		pSamResponse,
		wSamRespLen, 
		&pCardResponse,
		&wCardRespLen, 
		&bPiccErrCode);

	/* Verify the MAC. */
	if(wStatus != PH_ERR_SUCCESS)
	{
		/* Reset the Authentication States. */
		PH_CHECK_SUCCESS_FCT(wStatus1, phalMfIdentity_Sam_NonX_Int_ResetAuthStatus(pDataParams));
	}
	else
	{
		/* Do not call SM Verification command if the communication settings is PLAIN with value 0x20. */
		if(bFileOption != PHAL_MFIDENTITY_COMMUNICATION_PLAIN_1)
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_NonX_Int_VerifySM(
				pDataParams,
				(uint16_t) (PH_EXCHANGE_DEFAULT | PHALMFIDENTITY_SAM_NONX_EXCHANGE_PICC_STATUS),
				PHAL_MFIDENTITY_COMMUNICATION_MACD,
				NULL,
				0,
				bPiccErrCode,
				pCardResponse,
				wCardRespLen,
				&pSamResponse,
				&wSamRespLen));
		}
	}

	return wStatus;
}

phStatus_t phalMfIdentity_Sam_NonX_Int_ResetAuthStatus(void * pDataParams)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;

	/* Reset the Authmode and Key number */
	PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode	= 0xFF;
    PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo		= 0xFF;
	PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthType	= 0xFF;

	/* Kill the PICC Authentication. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_KillAuthentication(
		PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PHHAL_HW_SAMAV3_CMD_SAM_KILL_AUTHENTICATION_PARTIAL));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_NonX_Int_ComputeTMACSessionVectors(void * pDataParams, uint8_t bOption, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer, uint16_t wDstKeyNo, 
	uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aSV[16];
	uint8_t		PH_MEMLOC_REM aIV[16];
	uint8_t		PH_MEMLOC_REM bSvLen = 0;
	uint8_t		PH_MEMLOC_REM bSvBak = 0;
	uint16_t	PH_MEMLOC_REM wActTMC = 0;
	uint16_t	PH_MEMLOC_REM wSesTMC = 0;
	uint32_t	PH_MEMLOC_REM dwTMC = 0;

	/* Increment the TMC by 1. */
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthType == PHAL_MFIDENTITY_AUTH_TYPE_EV2)
	{
		dwTMC = (uint32_t) (pTMC[0] | (pTMC[1] << 8) | (pTMC[2] << 16) | (pTMC[3] << 24));
		dwTMC++;
	}
	else
	{
		wActTMC = (uint16_t) (pTMC[0] | (pTMC[1] << 8));
		wSesTMC = (uint16_t) (pTMC[2] | (pTMC[3] << 8));
		
		wActTMC++;
		wSesTMC++;
		dwTMC = (uint32_t) ((wSesTMC << 16) | wActTMC);
	}

	/* Clear the session vector SV. */
	memset(aSV, 0, 16);		/* PRQA S 3200 */
	memset(aIV, 0, 16);		/* PRQA S 3200 */

	/* Compute the session vector. */
	aSV[bSvLen++] = (uint8_t) ((bOption == PHALMFIDENTITY_SAM_NONX_SESSION_ENC) ? 0xA5 : 0x5A);
	aSV[bSvLen++] = 0x00;
	aSV[bSvLen++] = 0x01;
	aSV[bSvLen++] = 0x00;
	aSV[bSvLen++] = 0x80;

	/* Append the TMC information. */
	aSV[bSvLen++] = (uint8_t) (dwTMC & 0xFF);
	aSV[bSvLen++] = (uint8_t) ((dwTMC & 0xFF00) >> 8);
	aSV[bSvLen++] = (uint8_t) ((dwTMC & 0xFF0000) >> 16);
	aSV[bSvLen++] = (uint8_t) ((dwTMC & 0xFF000000) >> 24);

	/* Append the UID infomration. */
	memcpy(&aSV[bSvLen], pUid, bUidLen);	/* PRQA S 3200 */
	bSvLen = 16;

	/* Rotate the SV information by 1 for LRP. */
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthType == PHAL_MFIDENTITY_AUTH_TYPE_LRP)
	{
		bSvBak = aSV[0];
		memcpy(&aSV[0], &aSV[1], 15);	/* PRQA S 3200 */
		aSV[15] = bSvBak;
	}

	/* Exchange the session vector information to SAM. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_DeriveKey(
		PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
		(uint8_t) wSrcKeyNo,
		(uint8_t) wSrcKeyVer,
		(uint8_t) wDstKeyNo,
		aSV,
		bSvLen));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_NonX_Int_ComputeSDMSessionVectors(void * pDataParams, uint8_t bOption, uint8_t bSdmOption, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer, 
	uint16_t wDstKeyNo, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aSV[16];
	uint8_t		PH_MEMLOC_REM bSvLen = 0;
	uint32_t	PH_MEMLOC_REM dwSDMReadCtr = 0;

	/* Validate the Counter value. */
	dwSDMReadCtr = (uint32_t) (pSDMReadCtr[0] | (pSDMReadCtr[1] << 8) | (pSDMReadCtr[2] << 16) | (pSDMReadCtr[3] << 24));
	if (dwSDMReadCtr == 0xFFFFFF)
	{
		return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_AL_MFIDENTITY);
	}

	/* Clear the session vector SV. */
	memset(aSV, 0, 16);		/* PRQA S 3200 */

	/* Frame the default values in session vector. */
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyType)
	{
		aSV[bSvLen++] = 0x00;
		aSV[bSvLen++] = 0x01;
		aSV[bSvLen++] = 0x00;
		aSV[bSvLen++] = 0x80;

		aSV[14] = 0x1E;
		aSV[15] = 0xE1;
	}
	else
	{
		aSV[bSvLen++] = (uint8_t) ((bOption == PHALMFIDENTITY_SAM_NONX_SESSION_ENC) ? 0xC3 : 0x3C);
		aSV[bSvLen++] = (uint8_t) ((bOption == PHALMFIDENTITY_SAM_NONX_SESSION_ENC) ? 0x3C : 0xC3);
		aSV[bSvLen++] = 0x00;
		aSV[bSvLen++] = 0x01;
		aSV[bSvLen++] = 0x00;
		aSV[bSvLen++] = 0x80;
	}

	/* Append the UID */
	if(bSdmOption & PHAL_MFIDENTITY_VCUID_PRESENT)
	{
		memcpy(&aSV[bSvLen], pUid, bUidLen);	/* PRQA S 3200 */
		bSvLen += bUidLen;
	}

	/* Append the SDM ReadCtr information. */
	if(bSdmOption & PHAL_MFIDENTITY_RDCTR_PRESENT)
	{
		aSV[bSvLen++] = (uint8_t) (dwSDMReadCtr & 0xFF);
		aSV[bSvLen++] = (uint8_t) ((dwSDMReadCtr & 0xFF00) >> 8);
		aSV[bSvLen++] = (uint8_t) ((dwSDMReadCtr & 0xFF0000) >> 16);
	}

	/* Update the SV length to 16. */
	bSvLen = 16;

	/* Exchange the session vector information to SAM. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_DeriveKey(
		PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
		(uint8_t) wSrcKeyNo,
		(uint8_t) wSrcKeyVer,
		(uint8_t) wDstKeyNo,
		aSV,
		bSvLen));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_NonX_Int_LoadSDMInitVector(void * pDataParams, uint8_t * pSDMReadCtr)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bOption = 0;
	uint8_t		PH_MEMLOC_REM aIV[16];
	uint8_t *	PH_MEMLOC_REM pIV = NULL;
	uint16_t	PH_MEMLOC_REM wIvLen = 0;

	/* Clear the Initialization Vector. */
	memset(aIV, 0, 16);		/* PRQA S 3200 */

	/* Frame the default values in session vector. */
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyType)
	{
		/* IV computation is SDMReadCtr || 0x000000 */

		memcpy(&aIV[wIvLen], pSDMReadCtr, 3);	/* PRQA S 3200 */
		wIvLen += 3;

		/* Update IVLen to allocate three zeros. */
		wIvLen += 3;

		/* Set the Option. */
		bOption = PHHAL_HW_SAMAV3_CMD_SAM_LOAD_IV_MODE_SET_LRP_ENC_CTR;
	}
	else
	{
		/* IV computation is E(KSesSDMFileReadENC; SDMReadCtr || 0x00000000000000000000000000) */
		memcpy(&aIV[wIvLen], pSDMReadCtr, 3);	/* PRQA S 3200 */
		
		/* Set the IV length to 16. */
		wIvLen = 16;

		/* Encrypt the IV. */
		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_EncipherOfflineData(
			PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
			PH_EXCHANGE_DEFAULT,
			aIV,
			(uint8_t) wIvLen,
			&pIV,
			&wIvLen));

		/* Copy the enciphered data to local buffer. */
		memcpy(aIV, pIV, wIvLen);		/* PRQA S 3200 */		
		pIV = NULL;

		/* Set the Option. */
		bOption = PHHAL_HW_SAMAV3_CMD_SAM_LOAD_IV_MODE_SET_IV;
	}

	/* Load the IV. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_LoadInitVector(
			PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
			bOption,
			aIV,
			(uint8_t) wIvLen));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_NonX_Int_GetFrameLen(void * pDataParams, uint16_t * pFrameLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint16_t	PH_MEMLOC_REM wFrameLen = 0;

	/* Get the frame size that can be transmitted to PICC. */
	PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_GetConfig(
		PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
		0x04, /* Get the frame length of PICC and PCD. */
		&wFrameLen));

	/* Update the parameter. */
	switch((uint8_t) (wFrameLen & 0x000F))
	{
		case 0: *pFrameLen = 16; break;
		case 1: *pFrameLen = 24; break;
		case 2: *pFrameLen = 32; break;
		case 3: *pFrameLen = 40; break;
		case 4: *pFrameLen = 48; break;
		case 5: *pFrameLen = 64; break;
		case 6: *pFrameLen = 96; break;
		case 7: *pFrameLen = 128; break;
		case 8: *pFrameLen = 256; break;

		default:
			break;
	}

	/* Remove the ISO header. */
	*pFrameLen -= 4;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */
