﻿/*
 * Copyright 2017 - 2020, 2022 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software ICode Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 *
 * History:
 *  CHu: Generated 02. January 2017
 *
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <phpalSli15693.h>
#include <phalICode.h>

#ifdef NXPBUILD__PHAL_ICODE_SW

#ifdef NXPBUILD__PH_CRYPTOSYM
#include <phKeyStore.h>
#include <phCryptoSym.h>
#include <phCryptoRng.h>
#endif /* NXPBUILD__PH_CRYPTOSYM */

#include "phalICode_Sw.h"
#include "../phalICode_Int.h"

phStatus_t phalICode_Sw_Init(phalICode_Sw_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, void * pPalSli15693DataParams,
    void * pCryptoDataParams, void * pCryptoRngDataParams, void * pKeyStoreDataParams)
{
    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_ICODE);
    PH_ASSERT_NULL_PARAM(pPalSli15693DataParams, PH_COMP_AL_ICODE);

    /* Check the size. */
    if(sizeof(phalICode_Sw_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_ICODE);
    }

    /* Initialize the structure members. */
    pDataParams->wId                    = PH_COMP_AL_ICODE | PHAL_ICODE_SW_ID;
    pDataParams->pPalSli15693DataParams = pPalSli15693DataParams;
    pDataParams->pCryptoDataParams      = pCryptoDataParams;
    pDataParams->pCryptoRngDataParams   = pCryptoRngDataParams;
    pDataParams->pKeyStoreDataParams    = pKeyStoreDataParams;
    pDataParams->bBuffering             = PH_OFF;

    pDataParams->wChallange_Timeout = PHAL_ICODE_TIMEOUT_LONG_US;
    pDataParams->wChallange_Timeout_Unit = (uint16_t) (PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_US >> 1U);

    /* Reset the random number buffer. */
    (void) memset(pDataParams->aRnd_Challenge, 0x00, sizeof(pDataParams->aRnd_Challenge));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}





/* iCode Mandatory commands ---------------------------------------------------------------------------------------- */
phStatus_t phalICode_Sw_Inventory(phalICode_Sw_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t * pDsfid, uint8_t * pUid)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_Inventory(
        pDataParams->pPalSli15693DataParams,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        pDsfid,
        pUid));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_StayQuiet(phalICode_Sw_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_StayQuiet(
        pDataParams->pPalSli15693DataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}





/* iCode Optional commands ----------------------------------------------------------------------------------------- */
phStatus_t phalICode_Sw_ReadSingleBlock(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockNo,
    uint8_t ** ppData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadSingleBlock(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bBlockNo,
        ppData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_WriteSingleBlock(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockNo,
    uint8_t * pData, uint8_t bDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WriteSingleBlock(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bBlockNo,
        pData,
        bDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_LockBlock(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_LockBlock(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bBlockNo));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ReadMultipleBlocks(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockNo,
    uint8_t bNumBlocks, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    *pDataLen = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadMultipleBlocks(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        bOption,
        bBlockNo,
        bNumBlocks,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_Select(phalICode_Sw_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_Select(
        pDataParams->pPalSli15693DataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ResetToReady(phalICode_Sw_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_ResetToReady(
        pDataParams->pPalSli15693DataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_WriteAFI(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bAfi)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WriteAFI(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bAfi));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_LockAFI(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_LockAFI(
        pDataParams->pPalSli15693DataParams,
        bOption));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_WriteDSFID(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bDsfid)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WriteDSFID(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bDsfid));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_LockDSFID(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_LockDSFID(
        pDataParams->pPalSli15693DataParams,
        bOption));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_GetSystemInformation(phalICode_Sw_DataParams_t * pDataParams, uint8_t ** ppSystemInfo,
    uint16_t * pSystemInfoLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_GetSystemInformation(
        pDataParams->pPalSli15693DataParams,
        ppSystemInfo,
        pSystemInfoLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_GetMultipleBlockSecurityStatus(phalICode_Sw_DataParams_t * pDataParams, uint8_t bBlockNo,
    uint8_t bNoOfBlocks, uint8_t * pStatus, uint16_t * pStatusLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_GetMultipleBlockSecurityStatus(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        bBlockNo,
        bNoOfBlocks,
        pStatus,
        pStatusLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_FastReadMultipleBlocks(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockNo,
    uint8_t bNumBlocks, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_FastReadMultipleBlocks(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        bOption,
        bBlockNo,
        bNumBlocks,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ExtendedReadSingleBlock(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint16_t wBlockNo,
    uint8_t ** ppData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ExtendedReadSingleBlock(
        pDataParams->pPalSli15693DataParams,
        bOption,
        wBlockNo,
        ppData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ExtendedWriteSingleBlock(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint16_t wBlockNo,
    uint8_t * pData, uint8_t bDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ExtendedWriteSingleBlock(
        pDataParams->pPalSli15693DataParams,
        bOption,
        wBlockNo,
        pData,
        bDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ExtendedLockBlock(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint16_t wBlockNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ExtendedLockBlock(
        pDataParams->pPalSli15693DataParams,
        bOption,
        wBlockNo));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ExtendedReadMultipleBlocks(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint16_t wBlockNo,
    uint16_t wNumBlocks, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ExtendedReadMultipleBlocks(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        bOption,
        wBlockNo,
        wNumBlocks,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalICode_Sw_AuthenticateTAM(phalICode_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bAuthType, uint8_t bKeyNo,
    uint8_t bKeyVer, uint8_t bKeyID_CertNum, uint8_t * pDivInput, uint8_t bDivLen, uint8_t * pChallenge, uint16_t wChallengeLen,
    uint8_t * pAddInfo, uint16_t wAddInfoLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwTRnd_TAM1;
    uint16_t    PH_MEMLOC_REM wKeyType = 0;
    uint16_t    PH_MEMLOC_REM wGenRnd = 0;
    uint16_t    PH_MEMLOC_REM wDivOption = 0;
    uint16_t    PH_MEMLOC_REM wC_TAM1;
#ifdef NXPBUILD__PHAL_ICODE_INTERNAL
    uint16_t    PH_MEMLOC_REM wTimingData = 0;
#endif /* NXPBUILD__PHAL_ICODE_INTERNAL */
    uint16_t    PH_MEMLOC_REM wRespFlag = 0;
    uint8_t     PH_MEMLOC_REM bOption = 0;
    uint8_t     PH_MEMLOC_REM bRndSize = 0;

    uint8_t     PH_MEMLOC_REM aRespPlain[16];
    uint8_t     PH_MEMLOC_REM aIChallenge_TAM1[10];

    uint8_t     PH_MEMLOC_REM aIChallenge[PHAL_ICODE_RANDOM_NUMBER_MAX_SIZE];
    uint8_t     PH_MEMLOC_REM aKey[PH_KEYSTORE_KEY_TYPE_AES128_SIZE];

    uint8_t     PH_MEMLOC_REM aCmdBuff[70];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    PH_UNUSED_VARIABLE(pAddInfo);
    PH_UNUSED_VARIABLE(wAddInfoLen);
#ifndef NXPBUILD__PHAL_ICODE_INTERNAL
    PH_UNUSED_VARIABLE(ppResponse);
    PH_UNUSED_VARIABLE(pRespLen);
#endif /* NXPBUILD__PHAL_ICODE_INTERNAL */

    /* Extract Parameters */
    bOption = (uint8_t) (wOption & 0x000FU);
    wDivOption = (uint16_t) (wOption & 0x00F0U);
    wGenRnd = (uint16_t) (wOption & 0x0F00U);

    /* Validate Parameter */
    PHAL_ICODE_VALIDATE_AUTHTYPE(bAuthType);
    if((wGenRnd != PHAL_ICODE_GENERATE_CHALLENGE_INTERNAL) &&
        (wGenRnd != PHAL_ICODE_GENERATE_CHALLENGE_EXTERNAL))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    if((wDivOption != PHAL_ICODE_DIVERSIFICATION_DISABLED) &&
        (wDivOption != PHAL_ICODE_DIVERSIFICATION_ENABLED))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pDataParams->pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Clear all the local variables. */
    (void) memset(aCmdBuff, 0x00, (size_t)sizeof(aCmdBuff));
    (void) memset(aIChallenge, 0x00U, (size_t) sizeof(aIChallenge));

    /* Frame and exchange the TAM message based on Authentication type. */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_AUTHENTICATE;

    /* Prepare Message based on Authenticate Type */
#ifdef NXPBUILD__PHAL_ICODE_INTERNAL
    if((bAuthType == PHAL_ICODE_AUTHENTICATE_TAM_1_2) || (bAuthType == PHAL_ICODE_AUTHENTICATE_TAM_1_3))
    {
        aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_DH;
        if(bAuthType == PHAL_ICODE_AUTHENTICATE_TAM_1_3)
        {
            aCmdBuff[bCmdLen++] = PHAL_ICODE_AUTH_PARAM_TAM1_3;
            aCmdBuff[bCmdLen++] = bKeyID_CertNum;
        }
        else
        {
            aCmdBuff[bCmdLen++] = PHAL_ICODE_AUTH_PARAM_TAM1_2;
        }
    }
    else
#endif /* NXPBUILD__PHAL_ICODE_INTERNAL */
    {
        (void) memset(aKey, 0x00, PH_KEYSTORE_KEY_TYPE_AES128_SIZE);

        /* Get the key from key store. */
        PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
            pDataParams->pKeyStoreDataParams,
            bKeyNo,
            bKeyVer,
            (uint8_t) sizeof(aKey),
            aKey,
            &wKeyType));

        /* Check if key type is of type AES. */
        if(wKeyType != PH_KEYSTORE_KEY_TYPE_AES128)
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
        }

        /* Diversify the key. */
        if(wDivOption == PHAL_ICODE_DIVERSIFICATION_ENABLED)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DiversifyDirectKey(
                pDataParams->pCryptoDataParams,
                PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS,
                aKey,
                wKeyType,
                pDivInput,
                bDivLen,
                aKey));
        }

        /* Reverse the Key before loading to crypto params. */
        phalICode_Int_Reverse(aKey, (uint16_t) sizeof(aKey));

        /* Load the key to crypto params. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParams,
            aKey,
            wKeyType));

        aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_AES;
        aCmdBuff[bCmdLen++] = PHAL_ICODE_TAM_CUSTOMDATA_CLEAR | PHAL_ICODE_AUTHPROC_TAM;
        aCmdBuff[bCmdLen++] = bKeyID_CertNum;
    }

    /* Get the random number. */
    if(wGenRnd == PHAL_ICODE_GENERATE_CHALLENGE_INTERNAL)
    {
        /* Get the Random number size to use. */
        bRndSize = PHAL_ICODE_RANDOM_NUMBER_MIN_SIZE;
#ifdef NXPBUILD__PHAL_ICODE_INTERNAL
        switch(bAuthType)
        {
            case PHAL_ICODE_AUTHENTICATE_TAM_1_2:
            case PHAL_ICODE_AUTHENTICATE_TAM_1_3:
                bRndSize = PHAL_ICODE_RANDOM_NUMBER_MAX_SIZE;
                break;

            default:
                bRndSize = PHAL_ICODE_RANDOM_NUMBER_MIN_SIZE;
                break;
        }
#endif /* NXPBUILD__PHAL_ICODE_INTERNAL */

        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Rnd(
            pDataParams->pCryptoRngDataParams,
            bRndSize,
            aIChallenge));
    }
    else
    {
        (void) memcpy(aIChallenge, pChallenge, wChallengeLen);
        bRndSize = (uint8_t) wChallengeLen;
    }

    /* Add the random number. */
    (void) memcpy(&aCmdBuff[bCmdLen], aIChallenge, bRndSize);
    bCmdLen += bRndSize;

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pDataParams->pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pDataParams->pPalSli15693DataParams, wStatus));

    /* Validate response based on Authentication type. */
#ifdef NXPBUILD__PHAL_ICODE_INTERNAL
    if((bAuthType == PHAL_ICODE_AUTHENTICATE_TAM_1_2) || (bAuthType == PHAL_ICODE_AUTHENTICATE_TAM_1_3))
    {
        /* Check if the response consists of 3 bytes of data. */
        if(wRespLen != (1U /* Barker code. */ + 2U /* Timing information */))
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
        }

        /* Get the response flag from PAL layer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
            pDataParams->pPalSli15693DataParams,
            PHPAL_SLI15693_CONFIG_ADD_INFO,
            &wRespFlag));

        /* Check if barker code is valid. */
        if(!(((pResponse[0]) & 0x7FU /* Barker code extraction. */) == 0x27U /* Barker Code */))
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
        }

        /* Check if Done flag is not set and it's not the final response. */
        if((((pResponse[0]) & 0x80U /* Done Flag extraction. */) == 0x80U) ||
            (((wRespFlag) & 0x02U /* Response buffer flag extraction */) == 0x02U) ||
            (((wRespFlag) & 0x04U /* Final response flag extraction */) == 0x04U))
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
        }

        /* Extract timing information. */
        wTimingData = (uint16_t) ((pResponse[2] << 8U) | pResponse[1]);

        /* Update the timing received from tag */
        PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
            pDataParams->pPalSli15693DataParams,
            PHPAL_SLI15693_CONFIG_TIMEOUT_MS,
            wTimingData));

        /* Retrieve final data. */
        wStatus = phpalSli15693_Exchange(
            pDataParams->pPalSli15693DataParams,
            PH_EXCHANGE_DEFAULT,
            NULL,
            0,
            &pResponse,
            &wRespLen);

        /* Compute the status code. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pDataParams->pPalSli15693DataParams, wStatus));

        /* Check if barker code is valid. */
        if(!(((pResponse[0]) & 0x7FU /* Barker code extraction. */) == 0x27U /* Barker Code */))
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
        }

        /* Get the response flag from PAL layer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
            pDataParams->pPalSli15693DataParams,
            PHPAL_SLI15693_CONFIG_ADD_INFO,
            &wRespFlag));

        /* Check if Done flag is set and is it the final response. */
        if(!((((pResponse[0]) & 0x80U /* Done Flag extraction. */) == 0x80U) &&
            (((wRespFlag) & 0x02U /* Response buffer flag extraction */) == 0x02U) &&
            (((wRespFlag) & 0x04U /* Final response flag extraction */) == 0x04U)))
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
        }

        /* Remove Barker code. */
        --wRespLen;
        pResponse++;

        /* Copy the response to parameters. */
        *ppResponse = pResponse;
        *pRespLen = wRespLen;
    }
    else
#endif /* NXPBUILD__PHAL_ICODE_INTERNAL */
    {
        /* Check if the response consists of 17 bytes of data. */
        if(wRespLen != (1U /* Barker code. */ + 16U /* TAM1 Response */))
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
        }

        /* Get the response flag from PAL layer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
            pDataParams->pPalSli15693DataParams,
            PHPAL_SLI15693_CONFIG_ADD_INFO,
            &wRespFlag));

        /* Check if barker code is valid. */
        if(!(((pResponse[0]) & 0x7FU /* Barker code extraction. */) == 0x27U /* Barker Code */))
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
        }

        /* Check if Done flag is set and is it the final response. */
        if(!((((pResponse[0]) & 0x80U /* Done Flag extraction. */) == 0x80U) &&
            (((wRespFlag) & 0x02U /* Response buffer flag extraction */) == 0x02U) &&
            (((wRespFlag) & 0x04U /* Final response flag extraction */) == 0x04U)))
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
        }

        /* Decrement the received data length to exclude the barker code. */
        --wRespLen;

        /* Reverse the buffer. */
        phalICode_Int_Reverse(&pResponse[1], wRespLen);

        /* Clear all the local variables. */
        (void) memset(aRespPlain, 0x00, 16);
        (void) memset(aIChallenge_TAM1, 0x00, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);

        /* Decrypt the response to extract the random numbers. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
            pDataParams->pCryptoDataParams,
            (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_ECB),
            &pResponse[1],
            PH_CRYPTOSYM_AES_BLOCK_SIZE,
            aRespPlain));

        /* Reverse the buffer to load the correct values in the variables. This reversing is actually not required.
         * Its added because
         *      lets say the response after decryption is 96 C5 < 4byte TChallange> <10 byte IChallange>
         *      When memcpy is called to store the C_TAM1 (96 C5) data without reversing the decrypted
         *      buffer, the wC_TAM1 holds the data as C5 96. This should be then reversed. Also the
         *      IChallange should be reversed. To avoid multiple reversing of data, the decrypted data
         *      itself is reversed so that it gets stored to the local variables properly and they are
         *      compared easily.
         */
        phalICode_Int_Reverse(aRespPlain, PH_CRYPTOSYM_AES_BLOCK_SIZE);

        /* Extract constant and random numbers. */
        (void) memcpy(aIChallenge_TAM1, aRespPlain, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);
        (void) memcpy(&dwTRnd_TAM1, &aRespPlain[PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM], 4U);
        (void) memcpy(&wC_TAM1, &aRespPlain[PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM + 4U /* TChallenge */], 2U);

        /* Verify the received constant Tag authentication value. */
        if(wC_TAM1 != PHAL_ICODE_CONST_TAM1)
        {
            return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_ICODE);
        }

        /* Verify the IChallenge. */
        if(memcmp(aIChallenge, aIChallenge_TAM1, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM) != 0x00)
        {
            return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_ICODE);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_AuthenticateTAM1(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bKeyNo, uint8_t bKeyVer,
    uint8_t bKeyNoCard)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wKeyType = 0;
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint16_t    PH_MEMLOC_REM wRespFlag = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint16_t    PH_MEMLOC_REM wC_TAM1;
    uint32_t    PH_MEMLOC_REM dwTRnd_TAM1;

    uint8_t     PH_MEMLOC_REM aIChallenge[PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM];
    uint8_t     PH_MEMLOC_REM aKey[PH_KEYSTORE_KEY_TYPE_AES128_SIZE];
    uint8_t     PH_MEMLOC_REM aCmdBuff[14];
    uint8_t     PH_MEMLOC_REM aRespPlain[16];
    uint8_t     PH_MEMLOC_REM aIChallenge_TAM1[10];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pDataParams->pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Clear all the local variables. */
    (void) memset(aIChallenge, 0x00, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);
    (void) memset(aKey, 0x00, PH_KEYSTORE_KEY_TYPE_AES128_SIZE);

    /* Get the key from key store. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        bKeyNo,
        bKeyVer,
        (uint8_t) sizeof(aKey),
        aKey,
        &wKeyType));

    /* Check if key type is of type AES. */
    if(wKeyType != PH_KEYSTORE_KEY_TYPE_AES128)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Reverse the Key before loading to crypto params. */
    phalICode_Int_Reverse(aKey, (uint16_t) sizeof(aKey));

    /* Load the key to crypto params. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParams,
        aKey,
        wKeyType));

    /* Get the random number. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Rnd(
        pDataParams->pCryptoRngDataParams,
        PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM,
        aIChallenge));

    /* Frame and exchange the TAM1 message. */
    /* Clear all the local variables. */
    (void) memset(aCmdBuff, 0x00, (size_t)sizeof(aCmdBuff));

    /* Frame the command. */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_AUTHENTICATE;
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_AES;

    /* Prepare TAM1 Message */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_TAM_CUSTOMDATA_CLEAR | PHAL_ICODE_AUTHPROC_TAM;
    aCmdBuff[bCmdLen++] = bKeyNoCard;

    /* Add the random number. */
    (void) memcpy(&aCmdBuff[bCmdLen], aIChallenge, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);
    bCmdLen += PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM;

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pDataParams->pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pDataParams->pPalSli15693DataParams, wStatus));

    /* Check if the response consists of 17 bytes of data. */
    if(wRespLen != (1U /* Barker code. */ + 16U /* TAM1 Response */))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Get the response flag from PAL layer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_ADD_INFO,
        &wRespFlag));

    /* Check if barker code is valid. */
    if(!(((pResponse[0]) & 0x7FU /* Barker code extraction. */) == 0x27U /* Barker Code */))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Check if Done flag is set and is it the final response. */
    if(!((((pResponse[0]) & 0x80U /* Done Flag extraction. */) == 0x80U) &&
        (((wRespFlag) & 0x02U /* Response buffer flag extraction */) == 0x02U) &&
        (((wRespFlag) & 0x04U /* Final response flag extraction */) == 0x04U)))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Decrement the received data length to exclude the barker code. */
    --wRespLen;

    /* Reverse the buffer. */
    phalICode_Int_Reverse(&pResponse[1], wRespLen);

    /* Clear all the local variables. */
    (void) memset(aRespPlain, 0x00, 16);
    (void) memset(aIChallenge_TAM1, 0x00, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);

    /* Decrypt the response to extract the random numbers. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParams,
        (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_ECB),
        &pResponse[1],
        PH_CRYPTOSYM_AES_BLOCK_SIZE,
        aRespPlain));

    /* Reverse the buffer to load the correct values in the variables. This reversing is actually not required.
     * Its added because
     *      lets say the response after decryption is 96 C5 < 4byte TChallange> <10 byte IChallange>
     *      When memcpy is called to store the C_TAM1 (96 C5) data without reversing the decrypted
     *      buffer, the wC_TAM1 holds the data as C5 96. This should be then reversed. Also the
     *      IChallange should be reversed. To avoid multiple reversing of data, the decrypted data
     *      itself is reversed so that it gets stored to the local variables properly and they are
     *      compared easily. */
    phalICode_Int_Reverse(aRespPlain, PH_CRYPTOSYM_AES_BLOCK_SIZE);

    /* Extract constant and random numbers. */
    (void) memcpy(aIChallenge_TAM1, aRespPlain, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);
    (void) memcpy(&dwTRnd_TAM1, &aRespPlain[PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM], 4U);
    (void) memcpy(&wC_TAM1, &aRespPlain[PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM + 4U /* TChallenge */], 2U);

    /* Verify the received constant Tag authentication value. */
    if(wC_TAM1 != PHAL_ICODE_CONST_TAM1)
    {
        return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_ICODE);
    }

    /* Verify the IChallenge. */
    if(memcmp(aIChallenge, aIChallenge_TAM1, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM) != 0x00)
    {
        return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_ICODE);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_AuthenticateMAM(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bKeyNo, uint8_t bKeyVer,
    uint8_t bKeyNoCard, uint8_t bPurposeMAM2)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wKeyType = 0;
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint16_t    PH_MEMLOC_REM wRespFlag = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint16_t    PH_MEMLOC_REM wC_MAM1 = 0;

    uint8_t     PH_MEMLOC_REM aIChallenge[PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM];
    uint8_t     PH_MEMLOC_REM aKey[PH_KEYSTORE_KEY_TYPE_AES128_SIZE];
    uint8_t     PH_MEMLOC_REM aCmdBuff[20];
    uint8_t     PH_MEMLOC_REM aRespPlain[16];
    uint8_t     PH_MEMLOC_REM aIChallenge_MAM1[PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM];
    uint8_t     PH_MEMLOC_REM aTChallenge_MAM1[PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pDataParams->pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Clear all the local variables. */
    (void) memset(aIChallenge, 0x00, PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM);
    (void) memset(aKey, 0x00, PH_KEYSTORE_KEY_TYPE_AES128_SIZE);

    /* Get the key from key store. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        bKeyNo,
        bKeyVer,
        (uint8_t) sizeof(aKey),
        aKey,
        &wKeyType));

    /* Check if key type is of type AES. */
    if(wKeyType != PH_KEYSTORE_KEY_TYPE_AES128)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Reverse the Key before loading to crypto params. */
    phalICode_Int_Reverse(aKey, (uint16_t) sizeof(aKey));

    /* Load the key to crypto params. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParams,
        aKey,
        wKeyType));

    /* Get the random number. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Rnd(
        pDataParams->pCryptoRngDataParams,
        PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM,
        aIChallenge));

    /* Clear all the local variables. */
    (void) memset(aCmdBuff, 0x00, (size_t) sizeof(aCmdBuff));
    (void) memset(aRespPlain, 0x00, 16);
    (void) memset(aIChallenge_MAM1, 0x00, PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM);
    (void) memset(aTChallenge_MAM1, 0x00, PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM);

    /* Frame the command. */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_AUTHENTICATE;
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_AES;

    /* Frame the MAM1 message. */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_MAM1_STEP | PHAL_ICODE_AUTHPROC_MAM;
    aCmdBuff[bCmdLen++] = bKeyNoCard;

    /* Add the random number. */
    (void) memcpy(&aCmdBuff[bCmdLen], aIChallenge, PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM);
    bCmdLen += PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM;

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pDataParams->pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pDataParams->pPalSli15693DataParams, wStatus));

    /* Check if the response consists of 23 bytes of data. */
    if(wRespLen != (1U /* Barker code. */ + 22U /* Rest of MAM1 response. */))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Get the response flag from PAL layer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_FLAGS,
        &wRespFlag));

    /* Check if barker code is valid. */
    if(!(((pResponse[0]) & 0x7FU /* Barker code extraction. */) == 0x27U /* Barker Code */))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Check if Done flag is set and is it the final response. */
    if((((pResponse[0]) & 0x80U /* Done Flag extraction. */) == 0x80U) &&
        (((wRespFlag) & 0x02U /* Response buffer flag extraction */) == 0x02U) &&
        (((wRespFlag) & 0x04U /* Final response flag extraction */) == 0x04U))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Decrement the received data length to exclude the barker code. */
    --wRespLen;

    /* Reverse the buffer. */
    phalICode_Int_Reverse(&pResponse[1], wRespLen);

    /* Copy the un-encrypted part of Tag random number (TChallenge_MAM1) from the response. */
    (void) memcpy(aTChallenge_MAM1, &pResponse[wRespLen - 5U], (PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM - 4U /* TChallenge */));

    /* Decrypt the response to extract the random numbers. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParams,
        (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_ECB),
        &pResponse[1],
        PH_CRYPTOSYM_AES_BLOCK_SIZE,
        aRespPlain));

    /* Append the remaining Tag random number (TChallenge_MAM1) from the decrypted data. */
    (void) memcpy(&aTChallenge_MAM1[6], &aRespPlain[2], 4 /* TChallenge */);

    /* Reverse the buffer to load the correct values in the variables. This reversing is actually not required.
     * Its added because
     *      lets say the response after decryption is DA 83 < 4byte TChallange> <10 byte IChallange>
     *      When memcpy is called to store the C_TAM1 (DA 83) data without reversing the decrypted
     *      buffer, the wC_TAM1 holds the data as 83 DA. This should be then reversed. Also the
     *      IChallange should be reversed. To avoid multiple reversing of data, the decrypted data
     *      itself is reversed so that it gets stored to the local variables properly and they are
     *      compared easily. */
    phalICode_Int_Reverse(aRespPlain, PH_CRYPTOSYM_AES_BLOCK_SIZE);

    /* Extract constant and random numbers. */
    (void) memcpy(aIChallenge_MAM1, aRespPlain, PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM);
    (void) memcpy(&wC_MAM1, &aRespPlain[PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM + 4U /* TChallenge */], 2U);

    /* Verify the received constant Mutual authentication 1 value. */
    if(wC_MAM1 != PHAL_ICODE_CONST_MAM1)
    {
        return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_ICODE);
    }

    /* Verify the IChallenge. */
    if(memcmp(aIChallenge, aIChallenge_MAM1, PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM) != 0x00)
    {
        return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_ICODE);
    }

    /* Frame the second step of MAM authentication command. */
    bCmdLen = 0;
    (void) memset(aCmdBuff, 0x00, (size_t) sizeof(aCmdBuff));

    /* Frame the command. */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_AUTHENTICATE;

    aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_AES;
    aCmdBuff[bCmdLen++] = PHAL_ICODE_MAM2_STEP | PHAL_ICODE_AUTHPROC_MAM;

    /* Add the constant MAM 2 value. */
    aCmdBuff[bCmdLen++] = (uint8_t) ((PHAL_ICODE_CONST_MAM2 & 0xFF00U) >> 8U);
    aCmdBuff[bCmdLen++] = (uint8_t) ((PHAL_ICODE_CONST_MAM2 & 0x00FFU) | bPurposeMAM2);

    /* Add the random number. */
    phalICode_Int_Reverse(aIChallenge, PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM);
    (void) memcpy(&aCmdBuff[bCmdLen], &aIChallenge[6], 4);
    bCmdLen += 4U;

    /* Add the tag challenge. */
    (void) memcpy(&aCmdBuff[bCmdLen], aTChallenge_MAM1, PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM);
    bCmdLen = bCmdLen + PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM;

    /* Decrypt the response to extract the random numbers. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParams,
        (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_ECB),
        &aCmdBuff[3],
        PH_CRYPTOSYM_AES_BLOCK_SIZE,
        &aCmdBuff[3]));

    /* Reverse the buffer. */
    phalICode_Int_Reverse(&aCmdBuff[3], PH_CRYPTOSYM_AES_BLOCK_SIZE);

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pDataParams->pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pDataParams->pPalSli15693DataParams, wStatus));

    /* Check if there is no response message. */
    if(wRespLen != 0x01U /* Barker Code. */)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Get the response flag from PAL layer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_FLAGS,
        &wRespFlag));

    /* Check if barker code is valid. */
    if(!(((pResponse[0]) & 0x7FU /* Barker code extraction. */) == 0x27U /* Barker Code */))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

#ifdef NXPBUILD__PHAL_ICODE_INTERNAL
phStatus_t phalICode_Sw_WriteCertificate(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bCertNo, uint16_t wWordPtr,
    uint8_t * pCertificate, uint8_t bCertLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WriteCertificate(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bCertNo,
        wWordPtr,
        pCertificate,
        bCertLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ReadCertificate(phalICode_Sw_DataParams_t * pDataParams, uint8_t bCertNo, uint16_t wWordPtr, uint16_t wWordCount,
    uint8_t ** ppCertificate, uint16_t *pCertLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadCertificate(
        pDataParams->pPalSli15693DataParams,
        bCertNo,
        wWordPtr,
        wWordCount,
        ppCertificate,
        pCertLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}
#endif /* NXPBUILD__PHAL_ICODE_INTERNAL */

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalICode_Sw_Challenge(phalICode_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bChallengeType, uint8_t bKeyID_CertNum,
    uint8_t * pChallenge, uint16_t wChallengeLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_Challenge(
        pDataParams->pPalSli15693DataParams,
        pDataParams->pCryptoRngDataParams,
        wOption,
        bChallengeType,
        pDataParams->wChallange_Timeout,
        pDataParams->wChallange_Timeout_Unit,
        bKeyID_CertNum,
        pChallenge,
        wChallengeLen,
        pDataParams->aRnd_Challenge));

    /* Update ChallengeType to DataParams. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetChallengeType(pDataParams, bChallengeType));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ReadBuffer(phalICode_Sw_DataParams_t * pDataParams, uint8_t bVerify, uint8_t bKeyNo, uint8_t bKeyVer,
    uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadBuffer(
        PHAL_ICODE_SW_ID,
        pDataParams->pPalSli15693DataParams,
        pDataParams->pCryptoDataParams,
        pDataParams->pKeyStoreDataParams,
        pDataParams->aRnd_Challenge,
        pDataParams->bChallengeType,
        bVerify,
        bKeyNo,
        bKeyVer,
        ppResponse,
        pRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalICode_Sw_ExtendedGetSystemInformation(phalICode_Sw_DataParams_t * pDataParams, uint8_t bInfoParams, uint8_t ** ppSystemInfo,
    uint16_t * pSystemInfoLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ExtendedGetSystemInformation(
        pDataParams->pPalSli15693DataParams,
        bInfoParams,
        ppSystemInfo,
        pSystemInfoLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ExtendedGetMultipleBlockSecurityStatus(phalICode_Sw_DataParams_t * pDataParams, uint16_t wBlockNo,
    uint16_t wNoOfBlocks, uint8_t * pStatus, uint16_t * pStatusLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ExtendedGetMultipleBlockSecurityStatus(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        wBlockNo,
        wNoOfBlocks,
        pStatus,
        pStatusLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ExtendedFastReadMultipleBlocks(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint16_t wBlockNo,
    uint16_t wNumBlocks, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ExtendedFastReadMultipleBlocks(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        bOption,
        wBlockNo,
        wNumBlocks,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}





/* iCode Custom commands ------------------------------------------------------------------------------------------- */
phStatus_t phalICode_Sw_InventoryRead(phalICode_Sw_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bBlockNo, uint8_t bNoOfBlocks, uint8_t * pUid, uint8_t * pUidLen, uint8_t * pData,
    uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_InventoryRead(
        pDataParams->pPalSli15693DataParams,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        bBlockNo,
        bNoOfBlocks,
        pUid,
        pUidLen,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_InventoryReadExtended(phalICode_Sw_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bExtendedOptions, uint8_t * pCID, uint8_t bBlockNo, uint8_t bNoOfBlocks, uint8_t * pCDIDOut,
    uint8_t * pUid, uint8_t * pUidLen, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_InventoryReadExtended(
        pDataParams->pPalSli15693DataParams,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        bExtendedOptions,
        pCID,
        bBlockNo,
        bNoOfBlocks,
        pCDIDOut,
        pUid,
        pUidLen,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_FastInventoryRead(phalICode_Sw_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bBlockNo, uint8_t bNoOfBlocks, uint8_t * pUid, uint8_t * pUidLen, uint8_t * pData,
    uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_FastInventoryRead(
        pDataParams->pPalSli15693DataParams,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        bBlockNo,
        bNoOfBlocks,
        pUid,
        pUidLen,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_FastInventoryReadExtended(phalICode_Sw_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bExtendedOptions, uint8_t * pCID, uint8_t bBlockNo, uint8_t bNoOfBlocks, uint8_t * pCDIDOut,
    uint8_t * pUid, uint8_t * pUidLen, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_FastInventoryReadExtended(
        pDataParams->pPalSli15693DataParams,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        bExtendedOptions,
        pCID,
        bBlockNo,
        bNoOfBlocks,
        pCDIDOut,
        pUid,
        pUidLen,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_SetEAS(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetEAS(
        pDataParams->pPalSli15693DataParams,
        bOption));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ResetEAS(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ResetEAS(
        pDataParams->pPalSli15693DataParams,
        bOption));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_LockEAS(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_LockEAS(
        pDataParams->pPalSli15693DataParams,
        bOption));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_EASAlarm(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pEasIdValue, uint8_t bEasIdMaskLen,
    uint8_t ** ppEas, uint16_t * pEasLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_EASAlarm(
        pDataParams->pPalSli15693DataParams,
        bOption,
        pEasIdValue,
        bEasIdMaskLen,
        ppEas,
        pEasLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_PasswordProtectEAS(phalICode_Sw_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_PasswordProtectEAS(
        pDataParams->pPalSli15693DataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_PasswordProtectAFI(phalICode_Sw_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_PasswordProtectAFI(
        pDataParams->pPalSli15693DataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_WriteEAS_ID(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pEasIdValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WriteEAS_ID(
        pDataParams->pPalSli15693DataParams,
        bOption,
        pEasIdValue));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ReadEPC(phalICode_Sw_DataParams_t * pDataParams, uint8_t ** ppEpc, uint16_t * pEpcLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadEPC(
        pDataParams->pPalSli15693DataParams,
        ppEpc,
        pEpcLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_GetNXPSystemInformation(phalICode_Sw_DataParams_t * pDataParams, uint8_t ** ppSystemInfo, uint16_t * pSystemInfoLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_GetNXPSystemInformation(
        pDataParams->pPalSli15693DataParams,
        ppSystemInfo,
        pSystemInfoLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_InventoryPageRead(phalICode_Sw_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bPageNo, uint8_t bNoOfPages, uint8_t ** ppUid, uint8_t * pUidLen, uint8_t ** ppData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aUid[PHPAL_SLI15693_UID_LENGTH];
    uint8_t     PH_MEMLOC_REM aData[PHAL_ICODE_BLOCK_SIZE * PHAL_ICODE_MAX_BLOCKS];

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_InventoryPageRead(
        pDataParams->pPalSli15693DataParams,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        bPageNo,
        bNoOfPages,
        aUid,
        pUidLen,
        aData,
        pDataLen));

    /* Copy the data to parameters. */
    *ppUid = aUid;
    *ppData = aData;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_FastInventoryPageRead(phalICode_Sw_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bPageNo, uint8_t bNoOfPages, uint8_t ** ppUid, uint8_t * pUidLen, uint8_t ** ppData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aUid[PHPAL_SLI15693_UID_LENGTH];
    uint8_t     PH_MEMLOC_REM aData[PHAL_ICODE_BLOCK_SIZE * PHAL_ICODE_MAX_BLOCKS];

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_FastInventoryPageRead(
        pDataParams->pPalSli15693DataParams,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        bPageNo,
        bNoOfPages,
        aUid,
        pUidLen,
        aData,
        pDataLen));

    /* Copy the data to parameters. */
    *ppUid = aUid;
    *ppData = aData;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_GetRandomNumber(phalICode_Sw_DataParams_t * pDataParams, uint8_t ** ppRnd, uint16_t * pRndLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_GetRandomNumber(
        pDataParams->pPalSli15693DataParams,
        ppRnd,
        pRndLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_SetPassword(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bPwdIdentifier, uint8_t * pXorPwd)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetPassword(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bPwdIdentifier,
        pXorPwd));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_WritePassword(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bPwdIdentifier, uint8_t * pPwd)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WritePassword(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bPwdIdentifier,
        pPwd));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_LockPassword(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bPwdIdentifier)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_LockPassword(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bPwdIdentifier));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ProtectPage(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bPPAdd_PageNo, uint8_t bProtectionStatus)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ProtectPage(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bPPAdd_PageNo,
        bProtectionStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_LockPageProtectionCondition(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bPageNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_LockPageProtectionCondition(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bPageNo));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_GetMultipleBlockProtectionStatus(phalICode_Sw_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t bNoOfBlocks,
    uint8_t * pProtectionStates, uint16_t * pNumReceivedStates)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_GetMultipleBlockProtectionStatus(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        bBlockNo,
        bNoOfBlocks,
        pProtectionStates,
        pNumReceivedStates));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_Destroy(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pXorPwd)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_Destroy(
        pDataParams->pPalSli15693DataParams,
        bOption,
        pXorPwd));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_EnablePrivacy(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pXorPwd)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_EnablePrivacy(
        pDataParams->pPalSli15693DataParams,
        bOption,
        pXorPwd));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_64BitPasswordProtection(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_64BitPasswordProtection(
        pDataParams->pPalSli15693DataParams,
        bOption));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_StayQuietPersistent(phalICode_Sw_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_StayQuietPersistent(
        pDataParams->pPalSli15693DataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ReadSignature(phalICode_Sw_DataParams_t * pDataParams, uint8_t ** ppSign, uint16_t * pSignLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadSignature(
        pDataParams->pPalSli15693DataParams,
        ppSign,
        pSignLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ReadConfig(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockAddr, uint8_t bNoOfBlocks,
    uint8_t ** ppData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadConfig(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bBlockAddr,
        bNoOfBlocks,
        ppData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_WriteConfig(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockAddr, uint8_t * pData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WriteConfig(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bBlockAddr,
        pData));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_PickRandomID(phalICode_Sw_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_PickRandomID(
        pDataParams->pPalSli15693DataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ReadTT(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadTT(
        pDataParams->pPalSli15693DataParams,
        bOption,
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ParameterRequest(phalICode_Sw_DataParams_t * pDataParams, uint8_t * pBitRate, uint8_t * pTiming)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ParameterRequest(
        pDataParams->pPalSli15693DataParams,
        pBitRate,
        pTiming));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ParameterSelect(phalICode_Sw_DataParams_t * pDataParams, uint8_t bBitRate, uint8_t bTiming)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ParameterSelect(
        pDataParams->pPalSli15693DataParams,
        bBitRate,
        bTiming));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ReadSRAM(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockNo, uint8_t bNumBlocks,
    uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadSRAM(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        bOption,
        bBlockNo,
        bNumBlocks,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_WriteSRAM(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockNo, uint8_t bNumBlocks,
    uint8_t * pData, uint16_t wDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WriteSRAM(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        bOption,
        bBlockNo,
        bNumBlocks,
        pData,
        wDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_I2CMRead(phalICode_Sw_DataParams_t * pDataParams, uint8_t bI2CParam, uint16_t wDataLen, uint8_t * pData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_I2CMRead(
        pDataParams->pPalSli15693DataParams,
        bI2CParam,
        wDataLen,
        pData));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_I2CMWrite(phalICode_Sw_DataParams_t * pDataParams, uint8_t bI2CParam, uint8_t * pData, uint16_t wDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_I2CMWrite(
        pDataParams->pPalSli15693DataParams,
        bI2CParam,
        pData,
        wDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

#ifdef NXPBUILD__PHAL_ICODE_INTERNAL
phStatus_t phalICode_Sw_IncrCnt(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bCounter, uint16_t wValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_IncrCnt(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bCounter,
        wValue));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_ReadCnt(phalICode_Sw_DataParams_t * pDataParams, uint8_t bCounter, uint16_t * pValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadCnt(
        pDataParams->pPalSli15693DataParams,
        bCounter,
        pValue));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_WriteCnt(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bCounter, uint16_t wValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WriteCnt(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bCounter,
        wValue));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_LockCertificate(phalICode_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bCertNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_LockCertificate(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bCertNo));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_FastReadCertificate(phalICode_Sw_DataParams_t * pDataParams, uint8_t bCertNo, uint16_t wWordPtr, uint16_t wWordCount,
    uint8_t ** ppCertificate, uint16_t *pCertLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_FastReadCertificate(
        pDataParams->pPalSli15693DataParams,
        bCertNo,
        wWordPtr,
        wWordCount,
        ppCertificate,
        pCertLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_FastAuthenticateTAM(phalICode_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bAuthType,
    uint8_t bKeyID_CertNum, uint8_t * pChallenge, uint16_t wChallengeLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint16_t    PH_MEMLOC_REM wRxDataRate = 0;
    uint16_t    PH_MEMLOC_REM wTimingData = 0;
    uint16_t    PH_MEMLOC_REM wRespFlag = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[4];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    PH_UNUSED_VARIABLE(wOption);

    /* Validate Parameter */
    PHAL_ICODE_VALIDATE_AUTHTYPE(bAuthType);

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pDataParams->pPalSli15693DataParams,
        PH_ON,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Configure Fast DataRate. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetFast_DataRate(
        pDataParams->pPalSli15693DataParams,
        &wRxDataRate));

    /* Clear all the local variables. */
    (void) memset(aCmdBuff, 0x00, (size_t)sizeof(aCmdBuff));

    /* Frame and exchange the TAM message based on Authentication type. */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_FAST_AUTHENTICATE;

    /* Prepare Message based on Authenticate Type */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_DH;
    if(bAuthType == PHAL_ICODE_AUTHENTICATE_TAM_1_3)
    {
        aCmdBuff[bCmdLen++] = PHAL_ICODE_AUTH_PARAM_TAM1_3;
        aCmdBuff[bCmdLen++] = bKeyID_CertNum;
    }
    else
    {
        aCmdBuff[bCmdLen++] = PHAL_ICODE_AUTH_PARAM_TAM1_2;
    }

    /* Buffer command information. */
    wStatus = phpalSli15693_Exchange(
        pDataParams->pPalSli15693DataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmdBuff,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Buffer command information. */
    wStatus = phpalSli15693_Exchange(
        pDataParams->pPalSli15693DataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pChallenge,
        wChallengeLen,
        &pResponse,
        &wRespLen);

    /* Compute the status code. */
    wStatus = phalICode_Int_ComputeErrorCode(pDataParams->pPalSli15693DataParams, wStatus);

    /* Revert back DataRate  */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        PH_CHECK_SUCCESS_FCT(wStatus1, phpalSli15693_SetConfig(
            pDataParams->pPalSli15693DataParams,
            PHPAL_SLI15693_CONFIG_RXDATARATE,
            wRxDataRate));

        return wStatus;
    }

    /* Check if the response consists of 3 bytes of data. */
    if(wRespLen != (1U /* Barker code. */ + 2U /* Timing information */))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Get the response flag from PAL layer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_ADD_INFO,
        &wRespFlag));

    /* Check if barker code is valid. */
    if(!(((pResponse[0]) & 0x7FU /* Barker code extraction. */) == 0x27U /* Barker Code */))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Check if Done flag is not set and it's not the final response. */
    if((((pResponse[0]) & 0x80U /* Done Flag extraction. */) == 0x80U) ||
        (((wRespFlag) & 0x02U /* Response buffer flag extraction */) == 0x02U) ||
        (((wRespFlag) & 0x04U /* Final response flag extraction */) == 0x04U))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Extract timing information. */
    wTimingData = (uint16_t) ((pResponse[2] << 8U) | pResponse[1]);

    /* Update the timing received from tag */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_MS,
        wTimingData));

    /* Retrieve final data. */
    wStatus = phpalSli15693_Exchange(
        pDataParams->pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        NULL,
        0,
        &pResponse,
        &wRespLen);

    /* Compute the status code. */
    wStatus = phalICode_Int_ComputeErrorCode(pDataParams->pPalSli15693DataParams, wStatus);

    /* Revert back DataRate  */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        PH_CHECK_SUCCESS_FCT(wStatus1, phpalSli15693_SetConfig(
            pDataParams->pPalSli15693DataParams,
            PHPAL_SLI15693_CONFIG_RXDATARATE,
            wRxDataRate));

        return wStatus;
    }

    /* Check if barker code is valid. */
    if(!(((pResponse[0]) & 0x7FU /* Barker code extraction. */) == 0x27U /* Barker Code */))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Get the response flag from PAL layer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_ADD_INFO,
        &wRespFlag));

    /* Check if Done flag is set and is it the final response. */
    if(!((((pResponse[0]) & 0x80U /* Done Flag extraction. */) == 0x80U) &&
        (((wRespFlag) & 0x02U /* Response buffer flag extraction */) == 0x02U) &&
        (((wRespFlag) & 0x04U /* Final response flag extraction */) == 0x04U)))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Remove Barker code. */
    --wRespLen;
    pResponse++;

    /* Copy the response to parameters. */
    *ppResponse = pResponse;
    *pRespLen = wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_FastReadBuffer(phalICode_Sw_DataParams_t * pDataParams, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_FastReadBuffer(
        pDataParams->pPalSli15693DataParams,
        ppResponse,
        pRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}
#endif /* NXPBUILD__PHAL_ICODE_INTERNAL */





/* iCode Utility commands ------------------------------------------------------------------------------------------ */
phStatus_t phalICode_Sw_GetSerialNo(phalICode_Sw_DataParams_t * pDataParams, uint8_t ** ppUid, uint16_t * pUidLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bUidLen = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetSerialNo(
        pDataParams->pPalSli15693DataParams,
        *ppUid,
        &bUidLen));

    *pUidLen = bUidLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_SetSerialNo(phalICode_Sw_DataParams_t * pDataParams, uint8_t * pUid, uint8_t bUidLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Check if UID length is not proper. */
    if(bUidLen != PHPAL_SLI15693_UID_LENGTH)
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetSerialNo(
        pDataParams->pPalSli15693DataParams,
        pUid,
        bUidLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_GetConfig(phalICode_Sw_DataParams_t * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Retrieve the configuration settings. */
    switch(wConfig)
    {
        case PHAL_ICODE_CONFIG_ENABLE_BUFFERING:
            *pValue = pDataParams->bBuffering;
            break;

        case PHAL_ICODE_CONFIG_FLAGS:
        case PHAL_ICODE_CONFIG_ADD_INFO:
        case PHAL_ICODE_CONFIG_TIMEOUT_US:
        case PHAL_ICODE_CONFIG_TIMEOUT_MS:
            PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
                pDataParams->pPalSli15693DataParams,
                wConfig,
                pValue));
            break;

        case PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_US:
        case PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_MS:
            *pValue = pDataParams->wChallange_Timeout;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_SetConfig(phalICode_Sw_DataParams_t * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Update the configuration settings. */
    switch(wConfig)
    {
        case PHAL_ICODE_CONFIG_ENABLE_BUFFERING:
            pDataParams->bBuffering = (uint8_t) wValue;

            PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
                pDataParams->pPalSli15693DataParams,
                PHPAL_SLI15693_CONFIG_ENABLE_BUFFERING,
                wValue));
            break;

        case PHAL_ICODE_CONFIG_FLAGS:
            PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
                pDataParams->pPalSli15693DataParams,
                wConfig,
                wValue));
            break;

        case PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_US:
            pDataParams->wChallange_Timeout = wValue;
            pDataParams->wChallange_Timeout_Unit = (uint16_t) (PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_US >> 1U);
            break;

        case PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_MS:
            pDataParams->wChallange_Timeout = wValue;
            pDataParams->wChallange_Timeout_Unit = (uint16_t) (PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_MS >> 1U);
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sw_GetTagType(phalICode_Sw_DataParams_t * pDataParams, uint16_t * pTagType)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_GetTagType(
        pDataParams->pPalSli15693DataParams,
        pTagType));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

#endif /* NXPBUILD__PHAL_ICODE_SW */
