﻿/*
* Copyright 2017 - 2020, 2022 - 2025 NXP
* NXP Confidential and Proprietary.
* This software is owned or controlled by NXP and may only be used strictly
* in accordance with the applicable license terms. By expressly accepting
* such terms or by downloading, installing, activating and/or otherwise using
* the software, you are agreeing that you have read, and that you agree to
* comply with and are bound by, such license terms. If you do not agree to be
* bound by the applicable license terms, then you may not retain, install,
* activate or otherwise use the software.
*/

/** \file
* Sam NonX ICode Application Component of Reader Library Framework.
* $Author: Rajendran Kumar (nxp99556) $
* $Revision: 7467 $
* $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
*
* History:
*  CHu: Generated 02. January 2017
*
*/

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <phKeyStore.h>
#include <phCryptoSym.h>
#include <phCryptoRng.h>
#include <phpalSli15693.h>
#include <phalICode.h>

#ifdef NXPBUILD__PHAL_ICODE_SAM_NONX

#include <phhalHw_SamAV3_Cmd.h>
#include "phalICode_Sam_NonX.h"
#include "../phalICode_Int.h"

phStatus_t phalICode_SamAV3_NonX_Init(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, phhalHw_SamAV3_DataParams_t * pHalSamDataParams,
    void * pPalSli15693DataParams, void * pCryptoDataParams, void * pCryptoRngDataParams, void * pKeyStoreDataParams)
{
    /* Validate the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_ICODE);
    PH_ASSERT_NULL_DATA_PARAM(pHalSamDataParams, PH_COMP_AL_ICODE);
    PH_ASSERT_NULL_PARAM(pPalSli15693DataParams, PH_COMP_AL_ICODE);

    /* Check the size. */
    if(sizeof(phalICode_SamAV3_NonX_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_ICODE);
    }

    /* Initialize the structure members. */
    pDataParams->wId                    = PH_COMP_AL_ICODE | PHAL_ICODE_SAMAV3_NONX_ID;
    pDataParams->pHalSamDataParams      = pHalSamDataParams;
    pDataParams->pPalSli15693DataParams = pPalSli15693DataParams;
    pDataParams->pCryptoDataParams      = pCryptoDataParams;
    pDataParams->pCryptoRngDataParams   = pCryptoRngDataParams;
    pDataParams->pKeyStoreDataParams    = pKeyStoreDataParams;
    pDataParams->bBuffering             = PH_OFF;

    pDataParams->wChallange_Timeout = PHAL_ICODE_TIMEOUT_LONG_US;
    pDataParams->wChallange_Timeout_Unit = (uint16_t) (PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_US >> 1U);

    /* Reset the random number buffer. */
    (void) memset(pDataParams->aRnd_Challenge, 0x00U, sizeof(pDataParams->aRnd_Challenge));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}





/* iCode Mandatory commands ---------------------------------------------------------------------------------------- */
phStatus_t phalICode_Sam_NonX_Inventory(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t * pDsfid, uint8_t * pUid)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_Inventory(
        pDataParams->pPalSli15693DataParams,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        pDsfid,
        pUid));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_StayQuiet(phalICode_SamAV3_NonX_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_StayQuiet(pDataParams->pPalSli15693DataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}





/* iCode Optional commands ----------------------------------------------------------------------------------------- */
phStatus_t phalICode_Sam_NonX_ReadSingleBlock(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockNo, uint8_t ** ppData,
    uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadSingleBlock(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bBlockNo,
        ppData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_WriteSingleBlock(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockNo, uint8_t * pData,
    uint8_t bDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WriteSingleBlock(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bBlockNo,
        pData,
        bDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_LockBlock(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_LockBlock(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bBlockNo));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ReadMultipleBlocks(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockNo, uint8_t bNumBlocks,
    uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadMultipleBlocks(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        bOption,
        bBlockNo,
        bNumBlocks,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_Select(phalICode_SamAV3_NonX_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_Select(pDataParams->pPalSli15693DataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ResetToReady(phalICode_SamAV3_NonX_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_ResetToReady(pDataParams->pPalSli15693DataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_WriteAFI(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bAfi)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WriteAFI(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bAfi));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_LockAFI(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_LockAFI(
        pDataParams->pPalSli15693DataParams,
        bOption));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_WriteDSFID(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bDsfid)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WriteDSFID(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bDsfid));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_LockDSFID(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_LockDSFID(
        pDataParams->pPalSli15693DataParams,
        bOption));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_GetSystemInformation(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t ** ppSystemInfo, uint16_t * pSystemInfoLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_GetSystemInformation(
        pDataParams->pPalSli15693DataParams,
        ppSystemInfo,
        pSystemInfoLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_GetMultipleBlockSecurityStatus(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t bNoOfBlocks,
    uint8_t * pStatus, uint16_t * pStatusLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_GetMultipleBlockSecurityStatus(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        bBlockNo,
        bNoOfBlocks,
        pStatus,
        pStatusLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_FastReadMultipleBlocks(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockNo, uint8_t bNumBlocks,
    uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_FastReadMultipleBlocks(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        bOption,
        bBlockNo,
        bNumBlocks,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ExtendedReadSingleBlock(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint16_t wBlockNo,
    uint8_t ** ppData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ExtendedReadSingleBlock(
        pDataParams->pPalSli15693DataParams,
        bOption,
        wBlockNo,
        ppData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ExtendedWriteSingleBlock(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint16_t wBlockNo,
    uint8_t * pData, uint8_t bDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ExtendedWriteSingleBlock(
        pDataParams->pPalSli15693DataParams,
        bOption,
        wBlockNo,
        pData,
        bDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ExtendedLockBlock(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint16_t wBlockNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ExtendedLockBlock(
        pDataParams->pPalSli15693DataParams,
        bOption,
        wBlockNo));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ExtendedReadMultipleBlocks(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint16_t wBlockNo,
    uint16_t wNumBlocks, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ExtendedReadMultipleBlocks(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        bOption,
        wBlockNo,
        wNumBlocks,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_AuthenticateTAM(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint16_t wOption,
    uint8_t bKeyNo, uint8_t bKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wRespFlag = 0;
    uint8_t     PH_MEMLOC_REM bOption = 0;

    uint8_t     PH_MEMLOC_REM aIChallenge[PHAL_ICODE_RANDOM_NUMBER_MAX_SIZE];

    uint8_t     PH_MEMLOC_REM aCmdBuff[14U];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Extract Parameters */
    bOption = (uint8_t) (wOption & 0x000FU);

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pDataParams->pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Clear all the local variables. */
    (void) memset(aCmdBuff, 0x00, (size_t)sizeof(aCmdBuff));
    (void) memset(aIChallenge, 0x00U, (size_t) sizeof(aIChallenge));

    /* Receive the IChallange from SAM. ------------------------------------------------------------------------------ */
    wStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticateTAM1(
        pDataParams->pHalSamDataParams,
        PHHAL_HW_SAMAV3_CMD_TAM_GET_RND,
        bKeyNo,
        bKeyVer,
        pDivInput,
        bDivLen,
        &pResponse,
        &wRespLen);

    /* Verify if the response is not SUCCESS CHAINING. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
    {
        return wStatus;
    }

    /* Copy the IChallange received from SAM. */
    (void) memcpy(aIChallenge, pResponse, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);

    /* Frame TAM1 command to be sent to the card and receive TResponse. ---------------------------------------------- */
    /* Clear all the local variables. */
    bCmdLen = 0;
    (void) memset(aCmdBuff, 0x00, (size_t)sizeof(aCmdBuff));

    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_AUTHENTICATE;
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_AES;

    /* Prepare TAM1 Message */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_TAM_CUSTOMDATA_CLEAR | PHAL_ICODE_AUTHPROC_TAM;
    aCmdBuff[bCmdLen++] = bKeyNoCard;

    /* Reverse the random number received from SAM. */
    phalICode_Int_Reverse(aIChallenge, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);

    /* Add the random number. */
    (void) memcpy(&aCmdBuff[bCmdLen], aIChallenge, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);
    bCmdLen += PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM;

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pDataParams->pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pDataParams->pPalSli15693DataParams, wStatus));

    /* Get the response flag from PAL layer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_ADD_INFO,
        &wRespFlag));

    /* Check if barker code is valid. */
    if(!(((pResponse[0U]) & 0x7FU /* Barker code extraction. */) == 0x27U /* Barker Code */))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Check if Done flag is set and is it the final response. */
    if(!((((pResponse[0U]) & 0x80U /* Done Flag extraction. */) == 0x80U) &&
        (((wRespFlag) & 0x02U /* Response buffer flag extraction */) == 0x02U) &&
        (((wRespFlag) & 0x04U /* Final response flag extraction */) == 0x04U)))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Decrement the received data length to exclude the barker code. */
    --wRespLen;

    /* Reverse the response buffer */
    phalICode_Int_Reverse(&pResponse[1U], wRespLen);

    /* Send the received data from card (TResponse) to SAM for decryption and verification. -------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_AuthenticateTAM1(
        pDataParams->pHalSamDataParams,
        PHHAL_HW_SAMAV3_CMD_TAM_PROCESS_TRESPONE,
        0U,
        0U,
        &pResponse[1],
        (uint8_t) wRespLen,
        NULL,
        0U));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_AuthenticateTAM1(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bKeyNo, uint8_t bKeyVer,
    uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint16_t    PH_MEMLOC_REM wRespFlag = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    uint8_t     PH_MEMLOC_REM aIChallenge[PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM];
    uint8_t     PH_MEMLOC_REM aKey[PH_KEYSTORE_KEY_TYPE_AES128_SIZE];
    uint8_t     PH_MEMLOC_REM aCmdBuff[14];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pDataParams->pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Clear all the local variables. */
    (void) memset(aIChallenge, 0x00U, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);
    (void) memset(aKey, 0x00U, PH_KEYSTORE_KEY_TYPE_AES128_SIZE);

    /* Receive the IChallange from SAM. ------------------------------------------------------------------------------ */
    wStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticateTAM1(
        pDataParams->pHalSamDataParams,
        PHHAL_HW_SAMAV3_CMD_TAM_GET_RND,
        bKeyNo,
        bKeyVer,
        pDivInput,
        bDivLen,
        &pResponse,
        &wRespLen);

    /* Verify if the response is not SUCCESS CHAINING. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
    {
        return wStatus;
    }

    /* Copy the IChallange received from SAM. */
    (void) memset(aIChallenge, 0x00, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);
    (void) memcpy(aIChallenge, pResponse, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);

    /* Frame TAM1 command to be sent to the card and receive TResponse. ---------------------------------------------- */
    /* Clear all the local variables. */
    bCmdLen = 0;
    (void) memset(aCmdBuff, 0x00, (size_t)sizeof(aCmdBuff));

    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_AUTHENTICATE;
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_AES;

    /* Prepare TAM1 Message */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_TAM_CUSTOMDATA_CLEAR | PHAL_ICODE_AUTHPROC_TAM;
    aCmdBuff[bCmdLen++] = bKeyNoCard;

    /* Reverse the random number received from SAM. */
    phalICode_Int_Reverse(aIChallenge, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);

    /* Add the random number. */
    (void) memcpy(&aCmdBuff[bCmdLen], aIChallenge, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);
    bCmdLen += PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM;

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pDataParams->pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pDataParams->pPalSli15693DataParams, wStatus));

    /* Get the response flag from PAL layer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_ADD_INFO,
        &wRespFlag));

    /* Check if barker code is valid. */
    if(!(((pResponse[0U]) & 0x7FU /* Barker code extraction. */) == 0x27U /* Barker Code */))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Check if Done flag is set and is it the final response. */
    if(!((((pResponse[0U]) & 0x80U /* Done Flag extraction. */) == 0x80U) &&
        (((wRespFlag) & 0x02U /* Response buffer flag extraction */) == 0x02U) &&
        (((wRespFlag) & 0x04U /* Final response flag extraction */) == 0x04U)))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Decrement the received data length to exclude the barker code. */
    --wRespLen;

    /* Reverse the response buffer */
    phalICode_Int_Reverse(&pResponse[1U], wRespLen);

    /* Send the received data from card (TResponse) to SAM for decryption and verification. -------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_AuthenticateTAM1(
        pDataParams->pHalSamDataParams,
        PHHAL_HW_SAMAV3_CMD_TAM_PROCESS_TRESPONE,
        0U,
        0U,
        &pResponse[1],
        (uint8_t) wRespLen,
        NULL,
        0U));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_AuthenticateTAM2(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bKeyNo,
    uint8_t bKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivLen, uint8_t bBlockSize, uint8_t bBlockCount, uint8_t bProfile,
    uint8_t bProtMode, uint16_t wOffset, uint8_t * pCustomData, uint16_t * pCustomDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint16_t    PH_MEMLOC_REM wRespFlag = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    uint8_t     PH_MEMLOC_REM aCmdBuff[16];
    uint8_t     PH_MEMLOC_REM aIChallenge[PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pDataParams->pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Receive the IChallange from SAM. ------------------------------------------------------------------------------ */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_AuthenticateTAM2(
        pDataParams->pHalSamDataParams,
        PHHAL_HW_SAMAV3_CMD_TAM_GET_RND,
        bKeyNo,
        bKeyVer,
        pDivInput,
        bDivLen,
        bBlockSize,
        bBlockCount,
        bProtMode,
        &pResponse,
        &wRespLen));

    /* Copy the IChallange received from SAM. */
    (void) memset(aIChallenge, 0x00U, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);
    (void) memcpy(aIChallenge, pResponse, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);

/* Frame TAM2 command to be sent to the card and receive TResponse. ---------------------------------------------- */
    /* Clear all the local variables. */
    bCmdLen = 0U;
    (void) memset(aCmdBuff, 0x00U, (size_t)sizeof(aCmdBuff));

    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_AUTHENTICATE;
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_AES;

    /* Prepare TAM2 Message */
    aCmdBuff[bCmdLen++] = (uint8_t) ((bBlockSize << 3U) | (PHAL_ICODE_TAM_CUSTOMDATA_SET << 2U) | PHAL_ICODE_AUTHPROC_TAM);
    aCmdBuff[bCmdLen++] = bKeyNoCard;

    /* Reverse the random number received from SAM. */
    phalICode_Int_Reverse(aIChallenge, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);

    /* Add the random number. */
    (void) memcpy(&aCmdBuff[bCmdLen], aIChallenge, PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM);
    bCmdLen += PHAL_ICODE_RANDOM_NUMBER_SIZE_TAM;

    /* Append Profile, Offset, BlockCount and ProtMode to command buffer. */
    /* Decrementing BlockCount because the BlockCount ranges form 0 - 15 as per ISO29167 protocol. */
    aCmdBuff[bCmdLen++] = (uint8_t) (bProtMode << 4U) | (bBlockCount - 1U);
    aCmdBuff[bCmdLen++] = (uint8_t) wOffset;
    aCmdBuff[bCmdLen++] = (uint8_t) ((wOffset >> 8U) | (bProfile << 4U));

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pDataParams->pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pDataParams->pPalSli15693DataParams, wStatus));

    /* Get the response flag from PAL layer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_ADD_INFO,
        &wRespFlag));

    /* Check if barker code is valid. */
    if(!(((pResponse[0U]) & 0x7FU /* Barker code extraction. */) == 0x27U /* Barker Code */))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Check if Done flag is set and is it the final response. */
    if(!((((pResponse[0U]) & 0x80U /* Done Flag extraction. */) == 0x80U) &&
        (((wRespFlag) & 0x02U /* Response buffer flag extraction */) == 0x02U) &&
        (((wRespFlag) & 0x04U /* Final response flag extraction */) == 0x04U)))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Decrement the received data length to exclude the barker code. */
    --wRespLen;

    /* Reverse the response buffer */
    phalICode_Int_Reverse(&pResponse[1], wRespLen);

    /* Send the received data from card (TResponse) to SAM for decryption and verification. -------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_AuthenticateTAM2(
        pDataParams->pHalSamDataParams,
        PHHAL_HW_SAMAV3_CMD_TAM_PROCESS_TRESPONE,
        0U,
        0U,
        &pResponse[1U],
        (uint8_t) wRespLen,
        bBlockSize,
        bBlockCount,
        bProtMode,
        &pResponse,
        &wRespLen));


    /* Copy the custom data to internal buffer. */
    (void) memcpy(pCustomData, pResponse, wRespLen);

    /* Update the Custom data length value. */
    *pCustomDataLen = wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_AuthenticateMAM(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bKeyNo, uint8_t bKeyVer,
    uint8_t bKeyNoCard, uint8_t bPurposeMAM2, uint8_t * pDivInput, uint8_t bDivLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint16_t    PH_MEMLOC_REM wRespFlag = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    uint8_t     PH_MEMLOC_REM aIChallenge[PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM];
    uint8_t     PH_MEMLOC_REM aCmdBuff[20];
    uint8_t     PH_MEMLOC_REM aIResponse[16];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pDataParams->pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Receive the IChallange from SAM. ------------------------------------------------------------------------------ */
    wStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticateMAM1(
        pDataParams->pHalSamDataParams,
        bKeyNo,
        bKeyVer,
        pDivInput,
        bDivLen,
        bPurposeMAM2,
        &pResponse,
        &wRespLen);

    /* Verify if the response is not SUCCESS CHAINING. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
    {
        return wStatus;
    }

    /* Copy the IChallange received from SAM. */
    (void) memset(aIChallenge, 0x00, PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM);
    (void) memcpy(aIChallenge, pResponse, PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM);

    /* Frame MAM1 command to be sent to the card and receive TResponse. ---------------------------------------------- */
    /* Clear all the local variables. */
    bCmdLen = 0U;
    (void) memset(aCmdBuff, 0x00U, (size_t)sizeof(aCmdBuff));

    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_AUTHENTICATE;
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_AES;

    /* Frame the MAM1 message. */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_MAM1_STEP | PHAL_ICODE_AUTHPROC_MAM;
    aCmdBuff[bCmdLen++] = bKeyNoCard;

    /* Reverse the random number received from SAM. */
    phalICode_Int_Reverse(aIChallenge, PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM);

    /* Add the random number. */
    (void) memcpy(&aCmdBuff[bCmdLen], aIChallenge, PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM);
    bCmdLen += PHAL_ICODE_RANDOM_NUMBER_SIZE_MAM;

    /* Exchange the command to the card. */
    wStatus = phpalSli15693_Exchange(
        pDataParams->pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pDataParams->pPalSli15693DataParams, wStatus));

    /* Check if the response consists of 23 bytes of data. */
    if(wRespLen != (1U /* Barker code. */ + 22U /* Rest of MAM1 response. */))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Get the response flag from PAL layer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_FLAGS,
        &wRespFlag));

    /* Check if barker code is valid. */
    if(!(((pResponse[0]) & 0x7F /* Barker code extraction. */) == 0x27 /* Barker Code */))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Check if Done flag is set and is it the final response. */
    if((((pResponse[0U]) & 0x80U /* Done Flag extraction. */) == 0x80U) &&
        (((wRespFlag) & 0x02U /* Response buffer flag extraction */) == 0x02U) &&
        (((wRespFlag) & 0x04U /* Final response flag extraction */) == 0x04U))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Decrement the received data length to exclude the barker code. */
    --wRespLen;

    /* Reverse the TReponse Buffer .*/
    phalICode_Int_Reverse(&pResponse[1U], 22U);

    /* Send the received data from card (TResponse) to SAM for verification and framing IResponse --------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_AuthenticateMAM2(
        pDataParams->pHalSamDataParams,
        &pResponse[1U],
        (uint8_t) wRespLen,
        &pResponse,
        &wRespLen));

    /* Check if the response consists of 16 bytes of data. */
    if(wRespLen != 16U)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Copy the IResponse received from SAM. */
    (void) memset(aIResponse, 0x00U, wRespLen);
    (void) memcpy(aIResponse, pResponse, wRespLen);

    /* Frame MAM2 command to be sent to the card. -------------------------------------------------------------------- */
    bCmdLen = 0U;
    (void) memset(aCmdBuff, 0x00U, (size_t)sizeof(aCmdBuff));

    /* Frame the command. */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_AUTHENTICATE;

    aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_AES;
    aCmdBuff[bCmdLen++] = PHAL_ICODE_MAM2_STEP | PHAL_ICODE_AUTHPROC_MAM;

    /* Append the IResponse received from SAM to command buffer. */
    (void) memcpy(&aCmdBuff[bCmdLen], aIResponse, 16U);
    bCmdLen += 16U;

    /* Reverse the buffer. */
    phalICode_Int_Reverse(&aCmdBuff[3], 16);

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pDataParams->pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pDataParams->pPalSli15693DataParams, wStatus));

    /* Check if there is no response message. */
    if(wRespLen != 0x01U /* Barker Code. */)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Get the response flag from PAL layer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pDataParams->pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_FLAGS,
        &wRespFlag));

    /* Check if barker code is valid. */
    if(!(((pResponse[0U]) & 0x7FU /* Barker code extraction. */) == 0x27U /* Barker Code */))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

#ifdef NXPBUILD__PHAL_ICODE_INTERNAL
phStatus_t phalICode_Sam_NonX_WriteCertificate(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bCertNo,
    uint16_t wWordPtr, uint8_t * pCertificate, uint8_t bCertLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WriteCertificate(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bCertNo,
        wWordPtr,
        pCertificate,
        bCertLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ReadCertificate(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bCertNo, uint16_t wWordPtr,
    uint16_t wWordCount, uint8_t ** ppCertificate, uint16_t *pCertLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadCertificate(
        pDataParams->pPalSli15693DataParams,
        bCertNo,
        wWordPtr,
        wWordCount,
        ppCertificate,
        pCertLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}
#endif /* NXPBUILD__PHAL_ICODE_INTERNAL */

phStatus_t phalICode_Sam_NonX_Challenge(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint16_t wOption, uint8_t bChallengeType,
    uint8_t bKeyID_CertNum, uint8_t * pChallenge, uint16_t wChallengeLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_Challenge(
        pDataParams->pPalSli15693DataParams,
        pDataParams->pCryptoRngDataParams,
        wOption,
        bChallengeType,
        pDataParams->wChallange_Timeout,
        pDataParams->wChallange_Timeout_Unit,
        bKeyID_CertNum,
        pChallenge,
        wChallengeLen,
        pDataParams->aRnd_Challenge));

    /* Update ChallengeType to DataParams. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetChallengeType(pDataParams, bChallengeType));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ReadBuffer(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bVerify, uint8_t bKeyNo, uint8_t bKeyVer,
    uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadBuffer(
        PHAL_ICODE_SAMAV3_NONX_ID,
        pDataParams->pPalSli15693DataParams,
        pDataParams->pCryptoDataParams,
        pDataParams->pKeyStoreDataParams,
        pDataParams->aRnd_Challenge,
        0xFFU,
        bVerify,
        bKeyNo,
        bKeyVer,
        ppResponse,
        pRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ExtendedGetSystemInformation(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bInfoParams, uint8_t ** ppSystemInfo,
    uint16_t * pSystemInfoLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ExtendedGetSystemInformation(
        pDataParams->pPalSli15693DataParams,
        bInfoParams,
        ppSystemInfo,
        pSystemInfoLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ExtendedGetMultipleBlockSecurityStatus(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint16_t wBlockNo, uint16_t wNoOfBlocks,
    uint8_t * pStatus, uint16_t * pStatusLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ExtendedGetMultipleBlockSecurityStatus(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        wBlockNo,
        wNoOfBlocks,
        pStatus,
        pStatusLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ExtendedFastReadMultipleBlocks(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint16_t wBlockNo,
    uint16_t wNumBlocks, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ExtendedFastReadMultipleBlocks(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        bOption,
        wBlockNo,
        wNumBlocks,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}





/* iCode Custom commands ------------------------------------------------------------------------------------------- */
phStatus_t phalICode_Sam_NonX_InventoryRead(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bBlockNo, uint8_t bNoOfBlocks, uint8_t * pUid, uint8_t * pUidLen, uint8_t * pData,
    uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_InventoryRead(
        pDataParams->pPalSli15693DataParams,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        bBlockNo,
        bNoOfBlocks,
        pUid,
        pUidLen,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_InventoryReadExtended(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bExtendedOptions, uint8_t * pCID, uint8_t bBlockNo, uint8_t bNoOfBlocks, uint8_t * pCDIDOut,
    uint8_t * pUid, uint8_t * pUidLen, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_InventoryReadExtended(
        pDataParams->pPalSli15693DataParams,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        bExtendedOptions,
        pCID,
        bBlockNo,
        bNoOfBlocks,
        pCDIDOut,
        pUid,
        pUidLen,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_FastInventoryRead(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bBlockNo, uint8_t bNoOfBlocks, uint8_t * pUid, uint8_t * pUidLen, uint8_t * pData,
    uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_FastInventoryRead(
        pDataParams->pPalSli15693DataParams,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        bBlockNo,
        bNoOfBlocks,
        pUid,
        pUidLen,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_FastInventoryReadExtended(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bExtendedOptions, uint8_t * pCID, uint8_t bBlockNo, uint8_t bNoOfBlocks, uint8_t * pCDIDOut,
    uint8_t * pUid, uint8_t * pUidLen, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_FastInventoryReadExtended(
        pDataParams->pPalSli15693DataParams,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        bExtendedOptions,
        pCID,
        bBlockNo,
        bNoOfBlocks,
        pCDIDOut,
        pUid,
        pUidLen,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_SetEAS(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetEAS(pDataParams->pPalSli15693DataParams, bOption));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ResetEAS(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ResetEAS(pDataParams->pPalSli15693DataParams, bOption));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_LockEAS(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_LockEAS(pDataParams->pPalSli15693DataParams, bOption));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_EASAlarm(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pEasIdValue, uint8_t bEasIdMaskLen,
    uint8_t ** ppEas, uint16_t * pEasLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_EASAlarm(
        pDataParams->pPalSli15693DataParams,
        bOption,
        pEasIdValue,
        bEasIdMaskLen,
        ppEas,
        pEasLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_PasswordProtectEAS(phalICode_SamAV3_NonX_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_PasswordProtectEAS(pDataParams->pPalSli15693DataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_PasswordProtectAFI(phalICode_SamAV3_NonX_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_PasswordProtectAFI(pDataParams->pPalSli15693DataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_WriteEAS_ID(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pEasIdValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WriteEAS_ID(pDataParams->pPalSli15693DataParams, bOption, pEasIdValue));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ReadEPC(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t ** ppEpc, uint16_t * pEpcLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadEPC(pDataParams->pPalSli15693DataParams, ppEpc, pEpcLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_GetNXPSystemInformation(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t ** ppSystemInfo, uint16_t * pSystemInfoLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_GetNXPSystemInformation(
        pDataParams->pPalSli15693DataParams,
        ppSystemInfo,
        pSystemInfoLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_InventoryPageRead(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bPageNo, uint8_t bNoOfPages, uint8_t ** ppUid, uint8_t * pUidLen, uint8_t ** ppData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aUid[PHPAL_SLI15693_UID_LENGTH];
    uint8_t     PH_MEMLOC_REM aData[PHAL_ICODE_BLOCK_SIZE * PHAL_ICODE_MAX_BLOCKS];

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_InventoryPageRead(
        pDataParams->pPalSli15693DataParams,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        bPageNo,
        bNoOfPages,
        aUid,
        pUidLen,
        aData,
        pDataLen));

    /* Copy the data to parameters. */
    *ppUid = aUid;
    *ppData = aData;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_FastInventoryPageRead(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bPageNo, uint8_t bNoOfPages, uint8_t ** ppUid, uint8_t * pUidLen, uint8_t ** ppData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aUid[PHPAL_SLI15693_UID_LENGTH];
    uint8_t     PH_MEMLOC_REM aData[PHAL_ICODE_BLOCK_SIZE * PHAL_ICODE_MAX_BLOCKS];

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_FastInventoryPageRead(
        pDataParams->pPalSli15693DataParams,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        bPageNo,
        bNoOfPages,
        aUid,
        pUidLen,
        aData,
        pDataLen));

    /* Copy the data to parameters. */
    *ppUid = aUid;
    *ppData = aData;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_GetRandomNumber(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t ** ppRnd, uint16_t * pRndLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_GetRandomNumber(pDataParams->pPalSli15693DataParams, ppRnd, pRndLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_SetPassword(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bPwdIdentifier, uint8_t * pXorPwd)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetPassword(pDataParams->pPalSli15693DataParams, bOption, bPwdIdentifier, pXorPwd));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_WritePassword(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bPwdIdentifier, uint8_t * pPwd)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WritePassword(pDataParams->pPalSli15693DataParams, bOption, bPwdIdentifier, pPwd));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_LockPassword(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bPwdIdentifier)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_LockPassword(pDataParams->pPalSli15693DataParams, bOption, bPwdIdentifier));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ProtectPage(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bPPAdd_PageNo, uint8_t bProtectionStatus)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ProtectPage(pDataParams->pPalSli15693DataParams, bOption, bPPAdd_PageNo, bProtectionStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_LockPageProtectionCondition(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bPageNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_LockPageProtectionCondition(pDataParams->pPalSli15693DataParams, bOption, bPageNo));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_GetMultipleBlockProtectionStatus(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t bNoOfBlocks,
    uint8_t * pProtectionStates, uint16_t * pNumReceivedStates)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_GetMultipleBlockProtectionStatus(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        bBlockNo,
        bNoOfBlocks,
        pProtectionStates,
        pNumReceivedStates));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_Destroy(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pXorPwd)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_Destroy(pDataParams->pPalSli15693DataParams, bOption, pXorPwd));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_EnablePrivacy(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pXorPwd)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_EnablePrivacy(
        pDataParams->pPalSli15693DataParams,
        bOption,
        pXorPwd));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_64BitPasswordProtection(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_64BitPasswordProtection(pDataParams->pPalSli15693DataParams, bOption));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_StayQuietPersistent(phalICode_SamAV3_NonX_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_StayQuietPersistent(pDataParams->pPalSli15693DataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ReadSignature(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t ** ppSign, uint16_t * pSignLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadSignature(pDataParams->pPalSli15693DataParams, ppSign, pSignLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ReadConfig(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockAddr, uint8_t bNoOfBlocks,
    uint8_t ** ppData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadConfig(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bBlockAddr,
        bNoOfBlocks,
        ppData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_WriteConfig(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockAddr, uint8_t * pData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WriteConfig(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bBlockAddr,
        pData));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_PickRandomID(phalICode_SamAV3_NonX_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_PickRandomID(pDataParams->pPalSli15693DataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ReadTT(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadTT(
        pDataParams->pPalSli15693DataParams,
        bOption,
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ParameterRequest(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t * pBitRate, uint8_t * pTiming)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ParameterRequest(pDataParams->pPalSli15693DataParams, pBitRate, pTiming));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ParameterSelect(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bBitRate, uint8_t bTiming)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ParameterSelect(pDataParams->pPalSli15693DataParams, bBitRate, bTiming));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ReadSRAM(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockNo, uint8_t bNumBlocks,
    uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadSRAM(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        bOption,
        bBlockNo,
        bNumBlocks,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_WriteSRAM(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bBlockNo, uint8_t bNumBlocks,
    uint8_t * pData, uint16_t wDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WriteSRAM(
        pDataParams->pPalSli15693DataParams,
        pDataParams->bBuffering,
        bOption,
        bBlockNo,
        bNumBlocks,
        pData,
        wDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_I2CMRead(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bI2CParam, uint16_t wDataLen, uint8_t * pData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_I2CMRead(
        pDataParams->pPalSli15693DataParams,
        bI2CParam,
        wDataLen,
        pData));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_I2CMWrite(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bI2CParam, uint8_t * pData, uint16_t wDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_I2CMWrite(
        pDataParams->pPalSli15693DataParams,
        bI2CParam,
        pData,
        wDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

#ifdef NXPBUILD__PHAL_ICODE_INTERNAL
phStatus_t phalICode_Sam_NonX_IncrCnt(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bCounter, uint16_t wValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_IncrCnt(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bCounter,
        wValue));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_ReadCnt(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bCounter, uint16_t * pValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadCnt(
        pDataParams->pPalSli15693DataParams,
        bCounter,
        pValue));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_WriteCnt(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bCounter, uint16_t wValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_WriteCnt(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bCounter,
        wValue));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_LockCertificate(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bOption, uint8_t bCertNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_LockCertificate(
        pDataParams->pPalSli15693DataParams,
        bOption,
        bCertNo));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_FastReadCertificate(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t bCertNo, uint16_t wWordPtr,
    uint16_t wWordCount, uint8_t ** ppCertificate, uint16_t *pCertLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_FastReadCertificate(
        pDataParams->pPalSli15693DataParams,
        bCertNo,
        wWordPtr,
        wWordCount,
        ppCertificate,
        pCertLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_FastReadBuffer(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_FastReadBuffer(
        pDataParams->pPalSli15693DataParams,
        ppResponse,
        pRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}
#endif /* NXPBUILD__PHAL_ICODE_INTERNAL */





/* iCode Utility commands ------------------------------------------------------------------------------------------ */
phStatus_t phalICode_Sam_NonX_GetSerialNo(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t ** ppUid, uint16_t * pUidLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bUidLen = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetSerialNo(
        pDataParams->pPalSli15693DataParams,
        *ppUid,
        &bUidLen));

    *pUidLen = bUidLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_SetSerialNo(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint8_t * pUid, uint8_t bUidLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Check if UID length is not proper. */
    if(bUidLen != PHPAL_SLI15693_UID_LENGTH)
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetSerialNo(
        pDataParams->pPalSli15693DataParams,
        pUid,
        bUidLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_GetConfig(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Update the configuration settings. */
    switch(wConfig)
    {
        case PHAL_ICODE_CONFIG_ENABLE_BUFFERING:
            *pValue = pDataParams->bBuffering;
            break;

        case PHAL_ICODE_CONFIG_FLAGS:
        case PHAL_ICODE_CONFIG_ADD_INFO:
        case PHAL_ICODE_CONFIG_TIMEOUT_US:
        case PHAL_ICODE_CONFIG_TIMEOUT_MS:
            PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
                pDataParams->pPalSli15693DataParams,
                wConfig,
                pValue));
            break;

        case PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_US:
        case PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_MS:
            *pValue = pDataParams->wChallange_Timeout;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_SetConfig(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Update the configuration settings. */
    switch(wConfig)
    {
        case PHAL_ICODE_CONFIG_ENABLE_BUFFERING:
            pDataParams->bBuffering = (uint8_t) wValue;

            PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
                pDataParams->pPalSli15693DataParams,
                PHPAL_SLI15693_CONFIG_ENABLE_BUFFERING,
                wValue));
            break;

        case PHAL_ICODE_CONFIG_FLAGS:
            PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
                pDataParams->pPalSli15693DataParams,
                wConfig,
                wValue));
            break;

        case PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_US:
            pDataParams->wChallange_Timeout = wValue;
            pDataParams->wChallange_Timeout_Unit = (uint16_t) (PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_US >> 1U);
            break;

        case PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_MS:
            pDataParams->wChallange_Timeout = wValue;
            pDataParams->wChallange_Timeout_Unit = (uint16_t) (PHAL_ICODE_CONFIG_CHALLENGE_TIMEOUT_MS >> 1U);
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Sam_NonX_GetTagType(phalICode_SamAV3_NonX_DataParams_t * pDataParams, uint16_t * pTagType)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_GetTagType(
        pDataParams->pPalSli15693DataParams,
        pTagType));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}
#endif /* NXPBUILD__PHAL_ICODE_SAM_NONX */
