/*
 * Copyright 2016 - 2018, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic MIFARE PRIME NTAG Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHALMFPRIMENTAG_H
#define PHALMFPRIMENTAG_H

#include <ph_Status.h>
#include <phhalHw.h>
#include <phpalMifare.h>
#include <ph_TypeDefs.h>
#include <ph_RefDefs.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/***************************************************************************************************************************************/
/* Software Dataparams and Initialization Interface.																				   */
/***************************************************************************************************************************************/
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW

/** \defgroup phalMfPrimeNtag_Sw Component : Software
 * @{
 */

#define PHAL_MFPRIMENTAG_SW_ID												0x01U	/**< ID for Software MIFARE PRIME NTAG layer. */

/** \name Sizes */
/** @{ */
#define PHAL_MFPRIMENTAG_SIZE_TI											4U		/**< Size of Transaction Identifier. */
#define PHAL_MFPRIMENTAG_SIZE_MAC											16U		/**< Size of (untruncated) MAC. */
#define PHAL_MFPRIMENTAG_BLOCK_SIZE											16U		/**< Block Size */
#define PHAL_MFPRIMENTAG_MAX_WRITE_SIZE										0xFFFDU	/**< Max size in a Write function. */
/** @} */

/** \brief MIFARE PRIME NTAG Software parameter structure  */
typedef struct
{
	uint16_t wId;																	/**< Layer ID for this component, NEVER MODIFY! */
	void * pPalMifareDataParams;                                					/**< Pointer to the parameter structure of the palMifare component. */
	void * pKeyStoreDataParams;                                 					/**< Pointer to the parameter structure of the KeyStore layer. */
	void * pCryptoDataParamsEnc;                                					/**< Pointer to the parameter structure of the Crypto layer for encryption. */
	void * pCryptoDataParamsMac;                                					/**< Pointer to the parameter structure of the CryptoMAC. */
	void * pCryptoRngDataParams;                                					/**< Pointer to the parameter structure of the CryptoRng layer. */
	void * pHalDataParams;                                      					/**< Pointer to the HAL parameters structure. */
	uint8_t bSesAuthENCKey[24];                                 					/**< Session key for this authentication */
	uint8_t bKeyNo;                                             					/**< key number against which this authentication is done */
	uint8_t bIv[16];                                            					/**< Max size of IV can be 16 bytes */
	uint8_t bAuthMode;                                          					/**< Authenticate (0x0A), AuthISO (0x1A), AuthAES (0xAA) */
	uint8_t pAid[3];                                            					/**< Aid of the currently selected application */
	uint8_t bCryptoMethod;                                      					/**< DES,3DES, 3K3DES or AES, LRP  */
	uint8_t bWrappedMode;                                       					/**< Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 APDUs. */
	uint16_t wCrc;                                              					/**< 2 Byte CRC initial value in Authenticate mode. */
	uint32_t dwCrc;                                             					/**< 4 Byte CRC initial value in 0x1A, 0xAA mode. */
	uint16_t wAdditionalInfo;                                   					/**< Specific error codes for Desfire generic errors. */
	uint32_t dwPayLoadLen;                                       					/**< Amount of data to be read. Required for Enc read to verify CRC. */
	uint16_t wCmdCtr;                                          						/**< Command count within transaction. */
	uint8_t bTi[PHAL_MFPRIMENTAG_SIZE_TI];                      					/**< Transaction Identifier. */
	uint8_t bSesAuthMACKey[16];                                 					/**< Authentication MAC key for the session. */
	uint8_t bKeySessionAuthMaster[16];                          					/**< Session Auth master key. */
	uint8_t pUnprocByteBuff[PHAL_MFPRIMENTAG_SIZE_MAC];        						/**< Buffer containing unprocessed bytes for read mac answer stream. */
	uint8_t bNoUnprocBytes;                                     					/**< Amount of data in the pUnprocByteBuff. */
	uint8_t bLastBlockBuffer[16];                               					/**< Buffer to store last Block of encrypted data in case of chaining. */
	uint8_t bLastBlockIndex;                                    					/**< Last Block Buffer Index. */
	void * pTMIDataParams;                                      					/**< Pointer to the parameter structure for collecting TMI. */
	uint8_t bShortLenApdu;	     													/**< Parameter for force set Short Length APDU in case of BIG ISO read. */
	uint8_t bSdmOptionVal;															/**< Parameter to set SDM . */
} phalMfprimeNtag_Sw_DataParams_t;

/**
 * \brief Initialise this layer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phalMfprimeNtag_Sw_Init(
		phalMfprimeNtag_Sw_DataParams_t * pDataParams,								/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wSizeOfDataParams,													/**< [In] Specifies the size of the data parameter structure */
		void * pPalMifareDataParams,												/**< [In] Pointer to a palMifare component context. */
		void * pKeyStoreDataParams,													/**< [In] Pointer to Key Store data parameters. */
		void * pCryptoDataParamsEnc,												/**< [In] Pointer to a Crypto component context for encryption. */
		void * pCryptoDataParamsMac,												/**< [In] Pointer to a CryptoMAC component context. */
		void * pCryptoRngDataParams,												/**< [In] Pointer to a CryptoRng component context. */
		void * pTMIDataParams,														/**< [In] Pointer to a TMI component. */
		void * pHalDataParams														/**< [In] Pointer to the HAL parameters structure. */
	);

/**
 * end of group phalMfPrimeNtag_Sw
 * @}
 */

#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */

/***************************************************************************************************************************************/
/* Sam NonX Dataparams and Initialization Interface.																				   */
/***************************************************************************************************************************************/
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX

/**
 * \defgroup phalMfPrimeNtag_Sam_NonX Component : Sam_NonX
 * @{
 */

#define PHAL_MFPRIMENTAG_SAMAV3_NONX_ID										0x02U	/**< ID for SamAV3 Non X MIFARE PRIME NTAG layer. */

/** \brief MIFARE PRIME NTAG SAM AV3 NonX parameter structure  */
typedef struct
{
	uint16_t wId;																	/**< Layer ID for this component, NEVER MODIFY! */
	phhalHw_SamAV3_DataParams_t * pHalSamDataParams;    							/**< Pointer to the parameter structure of the hal of the SAM layer. */
	void * pPalMifareDataParams;                        							/**< Pointer to the parameter structure of the palMifare component. */
	uint8_t bAuthMode;                                  							/**< Auth. Mode 0x0A or Ox1A or OxAA. */
	uint8_t bKeyNo;                                     							/**< Key number against which authenticated. */
	uint8_t aAid[3];                                    							/**< Currently selected application Id. */
	uint16_t wAdditionalInfo;                           							/**< Specific error codes for Desfire generic errors or error codes returned by the card. */
} phalMfPrimeNtag_SamAV3_NonX_DataParams_t;

/**
 * \brief Initializes the MIFARE PRIME NTAG Sam AV3 component in Non X mode (S mode).
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS for Invaid Dataparams
 * \retval #PH_ERR_INVALID_PARAMETER for invalid parameter values.
 */
phStatus_t phalMfPrimeNtag_SamAV3_NonX_Init(
		phalMfPrimeNtag_SamAV3_NonX_DataParams_t * pDataParams,						/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wSizeOfDataParams,													/**< [In] Specifies the size of the data parameter structure */
		phhalHw_SamAV3_DataParams_t * pHalSamDataParams,							/**< [In] Pointer to the HAL parameter structure of the SAM. */
		void * pPalMifareDataParams													/**< [In] Pointer to Pal Mifare parameter structure. */
	);

/**
 * end of group phalMfPrimeNtag_Sam_NonX
 * @}
 */
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */

/***************************************************************************************************************************************/
/* SAM X Dataparams and Initialization Interface.																				       */
/***************************************************************************************************************************************/

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X

/**
 * \defgroup phalMfPrimeNtag_Sam_X Component : Sam_X
 * @{
 */

#define PHAL_MFPRIMENTAG_SAMAV3_X_ID										0x03U   /**< ID for SamAV3 X-Mode MIFARE PRIME NTAG layer. */

/** \brief MIFARE MIFARE PRIME NTAG SAM AV3 X parameter structure  */
typedef struct
{
	uint16_t wId;																	/**< Layer ID for this component, NEVER MODIFY! */
	phhalHw_SamAV3_DataParams_t * pHalSamDataParams;    							/**< Pointer to the parameter structure of the hal of the SAM layer. */
	uint8_t bAuthMode;                                  							/**< Auth Mode. 0x0A or 0x1A or 0xAA. */
	uint8_t bKeyNo;                                     							/**< Key number against which authenticated. */
	uint8_t aAid[3];                                    							/**< Currently selected application Id. */
	uint16_t wAdditionalInfo;                           							/**< Specific error codes for Desfire generic errors or error codes returned by the card. */
} phalMfPrimeNtag_SamAV3_X_DataParams_t;

/**
 * \brief Initialise SamAV3 X-Mode layer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phalMfPrimeNtag_SamAV3_X_Init(
		phalMfPrimeNtag_SamAV3_X_DataParams_t * pDataParams,						/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wSizeOfDataParams,													/**< [In] Specifies the size of the data parameter structure */
		phhalHw_SamAV3_DataParams_t * pHalSamDataParams								/**< [In] Pointer to the HAL parameter structure of the SAM. */
	);

/**
 * end of group phalMfPrimeNtag_Sam_X
 * @}
 */

#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */



/***************************************************************************************************************************************/
/* MIFARE PRIME NTAG Generic command declarations.     																			       */
/***************************************************************************************************************************************/
#ifdef NXPBUILD__PHAL_MFPRIMENTAG

/** \defgroup phalMfPrimeNtag MIFARE PRIME NTAG
 * \brief These Functions implement the MIFARE PRIME NTAG commands.
 * @{
 */

/** \name Other Options for various Functions */
/** @{ */
#define PHAL_MFPRIMENTAG_ORIGINALITY_KEY_FIRST								0x01U	/**< Originality key for AES */
#define PHAL_MFPRIMENTAG_ORIGINALITY_KEY_LAST								0x04U	/**< Originality key for AES  */
#define PHAL_MFPRIMENTAG_APP_KEY_LAST										0x02U	/**< Numebr of application keys. */
#define PHAL_MFPRIMENTAG_MAC_DATA_INCOMPLETE								0x01U	/**< Option for indicating more data to come in next call for MAC calculation */
#define PHAL_MFPRIMENTAG_COMMUNICATION_MAC_ON_RC							0x02U	/**< MAC is appended over response  */
#define PHAL_MFPRIMENTAG_AUTHENTICATE_RESET									0x08U	/**< authentication state shall be reset */
/** @} */

/** \name The communication mode to be used. */
/** @{ */
#define PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN								0x00U	/**< Plain mode of communication. */
#define PHAL_MFPRIMENTAG_COMMUNICATION_MACD									0x10U	/**< MAC mode of communication. */
#define PHAL_MFPRIMENTAG_COMMUNICATION_ENC									0x30U	/**< Enciphered mode of communication. */
/** @} */

/** \name Error Codes mapping to custom values. */
/** @{ */
#define PHAL_MFPRIMENTAG_ERR_FORMAT						(PH_ERR_CUSTOM_BEGIN + 0)	/**< MF PRIME NTAG Response - No changes done to backup files. */
#define PHAL_MFPRIMENTAG_ERR_NO_SUCH_KEY				(PH_ERR_CUSTOM_BEGIN + 2)	/**< MF PRIME NTAG Invalid key number specified. */
#define PHAL_MFPRIMENTAG_ERR_PERMISSION_DENIED			(PH_ERR_CUSTOM_BEGIN + 3)	/**< MF PRIME NTAG Current configuration/status does not allow the requested command. */
#define PHAL_MFPRIMENTAG_ERR_BOUNDARY_ERROR				(PH_ERR_CUSTOM_BEGIN + 5)	/**< MF PRIME NTAG Attempt to read/write data from/to beyond the files/record's limits. */
#define PHAL_MFPRIMENTAG_ERR_COMMAND_ABORTED			(PH_ERR_CUSTOM_BEGIN + 6)	/**< MF PRIME NTAG Previous cmd not fully completed. Not all frames were requested or provided by the PCD. */
#define PHAL_MFPRIMENTAG_ERR_DUPLICATE					(PH_ERR_CUSTOM_BEGIN + 8)	/**< MF PRIME NTAG File/Application with same number already exists. */
#define PHAL_MFPRIMENTAG_ERR_FILE_NOT_FOUND				(PH_ERR_CUSTOM_BEGIN + 9)	/**< MF PRIME NTAG Specified file number does not exist. */
#define PHAL_MFPRIMENTAG_ERR_PICC_CRYPTO				(PH_ERR_CUSTOM_BEGIN + 10)	/**< MF PRIME NTAG Crypto error returned by PICC. */
#define PHAL_MFPRIMENTAG_ERR_PARAMETER_ERROR			(PH_ERR_CUSTOM_BEGIN + 11)	/**< MF PRIME NTAG Parameter value error returned by PICC. */
#define PHAL_MFPRIMENTAG_ERR_DF_GEN_ERROR				(PH_ERR_CUSTOM_BEGIN + 12)	/**< MF PRIME NTAG DesFire Generic error. Check additional Info. */
#define PHAL_MFPRIMENTAG_ERR_DF_7816_GEN_ERROR			(PH_ERR_CUSTOM_BEGIN + 13)	/**< MF PRIME NTAG ISO 7816 Generic error. Check Additional Info. */
#define PHAL_MFPRIMENTAG_ERR_CMD_INVALID				(PH_ERR_CUSTOM_BEGIN + 14)	/**< MF PRIME NTAG ISO 7816 Generic error. Check Additional Info. */
/** @} */

/** \name Key Types */
/** @{ */
#define PHAL_MFPRIMENTAG_KEY_TYPE_2K3DES									0x00U	/**< 2 Key Triple Des */
#define PHAL_MFPRIMENTAG_KEY_TYPE_3K3DES									0x01U	/**< 3 Key Triple Des */
#define PHAL_MFPRIMENTAG_KEY_TYPE_AES128									0x02U	/**< AES 128 Key */
/** @} */




/* MIFARE PRIME NTAG secure messaging related commands. ------------------------------------------------------------------------------ */
/**
 * \defgroup phalMfPrimeNtag_SecureMessaging Commands_SecureMessaging
 * \brief These Components implement the MIFARE MFPRIMENTAG Secure Messaging related commands.
 * @{
 */

/** \name Authentication Modes */
/** @{ */
#define PHAL_MFPRIMENTAG_NOT_AUTHENTICATED									0xFFU	/**< No authentication. */
#define PHAL_MFPRIMENTAG_AUTHENTICATEEV2									0x71U	/**< EV2 First Authentication; 0x71. */
#define PHAL_MFPRIMENTAG_AUTHNONFIRST_NON_LRP								0x0000U	/**< Non First Auth in regular EV2 auth Mode. */
#define PHAL_MFPRIMENTAG_AUTHFIRST_NON_LRP									0x0001U	/**< First Auth in regular EV2 auth Mode. */
/** @} */

#ifdef NXPBUILD__PH_CRYPTOSYM
/** \name Diversification options to be used with Authenticate command. */
/** @{ */
#define PHAL_MFPRIMENTAG_NO_DIVERSIFICATION           						0xFFFFU	/**< No diversification. */
#define PHAL_MFPRIMENTAG_DIV_METHOD_ENCR			PH_CRYPTOSYM_DIV_MODE_DESFIRE	/**< Encryption based method of diversification. */
#define PHAL_MFPRIMENTAG_DIV_METHOD_CMAC		PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS	/**< CMAC based method of diversification. */
/** @} */

/**
 * \brief Performs an First or Non First Authentication depending upon bFirstAuth Parameter. This will be using the AES128 keys and will
 * generate and verify the contents based on generic AES algorithm.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfprimeNtag_AuthenticateEv2(
		void * pDataParams,															/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bFirstAuth,     													/**< [In] One of the below options.
																					 *			\arg #PHAL_MFPRIMENTAG_AUTHNONFIRST_NON_LRP
																					 *			\arg #PHAL_MFPRIMENTAG_AUTHFIRST_NON_LRP
																					 */
		uint16_t wOption,       													/**< [In] Diversification option can be one of
																					 *			\arg #PHAL_MFPRIMENTAG_NO_DIVERSIFICATION
																					 *			\arg #PHAL_MFPRIMENTAG_DIV_METHOD_ENCR
																					 *			\arg #PHAL_MFPRIMENTAG_DIV_METHOD_CMAC
																					 */
		uint16_t wKeyNo,        													/**< [In] Key number in keystore of software or SAM. */
		uint16_t wKeyVer,       													/**< [In] Key version in keystore of software or SAM. */
		uint8_t bKeyNoCard,     													/**< [In] Key number on card. */
		uint8_t * pDivInput,    													/**< [In] Diversification input. Can be NULL. */
		uint8_t bDivLen,        													/**< [In] Length of diversification input max 31B. */
		uint8_t bLenPcdCapsIn,  													/**< [In] Length of PcdCapsIn. Always zero for following authentication */
		uint8_t * bPcdCapsIn,    													/**< [In] PCD Capabilities. */
		uint8_t * bPcdCapsOut,   													/**< [Out] PCD Capabilities. */
		uint8_t * bPdCapsOut     													/**< [Out] PD Capabilities. */
	);
#endif /* NXPBUILD__PH_CRYPTOSYM */

/**
 * end of group phalMfPrimeNtag_SecureMessaging
 * @}
 */




/* MIFARE PRIME NTAG Memory and Configuration mamangement commands. ------------------------------------------------------------------ */
/**
 * \defgroup phalMfPrimeNtag_MemoryConfiguration Commands_MemoryConfiguration
 * \brief These Components implement the MIFARE PRIME NTAG Memory and Configuration Management related commands.
 * @{
 */

#ifdef NXPBUILD__PH_CRYPTOSYM
/** \name Options for SetConfiguration command. */
/** @{ */
#define PHAL_MFPRIMENTAG_SET_CONFIG_OPTION0									0x00U	/**< Option 0 Reserved. */
#define PHAL_MFPRIMENTAG_SET_CONFIG_OPTION1									0x01U	/**< Option 1 Reserved. */
#define PHAL_MFPRIMENTAG_SET_CONFIG_OPTION2									0x02U	/**< Option 2 Reserved. */
#define PHAL_MFPRIMENTAG_SET_CONFIG_OPTION3									0x03U	/**< Option 3 Reserved. */
#define PHAL_MFPRIMENTAG_SET_CONFIG_OPTION4									0x04U	/**< Option 4 for Secure Messaging Configuration. */
#define PHAL_MFPRIMENTAG_SET_CONFIG_OPTION5									0x05U	/**< Option 5 Reserved. */
#define PHAL_MFPRIMENTAG_SET_CONFIG_OPTION6									0x06U	/**< Option 6 Reserved. */
#define PHAL_MFPRIMENTAG_SET_CONFIG_OPTION7									0x07U	/**< Option 7 Reserved. */
#define PHAL_MFPRIMENTAG_SET_CONFIG_OPTION8									0x08U	/**< Option 8 Reserved. */
#define PHAL_MFPRIMENTAG_SET_CONFIG_OPTION9									0x09U	/**< Option 9 Reserved. */
#define	PHAL_MFPRIMENTAG_SET_CONFIG_OPTION10								0x0AU	/**< Option A Reserved. */
#define	PHAL_MFPRIMENTAG_SET_CONFIG_OPTION11								0x0BU	/**< Option B Reserved. */
/** @} */

/**
 * \brief Configures the card and pre personalizes the card with a key, defines if the UID or the random ID is sent back during communication setup and
 * configures the ATS string.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfprimeNtag_SetConfiguration(
		void * pDataParams,															/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,															/**< [In] Configuration Option. Define length and content of the Data parameter.
																					 *			\arg #PHAL_MFPRIMENTAG_SET_CONFIG_OPTION0
																					 *			\arg #PHAL_MFPRIMENTAG_SET_CONFIG_OPTION1
																					 *			\arg #PHAL_MFPRIMENTAG_SET_CONFIG_OPTION2
																					 *			\arg #PHAL_MFPRIMENTAG_SET_CONFIG_OPTION3
																					 *			\arg #PHAL_MFPRIMENTAG_SET_CONFIG_OPTION4
																					 *			\arg #PHAL_MFPRIMENTAG_SET_CONFIG_OPTION5
																					 *			\arg #PHAL_MFPRIMENTAG_SET_CONFIG_OPTION6
																					 *			\arg #PHAL_MFPRIMENTAG_SET_CONFIG_OPTION7
																					 *			\arg #PHAL_MFPRIMENTAG_SET_CONFIG_OPTION8
																					 *			\arg #PHAL_MFPRIMENTAG_SET_CONFIG_OPTION9
																					 *			\arg #PHAL_MFPRIMENTAG_SET_CONFIG_OPTION10
																					 *			\arg #PHAL_MFPRIMENTAG_SET_CONFIG_OPTION11
																					 */
		uint8_t * pData,															/**< [In] Data for the option specified.*/
		uint8_t bDataLen															/**< [In] Length of bytes available in Data buffer. */
	);
#endif /* NXPBUILD__PH_CRYPTOSYM */

/**
 * \brief Returns manufacturing related data of the PICC
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfprimeNtag_GetVersion(
		void * pDataParams,															/**< [In] Pointer to this layer's parameter structure. */
		uint8_t * pVerInfo															/**< [Out] The version information of the product. */
	);

/**
 * end of group phalMfPrimeNtag_MemoryConfiguration
 * @}
 */




/* MIFARE PRIME NTAG Key mamangement commands. -------------------------------------------------------------------------------------- */
/**
 * \defgroup phalMfPrimeNtag_KeyManagement Commands_KeyManagement
 * \brief These Components implement the MIFARE PRIME NTAG Key Management related commands.
 * @{
 */

#ifdef NXPBUILD__PH_CRYPTOSYM
/** \name Diversification options to be used with ChangeKey command. */
/** @{ */
#define PHAL_MFPRIMENTAG_CHGKEY_NO_DIVERSIFICATION            				0xFFFFU	/**< No diversification. */
#define PHAL_MFPRIMENTAG_CHGKEY_DIV_NEW_KEY            						0x0002U	/**< Bit 1. Indicating diversification of new key requred. */
#define PHAL_MFPRIMENTAG_CHGKEY_DIV_OLD_KEY            						0x0004U	/**< Bit 2 indicating old key was diversified. */
#define PHAL_MFPRIMENTAG_CHGKEY_DIV_NEW_KEY_ONERND     						0x0008U	/**< Bit 3 indicating new key diversification using one rnd. Default is two rounds. */
#define PHAL_MFPRIMENTAG_CHGKEY_DIV_OLD_KEY_ONERND     						0x0010U	/**< Bit 4 indicating old key diversification using one rnd. Default is two rounds. */
#define PHAL_MFPRIMENTAG_CHGKEY_DIV_METHOD_CMAC								0x0020U	/**< Bit 5 indicating key diversification method based on CMAC. Default is Encryption method */
 /** @} */

/**
 * \brief Changes any key on the PICC
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfprimeNtag_ChangeKey(
		void * pDataParams,															/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wOption,      														/**< [In] One of the below options.
																					 *			\arg #PHAL_MFPRIMENTAG_NO_DIVERSIFICATION
																					 *			\arg #PHAL_MFPRIMENTAG_CHGKEY_DIV_NEW_KEY | #PHAL_MFPRIMENTAG_CHGKEY_DIV_METHOD_CMAC
																					 *			\arg #PHAL_MFPRIMENTAG_CHGKEY_DIV_OLD_KEY | #PHAL_MFPRIMENTAG_CHGKEY_DIV_METHOD_CMAC
																					 *			\arg #PHAL_MFPRIMENTAG_CHGKEY_DIV_NEW_KEY | #PHAL_MFPRIMENTAG_CHGKEY_DIV_NEW_KEY_ONERND
																					 *			\arg #PHAL_MFPRIMENTAG_CHGKEY_DIV_OLD_KEY | #PHAL_MFPRIMENTAG_CHGKEY_DIV_OLD_KEY_ONERND
																					 *			\arg #PHAL_MFPRIMENTAG_CHGKEY_DIV_NEW_KEY | #PHAL_MFPRIMENTAG_CHGKEY_DIV_OLD_KEY
																					 *			\arg #PHAL_MFPRIMENTAG_CHGKEY_DIV_NEW_KEY | #PHAL_MFPRIMENTAG_CHGKEY_DIV_OLD_KEY |
																					 * 				 #PHAL_MFPRIMENTAG_CHGKEY_DIV_METHOD_CMAC
																					 *			\arg #PHAL_MFPRIMENTAG_CHGKEY_DIV_NEW_KEY | #PHAL_MFPRIMENTAG_CHGKEY_DIV_OLD_KEY |
																					 *				 #PHAL_MFPRIMENTAG_CHGKEY_DIV_NEW_KEY_ONERND | #PHAL_MFPRIMENTAG_CHGKEY_DIV_OLD_KEY_ONERND
																					 */
		uint16_t wOldKeyNo,															/**< [In] Old key number in keystore of software or SAM. */
		uint16_t wOldKeyVer,    													/**< [In] Old key number in keystore of software or SAM. */
		uint16_t wNewKeyNo,     													/**< [In] New key number in keystore of software or SAM. */
		uint16_t wNewKeyVer,    													/**< [In] New key number in keystore of software or SAM. */
		uint8_t bKeyNoCard,     													/**< [In] Key number on card. */
		uint8_t * pDivInput,    													/**< [In] Diversification input. Can be NULL. */
		uint8_t bDivLen         													/**< [In] Length of diversification input max 31B. */
	);
#endif /* NXPBUILD__PH_CRYPTOSYM */

/**
 * \brief Reads out the current key version of any key stored on the PICC
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfprimeNtag_GetKeyVersion(
		void * pDataParams,															/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bKeyNo,																/**< [In] Key number on card. */
		uint8_t bKeySetNo,															/**< [In] 1 byte Key set number. Optional as it is passed only when bit[6] of bKeyNo is set. */
		uint8_t * pKeyVersion,														/**< [Out] The version of the specified key. */
		uint8_t * bRxLen															/**< [Out] Length of bytes available in KeyVersion buffer. */
	);

/**
 * end of group phalMfPrimeNtag_KeyManagement
 * @}
 */




/* MIFARE PRIME NTAG File mamangement commands. -------------------------------------------------------------------------------------- */
/**
 * \defgroup phalMfPrimeNtag_FileManagement Commands_FileManagement
 * \brief These Components implement the MIFARE PRIME NTAG File Management related commands.
 * @{
 */

/**
 * \brief Get informtion on the properties of a specific file
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t  phalMfprimeNtag_GetFileSettings(
		void * pDataParams,															/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bFileNo,															/**< [In] The file number for which the setting to be retrieved. */
		uint8_t * pFSBuffer,														/**< [Out] The buffer containing the settings. The buffer should be 17 bytes. */
		uint8_t * bBufferLen														/**< [Out] The length of bytes available in FSBuffer buffer. */
	);

#ifdef NXPBUILD__PH_CRYPTOSYM
/**
 * \brief Returns manufacturing related data of the PICC
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t  phalMfprimeNtag_GetFileCountersSDM(
		void * pDataParams,															/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,															/**< [In] Indicates the mode of communication to be used while exchanging the data to PICC.
																					 *			\arg #PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN
																					 *			\arg #PHAL_MFPRIMENTAG_COMMUNICATION_ENC
																					 */
		uint8_t bFileNo,															/**< [In] File number for which the Counter information need to be received. */
		uint8_t * pFileCounters,													/**< [Out] The SDMReadCounter information returned by the PICC. */
		uint8_t * pRxLen															/**< [Out] Length of bytes available in FileCounters buffer. */
	);
#endif /* NXPBUILD__PH_CRYPTOSYM */

/** \name The File option and other flags for ChangeFileSettings command. */
/** @{ */
#define PHAL_MFPRIMENTAG_FILE_OPTION_PLAIN									0x00U	/**< Plain mode of communication. */
#define PHAL_MFPRIMENTAG_FILE_OPTION_MACD									0x01U	/**< MAC mode of communication. */
#define PHAL_MFPRIMENTAG_FILE_OPTION_ENC									0x03U	/**< Enciphered mode of communication. */
#define PHAL_MFPRIMENTAG_SDM_ENABLED										0x40u   /**< Secure Dynamic Messaging and Mirroring is enabled. */
#define PHAL_MFPRIMENTAG_VCUID_PRESENT										0x01U	/**< Only VCUID is considred for SDM MAC calculation */
#define PHAL_MFPRIMENTAG_RDCTR_PRESENT										0x02U	/**< Only RDCTR  is considred for SDM MAC calculation */
#define PHAL_MFPRIMENTAG_ENCODING_MODE_ASCII								0x01U	/**< Indicated the encoding mode as ASCII. */
/** @} */

/**
 * \brief Changes the access parameters of an existing file.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t  phalMfprimeNtag_ChangeFileSettings(
		void * pDataParams,															/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,															/**< [In] Indicates the mode of communication to be used while exchanging the data to PICC.
																					 *			\arg #PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN
																					 *			\arg #PHAL_MFPRIMENTAG_COMMUNICATION_MACD
																					 *			\arg #PHAL_MFPRIMENTAG_COMMUNICATION_ENC
																					 */
		uint8_t bFileNo,															/**< [In] File number for which the setting need to be updated. */
		uint8_t bCommSett,      													/**< [In] New communication settings for the file.
																					 *			\arg #PHAL_MFPRIMENTAG_FILE_OPTION_PLAIN
																					 *			\arg #PHAL_MFPRIMENTAG_FILE_OPTION_MACD
																					 * 			\arg #PHAL_MFPRIMENTAG_FILE_OPTION_ENC
																					 */
		uint8_t * pAccessRights,													/**< [In] The new access right to be applied for the file. Should be 2 byte. */
		uint8_t bNumAddARs,															/**< [In] Length of additional Access Rights. */
		uint8_t * pAddARs															/**< [In] The Additional Access Rights information. Should be 2 * NumAddARs. */
	);

/**
 * \brief Changes the access parameters of an existing file.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t  phalMfprimeNtag_ChangeFileSettingsSDM(
		void * pDataParams,															/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,															/**< [In] Indicates the mode of communication to be used while exchanging the data to PICC.
																					*			\arg #PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN
																					*			\arg #PHAL_MFPRIMENTAG_COMMUNICATION_MACD
																					*			\arg #PHAL_MFPRIMENTAG_COMMUNICATION_ENC
																					*/
		uint8_t bFileNo,															/**< [In] File number for which the setting need to be updated. */
		uint8_t bFileOption,      													/**< [In] New communication settings for the file.
																					*			\arg #PHAL_MFPRIMENTAG_FILE_OPTION_PLAIN
																					*			\arg #PHAL_MFPRIMENTAG_FILE_OPTION_MACD
																					*			\arg #PHAL_MFPRIMENTAG_FILE_OPTION_ENC
																					*		  \n
																					*		  Ored with \arg #PHAL_MFPRIMENTAG_SDM_ENABLED
																					*/
		uint8_t * pAccessRights,													/**< [In] The new access right to be applied for the file. Should be 2 byte. */
		uint8_t bSdmOptions,														/**< [In] One of the below values to be used. Can be ORed.
																					*			\arg #PHAL_MFPRIMENTAG_VCUID_PRESENT
																					*			\arg #PHAL_MFPRIMENTAG_RDCTR_PRESENT
																					* 			\arg #PHAL_MFPRIMENTAG_ENCODING_MODE_ASCII
																					*/
		uint8_t * pSdmAccessRights,													/**< [In] The SDM access rights to be applied. Should be 2 bytes. */
		uint8_t * pVCUIDOffset,														/**< [In] VCUID Offset information. Should be 3 bytes. */
		uint8_t * pSDMReadCtrOffset,												/**< [In] SDMReadCtrLimit value (LSB first). Should be 3 bytes. */
		uint8_t * pSDMMACInputOffset,												/**< [In] Offset in the file where the SDM MAC computation starts (LSB first). Should be 3 bytes. */
		uint8_t * pSDMMACOffset														/**< [In] SDMMAC mirror position (LSB first). Should be 3 bytes. */

	);

/**
 * end of group  phalMfPrimeNtag_FileManagement
 * @}
 */




/* MIFARE PRIME NTAG Data mamangement commands. -------------------------------------------------------------------------------------- */
/**
 * \defgroup phalMfPrimeNtag_DataManagement Commands_DataManagement
 * \brief These Components implement the MIFARE PRIME NTAG Data Management related commands.
 * @{
 */

/**
 * \brief Reads data from standard data files.
 *
 * \remarks
 *
 * Chaining upto the size of the HAL Rx buffer is handled within this function. If more data is to be read, the user has to call
 * this function again with bOption = #PH_EXCHANGE_RXCHAINING | [one of the communication options]
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING indicating more data to be read.
 * \retval Depending on implementation and underlying component.
 */
phStatus_t phalMfprimeNtag_ReadData(
		void * pDataParams,															/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,															/**< [In] Communication settings for the file.
																					 *			\arg #PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN
																					 *			\arg #PHAL_MFPRIMENTAG_COMMUNICATION_MACD
																					 *			\arg #PHAL_MFPRIMENTAG_COMMUNICATION_ENC
																					 *
																					 *			\arg #PH_EXCHANGE_RXCHAINING: To be Or'd with the above option
																					 *				flag if Chaining status is returned.
																					 */
		uint8_t bIns,																/**< [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		uint8_t bFileNo,															/**< [In] The file number from where the data to be read. */
		uint8_t * pOffset,															/**< [In] The offset from where the data should be read. Will be of 3 bytes with LSB first.
																					 *			\c If 0x10 need to be offset then it will be 10 00 00.
																					 */
		uint8_t * pLength,															/**< [In] The number of bytes to be read. Will be of 3 bytes with LSB first.
																					 *			\c If 0x10 bytes need to be read then it will be 10 00 00. \n
																					 *			\c If complete file need to be read then it will be 00 00 00.
																					 */
		uint8_t ** ppRxdata,														/**< [Out] The data retuned by the PICC. */
		uint16_t * pRxdataLen														/**< [Out] Length of bytes available in RxData buffer. */
	);

/**
 * \brief Writes data to standard data files.
 *
 * \remarks
 * Implements chaining to the card.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfprimeNtag_WriteData(
		void * pDataParams,															/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,															/**< [In] Communication settings for the file.
																					 *			\arg #PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN
																					 *			\arg #PHAL_MFPRIMENTAG_COMMUNICATION_MACD
																					 *			\arg #PHAL_MFPRIMENTAG_COMMUNICATION_ENC
																					 */
		uint8_t bIns,																/**< [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		uint8_t bFileNo,															/**< [In] The file number from where the data to be read. */
		uint8_t * pOffset,															/**< [In] The offset from where the data should be written. Will be of 3 bytes with LSB first.
																					 *			\c If 0x10 need to be offset then it will be 10 00 00.
																					 */
		uint8_t * pTxData,															/**< [In] The data to be written to the PICC. */
		uint8_t * pTxDataLen														/**< [In] The number of bytes to be written. Will be of 3 bytes with LSB first.
																					 *			\c If 0x10 bytes need to be written then it will be 10 00 00.
																					 */
	);

/**
 * end of group phalMfprimeNtag_DataManagement
 * @}
 */




/* MIFARE PRIME NTAG ISO7816-4 commands. --------------------------------------------------------------------------------------------- */
/**
 * \defgroup phalMfPrimeNtag_ISO7816 Commands_ISO7816
 * \brief These Components implement the MIFARE NTAG 42xDNA ISO/IEC 7816-4 Basic commands.
 * @{
 */

/** \name Options for ISOSelectFile command. */
/** @{ */
#define PHAL_MFPRIMENTAG_FCI_RETURNED										0x00U	/**< Option to indicate the return of FCI. */
#define PHAL_MFPRIMENTAG_FCI_NOT_RETURNED									0x0CU	/**< Option to indicate the no return of FCI. */
#define PHAL_MFPRIMENTAG_SELECTOR_0											0x00U	/**< Option to indicate Selection by 2 byte file Id. */
#define PHAL_MFPRIMENTAG_SELECTOR_1											0x01U	/**< Option to indicate Selection by child DF. */
#define PHAL_MFPRIMENTAG_SELECTOR_2											0x02U	/**< Option to indicate Select EF under current DF. Fid = EF id. */
#define PHAL_MFPRIMENTAG_SELECTOR_3											0x03U	/**< Option to indicate Select parent DF of the current DF. */
#define PHAL_MFPRIMENTAG_SELECTOR_4											0x04U	/**< Option to indicate Selection by DF Name. DFName and len is then valid. */
/** @} */

/**
 * \brief ISO Select. This command is implemented in compliance with ISO/IEC 7816-4.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfprimeNtag_IsoSelectFile(
		void * pDataParams,															/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOption,															/**< [In] Option for return / no return of FCI.
																					 *			\arg #PHAL_MFPRIMENTAG_FCI_RETURNED
																					 *			\arg #PHAL_MFPRIMENTAG_FCI_NOT_RETURNED
																					 */
		uint8_t bSelector,															/**< [In] The selector to be used.
																					 *			\arg #PHAL_MFPRIMENTAG_SELECTOR_0
																					 *			\arg #PHAL_MFPRIMENTAG_SELECTOR_1
																					 *			\arg #PHAL_MFPRIMENTAG_SELECTOR_2
																					 *			\arg #PHAL_MFPRIMENTAG_SELECTOR_3
																					 *			\arg #PHAL_MFPRIMENTAG_SELECTOR_4
																					 */
		uint8_t * pFid,																/**< [In] The ISO File number to be selected. */
		uint8_t * pDFname,															/**< [In] The ISO DFName to be selected. Valid only when bOption = 0x04. */
		uint8_t bDFnameLen,															/**< [In] Length of bytes available in DFname buffer. */
		uint8_t bExtendedLenApdu,													/**< [In] Flag for Extended Length APDU. 0x01 for Extended Length APDUs. 0x00 or
																					 *		  any other value for Short APDUs.
																					 */
		uint8_t ** ppFCI,															/**< [Out] The FCI information returned by the PICC. */
		uint16_t * pwFCILen															/**< [Out] Length of bytes available in FCI buffer. */
	);

/**
 * \brief ISO Read Binary. This command is implemented in compliance with ISO/IEC 7816-4.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING operation success with chaining.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfprimeNtag_IsoReadBinary(
		void * pDataParams,															/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wOption,															/**< [In] One of the below options.
																					 *			\arg #PH_EXCHANGE_DEFAULT   : To exchange the initial command to the PICC.
																					 *			\arg #PH_EXCHANGE_RXCHAINING: To exchange the chaining command to PICC in
																					 *										  case PICC returns AF as the status.
																					 */
		uint8_t bOffset,															/**< [In] The offset from where the data should be read. */
		uint8_t bSfid,																/**< [In] Short ISO File Id. Bit 7 should be 1 to indicate Sfid is supplied.
																					 *		  Else it is treated as MSB of 2Byte offset.
																					 */
		uint32_t dwBytesToRead,														/**< [In] Number of bytes to read. If 0, then entire file to be read. */
		uint8_t bExtendedLenApdu,													/**< [In] Flag for Extended Length APDU. 0x01 for Extended Length APDUs. 0x00 or
																					 *		  any other value for Short APDUs.
																					 */
		uint8_t ** ppRxBuffer,														/**< [Out] The data retuned by the PICC. */
		uint32_t * pBytesRead														/**< [Out] Length of bytes available in RxData buffer. */
	);

/**
 * \brief Iso Update Binary. This command is implemented in compliance with ISO/IEC 7816-4.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfprimeNtag_IsoUpdateBinary(
		void * pDataParams,															/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bOffset,															/**< [In] The offset from where the data should be updated. */
		uint8_t bSfid,																/**< [In] Short ISO File Id. Bit 7 should be 1 to indicate Sfid is supplied.
																					 *		  Else it is treated as MSB of 2Byte offset.
																					 */
		uint8_t bExtendedLenApdu,													/**< [In] Flag for Extended Length APDU. 0x01 for Extended Length APDUs. 0x00 or
																					 *		  any other value for Short APDUs.
																					 */
		uint8_t * pData,															/**< [In] Data to be updated. */
		uint32_t dwDataLen															/**< [In] Length of bytes available in Data buffer. */
	);

/**
 * end of group phalMfPrimeNtag_ISO7816
 * @}
 */




/* MIFARE PRIME NTAG Originality Check functions. ------------------------------------------------------------------------------------ */
/**
 * \defgroup phalMfPrimeNtag_OriginalityCheck Commands_OriginalityCheck
 * \brief These Components implement the MIFARE PRIME NTAG Originality Check functions.
 * @{
 */

/**
 * \brief Performs the originality check to verify the genuineness of chip
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfprimeNtag_ReadSign(
		void * pDataParams,															/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bAddr,																/**< [In] Value is always 00. Present for forward compatibility reasons */
		uint8_t ** pSignature														/**< [Out] The plain 56 bytes originality signature of the PICC. */
	);

/**
 * end of group phalMfPrimeNtag_OriginalityCheck
 * @}
 */




/* MIFARE PRIME NTAG Miscellaneous functions. ---------------------------------------------------------------------------------------- */
/**
 * \defgroup phalMfPrimeNtag_Miscellaneous Commands_Miscellaneous
 * \brief These Components implement the MIFARE PRIME NTAG Miscellaneous functions. These are not part of actual NTAG data sheet
 * rather its for internal purpose.
 * @{
 */

/** \name The configuration to be used for SetConfig / GetConfig. */
/** @{ */
#define PHAL_MFPRIMENTAG_ADDITIONAL_INFO									0x00A1U	/**< Option for getconfig to get additional info of a generic error. */
#define PHAL_MFPRIMENTAG_SHORT_LENGTH_APDU									0x00A3U	/**< Option to get / set Short Length APDU wrapping in ISO 7816-4 APDUs. */
/** @} */

/**
 * \brief Perform a GetConfig command.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfprimeNtag_GetConfig(
		void * pDataParams,															/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wConfig,															/**< [In] Configuration to read. Will be one of the below values.
																					 *			\arg #PHAL_MFPRIMENTAG_ADDITIONAL_INFO
																					 *			\arg #PHAL_MFPRIMENTAG_SHORT_LENGTH_APDU
																					 */
		uint16_t * pValue															/**< [Out] The value for the mentioned configuration. */
	);

/**
 * \brief Perform a SetConfig command.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfprimeNtag_SetConfig(
		void * pDataParams,															/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wConfig,															/**< [In] Configuration to set. Will be one of the below values.
																					 *			\arg #PHAL_MFPRIMENTAG_ADDITIONAL_INFO
																					 *			\arg #PHAL_MFPRIMENTAG_SHORT_LENGTH_APDU
																					 */
		uint16_t wValue																/**< [In] The value for the mentioned configuration. */
	);

/**
 * \brief Reset the authentication
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfprimeNtag_ResetAuthentication(
		void * pDataParams															/**< [In] Pointer to this layer's parameter structure. */
	);

#ifdef NXPBUILD__PH_CRYPTOSYM
/**
 * \brief Calculate MAC for SDM information.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfprimeNtag_CalculateMACSDM(
		void * pDataParams,														/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wOption,														/**< [In] Diversification option. 0xFFFF is for No Diversification. */
		uint8_t bSdmOption,														/**< [In] SDM Option to indicate which parameters to be considered. Can be ORed.
																				*			\arg #PHAL_MFPRIMENTAG_VCUID_PRESENT
																				*			\arg #PHAL_MFPRIMENTAG_RDCTR_PRESENT
																				*/
		uint16_t wSDMFileReadKeyNo,												/**< [In] Key number in SW key store key. */
		uint16_t wSDMFileReadKeyVer,											/**< [In] Key version in Sw key store. */
		uint16_t wRamKeyNo,														/**< [In] Key number of Destination Key where the computed session TMAC key will be stored.
																				 *		  To be used for SAM AV3 only.
																				 */
		uint16_t wRamKeyVer,													/**< [In] Key version of Destination Key where the computed session TMAC key will be stored.
																				 *		  To be used for SAM AV3 only.
																				 */
		uint8_t * pDivInput,													/**< [In] Diversification input to diversify the SDMFileReadKey. */
		uint8_t bDivInputLen,													/**< [In] Length of byte available in DivInput buffer. */
		uint8_t * pUid,         												/**< [In] UID of the card. */
		uint8_t bUidLen,														/**< [In] Length of bytes available in UID buffer. */
		uint8_t * pSDMReadCtr,	    											/**< [In] SDM Read Counter Input. */
		uint8_t * pInData,														/**< [In] Data read out of PICC from SDMMacInputoffset for ReadLength bytes */
		uint16_t wInDataLen,													/**< [In] Length of bytes available in InData buffer. */
		uint8_t * pRespMac														/**< [Out] The computed MAC information. */
	);
#endif /* NXPBUILD__PH_CRYPTOSYM */

/**
 * end of group phalMfprimeNtag_Miscellaneous
 * @}
 */

/**
 * end of group phalMfPrimeNtag
 * @}
 */

#endif /* NXPBUILD__PHAL_MFPRIMENTAG */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHALMFPRIMENTAG_H */
