/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/*
 * Dummy Android Component specific PAL-Component of Reader Library Framework.
 * $RCSfile $
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <ph_NxpBuild.h>

#ifdef NXPBUILD__PHPAL_I14443P4_ANDROID

#include "phpalI14443p4_Android.h"
#include "phpalI14443p4_Android_Int.h"

#ifndef _WIN32
#include <stdio.h>              /* PRQA S 5124 */
#else
#pragma warning(push)           /* PRQA S 3116 */
#pragma warning(disable:4001)   /* PRQA S 3116 */
#include <stdio.h>              /* PRQA S 5124 */
#pragma warning(pop)            /* PRQA S 3116 */
#endif

phStatus_t phpalI14443p4_Android_OpenConnection(
    phpalI14443p4_Android_DataParams_t * pDataParams,
    uint8_t bTechnologie
    )
{
    phStatus_t PH_MEMLOC_REM status;

    if (pDataParams->bIsOpen == PH_ON)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_PAL_ISO14443P4);
    }

    /* Loop until we get all tag notifications in the buffer */
    while ((status = phpalI14443p4_Android_GetNewTagMessage(pDataParams, 1)) == PH_ERR_SUCCESS);
    /* Last valid tag notification is now in abAvailableTechnologies */

    /* Close old connection that may be open (previous run) */
    phpalI14443p4_Android_CloseConnection(pDataParams, bTechnologie);

    status = phpalI14443p4_Android_Transceive(
        pDataParams,
        PHPAL_I14443P4_ANDROID_COMMAND_CONNECT,
        bTechnologie,
        NULL, /* No Data */
        0,
        NULL, /* No Response Expected */
        NULL,
        NULL); /* Timing not used */

    /* if open is ok store actual status */
    if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        pDataParams->bIsOpen = PH_ON;
        pDataParams->bUsedTechnologies = bTechnologie;
    }
    return status;
}

phStatus_t phpalI14443p4_Android_GetNewTagMessage(
    phpalI14443p4_Android_DataParams_t * pDataParams,
    uint16_t wTimeout
    )
{
    phStatus_t PH_MEMLOC_REM status;
    phStatus_t PH_MEMLOC_REM statusExchange;
    uint16_t wResponseLength = 0;
    uint16_t wIndex = 0;

    union
    {
        uint32_t i;
        uint8_t c[4];
    } uint32Union;

    /* Set timeout while waiting for tag notification */
    PH_CHECK_SUCCESS_FCT(status, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wTimeout));

    /* Wait for getting a tag found notification */
    statusExchange = phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        NULL,
        0,
        5,
        pDataParams->pExchangeBuffer,
        &wResponseLength);

    /* Restore timeout */
    PH_CHECK_SUCCESS_FCT(status, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 5000));

    PH_CHECK_SUCCESS(statusExchange);

    /* me must have received 5 bytes */
    if (wResponseLength != 5)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_ISO14443P4);
    }

    /* Get the length of data and receive rest */
    uint32Union.c[3] = pDataParams->pExchangeBuffer[1];
    uint32Union.c[2] = pDataParams->pExchangeBuffer[2];
    uint32Union.c[1] = pDataParams->pExchangeBuffer[3];
    uint32Union.c[0] = pDataParams->pExchangeBuffer[4];

    if (uint32Union.i > (uint32_t)(pDataParams->wExchangeBufSize - 5) || uint32Union.i > (uint32_t)(0xFFFF - 5))
    {
        phpalI14443p4_Android_ClearReceiveBuffer(pDataParams, uint32Union.i);
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_PAL_ISO14443P4);
    }

    /* Perform actual exchange (original timeout data should be still available) */
    statusExchange = phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        NULL,
        0,
        (uint16_t)uint32Union.i,
        pDataParams->pExchangeBuffer + 5,
        &wResponseLength);
    /* Since we provide the buffer, it's a PAL buffer overflow */
    if ((statusExchange & PH_ERR_MASK) == PH_ERR_BUFFER_OVERFLOW)
    {
        phpalI14443p4_Android_ClearReceiveBuffer(pDataParams, uint32Union.i);
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_PAL_ISO14443P4);
    }
    PH_CHECK_SUCCESS(statusExchange);

    /* Calculate length including header */
    wResponseLength += 5;

    /* Response must have at least Header, Length, Technologie, Time and on available technologie but not more than possible technoligies */
    if (wResponseLength < 15 || wResponseLength > 14 + PHPAL_I14443P4_ANDROID_TECHNOLOGIE_CNT)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_ISO14443P4);
    }

    /* check if length in header match with received data */
    if (uint32Union.i != (uint32_t)(wResponseLength - 5))
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_ISO14443P4);
    }

    /* Check response content */
    if (pDataParams->pExchangeBuffer[0] != PHPAL_I14443P4_ANDROID_COMMAND_TAG_PRESENT ||
        pDataParams->pExchangeBuffer[5] != PHPAL_I14443P4_ANDROID_TECHNOLOGIE_NONE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_ISO14443P4);
    }

    /* Check if all technologies are valid */
    for (wIndex = 14; wIndex < wResponseLength; wIndex++)
    {
        if (pDataParams->pExchangeBuffer[wIndex] == PHPAL_I14443P4_ANDROID_TECHNOLOGIE_NONE ||
            pDataParams->pExchangeBuffer[wIndex] >= PHPAL_I14443P4_ANDROID_TECHNOLOGIE_CNT)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_ISO14443P4);
        }
    }

    /* Copy available technologies into technologie buffer */
    memset(&pDataParams->abAvailableTechnologies, PHPAL_I14443P4_ANDROID_TECHNOLOGIE_NONE, PHPAL_I14443P4_ANDROID_TECHNOLOGIE_CNT);
    for (wIndex = 14; wIndex < wResponseLength; wIndex++)
    {
        pDataParams->abAvailableTechnologies[wIndex - 14] = pDataParams->pExchangeBuffer[wIndex];
    }
    return status;
}

phStatus_t phpalI14443p4_Android_CloseConnection(
    phpalI14443p4_Android_DataParams_t * pDataParams,
    uint8_t bTechnologie
    )
{
    phStatus_t PH_MEMLOC_REM status = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);

    if (bTechnologie != PHPAL_I14443P4_ANDROID_TECHNOLOGIE_NONE)
    {
        status = phpalI14443p4_Android_Transceive(
            pDataParams,
            PHPAL_I14443P4_ANDROID_COMMAND_CLOSE,
            bTechnologie,
            NULL, /* No Data */
            0,
            NULL, /* No Response Expected */
            NULL,
            NULL); /* Timing not used */

        pDataParams->bIsOpen = PH_OFF;
        pDataParams->bUsedTechnologies = PHPAL_I14443P4_ANDROID_TECHNOLOGIE_NONE;
    }
    return status;
}

phStatus_t phpalI14443p4_Android_Transceive(
    phpalI14443p4_Android_DataParams_t * pDataParams,
    uint8_t bCommandHeader,
    uint8_t bTechnologie,
    uint8_t *pData,
    uint16_t wDataLength,
    uint8_t **ppResponse,
    uint16_t *pwResponseLength,
    uint64_t *pqwTimeUsed
    )
{
    phStatus_t PH_MEMLOC_REM status;
    uint16_t wResponseLength = 0;
    uint8_t bResponseHeader = 0;
    uint8_t bResponseTechnologie = 0;
    uint16_t wAlreadyReceived = 0;

    union
    {
        uint32_t i;
        uint8_t c[4];
    } uint32Union;
    union
    {
        uint64_t i;
        uint8_t c[8];
    } uint64Union;

    /* Init return parameter */
    if (pqwTimeUsed != NULL)
    {
        *pqwTimeUsed = 0;
    }
    if (ppResponse != NULL)
    {
        *ppResponse = NULL;
    }
    if (pwResponseLength != NULL)
    {
        *pwResponseLength = 0;
    }

    /* Check parameter */
    if (pData == NULL && wDataLength != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
    }

    /* If the provided buffer overlabs with the used exchange buffer and it is not on the correct position -> Error */
    if ((pData >= pDataParams->pExchangeBuffer && pData < pDataParams->pExchangeBuffer + 6 + wDataLength) && pData != pDataParams->pExchangeBuffer + 6)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
    }

    /* Check if a new tag message is in the pipe */
    status = phpalI14443p4_Android_GetNewTagMessage(pDataParams, 1);
    if ((status & PH_ERR_MASK) != PH_ERR_IO_TIMEOUT)
    {
        /* If available close the connection and return error */
        pDataParams->bIsOpen = PH_OFF;
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_PAL_ISO14443P4);
    }

    /* Build command */
    pDataParams->pExchangeBuffer[0] = bCommandHeader;
    uint32Union.i = 1 + wDataLength;
    pDataParams->pExchangeBuffer[1] = uint32Union.c[3];
    pDataParams->pExchangeBuffer[2] = uint32Union.c[2];
    pDataParams->pExchangeBuffer[3] = uint32Union.c[1];
    pDataParams->pExchangeBuffer[4] = uint32Union.c[0];
    pDataParams->pExchangeBuffer[5] = bTechnologie;

    /* If the command buffer is not in the correct position copy the data */
    if (pData != NULL && pData != pDataParams->pExchangeBuffer + 6)
    {
        memcpy(pDataParams->pExchangeBuffer + 6, pData, wDataLength);
    }

    /* Perform actual exchange */
    PH_CHECK_SUCCESS_FCT(status, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        pDataParams->pExchangeBuffer,
        wDataLength + 6,
        5,
        pDataParams->pExchangeBuffer,
        &wResponseLength));

    /* me must have received 5 bytes */
    if (wResponseLength != 5)
    {
        pDataParams->bIsOpen = PH_OFF;
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_ISO14443P4);
    }

    /* Get the length of data and receive rest */
    uint32Union.c[3] = pDataParams->pExchangeBuffer[1];
    uint32Union.c[2] = pDataParams->pExchangeBuffer[2];
    uint32Union.c[1] = pDataParams->pExchangeBuffer[3];
    uint32Union.c[0] = pDataParams->pExchangeBuffer[4];

    if (uint32Union.i > (uint32_t)(pDataParams->wExchangeBufSize - 5) || uint32Union.i > (uint32_t)(0xFFFF - 5))
    {
        pDataParams->bIsOpen = PH_OFF;
        phpalI14443p4_Android_ClearReceiveBuffer(pDataParams, uint32Union.i);
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_PAL_ISO14443P4);
    }

    wAlreadyReceived = 0;
    do
    {
        /* Perform actual exchange */
        status = phbalReg_Exchange(
            pDataParams->pBalDataParams,
            PH_EXCHANGE_DEFAULT,
            NULL,
            0,
            (uint16_t)uint32Union.i - wAlreadyReceived,
            pDataParams->pExchangeBuffer + 5 + wAlreadyReceived,
            &wResponseLength);
        /* Since we provide the buffer, it's a PAL buffer overflow */
        if ((status & PH_ERR_MASK) == PH_ERR_BUFFER_OVERFLOW)
        {
            pDataParams->bIsOpen = PH_OFF;
            phpalI14443p4_Android_ClearReceiveBuffer(pDataParams, uint32Union.i);
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_PAL_ISO14443P4);
        }
        PH_CHECK_SUCCESS(status);
        wAlreadyReceived += wResponseLength;
    } while(wAlreadyReceived < uint32Union.i);

    /* Calculate length including header */
    wResponseLength = 5 + wAlreadyReceived;

    /* Response must have at least Header, Length, Technologie, Time */
    if (wResponseLength < 14)
    {
        pDataParams->bIsOpen = PH_OFF;
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_ISO14443P4);
    }

    /* check if length in header match with received data */
    if (uint32Union.i != (uint32_t)(wResponseLength - 5))
    {
        pDataParams->bIsOpen = PH_OFF;
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_ISO14443P4);
    }

    /* Response header must be the command header or an error */
    bResponseHeader = pDataParams->pExchangeBuffer[0];
    if (bResponseHeader != bCommandHeader && bResponseHeader != PHPAL_I14443P4_ANDROID_COMMAND_ERROR)
    {
        pDataParams->bIsOpen = PH_OFF;
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_ISO14443P4);
    }

    /* If we got an error we did not check technologie */
    if (bResponseHeader != PHPAL_I14443P4_ANDROID_COMMAND_ERROR)
    {
        /* Check technologie */
        bResponseTechnologie = pDataParams->pExchangeBuffer[5];
        if (bResponseTechnologie != bTechnologie)
        {
            pDataParams->bIsOpen = PH_OFF;
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_ISO14443P4);
        }
    }

    /* Get timimg if required */
    if (pqwTimeUsed != NULL)
    {
        uint64Union.c[7] = pDataParams->pExchangeBuffer[6];
        uint64Union.c[6] = pDataParams->pExchangeBuffer[7];
        uint64Union.c[5] = pDataParams->pExchangeBuffer[8];
        uint64Union.c[4] = pDataParams->pExchangeBuffer[9];
        uint64Union.c[3] = pDataParams->pExchangeBuffer[10];
        uint64Union.c[2] = pDataParams->pExchangeBuffer[11];
        uint64Union.c[1] = pDataParams->pExchangeBuffer[12];
        uint64Union.c[0] = pDataParams->pExchangeBuffer[13];
        *pqwTimeUsed = uint64Union.i;
    }

    /* Return RxBuffer pointer */
    if (ppResponse != NULL)
    {
        *ppResponse = pDataParams->pExchangeBuffer + 14; /* Remove the header, length. technologie and time */
    }

    /* Return RxBuffer length */
    if (pwResponseLength != NULL)
    {
        *pwResponseLength = wResponseLength - 14;
    }

    if (bResponseHeader != bCommandHeader)
    {
        pDataParams->bIsOpen = PH_OFF;
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_PAL_ISO14443P4);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Android_ClearReceiveBuffer(
    phpalI14443p4_Android_DataParams_t * pDataParams,
    uint32_t dwLength
    )
{
    phStatus_t PH_MEMLOC_REM status;
    uint16_t wResponseLength = 0;
    uint16_t wOldTimeout = 0;
    uint8_t aBuffer[100];
    uint32_t dwDataCleared = 0;
    uint16_t wDataToRead = 0;

    /* Set timeout while waiting for tag notification */
    PH_CHECK_SUCCESS_FCT(status, phbalReg_GetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wOldTimeout));

    /* Set timeout to 1 (zero is infinity) */
    PH_CHECK_SUCCESS_FCT(status, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 1));

    for (dwDataCleared = 0; dwDataCleared < dwLength; dwDataCleared += sizeof(aBuffer))
    {
        if (dwDataCleared + sizeof(aBuffer) > dwLength)
        {
            wDataToRead = (uint16_t)(dwLength - dwDataCleared);
        }
        else
        {
            wDataToRead = (uint16_t)sizeof(aBuffer);
        }

        /* Wait for getting a tag found notification */
        status = phbalReg_Exchange(
            pDataParams->pBalDataParams,
            PH_EXCHANGE_DEFAULT,
            NULL,
            0,
            wDataToRead,
            aBuffer,
            &wResponseLength);
    }

    /* Restore timeout */
    return phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wOldTimeout);
}
#endif /* NXPBUILD__PHPAL_I14443P4_ANDROID */
