/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software ISO14443-4 Component of Basic Function Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phpalI14443p4.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHPAL_I14443P4_ANDROID

#include "phpalI14443p4_Android.h"
#include "phpalI14443p4_Android_Int.h"

phStatus_t phpalI14443p4_Android_Init(
                                    phpalI14443p4_Android_DataParams_t * pDataParams,
                                    uint16_t wSizeOfDataParams,
                                    phbalReg_SocketWin_DataParams_t * pBalDataParams,
                                    uint8_t * pExchangeBuffer,
                                    uint16_t wExchangeBufSize
                                    )
{
    /* Check parameter */
    if (sizeof(phpalI14443p4_Android_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_ISO14443P4);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalDataParams);
    PH_ASSERT_NULL (pExchangeBuffer);
    PH_ASSERT_NULL (wExchangeBufSize);

    /* Init private data */
    pDataParams->wId            = PH_COMP_PAL_ISO14443P4 | PHPAL_I14443P4_ANDROID_ID;
    pDataParams->pBalDataParams = pBalDataParams;
    pDataParams->bIsOpen = PH_OFF;
    memset(&pDataParams->abAvailableTechnologies, PHPAL_I14443P4_ANDROID_TECHNOLOGIE_NONE, PHPAL_I14443P4_ANDROID_TECHNOLOGIE_CNT);
    pDataParams->bUsedTechnologies = PHPAL_I14443P4_ANDROID_TECHNOLOGIE_NONE;
    pDataParams->pExchangeBuffer = pExchangeBuffer;
    pDataParams->wExchangeBufSize = wExchangeBufSize;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Android_SetProtocol(
    phpalI14443p4_Android_DataParams_t * pDataParams,
    uint8_t   bCidEnable,
    uint8_t   bCid,
    uint8_t   bNadEnable,
    uint8_t   bNad,
    uint8_t   bFwi,
    uint8_t   bFsdi,
    uint8_t   bFsci
    )
{
    PH_ASSERT_NULL (pDataParams);

    /* satisfy compiler */
    if (bCidEnable || bCid || bNadEnable || bNad || bFwi || bFsdi || bFsci);

    return phpalI14443p4_Android_OpenConnection(pDataParams, PHPAL_I14443P4_ANDROID_TECHNOLOGIE_ISO_DEP);
}

phStatus_t phpalI14443p4_Android_ResetProtocol(
    phpalI14443p4_Android_DataParams_t * pDataParams
    )
{
    PH_ASSERT_NULL (pDataParams);
    /* satisfy compiler */
    if (pDataParams);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Android_Deselect(
                                        phpalI14443p4_Android_DataParams_t * pDataParams
                                        )
{
    return phpalI14443p4_Android_CloseConnection(pDataParams, pDataParams->bUsedTechnologies);
}

phStatus_t phpalI14443p4_Android_PresCheck(
    phpalI14443p4_Android_DataParams_t * pDataParams
    )
{
    PH_ASSERT_NULL (pDataParams);
    /* satisfy compiler */
    if (pDataParams);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Android_Exchange(
                                        phpalI14443p4_Android_DataParams_t * pDataParams,
                                        uint16_t wOption,
                                        uint8_t * pTxBuffer,
                                        uint16_t wTxLength,
                                        uint8_t ** ppRxBuffer,
                                        uint16_t * pRxLength
                                        )
{
    phStatus_t PH_MEMLOC_REM status;
    uint64_t qwTimeConsumed = 0;

    PH_ASSERT_NULL (ppRxBuffer);
    PH_ASSERT_NULL (pRxLength);

    if (pDataParams->bIsOpen != PH_ON)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_PAL_ISO14443P4);
    }

    if (wOption != PH_EXCHANGE_DEFAULT)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
    }

    status = phpalI14443p4_Android_Transceive(
        pDataParams,
        PHPAL_I14443P4_ANDROID_COMMAND_TRANSCEIVE,
        PHPAL_I14443P4_ANDROID_TECHNOLOGIE_ISO_DEP,
        pTxBuffer, /* Data */
        wTxLength,
        ppRxBuffer, /* Response */
        pRxLength,
        &qwTimeConsumed); /* Timing */

    if (status != PH_ERR_SUCCESS && *pRxLength != 0 && *ppRxBuffer != NULL)
    {
        if (memcmp(*ppRxBuffer, "java.io.IOException: Transceive failed", *pRxLength) == 0 ||
            memcmp(*ppRxBuffer, "android.nfc.TagLostException: Tag was lost.", *pRxLength) == 0 )
        {
            memset(&pDataParams->abAvailableTechnologies, PHPAL_I14443P4_ANDROID_TECHNOLOGIE_NONE, PHPAL_I14443P4_ANDROID_TECHNOLOGIE_CNT);
            return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_PAL_ISO14443P4);
        }
    }
    return status;
}

phStatus_t phpalI14443p4_Android_SetConfig(
    phpalI14443p4_Android_DataParams_t * pDataParams,
    uint16_t wConfig,
    uint16_t wValue
    )
{
    PH_ASSERT_NULL (pDataParams);

    /* satisfy compiler */
    if (pDataParams || wConfig || wValue);

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Android_GetConfig(
    phpalI14443p4_Android_DataParams_t * pDataParams,
    uint16_t wConfig,
    uint16_t * pValue
    )
{
    phStatus_t PH_MEMLOC_REM status;
    uint8_t *pRxBuffer;
    uint16_t wRxLength = 0;

    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pValue);

    switch (wConfig)
    {
    case PHPAL_I14443P4_ANDROID_CONFIG_SYS_INFO_SIZE:
        {
            PH_CHECK_SUCCESS_FCT(status, phpalI14443p4_Android_Transceive(
                pDataParams,
                PHPAL_I14443P4_ANDROID_COMMAND_GET_SYS_INFO,
                PHPAL_I14443P4_ANDROID_TECHNOLOGIE_NONE,
                NULL, /* Data */
                0,
                &pRxBuffer, /* Response */
                &wRxLength,
                NULL)); /* Timing */

            *pValue = wRxLength;
            break;
        }
    default:
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

#endif /* NXPBUILD__PHPAL_I14443P4_ANDROID */
