/*
 * Copyright 2015 - 2020, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * PCSC generic HAL interface.
 *
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PH_KEYSTORE
#include <phKeyStore.h>
#endif /* NXPBUILD__PH_KEYSTORE */

#ifdef NXPBUILD__PHHAL_HW_PCSC

#include <phhalHw_Pcsc_Cmd.h>
#include "phhalHw_Pcsc.h"
#include "phhalHw_Pcsc_Int.h"

phStatus_t phhalHw_Pcsc_Init(
                             phhalHw_Pcsc_DataParams_t * pDataParams,
                             uint16_t wSizeOfDataParams,
                             void * pBalDataParams,
                             void * pKeyStoreDataParams,
                             uint8_t * pTxBuffer,
                             uint16_t wTxBufSize,
                             uint8_t * pRxBuffer,
                             uint16_t wRxBufSize
                             )
{
    if (sizeof(phhalHw_Pcsc_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_HAL);
    }

    /* The buffers cannot be zero */
    if (wTxBufSize == 0U || wRxBufSize == 0U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Check all the pointers */
    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pBalDataParams);
    PH_ASSERT_NULL(pTxBuffer);
    PH_ASSERT_NULL(pRxBuffer);

    /* CHECK: Modify Tx-/Rx- Buffer pointers/lengths? */

    /* Initialize private data */
    pDataParams->wId                    = PH_COMP_HAL | PHHAL_HW_PCSC_ID;
    pDataParams->pBalDataParams         = pBalDataParams;
    pDataParams->pKeyStoreDataParams    = pKeyStoreDataParams;
    pDataParams->pTxBuffer              = pTxBuffer;
    pDataParams->wTxBufSize             = wTxBufSize;
    pDataParams->wTxBufLen              = 0U;
    pDataParams->pRxBuffer              = pRxBuffer;
    pDataParams->wRxBufSize             = wRxBufSize;
    pDataParams->wRxBufLen              = 0U;
    pDataParams->wRxBufStartPos         = 0U;
    pDataParams->wTxBufStartPos         = 0U;
    pDataParams->bTxLastBits            = 0U;
    pDataParams->dwFdtPc                = 0U;
    pDataParams->bCardType              = PHHAL_HW_CARDTYPE_ISO14443A;
    pDataParams->wErrorCode             = 0U;
    pDataParams->bRxLastBits            = 0U;
    pDataParams->bCollPos               = 0U;
    pDataParams->bTransparentSession    = PH_OFF;
    pDataParams->dwTimeoutUs            = 0U;
    pDataParams->bSak                   = 0U;
    pDataParams->wFieldOffTime          = PHHAL_HW_FIELD_OFF_DEFAULT;
    pDataParams->wFieldRecoveryTime     = PHHAL_HW_FIELD_RECOVERY_DEFAULT;
    pDataParams->wTxRxFlag              = 0x02U;
    pDataParams->bExchangeType          = PHHAL_HW_PCSC_NORMAL;
    pDataParams->wAdditionalInfo        = 0U;
    pDataParams->bKeyLocation           = PHHAL_HW_PCSC_KEY_LOC_VOLATILE;
    pDataParams->bTxDataRate            = 0U;
    pDataParams->bRxDataRate            = 0U;

    /* CHECK: Start transparent session here? */

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_FieldOn (
	phhalHw_Pcsc_DataParams_t * pDataParams
)
{
	phStatus_t PH_MEMLOC_REM statusTmp;

	if ( pDataParams->bTransparentSession == PH_ON )
	{
		/* RF on command */
		PH_CHECK_SUCCESS_FCT ( statusTmp, phhalHw_Pcsc_Cmd_RfOn (
			pDataParams ) );
	}
	else
	{
		/* No Open Port -> this is done when activate is called in the protocol layer */
		/* Open Port to start Connection
		status = phbalReg_OpenPort(pDataParams->pBalDataParams);*/
		phTools_Sleep ( pDataParams->wFieldRecoveryTime );
	}

	return PH_ADD_COMPCODE ( PH_ERR_SUCCESS, PH_COMP_HAL );
}

phStatus_t phhalHw_Pcsc_FieldOff(
                                 phhalHw_Pcsc_DataParams_t * pDataParams
                                 )
{
    phStatus_t PH_MEMLOC_REM statusTmp;

    if (pDataParams->bTransparentSession == PH_ON)
    {
        /* RF off command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_RfOff(
            pDataParams));
    }
    else
    {
        /* Close Port to exit Connection */
        phbalReg_ClosePort(pDataParams->pBalDataParams);
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Pcsc_FieldReset(
                                   phhalHw_Pcsc_DataParams_t * pDataParams
                                   )
{
    phStatus_t PH_MEMLOC_REM statusTmp;

    if (pDataParams->bTransparentSession == PH_ON)
    {
        /* RF reset (RF on + Timer + RF off) */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_RfReset(
            pDataParams));
    }
    else
    {
        /* Close Port to exit Connection */
        phbalReg_ClosePort(pDataParams->pBalDataParams);
        phTools_Sleep(pDataParams->wFieldOffTime);
    }

    /* Wait for field recovery time */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Wait(
        pDataParams,
        PHHAL_HW_TIME_MILLISECONDS,
        (pDataParams->wFieldRecoveryTime)));

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_ApplyProtocolSettings(
                                              phhalHw_Pcsc_DataParams_t * pDataParams,
                                              uint8_t bCardType
                                              )
{
    phStatus_t PH_MEMLOC_REM statusTmp;

    /* Store new card type */
    if (bCardType != PHHAL_HW_CARDTYPE_CURRENT)
    {
        pDataParams->bCardType = bCardType;
    }

    if (pDataParams->bTransparentSession != PH_ON)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    /* Switch to specified protocol */
    switch (pDataParams->bCardType)
    {
        /* ISO 14443A */
    case PHHAL_HW_CARDTYPE_ISO14443A:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_SwitchProtocol(
            pDataParams,
            PHHAL_HW_PCSC_ISO14443A,
            PHHAL_HW_PCSC_LAYER_2));
        break;

        /* ISO 15693 */
    case PHHAL_HW_CARDTYPE_ISO15693:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_SwitchProtocol(
            pDataParams,
            PHHAL_HW_PCSC_ISO15693,
            PHHAL_HW_PCSC_LAYER_2));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        break;
    }

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_SetConfig(
                                  phhalHw_Pcsc_DataParams_t * pDataParams,
                                  uint16_t wConfig,
                                  uint16_t wValue
                                  )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t *   PH_MEMLOC_REM pBuffer;
    uint16_t    PH_MEMLOC_REM wBufferSize;
    uint16_t    PH_MEMLOC_REM wBufferLen = 0;
    uint16_t    PH_MEMLOC_REM wDataRate = 0U;

    switch(wConfig)
    {
    case PHHAL_HW_CONFIG_PARITY:
        if (wValue == PH_ON)
        {
            pDataParams->wTxRxFlag &= (uint16_t)~(uint16_t)(PHHAL_HW_SET_PARITY);
        }
        else if (wValue == PH_OFF)
        {
            pDataParams->wTxRxFlag |= PHHAL_HW_SET_PARITY;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* Send the configuration to reader */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_TxRxFlag(
            pDataParams,
            pDataParams->wTxRxFlag));
        break;

    case PHHAL_HW_CONFIG_TXCRC:
        if (wValue == PH_ON)
        {
            pDataParams->wTxRxFlag &= (uint16_t)~(uint16_t)(PHHAL_HW_SET_TXCRC);
        }
        else if (wValue == PH_OFF)
        {
            pDataParams->wTxRxFlag |= PHHAL_HW_SET_TXCRC;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* Send the configuration to reader */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_TxRxFlag(
            pDataParams,
            pDataParams->wTxRxFlag));
        break;

    case PHHAL_HW_CONFIG_RXCRC:
        if (wValue == PH_ON)
        {
            pDataParams->wTxRxFlag |= PHHAL_HW_SET_RXCRC;
        }
        else if (wValue == PH_OFF)
        {
            pDataParams->wTxRxFlag &= (uint16_t)~(uint16_t)(PHHAL_HW_SET_RXCRC);
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* Send the configuration to reader */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_TxRxFlag(
            pDataParams,
            pDataParams->wTxRxFlag));
        break;

    case PHHAL_HW_CONFIG_TXLASTBITS:
        /* Check parameter */
        if (wValue > 0x07U)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        pDataParams->bTxLastBits = (uint8_t)wValue;
        break;

    case PHHAL_HW_CONFIG_ADDITIONAL_INFO:
        if (pDataParams->bExchangeType == PHHAL_HW_PCSC_TRANSPARENT)
        {
            /* CHECK: Change this if exchange algorithm is changed */
            pDataParams->wAdditionalInfo = (uint8_t)(wValue + 30U);
        }
        else
        {
            pDataParams->wAdditionalInfo = (uint8_t)wValue;
        }
        break;

    case PHHAL_HW_CONFIG_TXBUFFER:
        if (pDataParams->bExchangeType == PHHAL_HW_PCSC_TRANSPARENT)
        {
            /* CHECK: Change this if exchange algorithm is changed */
            if (pDataParams->wAdditionalInfo > (pDataParams->wTxBufLen + 30U))
            {
                return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
            }
        }
        else
        {
            if (pDataParams->wAdditionalInfo >= pDataParams->wTxBufLen)
            {
                return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
            }
        }

        pDataParams->pTxBuffer[pDataParams->wAdditionalInfo] = (uint8_t)wValue;
        break;

    case PHHAL_HW_CONFIG_RXBUFFER_STARTPOS:
        /* Boundary check */
        if (wValue >= pDataParams->wRxBufSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* Set start position */
        pDataParams->wRxBufStartPos = wValue;
        pDataParams->wRxBufLen = wValue;

        /* Preserve RxBuffer contents if needed */
        if (pDataParams->pTxBuffer == pDataParams->pRxBuffer)
        {
            pDataParams->wTxBufStartPos = pDataParams->wRxBufStartPos;
        }
        else
        {
            pDataParams->wTxBufStartPos = 0;
        }
        break;

    case PHHAL_HW_CONFIG_TXBUFFER_LENGTH:

        /* Retrieve TxBuffer parameters */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetTxBuffer(
            pDataParams,
            &pBuffer,
            &wBufferLen,
            &wBufferSize));

        /* Check parameter, must not exceed TxBufferSize */
        if (wValue > wBufferSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* Set buffer length */
        pDataParams->wTxBufLen = wValue;
        break;

    case PHHAL_HW_CONFIG_SETMINFDT:
        /* Set minimum FDT */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_SetMinFDT(
            pDataParams,
            wValue));
        break;

    case PHHAL_HW_CONFIG_FIELD_OFF_TIME:
        /* Parameter Check */
        if (wValue == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        pDataParams->wFieldOffTime = wValue;
        break;

    case PHHAL_HW_CONFIG_FIELD_RECOVERY_TIME:
        pDataParams->wFieldRecoveryTime = wValue;
        break;

    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_US:
        pDataParams->dwTimeoutUs = wValue;
        break;

    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS:
        pDataParams->dwTimeoutUs = (wValue * 1000);
        break;

    case PHHAL_HW_PCSC_CONFIG_FSDI:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_SetParam(
            pDataParams,
            0x01U,
            (uint8_t *)&wValue,
            0x01U));
        break;

    case PHHAL_HW_PCSC_CONFIG_FSCI:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_SetParam(
            pDataParams,
            0x02U,
            (uint8_t *)&wValue,
            0x01U));
        break;

    case PHHAL_HW_PCSC_CONFIG_FWTI:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_SetParam(
            pDataParams,
            0x03U,
            (uint8_t *)&wValue,
            0x01U));
        break;

    case PHHAL_HW_PCSC_CONFIG_MAX_SPEED:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_SetParam(
            pDataParams,
            0x04U,
            (uint8_t *)&wValue,
            0x01U));
        break;

    case PHHAL_HW_PCSC_CONFIG_CURRENT_SPEED:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_SetParam(
            pDataParams,
            0x05U,
            (uint8_t *)&wValue,
            0x01U));
        break;

    case PHHAL_HW_PCSC_CONFIG_MOD_INDEX:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_SetParam(
            pDataParams,
            0x06U,
            (uint8_t *)&wValue,
            0x01U));
        break;

    case PHHAL_HW_PCSC_CONFIG_PCB:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_SetParam(
            pDataParams,
            0x07U,
            (uint8_t *)&wValue,
            0x01U));
        break;

    case PHHAL_HW_PCSC_CONFIG_CID:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_SetParam(
            pDataParams,
            0x08U,
            (uint8_t *)&wValue,
            0x01U));
        break;

    case PHHAL_HW_PCSC_CONFIG_NAD:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_SetParam(
            pDataParams,
            0x09U,
            (uint8_t *)&wValue,
            0x01U));
        break;

    case PHHAL_HW_PCSC_CONFIG_DATA_CODING:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_SetParam(
            pDataParams,
            0x0BU,
            (uint8_t *)&wValue,
            0x01U));
        break;

    case PHHAL_HW_PCSC_CONFIG_EXCHANGE:
        /* Parameter Check */
        if ((wValue != PHHAL_HW_PCSC_TRANSPARENT) && (wValue != PHHAL_HW_PCSC_NORMAL))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        if ((wValue == PHHAL_HW_PCSC_TRANSPARENT) && (pDataParams->bTransparentSession == PH_OFF))
        {
            /* Start transparent session */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_StartTransparentSession(pDataParams));
        }

        if ((wValue == PHHAL_HW_PCSC_NORMAL) && (pDataParams->bTransparentSession == PH_ON))
        {
            /* End transparent session */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_EndTransparentSession(pDataParams));
        }

        pDataParams->bExchangeType = (uint8_t) wValue;
        break;

    case PHHAL_HW_PCSC_CONFIG_KEY_LOCATION:
        /* Parameter Check */
        if ((wValue != PHHAL_HW_PCSC_KEY_LOC_VOLATILE) && (wValue != PHHAL_HW_PCSC_KEY_LOC_NONVOLATILE))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        pDataParams->bKeyLocation = (uint8_t) wValue;
        break;

    case PHHAL_HW_CONFIG_TXDATARATE:
        if (wValue > PHHAL_HW_RF_DATARATE_848)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* Calculate data rate parameters */
        wDataRate = (wValue == pDataParams->bRxDataRate)? 0x80U : 0x00U;
        wDataRate |= ((pDataParams->bRxDataRate << 3U) & 0x18U) | (wValue & 0x03U);

        /* Apply new data rates */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_SetConfig(
            pDataParams,
            PHHAL_HW_PCSC_CONFIG_CURRENT_SPEED,
            wDataRate));

        /* Update value */
        pDataParams->bTxDataRate = (uint8_t)wValue;
        break;

    case PHHAL_HW_CONFIG_RXDATARATE:
        if (wValue > PHHAL_HW_RF_DATARATE_848)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* Calculate data rate parameters */
        wDataRate = (pDataParams->bTxDataRate == wValue)? 0x80U : 0x00U;
        wDataRate |= ((wValue << 3U) & 0x18U) | (pDataParams->bTxDataRate & 0x03U);

        /* Apply new data rates */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_SetConfig(
            pDataParams,
            PHHAL_HW_PCSC_CONFIG_CURRENT_SPEED,
            wDataRate));

        /* Update value */
        pDataParams->bRxDataRate = (uint8_t)wValue;
        break;

    case PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1:
    case PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL:
    case PHHAL_HW_CONFIG_RXALIGN:
    case PHHAL_HW_CONFIG_RXDEAFBITS:
        /* NA in PCSC mode */
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
        break;
    }

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_GetConfig(
                                  phhalHw_Pcsc_DataParams_t * pDataParams,
                                  uint16_t wConfig,
                                  uint16_t * pValue
                                  )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bLength = 0U;
    uint8_t *   PH_MEMLOC_REM pBuffer;
    uint16_t    PH_MEMLOC_REM wBufferLen;
    uint16_t    PH_MEMLOC_REM wBufferSize;

    switch(wConfig)
    {
    case PHHAL_HW_CONFIG_TXCRC:
        if ((pDataParams->wTxRxFlag & PHHAL_HW_SET_TXCRC) == PHHAL_HW_SET_TXCRC)
        {
            *pValue = PH_OFF;
        }
        else
        {
            *pValue = PH_ON;
        }
        break;

    case PHHAL_HW_CONFIG_RXCRC:
        if ((pDataParams->wTxRxFlag & PHHAL_HW_SET_RXCRC) == PHHAL_HW_SET_RXCRC)
        {
            *pValue = PH_ON;
        }
        else
        {
            *pValue = PH_OFF;
        }
        break;

    case PHHAL_HW_CONFIG_RXLASTBITS:
        *pValue = pDataParams->bRxLastBits;
        break;

    case PHHAL_HW_CONFIG_RXBUFFER_STARTPOS:
        *pValue = pDataParams->wRxBufStartPos;
        break;

    case PHHAL_HW_CONFIG_RXBUFFER_BUFSIZE:
        *pValue = pDataParams->wRxBufSize;
        break;

    case PHHAL_HW_CONFIG_TXBUFFER_BUFSIZE:
        *pValue = pDataParams->wTxBufSize - pDataParams->wTxBufStartPos;
        break;

    case PHHAL_HW_CONFIG_TXBUFFER:
        if (pDataParams->bExchangeType == PHHAL_HW_PCSC_TRANSPARENT)
        {
            /* CHECK: Change this if exchange algorithm is changed */
            if (pDataParams->wAdditionalInfo > (pDataParams->wTxBufLen + 30U))
            {
                return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
            }
        }
        else
        {
            if (pDataParams->wAdditionalInfo >= pDataParams->wTxBufLen)
            {
                return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
            }
        }

        *pValue = pDataParams->pTxBuffer[pDataParams->wAdditionalInfo];
        break;

    case PHHAL_HW_CONFIG_TXBUFFER_LENGTH:

        /* Retrieve TxBuffer parameters */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetTxBuffer(
            pDataParams,
            &pBuffer,
            &wBufferLen,
            &wBufferSize));

        /* Return parameter */
        *pValue = wBufferLen;
        break;

    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_US:
        if (pDataParams->dwTimeoutUs > 0xFFFFU)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }

        *pValue = (uint16_t)pDataParams->dwTimeoutUs;
        break;

    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS:
        *pValue = (uint16_t)(pDataParams->dwTimeoutUs / 1000U);
        break;

    case PHHAL_HW_PCSC_CONFIG_FSDI:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_GetParam(
            pDataParams,
            0x01U,
            (uint8_t *)pValue,
            &bLength));
        break;

    case PHHAL_HW_PCSC_CONFIG_FSCI:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_GetParam(
            pDataParams,
            0x02U,
            (uint8_t *)pValue,
            &bLength));
        break;

    case PHHAL_HW_PCSC_CONFIG_FWTI:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_GetParam(
            pDataParams,
            0x03U,
            (uint8_t *)pValue,
            &bLength));
        break;

    case PHHAL_HW_PCSC_CONFIG_MAX_SPEED:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_GetParam(
            pDataParams,
            0x04U,
            (uint8_t *)pValue,
            &bLength));
        break;

    case PHHAL_HW_PCSC_CONFIG_CURRENT_SPEED:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_GetParam(
            pDataParams,
            0x05U,
            (uint8_t *)pValue,
            &bLength));
        break;

    case PHHAL_HW_PCSC_CONFIG_MOD_INDEX:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_GetParam(
            pDataParams,
            0x06U,
            (uint8_t *)pValue,
            &bLength));
        break;

    case PHHAL_HW_PCSC_CONFIG_PCB:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_GetParam(
            pDataParams,
            0x07U,
            (uint8_t *)pValue,
            &bLength));
        break;

    case PHHAL_HW_PCSC_CONFIG_CID:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_GetParam(
            pDataParams,
            0x08U,
            (uint8_t *)pValue,
            &bLength));
        break;

    case PHHAL_HW_PCSC_CONFIG_NAD:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_GetParam(
            pDataParams,
            0x09U,
            (uint8_t *)pValue,
            &bLength));
        break;

    case PHHAL_HW_PCSC_CONFIG_DATA_CODING:
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_GetParam(
            pDataParams,
            0x0BU,
            (uint8_t *)pValue,
            &bLength));
        break;

    case PHHAL_HW_PCSC_CONFIG_STATUS_BYTES:
        *pValue = pDataParams->wErrorCode;
        break;

    case PHHAL_HW_PCSC_CONFIG_EXCHANGE:
        *pValue = pDataParams->bExchangeType;
        break;

    case PHHAL_HW_PCSC_CONFIG_KEY_LOCATION:
        *pValue = pDataParams->bKeyLocation;
        break;

    case PHHAL_HW_CONFIG_TXDATARATE:
        *pValue = pDataParams->bTxDataRate;
        break;

    case PHHAL_HW_CONFIG_RXDATARATE:
        *pValue = pDataParams->bRxDataRate;
        break;

    case PHHAL_HW_CONFIG_FIELD_OFF_TIME:
        *pValue = pDataParams->wFieldOffTime;
        break;

    case PHHAL_HW_CONFIG_FIELD_RECOVERY_TIME:
        *pValue = pDataParams->wFieldRecoveryTime;
        break;

    case PHHAL_HW_CONFIG_CARD_TYPE:
        *pValue = pDataParams->bCardType;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
        break;
    }

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Exchange(
                                 phhalHw_Pcsc_DataParams_t * pDataParams,
                                 uint16_t wOption,
                                 uint8_t * pTxBuffer,
                                 uint16_t wTxLength,
                                 uint8_t ** ppRxBuffer,
                                 uint16_t * pRxLength
                                 )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t *  PH_MEMLOC_REM pTmpRxBuffer = NULL;
    uint8_t *  PH_MEMLOC_REM pTmpTxBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wTmpBufferSize = 0U;
    uint16_t   PH_MEMLOC_REM wTmpBufferLen = 0U;

    /* Transparent HAL exchange */
    if (pDataParams->bExchangeType == PHHAL_HW_PCSC_TRANSPARENT)
    {
        /* Transparent exchange command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_Transceive(
            pDataParams,
            wOption,
            pTxBuffer,
            wTxLength,
            ppRxBuffer,
            pRxLength));
    }
    /* Normal HAL exchange */
    else
    {
        /* Check exchange option */
        if (wOption & (uint16_t)~(uint16_t)(PH_EXCHANGE_BUFFERED_BIT | PH_EXCHANGE_LEAVE_BUFFER_BIT))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* Clear internal buffer if not a continuous buffer */
        if ((wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT) == 0x00U)
        {
            pDataParams->wTxBufLen = 0;
        }

        /* Reset the receive length */
        if (pRxLength != NULL)
        {
            *pRxLength = 0;
        }

        /* Retrieve transmit buffer parameters */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetTxBuffer(
            pDataParams,
            &pTmpTxBuffer,
            &wTmpBufferLen,
            &wTmpBufferSize));

        /* If there is some data to transmit */
        if (wTxLength != 0U)
        {
            /* Check buffer size */
            if (wTmpBufferSize < (wTxLength + pDataParams->wTxBufLen))
            {
                /* Can't fit in the HAL buffer */
                pDataParams->wTxBufLen = 0U;

                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
            }

            /* Copy data to TX buffer */
            (void)memcpy(                                 /* PRQA S 3200 */
               &pTmpTxBuffer[pDataParams->wTxBufLen],
               pTxBuffer,
               wTxLength);

            /* Increment TX buffer length */
            pDataParams->wTxBufLen += wTxLength;
        }

        /* Return with success, if only buffering is requested. */
        if (wOption & PH_EXCHANGE_BUFFERED_BIT)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
        }

        /* Get RX buffer */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_GetRxBuffer(
            pDataParams,
            &pTmpRxBuffer,
            &wTmpBufferLen,
            &wTmpBufferSize));

        /* Exchange the data */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
            pDataParams->pBalDataParams,
            PH_EXCHANGE_DEFAULT,
            pTmpTxBuffer,
            pDataParams->wTxBufLen,
            wTmpBufferSize,
            pTmpRxBuffer,
            &wTmpBufferLen));

        /* Reset the TX parameters */
        pDataParams->wTxBufLen = 0;
        pDataParams->bTxLastBits = 0;

        /* Update received data length */
        pDataParams->wRxBufLen = pDataParams->wRxBufStartPos + wTmpBufferLen;

        /* Update out parameters */
        if (ppRxBuffer != NULL)
        {
            *ppRxBuffer = pDataParams->pRxBuffer;
        }
        if (pRxLength != NULL)
        {
            *pRxLength = pDataParams->wRxBufLen;
        }
    }

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Wait(
                             phhalHw_Pcsc_DataParams_t * pDataParams,
                             uint8_t bUnit,
                             uint16_t wTimeout
                             )
{
    /* Variables not used */
    PH_UNUSED_VARIABLE(pDataParams);

    if (bUnit == PHHAL_HW_TIME_MICROSECONDS)
    {
        (void)phTools_Sleep((wTimeout + 999) / 1000);
    }
    else if (bUnit == PHHAL_HW_TIME_MILLISECONDS)
    {
        (void)phTools_Sleep(wTimeout);
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Pcsc_MfcAuthenticateKeyNo(
                                             phhalHw_Pcsc_DataParams_t * pDataParams,
                                             uint8_t bBlockNo,
                                             uint8_t bKeyType,
                                             uint16_t wKeyNo,
                                             uint16_t wKeyVersion,
                                             uint8_t * pUid
                                             )
{
#ifdef NXPBUILD__PH_KEYSTORE
    phStatus_t  PH_MEMLOC_REM status;
    uint8_t     PH_MEMLOC_REM aKey[PHHAL_HW_MFC_KEY_LENGTH * 2];
    uint8_t *   PH_MEMLOC_REM pKey;
    uint16_t    PH_MEMLOC_REM bKeystoreKeyType;

    /* Bail out if we haven't got a keystore */
    if (pDataParams->pKeyStoreDataParams == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Retrieve KeyA & KeyB from keystore */
    status = phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wKeyNo,
        wKeyVersion,
        sizeof(aKey),
        aKey,
        &bKeystoreKeyType);

    /* Change component code for invalid paramter */
    if ((status & PH_ERR_MASK) == PH_ERR_INVALID_PARAMETER)
    {
        return PH_ADD_COMPCODE((status & PH_ERR_MASK), PH_COMP_HAL);
    }
    else
    {
        PH_CHECK_SUCCESS(status);
    }

    /* Check key type */
    if (bKeystoreKeyType != PH_KEYSTORE_KEY_TYPE_MIFARE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Evaluate which key to use */
    if (bKeyType == PHHAL_HW_MFC_KEYA)
    {
        /* Use KeyA */
        pKey = aKey;
    }
    else if (bKeyType == PHHAL_HW_MFC_KEYB)
    {
        /* Use KeyB */
        pKey = &aKey[PHHAL_HW_MFC_KEY_LENGTH];
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Call the authentication function using the key */
    return phhalHw_Pcsc_MfcAuthenticate(pDataParams, bBlockNo, bKeyType, aKey, pUid);
#else
    /* Satisfy compiler */
    if (pDataParams || bBlockNo || bKeyType || wKeyNo || wKeyVersion || pUid);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
#endif /* NXPBUILD__PH_KEYSTORE */
}

phStatus_t phhalHw_Pcsc_MfcAuthenticate(
                                        phhalHw_Pcsc_DataParams_t * pDataParams,
                                        uint8_t bBlockNo,
                                        uint8_t bKeyType,
                                        uint8_t * pKey,
                                        uint8_t * pUid
                                        )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t    PH_MEMLOC_REM bPcscKeyType = 0U;

    /* CHECK: Key number '0' used by default. Key number tracking and
     * management not implemented. */

    /* Variables not used */
    PH_UNUSED_VARIABLE(pUid);

    /* Check key type */
    if(bKeyType == PHHAL_HW_MFC_KEYA)
    {
        bPcscKeyType = PHHAL_HW_PCSC_KEY_A;
    }
    else if (bKeyType == PHHAL_HW_MFC_KEYB)
    {
        bPcscKeyType = PHHAL_HW_PCSC_KEY_B;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Load Key into reader; Key No '0' */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_LoadKeys(
        pDataParams,
        0U,
        0U,
        0U,
        pDataParams->bKeyLocation,
        0U,
        pKey,
        PHHAL_HW_MFC_KEY_LENGTH));

    /* Authenticate; With Key No '0' */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_GeneralAuthenticate(
        pDataParams,
        bPcscKeyType,
        0U,
        bBlockNo));

    return PH_ERR_SUCCESS;
}

#endif  /* NXPBUILD__PHHAL_HW_PCSC */
