/*
 * Copyright 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
* SAM (Secure Access Module) internal implementation via I2C interface for Reader Library
* $Author: Rajendran Kumar (nxp99556) $
* $Revision: 7467 $
* $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
*
* History:
*  Rajendran Kumar: Generated 08. Feb 2024
*
*/

#include <ph_Status.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHBAL_REG_SAM

#include "phbalReg_Sam_I2C.h"
#include "phbalReg_Sam_I2C_T1.h"

#include "../phbalReg_Sam.h"
#include "../phbalReg_Sam_Int.h"

phStatus_t phbalReg_Sam_Int_Exchange_I2C(phbalReg_Sam_DataParams_t * pDataParams, uint8_t bCommType, uint8_t bCmd,
    uint8_t * pData, uint16_t wDataLen, uint32_t dwBytesToRead, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint16_t    PH_MEMLOC_REM wOption = 0;
    uint16_t    PH_MEMLOC_REM wCmd = 0;
    uint16_t    PH_MEMLOC_REM wTotRespLen = 0;
    uint16_t    PH_MEMLOC_REM wRemRespLen = 0;
    uint16_t    PH_MEMLOC_REM wLEN = 0;
    uint16_t    PH_MEMLOC_REM wTxLen = 0;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;
    uint8_t     PH_MEMLOC_REM bRetries = 0U;
    uint8_t     PH_MEMLOC_REM bExchangeTx = PH_ON;

    /* Add command to TX buffer. */
    pDataParams->pTxBuffer[wTxLen++] = bCommType;
    pDataParams->pTxBuffer[wTxLen++] = bCmd;

    /* Frame the actual command. */
    wCmd = (uint16_t) ((bCommType << 8U) | bCmd);

    /* Add additional info to TxBuffer based on command. */
    switch(wCmd)
    {
        case PHBAL_SAM_I2C_CMD_EXCHANGE_DATA:
        case PHBAL_SAM_I2C_CMD_TRANSMIT_DATA:
        case PHBAL_SAM_I2C_CMD_RECEIVE_DATA:
            /* Compute the complete length
             * Default 7 bytes includes SlotNo(1Byte), Options(2Byte), I2CAddress(2Byte) and
             * NoOfBytesToREad(2Bytes)
             */
            wLEN = (uint16_t) (7U + wDataLen);

            /* Frame the command. */
            pDataParams->pTxBuffer[wTxLen++] = (uint8_t) wLEN;
            pDataParams->pTxBuffer[wTxLen++] = (uint8_t) ((wLEN & 0xFF00U) >> 8U);
            pDataParams->pTxBuffer[wTxLen++] = 0x00U;   /* Slot Number */
            pDataParams->pTxBuffer[wTxLen++] = 0x00U;   /* Option (LSB) */
            pDataParams->pTxBuffer[wTxLen++] = 0x00U;   /* Option (MSB) */
            pDataParams->pTxBuffer[wTxLen++] = (uint8_t) pDataParams->wI2C_SlaveAddr;
            pDataParams->pTxBuffer[wTxLen++] = (uint8_t) ((pDataParams->wI2C_SlaveAddr & 0xFF00U) >> 8U);

            /* Add RFU fields if Command is Transmit_Data */
            if(wCmd == PHBAL_SAM_I2C_CMD_TRANSMIT_DATA)
            {
                pDataParams->pTxBuffer[wTxLen++] = 0;
                pDataParams->pTxBuffer[wTxLen++] = 0;
            }
            else
            {
                pDataParams->pTxBuffer[wTxLen++] = (uint8_t) dwBytesToRead;
                pDataParams->pTxBuffer[wTxLen++] = (uint8_t) ((dwBytesToRead & 0xFF00U) >> 8U);
            }

            /* Read the Header Information. */
            wRemRespLen = (uint16_t) (PHBAL_SAM_FRAME_HEADER_LEN + ((dwBytesToRead == PHBAL_SAM_I2C_READ_UNKNOWN) ?
                PHBAL_SAM_MAX_FRAME_LEN : dwBytesToRead));
            break;

        case PHBAL_SAM_I2C_CMD_SET_CONFIGURATION:
            /* Compute the complete length
             * Default 5 bytes includes SlotNo(1Byte), ConfigID(2Byte) and ConfigValue(2Byte)
             */
            wLEN = 5U;

            /* Frame the command. */
            pDataParams->pTxBuffer[wTxLen++] = (uint8_t) wLEN;
            pDataParams->pTxBuffer[wTxLen++] = (uint8_t) ((wLEN & 0xFF00U) >> 8U);
            pDataParams->pTxBuffer[wTxLen++] = 0x00U;      /* Slot Number */

            /* Read the Header Information. */
            wRemRespLen = PHBAL_SAM_FRAME_HEADER_LEN;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }

    /* Add data if applicable. */
    (void) memcpy(&pDataParams->pTxBuffer[wTxLen], pData, wDataLen);
    wTxLen += wDataLen;

    /* Set Option info to be provided for lower BAL. */
    wOption = (uint16_t) PH_EXCHANGE_DEFAULT;

    /* Update information based on Lower BAL. */
    wOption |= (uint16_t) PHBAL_REG_SERIALWIN_SUPRESS_CHECKS;

    do
    {
        wStatus = phbalReg_Exchange(
            pDataParams->pLowerBalDataParams,
            wOption,
            bExchangeTx ? pDataParams->pTxBuffer : NULL,
            (uint16_t) (bExchangeTx ? wTxLen : 0U),
            wRemRespLen,
            &pDataParams->pRxBuffer[wTotRespLen],
            &wRspLen);

        wTotRespLen += wRspLen;
        bExchangeTx = PH_OFF;

        /* Check for error status from Reader */
        if(wRspLen >= PHBAL_SAM_FRAME_HEADER_LEN)
        {
            /* Retrieve wStatus */
            wStatus1 = PH_ADD_COMPCODE((uint16_t) pDataParams->pRxBuffer[PHBAL_SAM_FRAME_STATUS_POS],
                ((uint16_t) pDataParams->pRxBuffer[PHBAL_SAM_FRAME_STATUS_POS + 1] << 8));

            if((wStatus1 & PH_ERR_MASK) != PH_ERR_SUCCESS)
                break;
        }

        /* Reset Retry count */
        if(wRspLen != 0)
            bRetries = 0;

        /* Break the loop in case of error and max retries reached. */
        bRetries += (uint8_t) (((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) || (wRspLen == 0));
        if(bRetries == 10U)
            break;

    } while(wTotRespLen != wRemRespLen);

    if((wStatus1 & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* Check response */
        PH_CHECK_SUCCESS_FCT(wStatus, phbalReg_Sam_Int_CheckResponse(
            wCmd,
            pDataParams->pRxBuffer,
            wTotRespLen,
            NULL,
            &wRspLen));

        /* Move the response to parameters. */
        if(pRspLen != NULL)
        {
            if(*ppResponse == NULL)
            {
                *ppResponse = &pDataParams->pRxBuffer[PHBAL_SAM_FRAME_PAYLOAD_POS];
            }
            else
            {
                memcpy(ppResponse[0], &pDataParams->pRxBuffer[PHBAL_SAM_FRAME_PAYLOAD_POS], wRspLen);
            }

            *pRspLen = wRspLen;
        }
    }
    else
    {
        return wStatus1;
    }

    return wStatus;
}





phStatus_t phbalReg_Sam_ActivateSam_I2C(phbalReg_Sam_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint16_t    PH_MEMLOC_REM wConfig = 0;
    uint16_t    PH_MEMLOC_REM wValue = 0;

    uint8_t     PH_MEMLOC_REM aData[4U];

    uint8_t *   PH_MEMLOC_REM pAtr = NULL;
    uint16_t    PH_MEMLOC_REM wAtrLen = 0;

    /* Clear Transmit and Receive buffer */
    memset(pDataParams->pTxBuffer, 0x00U, pDataParams->wTxBufSize);
    memset(pDataParams->pRxBuffer, 0x00U, pDataParams->wRxBufSize);

    /* Perform Configuration Setting to switch off SAM Power */
    wConfig = PHBAL_SAM_I2C_CONFIG_POWER_SAM;
    memcpy(aData, (uint8_t *) &wConfig, 2U);

    wValue = PHBAL_SAM_I2C_POWER_OFF_SAM;
    memcpy(&aData[2U], (uint8_t *) &wValue, 2U);

    PH_CHECK_SUCCESS_FCT(wStatus, phbalReg_Sam_Int_Exchange_I2C(
        pDataParams,
        PHBAL_SAM_CMD_CLA_CONFIGURATION,
        PHBAL_SAM_CMD_INS_CONFIGURATION_SET,
        aData,
        4U,
        PHBAL_SAM_FRAME_HEADER_LEN,
        &pAtr,
        &wAtrLen));

    /* Perform Configuration Setting to switch on SAM Power */
    wValue = PHBAL_SAM_I2C_POWER_ON_SAM;
    memcpy(&aData[2U], (uint8_t *) &wValue, 2U);

    PH_CHECK_SUCCESS_FCT(wStatus, phbalReg_Sam_Int_Exchange_I2C(
        pDataParams,
        PHBAL_SAM_CMD_CLA_CONFIGURATION,
        PHBAL_SAM_CMD_INS_CONFIGURATION_SET,
        aData,
        4U,
        PHBAL_SAM_FRAME_HEADER_LEN,
        &pAtr,
        &wAtrLen));

    /* Enable PortOn Status */
    pDataParams->bIsPortOpened = PH_ON;

    /* Configure Timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phbalReg_SetConfig(
        pDataParams,
        PHBAL_REG_SAM_CONFIG_I2C_TIMEOUT_MS,
        pDataParams->wI2C_TimeoutMs));

    /* Get the ATR */
    PH_CHECK_SUCCESS_FCT(wStatus, phbalReg_Sam_Int_Exchange_I2C(
        pDataParams,
        PHBAL_SAM_CMD_CLA_COMMUNICATION_I2C,
        PHBAL_SAM_CMD_INS_I2C_EXCHANGE_DATA,
        NULL,
        0,
        28U,
        &pAtr,
        &wAtrLen));

    /* Store ATR */
    if(wAtrLen < pDataParams->wMaxAtrBufSize)
    {
        (void) memcpy(pDataParams->pAtrBuffer, pAtr, wAtrLen);
        pDataParams->wAtrBufSize = wAtrLen;
    }
    else
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
    }

    return wStatus;
}

phStatus_t phbalReg_Sam_DeActivateSam_I2C(phbalReg_Sam_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint16_t    PH_MEMLOC_REM wConfig = 0;
    uint16_t    PH_MEMLOC_REM wValue = 0;

    uint8_t     PH_MEMLOC_REM aData[4U];

    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Clear Transmit and Receive buffer */
    memset(pDataParams->pTxBuffer, 0x00U, pDataParams->wTxBufSize);
    memset(pDataParams->pRxBuffer, 0x00U, pDataParams->wRxBufSize);

    /* Perform Configuration Setting to switch off SAM Power */
    wConfig = PHBAL_SAM_I2C_CONFIG_POWER_SAM;
    memcpy(aData, (uint8_t *) &wConfig, 2U);

    wValue = PHBAL_SAM_I2C_POWER_OFF_SAM;
    memcpy(&aData[2U], (uint8_t *) &wValue, 2U);

    PH_CHECK_SUCCESS_FCT(wStatus, phbalReg_Sam_Int_Exchange_I2C(
        pDataParams,
        PHBAL_SAM_CMD_CLA_CONFIGURATION,
        PHBAL_SAM_CMD_INS_CONFIGURATION_SET,
        aData,
        4U,
        PHBAL_SAM_FRAME_HEADER_LEN,
        &pResponse,
        &wRspLen));

    return wStatus;
}

phStatus_t phbalReg_Sam_TransmitData_I2C(phbalReg_Sam_DataParams_t * pDataParams, uint8_t * pTxBuffer, uint16_t wTxBufLen,
    uint16_t wRxBufSize, uint8_t * pRxBuffer, uint16_t * pRxBufLen)
{
    PH_UNUSED_VARIABLE(pDataParams);
    PH_UNUSED_VARIABLE(wRxBufSize);

    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Build and exchange I-Block */
    if(pDataParams->bI2C_NextFrame == PHBAL_SAM_I2C_SEND_IFRAME)
    {
        wStatus = phbalReg_Sam_I2C_T1_BuildIBlock(pDataParams, pTxBuffer, &wTxBufLen);
    }

    /* Build and exchange S-Block */
    else if((pDataParams->bI2C_NextFrame == PHBAL_SAM_I2C_SEND_SFRAME_WTX_RESP) ||
        (pDataParams->bI2C_NextFrame == PHBAL_SAM_I2C_SEND_SFRAME_IFS))
    {
        /* S-Block */
        wStatus = phbalReg_Sam_I2C_T1_BuildSBlock(pDataParams, pTxBuffer[0]);
        pDataParams->bI2C_NextFrame = PHBAL_SAM_I2C_SEND_IFRAME;
    }
    else
    {
        /* Do nothing */
    }

    if(wStatus == PH_ERR_SUCCESS)
    {
        /* Perform Read and process the response  */
        wStatus = phbalReg_Sam_I2C_T1_ProcessResponse(
            pDataParams,
            wRxBufSize,
            pRxBuffer,
            pRxBufLen);
    }
    return wStatus;
}

#endif /* NXPBUILD__PHBAL_REG_SAM */
