/*
 * Copyright 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <ph_TypeDefs.h>
#include <phhalHw.h>
#include <phpalMifare.h>
#include <string.h>

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_X

#include "../phalMfPrimeNtag_Int.h"
#include "phalMfPrimeNtag_Sam_X.h"
#include "phalMfPrimeNtag_Sam_X_Int.h"
#include <phhalHw_SamAV3_Cmd.h>

phStatus_t phalMfPrimeNtag_SamAV3_X_Init(phalMfPrimeNtag_SamAV3_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, phhalHw_SamAV3_DataParams_t * pHalSamDataParams)
{
    /* data param check */
    if (sizeof(phalMfPrimeNtag_SamAV3_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
    }

    PH_ASSERT_NULL_DATA_PARAM (pDataParams, PH_COMP_AL_MFPRIMENTAG);
    PH_ASSERT_NULL_PARAM (pHalSamDataParams, PH_COMP_AL_MFPRIMENTAG);

    /* init private data */
	pDataParams->wId                  = PH_COMP_AL_MFPRIMENTAG | PHAL_MFPRIMENTAG_SAMAV3_X_ID;
    pDataParams->pHalSamDataParams    = pHalSamDataParams;
    pDataParams->bAuthMode            = PHAL_MFPRIMENTAG_NOT_AUTHENTICATED;
    pDataParams->bKeyNo               = 0xFF;
    pDataParams->wAdditionalInfo      = 0x0000;

    /* Default selected application */
    memset(pDataParams->aAid, 0x00, 3); /* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

/* MIFARE PRIME NTAG secure messaging related commands. ------------------------------------------------------------------------------ */
phStatus_t phalMfPrimeNtag_Sam_X_AuthenticateEv2(void *pDataParams, uint8_t bAuthType, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput,
	uint8_t bDivInputLen, uint8_t * pPcdCapsIn, uint8_t bPcdCapsInLen, uint8_t * pPcdCapsOut, uint8_t * pPdCapsOut)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM bAuthMode = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM aAppId[3] = {0x00, 0x00, 0x00};

	/* Check if First Auth parameter do not contain invalid values. */
	if(bAuthType > PHAL_MFPRIMENTAG_AUTHFIRST_NON_LRP)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}

	/* For EV2 mode the length should not exceed 6 bytes. */
	if(bPcdCapsInLen > 6)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}

#ifdef RDR_LIB_PARAM_CHECK
	if(memcmp(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00)
	{
		/* At PICC level valid key numbes are 1 - 4. */
		if((bKeyNoCard > PHAL_MFPRIMENTAG_ORIGINALITY_KEY_LAST ) || (bKeyNoCard < PHAL_MFPRIMENTAG_ORIGINALITY_KEY_FIRST))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
	}
	else
	{
		/* At App level valid key numbers are 0 - 4. */
		if((bKeyNoCard > PHAL_MFPRIMENTAG_APP_KEY_LAST))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
	}

	/* Change for valid diversification options. */
	if ((wOption != PHAL_MFPRIMENTAG_NO_DIVERSIFICATION) &&
		(wOption != PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Validate diversification input length. */
	if ((wOption != PHAL_MFPRIMENTAG_NO_DIVERSIFICATION) && (bDivInputLen > 31 ))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}
#endif

    /* Validate SAM Key number and version. */
    if ((wKeyNo > 0x7f) || (wKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

	/* Clear the PICC status code buffer. */
	memset(aPiccRetCode, 0x00, sizeof(aPiccRetCode));	/* PRQA S 3200 */

	/*
	 * Set EV2 Authentication type.
	 * EVx authentication type
	 * Authenticate first
	 * Authenticate non first
	 * AV2 compatibility mode key diversification methods, 3TDEA, AES key
	 */
	bAuthMode = (uint8_t) ((bAuthType == PHAL_MFPRIMENTAG_AUTHFIRST_NON_LRP) ? PHHAL_HW_CMD_SAMAV3_AUTH_MODE_EV2_FIRST_AUTH :
		PHHAL_HW_CMD_SAMAV3_AUTH_MODE_EV2_NON_FIRST_AUTH);

	/* Set Auth mode with diversification enabled. */
	bAuthMode = (uint8_t) (bAuthMode | ((wOption != PHAL_MFPRIMENTAG_NO_DIVERSIFICATION) ? PHHAL_HW_CMD_SAMAV3_KEY_DIVERSIFICATION_ON :
		PHHAL_HW_CMD_SAMAV3_KEY_DIVERSIFICATION_OFF));

	/* Set Diversification flags.
	 * AV2 compatibility mode key diversification methods, 3TDEA, AES key
	 */
    if (wOption == PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS)
    {
        bAuthMode = (uint8_t) (bAuthMode | PHHAL_HW_CMD_SAMAV3_KDF_AV2);
	}

	/* Set the secure messaging. */
	if((bKeyNoCard >= PHAL_MFPRIMENTAG_ORIGINALITY_KEY_FIRST) && (bKeyNoCard <= PHAL_MFPRIMENTAG_ORIGINALITY_KEY_LAST) &&
		(memcmp(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00))
	{
		bAuthMode = (uint8_t) (bAuthMode | PHHAL_HW_CMD_SAMAV3_SUPPRESS_SECURE_MESSAGING);
	}

	/* Reset the Authentication state. */
	if(bAuthType == PHAL_MFPRIMENTAG_AUTHFIRST_NON_LRP)
	{
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_ResetAuthentication(pDataParams));
	}

    wStatus = phhalHw_SamAV3_Cmd_DESFire_AuthenticatePICC(
		PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
        bAuthMode,
        PHHAL_HW_CMD_SAMAV3_ISO_MODE_ISO7816,
        bKeyNoCard,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
		bPcdCapsInLen,
		bPcdCapsInLen ? pPcdCapsIn : NULL,
        pDivInput,
        bDivInputLen,
		pPdCapsOut,
		pPcdCapsOut,
		aPiccRetCode);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccRetCode));

	/* Update the authentication states. */
	PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode = PHAL_MFPRIMENTAG_AUTHENTICATEEV2;
	PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo = bKeyNoCard;

	/* Do not update the auth state if originality keys are used. */
	if((bKeyNoCard >= PHAL_MFPRIMENTAG_ORIGINALITY_KEY_FIRST) && (bKeyNoCard <= PHAL_MFPRIMENTAG_ORIGINALITY_KEY_LAST) &&
		(memcmp(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00))
	{
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_ResetAuthentication(pDataParams));
	}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}




/* MIFARE PRIME NTAG Memory and Configuration mamangement commands. ------------------------------------------------------------------ */
phStatus_t phalMfPrimeNtag_Sam_X_SetConfiguration(void * pDataParams, uint8_t bOption, uint8_t * pData, uint8_t bDataSize)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[28];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

#ifdef RDR_LIB_PARAM_CHECK
	if (PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPRIMENTAG_NOT_AUTHENTICATED)
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPRIMENTAG);
	}
#endif

	/* Set the Length and Crypto information. */
	aAppData[wAppDataLen++] = 6;

	/* Set Extended Offset in Crypto mode. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET;

	/* Set Length Presence flag. */
	bLenPresent = PH_ON;

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFPRIMENTAG_CMD_SET_CONFIG;
	aAppData[wAppDataLen++] = bOption;

    memcpy(&aAppData[wAppDataLen], pData, bDataSize); /* PRQA S 3200 */
	wAppDataLen = (uint16_t) (wAppDataLen + bDataSize);

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_Iso7816Wrap(
		bLenPresent,
		1,
		PH_OFF,
		0,
		aAppData,
		&wAppDataLen));

	/* Frame the Crypto information. */
	bCrypto |= PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_FULL;

#ifndef RDR_LIB_PARAM_CHECK
	/* Set the Command Offset. */
	if(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode != PHAL_MFPRIMENTAG_AUTHENTICATEEV2)
	{
		bCrypto = (uint8_t) (bCrypto + (wAppDataLen - bDataSize));
	}
#endif

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
		PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_X_GetVersion(void * pDataParams, uint8_t * pVerInfo)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[7];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint16_t	PH_MEMLOC_REM wVerLen = 0;
	uint8_t		PH_MEMLOC_REM bFinished = 0;

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	if(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPRIMENTAG_AUTHENTICATEEV2)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFPRIMENTAG_CMD_GET_VERSION;

	do
	{
		/* Wrap the command if required. */
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_Iso7816Wrap(
			PH_OFF,
			0,
			PH_ON,
			0,
			aAppData,
			&wAppDataLen));

		/* Exchange the information to Sam. */
		wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
			PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			PH_EXCHANGE_DEFAULT,
			bCrypto,
			aAppData,
			(uint8_t) wAppDataLen,
			&pResponse,
			&wRespLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Evaluate the response. */
		wStatus = phalMfPrimeNtag_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccRetCode);

		/* Updated Finished to end the loop. */
		if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS))
		{
			PH_CHECK_SUCCESS(wStatus);
		}
		else
		{
			/* Update Command frame. */
			wAppDataLen = 0;
			aAppData[wAppDataLen++] = 0xAF;

			/* Copy the response information to the parameter. */
			memcpy(&pVerInfo[wVerLen], pResponse, wRespLen); /* PRQA S 3200 */
			wVerLen = (uint8_t) (wVerLen + wRespLen);

			/* Update Finish flag. */
			if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
			{
				bFinished = 1;
			}
		}
	}while(!bFinished);

	/* Do a Set Config of ADDITIONAL_INFO to set  the length(wLength) of the Version string */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_SetConfig(pDataParams, PHAL_MFPRIMENTAG_ADDITIONAL_INFO, wVerLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}




/* MIFARE PRIME NTAG Key mamangement commands. -------------------------------------------------------------------------------------- */
phStatus_t phalMfPrimeNtag_Sam_X_ChangeKey(void * pDataParams, uint16_t wOption, uint16_t wCurrKeyNo, uint16_t wCurrKeyVer, uint16_t wNewKeyNo, uint16_t wNewKeyVer, uint8_t bKeyNoCard,
	uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bKeyCompMeth = 0;
    uint8_t     PH_MEMLOC_REM bCfg = 0;
	uint8_t		PH_MEMLOC_REM aPiccRetCode[2];

#ifdef RDR_LIB_PARAM_CHECK
	uint8_t		PH_MEMLOC_REM aAppId[3] = {0x00, 0x00, 0x00};

	/* Only if seleted Aid is 0x000000. */
	if (memcmp(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00)
    {
		/* At PICC level ChangeKey cannot be performed. */
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }
    else
    {
		/* At App level, 0,1,2,3,4 are valid ones. */
		if(bKeyNoCard > PHAL_MFPRIMENTAG_APP_KEY_LAST)
		{
			/* Invalid application key specified */
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
    }

	if (PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPRIMENTAG_NOT_AUTHENTICATED)
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPRIMENTAG);
	}
#endif

    /* Check for valid SAM key number and version. */
    if ((wCurrKeyNo > 0x7f) || (wCurrKeyVer > 0xff) || (wNewKeyNo > 0x7f) || (wNewKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

	/* Clear the PICC status code buffer. */
	memset(aPiccRetCode, 0x00, sizeof(aPiccRetCode));	/* PRQA S 3200 */

    /* Set the key compilation method. */
	if(wOption == PHAL_MFPRIMENTAG_NO_DIVERSIFICATION)
	{
		bKeyCompMeth = 0x00;
	}
	else
	{
#ifdef RDR_LIB_PARAM_CHECK
		/* Change for valid diversification options. */
		if ((wOption & PHHAL_HW_CMD_SAMAV3_DIV_METHOD_AV2) != PHHAL_HW_CMD_SAMAV3_DIV_METHOD_AV2)
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}

		/* Validate diversification input length. */
		if ( bDivInputLen > 31 )
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
#endif

		/* Assin the option to local variable. */
		bKeyCompMeth = (uint8_t) wOption;
	}

    /* Desfire key number to be changed. */
	bCfg = bKeyNoCard;

    /* Set if PICC targeted key equal to PICC authenticated key. */
    if ((PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo) == bKeyNoCard)
    {
        bKeyCompMeth = (uint8_t) (bKeyCompMeth | PHHAL_HW_CMD_SAMAV3_CRYPTO_MODE_SAME_KEY);
    }

	/* Enable Iso7816 wrapping. */
    bCfg = (uint8_t) (bCfg | PHHAL_HW_CMD_SAMAV3_ISO_MODE_ISO7816);

	/* Exchange the command to PICC. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ChangeKeyPICC(
		PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		bKeyCompMeth,
		bCfg,
		0x00,
		0x00,
		(uint8_t)wCurrKeyNo,
		(uint8_t)wCurrKeyVer,
		(uint8_t)wNewKeyNo,
		(uint8_t)wNewKeyVer,
		pDivInput,
		bDivInputLen,
		aPiccRetCode);

    /* Evaluate the response. */
	wStatus = phalMfPrimeNtag_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccRetCode);

	/* Reset the Auth state. */
	if(wStatus != PH_ERR_SUCCESS)
	{
		/* Kill the PICC Authentication. */
		PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SamAV3_Cmd_SAM_KillAuthentication(
			PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
			PHHAL_HW_SAMAV3_CMD_SAM_KILL_AUTHENTICATION_PARTIAL));

		return wStatus;
	}
	else
	{
		/* Reset authentication status only if the key authenticated with is changed. */
		if ((PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo) == bKeyNoCard)
		{
			PH_CHECK_SUCCESS_FCT(wStatus1, phalMfPrimeNtag_Sam_X_ResetAuthentication(pDataParams));
		}
	}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_X_GetKeyVersion(void * pDataParams, uint8_t bKeyNo, uint8_t * pKeyVersion, uint8_t * pKeyVerLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

#ifdef RDR_LIB_PARAM_CHECK
	uint8_t		PH_MEMLOC_REM aAppId[3] = {0x00, 0x00, 0x00};

	/* Only if seleted Aid is 0x000000. */
	if (memcmp(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00)
    {
		/* At PICC level, 1,2,3,4 are valid ones. 0 is excluded */
		if (bKeyNo >= 0x05)
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
    }
	else
	{
		/* At App level, 0,1,2,3,4 are valid ones. */
		if (bKeyNo >= 0x03)
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
	}
#endif

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	if(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPRIMENTAG_AUTHENTICATEEV2)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFPRIMENTAG_CMD_GET_KEY_VERSION;
	aAppData[wAppDataLen++] = bKeyNo;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_Iso7816Wrap(
		PH_OFF,
		0,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccRetCode));

	/* Copy the data to the parameter */
	memcpy(pKeyVersion, pResponse, wRespLen);	/* PRQA S 3200 */
	*pKeyVerLen = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}




/* MIFARE PRIME NTAG File mamangement commands. -------------------------------------------------------------------------------------- */
phStatus_t phalMfPrimeNtag_Sam_X_GetFileSettings(void * pDataParams, uint8_t bFileNo, uint8_t * pFSBuffer, uint8_t * pBufferLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	if(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPRIMENTAG_AUTHENTICATEEV2)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFPRIMENTAG_CMD_GET_FILE_SETTINGS;
	aAppData[wAppDataLen++] = bFileNo;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_Iso7816Wrap(
		PH_OFF,
		0,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccRetCode));

	/* Copy the data to the parameter */
	memcpy(pFSBuffer, pResponse, wRespLen);	/* PRQA S 3200 */
	*pBufferLen = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_X_GetFileCountersSDM(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pFileCounters, uint8_t * pRespLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[11];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;

	/* Validate the parameters */
	if ((bOption != PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) &&
        (bOption != PHAL_MFPRIMENTAG_COMMUNICATION_ENC))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

	/* Append expected length to the command frame. */
	if (bOption == PHAL_MFPRIMENTAG_COMMUNICATION_ENC)
	{
		aAppData[wAppDataLen++] = 5;
		aAppData[wAppDataLen++] = 0;
		aAppData[wAppDataLen++] = 0;

		bLenPresent = PH_ON;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFPRIMENTAG_CMD_GET_SDM_FILE_COUNTER;
	aAppData[wAppDataLen++] = bFileNo;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_Iso7816Wrap(
		bLenPresent,
		3,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bOption,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccRetCode));

	/* Copy the data to the parameter */
	memcpy(pFileCounters, pResponse, wRespLen);	/* PRQA S 3200 */
	*pRespLen = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_X_ChangeFileSettings(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t bFileOption, uint8_t * pAccessRights)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[16];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;

	/* Frame the Crypto information.
	 * Masking the MSB bit becuase its not support by SAM.
	 * Its actually supported in Software only.
	 */
	bCrypto = (uint8_t) (bOption & 0xFE);

	/* Validate the parameters */
	if ((bCrypto != PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) &&
        (bCrypto != PHAL_MFPRIMENTAG_COMMUNICATION_ENC) &&
        (bCrypto != PHAL_MFPRIMENTAG_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

#ifdef RDR_LIB_PARAM_CHECK
	if ((bCrypto == PHAL_MFPRIMENTAG_COMMUNICATION_ENC) &&
		(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPRIMENTAG_NOT_AUTHENTICATED))
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPRIMENTAG);
	}
#endif

	/* Set the Length and Crypto information. */
	if(bCrypto == PHAL_MFPRIMENTAG_COMMUNICATION_ENC)
	{
		/* Start of data. */
		aAppData[wAppDataLen++] = 6;

		/* Set Extended Offset in Crypto mode. */
		bCrypto |= PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET;

		/* Set Length Presence flag. */
		bLenPresent = PH_ON;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFPRIMENTAG_CMD_CHANGE_FILE_SETTINGS;
	aAppData[wAppDataLen++] = bFileNo;
	aAppData[wAppDataLen++] = (uint8_t) (bFileOption >> 4);

    /* Append access rights. */
	memcpy(&aAppData[wAppDataLen], pAccessRights, 2); /* PRQA S 3200 */
	wAppDataLen += 2;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_Iso7816Wrap(
		bLenPresent,
		1,
		PH_OFF,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
		PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_X_ChangeFileSettingsSDM(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t bFileOption, uint8_t *pAccessRights,
	uint8_t bSdmOptions, uint8_t *pSdmAccessRights, uint8_t *pVCUIDOffset, uint8_t *pSDMReadCtrOffset, uint8_t *pSDMMACInputOffset, uint8_t *pSDMMACOffset)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[40];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;

	/* Validate the parameters */
#if RDR_LIB_PARAM_CHECK
	if(bFileNo != PHAL_MFPRIMENTAG_SDM_FILE_ID)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}

	if ((bOption == PHAL_MFPRIMENTAG_COMMUNICATION_ENC) &&
		(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPRIMENTAG_NOT_AUTHENTICATED))
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPRIMENTAG);
	}
#endif

	if ((bOption != PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) &&
        (bOption != PHAL_MFPRIMENTAG_COMMUNICATION_ENC) &&
        (bOption != PHAL_MFPRIMENTAG_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

	/* Set the Length and Crypto information. */
	if(bOption == PHAL_MFPRIMENTAG_COMMUNICATION_ENC)
	{
		/* Start of data. */
		aAppData[wAppDataLen++] = 6;

		/* Set Extended Offset in Crypto mode. */
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET;

		/* Set Length Presence flag. */
		bLenPresent = PH_ON;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFPRIMENTAG_CMD_CHANGE_FILE_SETTINGS;
	aAppData[wAppDataLen++] = bFileNo;
	aAppData[wAppDataLen++] = bFileOption;

    /* Append access rights. */
	memcpy(&aAppData[wAppDataLen], pAccessRights, 2); /* PRQA S 3200 */
	wAppDataLen += 2;

	/* Append SDMOption information. */
	aAppData[wAppDataLen++] = bSdmOptions;

	/* Append SDM access rights. */
	memcpy(&aAppData[wAppDataLen], pSdmAccessRights, 2); /* PRQA S 3200 */
	wAppDataLen += 2;

	/* Append VCUID Offset. */
	if(pVCUIDOffset != NULL)
	{
		memcpy(&aAppData[wAppDataLen], pVCUIDOffset, 3); /* PRQA S 3200 */
		wAppDataLen += 3;
	}

	/* Append SDM Read Ctr Offset. */
	if(pSDMReadCtrOffset != NULL)
	{
		memcpy(&aAppData[wAppDataLen], pSDMReadCtrOffset, 3); /* PRQA S 3200 */
		wAppDataLen += 3;
	}

	/* Append SDM MAC Input Offset. */
	if(pSDMMACInputOffset != NULL)
	{
		memcpy(&aAppData[wAppDataLen], pSDMMACInputOffset, 3); /* PRQA S 3200 */
		wAppDataLen += 3;
	}

	/* Append SDM MAC Offset. */
	if(pSDMMACOffset != NULL)
	{
		memcpy(&aAppData[wAppDataLen], pSDMMACOffset, 3); /* PRQA S 3200 */
		wAppDataLen += 3;
	}

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_Iso7816Wrap(
		bLenPresent,
		1,
		PH_OFF,
		0,
		aAppData,
		&wAppDataLen));

	/* Set the communication option. */
	bCrypto |= bOption;

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
		PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}




/* MIFARE PRIME NTAG Data mamangement commands. -------------------------------------------------------------------------------------- */
phStatus_t phalMfPrimeNtag_Sam_X_ReadData(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pLength, uint8_t ** ppResponse,
	uint16_t * pRespLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t		PH_MEMLOC_REM bFinished = 0;
	uint8_t		PH_MEMLOC_REM bBufferResp = PH_ON;
    uint8_t     PH_MEMLOC_REM bLengthPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[20];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t *   PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint16_t	PH_MEMLOC_REM wOffset = 0;
	uint16_t	PH_MEMLOC_REM wBuffOption = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	/* Validate the parameter. */
    if (((bOption & 0xF0) != PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) &&
        ((bOption & 0xF0) != PHAL_MFPRIMENTAG_COMMUNICATION_ENC) &&
        ((bOption & 0xF0) != PHAL_MFPRIMENTAG_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

	if ((bOption & 0x0FU) != PH_EXCHANGE_DEFAULT)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

	/* Frame the Crypto information. */
	bCrypto = (uint8_t) (bOption & 0xF0);

	/* Frame Presence of length information in the command frame.
	 * The first three bytes specifies number of bytes to be received from PICC.
	 */
	bLengthPresent = (uint8_t) (((bOption & 0xF0) == PHAL_MFPRIMENTAG_COMMUNICATION_ENC) ? PH_ON : PH_OFF);

	/* Add Read Length if communication mode is Encrypted. */
	if(bLengthPresent)
	{
		memcpy(&aAppData[wAppDataLen], pLength, 3); /* PRQA S 3200 */
		wAppDataLen += 3;
	}

    aAppData[wAppDataLen++] = PHAL_MFPRIMENTAG_CMD_READ_DATA_ISO;
    aAppData[wAppDataLen++] = bFileNo;

    memcpy(&aAppData[wAppDataLen], pOffset, 3); /* PRQA S 3200 */
    wAppDataLen += 3;

    memcpy(&aAppData[wAppDataLen], pLength, 3); /* PRQA S 3200 */
    wAppDataLen += 3;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_Iso7816Wrap(
		bLengthPresent,
		3,
		PH_ON,
		((pLength[0] | (pLength[1] << 8) | (pLength[2] << 8)) + 7),
		aAppData,
		&wAppDataLen));

	/* Frame the Buffer Option. */
	wBuffOption = (uint16_t) (PH_EXCHANGE_DEFAULT | PHHAL_HW_CMD_SAMAV3_ISO_CHAINING);

	do
	{
		/* Exchange the information to Sam. */
		wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
			PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			wBuffOption,
			bCrypto,
			aAppData,
			(uint8_t) wAppDataLen,
			&pResponse,
			&wRespLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Update the Buffer Option. */
		wBuffOption = (uint16_t) (PH_EXCHANGE_RXCHAINING | PHHAL_HW_CMD_SAMAV3_ISO_CHAINING);

		/* End the loop. */
		if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			bFinished = PH_ON;
		}
		else
		{
			/* End the loop if the status is success or PICC has provided a Chainng status. */
			if(wStatus == PH_ERR_SUCCESS)
			{
				bFinished = PH_ON;
			}

			/* Copy the data. */
			*pRespLen += wRespLen;
			if(bBufferResp)
			{
				*ppResponse = pResponse;
				bBufferResp = PH_OFF;
			}
			else
			{
				memcpy(&ppResponse[0][wOffset], pResponse, wRespLen);	/* PRQA S 3200 */
			}
			wOffset += wRespLen;

			/* Update the HAL buffer start position. */
			if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
			{
				PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
					PHHAL_HW_CONFIG_RXBUFFER_STARTPOS, wOffset));
			}

			/* Reset the Application Length. */
			wAppDataLen = 0;
		}
	}while(!bFinished);

	/* Update the HAL buffer start position to 0. */
	PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SetConfig(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PHHAL_HW_CONFIG_RXBUFFER_STARTPOS, 0));

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_X_WriteData(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pData, uint8_t * pDataLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	uint16_t	PH_MEMLOC_REM wBufferOption = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[PHAL_MFPRIMENTAG_MAX_WRITE_LEN + 20];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM bIsFirstFrame = PH_ON;
	uint8_t		PH_MEMLOC_REM bDataLen = 0;
	uint8_t		PH_MEMLOC_REM bOffset = 0;
	uint32_t	PH_MEMLOC_REM dwTotLen = 0;
	uint32_t	PH_MEMLOC_REM dwRemLen = 0;
	uint8_t		PH_MEMLOC_REM bExchangeComplete = 0;

#ifdef RDR_LIB_PARAM_CHECK
	/* Validate the parameters */
	if ((bOption != PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) &&
		(bOption != PHAL_MFPRIMENTAG_COMMUNICATION_ENC) &&
		(bOption != PHAL_MFPRIMENTAG_COMMUNICATION_MACD))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}

	if ((bOption == PHAL_MFPRIMENTAG_COMMUNICATION_ENC) &&
		PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPRIMENTAG_NOT_AUTHENTICATED)
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPRIMENTAG);
	}
#endif

	/* Compute the maximum length. */
	dwTotLen = (uint32_t) (pDataLen[0] | (pDataLen[1] << 8) | (pDataLen[2] << 16));

	/* Update Offset information in case FULL Mode. */
	if(bOption == PHAL_MFPRIMENTAG_COMMUNICATION_ENC)
	{
		aAppData[wAppDataLen++] = 12;

		/* Set Length Presence flag. */
		bLenPresent = PH_ON;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFPRIMENTAG_CMD_WRITE_DATA_ISO;
	aAppData[wAppDataLen++] = bFileNo;

	memcpy(&aAppData[wAppDataLen], pOffset, 3);	/* PRQA S 3200 */
	wAppDataLen += 3;

	memcpy(&aAppData[wAppDataLen], pDataLen, 3);	/* PRQA S 3200 */
	wAppDataLen += 3;

	/* Set the buffering flag. */
	wBufferOption = (uint16_t) (PH_EXCHANGE_DEFAULT | PH_EXCHANGE_TXCHAINING);

	/* Set the lengths. */
	dwRemLen = dwTotLen;

	do
	{
		/* Compute the maximum data to be exchanged.  */
		if(bIsFirstFrame)
		{
			bDataLen = (uint8_t) (PHAL_MFPRIMENTAG_MAX_WRITE_LEN - 8) /* The command header information is available. */;
			bDataLen = (uint8_t) ((bOption == PHAL_MFPRIMENTAG_COMMUNICATION_ENC) ? (bDataLen - 4) : bDataLen);
			bDataLen = (uint8_t) ((bOption == PHAL_MFPRIMENTAG_COMMUNICATION_ENC) ? (bDataLen - 16) : (bDataLen - 20));
		}
		else
		{
			bDataLen = (uint8_t) (PHAL_MFPRIMENTAG_MAX_WRITE_LEN - 5);
		}

		/* Set the completion flag. */
		if(dwRemLen <= bDataLen)
		{
			bDataLen = (uint8_t) dwRemLen;
			wBufferOption = PH_EXCHANGE_DEFAULT;
			bExchangeComplete = PH_ON;
		}

		/* Copy the data to the buffer. */
		memcpy(&aAppData[wAppDataLen], &pData[bOffset], bDataLen);	/* PRQA S 3200 */
		wAppDataLen += bDataLen;

		/* Wrap the command if required. */
		if(bIsFirstFrame)
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_Iso7816Wrap(
				bLenPresent,
				1,
				PH_OFF,
				(dwTotLen + 7),
				aAppData,
				&wAppDataLen));
		}

		/* Frame the Crypto information. */
		bCrypto = (uint8_t) (bOption | PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_ISO_CHAINING);

		/* Set the Offset information for MAC and FULL mode. */
		if(bOption != PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN)
		{
			bCrypto |= (uint8_t) (bIsFirstFrame ? 8 : 0);
			bCrypto |= (uint8_t) (bCrypto + 4);

			/* Reset the Crypto to zero . */
			if(!bIsFirstFrame)
			{
				bCrypto = 0;
			}
		}

		/* Set ExtendedOffset for Full mode and ISO Chaining. */
		if((bOption == PHAL_MFPRIMENTAG_COMMUNICATION_ENC) && bIsFirstFrame)
		{
			/* Mask out the previously set offset. */
			bCrypto = (uint8_t) (bCrypto & 0xF0);

			/* Set the Extended Offset bit. */
			bCrypto |= PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET;
		}

		/* Clear the first frame and Length Presence flag. */
		bIsFirstFrame = PH_OFF;
		bLenPresent = PH_OFF;

		/* Enchange the information to SAM. */
		wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
			PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			wBufferOption,
			bCrypto,
			aAppData,
			(uint8_t) wAppDataLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Evaluate the response. */
		wStatus = phalMfPrimeNtag_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccRetCode);

		/* Return the status if not SUCCESS or SUCCESS_CHAINING. */
		if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			return wStatus;
		}

		/* Update length. */
		bOffset += bDataLen;
		dwRemLen = (uint32_t) (dwRemLen - bDataLen);

		/* Set the Chaining satus. */
		wAppDataLen = 0;

	}while(!bExchangeComplete);

    return wStatus;
}




/* MIFARE PRIME NTAG ISO7816-4 commands. --------------------------------------------------------------------------------------------- */
phStatus_t phalMfPrimeNtag_Sam_X_IsoSelectFile(void * pDataParams, uint8_t bOption, uint8_t bSelector, uint8_t * pFid, uint8_t * pDFname, uint8_t bDFnameLen,
	uint8_t	bExtendedLenApdu, uint8_t ** ppFCI, uint16_t * pFCILen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
	uint8_t		PH_MEMLOC_REM aAppData[25];
	uint16_t	PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint8_t		PH_MEMLOC_REM aFileId[3] = {'\0'};
	uint8_t		PH_MEMLOC_REM aPiccDfName[7] = {0xD2, 0x76, 0x00, 0x00, 0x85, 0x01, 0x00};

	/* Validate the parameters. */
#ifdef RDR_LIB_PARAM_CHECK
	if (bDFnameLen > 16)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}
	/* If User intends to select through DFName, check for validity of DFName passed */
	if((bSelector ==  PHAL_MFPRIMENTAG_SELECTOR_4) && (bDFnameLen == 0x00))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}
	if ((bOption != PHAL_MFPRIMENTAG_FCI_RETURNED) && (bOption != PHAL_MFPRIMENTAG_FCI_NOT_RETURNED))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}
	if((bSelector != PHAL_MFPRIMENTAG_SELECTOR_4) && (bSelector != PHAL_MFPRIMENTAG_SELECTOR_3) && (pFid == NULL))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}
#endif

	/* Frame the command. */
	aAppData[wAppDataLen++] = 0x00;									/* CLA */
	aAppData[wAppDataLen++] = PHAL_MFPRIMENTAG_CMD_ISO7816_SELECT_FILE; /* INS */
	aAppData[wAppDataLen++] = bSelector;							/* P1 */
	aAppData[wAppDataLen++] = bOption;								/* P2 */

	/* Append LC. */
	if(bSelector != PHAL_MFPRIMENTAG_SELECTOR_3)
	{
		if(bExtendedLenApdu)
		{
			aAppData[wAppDataLen++] = 0;
			aAppData[wAppDataLen++] = 0;
		}
	}

	/* Append the payload and LC. */
	if(bSelector == PHAL_MFPRIMENTAG_SELECTOR_4)
	{
		/* Append LC. */
		aAppData[wAppDataLen++] = bDFnameLen;

		memcpy(&aAppData[wAppDataLen], pDFname, bDFnameLen); /* PRQA S 3200 */
		wAppDataLen += bDFnameLen;
	}
	else if((bSelector ==  PHAL_MFPRIMENTAG_SELECTOR_0) || (bSelector == PHAL_MFPRIMENTAG_SELECTOR_1) ||
		(bSelector == PHAL_MFPRIMENTAG_SELECTOR_2))
	{
		/* Append LC. */
		aAppData[wAppDataLen++] = 2;

		/* Select MF, DF or EF, by file identifier
		 * Select child DF
		 * Select EF under the current DF, by file identifier
		 * Select parent DF of the current DF
		 */
		aFileId[1] = aAppData[wAppDataLen++] = pFid[1];
		aFileId[0] = aAppData[wAppDataLen++] = pFid[0];
		aFileId[2] = 0;
	}
	else if(bSelector == PHAL_MFPRIMENTAG_SELECTOR_3)
	{
		/* For Selector as 0x03, there is nothing to be done. */
	}
	else
	{
		aAppData[wAppDataLen++] = 0;
	}

	/* Append LE only if there is an LC. */
	if(bSelector != PHAL_MFPRIMENTAG_SELECTOR_3)
	{
		if(bExtendedLenApdu)
		{
			aAppData[wAppDataLen++] = 0;
		}
		aAppData[wAppDataLen++] = 0;
	}

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SamAV3_Cmd_X_ISO14443_4_Exchange(
		PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen));

	/* Manipulate the status. */
	wStatus = phalMfprimeNtag_Int_ComputeErrorResponse(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams),
		((pResponse[wRespLen - 2] << 8) | pResponse[wRespLen - 1]));

	/* Decrement the RespLen to remove the status. */
	wRespLen -= 2;

	if(wStatus == PH_ERR_SUCCESS)
	{
		/* Reset Authentication should not be targeted for elementary file selection using file ID */
		if(bSelector !=  PHAL_MFPRIMENTAG_SELECTOR_2)
		{
			/* Reset Authentication Status here */
			PH_CHECK_SUCCESS_FCT(wStatus1, phalMfPrimeNtag_Sam_X_ResetAuthentication(pDataParams));
		}

		/* once the selection Success, update the File Id to master data structure if the selection is done through AID */
		if((bSelector ==  PHAL_MFPRIMENTAG_SELECTOR_0) || (bSelector == PHAL_MFPRIMENTAG_SELECTOR_1) ||
			(bSelector == PHAL_MFPRIMENTAG_SELECTOR_2))
		{
			memcpy(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->aAid, aFileId, sizeof(aFileId));	/* PRQA S 3200 */
		}
		else if((bSelector ==  PHAL_MFPRIMENTAG_SELECTOR_4))
		{
			/* Update the file ID to all zeros if DF Name is of PICC. */
			if(memcmp( pDFname, aPiccDfName, 7) == 0)
			{
				aFileId[0] = 0x00;
				aFileId[1] = 0x00;
				aFileId[2] = 0x00;
			}
			else
			{
				aFileId[0] = 0xff;
				aFileId[1] = 0xff;
				aFileId[2] = 0xff;
			}

			memcpy(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->aAid, aFileId, sizeof(aFileId));	/* PRQA S 3200 */
		}
		else
		{
			/* Nothing for Secector 0x03. */
		}
	}
	else
	{
		return wStatus;
	}

	/* Copy the response to the buffer */
	*ppFCI = pResponse;
	*pFCILen = wRespLen;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t  phalMfPrimeNtag_Sam_X_IsoReadBinary(void * pDataParams, uint16_t wOption, uint8_t bOffset, uint8_t bSfid, uint32_t dwBytesToRead, uint8_t bExtendedLenApdu,
	uint8_t ** ppResponse, uint32_t * pBytesRead)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
	uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t		PH_MEMLOC_REM bFinished = 0;
	uint8_t		PH_MEMLOC_REM bBufferResp = PH_ON;
	uint8_t     PH_MEMLOC_REM aAppData[7];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t *   PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint16_t	PH_MEMLOC_REM wOffset = 0;
	uint16_t	PH_MEMLOC_REM wBuffOption = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

#ifdef RDR_LIB_PARAM_CHECK
	/* Validate the parameter. */
	if(bSfid & 0x80)
	{
		/* Short file id is supplied */
		if((bSfid & 0x7FU) > 0x1F)
		{
			/* Error condition */
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
	}
#endif

	if(wOption != PH_EXCHANGE_DEFAULT)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;

	/* Frame the command information based on the option. */
	aAppData[wAppDataLen++] = 0x00;
	aAppData[wAppDataLen++] = PHAL_MFPRIMENTAG_CMD_ISO7816_READ_BINARY;
	aAppData[wAppDataLen++] = bSfid;
	aAppData[wAppDataLen++] = bOffset;

	if(bExtendedLenApdu)
	{
		aAppData[wAppDataLen++] = (uint8_t) ((dwBytesToRead & 0x00FF0000) >> 16);
		aAppData[wAppDataLen++] = (uint8_t) ((dwBytesToRead & 0x0000FF00) >> 8);
	}
	aAppData[wAppDataLen++] = (uint8_t) (dwBytesToRead & 0x000000FF);

	/* Frame the Buffer Option. */
	wBuffOption = (uint16_t) (wOption | PHHAL_HW_CMD_SAMAV3_ISO_CHAINING);

	do
	{
		/* Exchange the information to Sam. */
		wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
			PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			wBuffOption,
			bCrypto,
			aAppData,
			(uint8_t) wAppDataLen,
			&pResponse,
			&wRespLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Update the Buffer Option. */
		wBuffOption = (uint16_t) (PH_EXCHANGE_RXCHAINING | PHHAL_HW_CMD_SAMAV3_ISO_CHAINING);

		/* End the loop. */
		if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			bFinished = PH_ON;
		}
		else
		{
			/* End the loop if the status is success or PICC has provided a Chainng status. */
			if(wStatus == PH_ERR_SUCCESS)
			{
				bFinished = PH_ON;
			}

			/* Copy the data. */
			*pBytesRead += wRespLen;
			if(bBufferResp)
			{
				*ppResponse = pResponse;
				bBufferResp = PH_OFF;
			}
			else
			{
				memcpy(&ppResponse[0][wOffset], pResponse, wRespLen);	/* PRQA S 3200 */
			}
			wOffset += wRespLen;

			/* Update the HAL buffer start position. */
			if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
			{
				PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
					PHHAL_HW_CONFIG_RXBUFFER_STARTPOS, wOffset));
			}

			/* Reset the Application Length. */
			wAppDataLen = 0;
		}
	}while(!bFinished);

	/* Update the HAL buffer start position to 0. */
	PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SetConfig(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PHHAL_HW_CONFIG_RXBUFFER_STARTPOS, 0));

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccRetCode));

    return wStatus;
}

phStatus_t  phalMfPrimeNtag_Sam_X_IsoUpdateBinary(void * pDataParams, uint8_t bOffset, uint8_t bSfid, uint8_t bExtendedLenApdu, uint8_t * pData, uint32_t dwDataLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t     PH_MEMLOC_REM aAppData[7];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t	PH_MEMLOC_REM wPiccErrCode = 0;
	uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

#ifdef RDR_LIB_PARAM_CHECK
	/* Validate the parameters */
	if(bSfid & 0x80)
	{
		/* Short file id is supplied */
		if((bSfid & 0x7FU) > 0x1F)
		{
			/* Error condition */
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
	}
#endif

	/* Frame the command information. */
	aAppData[wAppDataLen++] = 0x00;
	aAppData[wAppDataLen++] = PHAL_MFPRIMENTAG_CMD_ISO7816_UPDATE_BINARY;
	aAppData[wAppDataLen++] = bSfid;
	aAppData[wAppDataLen++] = bOffset;

	if(bExtendedLenApdu)
	{
		aAppData[wAppDataLen++] = (uint8_t) ((dwDataLen & 0x00FF0000) >> 16);
		aAppData[wAppDataLen++] = (uint8_t) ((dwDataLen & 0x0000FF00) >> 8);
	}
	aAppData[wAppDataLen++] = (uint8_t) (dwDataLen & 0x000000FF);

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_DESFire_WriteX(
		PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_BUFFER_FIRST,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		NULL,
		NULL));

	wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
		PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_BUFFER_LAST,
		bCrypto,
		pData,
		(uint8_t) dwDataLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

	/* Convert the PICC status and validate it. */
	if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_DESFIRE_GEN)
	{
		wPiccErrCode = (phStatus_t) ((aPiccRetCode[0] << 8) | aPiccRetCode[1]);
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfprimeNtag_Int_ComputeErrorResponse(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams), wPiccErrCode));
	}

	return wStatus;
}




/* MIFARE PRIME NTAG Originality Check functions. ------------------------------------------------------------------------------------ */
phStatus_t phalMfPrimeNtag_Sam_X_ReadSign(void * pDataParams, uint8_t bAddr, uint8_t ** pSignature)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t     PH_MEMLOC_REM bLengthPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[12];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_FULL;
	if(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPRIMENTAG_NOT_AUTHENTICATED)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	}

	/* Add Length information is Communication mode is FULL. */
	if(bCrypto == PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_FULL)
	{
		aAppData[wAppDataLen++] = 56;
		aAppData[wAppDataLen++] = 0;
		aAppData[wAppDataLen++] = 0;

		/* Set presence of Length information. */
		bLengthPresent = PH_ON;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFPRIMENTAG_CMD_READ_SIG;
	aAppData[wAppDataLen++] = bAddr;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_Iso7816Wrap(
		bLengthPresent,
		3,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		pSignature,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}




/* MIFARE PRIME NTAG Miscellaneous functions. ---------------------------------------------------------------------------------------- */
phStatus_t phalMfPrimeNtag_Sam_X_GetConfig(void * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    switch (wConfig)
    {
		case PHAL_MFPRIMENTAG_ADDITIONAL_INFO:
			*pValue = PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->wAdditionalInfo;
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_X_SetConfig(void * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    switch (wConfig)
    {
		case PHAL_MFPRIMENTAG_ADDITIONAL_INFO:
			PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->wAdditionalInfo = wValue;
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_X_ResetAuthentication(void * pDataParams)
{
	return phalMfPrimeNtag_Sam_X_Int_ResetAuthentication(pDataParams);
}

phStatus_t phalMfPrimeNtag_Sam_X_CalculateMACSDM(void * pDataParams, uint8_t bSdmOption, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer,
	uint16_t wDstKeyNo, uint16_t wDstKeyVer, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr, uint8_t * pInData, uint16_t wInDataLen,
	uint8_t * pMac)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t *	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Validate the key information. */
	if ((wSrcKeyNo > 0x7f) || (wSrcKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

	if (((wDstKeyNo < 0xE0) || (wDstKeyNo > 0xE3)) || (wDstKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

	/* Derive SDM MAC (KSesSDMFileReadMAC) session key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_X_Int_ComputeSDMSessionVectors(
		pDataParams,
		bSdmOption,
		wSrcKeyNo,
		wSrcKeyVer,
		wDstKeyNo,
		pUid,
		bUidLen,
		pSDMReadCtr));

	/* Perform Offline activation using Ram Key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(
		PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
		(uint8_t) wDstKeyNo,
		(uint8_t) wDstKeyVer,
		NULL,
		0));

	/* Exchange the Input information to SAM. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_GenerateMAC(
		PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PH_EXCHANGE_DEFAULT,
		PHHAL_HW_SAMAV3_TRUNCATION_MODE_MFP,
		pInData,
		(uint8_t) wInDataLen,
		&pResponse,
		&wRespLen));

	/* Copy the Mac to the parameter. */
	memcpy(pMac, pResponse, wRespLen);	/* PRQA S 3200 */

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_X */
