/*
 * Copyright 2018, 2021, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <ph_TypeDefs.h>
#include <phhalHw.h>
#include <phpalMifare.h>
#include <string.h>

#ifdef NXPBUILD__PHAL_MFDF_SAM_X

#include "../phalMfdf_Int.h"
#include "phalMfdf_Sam_X.h"
#include "phalMfdf_Sam_X_Int.h"
#include <phhalHw_SamAV2_Cmd.h>

#ifdef NXPBUILD__PHAL_MFP_SAMAV3_X
#include <phhalHw_SamAV3_Cmd.h>
#endif /* NXPBUILD__PHAL_MFP_SAMAV3_X */

phStatus_t phalMfdf_SamAV2_X_Init(phalMfdf_SamAV2_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, phhalHw_SamAV2_DataParams_t * pHalSamDataParams)
{
    /* data param check */
    if (sizeof(phalMfdf_SamAV2_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFDF);
    }

    PH_ASSERT_NULL_DATA_PARAM (pDataParams, PH_COMP_AL_MFDF);
    PH_ASSERT_NULL_PARAM (pHalSamDataParams, PH_COMP_AL_MFDF);

    /* init private data */
	pDataParams->wId                  = PH_COMP_AL_MFDF | PHAL_MFDF_SAMAV2_X_ID;
    pDataParams->pHalSamDataParams    = pHalSamDataParams;
    pDataParams->bAuthMode            = PHAL_MFDF_NOT_AUTHENTICATED;
    pDataParams->bKeyNo               = 0xFF;
    pDataParams->bWrappedMode         = 0x00;
    pDataParams->wAdditionalInfo      = 0x0000;

    /* Default selected application */
    memset(pDataParams->pAid, 0x00, 3); /* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

#ifdef NXPBUILD__PHAL_MFDF_SAMAV3_X
phStatus_t phalMfdf_SamAV3_X_Init(phalMfdf_SamAV3_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, phhalHw_SamAV3_DataParams_t * pHalSamDataParams)
{
    /* data param check */
    if (sizeof(phalMfdf_SamAV3_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFDF);
    }

    PH_ASSERT_NULL_DATA_PARAM (pDataParams, PH_COMP_AL_MFDF);
    PH_ASSERT_NULL_PARAM (pHalSamDataParams, PH_COMP_AL_MFDF);

    /* init private data */
	pDataParams->wId                  = PH_COMP_AL_MFDF | PHAL_MFDF_SAMAV3_X_ID;
    pDataParams->pHalSamDataParams    = pHalSamDataParams;
    pDataParams->bAuthMode            = PHAL_MFDF_NOT_AUTHENTICATED;
    pDataParams->bKeyNo               = 0xFF;
    pDataParams->bWrappedMode         = 0x00;
    pDataParams->wAdditionalInfo      = 0x0000;

    /* Default selected application */
    memset(pDataParams->pAid, 0x00, 3); /* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}
#endif /* NXPBUILD__PHAL_MFDF_SAMAV3_X */




phStatus_t phalMfdf_Sam_X_Authenticate(void * pDataParams, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bWrappedMode = 0;
    uint8_t     PH_MEMLOC_REM bISOMode = 0;
	uint8_t     PH_MEMLOC_REM bAuthMode = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];

    /* Check for valid card key number. */
    if (bKeyNoCard > 0x0d)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

    /* Check for valid SAM keystore number and version. */
    if ((wKeyNo > 0x7f) || (wKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Change for valid diversification options. */
	if ((wOption != PHAL_MFDF_NO_DIVERSIFICATION) &&
		(wOption != (PH_CRYPTOSYM_DIV_MODE_DESFIRE | PH_CRYPTOSYM_DIV_OPTION_2K3DES_HALF)) &&
		(wOption != (PH_CRYPTOSYM_DIV_MODE_DESFIRE | PH_CRYPTOSYM_DIV_OPTION_2K3DES_FULL)) &&
		(wOption != PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* Validate diversification input length. */
	if ((wOption != PHAL_MFDF_NO_DIVERSIFICATION) && ((bDivInputLen > 31 ) || (bDivInputLen < 1)))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* Clear the PICC wStatus code buffer. */
	memset(aPiccRetCode, 0x00, sizeof(aPiccRetCode));	/* PRQA S 3200 */

	/* Set Auth mode with diversification enabled. */
	bAuthMode = (uint8_t) ((bDivInputLen > 0) ? PH_ON : PH_OFF);

	/* Set Diversification flags.
	 * For AV1 compatibility mode key diversification methods, TDEA Key, diversified using one encryption round
	 */
    if (wOption == PH_CRYPTOSYM_DIV_OPTION_2K3DES_HALF)
    {
        bAuthMode = (uint8_t) (bAuthMode | 0x08);
	}

	/* Set Diversification flags.
	 * AV2 compatibility mode key diversification methods, 3TDEA, AES key
	 */
    if (wOption == PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS)
    {
        bAuthMode = (uint8_t) (bAuthMode | 0x10);
	}

    /* Rest the current authentication wStatus. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_ResetAuthStatus(pDataParams));

    /* Set Native or ISO7816 PICC command format. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetWrappedMode(pDataParams, &bWrappedMode));
	bISOMode = (uint8_t) (bWrappedMode ? 0x40 : 0x00);

	/* Exchange the information to SAM. */
	wStatus = PHHAL_HW_SAM_CMD_DESFIRE_AUTHENTICATE(
		pDataParams,
		bAuthMode,
		bISOMode,
		bKeyNoCard,
		(uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        pDivInput,
        bDivInputLen,
		aPiccRetCode);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccRetCode));

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_SetAuthMode(pDataParams, PHAL_MFDF_AUTHENTICATE));
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_SetKeyNo(pDataParams, bKeyNoCard));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_AuthenticateISO(void * pDataParams, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bWrappedMode = 0;
    uint8_t     PH_MEMLOC_REM bISOMode = 0;
	uint8_t     PH_MEMLOC_REM bAuthMode = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];

    /* Check for valid card key number. */
    if (bKeyNoCard > 0x0d)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

    /* Check for valid SAM keystore number and version. */
    if ((wKeyNo > 0x7f) || (wKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Change for valid diversification options. */
	if ((wOption != PHAL_MFDF_NO_DIVERSIFICATION) &&
		(wOption != (PH_CRYPTOSYM_DIV_MODE_DESFIRE | PH_CRYPTOSYM_DIV_OPTION_2K3DES_HALF)) &&
		(wOption != (PH_CRYPTOSYM_DIV_MODE_DESFIRE | PH_CRYPTOSYM_DIV_OPTION_2K3DES_FULL)) &&
		(wOption != PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* Validate diversification input length. */
	if ((wOption != PHAL_MFDF_NO_DIVERSIFICATION) && ((bDivInputLen > 31 ) || (bDivInputLen < 1)))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* Clear the PICC wStatus code buffer. */
	memset(aPiccRetCode, 0x00, sizeof(aPiccRetCode));	/* PRQA S 3200 */

	/* Set Auth mode with diversification enabled. */
	bAuthMode = (uint8_t) ((bDivInputLen > 1) ? PH_ON : PH_OFF);

	/* Set Diversification flags.
	 * For AV1 compatibility mode key diversification methods, TDEA Key, diversified using one encryption round
	 */
    if (wOption == PH_CRYPTOSYM_DIV_OPTION_2K3DES_HALF)
    {
        bAuthMode = (uint8_t) (bAuthMode | 0x08);
	}

	/* Set Diversification flags.
	 * AV2 compatibility mode key diversification methods, 3TDEA, AES key
	 */
    if (wOption == PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS)
    {
        bAuthMode = (uint8_t) (bAuthMode | 0x10);
	}

    /* Reset the current authentication wStatus. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_ResetAuthStatus(pDataParams));

    /* Set Native or ISO7816 PICC command format. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetWrappedMode(pDataParams, &bWrappedMode));
	bISOMode = (uint8_t) (bWrappedMode ? 0x40 : 0x00);

	/* Exchange the information to SAM. */
	wStatus = PHHAL_HW_SAM_CMD_DESFIRE_AUTHENTICATE(
		pDataParams,
		bAuthMode,
		bISOMode,
		bKeyNoCard,
		(uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        pDivInput,
        bDivInputLen,
		aPiccRetCode);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccRetCode));

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_SetAuthMode(pDataParams, PHAL_MFDF_AUTHENTICATEISO));
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_SetKeyNo(pDataParams, bKeyNoCard));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_AuthenticateAES(void * pDataParams, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bWrappedMode = 0;
    uint8_t     PH_MEMLOC_REM bISOMode = 0;
	uint8_t     PH_MEMLOC_REM bAuthMode = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];

    /* Check for valid card key number. */
    if (bKeyNoCard > 0x0d)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

    /* Check for valid SAM keystore number and version. */
    if ((wKeyNo > 0x7f) || (wKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Change for valid diversification options. */
	if ((wOption != PHAL_MFDF_NO_DIVERSIFICATION) &&
		(wOption != (PH_CRYPTOSYM_DIV_MODE_DESFIRE | PH_CRYPTOSYM_DIV_OPTION_2K3DES_FULL)) &&
		(wOption != PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* Validate diversification input length. */
	if ((wOption != PHAL_MFDF_NO_DIVERSIFICATION) && ((bDivInputLen > 31 ) || (bDivInputLen < 1)))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* Clear the PICC wStatus code buffer. */
	memset(aPiccRetCode, 0x00, sizeof(aPiccRetCode));	/* PRQA S 3200 */

	/* Set Auth mode with diversification enabled. */
	bAuthMode = (uint8_t) ((bDivInputLen > 1) ? PH_ON : PH_OFF);

	/* Set Diversification flags.
	 * AV2 compatibility mode key diversification methods, 3TDEA, AES key
	 */
    if (wOption == PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS)
    {
        bAuthMode = (uint8_t) (bAuthMode | 0x10);
	}

    /* Reset the current authentication wStatus. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_ResetAuthStatus(pDataParams));

    /* Set Native or ISO7816 PICC command format. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetWrappedMode(pDataParams, &bWrappedMode));
	bISOMode = (uint8_t) (bWrappedMode ? 0x40 : 0x00);

	/* Exchange the information to SAM. */
	wStatus = PHHAL_HW_SAM_CMD_DESFIRE_AUTHENTICATE(
		pDataParams,
		bAuthMode,
		bISOMode,
		bKeyNoCard,
		(uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        pDivInput,
        bDivInputLen,
		aPiccRetCode);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccRetCode));

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_SetAuthMode(pDataParams, PHAL_MFDF_AUTHENTICATEAES));
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_SetKeyNo(pDataParams, bKeyNoCard));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_ChangeKeySettings(void * pDataParams, uint8_t bKeySettings)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bAuthMode = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetAuthMode(pDataParams, &bAuthMode));
	if(bAuthMode == PHAL_MFDF_NOT_AUTHENTICATED)
	{
		return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_AL_MFDF);
	}

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_CHANGE_KEY_SETTINGS;
	aAppData[bAppDataLen++] = bKeySettings;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_WriteX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_ENC,
		aAppData,
		bAppDataLen,
		1,
		PH_OFF));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_GetKeySettings(void * pDataParams, uint8_t * pKeySetting)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[6];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_GET_KEY_SETTINGS;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_ReadX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		PH_OFF,
		0,
		&pResponse,
		&wRespLen));

	/* Copy the data to the parameter */
	memcpy(pKeySetting, pResponse, wRespLen);	/* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_ChangeKey(void * pDataParams, uint16_t wOption, uint16_t wOldKeyNo, uint16_t wOldKeyVer, uint16_t wNewKeyNo, uint16_t wNewKeyVer, uint8_t bKeyNoCard,
	uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bKeyCompMeth = 0;
    uint8_t     PH_MEMLOC_REM bCfg = 0;
    uint8_t     PH_MEMLOC_REM bAuthMode = 0;
    uint8_t     PH_MEMLOC_REM bKeyNo = 0;
    uint8_t     PH_MEMLOC_REM bWrappedMode = 0;
    uint8_t     PH_MEMLOC_REM bISOMode = 0;
	uint8_t		PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM aAppId_Act[3] = { 0x00, 0x00, 0x00 };
    uint8_t		PH_MEMLOC_REM aAppId_Exp[3] = { 0x00, 0x00, 0x00 };

	/* Only if seleted Aid is 0x000000. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetAid(pDataParams, aAppId_Act));
	if((memcmp(aAppId_Exp, aAppId_Act, 3) == 0) && ((bKeyNoCard & 0x3FU) == 0x00))
    {
		/* Only if seleted Aid is 0x000000, and card key number is X0, then
         * it is likely to be the PICC master key that has to be changed.
		 */
		if ((bKeyNoCard != 0x80) && (bKeyNoCard != 0x40) && (bKeyNoCard != 0x00))
		{
			/* Invalid card key number supplied */
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
		}
    }
    else
    {
        if (bKeyNoCard > 0x0D)
        {
            /* Invalid application key specified */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
        }
    }

    /* Check for valid SAM key number and version. */
    if ((wOldKeyNo > 0x7f) || (wOldKeyVer > 0xff) || (wNewKeyNo > 0x7f) || (wNewKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Check for diversification options. */
	if((wOption != PHAL_MFDF_NO_DIVERSIFICATION) && (wOption != PHAL_MFDF_CHGKEY_DIV_NEW_KEY) && (wOption != PHAL_MFDF_CHGKEY_DIV_OLD_KEY) &&
		(wOption != (PHAL_MFDF_CHGKEY_DIV_NEW_KEY | PHAL_MFDF_CHGKEY_DIV_METHOD_CMAC)) &&
		(wOption != (PHAL_MFDF_CHGKEY_DIV_NEW_KEY | PHAL_MFDF_CHGKEY_DIV_NEW_KEY_ONERND)) &&
		(wOption != (PHAL_MFDF_CHGKEY_DIV_OLD_KEY | PHAL_MFDF_CHGKEY_DIV_METHOD_CMAC)) &&
		(wOption != (PHAL_MFDF_CHGKEY_DIV_OLD_KEY | PHAL_MFDF_CHGKEY_DIV_OLD_KEY_ONERND)) &&
		(wOption != (PHAL_MFDF_CHGKEY_DIV_OLD_KEY | PHAL_MFDF_CHGKEY_DIV_OLD_KEY_ONERND | PHAL_MFDF_CHGKEY_DIV_NEW_KEY)) &&
		(wOption != (PHAL_MFDF_CHGKEY_DIV_NEW_KEY | PHAL_MFDF_CHGKEY_DIV_OLD_KEY | PHAL_MFDF_CHGKEY_DIV_METHOD_CMAC)) &&
		(wOption != (PHAL_MFDF_CHGKEY_DIV_NEW_KEY | PHAL_MFDF_CHGKEY_DIV_OLD_KEY | PHAL_MFDF_CHGKEY_DIV_NEW_KEY_ONERND | PHAL_MFDF_CHGKEY_DIV_OLD_KEY_ONERND)))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Clear the PICC wStatus code buffer. */
	memset(aPiccRetCode, 0x00, sizeof(aPiccRetCode));	/* PRQA S 3200 */

    /* Set Native or ISO7816 PICC command format. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetWrappedMode(pDataParams, &bWrappedMode));
	bISOMode = (uint8_t) (bWrappedMode ? 0x40 : 0x00);

    /* Set the key compilation method. */
	if(wOption == PHAL_MFDF_NO_DIVERSIFICATION)
	{
		bKeyCompMeth = 0x00;
	}
	else
	{
		/* Validate diversification input length. */
		if ( bDivInputLen > 31 )
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
		}

		/* Assin the option to local variable. */
		bKeyCompMeth = (uint8_t) wOption;
	}

    /* Desfire key number to be changed. */
	bCfg = (uint8_t) (0x0f & bKeyNoCard);

    /* Set if PICC targeted key equal to PICC authenticated key. */
	PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdf_Sam_X_Int_GetKeyNo(pDataParams, &bKeyNo));
	if ((bKeyNoCard & 0x3f) == bKeyNo)
    {
        bKeyCompMeth = (uint8_t) (bKeyCompMeth | 0x01);
    }

    /* Include the key type in the cryptogram for Master Key */
    if(memcmp(aAppId_Exp, aAppId_Act, 3) == 0)
    {
        bCfg = (uint8_t) (bCfg | 0x10);
    }

	/* Enable Iso7816 wrapping. */
    bCfg = (uint8_t) (bCfg | bISOMode);

	/* Exchange the command to PICC. */
	wStatus = PHHAL_HW_SAM_CMD_DESFIRE_CHANGEKEY(
		pDataParams,
		bKeyCompMeth,
		bCfg,
		(uint8_t)wOldKeyNo,
		(uint8_t)wOldKeyVer,
		(uint8_t)wNewKeyNo,
		(uint8_t)wNewKeyVer,
		pDivInput,
		bDivInputLen,
		aPiccRetCode);

    /* Evaluate the response. */
	wStatus = phalMfdf_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccRetCode);

	/* Reset the Auth state. */
	if(wStatus != PH_ERR_SUCCESS)
	{
		PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdf_Sam_X_Int_GetAuthMode(pDataParams, &bAuthMode));
		if(bAuthMode != PHAL_MFDF_AUTHENTICATE)
		{
			PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdf_Sam_X_ResetAuthStatus(pDataParams));
		}

		return wStatus;
	}
	else
	{
		/* Reset authentication wStatus only if the key authenticated with is changed. */
		if (bKeyNo == (bKeyNoCard & 0x3FU))
		{
			PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdf_Sam_X_ResetAuthStatus(pDataParams));
		}
	}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_GetKeyVersion(void * pDataParams, uint8_t bKeyNo, uint8_t * pKeyVersion)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	 /* Only if seleted Aid is 0x000000. */
	if (bKeyNo > 0x0D)
    {
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_GET_KEY_VERSION;
	aAppData[bAppDataLen++] = bKeyNo;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_ReadX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		PH_OFF,
		0,
		&pResponse,
		&wRespLen));

	/* Copy the data to the parameter */
	memcpy(pKeyVersion, pResponse, wRespLen);	/* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}




phStatus_t phalMfdf_Sam_X_CreateApplication(void * pDataParams, uint8_t bOption, uint8_t * pAid, uint8_t bKeySettings1, uint8_t bKeySettings2, uint8_t * pISOFileId,
	uint8_t * pISODFName, uint8_t bISODFNameLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[30];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;

	/* Check for valid ISO DFName */
    if ((bISODFNameLen > 16) || (bOption == 0x02) || (bOption > 0x03))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_CREATE_APPLN;

	/* Buffer Application identifier to the command frame. */
    memcpy(&aAppData[bAppDataLen], pAid, 3); /* PRQA S 3200 */
	bAppDataLen += 3;

	/* Buffer Key settings information to command frame. */
    aAppData[bAppDataLen++] = bKeySettings1;
    aAppData[bAppDataLen++] = bKeySettings2;

    /* Buffer ISO FileID to exchange buffer. */
    if ((bOption & 0x01 ) == 0x01)
    {
		aAppData[bAppDataLen++] = pISOFileId[0];
		aAppData[bAppDataLen++] = pISOFileId[1];
    }

    /* Buffer ISO DFName to exchange buffer. */
    if ((bOption & 0x02) == 0x02)
    {
		memcpy(&aAppData[bAppDataLen], pISODFName, bISODFNameLen); /* PRQA S 3200 */
		bAppDataLen += bISODFNameLen;
    }

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_WriteX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		0,
		PH_OFF));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_DeleteApplication(void * pDataParams, uint8_t * pAid)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[10];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_DELETE_APPLN;

	/* Buffer Application identifier to the command frame. */
    memcpy(&aAppData[bAppDataLen], pAid, 3); /* PRQA S 3200 */
	bAppDataLen += 3;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_WriteX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		0,
		PH_OFF));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_GetApplicationIDs(void * pDataParams, uint8_t * pAidBuff, uint8_t * pNumAIDs)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[6];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint16_t	PH_MEMLOC_REM wOffset = 0;

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_GET_APPLN_IDS;

	do
	{
		/* Exchange the information to Sam. */
		wStatus = phalMfdf_Sam_X_Int_ReadX(
			pDataParams,
			PH_EXCHANGE_DEFAULT,
			PHAL_MFDF_COMMUNICATION_PLAIN,
			aAppData,
			bAppDataLen,
			PH_OFF,
			0,
			&pResponse,
			&wRespLen);

		/* Validate the expected wStatus. */
		if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			return wStatus;
		}

		/* Copy the data to the parameter */
		memcpy(&pAidBuff[wOffset], pResponse, wRespLen); /* PRQA S 3200 */
		*pNumAIDs += (uint8_t) (wRespLen / 3);

		/* Update the offset. */
		wOffset += wRespLen;

		/* Update with chaining flag. */
		bAppDataLen = 0;
		aAppData[bAppDataLen++] = PHAL_MFDF_CMD_CHAINING;

	}while(wStatus != PH_ERR_SUCCESS);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_GetDFNames(void * pDataParams, uint8_t bOption, uint8_t * pDFBuffer, uint8_t * bSize)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bAuthMode = 0;
    uint8_t     PH_MEMLOC_REM aAppData[6];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Validate the parameter. */
	if (((bOption & 0x0FU) != PH_EXCHANGE_DEFAULT) && ((bOption & 0x0FU) != PH_EXCHANGE_RXCHAINING))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetAuthMode(pDataParams, &bAuthMode));
	if ((bAuthMode == PHAL_MFDF_AUTHENTICATEISO) ||
		(bAuthMode == PHAL_MFDF_AUTHENTICATEAES))
	{
		/*
		 * Should return, invalid scenario error. Card will be disabled
         * in case this command is sent in these modes.
		 */
		return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_AL_MFDF);
	}

	/* Frame the command information. */
	aAppData[bAppDataLen++] = (uint8_t) (((bOption & 0x0FU) == PH_EXCHANGE_DEFAULT) ? PHAL_MFDF_CMD_GET_DF_NAMES : PHAL_MFDF_RESP_ADDITIONAL_FRAME);

	/* Exchange the information to Sam. */
	wStatus = phalMfdf_Sam_X_Int_ReadX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		PH_OFF,
		0,
		&pResponse,
		&wRespLen);

    /* Evaluate the response. */
	if((wStatus == PH_ERR_SUCCESS) || ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING))
	{
		/* Copy the data to the parameter */
		memcpy(pDFBuffer, pResponse, wRespLen);	/* PRQA S 3200 */
		*bSize = (uint8_t) wRespLen;
	}

	return wStatus;
}

phStatus_t phalMfdf_Sam_X_SelectApplication(void * pDataParams, uint8_t * pAppId)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[10];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;

	/* Reset the Auth states. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_ResetAuthStatus(pDataParams));

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_SELECT_APPLN;

	/* Add the primary application identifier */
    memcpy(&aAppData[bAppDataLen], pAppId, 3); /* PRQA S 3200 */
    bAppDataLen += 3;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_WriteX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		0,
		PH_OFF));

	/* Copy the AID to the params. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_SetAid(pDataParams, pAppId));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_FormatPICC(void * pDataParams)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[6];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_FORMAT_PICC;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_ReadX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		PH_OFF,
		0,
		&pResponse,
		&wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_GetVersion(void * pDataParams, uint8_t * pVerInfo)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bAuthMode = 0;
	uint8_t		PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[7];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint8_t		PH_MEMLOC_REM bVerLen = 0;
	uint8_t		PH_MEMLOC_REM bFinished = 0;

	/* A MAC is expected in the response. SAM should know this */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetAuthMode(pDataParams, &bAuthMode));
	if ((bAuthMode == PHAL_MFDF_AUTHENTICATEISO) || (bAuthMode == PHAL_MFDF_AUTHENTICATEAES))
	{
		bCrypto = PHAL_MFDF_COMMUNICATION_MACD;
	}

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_GET_VERSION;

	do
	{
		/* Exchange the information to Sam. */
		wStatus = phalMfdf_Sam_X_Int_ReadX(
			pDataParams,
			PH_EXCHANGE_DEFAULT,
			bCrypto,
			aAppData,
			bAppDataLen,
			PH_OFF,
			0,
			&pResponse,
			&wRespLen);

		/* Updated Finished to end the loop. */
		if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS))
		{
			PH_CHECK_SUCCESS(wStatus);
		}
		else
		{
			/* Update Command frame. */
			bAppDataLen = 0;
			aAppData[bAppDataLen++] = 0xAF;

			/* Copy the response information to the parameter. */
			memcpy(&pVerInfo[bVerLen], pResponse, wRespLen); /* PRQA S 3200 */
			bVerLen = (uint8_t) (bVerLen + wRespLen);

			/* Update Finish flag. */
			if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
			{
				bFinished = 1;
			}
		}
	}while(!bFinished);

	/* Do a Set Config of ADDITIONAL_INFO to set  the length(wLength) of the Version string */
	phalMfdf_Sam_X_SetConfig(pDataParams, PHAL_MFDF_ADDITIONAL_INFO, bVerLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_FreeMem(void * pDataParams, uint8_t * pMemInfo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[6];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_FREE_MEM;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_ReadX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		PH_OFF,
		0,
		&pResponse,
		&wRespLen));

	/* Copy the data to the parameter */
	memcpy(pMemInfo, pResponse, wRespLen);	/* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_SetConfiguration(void * pDataParams, uint8_t bOption, uint8_t * pData, uint8_t bDataSize)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bAuthMode = 0;
    uint8_t     PH_MEMLOC_REM aAppData[35];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;

	/* Validate the parameters. */
	switch(bOption)
	{
		case 0x00:
			if (bDataSize > 1)
			{
				return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
			}
			break;

		case 0x01:
			/*  check for data length 24 byte key and 1 byte key version*/
			if (bDataSize > 25)
			{
				return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
			}
			break;

		case 0x02:
			/* This command is not supported in Sam X mode since
			it requires Padding method 2 before encryption */
			return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFDF);
			break;

		default:
			/* Invalid configuration option*/
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetAuthMode(pDataParams, &bAuthMode));
	if(bAuthMode == PHAL_MFDF_NOT_AUTHENTICATED)
	{
		return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_AL_MFDF);
	}

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_SET_CONFIG;
	aAppData[bAppDataLen++] = bOption;

    memcpy(&aAppData[bAppDataLen], pData, bDataSize); /* PRQA S 3200 */
	bAppDataLen = (uint16_t) (bAppDataLen + bDataSize);

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_WriteX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_ENC,
		aAppData,
		bAppDataLen,
		bDataSize,
		PH_OFF));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_GetCardUID(void * pDataParams, uint8_t * pUid)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bAuthMode = 0;
    uint8_t     PH_MEMLOC_REM aAppData[11];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetAuthMode(pDataParams, &bAuthMode));
	if(bAuthMode == PHAL_MFDF_NOT_AUTHENTICATED)
	{
		return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_AL_MFDF);
	}

	/* Append expected length to the command frame. */
	aAppData[bAppDataLen++] = 7;
	aAppData[bAppDataLen++] = 0;
	aAppData[bAppDataLen++] = 0;

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_GET_CARD_UID;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_ReadX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_ENC,
		aAppData,
		bAppDataLen,
		PH_ON,
		3,
		&pResponse,
		&wRespLen));

	/* Copy the response to the parameter. */
	memcpy(pUid, pResponse, wRespLen);	/* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}




phStatus_t phalMfdf_Sam_X_GetFileIDs(void * pDataParams, uint8_t * pFid, uint8_t * bNumFIDs)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[6];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_GET_FILE_IDS;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_ReadX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		PH_OFF,
		0,
		&pResponse,
		&wRespLen));

	/* Copy the data to the parameter */
	memcpy(pFid, pResponse, wRespLen); /* PRQA S 3200 */
	*bNumFIDs = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_GetISOFileIDs(void * pDataParams, uint8_t * pFidBuffer, uint8_t * bNumFIDs)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[6];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint8_t		PH_MEMLOC_REM bFinished = 0;

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_GET_ISO_FILE_IDS;

	/* Set the number of FileID variable. */
	*bNumFIDs = 0;

	do
	{
		/* Exchange the information to Sam. */
		wStatus = phalMfdf_Sam_X_Int_ReadX(
			pDataParams,
			PH_EXCHANGE_DEFAULT,
			PHAL_MFDF_COMMUNICATION_PLAIN,
			aAppData,
			bAppDataLen,
			PH_OFF,
			0,
			&pResponse,
			&wRespLen);

		if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS))
		{
			PH_CHECK_SUCCESS(wStatus);
		}
		else
		{
			/* Update Command frame. */
			bAppDataLen = 0;
			aAppData[bAppDataLen++] = 0xAF;

			/* Copy the data to the parameter */
			memcpy(&pFidBuffer[*bNumFIDs * 2], pResponse, wRespLen);	/* PRQA S 3200 */
			*bNumFIDs += (uint8_t) (wRespLen / 2);

			/* Update Finish flag. */
			if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
			{
				bFinished = 1;
			}
		}
	}while(!bFinished);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_GetFileSettings(void * pDataParams, uint8_t bFileNo, uint8_t * pFSBuffer, uint8_t * pBufferLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Validate the parameters. */
	if (bFileNo > 0x1f)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_GET_FILE_SETTINGS;
	aAppData[bAppDataLen++] = bFileNo;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_ReadX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		PH_OFF,
		0,
		&pResponse,
		&wRespLen));

	/* Copy the data to the parameter */
	memcpy(pFSBuffer, pResponse, wRespLen);	/* PRQA S 3200 */
	*pBufferLen = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_ChangeFileSettings(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t bFileOption, uint8_t * pAccessRights)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[24];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;

	/* Validate the parameters */
	if ( bFileNo > 0x1f)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

    if (((bFileOption & 0x3f) != PHAL_MFDF_COMMUNICATION_PLAIN) &&
        ((bFileOption & 0x3f) != PHAL_MFDF_COMMUNICATION_ENC) &&
        ((bFileOption & 0x3f) != PHAL_MFDF_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	if ((bOption != PHAL_MFDF_COMMUNICATION_PLAIN) &&
        (bOption != PHAL_MFDF_COMMUNICATION_ENC) &&
        (bOption != PHAL_MFDF_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_CHANGE_FILE_SETTINGS;
	aAppData[bAppDataLen++] = bFileNo;
	aAppData[bAppDataLen++]   = (uint8_t) ((bFileOption & 0x30) >> 4);

    /* Append access rights. */
	memcpy(&aAppData[bAppDataLen], pAccessRights, 2); /* PRQA S 3200 */
	bAppDataLen += 2;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_WriteX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		(uint8_t) (((bOption & 0xF0U) == PHAL_MFDF_COMMUNICATION_ENC) ? PHAL_MFDF_COMMUNICATION_ENC : PHAL_MFDF_COMMUNICATION_PLAIN),
		aAppData,
		bAppDataLen,
		(uint8_t) (((bOption & 0xF0U) == PHAL_MFDF_COMMUNICATION_ENC) ? 3 : 0),
		PH_OFF));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_CreateStdDataFile(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pISOFileId, uint8_t bFileOption, uint8_t * pAccessRights,
    uint8_t * pFileSize)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[15];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;

	/* Validate the parameters. */
	if ((bFileNo > 0x1f) || (bOption > 0x01))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

    if ((bFileOption != PHAL_MFDF_COMMUNICATION_PLAIN) &&
        (bFileOption != PHAL_MFDF_COMMUNICATION_ENC) &&
        (bFileOption != PHAL_MFDF_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_CREATE_STD_DATAFILE;
	aAppData[bAppDataLen++] = bFileNo;

	/* Append ISOFileID is available. */
    if (bOption == 0x01)
    {
		memcpy(&aAppData[bAppDataLen], pISOFileId, 2); /* PRQA S 3200 */
		bAppDataLen += 2;
    }

    /* Append communication settings */
    aAppData[bAppDataLen++] = (uint8_t) (bFileOption >> 4);

    /* Append access rights. */
	memcpy(&aAppData[bAppDataLen], pAccessRights, 2); /* PRQA S 3200 */
	bAppDataLen += 2;

    /* Append FileSize. */
    memcpy(&aAppData[bAppDataLen], pFileSize, 3); /* PRQA S 3200 */
    bAppDataLen += 3;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_WriteX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		0,
		PH_OFF));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_CreateBackupDataFile(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pISOFileId, uint8_t bFileOption, uint8_t *pAccessRights,
	uint8_t * pFileSize)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[15];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;

	/* Validate the parameters. */
	if ((bFileNo > 0x1f) || (bOption > 0x01))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

    if ((bFileOption != PHAL_MFDF_COMMUNICATION_PLAIN) &&
        (bFileOption != PHAL_MFDF_COMMUNICATION_ENC) &&
        (bFileOption != PHAL_MFDF_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_CREATE_BKUP_DATAFILE;
	aAppData[bAppDataLen++] = bFileNo;

	/* Append ISOFileID is available. */
    if (bOption == 0x01)
    {
        memcpy(&aAppData[bAppDataLen], pISOFileId, 2); /* PRQA S 3200 */
		bAppDataLen += 2;
    }

    /* Append communication settings */
    aAppData[bAppDataLen++] = (uint8_t) (bFileOption >> 4);

    /* Append access rights. */
	memcpy(&aAppData[bAppDataLen], pAccessRights, 2); /* PRQA S 3200 */
	bAppDataLen += 2;

    /* Append FileSize. */
    memcpy(&aAppData[bAppDataLen], pFileSize, 3); /* PRQA S 3200 */
    bAppDataLen += 3;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_WriteX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		0,
		PH_OFF));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_CreateValueFile(void * pDataParams, uint8_t bFileNo, uint8_t bFileOption, uint8_t * pAccessRights, uint8_t * pLowerLmit, uint8_t * pUpperLmit,
	uint8_t * pValue, uint8_t bLimitedCredit)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[23];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;

	/* Validate the parameters. */
	if (bFileNo > 0x1f)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

    if ((bFileOption != PHAL_MFDF_COMMUNICATION_PLAIN) &&
        (bFileOption != PHAL_MFDF_COMMUNICATION_ENC) &&
        (bFileOption != PHAL_MFDF_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_CREATE_VALUE_FILE;
	aAppData[bAppDataLen++] = bFileNo;

    /* Append communication settings */
    aAppData[bAppDataLen++] = (uint8_t) (bFileOption >> 4);

    /* Append access rights. */
	memcpy(&aAppData[bAppDataLen], pAccessRights, 2); /* PRQA S 3200 */
	bAppDataLen += 2;

    /* Append lower limit. */
    memcpy(&aAppData[bAppDataLen], pLowerLmit, 4); /* PRQA S 3200 */
    bAppDataLen += 4;

    /* Append upper limit. */
    memcpy(&aAppData[bAppDataLen], pUpperLmit, 4); /* PRQA S 3200 */
    bAppDataLen += 4;

    /* Append value. */
    memcpy(&aAppData[bAppDataLen], pValue, 4); /* PRQA S 3200 */
    bAppDataLen += 4;

	/* Append LimitedCreditEnabled information */
	aAppData[bAppDataLen++] = bLimitedCredit;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_WriteX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		0,
		PH_OFF));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_CreateLinearRecordFile(void * pDataParams, uint8_t bOption, uint8_t  bFileNo, uint8_t * pISOFileId, uint8_t bFileOption, uint8_t * pAccessRights,
	uint8_t * pRecordSize, uint8_t * pMaxNoOfRec)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[18];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;

	/* Validate the parameters. */
	if ((bFileNo > 0x1f) || (bOption > 0x01))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

    if ((bFileOption != PHAL_MFDF_COMMUNICATION_PLAIN) &&
        (bFileOption != PHAL_MFDF_COMMUNICATION_ENC) &&
        (bFileOption != PHAL_MFDF_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_CREATE_LINEAR_RECFILE;
	aAppData[bAppDataLen++] = bFileNo;

	/* Append ISOFileID is available. */
    if (bOption == 0x01)
    {
        memcpy(&aAppData[bAppDataLen], pISOFileId, 2); /* PRQA S 3200 */
		bAppDataLen += 2;
    }

    /* Append communication settings */
    aAppData[bAppDataLen++] = (uint8_t) (bFileOption >> 4);

    /* Append access rights. */
	memcpy(&aAppData[bAppDataLen], pAccessRights, 2); /* PRQA S 3200 */
	bAppDataLen += 2;

    /* Append record size. */
	memcpy(&aAppData[bAppDataLen], pRecordSize, 3); /* PRQA S 3200 */
    bAppDataLen += 3;

    /* Append maximum number of records. */
	memcpy(&aAppData[bAppDataLen], pMaxNoOfRec, 3); /* PRQA S 3200 */
    bAppDataLen += 3;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_WriteX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		0,
		PH_OFF));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_CreateCyclicRecordFile(void * pDataParams, uint8_t bOption, uint8_t  bFileNo, uint8_t * pISOFileId, uint8_t bFileOption, uint8_t * pAccessRights,
	uint8_t * pRecordSize, uint8_t * pMaxNoOfRec)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[18];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;

	/* Validate the parameters. */
	if ((bFileNo > 0x1f) || (bOption > 0x01))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

    if ((bFileOption != PHAL_MFDF_COMMUNICATION_PLAIN) &&
        (bFileOption != PHAL_MFDF_COMMUNICATION_ENC) &&
        (bFileOption != PHAL_MFDF_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_CREATE_CYCLIC_RECFILE;
	aAppData[bAppDataLen++] = bFileNo;

	/* Append ISOFileID is available. */
    if (bOption == 0x01)
    {
        memcpy(&aAppData[bAppDataLen], pISOFileId, 2); /* PRQA S 3200 */
		bAppDataLen += 2;
    }

    /* Append communication settings */
    aAppData[bAppDataLen++] = (uint8_t) (bFileOption >> 4);

    /* Append access rights. */
	memcpy(&aAppData[bAppDataLen], pAccessRights, 2); /* PRQA S 3200 */
	bAppDataLen += 2;

    /* Append record size. */
	memcpy(&aAppData[bAppDataLen], pRecordSize, 3); /* PRQA S 3200 */
    bAppDataLen += 3;

    /* Append maximum number of records. */
	memcpy(&aAppData[bAppDataLen], pMaxNoOfRec, 3); /* PRQA S 3200 */
    bAppDataLen += 3;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_WriteX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		0,
		PH_OFF));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_DeleteFile(void * pDataParams, uint8_t bFileNo)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;

	/* Validate the parameters. */
	if (bFileNo > 0x1f)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_DELETE_FILE;
	aAppData[bAppDataLen++] = bFileNo;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_WriteX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		0,
		PH_OFF));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}




phStatus_t phalMfdf_Sam_X_ReadData(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pLength, uint8_t ** ppResponse,
	uint16_t * pRespLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bLengthPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[17];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint16_t	PH_MEMLOC_REM wOption = 0;
	uint8_t		PH_MEMLOC_REM bWrappedMode = 0;

	/* Validate the parameter. */
	if (bFileNo > 0x1f)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

    if (((bOption & 0xF0) != PHAL_MFDF_COMMUNICATION_PLAIN) &&
        ((bOption & 0xF0) != PHAL_MFDF_COMMUNICATION_ENC) &&
        ((bOption & 0xF0) != PHAL_MFDF_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	if (((bOption & 0x0FU) != PH_EXCHANGE_DEFAULT) && ((bOption & 0x0FU) != PH_EXCHANGE_RXCHAINING))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information based on the option. */
    if ((bOption & 0x0FU) == PH_EXCHANGE_RXCHAINING)
    {
		/* Frame additional frame code only for Native chaining. */
		aAppData[bAppDataLen++] = PHAL_MFDF_RESP_ADDITIONAL_FRAME;

		/* Set the Option information. */
		wOption = (uint16_t) (bOption & 0x0F);

		PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetWrappedMode(pDataParams, &bWrappedMode));
		if((PH_GET_COMPID(pDataParams) == PHAL_MFDF_SAMAV2_X_ID) && bWrappedMode)
		{
			wOption = PH_EXCHANGE_DEFAULT;
		}
    }
    else
    {
		/* Frame Presence of length information in the command frame.
		 * The first three bytes specifies number of bytes to be received from PICC.
		 */
		bLengthPresent = (uint8_t) (((bOption & 0xF0) == PHAL_MFDF_COMMUNICATION_ENC) ? PH_ON : PH_OFF);

		/* Add Read Length if communication mode is Encrypted. */
		if(bLengthPresent)
		{
			memcpy(&aAppData[bAppDataLen], pLength, 3); /* PRQA S 3200 */
			bAppDataLen += 3;
		}

        aAppData[bAppDataLen++] = PHAL_MFDF_CMD_READ_DATA;
        aAppData[bAppDataLen++] = bFileNo;

        memcpy(&aAppData[bAppDataLen], pOffset, 3); /* PRQA S 3200 */
        bAppDataLen += 3;

        memcpy(&aAppData[bAppDataLen], pLength, 3); /* PRQA S 3200 */
        bAppDataLen += 3;
    }

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_ReadX(
		pDataParams,
		wOption,
		(uint8_t) (bOption & 0xF0),
		aAppData,
		bAppDataLen,
		bLengthPresent,
		3,
		ppResponse,
		pRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_WriteData(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pData, uint8_t * pDataLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bAuthMode = 0;
    uint8_t     PH_MEMLOC_REM bWrappedMode = 0;
	uint8_t		PH_MEMLOC_REM bDataOffset = 0;
	uint16_t	PH_MEMLOC_REM wBufferOption = 0;
    uint8_t     PH_MEMLOC_REM aAppData[PHAL_MFDF_MAX_WRITE_LEN];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint8_t		PH_MEMLOC_REM bIsFirstFrame = PH_ON;
	uint8_t		PH_MEMLOC_REM bDataLen = 0;
	uint8_t		PH_MEMLOC_REM bOffset = 0;
	uint32_t	PH_MEMLOC_REM dwRemLen = 0;
	uint8_t		PH_MEMLOC_REM bExchangeComplete = 0;

	/* Validate the parameters */
	if (bFileNo > 0x1f)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}
	if ((bOption != PHAL_MFDF_COMMUNICATION_PLAIN) &&
		(bOption != PHAL_MFDF_COMMUNICATION_ENC) &&
		(bOption != PHAL_MFDF_COMMUNICATION_MACD))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_WRITE_DATA;
	aAppData[bAppDataLen++] = bFileNo;

	memcpy(&aAppData[bAppDataLen], pOffset, 3);
	bAppDataLen += 3;

	memcpy(&aAppData[bAppDataLen], pDataLen, 3);
	bAppDataLen += 3;

	/* Set the buffering flag. */
	wBufferOption = (uint16_t) (PH_EXCHANGE_DEFAULT | PH_EXCHANGE_TXCHAINING);

	/* Set the lengths. */
	dwRemLen = (uint32_t) (pDataLen[0] | (pDataLen[1] << 8) | (pDataLen[2] << 16));

	/* Get the parameter info. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetAuthMode(pDataParams, &bAuthMode));
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetWrappedMode(pDataParams, &bWrappedMode));

	do
	{
		/* Compute the maximum data to be exchanged.  */
		if(bIsFirstFrame)
		{
			bDataLen = (uint8_t) (PHAL_MFDF_MAX_WRITE_LEN - 8) /* The command header information is available. */;
			bDataLen = (uint8_t) ((bOption == PHAL_MFDF_COMMUNICATION_ENC) ? (bDataLen - 4) : bDataLen);

			if(bWrappedMode)
			{
				bDataLen = (uint8_t) ((bOption == PHAL_MFDF_COMMUNICATION_ENC) ? (bDataLen - 16) : (bDataLen - 20));
			}
		}
		else
		{
			bDataLen = (uint8_t) (PHAL_MFDF_MAX_WRITE_LEN - 1);
			if(bOption == PHAL_MFDF_COMMUNICATION_ENC)
			{
				bDataLen = (uint8_t) ((bAuthMode == PHAL_MFDF_AUTHENTICATEAES) ?
										(bDataLen - 11) : (bDataLen - 3));
			}

			bDataLen = (uint8_t) (bWrappedMode ? 32 : bDataLen);
		}

		/* Set the completion flag. */
		if(dwRemLen <= bDataLen)
		{
			bDataLen = (uint8_t) dwRemLen;
			wBufferOption = PH_EXCHANGE_DEFAULT;
			bExchangeComplete = PH_ON;
		}

		/* Copy the data to the buffer. */
		memcpy(&aAppData[bAppDataLen], &pData[bOffset], bDataLen);
		bAppDataLen += bDataLen;

		/* Set the Offset information for MAC and FULL mode. */
		if(bOption != PHAL_MFDF_COMMUNICATION_PLAIN)
		{
			bDataOffset = (uint8_t) (bIsFirstFrame ? 8 : 1);
			bDataOffset |= (uint8_t) (bWrappedMode ? (bDataOffset + 4) : bDataOffset);
		}

		/* Clear the first frame and Length Presence flag. */
		bIsFirstFrame = PH_OFF;

		/* Exchange the information to Sam. */
		wStatus = phalMfdf_Sam_X_Int_WriteX(
			pDataParams,
			wBufferOption,
			bOption,
			aAppData,
			bAppDataLen,
			bDataOffset,
			PH_ON);

		/* Return the wStatus if not SUCCESS or SUCCESS_CHAINING. */
		if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			return wStatus;
		}

		/* Update length. */
		bOffset += bDataLen;
		dwRemLen = (uint32_t) (dwRemLen - bDataLen);

		/* Set the Chaining satus. */
		bAppDataLen = 0;
		aAppData[bAppDataLen++] = PHAL_MFDF_RESP_ADDITIONAL_FRAME;

	}while(!bExchangeComplete);

    return wStatus;
}

phStatus_t phalMfdf_Sam_X_GetValue(void * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bLengthPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[11];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Validate the parameters. */
	if (bFileNo > 0x1f)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	if ((bCommOption != PHAL_MFDF_COMMUNICATION_PLAIN) &&
        (bCommOption != PHAL_MFDF_COMMUNICATION_ENC) &&
        (bCommOption != PHAL_MFDF_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Append expected length to the command frame. */
	if(bCommOption == PHAL_MFDF_COMMUNICATION_ENC)
	{
		aAppData[bAppDataLen++] = 4;
		aAppData[bAppDataLen++] = 0;
		aAppData[bAppDataLen++] = 0;

		bLengthPresent = PH_ON;
	}

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_GET_VALUE;
	aAppData[bAppDataLen++] = bFileNo;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_ReadX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		bCommOption,
		aAppData,
		bAppDataLen,
		bLengthPresent,
		3,
		&pResponse,
		&wRespLen));

	 /* Copy the value. */
    memcpy(pValue, pResponse, wRespLen); /* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_Credit(void * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[12];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;

	/* Validate the parameters */
	if ((bFileNo & 0x3f) > 0x1f)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

    if (((bCommOption & 0x3f) != PHAL_MFDF_COMMUNICATION_PLAIN) &&
        ((bCommOption & 0x3f) != PHAL_MFDF_COMMUNICATION_ENC) &&
        ((bCommOption & 0x3f) != PHAL_MFDF_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_CREDIT;
	aAppData[bAppDataLen++] = bFileNo;

    /* Append data. */
	memcpy(&aAppData[bAppDataLen], pValue, 4); /* PRQA S 3200 */
	bAppDataLen += 4;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_WriteX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		bCommOption,
		aAppData,
		bAppDataLen,
		4,
		PH_OFF));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_Debit(void * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[12];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;

	/* Validate the parameters */
	if ((bFileNo & 0x3f) > 0x1f)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

    if (((bCommOption & 0x3f) != PHAL_MFDF_COMMUNICATION_PLAIN) &&
        ((bCommOption & 0x3f) != PHAL_MFDF_COMMUNICATION_ENC) &&
        ((bCommOption & 0x3f) != PHAL_MFDF_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_DEBIT;
	aAppData[bAppDataLen++] = bFileNo;

    /* Append data. */
	memcpy(&aAppData[bAppDataLen], pValue, 4); /* PRQA S 3200 */
	bAppDataLen += 4;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_WriteX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		bCommOption,
		aAppData,
		bAppDataLen,
		4,
		PH_OFF));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_LimitedCredit(void * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[12];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;

	/* Validate the parameters */
	if ((bFileNo & 0x3f) > 0x1f)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

    if (((bCommOption & 0x3f) != PHAL_MFDF_COMMUNICATION_PLAIN) &&
        ((bCommOption & 0x3f) != PHAL_MFDF_COMMUNICATION_ENC) &&
        ((bCommOption & 0x3f) != PHAL_MFDF_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_LIMITED_CREDIT;
	aAppData[bAppDataLen++] = bFileNo;

    /* Append data. */
	memcpy(&aAppData[bAppDataLen], pValue, 4); /* PRQA S 3200 */
	bAppDataLen += 4;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_WriteX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		bCommOption,
		aAppData,
		bAppDataLen,
		4,
		PH_OFF));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_ReadRecords(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pRecNo, uint8_t * pRecCount, uint8_t * pRecSize,
	uint8_t ** ppResponse, uint16_t * pRespLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bLengthPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[17];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint16_t	PH_MEMLOC_REM wOption = 0;
	uint8_t		PH_MEMLOC_REM bWrappedMode = 0;

	/* Validate the parameter. */
	if (bFileNo > 0x1f)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

    if (((bOption & 0xF0) != PHAL_MFDF_COMMUNICATION_PLAIN) &&
        ((bOption & 0xF0) != PHAL_MFDF_COMMUNICATION_ENC) &&
        ((bOption & 0xF0) != PHAL_MFDF_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	if (((bOption & 0x0FU) != PH_EXCHANGE_DEFAULT) && ((bOption & 0x0FU) != PH_EXCHANGE_RXCHAINING))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information based on the option. */
    if ((bOption & 0x0FU) == PH_EXCHANGE_RXCHAINING)
    {
		/* Frame additional frame code only for Native chaining. */
		aAppData[bAppDataLen++] = PHAL_MFDF_RESP_ADDITIONAL_FRAME;

		/* Set the Option information. */
		wOption = (uint16_t) (bOption & 0x0F);

		PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetWrappedMode(pDataParams, &bWrappedMode));
		if((PH_GET_COMPID(pDataParams) == PHAL_MFDF_SAMAV2_X_ID) && bWrappedMode)
		{
			wOption = PH_EXCHANGE_DEFAULT;
		}
    }
    else
    {
		/* Frame Presence of length information in the command frame.
		 * The first three bytes specifies number of bytes to be received from PICC.
		 */
		bLengthPresent = (uint8_t) (((bOption & 0xF0) == PHAL_MFDF_COMMUNICATION_ENC) ? PH_ON : PH_OFF);

		/* Add Read Length if communication mode is Encrypted. */
		if(bLengthPresent)
		{
			memcpy(&aAppData[bAppDataLen], pRecSize, 3); /* PRQA S 3200 */
			bAppDataLen += 3;
		}

        aAppData[bAppDataLen++] = PHAL_MFDF_CMD_READ_RECORDS;
        aAppData[bAppDataLen++] = bFileNo;

		memcpy(&aAppData[bAppDataLen], pRecNo, 3); /* PRQA S 3200 */
        bAppDataLen += 3;

		memcpy(&aAppData[bAppDataLen], pRecCount, 3); /* PRQA S 3200 */
        bAppDataLen += 3;
    }

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_ReadX(
		pDataParams,
		wOption,
		(uint8_t) (bOption & 0xF0),
		aAppData,
		bAppDataLen,
		bLengthPresent,
		3,
		ppResponse,
		pRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_WriteRecord(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pData, uint8_t * pDataLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bAuthMode = 0;
    uint8_t     PH_MEMLOC_REM bWrappedMode = 0;
	uint16_t	PH_MEMLOC_REM wBufferOption = 0;
	uint8_t		PH_MEMLOC_REM bDataOffset = 0;
    uint8_t     PH_MEMLOC_REM aAppData[PHAL_MFDF_MAX_WRITE_LEN + 20];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint8_t		PH_MEMLOC_REM bIsFirstFrame = PH_ON;
	uint8_t		PH_MEMLOC_REM bDataLen = 0;
	uint8_t		PH_MEMLOC_REM bOffset = 0;
	uint32_t	PH_MEMLOC_REM dwRemLen = 0;
	uint8_t		PH_MEMLOC_REM bExchangeComplete = 0;

	/* Validate the parameters */
	if (bFileNo > 0x1f)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}
	if ((bOption != PHAL_MFDF_COMMUNICATION_PLAIN) &&
		(bOption != PHAL_MFDF_COMMUNICATION_ENC) &&
		(bOption != PHAL_MFDF_COMMUNICATION_MACD))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_WRITE_RECORD;
	aAppData[bAppDataLen++] = bFileNo;

	memcpy(&aAppData[bAppDataLen], pOffset, 3);
	bAppDataLen += 3;

	memcpy(&aAppData[bAppDataLen], pDataLen, 3);
	bAppDataLen += 3;

	/* Set the buffering flag. */
	wBufferOption = (uint16_t) (PH_EXCHANGE_DEFAULT | PH_EXCHANGE_TXCHAINING);

	/* Set the lengths. */
	dwRemLen = (uint32_t) (pDataLen[0] | (pDataLen[1] << 8) | (pDataLen[2] << 16));

	/* Get the parameter info. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetAuthMode(pDataParams, &bAuthMode));
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetWrappedMode(pDataParams, &bWrappedMode));

	do
	{
		/* Compute the maximum data to be exchanged.  */
		if(bIsFirstFrame)
		{
			bDataLen = (uint8_t) (PHAL_MFDF_MAX_WRITE_LEN - 8) /* The command header information is available. */;
			bDataLen = (uint8_t) ((bOption == PHAL_MFDF_COMMUNICATION_ENC) ? (bDataLen - 4) : bDataLen);

			if(bWrappedMode)
			{
				bDataLen = (uint8_t) ((bOption == PHAL_MFDF_COMMUNICATION_ENC) ? (bDataLen - 16) : (bDataLen - 20));
			}
		}
		else
		{
			bDataLen = (uint8_t) (PHAL_MFDF_MAX_WRITE_LEN - 1);
			if(bOption == PHAL_MFDF_COMMUNICATION_ENC)
			{
				bDataLen = (uint8_t) ((bAuthMode == PHAL_MFDF_AUTHENTICATEAES) ?
										(bDataLen - 11) : (bDataLen - 3));
			}

			bDataLen = (uint8_t) (bWrappedMode ? 32 : bDataLen);
		}

		/* Set the completion flag. */
		if(dwRemLen <= PHAL_MFDF_MAX_WRITE_LEN)
		{
			bDataLen = (uint8_t) dwRemLen;
			wBufferOption = PH_EXCHANGE_DEFAULT;
			bExchangeComplete = PH_ON;
		}

		/* Copy the data to the buffer. */
		memcpy(&aAppData[bAppDataLen], &pData[bOffset], bDataLen);
		bAppDataLen += bDataLen;

		/* Set the Offset information for MAC and FULL mode. */
		if(bOption != PHAL_MFDF_COMMUNICATION_PLAIN)
		{
			bDataOffset = (uint8_t) (bIsFirstFrame ? 8 : 1);
			bDataOffset |= (uint8_t) (bWrappedMode ? (bDataOffset + 4) : bDataOffset);
		}

		/* Clear the first frame and Length Presence flag. */
		bIsFirstFrame = PH_OFF;

		/* Exchange the information to Sam. */
		wStatus = phalMfdf_Sam_X_Int_WriteX(
			pDataParams,
			wBufferOption,
			bOption,
			aAppData,
			bAppDataLen,
			bDataOffset,
			PH_ON);

		/* Return the wStatus if not SUCCESS or SUCCESS_CHAINING. */
		if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			return wStatus;
		}

		/* Update the offset and length. */
		bOffset += bDataLen;
		dwRemLen = (uint32_t) (dwRemLen - bDataLen);

		/* Set the Chaining satus. */
		bAppDataLen = 0;
		aAppData[bAppDataLen++] = PHAL_MFDF_RESP_ADDITIONAL_FRAME;

	}while(!bExchangeComplete);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_ClearRecordFile(void * pDataParams, uint8_t bFileNo)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;

	/* Validate the parameters. */
	if (bFileNo > 0x1f)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_CLEAR_RECORDS_FILE;
	aAppData[bAppDataLen++] = bFileNo;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_WriteX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		0,
		PH_OFF));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}




phStatus_t phalMfdf_Sam_X_CommitTransaction(void * pDataParams)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[6];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_COMMIT_TXN;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_ReadX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		PH_OFF,
		0,
		&pResponse,
		&wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_AbortTransaction(void * pDataParams)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[6];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Frame the command information. */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_ABORT_TXN;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_ReadX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PHAL_MFDF_COMMUNICATION_PLAIN,
		aAppData,
		bAppDataLen,
		PH_OFF,
		0,
		&pResponse,
		&wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}




phStatus_t phalMfdf_Sam_X_IsoSelectFile(void * pDataParams, uint8_t bOption, uint8_t bSelector, uint8_t * pFid, uint8_t * pDFname, uint8_t bDFnameLen,
	uint8_t ** ppFCI, uint16_t * pFCILen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
	uint8_t		PH_MEMLOC_REM aAppData[25];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint16_t	PH_MEMLOC_REM wPiccErrCode = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM aFileId[3] = {'\0'};

#ifndef NXPBUILD__PHAL_MFDF_SAMAV3_X
	/* Actuall these parametes are not used when Sam is AV2 HAL. When Package manager is build using only SAM AV2 HAL
	 * these parameter are shown as warning resulting in error in Release mode. So package manager to run smoothly,
	 * initializating the unused parameters.
	 */
	PHAL_MFDF_UNUSED_VARIABLE(bPiccRetCodeLen);
#endif /* NXPBUILD__PHAL_MFDF_SAMAV3_X */

	/* Validate the parameters. */
	if( (bDFnameLen > 16) || ((bOption != 0x00) && (bOption != 0x0C)))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	if(bSelector > 0x04)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* Clear the PICC wStatus code buffer. */
	memset(aPiccRetCode, 0x00, sizeof(aPiccRetCode));	/* PRQA S 3200 */

	/* Reset the Authentication. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_ResetAuthStatus(pDataParams));

	/* Frame the command. */
	aAppData[bAppDataLen++] = 0x00;									/* CLA */
	aAppData[bAppDataLen++] = PHAL_MFDF_CMD_ISO7816_SELECT_FILE; /* INS */
	aAppData[bAppDataLen++] = bSelector;							/* P1 */
	aAppData[bAppDataLen++] = bOption;								/* P2 */

	/* Append the payload and LC. */
	if(bSelector == 0x04)
	{
		/* Append LC. */
		aAppData[bAppDataLen++] = bDFnameLen;

		memcpy(&aAppData[bAppDataLen], pDFname, bDFnameLen); /* PRQA S 3200 */
		bAppDataLen += bDFnameLen;
	}
	else
	{
		/* Append LC. */
		aAppData[bAppDataLen++] = 2;

		/* Select MF, DF or EF, by file identifier
		 * Select child DF
		 * Select EF under the current DF, by file identifier
		 * Select parent DF of the current DF
		 */
		aFileId[1] = aAppData[bAppDataLen++] = pFid[1];
		aFileId[0] = aAppData[bAppDataLen++] = pFid[0];
		aFileId[2] = 0;
	}
	aAppData[bAppDataLen++] = 0;

	/* Exchange the information to Sam. */
	wStatus = PHHAL_HW_SAM_CMD_DESFIRE_READX(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		0x00 /* Crypto Mode as PLAIN. */,
		aAppData,
		bAppDataLen,
		ppFCI,
		pFCILen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Convert the PICC wStatus and validate it. */
	if((wStatus & PH_ERR_MASK) == (PH_ERR_CUSTOM_BEGIN + 23) /* PHHAL_HW_SAMAV2_ERR_DESFIRE_GEN */)
	{
		/* Get the PICC error code form HAL for SAM AV2. */
		if(PH_GET_COMPID(pDataParams) == PHAL_MFDF_SAMAV2_X_ID)
		{
			PH_CHECK_SUCCESS_FCT(wStatus1, PHHAL_HW_GET_CONFIG(
				pDataParams,
				PHHAL_HW_CONFIG_ADDITIONAL_INFO,
				&wPiccErrCode));
		}
		else
		{
			wPiccErrCode = (uint16_t) ((aPiccRetCode[0] << 8) | aPiccRetCode[1]);
		}

		PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdf_Int_ComputeErrorResponse(pDataParams, wPiccErrCode));
	}

	/* Enable Wrapped Mode. */
	if(wStatus == PH_ERR_SUCCESS)
	{
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_SetWrappedMode(pDataParams, PH_ON));
	}

	return wStatus;
}

phStatus_t phalMfdf_Sam_X_IsoReadBinary(void * pDataParams, uint16_t wOption, uint8_t bOffset, uint8_t bSfid, uint8_t bBytesToRead,
	uint8_t ** ppResponse, uint16_t * pBytesRead)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
    phStatus_t	PH_MEMLOC_REM wStatus1 = 0;
    uint8_t		PH_MEMLOC_REM aAppData[7];
    uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
    uint8_t *	PH_MEMLOC_REM pResponse = 0;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint16_t	PH_MEMLOC_REM wHalBufSize = 0;
    uint16_t	PH_MEMLOC_REM wNextPos = 0;
    uint8_t		PH_MEMLOC_REM bFlag = 1;
	uint16_t	PH_MEMLOC_REM wPiccErrCode = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
    uint8_t     PH_MEMLOC_REM bPiccRetCodeLen = 0;

#ifndef NXPBUILD__PHAL_MFDF_SAMAV3_X
	/* Actuall these parametes are not used when Sam is AV2 HAL. When Package manager is build using only SAM AV2 HAL
	 * these parameter are shown as warning resulting in error in Release mode. So package manager to run smoothly,
	 * initializating the unused parameters.
	 */
	PHAL_MFDF_UNUSED_VARIABLE(bPiccRetCodeLen);
#endif /* NXPBUILD__PHAL_MFDF_SAMAV3_X */

	/* Clear the PICC wStatus code buffer. */
	memset(aPiccRetCode, 0x00, sizeof(aPiccRetCode));	/* PRQA S 3200 */

	/* Initialize the buffer*/
	memset(aAppData, 0x00, 7); /* PRQA S 3200 */

	if (bSfid & 0x80)
	{
		/* Short file id is supplied */
		if ((bSfid & 0x7FU) > 0x1F)
		{
			/* Error condition */
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
		}
	}

	if (wOption == PH_EXCHANGE_DEFAULT)
	{
		aAppData[bAppDataLen++] = 0x00;
		aAppData[bAppDataLen++] = PHAL_MFDF_CMD_ISO7816_READ_BINARY;
		aAppData[bAppDataLen++] = bSfid;
		aAppData[bAppDataLen++] = bOffset;
		aAppData[bAppDataLen++] = bBytesToRead;
	}
	else if (wOption == PH_EXCHANGE_RXCHAINING)
	{
		aAppData[bAppDataLen++] = PHAL_MFDF_RESP_ADDITIONAL_FRAME;
		wOption = PH_EXCHANGE_RXCHAINING;
	}
	else
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
	}

	/* Get the HAL buffer size*/
	PH_CHECK_SUCCESS_FCT(wStatus1, PHHAL_HW_GET_CONFIG(
		pDataParams,
		PHHAL_HW_CONFIG_RXBUFFER_BUFSIZE,
		&wHalBufSize));

    do
    {
		wStatus = PHHAL_HW_SAM_CMD_DESFIRE_READX(
			pDataParams,
			wOption,
			0x00 /* Crypto Mode as PLAIN. */,
			aAppData,
			bAppDataLen,
			&pResponse,
			&wRespLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Validate the wStatus. */
		if (((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING) &&
			((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS))
		{
			if((wStatus & PH_ERR_MASK) == (PH_ERR_CUSTOM_BEGIN + 23) /* PHHAL_HW_SAMAV2_ERR_DESFIRE_GEN */)
			{
				/* Get the PICC error code form HAL for SAM AV2. */
				if(PH_GET_COMPID(pDataParams) == PHAL_MFDF_SAMAV2_X_ID)
				{
					PH_CHECK_SUCCESS_FCT(wStatus1, PHHAL_HW_GET_CONFIG(
						pDataParams,
						PHHAL_HW_CONFIG_ADDITIONAL_INFO,
						&wPiccErrCode));
				}
				else
				{
					wPiccErrCode = (uint16_t) ((aPiccRetCode[0] << 8) | aPiccRetCode[1]);
				}

				PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Int_ComputeErrorResponse(pDataParams, wPiccErrCode));
			}
			else
			{
				return wStatus;
			}
		}

		/* Copy the response to the parameter. */
		*pBytesRead = (uint16_t) (*pBytesRead + wRespLen);
		wNextPos = (uint16_t) (wNextPos + wRespLen);

		if (bFlag)
		{
			*ppResponse = pResponse;
			bFlag = 0;
		}

		/* Update the command information to receive more data from SAM. */
		if ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
		{
			wOption = PH_EXCHANGE_RXCHAINING;
			bAppDataLen = 0;
			aAppData[bAppDataLen++] = PHAL_MFDF_RESP_ADDITIONAL_FRAME;

			PH_CHECK_SUCCESS_FCT(wStatus1, PHHAL_HW_SET_CONFIG(pDataParams,
				PHHAL_HW_CONFIG_RXBUFFER_STARTPOS,
				wNextPos));

			if ((wNextPos + PHAL_MFDF_MAX_FRAME_SIZE) >= wHalBufSize)
			{
				PH_CHECK_SUCCESS_FCT(wStatus1, PHHAL_HW_SET_CONFIG(pDataParams,
					PHHAL_HW_CONFIG_RXBUFFER_STARTPOS,
					0x00));
				return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_AL_MFDF);
			}
		}
    }while ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING);

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_IsoUpdateBinary(void * pDataParams, uint8_t bOffset, uint8_t bSfid, uint8_t * pData, uint8_t bDataLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM aAppData[7];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint16_t	PH_MEMLOC_REM wPiccErrCode = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

#ifndef NXPBUILD__PHAL_MFDF_SAMAV3_X
	/* Actuall these parametes are not used when Sam is AV2 HAL. When Package manager is build using only SAM AV2 HAL
	 * these parameter are shown as warning resulting in error in Release mode. So package manager to run smoothly,
	 * initializating the unused parameters.
	 */
	PHAL_MFDF_UNUSED_VARIABLE(bPiccRetCodeLen);
#endif /* NXPBUILD__PHAL_MFDF_SAMAV3_X */

	/* Clear the PICC wStatus code buffer. */
	memset(aPiccRetCode, 0x00, sizeof(aPiccRetCode));	/* PRQA S 3200 */

	/* Validate the parameters */
	if (bSfid & 0x80)
    {
        /* Short file id is supplied */
        if ((bSfid & 0x7FU) > 0x1F)
        {
            /* Error condition */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
        }
    }

	/* Frame the command information. */
	aAppData[bAppDataLen++] = 0x00;
    aAppData[bAppDataLen++] = PHAL_MFDF_CMD_ISO7816_UPDATE_BINARY;
    aAppData[bAppDataLen++] = bSfid;
    aAppData[bAppDataLen++] = bOffset;
	aAppData[bAppDataLen++] = bDataLen;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, PHHAL_HW_SAM_CMD_DESFIRE_WRITEX(
		pDataParams,
		PH_EXCHANGE_BUFFER_FIRST,
		0x00 /* Crypto Mode as PLAIN. */,
		aAppData,
		bAppDataLen,
		NULL,
		NULL));

	wStatus = PHHAL_HW_SAM_CMD_DESFIRE_WRITEX(
		pDataParams,
		PH_EXCHANGE_BUFFER_LAST,
		0x00,
		pData,
		bDataLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Convert the PICC wStatus and validate it. */
	if((wStatus & PH_ERR_MASK) == (PH_ERR_CUSTOM_BEGIN + 23) /* PHHAL_HW_SAMAV2_ERR_DESFIRE_GEN */)
	{
		/* Get the PICC error code form HAL for SAM AV2. */
		if(PH_GET_COMPID(pDataParams) == PHAL_MFDF_SAMAV2_X_ID)
		{
			PH_CHECK_SUCCESS_FCT(wStatus1, PHHAL_HW_GET_CONFIG(
				pDataParams,
				PHHAL_HW_CONFIG_ADDITIONAL_INFO,
				&wPiccErrCode));
		}
		else
		{
			wPiccErrCode = (uint16_t) ((aPiccRetCode[0] << 8) | aPiccRetCode[1]);
		}

		PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdf_Int_ComputeErrorResponse(pDataParams, wPiccErrCode));
	}

    return wStatus;
}

phStatus_t phalMfdf_Sam_X_IsoReadRecords(void * pDataParams, uint16_t wOption, uint8_t bRecNo, uint8_t bReadAllFromP1, uint8_t bSfid, uint8_t bBytesToRead,
	uint8_t ** ppResponse, uint16_t * pBytesRead)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM aAppData[7];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint16_t	PH_MEMLOC_REM wPiccErrCode = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

#ifndef NXPBUILD__PHAL_MFDF_SAMAV3_X
	/* Actuall these parametes are not used when Sam is AV2 HAL. When Package manager is build using only SAM AV2 HAL
	 * these parameter are shown as warning resulting in error in Release mode. So package manager to run smoothly,
	 * initializating the unused parameters.
	 */
	PHAL_MFDF_UNUSED_VARIABLE(bPiccRetCodeLen);
#endif /* NXPBUILD__PHAL_MFDF_SAMAV3_X */

	/* Clear the PICC wStatus code buffer. */
	memset(aPiccRetCode, 0x00, sizeof(aPiccRetCode));	/* PRQA S 3200 */

	/* Validate the parameter. */
	if (wOption == PH_EXCHANGE_DEFAULT)
	{
		if (bSfid > 0x1F)
		{
			/* Error condition */
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
		}
	}

	if ((wOption != PH_EXCHANGE_DEFAULT) && (wOption != PH_EXCHANGE_RXCHAINING))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information based on the option. */
    if (wOption == PH_EXCHANGE_DEFAULT)
    {
		aAppData[bAppDataLen++] = 0x00;
        aAppData[bAppDataLen++] = PHAL_MFDF_CMD_ISO7816_READ_RECORDS;
        aAppData[bAppDataLen++] = bRecNo;
        aAppData[bAppDataLen++] = (uint8_t) ((bSfid <<= 3) | (bReadAllFromP1 ? 0x05 : 0x04));
		aAppData[bAppDataLen++] = bBytesToRead;
    }
	else
	{
		aAppData[bAppDataLen++] = PHAL_MFDF_RESP_ADDITIONAL_FRAME;
	}

	/* Exchange the information to Sam. */
	wStatus = PHHAL_HW_SAM_CMD_DESFIRE_READX(
		pDataParams,
		wOption,
		0x00 /* Crypto Mode as PLAIN. */,
		aAppData,
		bAppDataLen,
		ppResponse,
		pBytesRead,
		aPiccRetCode,
		&bPiccRetCodeLen);

	/* Update the chaining wStatus code with Desfire component. */
	if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
	{
		return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_AL_MFDF);
	}

    /* Convert the PICC wStatus and validate it. */
	if((wStatus & PH_ERR_MASK) == (PH_ERR_CUSTOM_BEGIN + 23) /* PHHAL_HW_SAMAV2_ERR_DESFIRE_GEN */)
	{
		/* Get the PICC error code form HAL for SAM AV2. */
		if(PH_GET_COMPID(pDataParams) == PHAL_MFDF_SAMAV2_X_ID)
		{
			PH_CHECK_SUCCESS_FCT(wStatus1, PHHAL_HW_GET_CONFIG(
				pDataParams,
				PHHAL_HW_CONFIG_ADDITIONAL_INFO,
				&wPiccErrCode));
		}
		else
		{
			wPiccErrCode = (uint16_t) ((aPiccRetCode[0] << 8) | aPiccRetCode[1]);
		}

		PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdf_Int_ComputeErrorResponse(pDataParams, wPiccErrCode));
	}

    return wStatus;
}

phStatus_t phalMfdf_Sam_X_IsoAppendRecord(void * pDataParams, uint8_t bSfid, uint8_t * pData, uint8_t bDataLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM aAppData[7];
	uint8_t		PH_MEMLOC_REM bAppDataLen = 0;
	uint16_t	PH_MEMLOC_REM wPiccErrCode = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

#ifndef NXPBUILD__PHAL_MFDF_SAMAV3_X
	/* Actuall these parametes are not used when Sam is AV2 HAL. When Package manager is build using only SAM AV2 HAL
	 * these parameter are shown as warning resulting in error in Release mode. So package manager to run smoothly,
	 * initializating the unused parameters.
	 */
	PHAL_MFDF_UNUSED_VARIABLE(bPiccRetCodeLen);
#endif /* NXPBUILD__PHAL_MFDF_SAMAV3_X */

	/* Clear the PICC wStatus code buffer. */
	memset(aPiccRetCode, 0x00, sizeof(aPiccRetCode));	/* PRQA S 3200 */

	/* Short file id is supplied */
    if ((bSfid & 0x7FU) > 0x1F)
    {
        /* Error condition */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

	/* Frame the command information. */
	aAppData[bAppDataLen++] = 0x00;
    aAppData[bAppDataLen++] = PHAL_MFDF_CMD_ISO7816_APPEND_RECORD;
    aAppData[bAppDataLen++] = 0x00;
    aAppData[bAppDataLen++] = (uint8_t) (bSfid << 3);
	aAppData[bAppDataLen++] = bDataLen;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, PHHAL_HW_SAM_CMD_DESFIRE_WRITEX(
		pDataParams,
		PH_EXCHANGE_BUFFER_FIRST,
		0x00 /* Crypto Mode as PLAIN. */,
		aAppData,
		bAppDataLen,
		NULL,
		NULL));

	wStatus = PHHAL_HW_SAM_CMD_DESFIRE_WRITEX(
		pDataParams,
		PH_EXCHANGE_BUFFER_LAST,
		0x00,
		pData,
		bDataLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Convert the PICC wStatus and validate it. */
	if((wStatus & PH_ERR_MASK) == (PH_ERR_CUSTOM_BEGIN + 23) /* PHHAL_HW_SAMAV2_ERR_DESFIRE_GEN */)
	{
		/* Get the PICC error code form HAL for SAM AV2. */
		if(PH_GET_COMPID(pDataParams) == PHAL_MFDF_SAMAV2_X_ID)
		{
			PH_CHECK_SUCCESS_FCT(wStatus1, PHHAL_HW_GET_CONFIG(
				pDataParams,
				PHHAL_HW_CONFIG_ADDITIONAL_INFO,
				&wPiccErrCode));
		}
		else
		{
			wPiccErrCode = (uint16_t) ((aPiccRetCode[0] << 8) | aPiccRetCode[1]);
		}

		PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdf_Int_ComputeErrorResponse(pDataParams, wPiccErrCode));
	}

    return wStatus;
}

phStatus_t phalMfdf_Sam_X_IsoGetChallenge(void * pDataParams, uint16_t wKeyNo, uint16_t wKeyVer, uint32_t dwLe, uint8_t * pRPICC1)
{
    pRPICC1 = NULL;
    PHAL_MFDF_UNUSED_VARIABLE(pDataParams);
    PHAL_MFDF_UNUSED_VARIABLE(wKeyNo);
    PHAL_MFDF_UNUSED_VARIABLE(wKeyVer);
    PHAL_MFDF_UNUSED_VARIABLE(dwLe);
    PHAL_MFDF_UNUSED_VARIABLE(pRPICC1);

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_IsoExternalAuthenticate(void * pDataParams, uint8_t * pDataIn, uint8_t bInputLen, uint8_t * pDataOut, uint8_t * pOutLen)
{
    PHAL_MFDF_UNUSED_VARIABLE(pDataParams);
    PHAL_MFDF_UNUSED_VARIABLE(pDataIn);
    PHAL_MFDF_UNUSED_VARIABLE(bInputLen);
    PHAL_MFDF_UNUSED_VARIABLE(pDataOut);
    PHAL_MFDF_UNUSED_VARIABLE(pOutLen);

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_IsoInternalAuthenticate(void * pDataParams, uint8_t * pDataIn, uint8_t bInputLen, uint8_t * pDataOut, uint8_t * pOutLen)
{
    PHAL_MFDF_UNUSED_VARIABLE(pDataParams);
    PHAL_MFDF_UNUSED_VARIABLE(pDataIn);
    PHAL_MFDF_UNUSED_VARIABLE(bInputLen);
    PHAL_MFDF_UNUSED_VARIABLE(pDataOut);
    PHAL_MFDF_UNUSED_VARIABLE(pOutLen);

	/* This command is available only in AV1 Mode */
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_IsoAuthenticate(void * pDataParams, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bKeyNoCard, uint8_t bIsPICCkey)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
	uint8_t		PH_MEMLOC_REM bOption = 0;
	uint16_t	PH_MEMLOC_REM wPiccErrCode = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];

	PH_UNUSED_VARIABLE(bIsPICCkey);

    /* Check for valid card key number. */
    if (bKeyNoCard > 0x0d)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

    /* Check for valid SAM keystore number and version. */
    if ((wKeyNo > 0x7f) || (wKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDF);
    }

    /* Unset the current authentication wStatus. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_ResetAuthStatus(pDataParams));

	/* Frame the Option parameter. Always by Key Entry numbner. */
	bOption = 0x00;

	/* Exchange the information to SAM. */
    wStatus = PHHAL_HW_SAM_CMD_DESFIRE_AUTHENTICATE(
		pDataParams,
        bOption,
        0x80,
        bKeyNoCard,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
		NULL,
		0,
		aPiccRetCode);

	/* Convert the PICC wStatus and validate it. */
	if((wStatus & PH_ERR_MASK) == (PH_ERR_CUSTOM_BEGIN + 23) /* PHHAL_HW_SAMAV2_ERR_DESFIRE_GEN */)
	{
		/* Get the PICC error code form HAL for SAM AV2. */
		if(PH_GET_COMPID(pDataParams) == PHAL_MFDF_SAMAV2_X_ID)
		{
			PH_CHECK_SUCCESS_FCT(wStatus1, PHHAL_HW_GET_CONFIG(
				pDataParams,
				PHHAL_HW_CONFIG_ADDITIONAL_INFO,
				&wPiccErrCode));
		}
		else
		{
			wPiccErrCode = (uint16_t) ((aPiccRetCode[0] << 8) | aPiccRetCode[1]);
		}

		PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdf_Int_ComputeErrorResponse(pDataParams, wPiccErrCode));
	}

	/* Update the data params */
	if(wStatus == PH_ERR_SUCCESS)
	{
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_SetAuthMode(pDataParams, PHAL_MFDF_AUTHENTICATEISO));
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_SetKeyNo(pDataParams, bKeyNoCard));
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_SetWrappedMode(pDataParams, PH_ON));
	}

	return wStatus;
}




phStatus_t phalMfdf_Sam_X_GetConfig(void * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bValue = 0;

    switch (wConfig)
    {
		case PHAL_MFDF_ADDITIONAL_INFO:
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetAdditionalInfo(pDataParams, pValue));
			break;

		case PHAL_MFDF_WRAPPED_MODE:
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_GetWrappedMode(pDataParams, &bValue));
			*pValue = bValue;
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFDF);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_SetConfig(void * pDataParams, uint16_t wConfig, uint16_t wValue)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;

    switch (wConfig)
    {
		case PHAL_MFDF_ADDITIONAL_INFO:
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_SetAdditionalInfo(pDataParams, wValue));
			break;

		case PHAL_MFDF_WRAPPED_MODE:
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_SetWrappedMode(pDataParams, (uint8_t) wValue));
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFDF);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

phStatus_t phalMfdf_Sam_X_ResetAuthStatus(void * pDataParams)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Reset the Authmode and Key number */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_SetAuthMode(pDataParams, PHAL_MFDF_NOT_AUTHENTICATED));
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdf_Sam_X_Int_SetKeyNo(pDataParams, 0xFF));

	/* Reset PICC Authentication. */
	PH_CHECK_SUCCESS_FCT(wStatus, PHHAL_HW_KILL_AUTHENTICATION(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDF);
}

#endif /* NXPBUILD__PHAL_MFDF_SAM_X */
