/******************************************************************************
*                                                  
*  (c) copyright Freescale Semiconductor China Ltd. 2008
*  ALL RIGHTS RESERVED
*
*  File Name: USB_Handle.C
*                                                                          
*  Description: This file is to handle USB module for SD Card Reader       			    
*           
*                                                                          
*  Assembler:  Codewarrior for HC(S)08 V6.0
*                                            
*  Version: 1.0                                                         
*                                                                                
*                                                                          
*  Author: Patrick Yang                              
*                                                                                       
*  Location: Shanghai, P.R.China                                              
*                                                                                                                  
*                                                  
* UPDATED HISTORY:
*
* REV   YYYY.MM.DD  AUTHOR        DESCRIPTION OF CHANGE
* ---   ----------  ------        --------------------- 
* 1.0   2008.01.09  Patrick Yang  Initial version
* 
*
******************************************************************************/                                                                        
/* Freescale  is  not  obligated  to  provide  any  support, upgrades or new */
/* releases  of  the Software. Freescale may make changes to the Software at */
/* any time, without any obligation to notify or provide updated versions of */
/* the  Software  to you. Freescale expressly disclaims any warranty for the */
/* Software.  The  Software is provided as is, without warranty of any kind, */
/* either  express  or  implied,  including, without limitation, the implied */
/* warranties  of  merchantability,  fitness  for  a  particular purpose, or */
/* non-infringement.  You  assume  the entire risk arising out of the use or */
/* performance of the Software, or any systems you design using the software */
/* (if  any).  Nothing  may  be construed as a warranty or representation by */
/* Freescale  that  the  Software  or  any derivative work developed with or */
/* incorporating  the  Software  will  be  free  from  infringement  of  the */
/* intellectual property rights of third parties. In no event will Freescale */
/* be  liable,  whether in contract, tort, or otherwise, for any incidental, */
/* special,  indirect, consequential or punitive damages, including, but not */
/* limited  to,  damages  for  any loss of use, loss of time, inconvenience, */
/* commercial loss, or lost profits, savings, or revenues to the full extent */
/* such  may be disclaimed by law. The Software is not fault tolerant and is */
/* not  designed,  manufactured  or  intended by Freescale for incorporation */
/* into  products intended for use or resale in on-line control equipment in */
/* hazardous, dangerous to life or potentially life-threatening environments */
/* requiring  fail-safe  performance,  such  as  in the operation of nuclear */
/* facilities,  aircraft  navigation  or  communication systems, air traffic */
/* control,  direct  life  support machines or weapons systems, in which the */
/* failure  of  products  could  lead  directly to death, personal injury or */
/* severe  physical  or  environmental  damage  (High  Risk Activities). You */
/* specifically  represent and warrant that you will not use the Software or */
/* any  derivative  work of the Software for High Risk Activities.           */
/* Freescale  and the Freescale logos are registered trademarks of Freescale */
/* Semiconductor Inc.                                                        */ 
/*****************************************************************************/


#include "derivative.h"
#include "USB_Handle.h"
#include "USB_Desc.h"
#include "SCSI_Process.h"
#include "SD.h"



/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/

byte vUSB_Dev_State;
byte vUSB_Ctl_State;


byte vEP0IN_DataCnt;
byte *pEP0IN_Data;

byte vEP1State= kUDATA1;
byte vEP2State= kUDATA0;

word vEP1Idx;
word vEP2Idx;

byte vEP1Data[512];
byte vEP2Data[512];  

word vEP1_Cnt;

tBDT EP0Bi@0x1860;
tBDT EP0Bo@0x1863;
tBDT EP1Bi@0x1866;
tBDT EP2Bo@0x1869;


byte EP0IN_Data@0x1880;
tUSB_Setup Setup_Pkt@0x1890;

byte EP1IN_Data@0x18A0;
byte EP2OUT_Data@0x18E0;


byte vUSB_Trf_State;
byte vCBWBuf_flag;

                
/************************************************************************************
*************************************************************************************
* Global memory declarations
*************************************************************************************
************************************************************************************/

extern byte vCBW_Buf[31];
extern byte vCSWResult; 



/************************************************************************************
*************************************************************************************
*                                    Funcitons                                      *
*************************************************************************************
************************************************************************************/


/*********************************************************
* Name: USB_InitBulk
*
* Desc: Initialize EP1 and EP2 as Bulk only endpoint
*       EP1- IN  EP2-OUT
* 
* Parameter: None
*
* Return: None
*            
**********************************************************/

void USB_InitBulk(void) 
{
  EPCTL1=0x05;                          // endpoint 1 for IN only
  EPCTL2=0x09;                          // endpoint 2 for OUT only
  
  EP1Bi.Stat._byte= kUDATA1;
  EP1Bi.Cnt = 0x00;
  EP1Bi.Addr = cEP1INBuffAddr;          // EP1 IN buffer;
  
  EP2Bo.Stat._byte = kUDATA0;
  EP2Bo.Cnt = cEP2_BUFF_SIZE;
  EP2Bo.Addr = cEP2OUTBuffAddr;         // EP2 OUT buffer;
  
  vEP2Idx = 0;
  
}
  


/*********************************************************
* Name: USB_Init
*
* Desc: Initialize USB module, enable USB module
*
* Parameter: None
*
* Return: None
*             
**********************************************************/

void USB_Init(void)
{
  
  USBCTL0 = 0x80;                     // reset USB modult
  
  vUSB_Dev_State = cPOWER;            // Device in power state
  while(INTSTAT_USBRSTF); 
  

  USBCTL0 = 0x45;                     // internal pullup, regulator and PHY enable
  //USBCTL0 = 0x01;                     // external Reg & pullup and PHY enable
    
  ERRSTAT = 0xFF;                     // clear USB error flag
  INTSTAT = 0xBE;                     // clear USB interrupt, except for reset flag
  ERRENB = 0xBF;                      // disable all USB error interrupt sources

  INTENB_USBRST=1;                    // enable USB reset interrupt  

  CTL = 0x01;                         // enable USB module
  
  EPCTL0 = 0x0D;                      // enable endpoint 0         
}




/*********************************************************
* Name: EP0_Stall
*
* Desc: Stall Endpoint0
*
* Parameter: None
*
* Return: None
*             
**********************************************************/

void EP0_Stall(void)
{

   EPCTL0_EPSTALL = 1;                // stall EP0
   EP0Bo.Stat._byte = kUDATA0;        // endpoint 0 OUT initialization
   EP0Bo.Cnt = cEP0_BUFF_SIZE;              
   
}


/*********************************************************
* Name: EP0_Load
*
* Desc: Load EP0 data to EP0 IN buffer
*
* Parameter: None
*
* Return: None
*             
**********************************************************/

void EP0_Load(void)
{    
    byte Byte_To_Send;
    byte *pDst;
    
    if(vEP0IN_DataCnt < cEP0_BUFF_SIZE)
        Byte_To_Send = vEP0IN_DataCnt;
    else
        Byte_To_Send = cEP0_BUFF_SIZE;
    
    EP0Bi.Cnt = Byte_To_Send;
    
    vEP0IN_DataCnt = vEP0IN_DataCnt - Byte_To_Send;
                  
    pDst=(byte *)&EP0IN_Data;
    
    while(Byte_To_Send)
    {
       *pDst = *pEP0IN_Data;
       pDst++;
       pEP0IN_Data++;
       Byte_To_Send--;
    }
      
}     



/*********************************************************
* Name: USB_StdReq_Handler
*
* Desc: Handle USB standard request
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/

void USB_StdReq_Handler(void)
{
      
   byte length;
   
   switch(Setup_Pkt.bRequest) 
   {
      // not support
      //case mGET_STATUS:
      //case mCLR_FEATURE:
      //case mSET_FEATURE:   
      //  break;
 
      
      case mSET_ADDRESS:
        vUSB_Dev_State=cADR_PENDING;
        break;
      
      case mGET_DESC:
        
        switch(Setup_Pkt.wValue_h) 
        {
          case mDEVICE:
            pEP0IN_Data = (byte*)&Device_Descriptor;
            vEP0IN_DataCnt = sizeof(Device_Descriptor);
            break;
          
          case mCONFIGURATION:
            pEP0IN_Data = (byte*)&Configuration_Descriptor;
            vEP0IN_DataCnt = sizeof(Configuration_Descriptor);
            break;
        
          case mSTRING:
            pEP0IN_Data = (byte*)String_Table[Setup_Pkt.wValue_l];
            vEP0IN_DataCnt = *pEP0IN_Data;
            break;
          
          default:
            EP0_Stall();
            break;  
        }
        
        break;
      
      case mSET_CONFIG:
        if(Setup_Pkt.wValue_h+Setup_Pkt.wValue_l) // value is not zero
        {
          vUSB_Dev_State = cCONFIGURE;
         
          USB_InitBulk();                         // init EP1 and EP2
            
        } 
        else
          vUSB_Dev_State = cADDRESS;
      
      break;
      
      
      default:
        EP0_Stall();                              // not support
        break;
          
   }

   if(Setup_Pkt.bmRequestType &0x80)              // Device to Host
    {
       
       length = Setup_Pkt.wLength_l;
              
  		 if(length < vEP0IN_DataCnt)
         vEP0IN_DataCnt = length;
  			
       EP0_Load();
       
       EP0Bo.Stat._byte = kUDATA0; 
      
    }
    else    
    {
       EP0Bi.Cnt = 0;                             // return zero IN data
       EP0Bo.Stat._byte = kUDATA1;
    }
          
}


/*********************************************************
* Name: USB_CLsReq_Handler
*
* Desc: Handle USB class request
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/

void USB_ClsReq_Handler(void)
{
  
   if(mGET_MAXLUN == Setup_Pkt.bRequest)
   {
      EP0IN_Data = 0x01;
      EP0Bi.Cnt = 0x01;
      EP0Bo.Stat._byte = kUDATA0;
   } 
   else
   {
      EP0_Stall();
   }
}
   



/*********************************************************
* Name: USB_EP0_IN_Handler
*
* Desc: Handle USB EP0 IN Token
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/

 void USB_EP0_IN_Handler(void)
{
     
    if(vUSB_Dev_State == cADR_PENDING) {
        ADDR = Setup_Pkt.wValue_l;
        vUSB_Dev_State = cADDRESS;
    }
           

    EP0_Load();
          
    if(EP0Bi.Stat.McuCtlBit.DATA == 0)
        EP0Bi.Stat._byte = kUDATA1;
    else
        EP0Bi.Stat._byte = kUDATA0;

}


/*********************************************************
* Name: USB_EP0_OUT_Handler
*
* Desc: Handle USB EP0 OUT Token
*       EP0 OUT token is always Data1 and zero count
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/

void USB_EP0_OUT_Handler(void)
{
  
    EP0Bo.Cnt = cEP0_BUFF_SIZE; 
    EP0Bo.Stat._byte = kUDATA0;
    EP0Bi.Stat._byte = kUDATA1;      
}
      


/*********************************************************
* Name: USB_Setup_Handler
*
* Desc: Handle USB SETUP token
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/

void USB_Setup_Handler(void)
{
       
   vEP0IN_DataCnt= 0;
   
   if(0x00 == (Setup_Pkt.bmRequestType & 0x60))
    USB_StdReq_Handler();
   
   else if(0x20 == (Setup_Pkt.bmRequestType & 0x60))
    USB_ClsReq_Handler();       
    
   else
    EP0_Stall();  
   
   
   EP0Bi.Stat._byte = kUDATA1;  
   EP0Bo.Cnt  = cEP0_BUFF_SIZE;
   
   
   CTL_TSUSPEND = 0;
}



/*********************************************************
* Name: EP1_Load
*
* Desc: Load data to Endpoint 1 buffer
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/
void EP1_Load(void)
{
    byte i,counter;
    byte *pBuffer;
     
    pBuffer=(byte *)&EP1IN_Data;
    
    if(vEP1_Cnt > cEP1_BUFF_SIZE)
      counter = cEP1_BUFF_SIZE;
    else
      counter = (byte)vEP1_Cnt;
    
    for(i=0;i<counter;i++,vEP1Idx++)
      pBuffer[i]=vEP1Data[vEP1Idx];
       
    
    EP1Bi.Cnt = counter;
    vEP1State ^=0x40;  
    EP1Bi.Stat._byte= vEP1State;
    
    vEP1_Cnt = vEP1_Cnt - counter;
    
}


/*********************************************************
* Name: USB_EP1_IN_Handler
*
* Desc: Handle USB Endpoint1 IN token
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/
void USB_EP1_IN_Handler(void)
{
   
   if( vUSB_Trf_State == cCSW) 
   {
      Send_CSW();
   } 
   
   else if(vUSB_Trf_State == cEP1Tx)
   {
     
     EP1_Load();
     if(vEP1_Cnt == 0)
    {
      vUSB_Trf_State=cUSBWait; 
    } 
    
   }
   
   else
   {
      EP1Bi.Cnt = 0;
      EP1Bi.Stat._byte= kMCU;  
   }
    
}


/*********************************************************
* Name: USB_EP2_OUT_Handler
*
* Desc: Handle USB Endpoint2 OUT token
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/ 
void USB_EP2_OUT_Handler(void)
{
    byte i;
    byte *pCBW_Pkt;
    
    pCBW_Pkt=(byte *)&EP2OUT_Data;
      

    
    if(EP2Bo.Cnt ==31)        // CBW should be 31 bytes
    {
      vUSB_Trf_State = cCBW;
      vCBWBuf_flag=1;   
      
      for(i=0;i<EP2Bo.Cnt;i++)
      {
          vCBW_Buf[i]= pCBW_Pkt[i];
          
      }
             
      vEP2State ^=0x40;
      EP2Bo.Stat._byte=vEP2State;
      EP2Bo.Cnt = cEP2_BUFF_SIZE;
    }
      
    else
    {
         
      for(i=0;i<EP2Bo.Cnt;i++,vEP2Idx++)
      {
         
         vEP2Data[vEP2Idx]= pCBW_Pkt[i];
      }
         
      if(vEP2Idx == 512)
      {
        vUSB_Trf_State = cEP2Rx;
        EP2Bo.Cnt = cEP2_BUFF_SIZE;
        vEP2State ^=0x40;
        EP2Bo.Stat._byte= kMCU;
      } 
      else
      {
          
        vEP2State ^=0x40;
        EP2Bo.Stat._byte=vEP2State;
        EP2Bo.Cnt = cEP2_BUFF_SIZE;
      }  
        
    }
}




/*********************************************************
* Name: USB_Handler
*
* Desc: Handle all EP token request
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/

void USB_Handler(void) 
{
  byte stat;
  
  stat=STAT&0xF8;
      
  if(stat == mEP0_OUT) 
  {
    if(EP0Bo.Stat.RecPid.PID == mSETUP_TOKEN)
      USB_Setup_Handler();
    else
      USB_EP0_OUT_Handler();
  } 
  
  else if (stat==mEP0_IN) {
    USB_EP0_IN_Handler();
   
  }
  
  else if (stat==mEP1_IN){
    USB_EP1_IN_Handler();
    
  }
  
  else if (stat==mEP2_OUT) {
    USB_EP2_OUT_Handler();
    
  }
  
}


/*********************************************************
* Name: USB_Reset_Handler
*
* Desc: Handle USB reset command
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/

void USB_Reset_Handler(void)
{
    ERRSTAT = 0xFF;                 // clear USB error flag
    INTSTAT = 0xBF;                 // clear USB interrupt
    ERRENB = 0xBF;                  // disable all USB error interrupt sources
    INTENB = 0x88;                  // enable stall and token    
    
    ADDR = 0x00;                    // reset to default address 
  
    //Below is initialize the BDT
    EP0Bi.Stat._byte = kUDATA1;     // endpoint 0 IN buffer initialization
    EP0Bi.Cnt = 0x00;                      
    EP0Bi.Addr = cEP0INBuffAddr;                     
  
    EP0Bo.Stat._byte = kUDATA0;     // endpoint 0 OUT initialization
    EP0Bo.Cnt = cEP0_BUFF_SIZE;              
    EP0Bo.Addr = cEP0OUTBuffAddr;   // (EP0 OUT buffer-0x1860) >> 2;    

    
    EPCTL0 = 0x0D;                  // enable endpoint 0
    
  
    vUSB_Dev_State = cDEFAULT;
   
}



/*********************************************************
* Name: USB_Stall_Handler
*
* Desc: Handle USB stall info
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/

void USB_Stall_Handler(void) 
{
    
    if(EPCTL0_EPSTALL == 1)
        EPCTL0_EPSTALL = 0;
    
    INTSTAT_STALLF = 1;
}

/*********************************************************
* Name: USB_ISR
*
* Desc: USB interrupt service routine
* 
* Parameter: None
*
* Return: None            
*
**********************************************************/

interrupt void USB_ISR(void) 
{
  
   if(INTSTAT_USBRSTF && INTENB_USBRST) 
    {
      
      USB_Reset_Handler();
    }
         
  if(INTSTAT_STALLF && INTENB_STALL )
      USB_Stall_Handler();
   
  
  if(INTSTAT_TOKDNEF && INTENB_TOKDNE)
    {
      USB_Handler();
        
      INTSTAT_TOKDNEF = 1;    
    }
  
}