/******************************************************************************								
*    S12ZVH128 Demo Code                                                      *								
*                                                                             *								
*       Created on: 05/06/2013                                                *								
*       Author: B34981                                                        *								
*       Version: 1.0                                                          *								
*                                                                             *								
*       Demo code for the TRK-S12ZVH128 board. This code exercises some of    *								
*    the new features of the S12ZVH128 microcontroller, specifically aimed    *								
*    for Instrument Cluster applications.                                     *								
*                                                                             *								
*       This application:                                                     *								
*       - Configures the bus clock to run at 32 MHz from a 4 MHz oscillator   *								
*       - Animates a 4x40 (160) segment LCD                                   *								
*       - Displays pot value (12-bit) on the LCD (Odometer section)           *								
*       - Plays a single tone when pressing SW4 (PT0)                         *								
*       - Sends the MCU to STOP when pressing SW3 (PT1)                       *				
*       - Wakes the MCU from STOP when pressing SW1 (PT3)                     *				
*       - Displays the time from the RTC module on the LCD                    *								
*       - Toggles the LEDs with the Timer interrupt at different              *								
*    frequencies.                                                             *								
******************************************************************************/								


#include <hidef.h> /* for EnableInterrupts macro */
#include "derivative.h" /* include peripheral declarations */

#include "CMPU.h"
#include "LCD.h"
#include "SSG.h"
#include "GPIO.h"
#include "RTC.h"
#include "ADC.h"
#include "TIM.h"

/**********************************************************************************************
* External objects
**********************************************************************************************/


/**********************************************************************************************
* Constants and macros
**********************************************************************************************/


/**********************************************************************************************
* Global variables
**********************************************************************************************/
UINT16 i ,j;
UINT16 Lim = 500;
UINT16 Led_Freq[4] = {25000, 30000, 35000, 40000};


/**********************************************************************************************
* Local types
**********************************************************************************************/


/**********************************************************************************************
* Local function prototypes
*********************************************************************************************/
void RTC_ISR(void);
void TIM_CH0_ISR(void);
void TIM_CH1_ISR(void);
void TIM_CH2_ISR(void);
void TIM_CH3_ISR(void);
void PORTT_ISR(void);

/**********************************************************************************************
* Local variables
**********************************************************************************************/


/**********************************************************************************************
* Local functions
**********************************************************************************************/


/**********************************************************************************************
* Global functions
**********************************************************************************************/

/***********************************************************************************************
* Main routine
************************************************************************************************/  
void main(void) {

  EnableInterrupts;
 
  PLL_Init();			/* Initialize PLL to drive 4MHz OSC to 32 MHz Bus Clock */ 
  LCD_Init();			/* Initialize the LCD module to drive a 40 x 4 Segment LCD */
  RTC_Init();			/* Initialize the Real Time Clock module */
  GPIO_Init();			/* Initialize the buttons as inputs and the LEDS as outputs */   
  SSG_Init();			/* Initialize the SSG Module with automatic GONG DECAY control */  
  ADC_Init();			/* Initialize the ADC module to read only one element on the list (12-bit) */
  TIM_Init();			/* Initialize the TIM module with four channels executing output compares at different frequencies */
    
  asm(andcc #0x7F);		//Enable STOP mode  
    
  for(;;) {
    __RESET_WATCHDOG();	/* feeds the dog */    

  	LCD_Animate();		/* Animates de LCD */
  	SetClk(RTCHRR, RTCMINR, RTCSECR);		// Set the Real Time Clock to the LCD	
  	
	SetOdo(ADC_Read());	/* Reads the ADC and displays it to the Odometer section on the LCD */
    
		for(i = 0; i < 0xFF; i++){				/* Delay */
			for(j = 0; j < Lim; j++){
			
			    if(BTN0)						/* Is button 0 pressed? */
			    {    
			    	SSG_PlaySingleTone();		/* Play a single tone */
			    }
			    
			    if(BTN1){
			    	asm{STOP};					/* Enter STOP mode */			    
			    }				
				
			}
		}	
	} 
}


/**********************************************************************************************
* Interrupt Routines
**********************************************************************************************/
interrupt VectorNumber_Vrtc void RTC_ISR(){
	RTCS1_SECF = 1;		    //Clear seconds interrupt flag
	SetClk(RTCHRR, RTCMINR, RTCSECR);		// Set the Real Time Clock to the LCD	
}

interrupt VectorNumber_Vtim0ch0 void TIM_CH0_ISR(){
	TIM0TFLG1_C0F = 1;						// Clear interrupt flag
	TIM0TC0 = TIM0TC0 + Led_Freq[0];
	LED0 = ~LED0;							//Toggle LED
}

interrupt VectorNumber_Vtim0ch1 void TIM_CH1_ISR(){
	TIM0TFLG1_C1F = 1;						// Clear interrupt flag
	TIM0TC1 = TIM0TC1 + Led_Freq[1];
	LED1 = ~LED1;							//Toggle LED
}

interrupt VectorNumber_Vtim0ch2 void TIM_CH2_ISR(){
	TIM0TFLG1_C2F = 1;						// Clear interrupt flag
	TIM0TC2 = TIM0TC2 + Led_Freq[2];	
	LED2 = ~LED2;							//Toggle LED
}

interrupt VectorNumber_Vtim0ch3 void TIM_CH3_ISR(){
	TIM0TFLG1_C3F = 1;						// Clear interrupt flag
	TIM0TC3 = TIM0TC3 + Led_Freq[3];	
	LED3 = ~LED3;							//Toggle LED
}

interrupt VectorNumber_Vportt void PORTT_ISR(){
	PIFT_PIFT3 = 1;							// Clear interrupt flag
	/* Just Wake Up */
}
