/*******************************************************************************
* Freescale Semiconductor Inc.
* (c) Copyright 2004-2005 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
********************************************************************************
Services performed by FREESCALE in this matter are performed AS IS and without 
any warranty. CUSTOMER retains the final decision relative to the total design 
and functionality of the end product. FREESCALE neither guarantees nor will be 
held liable by CUSTOMER for the success of this project.
FREESCALE DISCLAIMS ALL WARRANTIES, EXPRESSED, IMPLIED OR STATUTORY INCLUDING, 
BUT NOT LIMITED TO, IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR 
A PARTICULAR PURPOSE ON ANY HARDWARE, SOFTWARE OR ADVISE SUPPLIED TO THE PROJECT
BY FREESCALE, AND OR NAY PRODUCT RESULTING FROM FREESCALE SERVICES. IN NO EVENT
SHALL FREESCALE BE LIABLE FOR INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF 
THIS AGREEMENT.

CUSTOMER agrees to hold FREESCALE harmless against any and all claims demands or 
actions by anyone on account of any damage, or injury, whether commercial, 
contractual, or tortuous, rising directly or indirectly as a result of an advise
or assistance supplied CUSTOMER in connection with product, services or goods 
supplied under this Agreement.
********************************************************************************
* File      main.c
* Version   1.0
* Classification   General Business Information
* Brief     ADC single conversion and usage of GPIO on port AD
********************************************************************************
* Detailed Description:
*
* Setting of ADC and starting of single conversion.
* Configuration of GPIO on port AD, usage of GPIO. 
*     
* - tested on: EVB9S12XEP100
*	- OSCCLK = 16MHz, BUSCLK = 8MHz
*	- Reference to documentation: MC9S12XEP100RMV1 Rev. 1.19
********************************************************************************


#include <hidef.h>      /* common defines and macros */
#include "derivative.h"      /* derivative-specific definitions */
#include <MC9S12XEP100.h>     /* derivative information */

unsigned int ADC_avrg = 0;
unsigned int ADC_POT = 0;
unsigned int direction = 0;
unsigned int LED = 1, DT_LED = 0, DT_LEDaux = 0;

/*******************************************************************************
* Local function prototypes
*******************************************************************************/
static void ADC_Init(void);
static void PWM_Init(void);
static void PWM_RGB(int a);
unsigned int ADC_Read(void);
static void PLL_Init(unsigned char pll_synr, unsigned char pll_refdv, unsigned char pll_postdiv);
void delay(long int time);          



#define LED_UP 0
#define LED_DOWN 1  
#define LED_MIN 1      


/*******************************************************************************
Function Name : ADC_Init
Parameters    : NONE
Modifies      : NONE
Returns       : NONE
Notes         : Initialization of the ADC. 
Issues        : NONE
*******************************************************************************/
void ADC_Init(void)
{

  ATD0CTL1 = 0B00000000;
    //Channel AN00 will be used as analog input  
  
  ATD0CTL1 = 0B00010000;
    //ETRIGSEL - External Trigger Source Select
    //SRES[1:0] - A/D Resolution Select 8bits
    //SMP_DIS - Discharge Before Sampling Bit
    //ETRIGCH[3:0] - External Trigger Channel Select   -
  

  ATD0CTL2 = 0B01000000;
    //Reserved
    //AFFC - ATD Fast Flag Clear All
    //ICLKSTP - Internal Clock in Stop Mode Bit
    //ETRIGLE - External Trigger Level/Edge Control
    //ETRIGP - External Trigger Polarity
    //ETRIGE - External Trigger Mode Enable
    //ASCIE - ATD Sequence Complete Interrupt Enable
    //ACMPIE - ATD Compare Interrupt Enable
  
  ATD0CTL3 = 0B10100010;
    //DJM - Result Register Data Justification (Right Justified)
    //S8C, S4C, S2C, S1C - Converison Sequence Length (4)
    //FIFO - Result Register FIFO Mode (0)
    //FRZ[1:0] - Background Debug Freeze Enable (Finish current converison, then freeze)
  
  ATD0CTL4 = 0B01000011;
    //SMP[2:0] - Sample Time Select (8)
    //PRS[4:0] - ATD Clock Prescaler (ATD clock 1MHz)   
    
    
    
  /****************************************************************************/
  //Let's set AN06 as digital input and channel AN07 as digital output:
  
  //AN06:
  //PER1AD0_PER1AD06 = 1; //pull up resistor enabled
  //DDR1AD0_DDR1AD06 = 0; //set channel AN06 as input

  //LEDRED:  
  //DDRP_DDRP3 = 1; //set channel PP3 as ouput    
  //PTP_PTP3 = 1;   

}


/*******************************************************************************
Function Name : PWM_Init
Parameters    : NONE
Modifies      : NONE
Returns       : NONE
Notes         : .
Issues        : NONE
*******************************************************************************/

void PWM_Init(void){
    /*************************/
  PWMCAE = 0x00; /* Set LEFT alignment for channels 0 - 3 */
  PWMPOL = 0x00; /* Set polarity for channels 3/4/6         */
  /**************************
    The above register setting should match the following truth table.
    PWM0: Left   Aligned=0, Polarity=0
    PWM1: Left   Aligned=0, Polarity=1
    PWM2: Center Aligned=1, Polarity=0
    PWM3: Center Aligned=1, Polarity=1
  ***************************/  
  
  /* PWM3 clk selection */
  PWMCLK_PCLK3     = 0;      /* selecet B as source clk */

  /* PWM4 clk selection */
  PWMCLK_PCLK4     = 0;      /* selecet A as source clk */

  /* PWM6 clk selection */
  PWMCLK_PCLK6     = 0;      /* selecet B as source clk */
  
  PWMPRCLK = 0x77;           // Bus clock [A & B] / 32
  
  PWMPER3 = 0xFF; /* Set chan 0 period length to the maximum.        */
  PWMPER4 = 0xFF; /* Set chan 1 period length to the maximum.        */
  PWMPER6 = 0xFF; /* Set chan 2 period length to the maximum.        */


  PWMDTY3 = 0x00; /* Set chan 0 duty cycle to half the maximum, 50%. */
  PWMDTY4 = 0x00; /* Set chan 1 duty cycle to half the maximum, 50%. */
  PWMDTY6 = 0x00; /* Set chan 2 duty cycle to half the maximum, 50%. */

  PWME = 0x58;    /* Enable PWM channels 0 - 3 (PP0-PP3) */

}

  
  



/*******************************************************************************
Function Name : PLL_Init
Parameters    : unsigned char pll_synr
                unsigned char pll_refdv
                unsigned char pll_postdivNONE
Modifies      : NONE
Returns       : NONE
Notes         : Sets BUSCLK = 40MHz from OSCCLK = 4MHz
Issues        : NONE
*******************************************************************************/
static void PLL_Init(unsigned char pll_synr, unsigned char pll_refdv, unsigned char pll_postdiv){
  PLLCTL = 0B00000001;    // CME=0,PLLON=0,FM1=0,FM2=0,FSTWKP=0,PRE=0,PCE=0,
                          // SCME=1
  CLKSEL = 0B00000011;    // PLLSEL=0,PSTP=0,PLLWAI=0,RTIWAI=1,COPWAI=1
  SYNR = pll_synr;        // Set the multiplier register [3]
  REFDV = pll_refdv;      // Set the divider register [1]
  POSTDIV = pll_postdiv;  // Set the post divider register [1]
  PLLCTL_PLLON = 1;       // Enable the Phase Lock Loop
  while(!CRGFLG_LOCK);    // Wait till the PLL VCO is within tolerance
  CLKSEL_PLLSEL = 1;      // Select clock source from PLLCLK
  
  ECLKCTL_NECLK=0;        // Enable the BusClk output at ECLK pin to see busclk
                          // if necessary
}



/***************************************************************************************
Function Name	: delay
Parameters	  	: time	
Returns		    : -	
Notes         	: Total delay time = (( 3 + 4 + (9.5 * count)) * Bus Clk time)
****************************************************************************************/

void delay (long int time)		// 3 cycles
{while(time>>0){--time;}			// Loop takes 9.5 cycles
}									            // 4 cycles


/***************************************************************************************
Function Name	: ADC_read
Parameters	  	: time	
Returns		    : -	
Notes         	: Total delay time = (( 3 + 4 + (9.5 * count)) * Bus Clk time)
****************************************************************************************/

unsigned int ADC_Read(void){
      ATD0CTL5 = 0B00100000;//start single conversion
      //Reserved
      //SC - Special Channel Conversion Bit 
      //SCAN - Continuous or Single conversion sequence (Single)
      //MULT - Multi-Channel Sample Mode (one channel)
      //CD, CC, CB, CA - Analog Input Channel Select Code (channel AN0)
           
    while(ATD0STAT0_SCF == 0);   //wait for end of conversion
    
    //we can do average of all four values 
    ADC_avrg = (ATD0DR0L+ATD0DR1L+ATD0DR2L+ATD0DR3L)/4;	
    return ADC_avrg;
}									           


/*******************************************************************************
Function Name : PWM_RGB
Parameters    : NONE
Modifies      : NONE
Returns       : NONE
Notes         : .
Issues        : NONE
*******************************************************************************/

void PWM_RGB(int a) {
	
	DT_LEDaux = a;
	if(DT_LED < DT_LEDaux){DT_LED = DT_LEDaux;}

	if(LED==1 && direction == LED_UP){						// LED Red
		  if(PWMDTY4<DT_LED){PWMDTY4++;}
	else if(PWMDTY4==DT_LED){direction = LED_DOWN;}
	} 
	else if(LED==1 && direction == LED_DOWN){
		  if(PWMDTY4>LED_MIN){PWMDTY4--;} else {direction = LED_UP; DT_LED = DT_LEDaux; LED++;}
	}
	
	
	
	else if(LED==2 && direction == LED_UP){					// LED Purple
		  if(PWMDTY4<DT_LED){PWMDTY4++;PWMDTY3++;}
	else if(PWMDTY4==DT_LED){direction = LED_DOWN;}
	} 
	else if(LED == 2 && direction == LED_DOWN){
		  if(PWMDTY4>LED_MIN){PWMDTY4--; PWMDTY3--;} else {direction = LED_UP; DT_LED = DT_LEDaux; LED++;}
	}
	

	
	else if(LED==3 && direction == LED_UP){					// LED Blue
		  if(PWMDTY3<DT_LED){PWMDTY3++;}
	else if(PWMDTY3==DT_LED){direction = LED_DOWN;}
	} 
	else if(LED==3 && direction == LED_DOWN){
		  if(PWMDTY3>LED_MIN){PWMDTY3--;} else {direction = LED_UP; DT_LED = DT_LEDaux; LED++;}
	}



	else if(LED==4 && direction == LED_UP){					// LED Ice blue
		  if(PWMDTY6<DT_LED){PWMDTY3++;PWMDTY6++;}
	else if(PWMDTY6==DT_LED){direction = LED_DOWN;}
	} 
	else if(LED == 4 && direction == LED_DOWN){
		  if(PWMDTY6>LED_MIN){PWMDTY3--; PWMDTY6--;} else {direction = LED_UP; DT_LED = DT_LEDaux; LED++;}
	}
	
	
	
	else if(LED==5 && direction == LED_UP){					// LED Green
		  if(PWMDTY6<DT_LED){PWMDTY6++;}
	else if(PWMDTY6==DT_LED){direction = LED_DOWN;}
	} 
	else if(LED==5 && direction == LED_DOWN){
		  if(PWMDTY6>LED_MIN){PWMDTY6--;} else {direction = LED_UP; DT_LED = DT_LEDaux; LED++;}
	}
	

	
	else if(LED==6 && direction == LED_UP){					// LED Yellow
		  if(PWMDTY6<DT_LED){PWMDTY6++;PWMDTY4++;}
	else if(PWMDTY6==DT_LED){direction = LED_DOWN;}
	} 
	else if(LED == 6 && direction == LED_DOWN){
		  if(PWMDTY6>LED_MIN){PWMDTY6--; PWMDTY4--;} 
		  else {direction = LED_UP; DT_LED = DT_LEDaux; LED=1;}
	}
	
	
}


/*******************************************************************************
Function Name : main
Parameters    : NONE
Modifies      : NONE
Returns       : NONE
Notes         : Main function.
Issues        : NONE
*******************************************************************************/
void main(void) {             
    
          
  
  PLL_Init(0x47, 0x01, 0x01); // 32MHz BUSCLK  from 8MHz OSCCLK
  ADC_Init();
  PWM_Init();
  

  for(;;) {
    ADC_POT = ADC_Read();
    delay(3500);
    PWM_RGB(ADC_POT);              
  } 
}
