;*************************************************************************
;* (c) Copyright Freescale Semiconductor, Inc 2009, All Rights Reserved  *
;*************************************************************************
;*************************************************************************
;*                 EEPROM Emulation Driver for LG32                      *
;*                                                                       *
;* FILE NAME     :  EED_MidLevel.asm                                     *
;* DATE          :  19.01.2009                                           *
;* AUTHOR        :  Reference Design Team                                *
;* EMAIL         :  rfdsnIDC@freescale.com                               *
;*************************************************************************
;******************************* CHANGES *********************************
;*   1.00   19.01.2009      Saurabh Jhamb           Initial Version
;*************************************************************************


    XDEF FSL_Erase, FSL_Program, FSL_CopyRecord, FSL_SearchRecord
    XDEF FSL_SwapSector, FSL_InitSector, DoHVCopyDown, FSL_SectorStatus
    XREF FlashProgram, FlashErase, DataVerify, HighVoltage
    XDEF FSL_AddSectorSize, FSL_GetAddr, FSL_SubSectorSize, FSL_SearchLoop, FSL_CallBack

    ;XREF.B HVSubSize, HVSubSizeN
    ;XREF.W recID, erasingCycles, failedAddress, destination
    ;XREF.W cbAddr, emuCount
    ;XREF.W emuStartAddr, emuEndAddr, emuBlank, hvonstack
    ;XREF.B size
    ;XREF.W startAddr, buffer, endAddr, cacheTable, hvPosition, emuBuffer, hvCopyFr, hvCopyTo
    XREF.B HVSubSize, HVSubSizeN
    XREF.B recID, erasingCycles, failedAddress, destination, cbAddr
    XREF.B emuStartAddr, emuEndAddr, emuBlank
    XREF.B hvonstack, hvPosition, size, emuBuffer, hvCopyFr, hvCopyTo 
    XREF.B startAddr, buffer, endAddr, cacheTable, emuCount

    INCLUDE "ssd_sgf.inc"
    INCLUDE "EED_Flash.inc"


MIDLEVELCODE: SECTION

;*********************************************************************
;* DoHVCopyDown:
;* Copy a high voltage function from Flash into stack 
;* (pointed by hvPosition).
;*
;* hvCopyTo the destination location for copy down
;* hvCopyFr the source location for copy down
;* COP was serviced once per copying #BYTES_PER_COP bytes 
;*********************************************************************
DoHVCopyDown:
    LDHX    hvPosition        ; 4 cycles
    STHX    hvCopyTo          ; 4 cycles

    ; HVSubSize, the size to be copied, is the size of HighVoltage
    ; Modification to low level high voltage functions is not suggested.
    LDA     #HVSubSize        ; 2 cycles
    STA     hvonstack         ; reused to store the HighVoltage size to 
                              ; be copied 3 cycles
    BEQ     Copy_Done         ; 3 cycles

Copy_Start:
    JSR     FSL_CallBack 

    CLRX                      ; 1 cycle
    LDA     hvonstack         ; 3 cycles
    CMP     #BYTES_PER_COP    ; 2 cycles
    BLO     Last_Part         ; 3 cycles
    SUB     #BYTES_PER_COP    ; get the remaining bytes, 2 cycles
    STA     hvonstack         ; 3 cycles
    TAX                       ; 1 cycle
    LDA     #BYTES_PER_COP    ; 2 cycles

Last_Part:
    PSHX                      ; push the remaining bytes onto stack

CopyLoop:
    LDHX    hvCopyFr          ; 4 cycles
    PSHA                      ; 2 cycles
    LDA     ,X                ; get the data, 3 cycles
    AIX     #1                ; 2 cycles
    STHX    hvCopyFr          ; update the source location, 4 cycles
    LDHX    hvCopyTo          ; 4 cycles
    STA     ,X                ; save the data, 2 cycles
    AIX     #1                ; 2 cycles
    STHX    hvCopyTo          ; udpate the destination location, 4 cycles
    PULA                      ; 3 cycles
    DBNZA   CopyLoop          ; 4 cycles

    PULA                      ; 3 cycles
    TSTA                      ; 1 cycles
    BNE     Copy_Start        ; 3 cycles

Copy_Done:
    RTS                       ; 6 cycles

;***************************************************************************
;* FSL_Erase:
;*     Erase and verify the Flash pages specified by startAddr and size.
;* Parameters required:
;*     startAddr,endAddr, size: Input
;*     failedAddress: Output
;* Changes:
;*     hvonstack: updated after SSD HighVoltage is copied into stack.
;*           
;*                      STACK USAGE 
;*                      -----------
;*  Addr                      Stack Contents
;*  ------                   -----------------
;*  
;*  Addr_StackTop   SP    --> 
;*                  SP+1  -->  failedAddress (16 Bits)
;*                  SP+3  -->  - (16 Bits)
;*                  SP+5  -->  size (16 Bits)      
;*                  SP+7  -->  dest (16 Bits)
;*                  SP+9  -->  1  (8 Bits)      
;*                  SP+10 --> ReturnAddress (16Bits)
;*
;* Possible return codes:
;*     EE_OK
;*     EE_ERROR_NOT_BLANK
;***************************************************************************
FSL_Erase:
 IF (EE_BOUNDARY_CHECK)
    LDHX    startAddr            ; check whether start address from where 
    CPHX    #FLASH_START_ADDRESS ; erase has to be started is more than the 
    BHS     checkEndAddr         ; flash start address
    LDA     #EE_ERROR_RANGE
    JMP     End_BC

checkEndAddr:    
    CPHX    #FLASH_END_ADDRESS   ; check whether start address from where 
    BLO     addSize              ; erase has to be started is more than the 
    LDA     #EE_ERROR_RANGE      ; flash end address
    JMP     End_BC
addSize:    
    LDA     size
continueAdd:    
    PSHA                          ; add size to be erased to check whether 
    JSR     FSL_AddSectorSize     ; it goes beyond flash end address
    PULA
    DBNZA   continueAdd
    AIX     #-1                   ; get the end address of the final sector
    CPHX    #FLASH_END_ADDRESS
    BLS     startErase
    LDA     #EE_ERROR_RANGE
    JMP     End_BC
 ENDIF    
    
startErase:    
    ; Check if the Erase SSD is already on stack
    LDA     hvonstack
    CMP     #EMU_HV_ONSTACK      ; check if HighVoltage is on stack
    BEQ     Begin_Erase

    LDHX    #HighVoltage
    STHX    hvCopyFr             ; hvCopyFr stores from where to copy
    JSR     DoHVCopyDown         ; call DoHVCopyDown to copy HighVoltage 
                                 ; function onto the stack
    LDA     #EMU_HV_ONSTACK
    STA     hvonstack            ; mark hvonstack as EMU_HV_ONSTACK

Begin_Erase:
    LDA     #CMD_PAGE_ERASE      ; Push erase command on stack
    PSHA
    LDHX    startAddr            ; push destination address on stack
    PSHX
    PSHH
    LDA     size                 ; push size on stack
    PSHA
    LDHX    hvPosition           ; push highvoltage address on stack
    PSHX
    PSHH
    LDHX    cbAddr               ; Load CallBack in HX
    JSR     FlashErase
    AIS     #6                   ; adjust the SP
    TSTA
    BNE     End_BC
           
    JSR     FSL_CallBack 
    
    LDA     #1                   ; verify type for blank check is 1
    PSHA
    LDHX    startAddr            ; push startAddr address on to stack
    PSHX
    PSHH
    LDA     endAddr+1            ; size to be checked = endAddr-startAddr
    SUB     startAddr+1
    PSHA                         ; push LSB of size on stack
    LDA     endAddr
    SBC     startAddr
    PSHA                         ; push MSB of size on stack
    LDHX    1,SP
    AIX     #1
    STHX    1,SP
    AIS     #-2
    LDHX    #failedAddress       ; push failed address on stack
    PSHX
    PSHH
    LDHX    cbAddr
    JSR     DataVerify           ; perform BlankCheck after FlashErase
    AIS     #9                   ; adjust the SP
    TSTA
    BEQ     End_BC
    LDA     #EE_ERROR_NOT_BLANK

End_BC:
    RTS

;***************************************************************************
;* FSL_Program:
;*     Program and verify data of size 'size' source data in emuBuffer into 
;*     Flash.
;*     Parameters required:
;*     destination: Input
;*     size: Input
;*     buffer: Input
;*     failedAddress: Output
;*     hvonstack: updated after SSD Highvoltage is copied into stack.
;*           
;*                      STACK USAGE 
;*                      -----------
;*  Addr                      Stack Contents
;*  ------                   -----------------
;*  
;*  Addr_StackTop   SP    --> 
;*                  SP+1  -->  failedAddress (16 Bits)
;*                  SP+3  -->  buffer (16 Bits)
;*                  SP+5  -->  size (16 Bits)      
;*                  SP+7  -->  dest (16 Bits)
;*                  SP+9  -->  0  (8 Bits)      
;*                  SP+10 -->  ReturnAddress (16Bits)
;*
;*
;* Possible return codes:
;*     EE_OK
;*     EE_ERROR_VERIFY
;***************************************************************************
FSL_Program:
  IF (EE_BOUNDARY_CHECK)
    LDHX    destination
    CPHX    #FLASH_START_ADDRESS ; address at which pgm has to be started is 
    BHS     checkEndPgmAddr      ; more than the flash start address
    LDA     #EE_ERROR_RANGE
    JMP     Program_End
checkEndPgmAddr:
    CPHX    #FLASH_END_ADDRESS   ; check whether start address from where 
    BLO     addPgmSize           ; erase has to be started is more than the 
    LDA     #EE_ERROR_RANGE      ; flash end address
    JMP     Program_End 
addPgmSize:
    LDA     size                 ; get the size
    DECA                         ; size decremented by 1 to get to the correct 
    PSHH                         ; memory location 
    PSHX
    ADD     1,SP                 ; add size to calculate the address
    STA     1,SP
    CLRA 
    ADC     2,SP
    STA     2,SP
    PULX
    PULH
    CPHX    #FLASH_END_ADDRESS   ; compare the size calculated with the
    BLS     startPgm             ; FLASH_END_ADDRESS
    LDA     #EE_ERROR_RANGE      ; flash end address
    JMP     Program_End 
 ENDIF 
startPgm:    
    ; one COP service is enough for programming at most 2 Bytes
    
    JSR     FSL_CallBack 
    
    ; Check if the Program is already on stack
    LDA     hvonstack
    CMP     #EMU_HV_ONSTACK      ; check if HighVoltage is on stack
    BEQ     Begin_Program

    LDHX    #HighVoltage
    STHX    hvCopyFr
    JSR     DoHVCopyDown         ; copy HighVoltage function onto the stack
    LDA     #EMU_HV_ONSTACK
    STA     hvonstack            ; mark hvonstack as EMU_HV_ONSTACK
                              
Begin_Program:
       
    LDHX    destination          ; push blank address onto stack
    PSHX
    PSHH
    LDA     size
    PSHA
    LDA     #0
    PSHA
    LDHX    buffer               ; push source address on stack     
    PSHX
    PSHH
    LDHX    hvPosition           ; push highvoltage on stack
    PSHX
    PSHH
    LDHX    cbAddr
    JSR     FlashProgram         ; execute the FlashProgram 
    AIS     #8                   ; adjust stack
    TSTA
    BNE     Program_End          ; if any error go to Pre_End_Prog

    
    
    LDA     #0                   ; Call DataVerify for program verify
    PSHA    
    LDHX    destination
    PSHX
    PSHH
    LDA     size
    PSHA
    LDA     #0
    PSHA
    LDHX    buffer        
    PSHX
    PSHH
    LDHX    #failedAddress
    PSHX
    PSHH
    LDHX    cbAddr
    JSR     DataVerify 
    AIS     #9                   ; adjust the stack pointer
    TSTA
    BEQ     Program_End
    LDA     #EE_ERROR_VERIFY
    
Program_End:
    
    RTS

;*********************************************************************
;* FSL_CopyRecord:
;*     Write user data into Flash pointed by emuBlank in record format
;* Parameters required:
;*     recID: Input
;*     failedAddress (reused as the start addr of raw data): Input
;*     emuBlank: Input
;*
;* Changes:  -
;*           
;*                      STACK USAGE 
;*                      -----------
;*  Addr                      Stack Contents
;*  ------                   -----------------
;*                  
;*  Addr_StackTop   SP    --> 
;*                  SP+1  -->  failedAddress (16 bits)
;*                  SP+3  -->  ReturnAddress (16 bits) 
;* 
;* Possible return codes:
;*     EE_OK
;*     EE_ERROR_VERIFY
;*********************************************************************
FSL_CopyRecord:
    LDHX    failedAddress   
    PSHH
    PSHX
    LDHX    emuBlank        ; store first blank address as destination      
    AIX     #1              ; increment X to avoid programming status byte
    STHX    destination
    LDA     #DATA_ID_SIZE
    STA     size            ; store size to be programmed
    LDHX    #recID          ; get the record ID
    STHX    buffer          ; store recID in buffer
    JSR     FSL_Program     ; program the record ID
    TSTA
    BNE     Pre_Exit_Copy

    LDA     #DATA_SIZE      ; get the user raw data length
    STA     size            ; store size to be programmed
    PULX
    PULH
    STHX    buffer          ; contains the address of the data

    LDHX    destination     ; increment destination by data Id size.
    AIX     #DATA_ID_SIZE
    STHX    destination
    JSR     FSL_Program     ; program the user data byte by byte
    TSTA
    BNE     Exit_Copy       ; check for error, if any go to Pre_Exit_Copy

    LDA     #1              ; get the user raw data length
    STA     size            ; store size to be programmed
 
    LDA     #RECORD_STATUS_COMPLETED
    STA     emuCount
    LDHX    #emuCount
    STHX    buffer          ; store status to be programmed in buffer

    LDHX    emuBlank        ; store first blank address as destination
    STHX    destination
    JSR     FSL_Program     ; update record status 
    BRA     Exit_Copy

Pre_Exit_Copy:
    AIS     #2              ; adjust the SP

Exit_Copy:
    RTS
;*********************************************************************
;* FSL_SearchRecord:
;*     Search record for a specific ID in a sector. For each record in the
;*     sector, check the ID first, if ID matches, check the record status.
;* Parameters required:
;*     startAddr, endAddr: Input
;*     failedAddr,emuBuffer:Output
;*
;* Changes:  -
;*           
;*                      STACK USAGE 
;*                      -----------
;*  Addr                      Stack Contents
;*  ------                   -----------------
;*                  
;*  Addr_StackTop   SP    --> 
;*                  SP+1  -->  endAddr       (16 bits) 
;*                  SP+3  -->  ReturnAddress (16 bits) 
;*
;*-------------------------------------------
;* Data Status   | Data Id | Status of record|
;*---------------|---------|-----------------|
;* $FF	         | $FF 	   | Erased.         |
;* $FF	         | XX 	   | under update.   |
;* XX	           | XX 	   | Valid.          |
;*--------------------------------------------
;*
;* 
;* Possible return codes:
;*     EE_OK
;*********************************************************************
FSL_SearchRecord:
    LDHX    #$FFFF
    STHX    buffer               ; clear the found history

    LDHX    startAddr 
    AIX     #3
    STHX    failedAddress        ; store the location of first 
                                 ; record in failedAddress
    JSR     FSL_AddSectorSize
    AIX     #-4
    PSHX
    PSHH
    
CheckRecID:
        
    JSR     FSL_CallBack 
    
    LDHX    failedAddress
    CPHX    1,SP
    BHS     Reach_End            ; Check if end of Sector is reached

    AIX    #$1
    IF (DATA_ID_SIZE == 1)       ; if user has configured Data ID size as 1 
    LDA     ,X
    CMP     #RECORD_ID_MAX       ; Check if recID is $FF
    BEQ     Reach_End            ; return with blank location in failedAddress
    CMP     recID                ; check if the ID matches ID under search.
    BNE     Update_To_NextRec
    AIX     #-1
    ELSE                         ; user has configured Data ID size as 2 
    LDHX     ,X
    CPHX    #RECORD_ID_MAX       ; Check if recID is $FFFF
    BEQ     Reach_End            ; return with blank location in failedAddress
    CPHX    recID                ; check if the ID matches ID under search.
    BNE     Update_To_NextRec
    LDHX    failedAddress
    ENDIF

    LDA     #RECORD_STATUS_ERASED; check if the status is erased.
    CMP     ,X
    BEQ     Update_To_NextRec    ; if yes go to check next record.
    STHX    buffer               ; record is valid. save location in emuBuffer

Update_To_NextRec:
    LDHX    failedAddress
    AIX     #RECORD_LENGTH       ; total length of a single record
    STHX    failedAddress        ; update failedAddress to next record
    BRA     CheckRecID

Reach_End:
    AIS     #2
    CLRA                         

Exit_Search:
    RTS


;*********************************************************************
;* FSL_InitSector:
;*     Initialize one sector to make it ready for EEPROM Emulation.
;*     The sector is erased and the erase cycles are programmed.
;*     The sector becomes alternative.
;* Parameters required:
;*     startAddr, endAddr: Input
;* Changes:
;*     erasingCycles: used to save the erasing cycles of the sector.
;*
;*                      STACK USAGE 
;*                      -----------
;*  Addr                      Stack Contents
;*  ------                   -----------------
;*                  
;*  Addr_StackTop   SP    --> 
;*                  SP+1  -->  erasingCycles (16 bits)
;*                  SP+3  -->  status byte (8 bits) 
;*                  SP+4  -->  ReturnAddress (16 bits) 
;*            
;* 
;* Possible return codes:
;*     EE_OK
;*     EE_ERROR_NOT_BLANK
;*     EE_ERROR_VERIFY
;*********************************************************************
FSL_InitSector:
    PSHA                         ; push the status to be programmed
    
    JSR     FSL_CallBack 
    
    ;checking for Sector status
    JSR    FSL_SectorStatus
    CMP    #EE_SECTOR_BLANK      ; if sector is blank, get the existing 
    BEQ    Zero_Cycles
    BRA    Getting_Cycles        ; erasing cycles otherwise fetch the 
Zero_Cycles:
    LDHX   #$00                  ; correct erasing cycles.
    PSHX
    PSHH
    BRA    Cycles_Got

Getting_Cycles:
    LDHX    startAddr
    LDHX    1,X                  ; get the erasingCycles of the sector. 
    AIX     #1                   
    PSHX                         ; store the cycles in stack
    PSHH
    
Cycles_Got:
    MOV     #$01,size            ; store number of sectors to be erased.
    JSR     FSL_Erase            ; erase the current sector
    TSTA
    BEQ     Program_EraseCycles
    AIS     #3                   ; correct stack and exit if erase failed.
    BRA     Exit_InitSector

Program_EraseCycles:       
    PULH
    PULX
    STHX    failedAddress
    LDHX    #failedAddress
    STHX    buffer               ; get erasing cycles in buffer
    LDHX    startAddr
    AIX     #1
    STHX    destination          ; store destination with erase cycle addr.
    LDA     #$02
    STA     size
    JSR     FSL_Program          ; program back the erasing cycles
    TSTA
    BEQ     Program_Status
    AIS     #1                   ; correct stack and exit if error
    BRA     Exit_InitSector

Program_Status:
    PULA
    STA     emuCount
    LDHX    #emuCount
    STHX    buffer               ; get status byte and store in buffer
    LDHX    startAddr
    STHX    destination          ; store destination with startAddr
    LDA     #01
    STA     size
    JSR     FSL_Program          ; update the sector status 
    TSTA
    BNE     Exit_InitSector

Done_InitSector:
    CLRA                         ; set return code as EE_OK

Exit_InitSector:
    RTS


;*********************************************************************
;* FSL_SectorStatus:
;*     Read the sector status and return the value.
;* Parameters required:
;*     startAddr, endAddr: Input
;*     erasingCycles: Output
;*
;*                      STACK USAGE 
;*                      -----------
;*  Addr                      Stack Contents
;*  ------                   -----------------
;*                  
;*  Addr_StackTop   SP    --> 
;*                  SP+1  -->  erasingCycles (16 bits)
;*                  SP+3  -->  ReturnAddress (16 bits) 
;*            
;*----------------------------------------------------------
;* Sector Status | erase cycles | Data Id | Status of sector|
;*---------------|--------------|---------|-----------------|
;* $FF	         |   $FFFF	    | $FF  	  | Blank.          |
;* $FF	         |   XXXX	    | $FF  	  | Alternative.    |
;* $FF	         |   XXXX	    | XX      | Update          |
;* XX 	         |   XXXX	    | XX      | Active.         |
;*-----------------------------------------------------------
;* Possible return codes:
;*     #EE_SECTOR_ACTIVE
;*     #EE_SECTOR_ALTERNATIVE
;*     #EE_SECTOR_BLANK
;*     #EE_SECTOR_UPDATE
;*********************************************************************
FSL_SectorStatus:
   ;checking for value of Data ID

    JSR     FSL_CallBack 
            
    LDHX   startAddr             ; Get the address of the sector in h:x
    AIX    #$01                  
    LDHX   ,X
    CPHX   #$FFFF                ; If erased cycles is 0xFFFF,  
    BEQ    Report_Blank          ; then the sector is Blank
    LDHX   startAddr
    LDA    ,X                    ; read the sector status byte in A
    CMP    #SECTOR_STATUS_ERASED ; See if the sector status byte is 0xFF
    BNE    Report_Active         ; if it is not, the sector is active
    AIX    #4
    IF (DATA_ID_SIZE == 1)       ; if user has configured Data ID size as 1 
    LDX    ,X
    CPX    #$FF                  ; check if ID is $FF
    BEQ    Report_Alt            ; Sector is 'Alternative'
    ELSE                         ; user has configured Data ID size as 2 
    LDHX   ,X                   
    CPHX   #$FFFF                ; check if ID is $FFFF
    BEQ    Report_Alt            ; Sector is 'Alternative'
    ENDIF
    
Report_Start:
    LDA    #EE_SECTOR_UPDATE     ; Return status as update      
    BRA    Exit_Status        
    
Report_Active:
    LDA    #EE_SECTOR_ACTIVE     ; Return status as active
    BRA    Exit_Status

Report_Blank:
    LDA    #EE_SECTOR_BLANK      ; Return status as Blank
    BRA    Exit_Status

Report_Alt:
    LDA    #EE_SECTOR_ALTERNATIVE; Return status as Alternative
    
Exit_Status:
    RTS
;****************************************************************************
;* FSL_SwapSector:
;*     Copy all latest valid records from oldest active sector to 
;*     oldest alternative sector.
;* Parameters required:
;*     startAddr, endAddr:Input
;*     emuStartAddr, emuEndAddr, emuBlank: Input/Output
;* Changes:
;*         erasingCycles reused to hold the starting of oldest active sector.
;*         recID reused to search and swap.
;*                      STACK USAGE 
;*                      -----------
;*  Addr                      Stack Contents
;*  ------                   -----------------
;*                  
;*  Addr_StackTop   SP    --> 
;*                  SP+1  -->  erasingCycles (16 bits)
;*                  SP+3  -->  ReturnAddress (16 bits) 
;*
;* Possible return codes:
;*     EE_OK
;*     EE_ERROR_NOT_BLANK
;*     EE_ERROR_VERIFY
;*     EE_ERROR_SSTAT
;****************************************************************************
FSL_SwapSector:
;*********************************************************
;The current active sectors address is in emuStartAddr and emuEndAddr.
;The ready sectors address is in startAddress and endAddress.
;The number of ready sectors are EED_READY_SECTORS. So the oldest active
;sector will be EED_READY_SECTORS sizes after the currentReady sector.
;This address of the oldest active shall be in emuBlank.

;***********Get the oldest active sector address in erasing cycles

; store the record ID in stack    
    IF (DATA_ID_SIZE == 1)       ; if user has configured Data ID size as 1 
    LDA     recID
    PSHA
    ELSE
    LDHX    recID
    PSHH
    PSHX
    ENDIF

    JSR     FSL_CallBack 
        
    LDHX    emuStartAddr
    LDA     #EED_READY_SECTORS

Find_Oldest_Sector:    
    PSHA                         ; use stack to loop and calculate the 
                                 ; address of oldest active
    CPHX    #(EFLASH_END_ADDRESS- EED_SECTOR_SIZE +1)
    BNE     End_Not_Reached
    LDHX    #EFLASH_START_ADDRESS       
    BRA     Search
    
End_Not_Reached:    
    JSR     FSL_AddSectorSize     
Search:
    STHX    startAddr
    JSR     FSL_SectorStatus
    CMP     #EE_SECTOR_ACTIVE
    BEQ     Oldest_Found
    CMP     #EE_SECTOR_ALTERNATIVE
    BNE     temp6
    LDHX    startAddr
    PULA
    DBNZA   Find_Oldest_Sector   ; 4 cycles

Oldest_Found:
    AIS     #$1                  ; Correct stack
    LDHX    startAddr
    AIX     #(3 - DATA_RECORD_SIZE)
    STHX    erasingCycles
    JSR     FSL_AddSectorSize
    AIX     #-(4 - DATA_RECORD_SIZE)    
    STHX    endAddr
    
;*******Read the record ID to be swapped

Read_RecordID:

    JSR     FSL_CallBack 
    
    LDHX    erasingCycles
    AIX     #(DATA_RECORD_SIZE)
    STHX    erasingCycles        ; store the address of the record 
                                 ; in erasingCycles (reused)
    CPHX    endAddr
    BHS     temp5
    LDA     ,X
    CMP     #$FF
    BEQ     Read_RecordID
    AIX     #1
    IF (DATA_ID_SIZE == 1)       ; if user has configured Data ID size as 1 
    LDA     ,X
    CMP     #$FF                 ; check if ID is $FF
    BEQ     Read_RecordID
    STA     recID
    ELSE                         ; user has configured Data ID size as 2 
    LDHX    ,X                   
    CPHX    #$FFFF               ; check if ID is $FFFF
    BEQ     Read_RecordID
    STHX    recID
    CPHX    #(RECORD_ID_MAX>>8)
    BHI     Record_NotIn_CacheTbl 
    ENDIF
;*****************Search for the record ID in Cache Table 
Search_Record:
    LDA    recID
    CMP    #EED_CACHETABLE_ENTRY
    BHS    Record_NotIn_CacheTbl
    LSLA                         ; double the recID to point correctly to 
                                 ; cache table entries
    LDHX   #cacheTable           ; get base address of cache table in HX
    PSHX
    ADD    1,SP
    STA    1,SP
    PULX
    BCC    No_Carry
    PSHH
    PULA
    ADD    #01
    PSHA
    PULH  
  
No_Carry:    
    LDHX   ,X
    STHX   buffer                ; store the address of that record in 
                                 ; buffer
    BRA    Store_Address
;*****************Search for the record ID in All active Sectors
Record_NotIn_CacheTbl:
    LDHX    emuStartAddr
    STHX    startAddr            ; store emuStartAddress in startAddr
    
    JSR     FSL_SearchLoop
    TSTA
    BEQ     Store_Address
    BRA     Read_RecordID
  
temp6:
    BRA    Exit_Swp

    ;the latest intended record location was saved in emuBuffer
Store_Address:
    CLRA
    CPHX    erasingCycles
    BLO     Read_RecordID
    CPHX    endAddr
    BLO     Swap_Record    
temp:
    BRA     Read_RecordID
temp5:
    BRA     Swp_Done
Swap_Record:    
    AIX     #(DATA_STATUS_SIZE+DATA_ID_SIZE)  
    STHX    failedAddress  
    JSR     FSL_CopyRecord       ; copy the data to alternative sector   
    TSTA
    BNE     temp1
    LDA     recID
    CMP     #EED_CACHETABLE_ENTRY
    BHS     Update_emuBlank
    LSLA                         ; double the recID to point correctly to 
                                 ; cache table entries
    LDHX    #cacheTable
    PSHH
    PSHX
    ADD     1,SP
    STA     1,SP
    LDA     2,SP
    ADC     #0
    STA     2,SP
    PULX
    PULH
                                
    MOV     destination,X+       ; store new address in cache table 
    MOV     destination+1,X+
    CLRA
Update_emuBlank:    
    LDHX    emuBlank
    AIX     #RECORD_LENGTH
    STHX    emuBlank             ; sector blank += RECORD_LENGTH
    BRA     temp
    
temp1:
    BRA     Exit_Swp

Swp_Done:
    IF (DATA_ID_SIZE == 1)       ; if user has configured Data ID size as 1 
    PULA
    STA     recID
    ELSE    
    PULH
    PULX
    STHX    recID
    ENDIF
    ; Program the status of the sector to active
    LDHX    emuStartAddr
    STHX    destination
    LDA     #1
    STA     size
    LDA     #00
    STA     emuCount
    LDHX    #emuCount
    LDHX    buffer
    JSR     FSL_Program
    TSTA
    BEQ     Cont_Erase
    BRA     Exit_Swp

Cont_Erase:    
    ; erase the oldest active sector
    LDHX    endAddr
    JSR     FSL_SubSectorSize
    AIX     #1
    STHX    startAddr
    LDA     #$FF
    JSR     FSL_InitSector

Exit_Swp:
        RTS
        
;****************************************************************************
;* FSL_SearchLoop:
;*     Loop across all the active sectors to search for the record ID. 
;*     
;* Parameters required:
;*     startAddr, endAddr:output
;*                      STACK USAGE 
;*                      -----------
;*  Addr                      Stack Contents
;*  ------                   -----------------
;*                  
;*  Addr_StackTop   SP    --> 
;*                  SP+1  -->  ReturnAddress (16 bits) 
;*
;****************************************************************************        
FSL_SearchLoop:

    JSR     FSL_CallBack 

Search_Loop:    
    JSR     FSL_SearchRecord     ; search for the recID
    LDHX    buffer               ; Check if buffer content is $FFFF
    CPHX    #$FFFF
    BNE     Suc_Exit_Srch        ; Exit as address found
    LDHX    startAddr
    CPHX    #EFLASH_START_ADDRESS; point to the next sector in the 
    BNE     Next_Search_Loop     ; round robin queue
    LDHX    #EFLASH_END_ADDRESS   
    AIX     #1
Next_Search_Loop:
    JSR     FSL_SubSectorSize    
    STHX    startAddr
    JSR     FSL_SectorStatus     ; proceed only if the sector is active
    CMP     #EE_SECTOR_ACTIVE
    BNE     Fail_Exit_Srch       ; if record not found store $FFFF in table
    BRA     Search_Loop   

Suc_Exit_Srch:
    LDA     #0
    BRA     Exit_Srch

Fail_Exit_Srch:
    LDA     #EE_ERROR_NOFND
    
Exit_Srch:
    RTS        
;****************************************************************************
;* FSL_GetAddr:
;*     Update startAddr and endAddr. 
;*     
;* Parameters required:
;*     startAddr, endAddr:output
;*                      STACK USAGE 
;*                      -----------
;*  Addr                      Stack Contents
;*  ------                   -----------------
;*                  
;*  Addr_StackTop   SP    --> 
;*                  SP+1  -->  ReturnAddress (16 bits) 
;*
;****************************************************************************        
FSL_GetAddr:
    STHX    startAddr            ; Store HX in startAddr
    JSR     FSL_AddSectorSize
    AIX     #-1
    STHX    endAddr              ; Store HX in endAddr
    RTS
;****************************************************************************
;* FSL_AddSectorSize:
;*     Incement H:X by a sector size . 
;*     
;* Parameters required:
;*     emuStartAddr,emuBlank,emuEndAddr:output
;*                      STACK USAGE 
;*                      -----------
;*  Addr                      Stack Contents
;*  ------                   -----------------
;*                  
;*  Addr_StackTop   SP    --> 
;*                  SP+1  -->  ReturnAddress (16 bits) 
;*
;****************************************************************************     
FSL_AddSectorSize:
    PSHH
    PULA
    ADD     #(EED_SECTOR_SIZE>>8); proceed to initialize next sector
    PSHA
    PULH
    RTS
;****************************************************************************
;* FSL_SubsectorSize:
;*     Subtract contents of HX with sector size (EED_SECTOR_SIZE). 
;*     
;* Parameters required:
;*     None
;*                      STACK USAGE 
;*                      -----------
;*  Addr                      Stack Contents
;*  ------                   -----------------
;*                  
;*  Addr_StackTop   SP    --> 
;*                  SP+1  -->  ReturnAddress (16 bits) 
;*
;****************************************************************************     
    
    
FSL_SubSectorSize:
    PSHH
    PULA
    SUB     #(EED_SECTOR_SIZE>>8); subtract sector size with endAddr
    PSHA
    PULH 
    RTS   

****************************************************************************
;* FSL_SubsectorSize:
;*     Calls back application function of Application 
;*     
;* Parameters required:
;*     None
;*                      STACK USAGE 
;*                      -----------
;*  Addr                      Stack Contents
;*  ------                   -----------------
;*                  
;*  Addr_StackTop   SP    --> 
;*                  SP+1  -->  ReturnAddress (16 bits) 
;*
;****************************************************************************  
FSL_CallBack:               
    LDHX    cbAddr               ; load callback address into H:X 
    CPHX    #NULL_CALLBACK       ; check if callback has to be bypassed      
    BEQ     SkipCB                ; if yes skip callback  
    JSR     ,x                   ; else call the callback subroutine

SkipCB:
    RTS          