;*************************************************************************
;* (c) Copyright Freescale Semiconductor, Inc 2009, All Rights Reserved  *
;*************************************************************************
;*************************************************************************
;*                 EEPROM Emulation Driver for LG32                      *
;*                                                                       *
;* FILE NAME     :  EED_HighLevel.asm                                    *
;* DATE          :  19.01.2009                                           *
;* AUTHOR        :  Reference Design Team                                *
;* EMAIL         :  rfdsnIDC@freescale.com                               *
;*************************************************************************
;******************************* CHANGES *********************************
;*   1.00   19.01.2009      Saurabh Jhamb           Initial Version
;*************************************************************************


    XDEF FSL_InitEeprom, FSL_ReadEeprom, FSL_WriteEeprom,FSL_DeinitEeprom
    XDEF FSL_ReportEepromStatus

    XREF FSL_Erase, FSL_Program, FSL_InitSector, FSL_CopyRecord
    XREF FSL_SearchRecord, FSL_SwapSector,FSL_SectorStatus, DoHVCopyDown
    XREF FSL_GetAddr, FSL_AddSectorSize, FSL_SubSectorSize, FSL_SearchLoop, FSL_CallBack
    
    XREF FlashInit

    ;XREF.B HVSubSize, HVSubSizeN
    ;XREF.W recID, erasingCycles, failedAddress, destination, cbAddr
    ;XREF.W emuStartAddr, emuEndAddr, emuBlank, emuCount
    ;XREF.B hvonstack, size
    ;XREF.W startAddr, buffer, endAddr, cacheTable, hvPosition, emuBuffer, hvCopyFr, hvCopyTo
    XREF.B HVSubSize, HVSubSizeN
    XREF.B recID, erasingCycles, failedAddress, destination, cbAddr
    XREF.B emuStartAddr, emuEndAddr, emuBlank
    XREF.B hvonstack, hvPosition, size, emuBuffer, hvCopyFr, hvCopyTo 
    XREF.B startAddr, buffer, endAddr, cacheTable, emuCount

    INCLUDE "EED_Flash.inc"
    
FLASH_CLOCK_DIVIDER:      equ   $44       ;Must be changed as per the bus clock, refer RM

HIGHLEVELCODE: SECTION

;*********************************************************************
;* FSL_InitEeprom:
;*     Initialize the EEPROM Emulation and determine the active Sector .
;* Parameters required:
;*     emuStartAddr, emuEndAddr, emuBlank, erasingCycles: Output
;* Changes:
;*     destination: reused to save the blank sector address.
;*     hvonstack, hvPosition: to save the high voltage function type 
;*     and location in stack.
;*
;* Possible return codes:
;*     EE_OK
;*     EE_ERROR_NOT_BLANK
;*     EE_ERROR_VERIFY
;*     (EE_ERROR_SECURITY_ENABLED | EE_ERROR_CLOCK_SETTING)
;*
;*             STACK USAGE OF FSL_InitEeprom
;*
;*  Addr                     Stack Contents
;*  ------                   -----------------
;*                  
;*  Addr_StackTop   SP    --> 
;*                  SP+1  -->  loop (8 bits)
;*                  SP+2  -->  Previous_Blank_Space (16 bits) 
;*                  SP+4  -->  High_Voltage_Function (29 bytes)
;*
;* Additional stack usage for addition - 1 byte.              
;*
;********************************************************************

FSL_InitEeprom:
    
    LDHX    #$0000
    STHX    emuStartAddr             
    STHX    emuEndAddr            

    JSR FSL_CallBack
    
;**************Stack initialization for local variable and local function***
      
    AIS     #HVSubSizeN          ; allocate some space for HV operation 
                                 ; on stack
    TSX
    STHX    hvPosition           ; store the stack address for HighVoltage
                                 ; function in "hvPosition"
    AIS     #-3                  ; allocate space for blank space and loop

;**************Global Variable initialization for re use****

    CLR     hvonstack            ; clear number of blank sectors 
    CLR     emuCount             ; clear number of active sectors
    LDHX    #$0000               ; clear previous 
    STHX    2,SP                  
    LDHX    #-1 
    STHX    erasingCycles        ; initialize erasingCycles
    STHX    destination          ; initialise blank sector address and 
    STHX    emuBuffer            ; updated sector address.

;**************Call FlashInit of Low Level to initialize the flash clock***

    LDA     #FLASH_CLOCK_DIVIDER ; Call the FlashInit function to initialize 
                                 ; the FLASH Clock .
    JSR     FlashInit
    TSTA                         ; Check for any error
    BEQ     Continue             ; If no error then continue 
    JMP     Exit_Init            ; else go to the Exit_Init

Continue:       
    IF (DATA_ID_SIZE == 1)       ; if user has configured Data ID size as 1 
    LDA     #RECORD_ID_MAX
    STA     recID                ; store $FF in recID
    ELSE                         ; user has configured Data ID size as 2 
    LDHX    #RECORD_ID_MAX
    STHX    recID                ; store $FFFF in recID
    ENDIF
    LDHX    #EFLASH_START_ADDRESS                   

;**************Find the status of all the sectors ***

Sector_Status:
    CPHX    #EFLASH_END_ADDRESS  ; Check if the end address is reached
    BHS     Check_Brownout       ; if yes, check for any brown out
    JSR     FSL_GetAddr
    AIX     #1
    STHX    endAddr
    JSR     FSL_SectorStatus     ; check the status of this sector   
    CMP     #EE_SECTOR_ACTIVE
    BEQ     Update_Active        ; Branch to the corresponding location
    CMP     #EE_SECTOR_BLANK
    BEQ     Update_Blank
    CMP     #EE_SECTOR_UPDATE
    BEQ     Update_Update
    BRA     Next_Loop
Update_Blank:
    INC     hvonstack            ; increment the blank sector count
    LDHX    startAddr
    STHX    destination          ; store blank sector address.
    BRA     Next_Loop
Update_Update:
    LDHX    startAddr
    STHX    emuBuffer            ; store updated sector address.
    BRA     Next_Loop
Update_Active:
    INC     emuCount             ; increment active sector count
    JSR     FSL_SearchRecord     ; search the active sector for blank space
    LDHX    failedAddress
    CPHX    endAddr
    BHI     Next_Loop            ; the active sector that contains the highest
    LDA     endAddr+1            ; blank space is the active sector that is
    SUB     failedAddress+1      ; to be used.
    PSHA
    PULX                         ; compare the blank space with the earlier
    LDA     endAddr              ; active sector's and update the emuBlank,
    SBC     failedAddress        ; emuStartAddr and emuEndAddr accordingly.
    PSHA
    PULH       
    CPHX    2,SP
    BLO     Next_Loop
    STHX    2,SP
    LDHX    failedAddress
    STHX    emuBlank             ; store the blank space
    LDHX    startAddr  
    STHX    emuStartAddr         ; store the emulation start address
    LDHX    endAddr
    AIX     #-1
    STHX    emuEndAddr           ; store the emulation end address
    BRA     Next_Loop

Next_Loop:
    LDHX    endAddr
    BRA     Sector_Status
Check_Brownout:                  ; check if there has been a brownout
    AIX     #-1                  ; check if no of ACTIVE clusters are  
    STHX    endAddr              ; equal to number determined at compile
    LDA     emuCount             ; time
    CMP     #EED_SECTOR_NUMBER
    BEQ     Correct_Active
    BLO     temp9
    JMP     Init_Oldest
temp9:
    JMP     Check_FirstTime
Correct_Active:
    LDHX    emuBuffer            ; number of ACTIVE clusters are same as specified
    CPHX    #$FFFF               ; continue initializing Alternative
    BNE     temp10
    LDA     hvonstack
    BEQ     Check_Active
    LDHX    destination
Continue_Alternative: 
    JSR     FSL_SectorStatus     ; check if the startAddr points to ACTIVE
    CMP     #EE_SECTOR_ACTIVE
    BEQ     temp13               ; Branch to the corresponding location
    LDA     hvonstack            ; Load Blank cluster count in stack
    PSHA
    LDA     #$FF 
    JSR     FSL_InitSector
    TSTA
    BNE     temp15
temp12: 
    LDHX    startAddr
    JSR     FSL_SubSectorSize
    STHX    startAddr
    LDHX    endAddr
    JSR     FSL_SubSectorSize
    STHX    endAddr
    PULA
    DBNZA   Continue_Alternative            
;**************Check if there had been any brown out and correct it ***
Check_Active:
    LDA     emuEndAddr+1         ; get lower byte
    SUB     emuBlank+1
    STA     emuBuffer+1          ; emuBuffer reused
    LDA     emuEndAddr           ; get higher byte
    SBC     emuBlank
    STA     emuBuffer
    LDHX    emuBuffer
    AIX     #1
    STHX    emuBuffer
    PSHX
    PSHH
    CPHX    #(EED_SECTOR_SIZE-3)
    BEQ     Check_Sector
Keep_Sector:
    PULH
    PULX
    CPHX    #DATA_RECORD_SIZE
    BHS     temp4
Get_NextActive:
    LDHX    emuEndAddr           ; Keep on looking for all other ACTIVE clusters
    AIX     #1
    STHX    startAddr
    JSR     FSL_SectorStatus
    CMP     #EE_SECTOR_ACTIVE
    BNE     temp11
    STHX    emuStartAddr
    JSR     FSL_AddSectorSize
    AIX     #-1 
    STHX    emuEndAddr
    JSR     FSL_SearchRecord
    LDHX    failedAddress
    STHX    emuBlank
    JMP     Get_NextActive
temp11:     
    JMP     Init_CacheTable
temp10:
    JMP     Update_Sector 
temp13:
    LDA     hvonstack            ; To maintain Equal Pushes and Pulls
    ADD     #01
    PSHA 
    JMP     temp12   
temp4:
    JMP     temp14 
temp15:
    AIS     #$1
    JMP     Exit_Init           
Check_Sector:
    LDHX    emuStartAddr
    CPHX    #EFLASH_START_ADDRESS
    BNE     Get_Sector
    LDHX    #(EFLASH_END_ADDRESS+1)
Get_Sector:
    AIX     #-1
    STHX    startAddr
    JSR     FSL_SubSectorSize
    AIX     #1
    STHX    startAddr
    JSR     FSL_SectorStatus
    CMP     #EE_SECTOR_ACTIVE
    BNE     Keep_Sector
    JSR     FSL_AddSectorSize
    AIX     #-1
    STHX    emuEndAddr
    JSR     FSL_SearchRecord
    LDA     emuEndAddr+1         ; get lower byte
    SUB     failedAddress+1
    STA     emuBuffer+1          ; emuBuffer reused
    LDA     emuEndAddr           ; get higher byte
    SBC     failedAddress
    STA     emuBuffer    
    LDHX    emuBuffer
    CPHX    #DATA_RECORD_SIZE
    BLO     Keep_Sector1
    LDHX    startAddr
    STHX    emuStartAddr
    LDHX    failedAddress
    STHX    emuBlank
    AIS     #$2
    JMP     Init_CacheTable

Keep_Sector1
    JMP     Keep_Sector

Update_Sector:
    JSR     FSL_GetAddr          ; get startAddr and endAddr   
    LDA     #$FF
    JSR     FSL_InitSector       ; initialize blank to alternative
    TSTA   
    BNE     temp2
temp14:
    BRA     temp7                ; Go to initialization of cache table after
                                 ; brown out correction.
Init_Oldest:
    LDA     #(EED_READY_SECTORS)
    PSHA
    LDHX    emuStartAddr
Find_Oldest_Sector:              ; This excess active is the oldest active.
    CPHX    #(EFLASH_END_ADDRESS- EED_SECTOR_SIZE +1)
    BEQ     Sector_End               ; so find out the oldest active sector.
    JSR     FSL_AddSectorSize
    BRA     Search
Sector_End:
    LDHX    #EFLASH_START_ADDRESS

Search:    
    JSR     FSL_GetAddr       ; get startAddr and endAddr 
    JSR     FSL_SectorStatus
    CMP     #EE_SECTOR_ACTIVE
    BEQ     Oldest_Found
    CMP     #EE_SECTOR_ALTERNATIVE
    BEQ     Continue_Alt
    AIS     #1
    BRA     Init_CacheTable
Continue_Alt:
    PULA
    DBNZA   Find_Oldest_Sector   ; 4 cycles

Oldest_Found:
    AIS     #1
    LDA     #$FF
    STHX    startAddr
    JSR     FSL_InitSector       ; initialize active to alternative
    TSTA   
    BEQ     Init_CacheTable
temp2:
    BRA     temp3
temp7:
    BRA     Init_CacheTable    
;**************Initialize all sectors if it is the first time***

Check_FirstTime:

    LDA     #EED_SECTOR_NUMBER
    LDHX    #EFLASH_START_ADDRESS
Init_As_Act:
    STA     1,SP                 ; store the number of active sectors in stack
    LDA     #$00                 ; initialize these sectors as active
    BRA     Init_Sector
Init_As_Alt:
    LDA     #$FF                 ; initialize the remaining as alternative
Init_Sector:
    PSHA
    CPHX    #EFLASH_END_ADDRESS  ; check if all sectors are initialized
    BHS     Init_emuStart        ; if yes, proceed to initialize cache table
    JSR     FSL_GetAddr          ; get startAddr and endAddr 
    STHX    endAddr    
    PULA                                    
    JSR     FSL_InitSector
    TSTA
    BEQ     Continue1                              
temp3:
    BRA     Exit_Init
Continue1:
    
    LDHX    endAddr
    AIX     #1 
    LDA     1,SP
    DBNZA   Init_As_Act          ; loop till all sectors are initialized.
    BRA     Init_As_Alt
Init_emuStart: 
    AIS     #1
    LDHX    #EFLASH_START_ADDRESS 
    STHX    emuStartAddr
    AIX     #3
    STHX    emuBlank
    JSR     FSL_AddSectorSize
    AIX     #-4
    STHX    emuEndAddr        
      

;**************Initialize Cache Table***
    
Init_CacheTable:
    LDHX    #cacheTable          ; get the address of cache table
    LDA     #EED_CACHETABLE_ENTRY; get the number of entries in the table
    STA     1,SP                 ; store it in stack to loop
    BEQ     Exit_Init
    LDA     #$0                  ; start with recID = 0

Cache_Loop:
    STHX    destination          ; store the cache table in destination.
    STA     recID
    LDHX    emuEndAddr
    STHX    endAddr
    LDHX    emuStartAddr
    
    JSR     FSL_SearchLoop
   
Store_Record:
    
    LDA     buffer               ; store the address in the cache table
    LDHX    destination
    STA     ,X
    AIX     #1
    LDA     buffer+1
    STA     ,X
    AIX     #1
    LDA     recID
    INCA        
    DBNZ    1,SP,Cache_Loop      ; quit the function if cahe is filled
    CLRA
    
Exit_Init:

    AIS     #$3
    AIS     #(HVSubSize)         ; restore the SP
    CLR     hvonstack
    RTS          

;***************************************************************************
;* FSL_ReadEeprom:
;*     Read the data from EEPROM, it just return the starting address of 
;*     This function makes use of the cache table for a faster return.
;*     user raw data.
;* Parameters required:
;*     recID: Input
;*     emuStartAddr, emuEndAddr, emuBlank: Input
;*     destination: Output
;*                 STACK USAGE 
;*                      -----------
;*  Addr                      Stack Contents
;*  ------                   -----------------
;*                  
;*  Addr_StackTop   SP    --> 
;*                  SP+1  -->  ReturnAddress (16 bits) 
;*
;* Possible return codes:
;*     EE_OK
;*     EE_ERROR_NOFND
;*     EE_ERROR_IDRNG
;***************************************************************************
FSL_ReadEeprom:
       ;checking for size of Data ID
        
    IF (DATA_ID_SIZE == 1)       ; if user has configured Data ID size as 1 
    LDA     recID
    CMP     #RECORD_ID_MAX       ; Check if the record ID is out of valid 
                                 ; range
    BEQ     ReadID_OutOfRange     
    ELSE                         ; user has configured Data ID size as 2 
    LDHX    recID
    CPHX    #RECORD_ID_MAX       ; Check if the record ID is out of valid 
                                 ; range
    BEQ     ReadID_OutOfRange     
    CPHX    #(RECORD_ID_MAX>>8)
    BHI     Record_NotIn_CacheTbl 
    ENDIF
     
    ;checking in cache table
Search_Record:
    LDA    recID                 ; Check if record ID is in cache table
    CMP    #EED_CACHETABLE_ENTRY
    BHS    Record_NotIn_CacheTbl
    LSLA                         ; if yes, double the recID to point correctly
                                 ; to cache table entries
    LDHX   #cacheTable           ; get base address of cache table in HX
    PSHX
    ADD    1,SP
    STA    1,SP                  ; Add the offset to base address to get to 
    PULX                         ; correct cache table location.
    BCC    No_Carry
    PSHH
    PULA
    ADD    #01
    PSHA
    PULH  
  
No_Carry:    
    LDHX   ,X
    CPHX   #$FFFF                 ; check cache table is initialized
    BEQ    Not_Found
    STHX   buffer                ; store the address of that record in 
                                 ; buffer  from cache table
    BRA    Store_Address
    
Record_NotIn_CacheTbl:
    LDHX    emuEndAddr           ; store emuEndAddress in endAddr
    STHX    endAddr
    LDHX    emuStartAddr
    
    STHX    startAddr
    JSR     FSL_SearchLoop
    TSTA
    BEQ     Store_Address
    BRA     Not_Found

    ;the latest intended record location was saved in buffer
Store_Address:
    CLRA
    LDHX    buffer
    AIX     #(DATA_STATUS_SIZE+DATA_ID_SIZE)  ; Data Id size is user 
                                 ; configurable
    STHX    destination          ; save the user raw data starting 
                                 ; address into "destination"
    BRA     Exit_Read

Not_Found:
    LDA     #EE_ERROR_NOFND      ; if not found set the return code as 
                                 ;"EE_ERROR_NOFND"
    BRA     Exit_Read

ReadID_OutOfRange:
    LDA     #EE_ERROR_IDRNG      ; if record ID is out of valid range set 
                                 ; return code as "EE_ERROR_IDRNG"
Exit_Read:
    RTS

;***************************************************************************
;* FSL_WriteEeprom:
;*     Encapsulate user's data into record format and write the record into
;*     the EEPROM. Calls Swap sector function if there is no enough space.
;* Parameters required:
;*     recID: Input
;*     emuStartAddr, emuEndAddr, emuBlank: Input/Output
;*     destination:output 
;*                 STACK USAGE 
;*                      -----------
;*  Addr                      Stack Contents
;*  ------                   -----------------
;*                  
;*  Addr_StackTop   SP    --> 
;*                  SP+1  -->  ReturnAddress (16 bits) 
;*
;* Possible return codes:
;*     EE_ERROR_NOMEM
;*     EE_ERROR_IDRNG
;***************************************************************************
FSL_WriteEeprom:

    JSR     FSL_CallBack

    ; At first, clear hvonstack and adjust SP
    CLR     hvonstack
    AIS     #HVSubSizeN          ; allocate some space for HV operation
                                 ; on Stack
    TSX                          ; H:X = SP + 1
    STHX    hvPosition           ; store the H:X in the hvPosition to 
                                 ; know where to copy the HighVoltage
    LDA     recID
    CMP     #RECORD_ID_MAX       ; Check if the recID is in the valid 
                                 ; range
    BEQ     temp8                ; if recID is not in valid range then 

    LDA     #(EED_SECTOR_NUMBER+1)
    PSHA
                                 ; go to WrtID_OutOfRange
Start_GetFree:

    ; buffer(free size) = emuEndAddr - emuBlank + 1 -- buffer is reused here.
    LDA     emuEndAddr+1         ; get lower byte
    SUB     emuBlank+1
    STA     buffer+1
    LDA     emuEndAddr           ; get higher byte
    SBC     emuBlank
    STA     buffer

    LDHX    buffer                  
    AIX     #1                   ; H:X save the free size
    
    CPHX    #RECORD_LENGTH       ; Compare the free size with the 
                                 ; record length
    BHS     Enough_Space         ; if enough space available then

    DEC     1,SP    
    BEQ     No_Memory

                             ; write the record
    LDHX    emuEndAddr         
    CPHX    #EFLASH_END_ADDRESS  ; check if emulatiom end has been 
                                 ; reached
    BNE     End_Not_Reached1
    LDHX    #EFLASH_START_ADDRESS; load emulation start address in HX
    BRA     Get_endAddr
End_Not_Reached1:
    AIX     #1                   ; add 1 to X to get the start address 
                                 ; of next 'Active' sector
Get_endAddr:
    STHX    startAddr            ; store start address of the current 
    STHX    emuStartAddr
    AIX     #3
    STHX    emuBlank
    JSR     FSL_AddSectorSize
    AIX     #-4
    STHX    emuEndAddr       
    STHX    endAddr              ; store the end address of active
    JSR     FSL_SectorStatus
    CMP     #EE_SECTOR_ACTIVE    ; check if the sector is active
    BEQ     Enough_Space
    JSR     FSL_SwapSector       ; else do a swapping of the clusters
    TSTA
    BEQ     Start_GetFree 
    AIS     #1 
    BRA     Exit_Wrt 
temp8:
    BRA     WrtID_OutOfRange   
Enough_Space:
    AIS     #1 
    LDHX    emuBlank
    STHX    destination         
    LDHX    emuBuffer 
    STHX    failedAddress        ; save data address in 
                                 ; failedAddress
    JSR     FSL_CopyRecord       ; call FSL_CopyRecord to copy the source 
                                 ; data into destination
    TSTA
    BNE     Exit_Wrt
    LDA     recID                ; Cache table has to be updated if recID 
    CMP     #EED_CACHETABLE_ENTRY; is a part of it.
    BHS     Update_emuBlank
    LSLA                         ; double the recID to point correctly to 
                                 ; cache table entries
    LDHX    #cacheTable
    PSHH
    PSHX
    ADD     1,SP
    STA     1,SP
    LDA     2,SP
    ADC     #0
    STA     2,SP
    PULX
    PULH
                                
    MOV     destination,X+       ; store new address in cache table 
    MOV     destination+1,X+
    
    ; update blank location
Update_emuBlank:    
    LDHX    emuBlank
    AIX     #RECORD_LENGTH
    STHX    emuBlank             ; sector blank += RECORD_LENGTH
    CLRA
    BRA     Exit_Wrt

No_Memory:
    AIS     #1 
    LDA     #EE_ERROR_NOMEM      ; if not enough memory, set return code  
                                 ; as "EE_ERROR_NOMEM"
    BRA     Exit_Wrt

WrtID_OutOfRange:
    LDA     #EE_ERROR_IDRNG      ; if record ID is out of valid range set 
                                 ; return code as "EE_ERROR_IDRNG"

Exit_Wrt:
    AIS     #HVSubSize           ; restore the SP
    CLR     hvonstack
    RTS


;***************************************************************************
;* FSL_DeinitEeprom:
;*     Release the ranges of Flash pages used for EEPROM emulation.
;* Parameters required:
;*     startAddr,endAddr,size:output
;* Changes:
;*     hvonstack, hvPosition: to save the high voltage function type 
;*     and location in stack.
;*
;*                      STACK USAGE 
;*                      -----------
;*  Addr                      Stack Contents
;*  ------                   -----------------
;*                  
;*  Addr_StackTop   SP    --> 
;*                  SP+1  -->  ReturnAddress (16 bits) 
;*
;* Possible return codes:
;* 
;***************************************************************************
FSL_DeinitEeprom:
    ;At first, clear hvonstack and adjust SP
    CLR     hvonstack
    AIS     #HVSubSizeN          ; allocate some space for HV operation 
                                 ; on Stack
    TSX                          ; H:X = SP + 1
    STHX    hvPosition           ; store the H:X in the hvPosition to 
                                 ; know where to copy the HighVoltage
     
           
    LDHX    #EFLASH_START_ADDRESS
    STHX    startAddr            ; store emulated Eeprom start address  
                                 ; in startAddr
                                      
    LDHX    #EFLASH_END_ADDRESS
    STHX    endAddr              ; store emulated Eeprom end address 
                                 ; in endAddr
    LDA     #EED_SECTORS_ALLOTED
    STA     size
    JSR     FSL_Erase            ; erase Emulated Eeprom


Exit_Deinit:
    AIS     #HVSubSize           ; restore the SP
    CLR     hvonstack
    RTS

;***************************************************************************
;* FSL_ReportEepromStatus:
;*     Check record status and return erase cycles of the emulation cluster.
;* Parameters required:
;*     
;*     erasingCycles: Output
;*     emuStartAddr, emuEndAddr, emuBlank: Input
;*
;*                      STACK USAGE 
;*                      -----------
;*  Addr                      Stack Contents
;*  ------                   -----------------
;*  Addr_StackTop   SP    --> 
;*                  SP+1  -->  ReturnAddress (16 bits) 
;*
;* Possible return codes:
;*     EE_ERROR_SSTAT
;*
;***************************************************************************
FSL_ReportEepromStatus:

    
    CLRA
    LDHX    emuStartAddr         ; get emuStartAddr in startAddr
    STHX    startAddr
    LDHX    emuEndAddr           ; get emuEndAddr in endAddr
    STHX    endAddr
    JSR     FSL_SectorStatus     ; call FSL_SectorStatus
    CMP     #EE_SECTOR_ACTIVE    ; compare with Active sector status
    BNE     Invalid_Status
    AIX     #1                   ; erasing cycles location = emuStartAddr+1
    LDHX    ,X
    STHX    erasingCycles        ; get the erase cycles of the sector in 
                                 ; erasingCycles
    BRA     Exit_Report


Invalid_Status:
    LDA     #EE_ERROR_SSTAT

Exit_Report:
    RTS
