/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2011-2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file iic.c
*
* @author B37811
*
* @version 0.0.1
*
* @date Sep 19, 2011
*
* @brief application entry point which performs application specific tasks. 
*
*******************************************************************************
*
* provide a demo for how to initialize the PT60, output messages via SCI, 
* flash operations, etc.
* NOTE:
*	printf call may occupy a lot of memory (around 1924 bytes), so please
*	consider your code size before using printf.
******************************************************************************/

#include <hidef.h> /* for EnableInterrupts macro */
#include "derivative.h" /* include peripheral declarations */
#include "iic.h"

#pragma MESSAGE DISABLE C4002 /* Warning C4002: Result not used */  


/***********************************************************************************************\
* Private prototypes
\***********************************************************************************************/

void I2C_Start(void);
void I2C_Stop(void);
void I2C_RepeatStart(void);
void I2C_Delay(void);
void I2C_CycleWrite(byte bout);
byte I2C_CycleRead(byte ack);

/***********************************************************************************************\
* Private memory declarations
\***********************************************************************************************/

#pragma DATA_SEG __SHORT_SEG _DATA_ZEROPAGE

static byte error;
static word timeout;

#pragma DATA_SEG DEFAULT

#define BUFFER_OUT_SIZE       8


/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: init_I2C
   *
   * @brief description: I2C Initialization, Set Baud Rate and turn on I2C
   *        
   * @parameter:  none 
   *
   * @return: none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void Init_I2C(void)
{
    /* Configure GPIO for I2C function */
	//SYS_SOPT1_IICPS = 0; // default, IIC SCL and SDA are mapped on PTA3 and PTA2 or PTB7 and PTB6
	I2C_F  = 0x1D;   // 100K baud rate
    I2C_C1 = I2C_C1_IICEN_MASK |    // enable IIC 
             I2C_C1_MST_MASK   |    // master
             I2C_C1_TX_MASK;           
}



/***********************************************************************************************\
* Private functions
\***********************************************************************************************/

/*********************************************************\
* Initiate I2C Start Condition
\*********************************************************/
void I2C_Start(void)
{
  I2C_C1_MST = 1;
  timeout = 0;
  while ((!I2C_S_BUSY) && (timeout<10000))
    timeout++;
  if (timeout >= 10000)
    error |= 0x01;
} //*** Wait until BUSY=1


/*********************************************************\
* Initiate I2C Stop Condition
\*********************************************************/
void I2C_Stop(void)
{
  I2C_C1_MST = 0;
  timeout = 0;
  while ( (I2C_S_BUSY) && (timeout<10000))
    timeout++;
  if (timeout >= 10000)
    error |= 0x02;
} //*** Wait until BUSY=0


/*********************************************************\
* Initiate I2C Repeat Start Condition
\*********************************************************/
void I2C_RepeatStart(void)
{
  I2C_C1_RSTA = 1;
  timeout = 0;
  while ((!I2C_S_BUSY) && (timeout<10000))
    timeout++;
  if (timeout >= 10000)
    error |= 0x04;
} //*** Wait until BUSY=1


/*********************************************************\
* I2C Delay
\*********************************************************/
void I2C_Delay(void)
{
  byte I2Cd;
  for (I2Cd=0; I2Cd<100; I2Cd++);
}


/*********************************************************\
* I2C Cycle Write
\*********************************************************/
void I2C_CycleWrite(byte bout)
{
  timeout = 0;
  while ((!I2C_S_TCF) && (timeout<10000))
    timeout++;
  if (timeout >= 10000)
    error |= 0x08; 
  I2C_D = bout; 
  timeout = 0;
  while ((!I2C_S_IICIF) && (timeout<10000))
    timeout++;
  if (timeout >= 10000)
    error |= 0x10;
  I2C_S_IICIF = 1; 
  if (I2C_S_RXAK)
    error |= 0x20;
}


/*********************************************************\
* I2C Cycle Read
\*********************************************************/
byte I2C_CycleRead(byte ack)
{
  byte bread; 
  timeout = 0;
  while ((!I2C_S_TCF) && (timeout<10000))
    timeout++;
  if (timeout >= 10000)
    error|=0x08;
  I2C_C1_TX = 0;
  I2C_C1_TXAK = ack;
  bread = I2C_D; 
  timeout = 0;
  while ((!I2C_S_IICIF) && (timeout<10000))
    timeout++;
  if (timeout >= 10000)
    error |= 0x10;
  I2C_S_IICIF=1;
  return bread;
}

/*
 * end here
 */
