/**************************************************************************
*
* Copyright 2005-2011 by Andrey Butok. Freescale Semiconductor, Inc.
*
**********************************************************************/ /*!
*
* @file fnet_checksum.c
*
* @date May-31-2011
*
* @version 0.1.15.0
*
* @brief Internet checksum implementation.
*
***************************************************************************/

#include "fnet_config.h"
#include "fnet_checksum.h"
#include "fnet_ip.h"

/*RFC:
    The checksum field is the 16 bit one's complement of the one's
    complement sum of all 16 bit words in the header and text.  If a
    segment contains an odd number of header and text octets to be
    checksummed, the last octet is padded on the right with zeros to
    form a 16 bit word for checksum purposes.  The pad is not
    transmitted as part of the segment.  While computing the checksum,
    the checksum field itself is replaced with zeros.
*/



/************************************************************************
* NAME: fnet_checksum_low
*
* DESCRIPTION: Calculates Internet checksum of nb chain.
*
*************************************************************************/
#if !FNET_CFG_OVERLOAD_CHECKSUM_LOW

static unsigned long fnet_checksum_low(unsigned long sum, int current_length, unsigned short *d_ptr)
{
        unsigned short p_byte1;
        
        while((current_length -= 32) >= 0)
        {
            sum += *d_ptr++;
            sum += *d_ptr++;
            sum += *d_ptr++;
            sum += *d_ptr++;
            sum += *d_ptr++;
            sum += *d_ptr++;
            sum += *d_ptr++;
            sum += *d_ptr++;
            sum += *d_ptr++;
            sum += *d_ptr++;
            sum += *d_ptr++;
            sum += *d_ptr++;
            sum += *d_ptr++;
            sum += *d_ptr++;
            sum += *d_ptr++;
            sum += *d_ptr++;
        }
        current_length += 32;

        while((current_length -= 8) >= 0)
        {
            sum += *d_ptr++;
            sum += *d_ptr++;
            sum += *d_ptr++;
            sum += *d_ptr++;
        }
        current_length += 8;

        while((current_length -= 2) >= 0)
            sum += *d_ptr++;  
            
        if(current_length += 2)
        {
            p_byte1 = (unsigned short)((*((unsigned short *)d_ptr)) & FNET_NTOHS(0xFF00));
           
            sum += (unsigned short)p_byte1;
        }
        return sum;
} 
#else

extern unsigned long fnet_checksum_low(unsigned long sum, int current_length, unsigned short *d_ptr);

#endif

static unsigned long fnet_checksum_nb(fnet_netbuf_t * nb, int len)
{
    fnet_netbuf_t *tmp_nb;
    unsigned short *d_ptr;
    unsigned short p_byte2;
    unsigned long sum = 0;
    int current_length;

    tmp_nb = nb;

    d_ptr = tmp_nb->data_ptr;        /* Store the data pointer of the 1st net_buf.*/

    if(nb->length > len)
        current_length = len;        /* If no more net_bufs to proceed.*/
    else
        current_length = (int)nb->length; /* Or full first net_buf.*/

    while(len)
    {
        len -= current_length;       /* current_length bytes should be proceeded.*/

        sum = fnet_checksum_low(sum, current_length, d_ptr); 

	/* Fix for Calypso */
        if(len == 0)
	{
	    return sum;
	}
	/* Fix for Calypso */
	
        tmp_nb = tmp_nb->next;
        d_ptr = tmp_nb->data_ptr;
        
        if(current_length & 1)
        {
            if(len)
            {
                /* If previous fragment was odd, add in first byte in lower 8 bits. */
                p_byte2 = fnet_ntohs((unsigned short)((*((unsigned char *)d_ptr)) & 0x00FF));
                d_ptr = (unsigned short *)((unsigned char *)d_ptr + 1);
                
                sum += (unsigned short)p_byte2;
                len--;
                current_length = -1;
            }
            else
                current_length = 0;
        }
        else
        {
            current_length = 0;
        }
          

        if(tmp_nb->length > len)
            current_length = len;
        else
            current_length += tmp_nb->length;
    }

    return sum;
}

/************************************************************************
* NAME: fnet_checksum
*
* DESCRIPTION: Calculates one's complement (Internet) checksum.
*
*************************************************************************/
unsigned short fnet_checksum(fnet_netbuf_t * nb, int len)
{
    unsigned long sum = fnet_checksum_nb(nb, len);

    /* Add potential carries - no branches. */

    sum = (sum >> 16) + (sum & 0xffff); /* Add in accumulated carries */
    sum += sum >> 16;                   /* Add potential last carry   */

    return (unsigned short)(0xffff & ~sum);
}

/************************************************************************
* NAME: fnet_checksum_pseudo
*
* DESCRIPTION: Calculates  one's complement (Internet) checksum of 
*              the IP pseudo header
*
*************************************************************************/
unsigned short fnet_checksum_pseudo( fnet_netbuf_t *nb,
                                     unsigned long ip_src,   unsigned long ip_dest,
                                     unsigned char protocol, unsigned short protocol_len )
{
    unsigned long sum = fnet_checksum_nb(nb, protocol_len);
    
    sum += (ip_src >> 16) + (ip_src & 0x0000FFFF) + 
            (ip_dest >> 16) + (ip_dest & 0x0000FFFF) + 
            fnet_htons((unsigned short)protocol) + fnet_htons(protocol_len);

    sum = (sum >> 16) + (sum & 0xffff); /* Add in accumulated carries */
    sum += sum >> 16;                   /* Add potential last carry   */

    return (unsigned short)(0xffff & ~sum);
}


