/**************************************************************************
* 
* Copyright 2005-2013 by Andrey Butok and Arturo Inzunza.
*                        Freescale Semiconductor, Inc.
*
**********************************************************************/ /*!
*
* @file fnet_mpc_timer.c
*
* @author Gordon Jahn
*
* @date September-05-2011
*
* @version 0.1.15.0
*
* @brief Bolero specific timers implementation.
*
***************************************************************************/

#include "fnet_config.h"

#if FNET_MPC
#include "fnet.h"
#include "fnet_timer_prv.h"
#include "fnet_isr.h"
#include "fnet_mpc.h"

#if FNET_CFG_DEBUG_STACK
    extern unsigned char __SP_INIT[];
    #define SP_INIT    (unsigned long)__SP_INIT
    unsigned long fnet_dbg_stack_max = 0;
#endif

#define FNET_TIMER_VECTOR_NUMBER FNET_CFG_CPU_TIMER_VECTOR_NUMBER   /* Number of timer interrupt vector.*/
#define FNET_TIMER_INT_LEVEL   (FNET_CFG_CPU_TIMER_VECTOR_PRIORITY) /* Timer interrupt level. Ignored for MCF V1 */

#if FNET_CFG_MPC_TIMER_DTIM     /* DMA Timer (Default for MPC) */
    #define FNET_TIMER_NUMBER      (FNET_CFG_CPU_TIMER_NUMBER)    /* Timer number (according to UM) */
    #define FNET_TIMER_CLKIN_PER_MS (FNET_CPU_CLOCK_KHZ)
#endif

/************************************************************************
* NAME: fnet_cpu_timer_handler_top
*
* DESCRIPTION: Top interrupt handler. Increment fnet_current_time 
*              and interrupt flag. 
*************************************************************************/
static void fnet_cpu_timer_handler_top( void )
{
#if FNET_CFG_MPC_TIMER_DTIM     /* PIT Timer is used normally */  
    /* Clear timer event condition.
     */
   	FNET_MPC_PITRTI_TFLG(FNET_TIMER_NUMBER) = 0x1;
#endif    
    
    /* Update RTC counter. 
     */
    fnet_timer_ticks_inc(); 
    
#if FNET_CFG_DEBUG_STACK    
    {
        unsigned long stack_value;
        unsigned long stack_usage;
        
        stack_value = fnet_MPC_sp_rd();
        
        stack_usage = SP_INIT - stack_value;
        if(stack_usage > fnet_dbg_stack_max)
            fnet_dbg_stack_max = stack_usage;    
    }
#endif 
}

/************************************************************************
* NAME: fnet_cpu_timer_init
*
* DESCRIPTION: Starts TCP/IP hardware timer. delay_ms - period of timer (ms)
*         e.g. Time-out period = (1/FNET_CFG_SYSTEM_CLOCK_KHZ)x(1)x(124+1)x528x100 = 100 ms
*************************************************************************/
int fnet_cpu_timer_init( unsigned int period_ms )
{
    int result;
    
    /* Install interrupt handler.
     */

    result = fnet_isr_vector_init(FNET_TIMER_VECTOR_NUMBER, fnet_cpu_timer_handler_top,
                                              fnet_timer_handler_bottom, FNET_TIMER_INT_LEVEL);

    if(result == FNET_OK)
    {
    #if FNET_CFG_MPC_TIMER_DTIM     /* DMA Timer (Default for MPC) */  

	FNET_MPC_PITRTI_MCR = 0x04;

	FNET_MPC_PITRTI_TCTRL(FNET_TIMER_NUMBER) = 0x0;

// On Panther, PIT runs on peripheral bridge clock 0, which is divided by MC_CGM_SC_DC0 (in this case over 4)
	FNET_MPC_PITRTI_LDVAL(FNET_TIMER_NUMBER) = period_ms * (FNET_TIMER_CLKIN_PER_MS/4); 
	
	FNET_MPC_PITRTI_TCTRL(FNET_TIMER_NUMBER) = 0x3;
    #endif
    
    }

    return result;
}

/************************************************************************
* NAME: fnet_cpu_timer_release
*
* DESCRIPTION: Relaeses TCP/IP hardware timer.
*              
*************************************************************************/
void fnet_cpu_timer_release( void )
{
#if FNET_CFG_MPC_TIMER_DTIM     /* DMA Timer (Default for MCF) */  
	FNET_MPC_PITRTI_TCTRL(FNET_TIMER_NUMBER) = 0x0;
   	FNET_MPC_PITRTI_TFLG(FNET_TIMER_NUMBER) = 0x1;
#endif    
    
    /* Free interrupt handler res.
     */
    fnet_isr_vector_release(FNET_TIMER_VECTOR_NUMBER);
}

#endif /*FNET_MPC*/ 
