/**************************************************************************
* 
* Copyright 2005-2011 by Andrey Butok. Freescale Semiconductor, Inc.
*
**********************************************************************/ /*!
*
* @file fnet_poll.h
*
* @author Andrey Butok
*
* @date Sep-2-2011
*
* @version 0.1.14.0
*
* @brief FNET Services polling mechanism API.
*
***************************************************************************/

#ifndef _FNET_POLL_H_

#define _FNET_POLL_H_

/*! @addtogroup fnet_polling 
* The polling mechanism enables the execution of registered services
* (DHCP client, TFTP client, Shell, Telnet server or HTTP server) in 
* "background" - during the application 
* idle time. Typically, the polling callback is registered during the service
* initialization (for example the @ref fnet_http_init() or other service initialization
* function).
* In order to make the polling mechanism work, the user application should 
* call the @ref fnet_poll() API function periodically, during the idle time.@n
* @n
* Configuration parameters:
* - @ref FNET_CFG_POLL_MAX  
*/
/*! @{ */

/**************************************************************************/ /*!
 * @brief Descriptor of a registered service.
 ******************************************************************************/
typedef unsigned int fnet_poll_desc_t;

/**************************************************************************/ /*!
 * @brief Service callback function prototype.
 *
 * @param service_param   This parameter is assigned during 
 *                        a service registration by the @ref 
 *                        fnet_poll_register().
 *                        
 ******************************************************************************/
typedef void (* fnet_poll_service_t)(void* service_param);

/***************************************************************************/ /*!
 *
 * @brief    Main polling function.
 *
 * @see fnet_poll_register()
 * 
 ******************************************************************************
 *
 * This function calls all registered service routines.@n
 * The user application should call this function periodically, after any service 
 * initialization.
 *
 ******************************************************************************/
void fnet_poll(void);


/***************************************************************************/ /*!
 *
 * @brief    Unregisters all registered service routines.
 *
 * @see fnet_poll_register(), fnet_poll_unregister()
 *
 ******************************************************************************
 *
 * This function unregisters all registered service routines from 
 * the polling list.
 *
 ******************************************************************************/ 
void fnet_poll_release(void);

/***************************************************************************/ /*!
 *
 * @brief    Registers the service routine in the polling list.
 *
 * @param service        Pointer to the service-polling routine.
 *
 * @param service_param  Service-polling-routine-specific parameter.
 *
 * @return This function returns:
 *   - Service descriptor, if no error occurs.
 *   - @ref FNET_ERR, if an error occurs.
 *
 * @see fnet_poll_unregister()
 *
 ******************************************************************************
 *
 * This function adds the service routine into the polling list.@n
 * This function is usually called during a service initialization. 
 * User application should not call this function directly.
 *
 ******************************************************************************/
fnet_poll_desc_t fnet_poll_register( fnet_poll_service_t service, void *service_param );

/***************************************************************************/ /*!
 *
 * @brief    Unregisters the service routine.
 *
 * @param desc       Service descriptor to be unregistered.
 *
 * @return This function returns:
 *   - @ref FNET_OK, if no error occurs.
 *   - @ref FNET_ERR, if an error occurs.
 *
 * @see fnet_poll_register()
 *
 ******************************************************************************
 *
 * This function unregisters the service routine assigned to the @c desc 
 * descriptor.@n
 * This function is usually called during a service release. 
 * User application should not call this function directly.
 *
 ******************************************************************************/
int fnet_poll_unregister( fnet_poll_desc_t desc );

/*! @} */

#endif
